%% Define a Custom Error Model Using a Function Handle  
% This example defines a custom error model using a function handle and
% adds noise to simulation data of a radioactive decay model with the
% first-order reaction  ${{dz} \over {dt}} = c \cdot x$, where |x| and |z|
% are species, and |c| is the forward rate constant.

%% 
% Load the sample project containing the radiodecay model |m1|. 
sbioloadproject radiodecay;  

%% 
% Suppose you have a simple custom error model with a standard mean-zero
% and unit-variance (Gaussian) normal variable |e|, simulation results |f|,
% and two parameters |p1| and |p2|: $y = f + p1 + p2 * e$
%
% Define a function handle that represents the error model. 
em = @(y,p1,p2) y+p1+p2*randn(size(y));  

%% 
% Simulate the model. 
[t,sd,names] = sbiosimulate(m1);  

%% 
% Plot the simulation results and hold the plot. 
plot(t,sd);
legend(names,'AutoUpdate','off');
hold on   

%% 
% Sample the error using the previously defined custom function with two
% parameters set to 0.5 and 30, respectively. 
sdNoisy = sbiosampleerror(sd,em,{0.5,30});  

%% 
% Plot the noisy simulation data. 
plot(t,sdNoisy);   

%% 
% You can also apply a different error model to each state, which is a column
% in |sd|. Suppose you want to apply the custom error model (|em|) to the
% first column (species |x| data) and the proportional error model to the
% second column (species |z| data). 
hold off
sdNoisy = sbiosampleerror(sd,{em,'proportional'},{{0.5,30},0.3});
plot(t,sd);
legend(names,'AutoUpdate','off');
hold on
plot(t,sdNoisy);    



%% 
% Copyright 2012 The MathWorks, Inc.