#ifndef __RANDOM_FUNCTIONS_INCLUDED__
	#define __RANDOM_FUNCTIONS_INCLUDED__
	#include "parametersAndLibraries.h"
	#include "classesAndFunctions.h"
			
	class cell{
		public:
			cell(int type_in){//default constructor
				type=type_in;
				timePositionUpdated=0;
				x=-1; y=-1; z=-1;//this must be set later
				vx=0; vy=0; vz=0;//0 velocity
				freePathRemaining=0; //failedContact=-1;
				cogAgRatio=0; probActivation=0; timeAntigenCountLastUpdated=0; //Prob activation also doubles as 'activation state' for T cells
			}

			double get_time_position_updated(){ return timePositionUpdated; }
			void set_position(double t, double nx, double ny, double nz){
				x=nx; y=ny; z=nz; timePositionUpdated=t;
			}
			double get_x() const {return x;}
			double get_y() const {return y;}
			double get_z() const {return z;}
			vector<double> get_pos() const {return {x,y,z};}

			void set_velocity(double nvx, double nvy, double nvz){
				vx=nvx; vy=nvy; vz=nvz;
			}
			double get_vx() const {return vx;}
			double get_vy() const {return vy;}
			double get_vz() const {return vz;}
			vector<double> get_vel() const {return {vx,vy,vz};}
		
			void set_free_path_remaining(double fpr) {freePathRemaining=fpr;}
			double reduce_free_path_remaining(double num) {return(freePathRemaining-=num);}
			double get_free_path_remaining() {return freePathRemaining;}

			void set_failed_interaction_ID(int dNum) {failedContact=dNum;}
			void unset_failed_interaction_ID() 		{failedContact=-1;}
			int get_failed_interaction_ID() 		{return failedContact;}

			void set_num_cog_antigen(double t, double cogRatio, double cogPrecision, double *cognateAgProbabilityTable){
				cogAgRatio=cogRatio;
				probActivation=cognateAgProbabilityTable[int(cogRatio/cogPrecision)];
				timeAntigenCountLastUpdated=t;
			}
			double get_cog_antigen_ratio() const { return cogAgRatio; }
			double get_prob_activation() const {return probActivation;}

			void update_num_cog_antigen(double time, double antigenDecayRate, double cogPrecision, double *cognateAgProbabilityTable){
				cogAgRatio *= exp(-antigenDecayRate*(time-timeAntigenCountLastUpdated)); //Exponential decay
				probActivation = cognateAgProbabilityTable[int(cogAgRatio/cogPrecision)];
				timeAntigenCountLastUpdated=time;
			}
			double get_time_antigen_updated(){ return timeAntigenCountLastUpdated; }
			void set_timeAntigenUpdated(double t) {timeAntigenCountLastUpdated=t;}

			void TC_set_activation_state(int input){ probActivation=input; }
			int TC_get_activation_state(){ return probActivation; }
			int DC_cannot_activate_t_cells(){ return probActivation<PROBABILITY_TOLERANCE; }

			double get_next_velocity_change(){ return nextVelocityChange; }
			void set_next_velocity_change(double input) { nextVelocityChange=input; }
		private:
			double x,y,z, vx,vy,vz, freePathRemaining, timePositionUpdated=0;
			unsigned int type; int failedContact=-1;
			double cogAgRatio, probActivation;//the number of cognate antigen on this DC, and so the probability it will activate a bound Tcell
			double timeAntigenCountLastUpdated; //The last time that the number of cognate antigen on this DC was correct - between then and now, some will have unbound.
			double nextVelocityChange; //To keep track of when an event will change this cell's velocity, so as not to add events that will never occur to the queue
	};

	//Cell definitions
		#define TCELL_TYPE (0)
		#define DCELL_TYPE (1)
		#define NOTCELL_TYPE (2)

	//Event definitions
		#define VEL_CHANGE_EVENT (0)
		#define SPHERE_EDGE_EVENT (1)
		#define INTERACT_EVENT (2)
		#define DC_INTERACT_EVENT (3)
		#define ENTER_POS_EVENT (4)
		#define LEAVE_POS_EVENT (5)
		#define DC_ENTRY_EVENT (6)
		#define DC_DEATH_EVENT (7)
		#define UNSET_FAILURE_ID_EVENT (8)
		#define MEASURE_OBSERVABLES_EVENT (9)
		#define GRAPHICAL_UPDATE_EVENT (10)

		#define X_NEG (0)
		#define X_POS (1)
		#define Y_NEG (2)
		#define Y_POS (3)
		#define Z_NEG (4)
		#define Z_POS (5)

	//Flags
		#define USING_SOURCE (0)

	class eventTime{
		public:
			//Explicit constructor
			eventTime(double t_in, unsigned int cell_type_in, unsigned int cell_in, unsigned int event_type_in, unsigned int otherCell_in, unsigned int dir_in){
				t=t_in; eventType=event_type_in; cellType=cell_type_in; cell=cell_in; otherCell=-1; dir=-1;
				if(eventType==INTERACT_EVENT || eventType==DC_INTERACT_EVENT) otherCell=otherCell_in;
				else if(eventType==ENTER_POS_EVENT || eventType==LEAVE_POS_EVENT) dir=dir_in;
				else throw std::runtime_error("Invalid value "+to_string(eventType)+" given to event class");
			}

			//Explicit constructor for non-cell events
			eventTime(double t_in, unsigned int event_type_in){
				t=t_in; eventType=event_type_in; cell=-1; cellType=NOTCELL_TYPE; otherCell=-1; dir=-1;
			}
			//Explicit constructor for non-cell events with an additional argument
			eventTime(double t_in, unsigned int event_type_in, unsigned int extraInput){
				t=t_in; eventType=event_type_in; cell=extraInput; cellType=NOTCELL_TYPE; otherCell=-1; dir=-1;
			}
			//Implicit instructor for types that need no extra argument: WARNING: There is no error checking on this constructor!
			eventTime(double t_in, unsigned int cell_type_in, unsigned int cell_in, unsigned int event_type_in){
				t=t_in; cellType=cell_type_in; cell=cell_in; eventType=event_type_in; otherCell=-1; dir=-1;
			}
			//Implicit instructor for types with an extra argument
			eventTime(double t_in, unsigned int cell_type_in, unsigned int cell_in, unsigned int event_type_in, unsigned int extraInput){
				t=t_in; eventType=event_type_in; cellType=cell_type_in; cell=cell_in;
				if(eventType==INTERACT_EVENT || eventType==DC_INTERACT_EVENT) otherCell=extraInput;
				else if(eventType==ENTER_POS_EVENT || eventType==LEAVE_POS_EVENT) dir=extraInput;
				else throw std::runtime_error("Invalid value "+to_string(eventType)+" given to event class");
			}
			bool cellEqual(unsigned int inputCell)			{ return cell==inputCell;}
			bool partnerCellEqual(unsigned int inputCell)	{ return otherCell==inputCell;}
			bool operator<(eventTime other) const{ return t > other.t; } //Defined for the sake of std::sort. Note that it is in reverse order.
			bool operator>(eventTime other) const{ return t < other.t; }
			friend ostream& operator<< (ostream &out, eventTime &et); //Allows ostream's "<<" to access the private members of this class.
			double getTime() {return t;}
			unsigned int getEventType() {return eventType;}
			unsigned int getCellType() {return cellType;}
			unsigned int getCellNum() {return cell;}
			unsigned int getPartnerCellNum() {return otherCell;}
			unsigned int getSiteChangeDirection() {return dir;}
			void changeCellNum(unsigned int cell_in){cell=cell_in;}
			void changePartnerCellNum(unsigned int cell_in){otherCell=cell_in;}
		private:
			double t;
			unsigned int eventType,cell,cellType,otherCell,dir;
	};
	ostream& operator<< (ostream &out, eventTime &et){//Overloading operator
		out << et.t << ": ";
		string dir; if(et.dir==X_POS) dir="x+"; else if(et.dir==X_NEG) dir="x-"; else if(et.dir==Y_POS) dir="y+"; else if(et.dir==Y_NEG) dir="y-"; else if(et.dir==Z_POS) dir="z+"; else if(et.dir==Z_NEG) dir="z-"; 
		if(et.eventType==VEL_CHANGE_EVENT)					out << "VEL_CHANGE "				<< (et.cellType==TCELL_TYPE?'T':'D') << et.cell;
		else if(et.eventType==SPHERE_EDGE_EVENT)			out << "SPHERE_EDGE "				<< (et.cellType==TCELL_TYPE?'T':'D') << et.cell;
		else if(et.eventType==INTERACT_EVENT)				out << "INTERACT "					<< (et.cellType==TCELL_TYPE?'T':'D') << et.cell << " (" << et.otherCell << ")";
		else if(et.eventType==DC_INTERACT_EVENT)			out << "DC_INTERACT "				<< (et.cellType==TCELL_TYPE?'T':'D') << et.cell << " (" << et.otherCell << ")";
		else if(et.eventType==ENTER_POS_EVENT)				out << "GRIDCELL_ENTRY "			<< (et.cellType==TCELL_TYPE?'T':'D') << et.cell << " (" << dir << ")";
		else if(et.eventType==LEAVE_POS_EVENT)				out << "GRIDCELL_EXIT "				<< (et.cellType==TCELL_TYPE?'T':'D') << et.cell << " (" << dir << ")";
		else if(et.eventType==DC_ENTRY_EVENT)				out << "DC_ENTRY ";
		else if(et.eventType==UNSET_FAILURE_ID_EVENT)		out << "UNSET_FAILURE_ID_EVENT "	<< "T" << et.cell;
		else if(et.eventType==MEASURE_OBSERVABLES_EVENT)	out << "MEASURE_OBSERVABLES ";
		else if(et.eventType==GRAPHICAL_UPDATE_EVENT)		out << "GRAPHICAL_UPDATE ";
		return out;
	}

	//Coordinates
		//#define COORD(x) (int((x+radius)*contractGridReductionFactor/cellSide))
		#define COORD(x) (int((x+radius)/cellSide))
		// #define OUTSIDE_SPHERE(x,y,z) ((x)*(x) + (y)*(y) + (z)*(z) > radius*radius)

	//Event consequences
		#define INTERACTION_EVENT_EXISTS(tCell,DC) (														\
			uint(distance(eventList.begin(), find_if(eventList.begin(),eventList.end(),[=](eventTime &et){	\
				return (et.cellEqual(tCell) && et.partnerCellEqual(DC));									\
			})))!=eventList.size()																			\
		)
		#define REMOVE_EVENT(cellType,cellNum,eventType){																	\
			inddist=0;																										\
			if(eventType==INTERACT_EVENT && cellType==DCELL_TYPE){															\
				_REMOVE_EVENT_BY_PARTNER(cellType,cellNum,eventType);														\
			}																												\
			else if(eventType==DC_INTERACT_EVENT){																			\
				_REMOVE_DC_INTERACT_EVENT(cellNum);																			\
			}																												\
			else { _REMOVE_EVENT(cellType,cellNum,eventType);}																\
		}
		#define _REMOVE_EVENT(cellType,cellNum,eventType){																	\
			while((inddist=uint(distance(																					\
					eventList.begin(), find_if(																				\
						eventList.begin()+inddist,eventList.end(),[=](eventTime &et){										\
							return et.getCellType()==cellType && et.getCellNum()==cellNum && et.getEventType()==eventType;	\
						}																									\
					)																										\
				)																											\
			))!=eventList.size()) {																							\
				if(inddist+1==oldEventListSize) {inddist++; continue;}/*Note that we do NOT remove the event if it is the current (last) one, as it will be removed at the end of the time loop.*/\
				if(eventList.size()==oldEventListSize) oldEventListSize=inddist+1; /*If we move the current event, keep track of it for deletion later*/ \
				eventList[inddist]=eventList.back(); eventList.pop_back();													\
			}																												\
		}
		#define _REMOVE_EVENT_BY_PARTNER(cellType,cellNum,eventType){																\
			while((inddist=uint(distance(																							\
					eventList.begin(), find_if(																						\
						eventList.begin()+inddist,eventList.end(),[=](eventTime &et){												\
							return et.getCellType()==cellType && et.getPartnerCellNum()==cellNum && et.getEventType()==eventType;	\
						}																											\
					)																												\
				)																													\
			))!=eventList.size()) {																									\
				if(inddist+1==oldEventListSize) {inddist++; continue;}/*Note that we do NOT remove the event if it is the current (last) one, as it will be removed at the end of the time loop.*/\
				if(eventList.size()==oldEventListSize) oldEventListSize=inddist+1; /*If we move the current event, keep track of it for deletion later*/ \
				eventList[inddist]=eventList.back(); eventList.pop_back();															\
			}																														\
		}
		#define _REMOVE_DC_INTERACT_EVENT(cellNum){																							\
			while((inddist=uint(distance(																									\
					eventList.begin(), find_if(																								\
						eventList.begin()+inddist,eventList.end(),[=](eventTime &et){														\
							return et.getEventType()==DC_INTERACT_EVENT && (et.getCellNum()==cellNum || et.getPartnerCellNum()==cellNum);	\
						}																													\
					)																														\
				)																															\
			))!=eventList.size()) {																											\
				if(inddist+1==oldEventListSize) {inddist++; continue;}/*Note that we do NOT remove the event if it is the current (last) one, as it will be removed at the end of the time loop.*/\
				if(eventList.size()==oldEventListSize) oldEventListSize=inddist+1; /*If we move the current event, keep track of it for deletion later*/ \
				eventList[inddist]=eventList.back(); eventList.pop_back();																	\
			}																																\
		}																																	\

		#define ADD_EVENTS_AFTER_VELOCITY_CHANGE(cellList,otherCellList,cellSiteArr,otherCellSiteArr, cellType,cellNum, tim, x,y,z,cx,cy,cz, vx,vy,vz,mfp,vmag){	\
			/*cout << "Event times for " << (cellType==TCELL_TYPE?"T":"D") << cellNum << endl;		*/														\
			eventList.push_back(eventTime(tim+mfp/vmag,cellType,cellNum,VEL_CHANGE_EVENT));																	\
			cellList[cellNum].set_next_velocity_change(tim+mfp/vmag);																						\
			/*cout << "Time until velocity change: " << tim<<"+"<<mfp/vmag<<"="<<tim+mfp/vmag << endl;		*/												\
																																							\
			/*Find time the cell with leave the sphere*/ 																									\
			magsq=SQMAGNITUDE(x,y,z); vivi=SQMAGNITUDE(vx,vy,vz); xivi=vx*x+vy*y+vz*z;																		\
			spheretime = -1*xivi + sqrt(xivi*xivi - vivi*(magsq-radius*radius));/*using positive root: negative root is always <0*/							\
			spheretime/=vivi; 																																\
			if(spheretime<mfp/vmag){ /*No need to add an event that will never resolve.*/																	\
				eventList.pop_back(); /*Get rid of the velocity change we just added, because we will change velocity when we hit the sphere edge*/			\
				eventList.push_back(eventTime(tim+spheretime,cellType,cellNum,SPHERE_EDGE_EVENT)); 															\
				cellList[cellNum].set_next_velocity_change(tim+spheretime);																					\
			}																																				\
																																							\
			CALCULATE_CONTACTABLE_GRID_SITES(x,y,z,cx,cy,cz);																								\
			nearbyPartners.clear();	if(cellType==DCELL_TYPE) {nearbyDCs.clear();}																			\
			for(int i=0; i<3; i++){																															\
				if(cx+i-1 >= numPositions) {break;}																											\
				else if(cx+i-1 < 0) {continue;}																												\
																																							\
				for(int j=0; j<3; j++){																														\
					if(cy+j-1 >= numPositions) {break;}																										\
					else if(cy+j-1 < 0) {continue;}																											\
																																							\
					for(int k=0; k<3; k++){																													\
						if(cz+k-1 >= numPositions) {break;}																									\
						else if(cz+k-1 < 0) {continue;}																										\
						if(localNeighbourhood[i+3*j+9*k]==1){																								\
							ind=THREED_TO_ONED(cx+i-1,cy+j-1,cz+k-1);																						\
							for (unsigned int onum: otherCellSiteArr[ind]){																					\
								if(!IS_IN_VECTOR(nearbyPartners,onum)) nearbyPartners.push_back(onum);														\
							}																																\
							if(cellType==DCELL_TYPE){for (unsigned int onum: cellSiteArr[ind]){																\
								if(onum!=cellNum && !IS_IN_VECTOR(nearbyDCs,onum)) nearbyDCs.push_back(onum);												\
							}}																																\
						}																																	\
					}																																		\
				}																																			\
			}																																				\
			ADD_COLLISION_EVENTS(cellList,otherCellList,nearbyDCs,nearbyPartners, cellType,cellNum, tim, x,y,z, vx,vy,vz, mfp,vmag);						\
																																							\
			/*Find time that the cell's contact region will enter or leave a row of x,y or z.*/																\
			etime=(cellSide*(cx+0.25)-x-radius)/vx; etime2=(cellSide*(cx+0.75)-x-radius)/vx;																\
			if(etime<0) etime+=cellSide/abs(vx); if(etime2<0) etime2+=cellSide/abs(vx);																		\
			if(vx>0){																																		\
				if(etime<mfp/vmag && etime<spheretime)  eventList.push_back(eventTime(tim+etime, cellType,cellNum,LEAVE_POS_EVENT,X_POS));					\
				if(etime2<mfp/vmag && etime2<spheretime) eventList.push_back(eventTime(tim+etime2,cellType,cellNum,ENTER_POS_EVENT,X_POS));					\
			/*	cout << "Time to leave a box in x: " << (vx>0) << " (" << cellSide << "*" << "(" << cx << "+" << 0.25 << ")-" << x << ")/" << vx << " " << cellSide << "*" << "(" << cx << "+" << 0.75 << ")-" << x  << "/" << vx << " "<< tim<<"+"<<etime<<"="<<tim+etime << endl;	*/\
			/*	cout << "Time to enter a box in x: " << (vx>0) << " (" << cellSide << "*" << "(" << cx << "+" << 0.75 << ")-" << x << ")/" << vx << " " << cellSide << "*" << "(" << cx << "+" << 0.25 << ")-" << x  << "/" << vx << " "<< tim<<"+"<<etime2<<"="<<tim+etime2 << endl;*/\
			}																																				\
			else{																																			\
				if(etime2<mfp/vmag && etime2<spheretime) eventList.push_back(eventTime(tim+etime2,cellType,cellNum,LEAVE_POS_EVENT,X_NEG));					\
				if(etime<mfp/vmag && etime<spheretime)  eventList.push_back(eventTime(tim+etime, cellType,cellNum,ENTER_POS_EVENT,X_NEG));					\
			/*	cout << "Time to leave a box in x: " << (vx>0) << " (" << cellSide << "*" << "(" << cx << "+" << 0.75 << ")-" << x << ")/" << vx << " " << cellSide << "*" << "(" << cx << "+" << 0.25 << ")-" << x  << "/" << vx << " " << tim<<"+"<<etime2<<"="<<tim+etime2 << endl; 	*/\
			/*	cout << "Time to enter a box in x: " << (vx>0) << " (" << cellSide << "*" << "(" << cx << "+" << 0.25 << ")-" << x << ")/" << vx << " " << cellSide << "*" << "(" << cx << "+" << 0.75 << ")-" << x  << "/" << vx << " "<< tim<<"+"<<etime<<"="<<tim+etime << endl;*/\
			}																																				\
																																							\
			etime=(cellSide*(cy+0.25)-y-radius)/vy; etime2=(cellSide*(cy+0.75)-y-radius)/vy;																\
			if(etime<0) etime+=cellSide/abs(vy); if(etime2<0) etime2+=cellSide/abs(vy);																		\
			if(vy>0){																																		\
				if(etime<mfp/vmag && etime<spheretime)  eventList.push_back(eventTime(tim+etime, cellType,cellNum,LEAVE_POS_EVENT,Y_POS));					\
				if(etime2<mfp/vmag && etime2<spheretime) eventList.push_back(eventTime(tim+etime2,cellType,cellNum,ENTER_POS_EVENT,Y_POS));					\
			/*	cout << "Time to leave a box in y: " << (vy>0) << " (" << cellSide << "*" << "(" << cy << "+" << 0.25 << ")-" << y << ")/" << vy << " " << cellSide << "*" << "(" << cy << "+" << 0.75 << ")-" << y  << "/" << vy << " " << tim<<"+"<<etime<<"="<<tim+etime << endl;	*/\
			/*	cout << "Time to enter a box in y: " << (vy>0) << " (" << cellSide << "*" << "(" << cy << "+" << 0.75 << ")-" << y << ")/" << vy << " " << cellSide << "*" << "(" << cy << "+" << 0.25 << ")-" << y  << "/" << vy << " " << tim<<"+"<<etime2<<"="<<tim+etime2 << endl; */\
			}																																				\
			else{																																			\
				if(etime2<mfp/vmag && etime2<spheretime) eventList.push_back(eventTime(tim+etime2,cellType,cellNum,LEAVE_POS_EVENT,Y_NEG));					\
				if(etime<mfp/vmag && etime<spheretime)  eventList.push_back(eventTime(tim+etime, cellType,cellNum,ENTER_POS_EVENT,Y_NEG));					\
			/*	cout << "Time to leave a box in y: " << (vy>0) << " (" << cellSide << "*" << "(" << cy << "+" << 0.75 << ")-" << y << ")/" << vy << " " << cellSide << "*" << "(" << cy << "+" << 0.25 << ")-" << y  << "/" << vy << " " << tim<<"+"<<etime2<<"="<<tim+etime2 << endl; 	*/\
			/*	cout << "Time to enter a box in y: " << (vy>0) << " (" << cellSide << "*" << "(" << cy << "+" << 0.25 << ")-" << y << ")/" << vy << " " << cellSide << "*" << "(" << cy << "+" << 0.75 << ")-" << y  << "/" << vy << " " << tim<<"+"<<etime<<"="<<tim+etime << endl; */\
			}																																				\
																																							\
			etime=(cellSide*(cz+0.25)-z-radius)/vz; etime2=(cellSide*(cz+0.75)-z-radius)/vz;																\
			if(etime<0) etime+=cellSide/abs(vz); if(etime2<0) etime2+=cellSide/abs(vz);																		\
			if(vz>0){																																		\
				if(etime<mfp/vmag && etime<spheretime)  eventList.push_back(eventTime(tim+etime, cellType,cellNum,LEAVE_POS_EVENT,Z_POS));					\
				if(etime2<mfp/vmag && etime2<spheretime) eventList.push_back(eventTime(tim+etime2,cellType,cellNum,ENTER_POS_EVENT,Z_POS));					\
			/*	cout << "Time to leave a box in z: " << (vz>0) << " (" << cellSide << "*" << "(" << cz << "+" << 0.25 << ")-" << z << ")/" << vz << " " << cellSide << "*" << "(" << cz << "+" << 0.75 << ")-" << z  << "/" << vz << " " << tim<<"+"<<etime<<"="<<tim+etime << endl;	*/\
			/*	cout << "Time to enter a box in z: " << (vz>0) << " (" << cellSide << "*" << "(" << cz << "+" << 0.75 << ")-" << z << ")/" << vz << " " << cellSide << "*" << "(" << cz << "+" << 0.25 << ")-" << z  << "/" << vz << " " << tim<<"+"<<etime2<<"="<<tim+etime2 << endl; */\
			}																																				\
			else{																																			\
				if(etime2<mfp/vmag && etime2<spheretime) eventList.push_back(eventTime(tim+etime2,cellType,cellNum,LEAVE_POS_EVENT,Z_NEG));					\
				if(etime<mfp/vmag && etime<spheretime)  eventList.push_back(eventTime(tim+etime, cellType,cellNum,ENTER_POS_EVENT,Z_NEG));					\
			/*	cout << "Time to leave a box in z: " << (vz>0) << " (" << cellSide << "*" << "(" << cz << "+" << 0.75 << ")-" << z << ")/" << vz << " " << cellSide << "*" << "(" << cz << "+" << 0.25 << ")-" << z  << "/" << vz << " " << tim<<"+"<<etime2<<"="<<tim+etime2 << endl;	*/\
			/*	cout << "Time to enter a box in z: " << (vz>0) << " (" << cellSide << "*" << "(" << cz << "+" << 0.25 << ")-" << z << ")/" << vz << " " << cellSide << "*" << "(" << cz << "+" << 0.75 << ")-" << z  << "/" << vz << " " << tim<<"+"<<etime<<"="<<tim+etime << endl; */\
			}																																				\
		}
		#define ADD_COLLISION_EVENTS(cellList,otherCellList,nearbyLikeCells,nearbyUnlikeCells, cellType,cellNum, tim, x,y,z, vx,vy,vz, mfp,vmag ){			\
			for (unsigned int onum: nearbyUnlikeCells){																										\
				if(INTERACTION_EVENT_EXISTS(onum,cellNum) || INTERACTION_EVENT_EXISTS(cellNum,onum)){														\
					continue;																																\
				} 																																			\
				auto oc=&otherCellList[onum];																												\
				if( (cellList[cellNum].get_failed_interaction_ID()==int(onum)) || ((*oc).get_failed_interaction_ID()==int(cellNum)) ){						\
					continue;																																\
				} 																																			\
				/*Get position & velocity of other cell in the grid*/																						\
				told=(*oc).get_time_position_updated();																										\
				l_ox=(*oc).get_x(); l_oy=(*oc).get_y(); l_oz=(*oc).get_z();																					\
				l_ovx=(*oc).get_vx(); l_ovy=(*oc).get_vy(); l_ovz=(*oc).get_vz(); 																			\
				dx=l_ovx*(t-told);  dy=l_ovy*(t-told); dz=l_ovz*(t-told);																					\
				l_ox+=dx; l_oy+=dy; l_oz+=dz;																												\
				(*oc).set_position(t,l_ox,l_oy,l_oz); (*oc).reduce_free_path_remaining(MAGNITUDE(dx,dy,dz));												\
				/*Calculate difference between all of these*/																								\
				dx=x-l_ox; dy=y-l_oy; dz=z-l_oz; dvx=vx-l_ovx; dvy=vy-l_ovy; dvz=vz-l_ovz;																	\
				vivi=SQMAGNITUDE(dvx,dvy,dvz);																												\
				xivi=dvx*dx+dvy*dy+dvz*dz; 																													\
				/*Finally, calculate time of collision and ensure it's before either cell changes velocity*/												\
				etime=-1*xivi - sqrt( xivi*xivi - vivi*(dx*dx+dy*dy+dz*dz-contactRadius*contactRadius ) ); /* using negative root: positive is behind the other cell*/\
				etime/=vivi; 																																\
				if(etime>0 && tim+etime<cellList[cellNum].get_next_velocity_change() && tim+etime<(*oc).get_next_velocity_change() ){						\
					if(cellType==TCELL_TYPE){																												\
						eventList.push_back(eventTime(tim+etime,TCELL_TYPE,cellNum,INTERACT_EVENT,onum));/*For the sake of indexing these events, always define it as a T cell event*/\
						cellList[cellNum].set_next_velocity_change(tim+etime);																				\
					}																																		\
					else{																																	\
						eventList.push_back(eventTime(tim+etime,TCELL_TYPE,onum,INTERACT_EVENT,cellNum));													\
						(*oc).set_next_velocity_change(tim+etime);																							\
					} 																																		\
				}																																			\
			}																																				\
			if(cellType==DCELL_TYPE){																														\
				for (unsigned int onum: nearbyLikeCells){																									\
					if(INTERACTION_EVENT_EXISTS(onum,cellNum) || INTERACTION_EVENT_EXISTS(cellNum,onum)) continue;											\
					auto oc=&cellList[onum];																												\
					told=(*oc).get_time_position_updated();																									\
					l_ox=(*oc).get_x(); l_oy=(*oc).get_y(); l_oz=(*oc).get_z();																				\
					l_ovx=(*oc).get_vx(); l_ovy=(*oc).get_vy(); l_ovz=(*oc).get_vz(); 																		\
					if(t!=told){																															\
						dx=l_ovx*(t-told);  dy=l_ovy*(t-told); dz=l_ovz*(t-told);																			\
						l_ox+=dx; l_oy+=dy; l_oz+=dz;																										\
						(*oc).set_position(t,l_ox,l_oy,l_oz); (*oc).reduce_free_path_remaining(MAGNITUDE(dx,dy,dz));										\
					}																																		\
					dx=x-l_ox; dy=y-l_oy; dz=z-l_oz; dvx=vx-l_ovx; dvy=vy-l_ovy; dvz=vz-l_ovz;																\
					vivi=SQMAGNITUDE(dvx,dvy,dvz);																											\
					xivi=dvx*dx+dvy*dy+dvz*dz; 																												\
					/*Finally, calculate time of collision and ensure it's before either cell changes velocity*/											\
					etime=-1*xivi - sqrt( xivi*xivi - vivi*(dx*dx+dy*dy+dz*dz-contactRadius*contactRadius) ); /* using negative root: positive is behind the other cell*/				\
					etime/=vivi; 																															\
					if(etime>0 && tim+etime<cellList[cellNum].get_next_velocity_change() && tim+etime<(*oc).get_next_velocity_change()) {							\
						eventList.push_back(eventTime(tim+etime,cellType,cellNum,DC_INTERACT_EVENT,onum));													\
						cellList[cellNum].set_next_velocity_change(tim+etime);																				\
						(*oc).set_next_velocity_change(tim+etime);																							\
					}																																		\
				}																																			\
			}																																				\
		}
		#define GENERATE_DC_VMAG() genrand_dc_velocity(engine);
		#define CHANGE_VELOCITY(cellList, cellType,cellNum, vx,vy,vz,mfp,vmag){																			\
			if(cellType==TCELL_TYPE){																													\
				vmag = genrand_tc_velocity(engine);																										\
					cellList[cellNum].set_free_path_remaining(mfp=genrand_tc_freepath(engine));																\
			} 																																			\
			else {																																		\
				vmag=GENERATE_DC_VMAG();																												\
				cellList[cellNum].set_free_path_remaining(mfp=genrand_dc_freepath(engine));																\
			}																																			\
			assert(mfp>0);																																\
			theta=acos(2*genrand_uniform_pos(engine)-1); phi=6.283185307*genrand_uniform_pos(engine); /*theta between 0 and pi, phi between 0 and 2pi*/	\
			cellList[cellNum].set_velocity(vx=vmag*sin(theta)*cos(phi), vy=vmag*sin(theta)*sin(phi), vz=vmag*cos(theta));								\
		}		
		#define REFLECT_FROM_SPHERE_EDGE(cellList, cellType, cellNum, x,y,z,vx,vy,vz,mfp){																				\
			if(cellType==TCELL_TYPE){																																	\
				vmag = genrand_tc_velocity(engine);																														\
				cellList[cellNum].set_free_path_remaining(mfp=genrand_tc_freepath(engine));																				\
			} 																																							\
			else {																																						\
				vmag=GENERATE_DC_VMAG();																																\
				cellList[cellNum].set_free_path_remaining(mfp=genrand_dc_freepath(engine));																				\
			}																																							\
			theta=acos(genrand_uniform_pos(engine)); phi=6.283185307*genrand_uniform_pos(engine);/*theta between 0 and pi/2, phi between 0 and 2pi*/					\
			magsq = MAGNITUDE(x,y,z); /*Get magnitude of position vector to normalise*/																					\
			unx=x/magsq; uny=y/magsq; unz=z/magsq; /*Set as a unit vector*/																								\
			vx=-vmag*unx; vy=-vmag*uny; vz=-vmag*unz; /*Set velocity direction opposite to the position vector (i.e. pointing to centre of the sphere)*/				\
			arbitraryAxisRotation(unz, uny, -unx, vx, vy, vz, theta); /*Choosing the radial axis as the z axis, this performs a rotation by theta about the x axis.*/	\
			arbitraryAxisRotation(unx, uny, unz,  vx, vy, vz, phi); /*Using the same axes as before, this performs a rotation by phi about the z axis.*/				\
			cellList[cellNum].set_velocity(vx, vy, vz);																													\
		}
		#define ATTEMPT_INTERACTION(tCellList,dCellList,tSites,dSites, tCellNum,dCellNum, tx,ty,tz,dendx,dendy,dendz, tcx,tcy,tcz, dendcx,dendcy,dendcz, vx,vy,vz){					\
			/*Update the DC's cognate antigen total*/																																\
			dCellList[dCellNum].update_num_cog_antigen(t, antigenDecayRate, cognateAgPrecision, cognateAgProbabilityTable);															\
			if(dCellList[dCellNum].DC_cannot_activate_t_cells()) {/*If there is no longer enough cognate Ag, there is little point simulating the DC anymore...*/					\
				REMOVE_CELL_FROM_SIMULATION(dSites,DCELL_TYPE,dCellNum,ox,oy,oz,ocx,ocy,ocz); 																						\
			}																																										\
																																													\
			/*Now, see if the T cell has been activated*/																															\
			else if(genrand_uniform_pos(engine)<dCellList[dCellNum].get_prob_activation()){/*Activation successful*/																\
				REMOVE_CELL_FROM_SIMULATION(tSites,TCELL_TYPE,tCellNum,tx,ty,tz,cx,cy,cz);																							\
				/*Finally, record date for analysis*/																																\
				this_numActivated++;																																				\
				tCellList[tCellNum].TC_set_activation_state(1);																														\
				if(timeUntilFirstActivation[chunk*numRepeats+repeat]==maxTime && this_numActivated>=1)						timeUntilFirstActivation[chunk*numRepeats+repeat] = t;	\
				if(timeUntil25pcActivation[chunk*numRepeats+repeat]==maxTime  && this_numActivated>=int(0.25*numTCells))	timeUntil25pcActivation[chunk*numRepeats+repeat] = t;	\
				if(timeUntil50pcActivation[chunk*numRepeats+repeat]==maxTime  && this_numActivated>=int(0.5*numTCells)) 	timeUntil50pcActivation[chunk*numRepeats+repeat] = t;	\
				if(timeUntil75pcActivation[chunk*numRepeats+repeat]==maxTime  && this_numActivated>=int(0.75*numTCells))	timeUntil75pcActivation[chunk*numRepeats+repeat] = t;	\
				if(timeUntil90pcActivation[chunk*numRepeats+repeat]==maxTime  && this_numActivated>=int(0.9*numTCells))		timeUntil90pcActivation[chunk*numRepeats+repeat] = t;	\
				if(this_numActivated==numTCells)																			timeUntil100pcActivation[chunk*numRepeats+repeat] = t;	\
			}																																										\
			else{ /*Activation failed - send the T cell away from the DC.*/																											\
				/*Remove existing events before adding more*/																														\
				REMOVE_EVENT(TCELL_TYPE,tCellNum,VEL_CHANGE_EVENT); REMOVE_EVENT(TCELL_TYPE,tCellNum,SPHERE_EDGE_EVENT);															\
				REMOVE_EVENT(TCELL_TYPE,tCellNum,INTERACT_EVENT); 																													\
				tCellList[tCellNum].set_failed_interaction_ID(dCellNum);																											\
				/*Regenerate velocity to move the T cell away from the DC after interaction. */																						\
				vmag = genrand_tc_velocity(engine); theta=acos(genrand_uniform_pos(engine)); phi=6.283185307*genrand_uniform_pos(engine); 											\
				dx=tx-dendx; dy=ty-dendy; dz=tz-dendz; /*Defines the position vector from DC to T cell. */																			\
				magsq = MAGNITUDE(dx,dy,dz); /*Get magnitude (not squared) to normalise*/																							\
				dx/=magsq; dy/=magsq; dz/=magsq; /*Set as a unit vector*/																											\
				vx=vmag*dx; vy=vmag*dy; vz=vmag*dz; /*Magnitude of velocity, currently parallel to position vector*/																\
				arbitraryAxisRotation(dz, dy, -dx, vx,vy,vz, theta); /*Setting position vector as z axis, this performs a rotation by theta about the x axis.*/						\
				arbitraryAxisRotation(dx, dy, dz,  vx,vy,vz, phi); /*Using the same axes as before, this performs a rotation by phi about the z axis.*/								\
				tCellList[tCellNum].set_velocity(vx,vy,vz);																															\
				tCellList[tCellNum].set_free_path_remaining(mfp=genrand_tc_freepath(engine));																						\
				eventList.push_back(eventTime(t+mfp/vmag,TCELL_TYPE,tCellNum,UNSET_FAILURE_ID_EVENT)); 																				\
				ADD_EVENTS_AFTER_VELOCITY_CHANGE(tCellList,dCellList,tSites,dSites, TCELL_TYPE,tCellNum, t, tx,ty,tz,tcx,tcy,tcz, vx,vy,vz,mfp,vmag);								\
			}																																										\
		}
		#define DC_INTERACTION(dCellList,dSites, cellNum,oCellNum, x,y,z,ox,oy,oz, cx,cy,cz, ocx,ocy,ocz, vx,vy,vz){															\
			dx2=x-ox; dy2=y-oy; dz2=z-oz; /*Defines the position vector between the DCs. */																						\
			/*cout << "d: " << dx2 << "," << dy2 << "," << dz2 << endl;		*/																									\
			magsq = MAGNITUDE(dx2,dy2,dz2); /*Get magnitude (not squared) to normalise*/																						\
			dx2/=magsq; dy2/=magsq; dz2/=magsq; /*Set as a unit vector*/																										\
			/*Set first DC's velocity.*/																																		\
			vmag=GENERATE_DC_VMAG(); theta=acos(genrand_uniform_pos(engine)); phi=6.283185307*genrand_uniform_pos(engine); 														\
			vx=vmag*dx2; vy=vmag*dy2; vz=vmag*dz2; /*Magnitude of velocity, currently parallel to position vector*/																\
			/*cout << "\tmag and d: "<< magsq << " " << dx2 << "," << dy2 << "," << dz2 << endl;		*/																		\
			/*cout << "\tvmag and v: " << vmag << " " << vx << "," << vy << "," << vz << endl;		*/																			\
			arbitraryAxisRotation(dz2, dy2, -dx2, vx,vy,vz, theta); /*Setting position vector as z axis, this performs a rotation by theta about the x axis.*/					\
			arbitraryAxisRotation(dx2, dy2, dz2,  vx,vy,vz, phi); /*Using the same axes as before, this performs a rotation by phi about the z axis.*/							\
			/*cout << "\tpost rotation vmag and v: " << vmag << " " << vx << "," << vy << "," << vz << endl;*/																	\
			dCellList[cellNum].set_velocity(vx,vy,vz);																															\
			dCellList[cellNum].set_free_path_remaining(mfp=genrand_dc_freepath(engine));																						\
			dCellList[oCellNum].set_velocity(0,0,0);/*The 2' cell hasn't had its velocity set yet. Just in case, make sure it is stationary to not predict a collision...*/		\
			REMOVE_EVENT(DCELL_TYPE,cellNum,VEL_CHANGE_EVENT); 	REMOVE_EVENT(DCELL_TYPE,cellNum,SPHERE_EDGE_EVENT);																\
			REMOVE_EVENT(DCELL_TYPE,cellNum,INTERACT_EVENT);	REMOVE_EVENT(DCELL_TYPE,cellNum,DC_INTERACT_EVENT);																\
			ADD_EVENTS_AFTER_VELOCITY_CHANGE(dCellList,tCellList,dSites,tSites, DCELL_TYPE,cellNum, t, x,y,z,cx,cy,cz, vx,vy,vz,mfp,vmag);										\
			/*Reverse position vector and set the second */																														\
			dx2=-dx2; dy2=-dy2; dz2=-dz2;																																		\
			vmag=GENERATE_DC_VMAG(); theta=acos(genrand_uniform_pos(engine)); phi=6.283185307*genrand_uniform_pos(engine); 														\
			vx=vmag*dx2; vy=vmag*dy2; vz=vmag*dz2; /*Magnitude of velocity, currently parallel to position vector*/																\
			/*cout << "\tother vmag and v: " << vmag << " " << vx << "," << vy << "," << vz << endl;	*/																		\
			arbitraryAxisRotation(dz2, dy2, -dx2, vx,vy,vz, theta); /*Setting position vector as z axis, this performs a rotation by theta about the x axis.*/					\
			arbitraryAxisRotation(dx2, dy2, dz2,  vx,vy,vz, phi); /*Using the same axes as before, this performs a rotation by phi about the z axis.*/							\
			/*cout << "\tpost rotation vmag and v: " << vmag << " " << vx << "," << vy << "," << vz << endl;	*/																\
			dCellList[oCellNum].set_velocity(vx,vy,vz);																															\
			dCellList[oCellNum].set_free_path_remaining(mfp=genrand_dc_freepath(engine));																						\
			REMOVE_EVENT(DCELL_TYPE,oCellNum,VEL_CHANGE_EVENT);	 	REMOVE_EVENT(DCELL_TYPE,oCellNum,SPHERE_EDGE_EVENT);														\
			REMOVE_EVENT(DCELL_TYPE,oCellNum,INTERACT_EVENT);		REMOVE_EVENT(DCELL_TYPE,oCellNum,DC_INTERACT_EVENT);														\
			ADD_EVENTS_AFTER_VELOCITY_CHANGE(dCellList,tCellList,dSites,tSites, DCELL_TYPE,oCellNum, t, ox,oy,oz,ocx,ocy,ocz, vx,vy,vz,mfp,vmag);								\
		}
		#define CONTACT_NEW_GRID_CELLS(cellList,otherCellList,cellSiteArr,otherCellSiteArr, cellType,cellNum, tim, x,y,z, vx,vy,vz, dir,mfp,vmag){	\
			cx=COORD(x); cy=COORD(y); cz=COORD(z); memcpy(localNeigh2D,Ones2D,9*sizeof(int)); 								\
			if(dir==X_POS)		{ /*M_p=x;*/ M_q=y; M_r=z; M_vp=vx; M_cp=cx+1; M_cq=cy; M_cr=cz; }							\
			else if(dir==X_NEG)	{ /*M_p=x;*/ M_q=y; M_r=z; M_vp=vx; M_cp=cx-1; M_cq=cy; M_cr=cz; }							\
			else if(dir==Y_POS)	{ /*M_p=y;*/ M_q=x; M_r=z; M_vp=vy; M_cp=cy+1; M_cq=cx; M_cr=cz; }							\
			else if(dir==Y_NEG)	{ /*M_p=y;*/ M_q=x; M_r=z; M_vp=vy; M_cp=cy-1; M_cq=cx; M_cr=cz; }							\
			else if(dir==Z_POS)	{ /*M_p=z;*/ M_q=x; M_r=y; M_vp=vz; M_cp=cz+1; M_cq=cx; M_cr=cy; }							\
			else if(dir==Z_NEG)	{ /*M_p=z;*/ M_q=x; M_r=y; M_vp=vz; M_cp=cz-1; M_cq=cx; M_cr=cy; }							\
			if(M_cp<numPositions && M_cp >= 0){	/*equal to 0 is okay if e.g. we set M_cp=cx-1*/	 							\
				/*Find next time we will enter another new site and add if it will happen before velocity changes*/			\
				etime=cellSide/abs(M_vp);																					\
				if(etime<mfp/vmag) eventList.push_back(eventTime(tim+etime,cellType,cellNum,ENTER_POS_EVENT,dir));			\
																															\
				/*Calculate which adjacent sites need marking*/																\
				if ( (diffFromCoord=(M_q+radius)/cellSide-M_cq)>leftOverlap ) {												\
					for(int k=0; k<3; k++) localNeigh2D[k]=0; 		/*0 for all r(k), if q(j)=q(0) */						\
				}																											\
				if ( diffFromCoord<rightOverlap ) {																			\
					for(int k=0; k<3; k++) localNeigh2D[k+6]=0; 	/*0 for all r(k) if q(j)=q(2) */						\
				}																											\
				if ( (diffFromCoord=(M_r+radius)/cellSide-M_cr)>leftOverlap ) {												\
					for(int j=0; j<3; j++) localNeigh2D[3*j]=0;		/*0 for all q(j) if r(k)=r(0) */						\
				}																											\
				if ( diffFromCoord<rightOverlap ) {																			\
					for(int j=0; j<3; j++) localNeigh2D[3*j+2]=0; 	/*0 for all q(j) if r(k)=r(2) */						\
				}																											\
																															\
				/*Mark new sites, and then find out if the new sites have any cells we can/will interact with*/				\
				nearbyPartners.clear();	if(cellType==DCELL_TYPE) {nearbyDCs.clear();}										\
				for(int j=0; j<3; j++){																						\
					if(M_cq+j-1 >= numPositions) {break;}																	\
					else if(M_cq+j-1 < 0) {continue;}																		\
					for(int k=0; k<3; k++){																					\
						if(M_cr+k-1 >= numPositions) {break;}																\
						else if(M_cr+k-1 < 0) {continue;}																	\
						if(localNeigh2D[3*j+k]==1) {																		\
							/*cout << (cellType==DCELL_TYPE?"D":"T") << " entered p,q,r=" << M_cp << "," << M_cq+j-1 << "," << M_cr+k-1 << " " << entryExitLam(j,k) << endl;*/\
							for (unsigned int onum: otherCellSiteArr[entryExitLam(j,k)]){									\
								if(!IS_IN_VECTOR(nearbyPartners,onum)) nearbyPartners.push_back(onum);						\
							}																								\
							if(cellType==DCELL_TYPE){for (unsigned int onum: cellSiteArr[entryExitLam(j,k)]){				\
								if(onum!=cellNum && !IS_IN_VECTOR(nearbyDCs,onum)) nearbyDCs.push_back(onum);				\
							}}																								\
							cellSiteArr[entryExitLam(j,k)].push_back(cellNum);												\
						}																									\
					}																										\
				}																											\
				ADD_COLLISION_EVENTS(cellList,otherCellList,nearbyDCs,nearbyPartners, cellType,cellNum, tim, x,y,z, vx,vy,vz, mfp,vmag  );\
			}																												\
		}		
		#define LEAVE_GRID_CELLS(cellSiteArr, cellType,cellNum, tim, x,y,z, vx,vy,vz, dir,mfp,vmag){						\
			cx=COORD(x); cy=COORD(y); cz=COORD(z);																			\
			if(dir==X_POS)		{ /*M_p=x;*/ M_q=y; M_r=z; M_vp=vx; M_cp=cx-1; M_cq=cy; M_cr=cz; }							\
			else if(dir==X_NEG)	{ /*M_p=x;*/ M_q=y; M_r=z; M_vp=vx; M_cp=cx+1; M_cq=cy; M_cr=cz; }							\
			else if(dir==Y_POS)	{ /*M_p=y;*/ M_q=x; M_r=z; M_vp=vy; M_cp=cy-1; M_cq=cx; M_cr=cz; }							\
			else if(dir==Y_NEG)	{ /*M_p=y;*/ M_q=x; M_r=z; M_vp=vy; M_cp=cy+1; M_cq=cx; M_cr=cz; }							\
			else if(dir==Z_POS)	{ /*M_p=z;*/ M_q=x; M_r=y; M_vp=vz; M_cp=cz-1; M_cq=cx; M_cr=cy; }							\
			else if(dir==Z_NEG)	{ /*M_p=z;*/ M_q=x; M_r=y; M_vp=vz; M_cp=cz+1; M_cq=cx; M_cr=cy; }							\
																															\
			if(M_cp<numPositions && M_cp >= 0){																				\
				/*Find next time we will leave another new site and add if it will happen before velocity changes*/			\
				etime=cellSide/abs(M_vp);																					\
				if(etime<mfp/vmag) eventList.push_back(eventTime(tim+etime,cellType,cellNum,LEAVE_POS_EVENT,dir));			\
																															\
				/*Remove sites*/																							\
				for(int j=0; j<3; j++){																						\
					if(M_cq+j-1 >= numPositions) {break;}																	\
					else if(M_cq+j-1 < 0) {continue;}																		\
					for(int k=0; k<3; k++){																					\
						if(M_cr+k-1 >= numPositions) {break;}																\
						else if(M_cr+k-1 < 0) {continue;}																	\
							/*cout << (cellType==DCELL_TYPE?"D":"T") << " left p,q,r=" << M_cp << "," << M_cq+j-1 << "," << M_cr+k-1 << " " << entryExitLam(j,k) << endl;*/\
							REMOVE_ELEMENT_FROM_VECTOR_IF_PRESENT(entryExitLam(j,k), cellSiteArr, cellNum);					\
					}																										\
				}																											\
			}																												\
		}	
		// #define HAS_CONTACTED_DC(tCellNum,dCellNum) (tCellList[tCellNum].get_failed_interaction_ID()==dCellNum)
		#define CALCULATE_CONTACTABLE_GRID_SITES(x,y,z,cx,cy,cz){											\
			memcpy(localNeighbourhood,Ones,27*sizeof(int)); 												\
			/* 	2D:		nx-1	nx	nx+1 	| 			nx-1	nx-1	nx+1								\
				ny-1	0		1	2 		|	nz-1	0 		1 		2 									\
				ny 		3 		4 	5 		|	nz 		9 		10 		11 									\
				ny+1 	6 		7 	8 		|	nz+1 	18 		19 		20				 		*/ 			\
			if ( (diffFromCoord=(x+radius)/cellSide-cx)>leftOverlap ) {										\
				for(int i=0; i<9; i++) localNeighbourhood[3*i]=0; /*0,3,6, 9,12,15, 18,21,24*/				\
			}																								\
			if ( diffFromCoord<rightOverlap ) {																\
				for(int i=0; i<9; i++) localNeighbourhood[3*i+2]=0; /*2,5,8, 11,14,17, 20,23,26*/			\
			}																								\
			if ( (diffFromCoord=(y+radius)/cellSide-cy)>leftOverlap ) {										\
				for(int j=0; j<9; j++) localNeighbourhood[9*(j/3)+j%3]=0; /*0,1,2, 9,10,11, 18,19,20*/		\
			}																								\
			if ( diffFromCoord<rightOverlap ) {																\
				for(int j=0; j<9; j++) localNeighbourhood[9*(j/3)+j%3+6]=0; /*6,7,8, 15,16,17, 24,25,26*/	\
			}																								\
			if ( (diffFromCoord=(z+radius)/cellSide-cz)>leftOverlap ) {										\
				for(int k=0; k<9; k++) localNeighbourhood[k]=0; /*0,1,2, 3,4,5, 6,7,8*/						\
			}																								\
			if ( diffFromCoord<rightOverlap ) {																\
				for(int k=0; k<9; k++) localNeighbourhood[k+18]=0; /*18,19,20, 21,22,23, 24,25,26*/			\
			}																								\
		}
		#define ADD_DC_TO_SIMULATION_AT_SOURCE(dCellList,tCellList,dSites,tSites, dcnum, tim, x,y,z,cx,cy,cz, vx,vy,vz, mfp,vmag){											\
			if(dCellList[dcnum].get_prob_activation()>0) {																													\
				/*Set position & velocity*/																																	\
				theta=acos(genrand_uniform_pos(engine)); phi=3.141592654*genrand_uniform_pos(engine);/*Random theta between 0 and pi, phi between 0 and pi -> a hemisphere*/\
				sourceX=radius*sin(theta)*cos(phi); sourceY=radius*sin(theta)*sin(phi); sourceZ=radius*cos(theta);															\
				dCellList[dcnum].set_position(tim, x=sourceX, y=sourceY, z=sourceZ);																						\
				vmag=GENERATE_DC_VMAG(); dCellList[dcnum].set_free_path_remaining(mfp=contactRadius+0.01);/*Make them clear the source area immediately*/					\
				theta=acos(genrand_uniform_pos(engine)); phi=6.283185307*genrand_uniform_pos(engine);/*theta between 0 and pi/2, phi between 0 and 2pi*/					\
				magsq = MAGNITUDE(x,y,z); /*Get magnitude of position vector to normalise*/																					\
				unx=x/magsq; uny=y/magsq; unz=z/magsq; /*Set as a unit vector*/																								\
				vx=-vmag*unx; vy=-vmag*uny; vz=-vmag*unz; /*Set velocity direction opposite to the position vector (i.e. pointing to centre of the sphere)*/				\
				arbitraryAxisRotation(unz, uny, -unx, vx, vy, vz, theta); /*Choosing the radial axis as the z axis, this performs a rotation by theta about the x axis.*/	\
				arbitraryAxisRotation(unx, uny, unz,  vx, vy, vz, phi); /*Using the same axes as before, this performs a rotation by phi about the z axis.*/				\
				dCellList[dcnum].set_velocity(vx, vy, vz);																													\
				/*Find which discrete sites this cell can contact, and mark them*/																							\
				cx=COORD(x); cy=COORD(y); cz=COORD(z);																														\
				CALCULATE_CONTACTABLE_GRID_SITES(x,y,z,cx,cy,cz);																											\
				for(int i=0; i<3; i++){																																		\
					if(cx+i-1 >= numPositions) {break;}																														\
					else if(cx+i-1 < 0) {continue;}																															\
																																											\
					for(int j=0; j<3; j++){																																	\
						if(cy+j-1 >= numPositions) {break;}																													\
						else if(cy+j-1 < 0) {continue;}																														\
																																											\
						for(int k=0; k<3; k++){																																\
							if(cz+k-1 >= numPositions) {break;}																												\
							else if(cz+k-1 < 0) {continue;}																													\
							if(localNeighbourhood[i+3*j+9*k]==1){																											\
								dSites[THREED_TO_ONED(cx+i-1,cy+j-1,cz+k-1)].push_back(dcnum);																				\
							}																																				\
						}																																					\
					}																																						\
				}																																							\
				ADD_EVENTS_AFTER_VELOCITY_CHANGE(dCellList,tCellList,dSites,tSites, DCELL_TYPE,dcnum, tim, x,y,z,cx,cy,cz, vx,vy,vz,mfp,vmag); 								\
			}																																								\
		}
		#define ADD_DC_TO_SIMULATION_AT_RANDOM(dCellList,tCellList,dSites,tSites, dcnum, tim, x,y,z,cx,cy,cz, vx,vy,vz, mfp,vmag){											\
			if(dCellList[dcnum].get_prob_activation()>0) {																													\
				/*Set position & velocity*/																																	\
				while(1==1){ 																																				\
					GENERATE_COORDS; /*Gets random values of x, y and z between -radius and +radius (coordinates from a cube of side = diameter)*/							\
					if(OUTSIDE_SPHERE(x,y,z)) continue;																														\
					inContact=0; 																																			\
					cx=COORD(x); cy=COORD(y); cz=COORD(z);																													\
					for (auto edcnum: dSites[THREED_TO_ONED(cx,cy,cz)]) {																									\
						ox=dCellList[edcnum].get_x(); oy=dCellList[edcnum].get_y(); oz=dCellList[edcnum].get_z(); 															\
						if(SQMAGNITUDE(x-ox,y-oy,z-oz)<contactRadius*contactRadius) inContact=1; /*Too close to another DC - need to try again*/							\
					}																																						\
					if( (posFailures++)>POSITION_OOB_TOLERANCE ){cerr << "ERROR: Too many attempts to place a DC onto the grid without contacting another. Check parameter choices." << endl; return 1;}	\
					else if(inContact==0) break;																															\
				}																																							\
				dCellList[dcnum].set_position(tim, x, y, z);																												\
				vmag=GENERATE_DC_VMAG(); theta=acos(2*genrand_uniform_pos(engine)-1); phi=6.283185307*genrand_uniform_pos(engine);											\
				dCellList[dcnum].set_velocity(vx=vmag*sin(theta)*cos(phi), vy=vmag*sin(theta)*sin(phi), vz=vmag*cos(theta));												\
				dCellList[dcnum].set_free_path_remaining(mfp=genrand_dc_freepath(engine));																					\
				/*Find which discrete sites this cell can contact, and mark them*/																							\
				cx=COORD(x); cy=COORD(y); cz=COORD(z);																														\
				CALCULATE_CONTACTABLE_GRID_SITES(x,y,z,cx,cy,cz);																											\
				for(int i=0; i<3; i++){																																		\
					if(cx+i-1 >= numPositions) {break;}																														\
					else if(cx+i-1 < 0) {continue;}																															\
																																											\
					for(int j=0; j<3; j++){																																	\
						if(cy+j-1 >= numPositions) {break;}																													\
						else if(cy+j-1 < 0) {continue;}																														\
																																											\
						for(int k=0; k<3; k++){																																\
							if(cz+k-1 >= numPositions) {break;}																												\
							else if(cz+k-1 < 0) {continue;}																													\
							if(localNeighbourhood[i+3*j+9*k]==1){																											\
								dSites[THREED_TO_ONED(cx+i-1,cy+j-1,cz+k-1)].push_back(dcnum);																				\
							}																																				\
						}																																					\
					}																																						\
				}																																							\
				ADD_EVENTS_AFTER_VELOCITY_CHANGE(dCellList,tCellList,dSites,tSites, DCELL_TYPE,dcnum, tim, x,y,z,cx,cy,cz, vx,vy,vz,mfp,vmag); 								\
			}																																								\
		}
		#define REMOVE_CELL_FROM_SIMULATION(cellSites,cellType,cellNum,x,y,z,cx,cy,cz){															\
			for (uint i=0; i<eventList.size(); i++){																							\
				if (																															\
						(eventList[i].getCellType()==cellType && eventList[i].getCellNum()==cellNum && i+1!=oldEventListSize) ||				\
						(eventList[i].getEventType()==INTERACT_EVENT && eventList[i].getPartnerCellNum()==cellNum && cellType==DCELL_TYPE) ||	\
						(eventList[i].getEventType()==DC_INTERACT_EVENT && eventList[i].getPartnerCellNum()==cellNum && cellType==DCELL_TYPE)	\
					){																															\
					if(eventList.size()==oldEventListSize) oldEventListSize=i+1; /*We are moving the current event, which we want to delete*/ 	\
					eventList[i]=eventList.back(); eventList.pop_back();																		\
					/*Note that we do NOT remove the event if it is the current one, as it will be removed at the end of the time loop.*/		\
				}																																\
			}																																	\
			for(int i=-1; i<=1; i++){																											\
				if(cx+i >= numPositions) {break;}																								\
				else if(cx+i < 0) {continue;}																									\
																																				\
				for(int j=-1; j<=1; j++){																										\
					if(cy+j >= numPositions) {break;}																							\
					else if(cy+j < 0) {continue;}																								\
																																				\
					for(int k=-1; k<=1; k++){																									\
						if(cz+k >= numPositions) {break;}																						\
						else if(cz+k < 0) {continue;}																							\
						REMOVE_ELEMENT_FROM_VECTOR_IF_PRESENT(THREED_TO_ONED(cx+i,cy+j,cz+k),cellSites,cellNum);								\
					}																															\
				}																																\
			}																																	\
		}

	//Vector operations
		#define REMOVE_ELEMENT_FROM_VECTOR_C(cx, cy, cz, arr, ele){	\
			vecIt = &arr[THREED_TO_ONED((cx),(cy),(cz))];			\
			(*vecIt).erase( 										\
				(*vecIt).begin()+distance(							\
					(*vecIt).begin(), find(							\
						(*vecIt).begin(), (*vecIt).end(), (ele)		\
					)/*erases the element 'ele' from the vector*/	\
				)													\
			);														\
		}
		#define REMOVE_ELEMENT_FROM_VECTOR_I(ind, arr, ele){		\
			vecIt = &arr[ind];										\
			(*vecIt).erase( 										\
				(*vecIt).begin()+distance(							\
					(*vecIt).begin(), find(							\
						(*vecIt).begin(), (*vecIt).end(), (ele)		\
					)/*erases the element 'ele' from the vector*/	\
				)													\
			);														\
		}
		#define REMOVE_ELEMENT_FROM_VECTOR_IF_PRESENT(ind, arr, ele){\
			vecIt = &arr[ind];										\
			if((dist=uint(distance(									\
					(*vecIt).begin(), find(							\
						(*vecIt).begin(), (*vecIt).end(), (ele)		\
					)/*erases the element 'ele' from the vector*/	\
				)))!=(*vecIt).size()								\
			){ 														\
				(*vecIt).erase((*vecIt).begin()+dist); }			\
		}

		#define REGENERATE_CELL_POSITIONS_AND_ANTIGEN(numWavesToGen){																									\
			/*Place DCs into LN and give them a random number of antigen*/																													\
				dCellsPresent=0;  for(uint wav=0; wav<numWaves; wav++) {dCellsPresentPerWave[wav]=0;}																																								\
				for(uint mdci=0; mdci<numDCells*numWavesToGen; mdci++){																														\
					/*Set the number of cognate antigen that this DC has*/																													\
					dCellList[mdci].set_num_cog_antigen(waveTimes[waveNumber], cogAgOnArrival, cognateAgPrecision, cognateAgProbabilityTable);												\
																																															\
					/*Add to grid, if we want them to all appear at once*/																													\
					if(timeBetweenDCArrival==0){																																			\
						if(USING_SOURCE) { 	ADD_DC_TO_SIMULATION_AT_SOURCE(dCellList,tCellList,dSites,tSites, dCellsPresent, waveTimes[waveNumber], x,y,z,cx,cy,cz, vx,vy,vz, mfp,vmag);}	\
						else {				ADD_DC_TO_SIMULATION_AT_RANDOM(dCellList,tCellList,dSites,tSites, dCellsPresent, waveTimes[waveNumber], x,y,z,cx,cy,cz, vx,vy,vz, mfp,vmag);}	\
					}																																										\
				}																																											\
				if(timeBetweenDCArrival==0){dCellsPresent=numDCells; dCellsPresentPerWave[waveNumber]=numDCells;}																			\
				/*Place events to remove and add additional waves of DCs*/																													\
				if(timeBetweenDCArrival!=0) {eventList.push_back(eventTime(waveTimes[waveNumber],DC_ENTRY_EVENT,waveNumber));} /*Time=0 for this event for the first wave, and then the relevant vaccination time thereafter*/	\
				if(numWaves>1) {eventList.push_back(eventTime(waveTimes[waveNumber]+activationTimeLimit,DC_DEATH_EVENT,waveNumber));} /*Which DCs to kill will be worked out automatically from the # of active waves*/			\
				for(uint w=1; waveNumber+w<numWaves; w++){																																				\
					eventList.push_back(eventTime(waveTimes[waveNumber+w],DC_ENTRY_EVENT,waveNumber+w));																					\
					eventList.push_back(eventTime(waveTimes[waveNumber+w]+activationTimeLimit,DC_DEATH_EVENT,waveNumber+w));																\
				}																																											\
				/*Place T cells on the grid, ensuring they are away from the DCs*/																											\
				posFailures=0;																																								\
				for(uint tcnum=0; tcnum<numTCells; tcnum++){																																\
					if(tCellList[tcnum].TC_get_activation_state()==1) continue;																												\
					while(1==1){ /*More efficient than the better practice of while(inContact==1), because of the need to continue if the random coordinates generated are bad.*/			\
						GENERATE_COORDS; /*Gets random values of x, y and z between -radius and +radius (coordinates from a cube of side = diameter)*/										\
						if(OUTSIDE_SPHERE(x,y,z)) continue;																																	\
						inContact=0; 																																						\
						cx=COORD(x); cy=COORD(y); cz=COORD(z);																																\
						for (auto mdci: dSites[THREED_TO_ONED(cx,cy,cz)]) {																													\
							ox=dCellList[mdci].get_x(); oy=dCellList[mdci].get_y(); oz=dCellList[mdci].get_z(); 																			\
							if(SQMAGNITUDE(x-ox,y-oy,z-oz)<contactRadius*contactRadius) inContact=1; /*Too close to DC - need to try again*/												\
						}																																									\
						if( (posFailures++)>POSITION_OOB_TOLERANCE ){cerr << "ERROR: Too many attempts to place a T-cell onto the grid without contacting a DC. Check parameter choices." << endl; return 1;}	\
						else if(inContact==0) break;																																		\
					}																																										\
					/*Reset activation state*/																																				\
					tCellList[tcnum].TC_set_activation_state(0);																															\
					tCellList[tcnum].unset_failed_interaction_ID();																															\
					/*Set position and velocity*/																																			\
					tCellList[tcnum].set_position(waveTimes[waveNumber], x, y, z); tCellInitialX[tcnum]=x; tCellInitialY[tcnum]=y; tCellInitialZ[tcnum]=z;									\
					CHANGE_VELOCITY(tCellList, TCELL_TYPE,tcnum, vx,vy,vz,mfp,vmag);																										\
					/*Find which discrete sites this cell can contact, and mark them*/																										\
					cx=COORD(x); cy=COORD(y); cz=COORD(z);																																	\
					CALCULATE_CONTACTABLE_GRID_SITES(x,y,z,cx,cy,cz);																														\
					for(int i=0; i<3; i++){																																					\
						if(cx+i-1 >= numPositions) {break;}																																	\
						else if(cx+i-1 < 0) {continue;}																																		\
						for(int j=0; j<3; j++){																																				\
							if(cy+j-1 >= numPositions) {break;}																																\
							else if(cy+j-1 < 0) {continue;}																																	\
							for(int k=0; k<3; k++){																																			\
								if(cz+k-1 >= numPositions) {break;}																															\
								else if(cz+k-1 < 0) {continue;}																																\
								if(localNeighbourhood[i+3*j+9*k]==1){																														\
									tSites[THREED_TO_ONED(cx+i-1,cy+j-1,cz+k-1)].push_back(tcnum);																							\
								}																																							\
							}																																								\
						}																																									\
					}																																										\
					ADD_EVENTS_AFTER_VELOCITY_CHANGE(tCellList,dCellList,tSites,dSites, TCELL_TYPE,tcnum,  waveTimes[waveNumber], x,y,z,cx,cy,cz, vx,vy,vz, mfp,vmag);						\
				}																																											\
				/*Push back events for recording data and plotting the current output*/																										\
				if(plot){																																									\
					PLOT_CELLS;																																						\
					eventList.push_back(eventTime(waveTimes[waveNumber],GRAPHICAL_UPDATE_EVENT));																						\
				}																																											\
				eventList.push_back(eventTime(waveTimes[waveNumber],MEASURE_OBSERVABLES_EVENT));																							\
				sort(eventList.begin(), eventList.end());																																	\
		}

	//Macro to output moment data once per chunk
		#define OUTPUT_MOMENT_DATA 																														\
			cout << "#timeUntilFirstActivation "; for(int i=0; i<numRepeats*numChunks;i++) cout << timeUntilFirstActivation[i] << " "; cout << endl;	\
			cout << "#timeUntil25pcActivation ";  for(int i=0; i<numRepeats*numChunks;i++) cout << timeUntil25pcActivation[i]  << " "; cout << endl;	\
			cout << "#timeUntil50pcActivation ";  for(int i=0; i<numRepeats*numChunks;i++) cout << timeUntil50pcActivation[i]  << " "; cout << endl;	\
			cout << "#timeUntil75pcActivation ";  for(int i=0; i<numRepeats*numChunks;i++) cout << timeUntil75pcActivation[i]  << " "; cout << endl;	\
			cout << "#timeUntil90pcActivation ";  for(int i=0; i<numRepeats*numChunks;i++) cout << timeUntil90pcActivation[i]  << " "; cout << endl;	\
			cout << "#timeUntil100pcActivation "; for(int i=0; i<numRepeats*numChunks;i++) cout << timeUntil100pcActivation[i] << " "; cout << endl;	\
			cout << "#numActivationsPerRepeat ";  for(int i=0; i<numRepeats*numChunks;i++) cout << numActivationsPerRepeat[i]  << " "; cout << endl;	\
			MOMENT_HEADER;																																\
			MOMENT_OUT(atLeastOneActivated);																											\
			MOMENT_OUT(realTimeTaken);																													\
			MOMENT_TM_HEADER;																															\
			MOMENT_TM_OUT_TC(numActivated);																												\
			MOMENT_TM_OUT_TC(tCellDisplacementFromStart);																								\
			cout << "#data_end" << endl;
#endif 
	//__FUNCTIONS_INCLUDED__
