//g++ LN.cpp -o LN -std=c++11 -lboost_system -lboost_iostreams
#define __LINEAR_MOVEMENT__
//#include "LN.h" //At bottom of file.
#include "linearFunctions.h"
#include "parametersAndLibraries.h"
#include "graphicalOutput.h"

//C++ libraries that do not work in MEX
	#include <assert.h>
	//#define NDEBUG //Disables assert statements

//Prototype (with variable names intact for sanity)
	int simulation(long unsigned int numTimeSteps, double timeStep, uint numTimeMeasurements, vector<double> waveTimes, double radius, 
		double contactRadius, double tCellAgSpecFreq, uint numTCells, uint numDCells, int numAntigenInContactArea, 
		int numAntigenOnDC, int tCellActivationThreshold, double cogAgInDermis, double cogAgOnArrival, 
		unsigned int firstDCArrival, unsigned int DCArrivalDuration, double antigenDecayRate, double tVelocityMean, 
		double tVelocityStDev, double tGammaShape, double tGammaScale, double freePathMean, double freePathStDev, 
		double dCellVelocity, int seed, int numRepeats, int numChunks, MOMENT_TYPE *mom_numActivated, 
		int mom_max_numActivated, const int mom_tm_maxnumActivated, MOMENT_TYPE *mom_numInteractions, 
		int mom_max_numInteractions, const int mom_tm_maxnumInteractions, MOMENT_TYPE *mom_numUniqueInteractions, 
		int mom_max_numUniqueInteractions, const int mom_tm_maxnumUniqueInteractions, 
		MOMENT_TYPE *mom_tCellDisplacementFromStart, int mom_max_tCellDisplacementFromStart, 
		const int mom_tm_maxtCellDisplacementFromStart, MOMENT_TYPE *mom_realTimeTaken, int mom_max_realTimeTaken, 
		MOMENT_TYPE *mom_atLeastOneActivated, int mom_max_atLeastOneActivated, double *timeUntilFirstActivation,  
		double *timeUntil25pcActivation, double *timeUntil50pcActivation,  double *timeUntil75pcActivation, 
		double *timeUntil90pcActivation, double *timeUntil100pcActivation, int *numActivationsPerRepeat, int plot, 
		int time_steps_per_graphical_update);

//Entry point for c++
	int main(int argc, char *argv[]){
		//Initialise params
		double timeStep = DEFAULT_TIMESTEP;
		long unsigned int numTimeSteps = NUM_TIMESTEPS_2DAY(timeStep);
		double maxTime=timeStep*numTimeSteps;
		uint numTimeMeasurements=DEFAULT_NUM_TIME_MEASUREMENTS;
		double radius = DEFAULT_RADIUS;
		double contactRadius = DEFAULT_CONTACT_RADIUS;
		double tCellAgSpecFreq = DEFAULT_AGSPEC_FREQ; //n.b. these are antigen-specific T-cells, rho*phi, where rho is the density of all T-cells and phi is the number specific to the antigen of interest
		uint numTCells = TOTAL_TNUM * tCellAgSpecFreq;
		uint numDCells = DEFAULT_DENDNUM;
		int numAntigenInContactArea = DEFAULT_NUM_ANTIGEN_IN_CONTACT_AREA;
		int numAntigenOnDC = numAntigenInContactArea*2000.0/7.8;
		int tCellActivationThreshold = DEFAULT_TCELL_ACTIVATION_THRESHOLD;
		double cogAgInDermis = DEFAULT_COGNATE_RATIO_MEAN_AT_DERMIS;
		double cogAgOnArrival = DEFAULT_COGNATE_RATIO_MEAN_AT_DERMIS;
		// double cognateAntigenRatioStDev = DEFAULT_COGNATE_RATIO_STDEV;
		unsigned int firstDCArrival = DEFAULT_FIRST_DC_ARRIVAL;
		unsigned int DCArrivalDuration = DEFAULT_DC_ARRIVAL_DURATION;
		double tVelocityMean=DEFAULT_T_VELOCITY_MEAN;
		double tVelocityStDev=DEFAULT_T_VELOCITY_STDEV;
		double tGammaShape=T_GAMMA_SHAPE;
		double tGammaScale=T_GAMMA_SCALE(DEFAULT_T_VELOCITY_MEAN);
		double freePathMean=DEFAULT_T_FREE_PATH_MEAN;
		double freePathStDev=DEFAULT_T_FREE_PATH_STDEV;
		double dCellVelocity=DEFAULT_D_VELOCITY_MEAN;
		int seed = DEFAULT_SEED;
		int numRepeats = DEFAULT_NUM_REPEATS;
		int numChunks = DEFAULT_NUM_CHUNKS;
		int plot=0, pause=0;	
		int time_steps_per_graphical_update=DEFAULT_TIME_STEPS_PER_GRAPHICAL_UPDATE;
		double antigenDecayRate = DEFAULT_ANTIGEN_DECAY_RATE;
		string timeStepOpt;
		bool TCellsSet=false, initialAntigenSet=false;
		vector<double> waveTimes = {0.0};

		//Get arguments
		cout << "#Input: "; for(int i=0; i<argc; i++) cout << argv[i] << " "; cout << endl;
		int opt;
		while ((opt = getopt(argc,argv, "R:b:T:t:D:N:n:A:@:P:p:k:V:v:F:f:W:s:M:m:,:w:r:c:xyz:")) != -1){
			switch(opt){
				case 'R':
					radius = atof(optarg);
					if(!TCellsSet) numTCells = TOTAL_TNUM * tCellAgSpecFreq;
					cout << "#Info: Radius from command line: " << radius << endl;
					break;

				case 'b':
					contactRadius = atof(optarg);
					cout << "#Info: Contact radius from command line: " << contactRadius << endl;
					break;

				case 'T':
					if(atoi(optarg)<=0) {cerr << "T cell count must be a positive integer. Aborting." << endl; return 1;}
					TCellsSet=true;
					numTCells = atoi(optarg);
					tCellAgSpecFreq = double(numTCells)/TOTAL_TNUM;
					cout << "#Info: Number of T cells from command line: " << numTCells << endl;
					break;
				
				case 't':
					tCellAgSpecFreq = atof(optarg);
					numTCells = TOTAL_TNUM * tCellAgSpecFreq;
					cout << "#Info: Antigen-specific T cell density from command line: " << tCellAgSpecFreq << endl;
					break;

				case 'D':
					if(atoi(optarg)<=0) {cerr << "D cell count must be a positive integer. Aborting." << endl; return 1;}
					numDCells = atoi(optarg);
					cout << "#Info: Number of dendritic cells from command line: " << numDCells << endl;
					break;

				case 'N':
					numAntigenInContactArea = atoi(optarg);
					numAntigenOnDC=2000.0/7.8 * numAntigenInContactArea;
					cout << "#Info: Number of antigen in contact area from command line: " << numAntigenInContactArea << endl;
					cout << "#Info: Number of antigen on DC from command line: " << numAntigenOnDC << endl;
					break;

				case 'n':
					tCellActivationThreshold = atoi(optarg);
					cout << "#Info: T cell activation threshold from command line: " << tCellActivationThreshold << endl;
					break;

				case 'A':
					initialAntigenSet=true;
					cogAgOnArrival = atof(optarg);
					cout << "#Info: Cognate antigen ratio on arrival from command line: " << cogAgOnArrival << endl;
					// if(cognateAntigenRatioStDev==DEFAULT_COGNATE_RATIO_STDEV){
					// 	cognateAntigenRatioStDev=sqrt(cognateAntigenRatioMean/NUM_ANTIGEN_ON_DC); //Which is how the default was calculated
					// 	cout << "#Info: Standard deviation of cognate antigen ratio set to " << cognateAntigenRatioStDev << endl;
					// }
					break;

				case '@':
					cogAgInDermis = atof(optarg);
					cout << "#Info: Cognate antigen ratio in dermis from command line: " << cogAgInDermis << endl;
					break;

				// case 'a':
				// 	cognateAntigenRatioStDev = atof(optarg);
				// 	cout << "#Info: Standard deviation cognate antigen ratio from command line: " << cognateAntigenRatioStDev << endl;
				// 	break;

				case 'P':
					firstDCArrival = atoi(optarg);
					cout << "#Info: First DC arrival time from command line: " << firstDCArrival << endl;
					break;

				case 'p':
					DCArrivalDuration = atoi(optarg);
					cout << "#Info: DC arrival period from command line: " << DCArrivalDuration << endl;
					break;

				case 'k':
					antigenDecayRate = atof(optarg);
					cout << "#Info: Antigen decay rate from command line: " << antigenDecayRate << endl;
					break;

				#ifdef _GAUSS_
					case 'V':
						tVelocityMean = atof(optarg);
						cout << "#Info: T cell velocity mean from command line: " << tVelocityMean << endl;
						break;
					case 'v':
						tVelocityStDev = atof(optarg);
						cout << "#Info: Gaussian T cell velocity standard deviation from command line: " << tVelocityStDev << endl;
						break;
				#else
					case 'V':
						tVelocityMean = atof(optarg);
						tGammaScale = T_GAMMA_SCALE(tVelocityMean);
						cout << "#Info: T cell velocity mean from command line: " << tVelocityMean << endl;
						break;
				#endif

				case 'F':
					freePathMean = atof(optarg);
					cout << "#Info: T cell mean free path from command line: " << freePathMean << endl;
					if(freePathStDev==DEFAULT_T_FREE_PATH_STDEV){
						freePathStDev = freePathMean*3/25;
						cout << "#Info: T cell standard deviation of free path set to: " << freePathStDev << endl;
					}
					break;

				case 'f':
					freePathStDev = atof(optarg);
					cout << "#Info: T cell standard deviation of free path from command line: " << freePathStDev << endl;
					break;

				case 'W':
					dCellVelocity = atof(optarg);
					cout << "#Info: D cell velocity mean from command line: " << dCellVelocity << endl;
					break;
					
				case 's':
					seed = atoi(optarg);
					cout << "#Info: Seed from command line: " << seed << endl;
					break;

				case 'M':
					timeStepOpt = optarg;
					if (isalpha(timeStepOpt[0])){
						transform(timeStepOpt.begin(), timeStepOpt.end(), timeStepOpt.begin(), ::tolower); //From <algorithm>
						if(timeStepOpt=="hour") 	{numTimeSteps = NUM_TIMESTEPS_HOUR(timeStep); maxTime=60;}
						else if(timeStepOpt=="day")	{numTimeSteps = NUM_TIMESTEPS_DAY(timeStep); maxTime=24*60;}
						else if(timeStepOpt=="2day"){numTimeSteps = NUM_TIMESTEPS_2DAY(timeStep); maxTime=48*60;}
						else {						cerr <<"Unknown value for timeStepOpt="<<timeStepOpt<<". Aborting." << endl; return 1;}
					}
					else {maxTime=stof(timeStepOpt); numTimeSteps=maxTime/timeStep;}
					cout << "#Info: Max time from command line: " << maxTime << endl;
					cout << "#Info: Number of time steps from command line: " << numTimeSteps << endl;
					break;

				case 'm':
					timeStep = atof(optarg);
					if(timeStep>1 || timeStep<0) cerr << "Simulation requires 0<timeStep<1. Aborting." << endl;
					else cout << "#Info: timestep from command line: " << timeStep << endl;

					if(timeStepOpt!="unset"){
						if (isalpha(timeStepOpt[0])){
							transform(timeStepOpt.begin(), timeStepOpt.end(), timeStepOpt.begin(), ::tolower); //From <algorithm>
							if(timeStepOpt=="hour") 	{numTimeSteps = NUM_TIMESTEPS_HOUR(timeStep); maxTime=60;}
							else if(timeStepOpt=="day")	{numTimeSteps = NUM_TIMESTEPS_DAY(timeStep); maxTime=24*60;}
							else if(timeStepOpt=="2day"){numTimeSteps = NUM_TIMESTEPS_2DAY(timeStep); maxTime=48*60;}
							else {						cerr <<"Unknown value for timeStepOpt="<<timeStepOpt<<". Aborting." << endl; return 1;}
						}
						else {maxTime=stof(timeStepOpt); numTimeSteps=maxTime/timeStep;}
					}
					break;

				case ',':
					opt= atoi(optarg); //reusing the argument!
					if(opt<0){cerr << "Number of time measurements must be a positive number." << endl; return 1;}
					numTimeMeasurements=opt;
					cout << "#Info: Num time measurements from command line: " << numTimeMeasurements << endl;
					break;

				case 'w':
				{ //Brackets to close scope of the new variables introduced in here
					stringstream waveSs; waveSs.str(optarg);
					int bufsize=100; char buffer[bufsize];
					// Count number of delimeters
					size_t numWaves=0; char *ptr=optarg;
					while( (ptr=strchr(ptr,',')) != NULL){
						numWaves++; ptr++;
					}
					numWaves++; //If  there are two commas, thee times were given, etc
					waveTimes = vector<double>(numWaves);
					//Fill vector with the times given by the user
					for(int i=0;waveSs.good();){
						waveSs.getline(buffer,bufsize,',');
						if (buffer[0]!='\0'){
							try{waveTimes[i++]=stod(buffer);}
							catch(const invalid_argument&){
								cerr << "Invalid argument for std::stod when attempting to read vaccination times "
									<< "from buffer " << buffer << " from stringstream " << optarg << endl;
								terminate();
							}
							if (waveTimes[i-1]<0){
								cerr << "Injection times must be greater than 0. Last value given: " << waveTimes[i-1]
									<< " Argument given: " << optarg << endl;
								return 1;
							}
						}
					}
					sort(waveTimes.begin(), waveTimes.end());
					//If 0.0 not given, add it
					if (waveTimes[0]-0.0>1e-2){
						waveTimes.insert(waveTimes.begin(),0.0);
						numWaves++;
					}
					cout << "#Info: Injection times from command line:";
					for (auto e: waveTimes) {cout << e << ",";} cout << endl;
					break;
				}

				case 'r':
					numRepeats = atoi(optarg);
					if(plot==1) cerr << "#WARNING: Num repeats set to " << numRepeats << " with plotting enabled. Was this intentional?" << endl;
					else		cout << "#Info: Num repeats from command line: " << numRepeats << endl;
					break;

				case 'c'://We do r repeats per chunk and output data after each chunk
					numChunks = atoi(optarg);
					if(plot==1) cerr << "#WARNING: Num chunks set to " << numChunks << "with plotting enabled. Was this intentional?" << endl;
					else		cout << "#Info: Num chunks from command line: " << numChunks << endl;
					break;

				case 'x':
					pause=1;
					cerr << "#Info: Graphical output will start paused, if using GLSC3D." << endl;
					break;

				case 'y':
					plot=1;
					if(numRepeats>1)cerr << "#WARNING: Plotting enabled with more than one repeat of simulation. Was this intentional?" << endl;
					else 			cout << "#Info: Plotting enabled" << endl;
					break;

				case 'z':
					time_steps_per_graphical_update = atoi(optarg);
					if(time_steps_per_graphical_update<=0){
						cerr << "#ERROR: time_steps_per_graphical_update must be >0. Value given: " << time_steps_per_graphical_update << endl;
						return 1;
					}
					cout << "#Info: Time steps per graphical update: " << time_steps_per_graphical_update << endl;
					break;

				default:
					cerr << "R:\t"<<"radius\t\t\t"<<				"default: "	<<DEFAULT_RADIUS << endl;
					cerr << "b:\t"<<"contactRadius\t\t"<<			"default: "	<<DEFAULT_CONTACT_RADIUS << endl;
					cerr << "T:\t"<<"numTCells\t\t"<<				"default: "	<<DEFAULT_AGSPEC_FREQ*TOTAL_TNUM << endl;
					cerr << "t:\t"<<"tCellDensity\t\t"<<			"default: "	<<DEFAULT_AGSPEC_FREQ << endl;
					cerr << "D:\t"<<"numDendriticCells\t"<<			"default: "	<<DEFAULT_DENDNUM << endl;
					cerr << "N:\t"<<"numAntigenInContactArea\t"<<	"default: "	<<DEFAULT_NUM_ANTIGEN_IN_CONTACT_AREA << endl;
					cerr << "n:\t"<<"tCellActivationThreshold\t"<<	"default: "	<<DEFAULT_TCELL_ACTIVATION_THRESHOLD << endl;
					cerr << "@:\t"<<"cogAgInDermis\t"<<				"default: "	<<DEFAULT_COGNATE_RATIO_MEAN_AT_DERMIS << endl;
					cerr << "A:\t"<<"cogAgOnArrival\t"<<			"default: "	<<DEFAULT_COGNATE_RATIO_MEAN_AT_DERMIS << "*exp(-antigenDecayRate*firstDCArrival) = " << DEFAULT_COGNATE_RATIO_MEAN_AT_DERMIS*exp(-DEFAULT_ANTIGEN_DECAY_RATE*DEFAULT_FIRST_DC_ARRIVAL) << endl;
					// cerr << "a:\t"<<"cognateAgRatioStDev\t"<<		"default: "	<<DEFAULT_COGNATE_RATIO_STDEV << endl;
					cerr << "P:\t"<<"firstDCArrival\t"<<			"default: "	<<DEFAULT_FIRST_DC_ARRIVAL << endl;
					cerr << "p:\t"<<"DCArrivalDuration\t"<<			"default: "	<<DEFAULT_DC_ARRIVAL_DURATION << endl;
					cerr << "k:\t"<<"antigenDecayRate\t" <<			"default "	<<DEFAULT_ANTIGEN_DECAY_RATE << endl;
					cerr << "V:\t"<<"tVelocityMean\t" << 			"default: " <<DEFAULT_T_VELOCITY_MEAN << endl;
					#ifdef _GAUSS_
						cerr << "v:\t"<<"tVelocityStDev\t" << 			"default: " <<DEFAULT_T_VELOCITY_STDEV << endl;
					#endif
					cerr << "F:\t"<<"freePathMean\t" <<				"default: " <<DEFAULT_T_FREE_PATH_MEAN << endl;
					cerr << "f:\t"<<"freePathStDev\t" <<			"default: " <<DEFAULT_T_FREE_PATH_STDEV << endl;
					cerr << "W:\t"<<"dCellVelocity\t" <<			"default: " <<DEFAULT_D_VELOCITY_MEAN << endl;
					cerr << "s:\t"<<"seed\t\t\t"<<					"default: "	<<DEFAULT_SEED << endl;
					cerr << "M:\t"<<"maxTime (numTimeSteps)\t\t"<<	"default: day ; possible values: hour, day or 2day, or a float. If there are secondary vaccinations, this will be automatically increased to be e.g. a day after the last vaccination." << endl;
					cerr << "m:\t"<<"timeStep\t\t"<<				"default: " << DEFAULT_TIMESTEP << endl;
					cerr << ",:\t"<<"numTimeMeasurements\t\t"<<		"default: " << DEFAULT_NUM_TIME_MEASUREMENTS << endl;
					cerr << "w:\t"<<"waveTimes\t\t"<<				"default: " << 0.0  << ". This argument is given as a comma separated list." << endl;
					cerr << "r:\t"<<"numRepeats\t\t"<<				"default: "	<<DEFAULT_NUM_REPEATS << endl;
					cerr << "c:\t"<<"numChunks\t\t"<<				"default: "	<<DEFAULT_NUM_CHUNKS << endl;
					cerr << "y \t"<<"plotting=ON\t\t"<<				"default: "	<<"OFF" << endl;
					cerr << "z:\t"<<"timePerGraphicUpdate\t"<<		"default: "	<<DEFAULT_TIME_STEPS_PER_GRAPHICAL_UPDATE << endl;
					cerr << endl;
					return 1;
			}
		}

		//Error check
		if(numTCells<=0){
			cerr << "Error: the number of T cells to be simulated, " << numTCells << " (density " << outputDoubleWith1000Seps(T_CELL_DENSITY)
			<< "/mm^3 and precursor freq " << tCellAgSpecFreq << " in volume " << boost::format ("%0.1g") % (4.0*0.3333*3.1415926 * radius*radius*radius * 1E-9) 
			<< "mm^3), is not a positive number. Increase LN radius or T cell density." << endl;
			return 1;
		}
		if(cogAgInDermis!=DEFAULT_COGNATE_RATIO_MEAN_AT_DERMIS && initialAntigenSet==true){
			cerr << "Error: attempted to change the initial amount of antigen in the dermis and the amount upon arrival in the LN simultaneously. Aborting." << endl;
			return 1;
		}
		for(uint i=0; i<waveTimes.size()-1; i++){
			if(waveTimes[i+1]-waveTimes[i]<timeStep){
				cerr << "Cannot have any two vaccination times be the same. Argument given: " << optarg << endl;
				return 1;
			}
		}

		//Calculate amount of antigen upon DC arrival in dermis
		if(initialAntigenSet==false){
			cogAgOnArrival=cogAgInDermis*exp(-antigenDecayRate*firstDCArrival);
			cout << "#Info: cognate antigen ratio upon LN arrival set to " << cogAgInDermis << " * exp(-" << antigenDecayRate << "*" << firstDCArrival << ") = " << cogAgOnArrival << endl;	
		} 

		//Increase numTimeSteps if there are secondary vaccinations
		if (waveTimes.back()!=0.0){
			cout << "#Info: numTimeSteps increased from " << numTimeSteps << " to ";
			numTimeSteps+=waveTimes.back()/timeStep;
			cout << numTimeSteps << endl;
		}

		//If paused activated and plotting is on, change "plot" variable to indicate this. It will be dealt with in the code later.
		if(pause && plot) plot=2;

		//Initialise output data arrays
		//Declare a 2-moment (avg and variance) for the number of activated T-cells as a fctn of time and the overall displacement from begging for each cell
		MOMENT_TM_DECL(numActivated,2,numTimeMeasurements+1); //+1 to get both 0 time and final time
		MOMENT_TM_DECL(tCellDisplacementFromStart,2,numTimeMeasurements+1);
		MOMENT_TM_DECL(numInteractions,2,numTimeMeasurements+1);
		MOMENT_TM_DECL(numUniqueInteractions,2,numTimeMeasurements+1);
		
		//Declare 2-moment for time taken per iteration
		MOMENT_DECL(realTimeTaken,2);
		MOMENT_DECL(atLeastOneActivated,2);

		double timeUntilFirstActivation[numChunks * numRepeats], timeUntil25pcActivation[numChunks*numRepeats], timeUntil50pcActivation[numChunks * numRepeats], 
		timeUntil75pcActivation[numChunks * numRepeats], timeUntil90pcActivation[numChunks * numRepeats], timeUntil100pcActivation[numChunks * numRepeats];
		int numActivationsPerRepeat[numChunks * numRepeats];
		for(int i=0; i<numChunks*numRepeats; i++){
			timeUntilFirstActivation[i]=numTimeSteps*timeStep;
			timeUntil25pcActivation[i]=numTimeSteps*timeStep;
			timeUntil50pcActivation[i]=numTimeSteps*timeStep;
			timeUntil75pcActivation[i]=numTimeSteps*timeStep;
			timeUntil90pcActivation[i]=numTimeSteps*timeStep;
			timeUntil100pcActivation[i]=numTimeSteps*timeStep;
			numActivationsPerRepeat[i]=0;
		}

		return simulation(numTimeSteps, timeStep, numTimeMeasurements, waveTimes, radius, contactRadius, tCellAgSpecFreq, numTCells, 
			numDCells, numAntigenInContactArea, numAntigenOnDC, tCellActivationThreshold, cogAgInDermis, cogAgOnArrival, 
			firstDCArrival, DCArrivalDuration, antigenDecayRate, tVelocityMean, tVelocityStDev, tGammaShape, tGammaScale, 
			freePathMean, freePathStDev, dCellVelocity, seed, numRepeats, numChunks, mom_numActivated, 
			mom_max_numActivated, mom_tm_maxnumActivated, mom_numInteractions, mom_max_numInteractions, mom_tm_maxnumInteractions,
			mom_numUniqueInteractions, mom_max_numUniqueInteractions, mom_tm_maxnumUniqueInteractions, 
			mom_tCellDisplacementFromStart, mom_max_tCellDisplacementFromStart, mom_tm_maxtCellDisplacementFromStart, 
			mom_realTimeTaken, mom_max_realTimeTaken, mom_atLeastOneActivated, mom_max_atLeastOneActivated, 
			timeUntilFirstActivation, timeUntil25pcActivation, timeUntil50pcActivation, timeUntil75pcActivation, 
			timeUntil90pcActivation, timeUntil100pcActivation, numActivationsPerRepeat, plot, time_steps_per_graphical_update);
	}

#include "LN.h"