#include "parametersAndLibraries.h"
#include "classesAndFunctions.h"
// #include "graphicalOutput.h"

int simulation(long unsigned int numTimeSteps, double timeStep, uint numTimeMeasurements, vector<double> waveTimes, double radius, 
	double contactRadius, double tCellAgSpecFreq, uint numTCells, uint numDCells, int numAntigenInContactArea, 
	int numAntigenOnDC, int tCellActivationThreshold, double cogAgInDermis, double cogAgOnArrival, 
	unsigned int firstDCArrival, unsigned int DCArrivalDuration, double antigenDecayRate, double tVelocityMean, 
	double tVelocityStDev, double tGammaShape, double tGammaScale, double freePathMean, double freePathStDev, 
	double dCellVelocity, int seed, int numRepeats, int numChunks,  MOMENT_TYPE *mom_numActivated, 
	int mom_max_numActivated, const int mom_tm_maxnumActivated, MOMENT_TYPE *mom_numInteractions, 
	int mom_max_numInteractions, const int mom_tm_maxnumInteractions, MOMENT_TYPE *mom_numUniqueInteractions, 
	int mom_max_numUniqueInteractions, const int mom_tm_maxnumUniqueInteractions, 
	MOMENT_TYPE *mom_tCellDisplacementFromStart, int mom_max_tCellDisplacementFromStart, 
	const int mom_tm_maxtCellDisplacementFromStart, MOMENT_TYPE *mom_realTimeTaken, int mom_max_realTimeTaken, 
	MOMENT_TYPE *mom_atLeastOneActivated, int mom_max_atLeastOneActivated, double *timeUntilFirstActivation, 
	double *timeUntil25pcActivation, double *timeUntil50pcActivation, double *timeUntil75pcActivation, 
	double *timeUntil90pcActivation, double *timeUntil100pcActivation, int *numActivationsPerRepeat, int plot=0, 
	int time_steps_per_graphical_update=10){

	//Output for analysis scripts' sake
	cout << "#Info: radius="					<< radius << endl;
	cout << "#Info: contactRadius=" 			<< contactRadius << endl;
	cout << "#Info: tCellAgSpecFreq="			<< tCellAgSpecFreq << endl;
	cout << "#Info: numTCells=" 				<< numTCells << endl;
	cout << "#Info: numDCells=" 				<< numDCells << endl;
	cout << "#Info: numAntigenInContactArea="	<< numAntigenInContactArea << endl;
	cout << "#Info: numAntigenOnDC="			<< numAntigenOnDC << endl;
	cout << "#Info: tCellActivationThreshold="	<< tCellActivationThreshold << endl;
	cout << "#Info: cogAgInDermis="				<< cogAgInDermis << endl;
	cout << "#Info: cogAgOnArrival="			<< cogAgOnArrival << endl;
	// cout << "#Info: cognateAgRatioStDev="		<< cognateAntigenRatioStDev << endl;
	cout << "#Info: firstDCArrival="			<< firstDCArrival << endl;
	cout << "#Info: DCArrivalDuration="			<< DCArrivalDuration << endl;
	cout << "#Info: antigenDecayRate="			<< antigenDecayRate << endl;
	#ifdef _GAUSS_
		cout << "#Info: VelocityDistribution: gaussian" << endl;
		cout << "#Info: tVelocityMean="				<< tVelocityMean << endl;
		cout << "#Info: tVelocityStDev="			<< tVelocityStDev << endl;
	#else
		cout << "#Info: VelocityDistribution: gamma"<< endl;
		cout << "#Info: tVelocityMean="				<< tVelocityMean << endl;
		cout << "#Info: tVelocityStDev="			<< sqrt(tGammaShape*tGammaScale*tGammaScale) << endl;
	#endif
	cout << "#Info: freePathMean="				<< freePathMean << endl;
	cout << "#Info: freePathStDev="				<< freePathStDev << endl;
	cout << "#Info: dCellVelocity="				<< dCellVelocity << endl;
	cout << "#Info: seed=" 						<< seed << endl;
	cout << "#Info: numTimeSteps=" 				<< numTimeSteps << endl;
	cout << "#Info: timeStep="					<< timeStep << endl;
	cout << "#Info: numTimeMeasurements="		<< numTimeMeasurements << endl;
	cout << "#Info: maxTime="					<< numTimeSteps*timeStep << endl;
	cout << "#Info: numWaves="					<< waveTimes.size() << endl;
	cout << "#Info: waveTimes=";				for (auto e: waveTimes) cout << e << ","; cout << endl;
	cout << "#Info: numRepeats=" 				<< numRepeats << endl;
	cout << "#Info: numChunks=" 				<< numChunks << endl;

	//More parameters used to hold data later
	double x, y, z, dx, dy, dz, newx, newy, newz, dendx, dendy, dendz, vmag, theta, phi, magsq, dist, cognateAgPrecision, minimumCognateAg=0, maxTime=numTimeSteps*timeStep;
	int tCellNum=0, dCellNum=0, dCellsPresent=0, coordX, coordY, coordZ, dcoordX, dcoordY, dcoordZ, inContact=0, cognateAgProbabilityTable_size, numPositions, numPositionsSq;
	uint activationTimeLimit=numTimeSteps, maxSimultaneousDCs=numDCells, numWaves=waveTimes.size(), waveNumber=0, numWavesToRegen=0, posFailures=0, this_numActivated=0, this_numUniqueInteractions=0, this_numInteractions=0;
	vector<unsigned long int> waveTimeSteps(numWaves); for(uint i=0; i<numWaves; i++) waveTimeSteps[i]=waveTimes[i]/timeStep;
	// vector<double> nextArrivals(numWaves); for(uint i=0; i<numWaves; i++) nextArrivals[i]=waveTimeSteps[i];
	vector<uint> dCellsPresentPerWave(numWaves); for(uint i=0; i<numWaves; i++) dCellsPresentPerWave[i]=0;
	unsigned long int t=0;
	vector<double> radialVec, parallel, reflectedVec, position, velocity;
	vector<int> nearbyDCs; vector<int> *vecIt;
	double *cognateAgProbabilityTable;
	clock_t repeatStart, repeatEnd;

	//Parameters for linear code
	#ifdef __LINEAR_MOVEMENT__
		double entryTime, timeTaken, timePerCoordX, timePerCoordY, timePerCoordZ, sourceX, sourceY, sourceZ, sourceT, timeLastUpdated, tarx, tary, tarz, contactRegionClearanceTime=2*double(contactRadius)/(dCellVelocity*timeStep);
		int canActivate, stillMoving, discreteGridUpdatedThisTimestep;
		uint elementPosition;
		vector<double>::iterator cst_it;
	#endif
	
	//Declare variables for plotting, then send commands if needed
	PLOT_OPEN;

	//Find lookup table for the probability of activation given ratio of cognate antigen; make one if it is not present
	char CALTFilename[400]; snprintf(CALTFilename, sizeof(CALTFilename), "%s_%d_%d.dat", COGNATE_ANTIGEN_RATIO_LOOKUP_TABLE_PREFIX, numAntigenInContactArea, tCellActivationThreshold);
	if(fileExists(CALTFilename)){
		cout << "#Reading file " << CALTFilename << " for cognate antigen ratio probability table." << endl;
		int retVal;
		if((retVal=readLookupTable(CALTFilename, cognateAgPrecision, cognateAgProbabilityTable_size, &cognateAgProbabilityTable))) return retVal;
	}
	else{
		cout << "#Could not find file " << CALTFilename << ", generating cognate antigen ratio probability table." << endl;
		int retVal;
		if((retVal=generateLookupTable(CALTFilename, numAntigenOnDC, numAntigenInContactArea, tCellActivationThreshold))) return retVal;
		if((retVal=readLookupTable(CALTFilename, cognateAgPrecision, cognateAgProbabilityTable_size, &cognateAgProbabilityTable))) return retVal; //A waste of time, but the time involved is miniscule so I care not	
	}
	for(int i=0; i<cognateAgProbabilityTable_size; i++) if(cognateAgProbabilityTable[i]<PROBABILITY_TOLERANCE) minimumCognateAg=cognateAgPrecision*i; //Find also the cognate ag ratio corresponding to our effective zero probability

	//Calculate time that the LN cannot activate any T cells anymore.
	if(cogAgOnArrival>minimumCognateAg && minimumCognateAg!=0 && antigenDecayRate!=0)	activationTimeLimit = int(log(cogAgOnArrival/minimumCognateAg)/(antigenDecayRate*timeStep)); 
	else if(cogAgOnArrival<minimumCognateAg)		activationTimeLimit = 0;
	else 											activationTimeLimit = numTimeSteps;
	cout << "#Data_cognateAntigenThresholdAndTime: " << minimumCognateAg << " " << activationTimeLimit << "/" << numTimeSteps << endl;

	//If multiple doses are being administered, check if any of them will overlap. If so, the memory required to place all the DCs at once will be increased.
	uint maxOverlap=0;
	for(uint s1=0; s1<waveTimeSteps.size(); s1++){
		for(uint s2=s1+1; s2<waveTimeSteps.size(); s2++){
			if (waveTimeSteps[s2]-waveTimeSteps[s1] > activationTimeLimit) break;
			maxOverlap = (s2-s1>maxOverlap) ? s2-s1 : maxOverlap;
		}
	}
	maxSimultaneousDCs=(maxOverlap+1)*numDCells;

	//Create containers for our T cells and DCs
	vector<tCell> tCellList(numTCells);
	#ifdef __LINEAR_MOVEMENT__
		vector<dCell> dCellList(maxSimultaneousDCs+1, dCell(numTimeSteps));
	#else
		vector<dCell> dCellList(maxSimultaneousDCs);
	#endif

	//Create a discrete grid to reduce the search space for T cells looking for DCs. However, make this grid iteratively more coarse until the memory usage is less than 1GB.
	long int occupiedPositionsArraySize=0; double contractGridReductionFactor=2.0;//actually starts at 1.0 due to division
	do{
		//Reduce memory usage of grid.
		if(contractGridReductionFactor < MINIMUM_CONTACT_GRID_MEMORY_REDUCTION_FACTOR){
			cerr << "Could not produce a discrete position grid of small enough footprint. Check parameters." << endl
			<< "\nnumPositions: " << radius << "\ncontactRadius: " << contactRadius << "\nFull memory usage: " <<  10*(radius / contactRadius+1)*sizeof(int)*1E-6 << "MB"
			<< "\nMinmimum reduction factor: " << MINIMUM_CONTACT_GRID_MEMORY_REDUCTION_FACTOR << "\nMinmum memory usage: " << 10*sizeof(int)*occupiedPositionsArraySize*1E-6 << "MB" << endl;//The 10 estimates how many DCs might take a spot, and the overhead
			return 1;
		}
		else contractGridReductionFactor/=2;

		//Set number of positions and calculate memory usage.
		numPositions = 2*contractGridReductionFactor*radius / contactRadius+1; //e.g. radius = 4, contact radius = 2 -> there are 2*4/2 +1 = 5 positions per dimension, at x=0,2,4,-2,-4
		numPositionsSq = numPositions*numPositions;
		occupiedPositionsArraySize = numPositions*numPositions*numPositions;
	} while(occupiedPositionsArraySize*10*sizeof(int)>2E9);
	double cellSide=contactRadius/contractGridReductionFactor;

	//Allocate memory and initialise grids
	total_malloced+= (long long int)(occupiedPositionsArraySize*sizeof(int)); 
	printf("#Info: numPositions: %d; allocating %lli bytes for occupiedPositions using reduction factor %g; total %gMB\n", 
		numPositions, (long long int)(occupiedPositionsArraySize*sizeof(int)), contractGridReductionFactor, 1.0*total_malloced/1E6);
	vector<vector<int> > occupiedPositions(occupiedPositionsArraySize);

	//More containers for T cells.
	vector<int> cellMovementOrder(numTCells); 
	vector<double> freePathRemaining(numTCells);
	double timeStepsBetweenDCArrival = (numDCells>1) ? 1.0*(DCArrivalDuration)/timeStep/(numDCells-1) : 0;//e.g. with a time gap of 6 hours, 11 DCs will arrive at 0, 0.6, 1.2, ... 5.4, 6.0.
	vector<double> tCellInitialX(numTCells),tCellInitialY(numTCells),tCellInitialZ(numTCells);

	//Start random num generator
	mt19937_64 engine(seed); //Random number generator (Mersenne Twister) initialised with seed
	uniform_int_distribution<int> genrand_int(0,1); //generates 1 or 0
	uniform_real_distribution<double> genrand_uniform_posneg(-1,1); //generates doubles between -1 and 1
	uniform_real_distribution<double> genrand_uniform_pos(0,1);
	#ifdef _GAUSS_
		normal_distribution<double> genrand_t_velocity(tVelocityMean, tVelocityStDev);
	#else
		gamma_distribution<double> genrand_t_velocity(tGammaShape,tGammaScale); 
	#endif
	normal_distribution<double> genrand_t_freepath(freePathMean, freePathStDev);
	// exponential_distribution<double> genrand_exp((numDCells-1)/(DCArrivalDuration/timeStep)); //if DCArrivalDuration is 0, this behaves as expected: only produces variates equal to 0
	
	//Loop over "chunks": we will output data after each of these, in case of many repeats.
	for(int chunk=0; chunk<numChunks; chunk++){
		//Reinitialise moments once per chunk
		MOMENT_TM_INIT(numActivated);
		MOMENT_TM_INIT(tCellDisplacementFromStart);
		// MOMENT_TM_INIT(numInteractions);
		// MOMENT_TM_INIT(numUniqueInteractions);
		MOMENT_INIT(realTimeTaken);
		MOMENT_INIT(atLeastOneActivated);

		for(int repeat=0; repeat<numRepeats; repeat++){
			repeatStart=clock();
			if(numChunks>1 && numRepeats>1) {cerr << "#c=" << chunk << "/" << numChunks << ",";}
			if(numRepeats>1) {cerr << "r=" << repeat << "/" << numRepeats << "\t";}
			//Reinitialise number of activated t-cells and positions of DCs
			this_numActivated=0; this_numUniqueInteractions=0; this_numInteractions=0; waveNumber=0;
			if(repeat>0) for(int i=0; i<occupiedPositionsArraySize; i++) occupiedPositions[i].clear(); // = emptyVector; //clear() is marginally faster than the equality. This loop still makes it slow, however.

			//Initialise a vector of integers that we will use to randomise movement of the T-cells, and a vector that indicates how long a T cell will move before it "collides"
			cellMovementOrder.resize(numTCells);
			for(uint i=0; i<numTCells; i++) cellMovementOrder[i]=i; //No need to randomise for first use, since the cells themselves were randomly generated
			
			//Place cells at random in the sphere and give them a random number of antigen
			REGENERATE_CELL_POSITIONS_AND_ANTIGEN(0,int(maxSimultaneousDCs/numDCells)); //also sets dCellsPresent=0

			//Move cells until end of time
			for(t=0; t<numTimeSteps; t++){
				//Progress indicator
				// if(t%(numTimeSteps/100)==0) cerr << waveNumber << " " << t << "\t" << cellMovementOrder.size() << " + " << this_numActivated << " = " << numTCells << endl; 

				//Record the number of T-cells that are activated at this time step (here because "0" happens before any change, then "1" happens after the first change, ...)
				if(t%(numTimeSteps/numTimeMeasurements)==0){
					MOMENT_TM(numActivated, this_numActivated, t*numTimeMeasurements/numTimeSteps);
					// MOMENT_TM(numUniqueInteractions,this_numUniqueInteractions,t*numTimeMeasurements/numTimeSteps)
					// MOMENT_TM(numInteractions,this_numInteractions,t*numTimeMeasurements/numTimeSteps)
				}

				//Break if we have exhausted all of the t-cells or if the last wave of DCs are out of cognate antigen
				if(cellMovementOrder.size()==0 || t>waveTimeSteps.back()+activationTimeLimit){
					for(uint t2=t*numTimeMeasurements/numTimeSteps+1; t2<numTimeMeasurements; t2++){
						MOMENT_TM(numActivated, this_numActivated, t2);
						// MOMENT_TM(numUniqueInteractions, this_numUniqueInteractions, t2);
						// MOMENT_TM(numInteractions, this_numInteractions, t2);
					}
					cerr << "**Early exit at time: " << t << "/" << numTimeSteps << " ";
					if(t>activationTimeLimit) cerr << "> t*, ";
					break;
				}

				//Check if the current wave of DCs has run out of antigen or if 48 hours have passed
				if(t>waveTimeSteps[waveNumber]+activationTimeLimit && waveNumber+1<numWaves){
					numWavesToRegen=int((dCellsPresent-1)/numDCells)+1; //Needed if we are going to be skipping time. e.g. have only a single complete wave -> get 0+1. have an incomplete wave -> also get 0+1. have two simultaneous waves -> get 1+1.
					// cerr << numWavesToRegen << "=int(" << (dCellsPresent-1) << "/" << numDCells <<")+1" << endl; return 1;
					dCellsPresent-=dCellsPresentPerWave[waveNumber];
					dCellsPresentPerWave[waveNumber]=0;
					assert(numWavesToRegen<=numWaves);

					//Check consecutive waves until we've found we're not past the activation limit of one of them
					while(t>waveTimeSteps[waveNumber]+activationTimeLimit && waveNumber<numWaves) waveNumber++; 
					//If we haven't already begun this wave, increase time and regenerate all DCs
					if (waveNumber!=numWaves && t<waveTimeSteps[waveNumber]){ 
						for(uint t2=t*numTimeMeasurements/numTimeSteps+1; t2<=waveTimeSteps[waveNumber]*numTimeMeasurements/numTimeSteps; t2++){
							MOMENT_TM(numActivated, this_numActivated, t2);
							// MOMENT_TM(numUniqueInteractions, this_numUniqueInteractions, t2);
							// MOMENT_TM(numInteractions, this_numInteractions, t2);
						}
						for(int i=0; i<occupiedPositionsArraySize; i++) occupiedPositions[i].clear();
						t=waveTimeSteps[waveNumber]-1; //-1 because it will suffer a t++ at the end of this loop
						REGENERATE_CELL_POSITIONS_AND_ANTIGEN((t+1), numWavesToRegen); //Regen DC positions and randomise T-cells to avoid them
					}
					//Even if we are continuing because of the presence of a second wave, there is a chance the "finished" wave still has antigen, i.e. 48 hours was the limit rather than the antigen decay rate. Thus, we remove the first numDCells elements from the vector and subtract the difference from all numbers on the discrete grid(!).
					else{
						for(int i=0; i<occupiedPositionsArraySize; i++) occupiedPositions[i].clear();
						#ifdef __LINEAR_MOVEMENT__
							vector<dCell>(dCellList.begin()+numDCells,dCellList.end()).swap(dCellList); //Form a temporary vector which is missing the first numDCells elements and swap with our own
							dCellList.resize(dCellList.size()+numDCells, dCell(numTimeSteps)); //add back empty elements onto the end
							//Place DCs back onto discrete grid
							for(int i=1; i<=dCellsPresent; i++){
								canActivate=!dCellList[i].update_num_cog_antigen(t, timeStep, antigenDecayRate, cognateAgPrecision, cognateAgProbabilityTable);
								//Work out if the DC is still moving
								sourceT	= dCellList[i].get_entry_time(); velocity= dCellList[i].get_vel();
								sourceX=dCellList[i].get_source_x(); sourceY=dCellList[i].get_source_y(); sourceZ=dCellList[i].get_source_z();
								x=dCellList[i].get_dest_x(); y=dCellList[i].get_dest_y(); z=dCellList[i].get_dest_z();
								timeTaken = MAGNITUDE( (x-sourceX),(y-sourceY),(z-sourceZ) )/(timeStep*dCellVelocity);
								
								//If it is moving, then place down the path from its current position to its destination
								if(t-dCellList[i].get_entry_time()<timeTaken) { 
									sourceX+=velocity[0]*(t-sourceT)*timeStep; sourceY+=velocity[1]*(t-sourceT)*timeStep; sourceZ+=velocity[2]*(t-sourceT)*timeStep;
									dCellList[i].set_position(sourceX,sourceY,sourceZ); //dCellList[i].set_source(sourceX,sourceY,sourceZ); dCellList[i].set_entry_time(t);
									if(canActivate) {PLACE_DC_PATH_ON_DISCRETE_GRID(i,sourceX,sourceY,sourceZ, x,y,z, t);}
								}
								//if not, place it at its destination
								else if(canActivate){
									SET_COORDINATES(x,y,z, coordX,coordY,coordZ);
									PLACE_DC_ON_DISCRETE_GRID(coordX,coordY,coordZ,i);
								}
							}
						#else
							vector<dCell>(dCellList.begin()+numDCells,dCellList.end()).swap(dCellList); //Form a temporary vector which is missing the first numDCells elements and swap with our own
							dCellList.resize(dCellList.size()+numDCells); //add back empty elements onto the end
							for(int i=0; i<dCellsPresent; i++){
								SET_COORDINATES(dCellList[i].get_x(),dCellList[i].get_y(),dCellList[i].get_z(), coordX,coordY,coordZ);
								if(dCellList[i].get_probActivation()>0) {PLACE_DC_ON_DISCRETE_GRID(coordX, coordY, coordZ, i);}
							}
						#endif
					}
				}
				
				//Spawn new DCs
				for(uint w=waveNumber; w<numWaves; w++){ //waveNumber is the current DC wave
					while(t>= waveTimeSteps[w] && t*timeStep<=waveTimes[w]+DCArrivalDuration && DCArrivalDuration!=0 && t-waveTimeSteps[w]>=dCellsPresentPerWave[w]*timeStepsBetweenDCArrival){
					// while(t>=nextArrivals[w] && dCellsPresentPerWave[w]<numDCells){ //uncomment this line and the nextArrivals[w]+ line below for random DC arrival
						dCellNum=(w-waveNumber)*numDCells+dCellsPresentPerWave[w];
						#ifdef __LINEAR_MOVEMENT__
							dCellNum++; //because index 0 is not valid in this version of the code
							dCellList[dCellNum].set_timeAntigenUpdated(waveTimeSteps[w]); //This needs to be done for simultaneous waves arriving, because there is no efficient apriori way of finding out which waves will be simultaneous
							x=dCellList[dCellNum].get_dest_x(); y=dCellList[dCellNum].get_dest_y(); z=dCellList[dCellNum].get_dest_z(); //"1+" so that the number "0" is not used, because (0 == -0) being TRUE is a problem
							sourceX=dCellList[dCellNum].get_x(); sourceY=dCellList[dCellNum].get_y(); sourceZ=dCellList[dCellNum].get_z(); 
							if(dCellList[dCellNum].get_probActivation()>0) {
								dCellList[dCellNum].set_entry_time(t); dCellList[dCellNum].set_timeMovementUpdated(t);
								PLACE_DC_PATH_ON_DISCRETE_GRID(dCellNum,sourceX,sourceY,sourceZ, x,y,z, t);
							}
						#else
							dCellList[dCellNum].set_timeAntigenUpdated(waveTimeSteps[w]);
							x=dCellList[dCellNum].get_x(); y=dCellList[dCellNum].get_y(); z=dCellList[dCellNum].get_z(); //Note that the non-vector version of dCellNum is used here, because we have already ensured that each cell should get a unique position
							SET_COORDINATES(x,y,z,coordX,coordY,coordZ);
							if(dCellList[dCellNum].get_probActivation()>0) {PLACE_DC_ON_DISCRETE_GRID(coordX, coordY, coordZ, dCellNum);}
						#endif
						dCellsPresent++; dCellsPresentPerWave[w]++;
						// nextArrivals[w]+=genrand_exp(engine);
					}
				}

				//Shuffle the list of cells so that we move them in a random order
				// shuffle(cellMovementOrder.begin(),cellMovementOrder.end(),engine); //--This is not necessary because the T-cells are simulated independently of each other.
				
				for(auto iter = cellMovementOrder.begin(); iter != cellMovementOrder.end(); iter++){
					tCellNum = *iter;
					//Get current position and record how far it has gone from the beginning
					x=tCellList[tCellNum].get_x(); y=tCellList[tCellNum].get_y(); z=tCellList[tCellNum].get_z(); //position = tCellList[tCellNum].get_pos(); x=position[0]; y=position[1]; z=position[2];
					if(t%(numTimeSteps/numTimeMeasurements)==0){
						dist = MAGNITUDE( x-tCellInitialX[tCellNum],y-tCellInitialY[tCellNum],z-tCellInitialZ[tCellNum] );
						MOMENT_TM(tCellDisplacementFromStart, dist, t*numTimeMeasurements/numTimeSteps);
					}

					//Move the cell: get current velocity & add it to position
					velocity = tCellList[tCellNum].get_vel();
					dx=velocity[0]*timeStep; dy=velocity[1]*timeStep; dz=velocity[2]*timeStep;
					newx=x+dx; newy=y+dy; newz=z+dz;
					freePathRemaining[tCellNum] -= MAGNITUDE(dx,dy,dz);	
				
					//Check if we have tried to left the sphere or the T cell has reached the end of its mean free path
					if(OUTSIDE_SPHERE(newx,newy,newz)) {
						//Regenerate velocity to move the T cell away from the sphere surface.
						vmag = genrand_t_velocity(engine); theta=acos(genrand_uniform_pos(engine)); phi=6.283185307*genrand_uniform_pos(engine); //random velocity; theta[0,pi/2] and phi[0,2pi]
						magsq = MAGNITUDE(newx,newy,newz); //Get magnitude of position vector to normalise
						x/=magsq; y/=magsq; z/=magsq; //Set as a unit vector
						velocity[0]=-vmag*x; velocity[1]=-vmag*y; velocity[2]=-vmag*z; //Set velocity direction opposite to the position vector (i.e. pointing to centre of the sphere)
						arbitraryAxisRotation(z, y, -x, velocity[0], velocity[1], velocity[2], theta); //Choosing the radial axis as the z axis, this performs a rotation by theta about the x axis.
						arbitraryAxisRotation(x, y, z,  velocity[0], velocity[1], velocity[2], phi); //Using the same axes as before, this performs a rotation by phi about the z axis.
						tCellList[tCellNum].set_velocity(velocity[0], velocity[1], velocity[2]);
						freePathRemaining[tCellNum] = genrand_t_freepath(engine); //Set how long until the particle stops and must choose a new velocity
						
						//Finally, reverse the previous step we tried to make and take a new one
						newx+= velocity[0]*timeStep-dx; newy+= velocity[1]*timeStep-dy; newz+= velocity[2]*timeStep-dz; 
					}
					else if(freePathRemaining[tCellNum]<=0) {//Check if the cell has exceeded the free path of its previous velocity vector and needs a new one generated
						vmag = genrand_t_velocity(engine); theta=acos(2*genrand_uniform_pos(engine)-1); phi=6.283185307*genrand_uniform_pos(engine); //theta between 0 and pi, phi between 0 and 2pi
						tCellList[tCellNum].set_velocity(vmag*sin(theta)*cos(phi), vmag*sin(theta)*sin(phi), vmag*cos(theta)); //Set velocity
						freePathRemaining[tCellNum] = genrand_t_freepath(engine); //Set how long until the particle stops and must choose a new velocity
					}

					//Check for DCs nearby to activate T-cell and either update position or remove from simulation
					inContact=0;
					CHECK_CONTACT_WITH_DENDRITES_T(newx,newy,newz,inContact);		
					if(inContact==1){//If this T cell encountered some dendrites, check to see if the interaction was successful
						//First, update the number of antigen on this DC, as some will have unbound since we last saw the DC.
						dCellList[dCellNum].update_num_cog_antigen(t, timeStep, antigenDecayRate, cognateAgPrecision, cognateAgProbabilityTable);
						SET_COORDINATES(dendx,dendy,dendz, dcoordX, dcoordY, dcoordZ);
						
						// this_numInteractions++;//NOTE: This will be an underestimate because we remove DCs from the discrete grid, unless all DCs are removed at once!
						// cerr << t*timeStep << " " << (t + firstDCArrival)*timeStep << " " << tCellNum << " " << dCellNum << " " << dCellList[dCellNum].get_probActivation() << endl;
						if (1==tCellList[tCellNum].increment_num_interactions()) {this_numUniqueInteractions++;} //increment_num_interactions returns the new value after incrementation
						
						if(dCellList[dCellNum].cannot_activate_t_cells()){
							REMOVE_DC_FROM_DISCRETE_GRID(dcoordX, dcoordY, dcoordZ, dCellNum); //If there is no longer enough cognate Ag, there is little point simulating the DC anymore...
						}

						//Now, see if the T cell has been activated
						else if(genrand_uniform_pos(engine)<dCellList[dCellNum].get_probActivation()){
							//Activated: we no longer want to track this cell. Erase it and pull the iterator back to not accidentally skip the next T cell
							cellMovementOrder.erase(cellMovementOrder.begin()+distance(cellMovementOrder.begin(), iter)); iter--;	
							//Finally, record date for analysis
							this_numActivated++;
							if(timeUntilFirstActivation[chunk*numRepeats+repeat]==maxTime && this_numActivated>=1)						timeUntilFirstActivation[chunk*numRepeats+repeat] = t*timeStep;
							if(timeUntil25pcActivation[chunk*numRepeats+repeat]==maxTime  && this_numActivated>=int(0.25*numTCells))	timeUntil25pcActivation[chunk*numRepeats+repeat] = t*timeStep;
							if(timeUntil50pcActivation[chunk*numRepeats+repeat]==maxTime  && this_numActivated>=int(0.5*numTCells)) 	timeUntil50pcActivation[chunk*numRepeats+repeat] = t*timeStep;
							if(timeUntil75pcActivation[chunk*numRepeats+repeat]==maxTime  && this_numActivated>=int(0.75*numTCells))	timeUntil75pcActivation[chunk*numRepeats+repeat] = t*timeStep;
							if(timeUntil90pcActivation[chunk*numRepeats+repeat]==maxTime  && this_numActivated>=int(0.9*numTCells))		timeUntil90pcActivation[chunk*numRepeats+repeat] = t*timeStep;
							if(this_numActivated==numTCells)																			timeUntil100pcActivation[chunk*numRepeats+repeat] = t*timeStep;
						}
						else{ //Activation failed - mark this T cell-DC pair so that they won't repeatedly try to interact.
							tCellList[tCellNum].set_failed_interaction_ID(dCellNum);
							//Regenerate velocity to move the T cell away from the DC after interaction. 
							vmag = genrand_t_velocity(engine); theta=acos(genrand_uniform_pos(engine)); phi=6.283185307*genrand_uniform_pos(engine); //random velocity; theta[0,pi/2] and phi[0,2pi]
							x = newx-dendx; y = newy-dendy; z = newz-dendz; //Defines the position vector from DC to T cell. 
							magsq = MAGNITUDE(x,y,z); //Get magnitude (not squared) to normalise
							x/=magsq; y/=magsq; z/=magsq; //Set as a unit vector
							velocity[0]=vmag*x; velocity[1]=vmag*y; velocity[2]=vmag*z; //Magnitude of velocity, currently parallel to position vector
							arbitraryAxisRotation(z, y, -x, velocity[0], velocity[1], velocity[2], theta); //Setting position vector as z axis, this performs a rotation by theta about the x axis.
							arbitraryAxisRotation(x, y, z,  velocity[0], velocity[1], velocity[2], phi); //Using the same axes as before, this performs a rotation by phi about the z axis.
							tCellList[tCellNum].set_velocity(velocity[0], velocity[1], velocity[2]);
							freePathRemaining[tCellNum] = genrand_t_freepath(engine); //Set how long until the particle stops and must choose a new velocity
						}
					}
					else /*if(inContact==0)*/{ //If it is not 0 - although this case should never happen!
						// tCellList[tCellNum].unset_failed_interaction_ID(); //Get rid of the failed interaction ID if it is there
					}

					//Update cell position (including for ones just removed)
					tCellList[tCellNum].set_position(newx, newy, newz);		

				}//End of T-cell loop
				if(plot && t%time_steps_per_graphical_update==0){PLOT_CELLS;}

			}//End of time loop

			//Record data for final time
			cerr << "numActivated=" << this_numActivated << "/" << numTCells << endl;
			MOMENT_TM(numActivated,this_numActivated,numTimeMeasurements); 
			// MOMENT_TM(numUniqueInteractions, this_numUniqueInteractions, numTimeMeasurements);
			// MOMENT_TM(numInteractions, this_numInteractions, numTimeMeasurements);
			numActivationsPerRepeat[chunk*numRepeats+repeat]=this_numActivated;
			MOMENT(atLeastOneActivated, (this_numActivated>0));
			repeatEnd=clock(); MOMENT(realTimeTaken, 1.*(repeatEnd-repeatStart)/(CLOCKS_PER_SEC));
		}//End of repeat loop

		OUTPUT_MOMENT_DATA; //If in C++, this goes to stdout
	} //End of chunk loop

	//Free all memory we took
	free(cognateAgProbabilityTable);
	return 0;
}