/* utils.c */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* utils.c: utils for niiT1vfa                                               */
/*                                                                           */
/* Copyright (C) 2014 Paul Kinchesh                                          */
/*                                                                           */
/* niiT1vfa is free software: you can redistribute it and/or modify          */
/* it under the terms of the GNU General Public License as published by      */
/* the Free Software Foundation, either version 3 of the License, or         */
/* (at your option) any later version.                                       */
/*                                                                           */
/* niiT1vfa is distributed in the hope that it will be useful,               */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of            */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the              */
/* GNU General Public License for more details.                              */
/*                                                                           */
/* You should have received a copy of the GNU General Public License         */
/* along with niiT1vfa. If not, see <http://www.gnu.org/licenses/>.          */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/**/

#include "niiT1vfa.h"

/*------------------------*/
/*---- Find threshold ----*/
/*------------------------*/
/* Method gifted in imagemath by Maj Hedehus (Genentech, Varian, Stanford) */
float threshold(struct data *d,int vol) 
{
  float meanN,meanS;
  float thr0,thr1;
  long offset,i;

  /* Check vol is sensible */
  if (vol>d->nvols-1 || vol<0) {
    fprintf(stderr,"\n%s: %s()\n",__FILE__,__FUNCTION__);
    fprintf(stderr,"  Specified threshold volume should be >= 0 and < %d\n\n",d->nvols);
    fprintf(stderr,"  Aborting ...\n\n");
    fflush(stderr);
    exit(1);
  }

  /* For initial noise estimate, assume that the volume edges are noise */        
  meanN = 0;
  for (i=0;i<d->nfh.dim[1];i++) meanN += d->M[vol][i];
  offset=(d->nfh.dim[2]-1)*d->nfh.dim[1];
  for (i=0;i<d->nfh.dim[1];i++) meanN += d->M[vol][offset+i];
  offset=(d->nfh.dim[3]-1)*d->nfh.dim[2]*d->nfh.dim[1];
  for (i=0;i<d->nfh.dim[1];i++) meanN += d->M[vol][offset+i];
  offset+=(d->nfh.dim[2]-1)*d->nfh.dim[1];
  for (i=0;i<d->nfh.dim[1];i++) meanN += d->M[vol][offset+i];
  meanN /= (4*d->nfh.dim[1]);

  /* Initial rough estimate of mean signal intensity */
  meanS = 0;
  for (i=0;i<d->nvoxels;i++) meanS += d->M[vol][i];
  meanS /= d->nvoxels;

#ifdef DEBUG
  fprintf(stdout,"\n%s: %s()\n",__FILE__,__FUNCTION__);
  fprintf(stdout,"  Initial noise estimate = %f\n",meanN);
  fprintf(stdout,"  Initial signal estimate = %f\n",meanS);
#endif

  /* Initial guesses for threshold between the two classes */
  thr0 = (meanS+meanN)/2.0;
  thr1 = getnewthr(d,vol,thr0);

  /* Now iterate */
  while (fabs((thr1-thr0)/(thr1+thr0)) > 1e-5) { /* 1e-5 seems reasonable */
    thr0 = thr1;
    thr1 = getnewthr(d,vol,thr0);
  }

  /* Typically threshold is overestimated */
  thr1 /=3.0;

#ifdef DEBUG
  fprintf(stdout,"  Threshold typically overestimated so reduced 3-fold to %f\n",thr1);
#endif

  return(thr1);
}

/*----------------------------*/
/*---- Find new threshold ----*/
/*----------------------------*/
float getnewthr(struct data *d,int vol,float thr) 
{
  float mean1,mean2,M;
  long  n1,n2,i;

  mean1=mean2=0;
  n1=n2=0;
  for (i=0;i<d->nvoxels;i++) {
    M = d->M[vol][i];
    if (M < thr) {
      mean1 += M;
      n1++;
    }
    else {
      mean2 += M;
      n2++;
    }
  }
  mean1 /= n1;
  mean2 /= n2;

  /* Next guess for threshold */
  thr = (mean1+mean2)/2;

#ifdef DEBUG
  fprintf(stdout,"  Iterating to estimate threshold ... %f\n",thr);
#endif

  return(thr);
}

/*-----------------------*/
/*---- Out of memory ----*/
/*-----------------------*/
int nomem(char *file,const char *function,int line)
{
  fprintf(stderr,"\n%s: %s() line %d\n",file,function,line);
  fprintf(stderr,"  Insufficient memory for program!\n\n");
  fflush(stderr);
  exit(1);
}

