/* niiT1vfa.h */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* niiT1vfa.h: Variable Flip Angle T1 fitting header                         */
/*                                                                           */
/* Copyright (C) 2014 Paul Kinchesh                                          */
/*                                                                           */
/* niiT1vfa is free software: you can redistribute it and/or modify          */
/* it under the terms of the GNU General Public License as published by      */
/* the Free Software Foundation, either version 3 of the License, or         */
/* (at your option) any later version.                                       */
/*                                                                           */
/* niiT1vfa is distributed in the hope that it will be useful,               */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of            */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the              */
/* GNU General Public License for more details.                              */
/*                                                                           */
/* You should have received a copy of the GNU General Public License         */
/* along with niiT1vfa. If not, see <http://www.gnu.org/licenses/>.          */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/**/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Usage: niiT1vfa [-f FA1,..,FAn] [-r TR] [-m maxT1] [-t thr]               */
/*                 [-a AFI.nii] [-p AFIflip] [-s] VFA.nii                    */
/*          VFA.nii is the file containing the VFA data                      */
/*          Options:                                                         */
/*            [-f FA1,FA2,..,FAn] specify flip angles                        */
/*            [-r TR]             specify TR/s (default 0.0028 s)            */
/*            [-m maxT1]          maximum T1/s (default 10.0 s)              */
/*            [-t thr]            threshold (default automatic)              */
/*            [-a AFI.nii]        use AFI FA file to correct prescribed FAs  */
/*            [-p AFIflip]        prescribed AFI flip/deg (default 64 deg)   */
/*            [-s]                generate T1 & iteration stats              */
/*          Output:                                                          */
/*            T1_VFA.nii          T1 estimate                                */
/*            M0_VFA.nii          M0* estimate, M0* = Mo.exp(-TE/T2*)        */
/*            NIT_VFA.nii         Number of iterations (-s option only)      */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/**/

#ifndef _NIIT1VFA_HEADER_
#define _NIIT1VFA_HEADER_

/*----------------------------------*/
/*---- Include standard headers ----*/
/*----------------------------------*/
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <stddef.h>
#include <sys/time.h>

/*-----------------------------*/
/*---- Include GSL headers ----*/
/*-----------------------------*/
#include <gsl/gsl_blas.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_eigen.h>
#include <gsl/gsl_linalg.h>
#include <gsl/gsl_multifit_nlin.h>

/*--------------------------------*/
/*---- Include NIfTI-1 header ----*/
/*--------------------------------*/
#include "nifti1.h"

#define TRUE       1
#define FALSE      0

#define  MAX_ITER   100   /* Max iterations to fit T1 */
#define  ITER_LIMIT 1e-6  /* Limit for stopping fit iterations */

/*--------------------*/
/*---- Data types ----*/
/*--------------------*/
#define NONE       0  /* No data */
#define T1         1  /* T1 data */
#define M0         2  /* M0 data */
#define SV         3  /* Single Volume data */
#define VFA        4  /* VFA data */
#define AFI        5  /* AFI data */

/*-----------------------------------------------------------*/
/*---- So we can simply include niiT1vfa.h in niiT1vfa.c ----*/
/*-----------------------------------------------------------*/
#ifdef LOCAL
#define EXTERN
#else
#define EXTERN extern
#endif

/*--------------------------------*/
/*---- Structure to hold data ----*/
/*--------------------------------*/
struct data 
{
  struct nifti_1_header nfh; /* NIfTI-1 header */
  int type;                  /* data type */
  long nvoxels;              /* # voxels in each volume */
  int nvols;                 /* # volumes */
  float  **M;                /* data */
};

/*------------------------------------*/
/*---- Structure to hold fit data ----*/
/*------------------------------------*/
struct fitdata
{
  size_t n;  /* # FAs */
  double *F; /* FAs */
  double *S; /* data */
};

/*--------------------------------------*/
/*---- Structure to hold input data ----*/
/*--------------------------------------*/
struct inputdata 
{
  char   *VFAfile; /* VFA nii file */
  int    nFAs;     /* # FAs */
  double *FA;      /* FAs */
  double TR;       /* TR */
  double maxT1;    /* max T1 */
  double thr;      /* threshold */
  char   *AFIfile; /* AFI nii file */
  double AFIflip;  /* AFI flip */
};

/*-------------------*/
/*---- Functions ----*/
/*-------------------*/
/* niiT1vfa.c */
int T1_f(const gsl_vector *x,void *data,gsl_vector *f);
int T1_df(const gsl_vector *x,void *data,gsl_matrix *J);
int T1_fdf(const gsl_vector *x,void *data,gsl_vector *f,gsl_matrix *J);
void print_state(int iter,gsl_multifit_fdfsolver *s,double TR);
int setniidata(struct data *d1,struct data *d2);
int loadniidata(struct data *d,struct inputdata *in);
void getinput(struct inputdata *in,int argc,char *argv[]);
int usage();
/* nifti1.c */
int writeniidata(struct data *d,char *file);
int readniidata(struct data *d,char *file);
int freeniidata(struct data *d);
void copynfh(struct nifti_1_header *nfh1,struct nifti_1_header *nfh2);
int diffvolsize(struct data *d1,struct data *d2);
/* utils.c */
float threshold(struct data *d,int vol);
float getnewthr(struct data *d,int vol,float thr);
int nomem(char *file,const char *function,int line);

#endif /* _NIIT1VFA_HEADER_ */

