/* niiT1vfa.c */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* niiT1vfa.c: Variable Flip Angle T1 fitting                                */
/*                                                                           */
/* Copyright (C) 2014 Paul Kinchesh                                          */
/*                                                                           */
/* niiT1vfa is free software: you can redistribute it and/or modify          */
/* it under the terms of the GNU General Public License as published by      */
/* the Free Software Foundation, either version 3 of the License, or         */
/* (at your option) any later version.                                       */
/*                                                                           */
/* niiT1vfa is distributed in the hope that it will be useful,               */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of            */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the              */
/* GNU General Public License for more details.                              */
/*                                                                           */
/* You should have received a copy of the GNU General Public License         */
/* along with niiT1vfa. If not, see <http://www.gnu.org/licenses/>.          */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/**/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Usage: niiT1vfa [-f FA1,..,FAn] [-r TR] [-m maxT1] [-t thr]               */
/*                 [-a AFI.nii] [-p AFIflip] [-s] VFA.nii                    */
/*          VFA.nii is the file containing the VFA data                      */
/*          Options:                                                         */
/*            [-f FA1,FA2,..,FAn] specify flip angles                        */
/*            [-r TR]             specify TR/s (default 0.0028 s)            */
/*            [-m maxT1]          maximum T1/s (default 10.0 s)              */
/*            [-t thr]            threshold (default automatic)              */
/*            [-a AFI.nii]        use AFI FA file to correct prescribed FAs  */
/*            [-p AFIflip]        prescribed AFI flip/deg (default 64 deg)   */
/*            [-s]                generate T1 & iteration stats              */
/*          Output:                                                          */
/*            T1_VFA.nii          T1 estimate                                */
/*            M0_VFA.nii          M0* estimate, M0* = Mo.exp(-TE/T2*)        */
/*            NIT_VFA.nii         Number of iterations (-s option only)      */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/**/

#define LOCAL         /* EXTERN globals will not be 'extern' */
#include "niiT1vfa.h" /* headers, structure and funtion defines */

static int T1stats=FALSE;

int main(int argc,char *argv[])
{
  /* Get arguments from command line */
  struct inputdata input;
  getinput(&input,argc,argv);

  /* Load input NIfTI data */
  struct data vfadata,afidata;
  vfadata.type=VFA;
  loadniidata(&vfadata,&input);
  afidata.type=AFI;
  loadniidata(&afidata,&input);

  /* Check AFI data is compatible */
  if (afidata.M != NULL) {
    if (diffvolsize(&vfadata,&afidata)) {
      fprintf(stderr,"\n%s: %s()\n",__FILE__,__FUNCTION__);
      fprintf(stderr,"  AFI data in %s appears to be incompatible\n\n",input.AFIfile);
      fprintf(stderr,"  Aborting ...\n\n");
      fflush(stderr);
      exit(1);
    }
    #ifdef DEBUG
      fprintf(stdout,"\n%s: %s()\n",__FILE__,__FUNCTION__);
      fprintf(stdout,"  AFI data is compatible\n");
    #endif
  }

  /* Check input FAs */
  if (input.nFAs != vfadata.nvols) {
    fprintf(stderr,"\n%s: %s()\n",__FILE__,__FUNCTION__);
    fprintf(stderr,"  # FAs appears to be incompatible with VFA data in %s \n\n",input.VFAfile);
    fprintf(stderr,"  Aborting ...\n\n");
    fflush(stderr);
    exit(1);
  }

  /* Set header & storage for output NIfTI data */
  struct data T1data,M0data,NITdata;
  T1data.type=T1;
  setniidata(&T1data,&vfadata);
  M0data.type=M0;
  setniidata(&M0data,&vfadata);
  if (T1stats) {
    NITdata.type=SV;
    setniidata(&NITdata,&vfadata);
  }
  /* Get threshold from first VFA volume */
  float thr;
  if (input.thr>0.0) thr=input.thr;
  else thr=threshold(&vfadata,0);

  /* Nonlinear Least-Squares Fit to get T1 */
  /* Specify # observations n and # fit parameters p (n must be >= p) */
  const size_t n=vfadata.nvols; /* # FAs */
  const size_t p=2;             /* # fit parameters, T1 and M0 */

  /* Set Levenberg-Marquardt derivative solver type which performs
     a QR decomposition of the Jacobian matrix of derivatives */
  const gsl_multifit_fdfsolver_type *T;
  T = gsl_multifit_fdfsolver_lmsder;

  /* Set derivative solver s for type T with n observations and p parameters */
  gsl_multifit_fdfsolver *s;
  s = gsl_multifit_fdfsolver_alloc(T,n,p);

  /* Storage for signals S and flips F to fit and the nominal FAs */
  double S[n],F[n],FA[n];

  /* Set nominal FAs (and F) in radians */
  long i;
  for (i=0;i<n;i++) {
    FA[i] = (double)input.FA[i]*M_PI/180.0;
    F[i] = FA[i];
  }

  /* Structure for the fit data */
  struct fitdata fd = {n,F,S};

  /* The gsl_multifit_function_fdf type defines a general system of 
     functions with arbitrary parameters and the corresponding 
     Jacobian matrix of derivatives */
  gsl_multifit_function_fdf f;
  f.f = &T1_f;      /* function to calculate the signal for each FA given T1 and M0 */
  f.df = &T1_df;    /* function to calculate the n*p Jacobian matrix of derivatives */
  f.fdf = &T1_fdf;  /* function to set the values of f and df at the same time for speed */
  f.n = n;          /* the number of components of the vector f */
  f.p = p;          /* the number of independent variables */
  f.params = &fd;   /* pointer to the arbitrary parameters of the function */

  /* Storage for initial estimate */
  double x_init[p];
  gsl_vector_view x;

  /* Loop over voxels to fit data */
  int status,imax;
  size_t iter=0;
  double Smax,sm,cm,E1;
  long j;
  for (j=0;j<vfadata.nvoxels;j++) {
    if (vfadata.M[0][j]>thr) {
      /* Set fit data and initial estimate */
      Smax=0.0; imax=0;
      for (i=0;i<n;i++) {
        S[i] = (double)vfadata.M[i][j];
        if (S[i]>Smax) { Smax=S[i]; imax=i; }
      }
      /* Correct FAs with AFI scan if available */
      if (afidata.M != NULL) {
        for (i=0;i<n;i++) F[i]=FA[i]*afidata.M[0][j]/input.AFIflip;
      }
      /* Set initial estimate */
      sm=sin(F[imax]);
      cm=cos(F[imax]);
      x_init[0] = S[imax]*(1-cm*cm)/((1-cm)*sm);  /* S0 */
      x_init[1] = cm;                             /* E1 = cos(ErnstAngle) */
      x = gsl_vector_view_array(x_init,p);     
      /* Set solver s to use the function f and the initial estimate x */
      gsl_multifit_fdfsolver_set(s,&f,&x.vector);
      /* Now iterate to fit T1 and M0 */
      iter=0;
      do {
        iter++;
        status = gsl_multifit_fdfsolver_iterate(s);
        if (status) break;
        /* print_state(iter,s,input.TR); // inspect results */
        status = gsl_multifit_test_delta(s->dx,s->x,ITER_LIMIT,ITER_LIMIT);
      } while (status == GSL_CONTINUE && iter < MAX_ITER);
      /* Store the results */
      E1 = gsl_vector_get(s->x,1); 
      T1data.M[0][j] = (float)-input.TR/log(E1);
      if (T1data.M[0][j]>input.maxT1) {
        T1data.M[0][j] = 0.0;
        M0data.M[0][j] = 0.0;
      } else {
        T1data.M[0][j] *= 1000.0; /* T1 in ms */
        M0data.M[0][j] = (float)gsl_vector_get(s->x,0); /* M0 */
      }
      if (T1stats) { /* for analysis of iterations */
        if (T1data.M[0][j]>0.0) NITdata.M[0][j] = (float)iter;
        else NITdata.M[0][j] = 0.0;
      }
    } else {
      T1data.M[0][j] = 0.0;
      M0data.M[0][j] = 0.0;
      if (T1stats) NITdata.M[0][j] = 0.0;
    }
  }
  gsl_multifit_fdfsolver_free(s);

  /* Output the data */
  char filename[MAXPATHLEN];
  sprintf(filename,"T1_%s",input.VFAfile);
  writeniidata(&T1data,filename);
  if (T1stats) {
    double t=0.0,t2=0.0,nt=0.0,besselcor;
    for (i=0;i<T1data.nvoxels;i++) {
      if (T1data.M[0][i]>0.0) {
        t+=T1data.M[0][i];
        t2+=T1data.M[0][i]*T1data.M[0][i];
        nt++;
      }
    }
    t /=nt;
    t2 /=nt;
    besselcor=sqrt(nt/(nt-1));
    fprintf(stdout,"\n%s: %s()\n",__FILE__,__FUNCTION__);
    fprintf(stdout,"  %d voxel fits, T1 (mean +/-stdev) = %lf +/- %lf ms\n",(int)nt,t,besselcor*sqrt(t2-t*t));
  }
  freeniidata(&T1data);
  sprintf(filename,"M0_%s",input.VFAfile);
  writeniidata(&M0data,filename);
  freeniidata(&M0data);
  if (T1stats) {
    double t=0.0,t2=0.0,nt=0.0,besselcor;
    for (i=0;i<NITdata.nvoxels;i++) {
      if (NITdata.M[0][i]>0.0) {
        t+=NITdata.M[0][i];
        t2+=NITdata.M[0][i]*NITdata.M[0][i];
        nt++;
      }
    }
    t /=nt;
    t2 /=nt;
    besselcor=sqrt(nt/(nt-1));
    fprintf(stdout,"\n%s: %s()\n",__FILE__,__FUNCTION__);
    fprintf(stdout,"  %d voxel fits, # iterations (mean +/-stdev) = %lf +/- %lf\n",(int)nt,t,besselcor*sqrt(t2-t*t));
    sprintf(filename,"NIT_%s",input.VFAfile);
    writeniidata(&NITdata,filename);
    freeniidata(&NITdata);
  }
  exit(0);
}

/*------------------------------------------------------*/
/*---- Calculate signal for each FA given T1 and M0 ----*/
/*------------------------------------------------------*/
int T1_f(const gsl_vector *x,void *data,gsl_vector *f) 
{
  int     n  = ((struct fitdata *)data)->n;
  double *S  = ((struct fitdata *)data)->S;
  double *F =  ((struct fitdata *)data)->F;

  /* Model S = S0*sin(F)*(1-E1)/(1-E1*cos(F))
       S:  Signal
       S0: Equilibrium magnetization
       F:  Flip
       E1: exp(-TR/T1)
     We array F, measure S and fit S0 and E1
  */

  double S0,E1,Scalc,sF,cF;
  int i;

  S0 = gsl_vector_get(x,0);
  E1 = gsl_vector_get(x,1);

  for (i=0;i<n;i++) {
    sF = sin(F[i]);
    cF = cos(F[i]);
    Scalc = S0*sF*(1-E1)/(1-E1*cF);
    gsl_vector_set(f,i,Scalc-S[i]);
  }
  return GSL_SUCCESS;
}

/*------------------------------------------------------------------*/
/*---- Calculate Jacobian matrix of derivatives given T1 and M0 ----*/
/*------------------------------------------------------------------*/
int T1_df(const gsl_vector *x,void *data,gsl_matrix *J)
{
  int     n  = ((struct fitdata *)data)->n;
  double *F =  ((struct fitdata *)data)->F;

  double S0,E1,sF,cF;
  int i;

  S0 = gsl_vector_get(x,0);
  E1 = gsl_vector_get(x,1);

  for (i=0;i<n;i++) {
    sF = sin(F[i]);
    cF = cos(F[i]);
    gsl_matrix_set(J,i,0,sF*(1-E1)/(1-E1*cF));                 /* d/dS0 (S) */
    gsl_matrix_set(J,i,1,S0*sF*(cF-1)/((1-E1*cF)*(1-E1*cF)));  /* d/dE1 (S) */
  }
  return GSL_SUCCESS;
}

/*-------------------------------------------------*/
/*---- Set f and df at the same time for speed ----*/
/*-------------------------------------------------*/
int T1_fdf(const gsl_vector *x,void *data,gsl_vector *f,gsl_matrix *J)
{
  T1_f (x,data,f);
  T1_df(x,data,J);
  return GSL_SUCCESS;
}

/*-------------------------------------------*/
/*---- Print iteration and result of fit ----*/
/*-------------------------------------------*/
void print_state(int iter,gsl_multifit_fdfsolver *s,double TR)
{
    fprintf(stdout,"ITERATION %d: %.8f %.8f\n",iter,gsl_vector_get(s->x,0),-TR/log(gsl_vector_get(s->x,1)));
}

/*------------------------*/
/*---- Set NIfTI data ----*/
/*------------------------*/
int setniidata(struct data *d1,struct data *d2)
{
  int i;

  /* Copy NIfTI data header */
  copynfh(&d2->nfh,&d1->nfh); /* copy d2->d1 */
  d1->nvoxels=d2->nvoxels;    /* # voxels in each volume */
  switch (d1->type) {
    case T1:
      d1->nvols=1;
      strcpy(d1->nfh.intent_name,"T1");
      break;
    case M0:
      d1->nvols=1;
      strcpy(d1->nfh.intent_name,"M0");
      break;
    case SV:
      d1->nvols=1;
      break;
    default:
      d1->nvols=d2->nvols; /* # volumes */
  }
  d1->nfh.datatype=NIFTI_TYPE_FLOAT32;

  if ((d1->M = (float **)malloc(d1->nvols*sizeof(float *))) == NULL) 
    nomem(__FILE__,__FUNCTION__,__LINE__);
  for (i=0;i<d1->nvols;i++) {
    if ((d1->M[i] = (float *)malloc(d1->nvoxels*sizeof(float))) == NULL) 
      nomem(__FILE__,__FUNCTION__,__LINE__);
  }

#ifdef DEBUG
  fprintf(stdout,"\n%s: %s()\n",__FILE__,__FUNCTION__);
  fprintf(stdout,"  %s has %d volume(s) of %ld voxels\n",d1->nfh.intent_name,d1->nvols,d1->nvoxels);
#endif

  return(0);
}

/*-------------------------*/
/*---- Load NIfTI data ----*/
/*-------------------------*/
int loadniidata(struct data *d,struct inputdata *in)
{
  char *file=NULL;

  /* Initialise data struct */
  d->nvoxels=0;
  d->nvols=0;
  d->M=NULL;

  /* Set file */
  switch(d->type) {
    case VFA:
      file=in->VFAfile;
      break;
    case AFI:
      file=in->AFIfile;
      break;
  }
  if (file==NULL) return(1);

  readniidata(d,file);
  return(0);
}

/*-------------------------------------*/
/*---- Get input from command line ----*/
/*-------------------------------------*/
void getinput(struct inputdata *in,int argc,char *argv[]) 
{
  char str[MAXPATHLEN];
  char *tok;
  int i;

  /* Initialise input struct */
  in->VFAfile=NULL;
  in->nFAs=16;
  if ((in->FA = (double *)malloc(in->nFAs*sizeof(double))) == NULL) 
    nomem(__FILE__,__FUNCTION__,__LINE__);
  for (i=0;i<16;i++) in->FA[i]=2.0+i*0.4;
  in->TR=0.0028;
  in->maxT1=10.0;
  in->thr=0.0;
  in->AFIfile=NULL;
  in->AFIflip=64.0;

  /* Read arguments of form -x aaa -y bbb and of form -xyz */
  while ((--argc > 0) && ((*++argv)[0] == '-')) {
    char *s;

    /* Interpret each non-null character after the '-' (ie argv[0]) as an option */
    for (s=argv[0]+1;*s;s++) {
      switch (*s) {
        case 'f': /* FAs */
          sscanf(*++argv,"%s",str);
          argc--;
          in->nFAs=1;
          for (i=0;i<strlen(str);i++) if (str[i]==',') in->nFAs++;
          if ((in->FA = (double *)malloc(in->nFAs*sizeof(double))) == NULL) 
            nomem(__FILE__,__FUNCTION__,__LINE__);
          tok=strtok(str,",");
          i=0;
          while (tok != NULL && i<in->nFAs) {
            in->FA[i++]=atof(tok);
            tok=strtok(NULL,",");
          }
          break;
        case 'r': /* TR */
          sscanf(*++argv,"%lf",&in->TR);
          argc--;
          break;
        case 'm': /* max T1 */
          sscanf(*++argv,"%lf",&in->maxT1);
          argc--;
          break;
        case 't': /* threshold */
          sscanf(*++argv,"%lf",&in->thr);
          argc--;
          break;
        case 'a': /* AFI data */
          sscanf(*++argv,"%s",str);
          if ((in->AFIfile = (char *)malloc((strlen(str)+1)*sizeof(char))) == NULL) 
            nomem(__FILE__,__FUNCTION__,__LINE__);
          strcpy(in->AFIfile,str);
          argc--;
          break;
        case 'p': /* prescribed AFI flip */
          sscanf(*++argv,"%lf",&in->AFIflip);
          argc--;
          break;
        case 's': /* T1 & iteration stats */
          T1stats=TRUE;
          break;
        default:
          fprintf(stderr,"\n%s: %s()\n",__FILE__,__FUNCTION__);
          fprintf(stderr,"  Illegal switch option\n");
          fflush(stderr);
          usage();
          break;
      }
    }
  }

  /* VFA data file */
  if (argc>0) {
    sscanf(*argv,"%s",str);
    if ((in->VFAfile = (char *)malloc((strlen(str)+1)*sizeof(char))) == NULL) 
      nomem(__FILE__,__FUNCTION__,__LINE__);
    strcpy(in->VFAfile,str);
    argc--;
  }
  if (in->VFAfile == NULL) usage();

#ifdef DEBUG
  fprintf(stdout,"\n%s: %s()\n",__FILE__,__FUNCTION__);
  fprintf(stdout,"  VFA file = %s\n",in->VFAfile);
  fprintf(stdout,"  # FAs = %d\n",in->nFAs);
  fprintf(stdout,"  FA = %lf",in->FA[0]);
  for (i=1;i<in->nFAs;i++) fprintf(stdout,", %lf",in->FA[i]);
  fprintf(stdout,"\n");
  fprintf(stdout,"  TR = %lf s\n",in->TR);
  fprintf(stdout,"  max T1 = %lf s\n",in->maxT1);
  if (in->thr > 0.0) fprintf(stdout,"  threshold = %lf\n",in->thr);
  if (in->AFIfile != NULL) {
    fprintf(stdout,"  AFI file = %s\n",in->AFIfile);
    fprintf(stdout,"  AFI flip = %lf deg\n",in->AFIflip);
  }
  if (T1stats) fprintf(stdout,"  reporting T1 stats\n");
  fflush(stdout);
#endif
}

/*---------------*/
/*---- Usage ----*/
/*---------------*/  
int usage()
{
  fprintf(stdout,"\n  niiT1vfa [-f FA1,..,FAn] [-r TR] [-m maxT1] [-t thr]"); 
  fprintf(stdout,"\n           [-a AFI.nii] [-p AFIflip] [-s] VFA.nii");
  fprintf(stdout,"\n    VFA.nii is the file containing the VFA data");
  fprintf(stdout,"\n    Options:");
  fprintf(stdout,"\n      [-f FA1,FA2,..,FAn] specify flip angles");
  fprintf(stdout,"\n      [-r TR]             specify TR/s (default 0.0028 s)");
  fprintf(stdout,"\n      [-m maxT1]          maximum T1/s (default 10.0 s)");
  fprintf(stdout,"\n      [-t thr]            threshold (default automatic)");
  fprintf(stdout,"\n      [-a AFI.nii]        use AFI FA file to correct prescribed FAs");
  fprintf(stdout,"\n      [-p AFIflip]        prescribed AFI flip/degrees (default 64.0 deg)");
  fprintf(stdout,"\n      [-s]                generate T1 & iteration stats");
  fprintf(stdout,"\n    Output:");
  fprintf(stdout,"\n      T1_VFA.nii          T1 estimate");
  fprintf(stdout,"\n      M0_VFA.nii          M0* estimate, M0* = Mo.exp(-TE/T2*)");
  fprintf(stdout,"\n      NIT_VFA.nii         Number of iterations (-s option only)\n\n");
  exit(1);
}

