/* nifti1.c */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* nifti1.c: NIfTI-1 routines for niiT1vfa                                   */
/*                                                                           */
/* Copyright (C) 2014 Paul Kinchesh                                          */
/*                                                                           */
/* niiT1vfa is free software: you can redistribute it and/or modify          */
/* it under the terms of the GNU General Public License as published by      */
/* the Free Software Foundation, either version 3 of the License, or         */
/* (at your option) any later version.                                       */
/*                                                                           */
/* niiT1vfa is distributed in the hope that it will be useful,               */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of            */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the              */
/* GNU General Public License for more details.                              */
/*                                                                           */
/* You should have received a copy of the GNU General Public License         */
/* along with niiT1vfa. If not, see <http://www.gnu.org/licenses/>.          */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/**/

#include "niiT1vfa.h"

/*--------------------------------*/
/*---- Write NIfTI-1 nii data ----*/
/*--------------------------------*/
int writeniidata(struct data *d,char *file)
{
  FILE *fp;
  float fourbytespace=0;

  /* Write header */
  if ((fp = fopen(file, "w")) == NULL) {
    fprintf(stderr,"  Unable to write to %s\n",file);
    fflush(stderr);
    return(1);
  }
  fwrite(&d->nfh,1,sizeof(nifti_1_header),fp); /* write the 348 bytes */
  fwrite(&fourbytespace,1,sizeof(float),fp);
  /* Write data */
  fwrite(d->M[0],sizeof(float),d->nvoxels,fp);
  /* Close file */
  fclose(fp);  

  return(0);
}

/*-------------------------------*/
/*---- Read NIfTI-1 nii data ----*/
/*-------------------------------*/
int readniidata(struct data *d,char *file)
{
  FILE *fp;
  int i;

  /* Initialise data struct */
  d->nvoxels=0;
  d->nvols=0;
  d->M=NULL;

  /* Open input file */
  if ((fp=fopen(file,"r")) == NULL) {
    fprintf(stderr,"\n%s: %s()\n",__FILE__,__FUNCTION__);
    fprintf(stderr,"  Unable to open %s\n\n",file);
    fprintf(stderr,"  Aborting ...\n\n");
    fflush(stderr);
    exit(1);
  }

  /* read 348 bytes */
  fread(&d->nfh,1,sizeof(nifti_1_header),fp);

  /* nii files have 352 byte offset to data */
  fseek(fp,352.0,SEEK_SET);

  /* Calculate # voxels/volume and # volumes */
  d->nvoxels=(long)(d->nfh.dim[1]*d->nfh.dim[2]*d->nfh.dim[3]);
  d->nvols=1;
  if (d->nfh.dim[4]>1) d->nvols *= d->nfh.dim[4];
  if (d->nfh.dim[5]>1) d->nvols *= d->nfh.dim[5];

  /* malloc and read data */
  switch (d->nfh.datatype) {
    case NIFTI_TYPE_FLOAT32:
      if ((d->M = (float **)malloc(d->nvols*sizeof(float *))) == NULL) 
        nomem(__FILE__,__FUNCTION__,__LINE__);
      for (i=0;i<d->nvols;i++) {
        if ((d->M[i] = (float *)malloc(d->nvoxels*sizeof(float))) == NULL) 
          nomem(__FILE__,__FUNCTION__,__LINE__);
      }
      for (i=0;i<d->nvols;i++) {
        fread(d->M[i],d->nvoxels,sizeof(float),fp);
      }
      break;
    default:
      fprintf(stderr,"\n%s: %s()\n",__FILE__,__FUNCTION__);
      fprintf(stderr,"  Only NIFTI_TYPE_FLOAT32 data supported\n\n");
      fprintf(stderr,"  Aborting ...\n\n");
      fflush(stderr);
      exit(1);    
      break;
  }

#ifdef DEBUG
  fprintf(stdout,"\n%s: %s()\n",__FILE__,__FUNCTION__);
  fprintf(stdout,"  %s has %d volume(s) of %ld voxels\n",file,d->nvols,d->nvoxels);
#endif

  return(0);
}

/*-------------------------------*/
/*---- Free NIfTI-1 nii data ----*/
/*-------------------------------*/
int freeniidata(struct data *d)
{
  int i;
  for (i=0;i<d->nvols;i++) free (d->M[i]);
  free (d->M);
  d->M=NULL; 
  d->nvoxels=0;
  d->nvols=0;
  d->type=NONE;
  return(0);
}

/*-----------------------------*/
/*---- Copy NIfTI-1 header ----*/
/*-----------------------------*/
void copynfh(struct nifti_1_header *nfh1,struct nifti_1_header *nfh2)
{
  int i;

  /* Copy from nfh1 -> nfh2 */
  nfh2->sizeof_hdr     = nfh1->sizeof_hdr;
  strcpy(nfh2->data_type,"");                 /* UNUSED char[10] */
  strcpy(nfh2->db_name,"");                   /* UNUSED char[18] */
  nfh2->extents        = nfh1->extents;       /* UNUSED int */
  nfh2->session_error  = nfh1->session_error; /* UNUSED short */
  nfh2->regular        = nfh1->regular;       /* UNUSED char */
  nfh2->dim_info       = nfh1->dim_info;
  for (i=0;i<8;i++) 
    nfh2->dim[i]       = nfh1->dim[i];
  nfh2->intent_p1      = nfh1->intent_p1;
  nfh2->intent_p2      = nfh1->intent_p2;
  nfh2->intent_p3      = nfh1->intent_p3;
  nfh2->intent_code    = nfh1->intent_code;  
  nfh2->datatype       = nfh1->datatype;
  nfh2->bitpix         = nfh1->bitpix;
  nfh2->slice_start    = nfh1->slice_start;
  for (i=0;i<8;i++) 
    nfh2->pixdim[i]    = nfh1->pixdim[i];
  nfh2->vox_offset     = nfh1->vox_offset;
  nfh2->scl_slope      = nfh1->scl_slope;
  nfh2->scl_inter      = nfh1->scl_inter;
  nfh2->slice_end      = nfh1->slice_end;
  nfh2->slice_code     = nfh1->slice_code; 
  nfh2->xyzt_units     = nfh1->xyzt_units;
  nfh2->cal_max        = nfh1->cal_max;
  nfh2->cal_min        = nfh1->cal_min;
  nfh2->slice_duration = nfh1->slice_duration;
  nfh2->toffset        = nfh1->toffset;
  nfh2->glmax          = nfh1->glmax;
  nfh2->glmin          = nfh1->glmin;
  strcpy(nfh2->descrip,nfh1->descrip);  
  strcpy(nfh2->aux_file,""); /* not used */
  nfh2->qform_code     = nfh1->qform_code;
  nfh2->sform_code     = nfh1->sform_code;
  nfh2->quatern_b      = nfh1->quatern_b;
  nfh2->quatern_c      = nfh1->quatern_c;
  nfh2->quatern_d      = nfh1->quatern_d;
  nfh2->qoffset_x      = nfh1->qoffset_x;
  nfh2->qoffset_y      = nfh1->qoffset_y;
  nfh2->qoffset_z      = nfh1->qoffset_z;
  for (i=0;i<4;i++) {
    nfh2->srow_x[i]    = nfh1->srow_x[i];
    nfh2->srow_y[i]    = nfh1->srow_y[i];
    nfh2->srow_z[i]    = nfh1->srow_z[i];
  }
  strcpy(nfh2->intent_name,"");
  strcpy(nfh2->magic,"n+1");
}

int diffvolsize(struct data *d1,struct data *d2)
{
  if (d1->nfh.dim[1] != d2->nfh.dim[1]) return(1);
  if (d1->nfh.dim[2] != d2->nfh.dim[2]) return(1);
  if (d1->nfh.dim[3] != d2->nfh.dim[3]) return(1);
  if (d1->nfh.dim[1] != d2->nfh.dim[1]) return(1);
  if (d1->nvoxels != d2->nvoxels) return(1);
  return(0);
}

