%% Program generateZeemandecel_N
% as other trajectory programs, but for the deceleration of metastable N,
% N(^2D)

% metastable N = states ^2D_3/2 or ^2D_5/2
% MJ = -3/2,-1/2,1/2,3/2 for ^2D_3/2
% MJ = -5/2,-3/2,-1/2,1/2,3/2,5/2 for ^2D_5/2
% always choose one spin-orbit state for deceleration

% 08/03/2014
% removed trapping
% removed spin flip estimate
% removed bias coil
% removed reading from experimental data files
% use adaptive phase angle (if desired, compare to normal phase angle) 
% note: switch-on time for coil 1 rather arbitrary and does not change
% final velocity significantly - choose such that it is a little bit
% shorter (or equal to coil 2); maximum Babs for last coil a little bit
% higher than for all the other coils, since mean acceleration not valid
% (not pulse overlap with coil behind it)

% LINUX command line input
% 1) cd folder_with_mfile
% 2) nohup \matlab -r generateZeemandecel_He &

% notes:
% LINUX will create a nohup.out file - DELETE DURING EXECUTION OF SCRIPT!! (contains massive file
% sizes; delete inside the MATLAB script)
% exit MATLAB after the script has finished execution, do this using
% exit;


%% Prepare Matlab to go for it...
%

% Matlab can indeed hold onto some variables and other settings "under the hood". 
% I have the following set up as a short-cut to purge it back to a "just switched on" 
% state (the one that can really catch you out is that clear functions is not a subset 
% of clear all).

restoredefaultpath;
clc;
clear all;
close all;
clear functions;
bdclose('all');
fclose('all');

timerStart = tic;

%% Physical Constants
%

kB = 1.3806504E-23; % Boltzmann constant (in J/K)
amu = 1.6605402E-27; % Atomic mass unit (in kg)
muB = 9.2740154E-24; % Bohr magneton in J/T


fprec = 6; % number of significant digits for output files


%% Read input data, define constants and variables
%

defaultfolder = pwd;

[names, inputvals] = textread('input.txt', '%s %f');

realnumbcoils = inputvals(1);
% total number of coils
amount2decel = inputvals(2);
% amount of deceleration/acceleration relative to the maximum amount of
% deceleration/acceleration possible at this current (for Oxsim = 2)
phasedeg = inputvals(3);
% phase angle for deceleration (for Oxsim = 0 or Oxsim = 1)
vz0dec = inputvals(4)/1000;
% initial velocity synchronous particle in z direction (in mm/micros)
current = inputvals(5);
% actual current though deceleration coils (in A)
addtodetect = inputvals(6);
% distance to detector after leaving last coil (in mm);
% first guess: 20 mm after end of last coil
dist2coil1 = inputvals(7);
% distance to the first coil (mm)
incpltime = inputvals(8);
% incoupling time decelerator (mus)
ZeemanN = inputvals(9);
% spin orbit state used for deceleration
fromfile = inputvals(10);
% load external coil.txt file (Atreju calc. evolutionary algorithm)


dircurrent = strcat('axaz_',num2str(current),'A');
if exist(dircurrent,'dir') == 0
	error('Err:curr','For this input current, a folder for axaz does not exist. Calculate mean acceleration first.')
end


if ZeemanN == 52
	gj = 1.200318; % ^2D_5/2, see Beltran-Lopez et al., Phys. Rev. A 39(1), 1989, 58
    mj = 5/2;
else if ZeemanN == 32
	gj = 0.799483; % ^2D_3/2, see Beltran-Lopez et al., Phys. Rev. A 39(1), 1989, 58
    mj = 3/2;
else
    error('Err:Zeeman_state','For N atom deceleration, only states 32 or 52 possible.')
    end
end

% ensure that all timings are multiples of 10 ns

% decelerator:

if mod(round(incpltime*1000*1E4)/1E4,10) ~= 0
    error('Err:incpltime','Incoupling time must be multiple of 10 ns!')
end


[names, inputvals] = textread('deceleration_param.txt', '%s %f');   

z0 = inputvals(1);
% initial position synchronous particle on z axis (in mm)
simcurrent = inputvals(2);
% current used to simulate B field of the deceleration coils
mass = inputvals(3)*amu; % mass of the atom/molecule (in kg)
facfirstswitch = inputvals(4); 
% empirical scale factor for the switching of the first coil
coillength = inputvals(5);
% length of one coil in z direction (in mm);
coildist = inputvals(6);
% distance from the middle of one coil to the middle of the next (in mm)
maxpulselength = inputvals(7);
% maximal pulselength of first coil (in mus)
R = inputvals(8); % resistance in Ohm
L = inputvals(9); % inductance in Henry
timeoverlap = inputvals(10);
% overlap between coil pulses
rampcoil = inputvals(11);
% normal rise/fall time B field deceleration coils (in mus)
ramp1 = inputvals(12);
% shorter rise/fall time B field deceleration coils (in mus) (when
% overlapped)
h1 = inputvals(13);
% scaled current height kick-hold 
h2 = inputvals(14);
% scaled current height at switch off

% for rampfactor:
% here: timeoverlap only relevant for coils with current in same direction
% anyway
m1 = h1/ramp1;
% n1 = 0
m2 = (1-h1)/timeoverlap;
n2 = h1-m2*ramp1;
m3 = -(1-h2)/timeoverlap;
m4 = -h2/ramp1;

timesteppulse = inputvals(15)/1000;
% numerical timestep for pulse generation, in ns, should be between 1-10 ns
fm = inputvals(16); % focusing mode
                           % 0 = OFF
                           % 4 = neg. current after switch-off
Oxsim = inputvals(17); % switching mode (phase angle mode)
                           % 0 = Zurich (AW)
                           % 1 = Oxford (KD)
                           % 2 = Oxford adaptive (adaptive)
negcurrent = inputvals(18); % current for re-switching (negative!)


if mod(round(rampcoil*1000*1E4)/1E4,10) ~= 0
    error('Err:rampcoil','Ramp time for decelerator coils must be multiple of 10 ns!')
end

if mod(round(timeoverlap*1000*1E4)/1E4,10) ~= 0
    error('Err:timeoverlap','Pulse overlap between coils must be multiple of 10 ns!')
end


[names, inputvals] = textread('inputfor3D.txt', '%s %f');   

no_particles = inputvals(1);
% number of particles
% if no_particles = 1: simulation set to synchronous particle by default
seed = inputvals(2);
% determines whether random numbers are seeded, 1 == true
timestep = inputvals(3)/1000;
% numerical timestep for trajectory simulation, in ns, should be 10-100 ns
disttowall = inputvals(4);
% inner radius of one coil, i.e., distance to glass tube or PEEK (in mm)
radius_laser = inputvals(5)/2;
% radius of detection laser along y axis, e.g., radius_laser = 0.35/2 (mm)
laserfrequency = inputvals(6);
% spacing for the time delay between photolysis and detection laser
Tz = inputvals(7); 
% longitudinal temperature (in K) needed to estimate velocity distribution
% along z axis
Tr = inputvals(8); 
% transverse temperature (in K) needed to estimate velocity distribution
% along r axis
l0 = inputvals(9);
% initial beam length in x dir (in mm) that is covered by the egun
% negligible (5E-5 mm)!
degun = inputvals(10);
% diameter egun beam (in mm)
vz0 = inputvals(11)/1000;
% initial longitudinal beam (!) velocity (in mm/micros)
% this is different from vz0dec - vz0dec cuts out an arbitrary slice from
% the initial velocity distribution
vr0 = inputvals(12)/1000;
% initial transverse beam velocity (in mm/micros)
disttoskimmer = inputvals(13); % distance nozzle - skimmer (in mm)
skimmerradius = inputvals(14)/2; % skimmer radius close to nozzle (in mm)
skimmeropenradius = inputvals(15)/2; % skimmer opening radius (in mm)
skimmerlength = inputvals(16); % skimmer length (in mm)
alphaskim = atan((skimmeropenradius-skimmerradius)/skimmerlength); % angle skimmer opening (in rad)
% account for the fact that decelerator may be turned off for
% acquisition but that pulse sequence was calculated with them included

decon = inputvals(17); % decelerator ON (= 1) or OFF (=0)
fhexon = inputvals(18); % means to include (= 1) or exclude (=0) the font Halbach hexapole array

% get additional information?!
get_phasespace = inputvals(19); % get phase space distribution ON (= 1) or OFF (=0)
% countp is activated by default
get_trajectories = inputvals(20); % get phase space trajectories ON (= 1) or OFF (=0)
get_realtrajectories = inputvals(21); % get real particle trajectories ON (= 1) or OFF (=0)
get_video = inputvals(22); % video making option ON (= 1) or OFF (=0)

stop_time = inputvals(23); % time at which the simulation is stopped (in case there are still particles remaining)

egun_pulsedur = inputvals(24); % egun pulse duration (mus)
egun_real = inputvals(25); % use real egun pulse profile (from FC) ON (= 1) or OFF (=0)
if egun_real == 1 && (egun_pulsedur ~= 20 && egun_pulsedur ~= 50)
	error('Err:egunprofile','Got experimental FC traces for 20 mus and 50 mus only. Use egun_real = 0 instead?')
end
vx_defl = inputvals(26)/1000; % deflection velocity in xdir (in mm/mus)

get_focus = inputvals(27); % get properties of beam at different positions along the beam axis, to compare with front Halbach (= 1) or OFF (=0)

if decon == 0 || current == 0
    decon = 0;
    current = 0;
    warning('WarnMe:decel','Decelerator off.')
end

if vr0 == 0
    bivar_rnd = 1;
    warning('WarnMe:bivar_rnd','vr0 != 0: Bivariate normal distribution used!')
end

if get_realtrajectories == 1
       if no_particles > 3E4
           error('Err:No_particles1','get_realtrajectories == 1: Number of particles limited to 3E4!')
       end
else if get_focus == 1
       if no_particles > 5E4
           error('Err:No_particles2','get_focus == 1: Number of particles limited to 5E4!')
       end
else if get_trajectories == 1 && get_realtrajectories == 0
       if no_particles > 1E5
           error('Err:No_particles3','get_trajectories == 1: Number of particles limited to 1E5!')
       end
    end
    end
end

% check that laserfrequency is a multiple of timestep
if rem(laserfrequency,timestep) ~= 0
    error('err:timestep','laser frequency must be a multiple of time step, change time step in input file and continue')
end


%% Hexapole properties
% here: NdFeB, Br = 1.3 T, r0 = 3 mm

[names, inputvals] = textread('input_Halbach.txt', '%s %f');   

Brem = inputvals(1); % remanence (in T)
r0hex = inputvals(2); % inner radius (in mm)
Lhex = inputvals(3); % length of the hexapole, z dir (in mm)
Lhexblock = inputvals(4); % length of the hexapole block, aluminium shell, z dir (in mm)

% multipole coefficients: a3=0; a15=0;
b3=inputvals(5);
b15=inputvals(6);

% coefficient for Glaser function:
% note: use simplified Glaser funtion of the form:
% y = 1./((1+(z/a).^4).^2) with a = d/(sqrt(sqrt(2)-1));
dGlaser = inputvals(7);
aGlaser = dGlaser/(sqrt(sqrt(2)-1));

dist2hex = disttoskimmer + skimmerlength + inputvals(8); % distance to where the aluminium block of the hexapole starts
dist2midfhex = dist2hex + (Lhexblock-Lhex) + Lhex/2;
% distance to the centre of the front hexapole = 
% distance to the edge of the hexapole + aluminium wall + length hexapole/2

bfhexextend = 3*Lhex; % limit dimensions of hexapole B field along decelerator z axis (in mm)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Generate the pulse sequence
% using Zeeman effect = 1 (lfs, linear)
% note: any Halbach hexapole arrays will not affect the motion of the 
% synchronous particle since Br=Bphi=Bz= 0 on axis

% other variables:
% gradB = sum(gradBz) = total B field gradient in a timestep
% |B| = sum(Bz) = total B field in a timestep
% rampfactor = to account for the finite rise/fall times 
%       of the coil B fields 
% accel, acc = acceleration of the synchronous particle
% particle = generates a matrix for the particle [position velocity]
% tol = tolerance for difference in switching time from previous and 
%       current iteration
% s = timestep counter
% oldcoil = switch-off time from last loop
% lastcount = saves the last counter from the current loop

negcurrent
fm
Oxsim

if exist('nohup.out','file') ~= 0
    delete('nohup.out')
end

dist2mid1stcoil = dist2coil1+coillength/2; 

% coil = matrix for the coil positions 
%       [position switch-on_time switch-off_time pulse_duration]
coil = zeros(realnumbcoils,4);

% define the coil centres
coil(1,1) = dist2mid1stcoil;
for i = 1:realnumbcoils-1
    coil(i+1,1) = coil(1,1)+ i*coildist;
end

sizcoil = size(coil,1);

% skip generation of the pulse sequence for external file read-in (Atreju 
% evolutionary algorithm)

if fromfile == 1

    warning('WarnMe:extseq','External pulse sequence used! No pulse sequence generated.')
    
	fileID = fopen('input.txt');
    C = textscan(fileID,'%s','delimiter','\n');
    file2use=C{1}{10};
    fclose(fileID);
    [file2use,~] = strtok(file2use); % cut to filename
    
	if exist(strcat('extfolder/',file2use),'file') ~= 0
        cd('extfolder')
            coil2use = dlmread(file2use,'\t');
        cd(defaultfolder)
	end
    
    if size(coil2use,1) ~= realnumbcoils
        error('Err:filesize','Number of coils from evolutionary algorithm does not match with program input!') 
    end
	if size(coil2use,2) ~= 4
        error('Err:filesize','Number of columns from evolutionary algorithm does not match with program!') 
	end
    coil = [coil(:,1) coil2use(:,2:3) coil(:,4)];

else

%% B field along z axis 
% from FEMM or Comsol file

cd('sim_files')
bfieldz = load('bonzaxis.txt'); % analytic solution
cd(defaultfolder)

bdist = bfieldz(2,1) - bfieldz(1,1); % spacing B field (in T)
bextend = -bfieldz(1,1); % dimension B field along decelerator axis (in mm)


%% Data for velocity dependence of kappa_0

cd(dircurrent)
acclevel = load('acclevel.txt'); % mean acceleration in z direction, at r = 0 (in m/s^2)
kap4level = load('kap4level.txt'); % kappa_0
vel4level = load('vel4level.txt'); % velocity (in mm/mus) corresponding to kappa_0
cd(defaultfolder)

acclevel = acclevel/max(acclevel); % scale mean acceleration to maximum value 


%% Zeeman effect
% derivative of the Zeeman effect in N(2D):
% for pulse generation: take only lfs state with highest MJ (linear),
% choose spin-orbit state

dEZee = mj*gj*muB; 

%% Coil properties and switching positions along decelerator axis
%

if Oxsim == 2 % else: use phasedeg specified in input file
        % here: use an adaptive phase angle:
    
        % find contour line that is closest to the amount of
        % deceleration/acceleration specified
        [dummy ind2dec] = min(abs(acclevel-amount2decel));
        trueamount2decel = acclevel(ind2dec)

        phasearr = [vel4level(:,ind2dec) kap4level(:,ind2dec)]; 
        phasearr = phasearr(~any(isnan(phasearr),2),:); %remove NaN values
        % convert kappa to phase angle
        % phiAW = (phasearr(:,2)+1/2-phasearr(:,1)*1000*rampcoil*1E-6/(coildist/1000))*180; % Alex Wiederkehr phase angle
        phiKD = (phasearr(:,2)+1/2)*180; % Katrin Dulitz phase angle
        phasearr(:,3) = phiKD; % use the same nomenclature as for Oxsim = 1
 
        phasearr
        
        if vz0dec > max(phasearr(:,1)) || vz0dec < min(phasearr(:,1))
            error('Err:curr2','Velocities to calculate phase angle are out of range. Calculate mean acceleration first if desired.')
        end
        phasedeg = interp1(phasearr(:,1),phasearr(:,3),vz0dec);
        
        % position of the synchr. particle when coils are switched off
        phaseangle = coil(:,1) - (coildist/180)*(90-phasedeg);
    
        phasecheck = zeros(sizcoil,3); % record phase angle and particle velocity at every switch-off (decay to zero)
        phasecheck(1,:) = [vz0dec*1000 phasedeg phaseangle(1,1)];
        warning('WarnMe:phasedeg','Adaptive phase angle used instead of phase angle specified in input file.')
else
        % position of the synchr. particle when coils are switched off
        phaseangle = coil(:,1) - (coildist/180)*(90-phasedeg);
        
        phasecheck = [vz0dec*1000 phasedeg  phaseangle(1,1)];
end

    
% tfirstcoil = (phaseangle(1)-coildist)/vz0dec - rampcoil; % this was used for H atom deceleration
tfirstcoil = (phaseangle(1)-coildist)/vz0dec - facfirstswitch*rampcoil; 
% for deceleration with 12 coils

ff = 500;
tolz = 0.005; % phase angle tolerance (in mm)

coil(1,2) = tfirstcoil;
coil(1,3) = coil(1,2) + rampcoil + ff*timesteppulse;
coil(2,2) = coil(1,3) - timeoverlap;
coil(2,3) = coil(2,2) + ramp1 + timeoverlap + ff*timesteppulse;


%% Preallocating vectors for speeding up the program
%

time = 0;

%% Optimization
%

s = 0; % time counter
zabs = z0; % initial position
vz = vz0dec; % initial velocity
vhz = 0; % dummy velocity half-step
cycles = 0; % number of cycles needed to get the pulse sequence

% vvv = [0 0 0];
% getv = 0;

foundalltimes = 0;

% Oxsim = 0: Zurich phase angle
% Oxsim = 1: (old) Oxford phase angle (ramp to zero)
% Oxsim = 2: adaptive phase angle


for j = 1:realnumbcoils  

ii = 1;
gottime = 0;
stopit = 0;

while stopit == 0      

    if cycles > 1E3
        error('Err:cycles','Maximum number of iterations reached without convergence. Change initial parameters.')
    end

s = s + 1; % begin with s = 1 since time(s=1) = 0*timestep;
time = s*timesteppulse; % time in microsec


%% B field ramp, B field and gradient
%

% determines B field strength (Bz) along decelerator axis
% next coil can only be switched on after the switch-off trigger for the
% previous coil

% only coil j, j-1 and j+1 since coils are not overlapped

% predefine B field values!!
% gradBtot_z = 0;
Bz1 = 0;
Bz2 = 0;

for ico = 1:4

jj = j+ico-2;

	if jj > 0 && jj <= realnumbcoils
        
        if coil(jj,2) ~= 0 && (zabs < (coil(jj,1) + bextend)) && (zabs > (coil(jj,1) - bextend))
        
        % decide here which coil gets normal rise and fall times, and for which
        % coils edges are required
        % edgemode = 0 means: normalrise = 0; normalfall = 0;
        % edgemode = 1 means: normalrise = 1; normalfall = 0;
        % edgemode = 2 means: normalrise = 0; normalfall = 1;
        % edgemode = 3 means: normalrise = 1; normalfall = 1;
                if jj == 1
                    edgemode = 1;
                else if jj == realnumbcoils
                    edgemode = 2;
                else
                    edgemode = 0;
                    end
                end
            
                if edgemode == 1
                        if time <= coil(jj,2) || coil(jj,2) == coil(jj,3)
                            rampfactor = 0;
                        else if time > coil(jj,2) && time < coil(jj,2)+rampcoil % normal rise
                            rampfactor = (current/simcurrent)*(1/rampcoil)*(time-coil(jj,2));  
                        else if time >= coil(jj,2)+rampcoil && time < coil(jj,3)-timeoverlap % constant level
                            rampfactor = (current/simcurrent);                
                        else if time >= coil(jj,3)-timeoverlap && time < coil(jj,3) % overlap fall
                            rampfactor = (current/simcurrent)*(m3*(time-coil(jj,2))+(h2-m3*(coil(jj,3)-coil(jj,2))));               
                        else if time >= coil(jj,3) && time < coil(jj,3)+ramp1 % rise 1 fall
                            rampfactor = (current/simcurrent)*(m4*(time-coil(jj,2))-m4*((coil(jj,3)-coil(jj,2))+ramp1));
                            else 
                            rampfactor = 0;
                            end
                            end
                            end
                            end
                        end
                else if edgemode == 2
                        if time <= coil(jj,2) || coil(jj,2) == coil(jj,3)
                            rampfactor = 0;    
                        else if time > coil(jj,2) && time < coil(jj,2)+ramp1 % rise 1 rise
                            rampfactor = (current/simcurrent)*(m1*(time-coil(jj,2)));    
                        else if time >= coil(jj,2)+ramp1 && time < coil(jj,2)+ramp1+timeoverlap % overlap rise
                            rampfactor = (current/simcurrent)*(m2*(time-coil(jj,2))+n2);
                        else if time >= coil(jj,2)+ramp1+timeoverlap && time < coil(jj,3) % constant level
                            rampfactor = (current/simcurrent);                
                        else if time >= coil(jj,3) && time < coil(jj,3)+rampcoil % normal fall
                            rampfactor = (current/simcurrent)*(1/rampcoil)*(coil(jj,3)+rampcoil-time);
                            else
                        	rampfactor = 0;
                            end
                            end
                            end
                            end
                        end
                    else if edgemode == 0
                        if time <= coil(jj,2) || coil(jj,2) == coil(jj,3)
                            rampfactor = 0;    
                        else if time > coil(jj,2) && time < coil(jj,2)+ramp1 % rise 1 rise
                            rampfactor = (current/simcurrent)*(m1*(time-coil(jj,2)));    
                        else if time >= coil(jj,2)+ramp1 && time < coil(jj,2)+ramp1+timeoverlap % overlap rise
                            rampfactor = (current/simcurrent)*(m2*(time-coil(jj,2))+n2);
                        else if time >= coil(jj,2)+ramp1+timeoverlap && time < coil(jj,3)-timeoverlap % constant level
                            rampfactor = (current/simcurrent);                
                        else if time >= coil(jj,3)-timeoverlap && time < coil(jj,3) % overlap fall
                            rampfactor = (current/simcurrent)*(m3*(time-coil(jj,2))+(h2-m3*(coil(jj,3)-coil(jj,2))));               
                        else if time >= coil(jj,3) && time < coil(jj,3)+ramp1 % rise 1 fall
                            rampfactor = (current/simcurrent)*(m4*(time-coil(jj,2))-m4*((coil(jj,3)-coil(jj,2))+ramp1));
                            else
                        	rampfactor = 0;
                            end
                            end
                            end
                            end
                            end
                        end
                        else % edgemode == 3
                            if time <= coil(jj,2) || coil(jj,2) == coil(jj,3)
                                rampfactor = 0;
                            else if time > coil(jj,2) && time < coil(jj,2)+rampcoil % normal rise
                                rampfactor = (current/simcurrent)*(1/rampcoil)*(time-coil(jj,2));  
                            else if time >= coil(jj,2)+rampcoil && time < coil(jj,3) % constant level
                                rampfactor = (current/simcurrent);                
                            else if time >= coil(jj,3) && time < coil(jj,3)+rampcoil % normal fall
                                rampfactor = (current/simcurrent)*(1/rampcoil)*(coil(jj,3)+rampcoil-time);
                                else
                                rampfactor = 0;
                                end
                                end
                                end
                            end                            
                        end
                    end
                end
                    
% normal:                    
%                 if it == 1      % rising edge
%                     rampfactor = (current/simcurrent)*(1/rampcoil)*(time-coil(jj,2));    
%                 else if it == 2 % constant level
%                     rampfactor = (current/simcurrent);
%                 else if it == 3 % falling edge
%                     rampfactor = (current/simcurrent)*(1/rampcoil)*(coil(jj,3) + rampcoil - time);                
%                     end
%                     end
%                 end
                
                if rampfactor ~= 0
                    index = ceil((zabs - coil(jj,1) + bextend)/bdist);
                    Bz1 = Bz1+rampfactor*bfieldz(index,2);                
                    Bz2 = Bz2+rampfactor*bfieldz(index+1,2);
                end
                
        end

	end

end

if fm == 4 % re-switch coil kk between switch-off coil kk+1 and switch-off coil kk+2
           % override zero rampfactors if needed
           % instantaneous switch-on
  for kk = 1:realnumbcoils-1
	if coil(kk+1,3) ~= 0 && zabs < (coil(kk,1) + bextend) && zabs > (coil(kk,1) - bextend)

        if kk < realnumbcoils-1
            if coil(kk+2,3) ~= 0 && time >= coil(kk+1,3) && time <= coil(kk+2,3) 
                rampfactor = (negcurrent/simcurrent); % constant level
            else if coil(kk+2,3) == 0 && time >= coil(kk+1,3)
                rampfactor = (negcurrent/simcurrent);
                end
            end
        else if kk == realnumbcoils-1
            if time >= coil(kk+1,3) && time <= coil(kk+1,3) + rampcoil
                rampfactor = (negcurrent/simcurrent);
            end
            end
        
            index = ceil((zabs - coil(kk,1) + bextend)/bdist);
            Bz1 = Bz1+rampfactor*bfieldz(index,2);                
            Bz2 = Bz2+rampfactor*bfieldz(index+1,2);
        end

	end
  end
end


% total gradient of B (only Bz component needed)
gradBtot_z =(sqrt(Bz2^2)-sqrt(Bz1^2))/bdist;


%% Determine acceleration
%

accsum_z = -(gradBtot_z/0.001)*dEZee/mass*1E-9;


%% Numerical integration of the equations of motion
% using the Velocity Verlet method
% remember: zabs = zabs(s), vz = vz(s-1)


% determine overlap time with next coil
tinq = timeoverlap;


if gottime == 0
    if time >= coil(j,2) % same result as with "if time >= 0"
        sagain = s-1;
        vzlast = vz;
        vhzlast = vhz;
        zabslast = zabs;
        gottime = 1;
    end
end


if s > 1
    vz = vhz + 0.5*timesteppulse*accsum_z;
end
    vhz = vz + 0.5*timesteppulse*accsum_z;
    zabs = zabs + timesteppulse*vhz;

    
% scheme: want phaseangle(j) to be the position at which the coil is
% completely switched off --> hence, guess coil(j,3) and then iterate until
% coil(j,3) + rampcoil, if phaseangle(j) is not reached, add some time to
% coil(j,3) and go back to old coil(j,3) and integrate from there again

if Oxsim == 1 || Oxsim == 2 % Oxford or adaptive phase angle
    if j == realnumbcoils
        coiloff = coil(j,3) + rampcoil;
    else    
        coiloff = coil(j,3) + ramp1;
    end
else % Oxsim == 0; Zurich
	coiloff = coil(j,3);
end


if time >= coiloff && foundalltimes == 0
    
    if zabs < phaseangle(j) % particle position < phaseangle

        coiloffold = coil(j,3);
        cycles = cycles + 1;
        sold = sagain;
        vzold = vzlast;
        vhzold = vhzlast;
        zabsold = zabslast;            

        coil(j,3) = coil(j,3) + ff*timesteppulse;

        s = sagain;
        vz = vzlast;
        vhz = vhzlast;
        zabs = zabslast;
        gottime = 0;
        
    else if zabs >= phaseangle(j) && zabs <= phaseangle(j) + tolz % particle position = phaseangle

        if Oxsim == 2 && j < realnumbcoils
            % for adaptive phase angle: 
            % adapt the switch-off position of the next coil (= phase angle)
            % according to the current velocity of the particle
            % goal: always stay in the phase-stable region during deceleration
            if vz > max(phasearr(:,1)) || vz < min(phasearr(:,1))
                error('Err:curr2','Velocities to calculate phase angle are out of range. Calculate mean acceleration first if desired.')
            end
            phasedeg = interp1(phasearr(:,1),phasearr(:,3),vz);

            phaseangle(j+1) = coil(j+1,1) - (coildist/180)*(90-phasedeg);
            phasecheck(j+1,:) = [vz*1000 phasedeg phaseangle(j+1)];
        end
        
        stopit = 1;
        if j == realnumbcoils
            foundalltimes = 1;
            if Oxsim == 0
                stopit = 0; % undo stopit for Zurich sim!
            end
        end

    else if zabs > phaseangle(j) + tolz % particle position >> phaseangle

        coil(j,3) = coiloffold + (ff/(2^ii))*timesteppulse; % try smaller stepsize
        ii = ii + 1;

        s = sold;
        vz = vzold;
        vhz = vhzold;
        zabs = zabsold;            
        gottime = 0;

        end
        end
    end
    
    
	if j < realnumbcoils
            % in order to avoid Majorana transitions, pulses will be overlapped
            % this yields the switch-on time for the next coil 
            coil(j+1,2) = coil(j,3) - tinq;
            coil(j+1,3) = coil(j+1,2) + rampcoil + ff*timesteppulse;
            % next coil turned off after the (shorter) ramptime plus some arb.
            % shift (guess); difference between ramp1 and ramptime irrelevant
	end

end


% for Zurich simulation, let particle fly until the last coil has decayed
% to zero --> goal: get final velocity!
if Oxsim == 0
    if j == realnumbcoils && foundalltimes == 1
        if time >= coil(j,3)+rampcoil
            stopit = 1; % stopit!
        end
    end
end


% % DO THIS FOR FIRST COIL ONLY:
% % first pulse should not be longer than "maxpulselength", 
% % else coil might explode
% % hence: switch-on time of first coil must be delayed 
% if coil(1,3)-coil(1,2) > maxpulselength
%         coil(1,2) = coil(1,3) - maxpulselength;
% end

% getv=getv+1;
% if getv == 100
%     vvv = vertcat(vvv,[time sqrt(Bz2^2) zabs]); 
%     getv = 0;
% end
% plot(vvv(:,3),vvv(:,2),'k.')

end % end while

end % end j


end


% some high-end security:

% pulse duration:
coil(:,4) = coil(:,3) - coil(:,2);

% pulses must not be longer than "maxpulselength", 
% else coils might explode
mpl = uint32(find(coil(:,4)>maxpulselength));
if isempty(mpl) ~= 1
    if size(mpl,1) == 1 && mpl == size(coil,1)
        coil(end,3) = coil(end,2) + maxpulselength;
        warning('Warn:DurLastCoil','Length of last pulse reduced to maximum pulse length. Final velocity higher than expected!')
    else
        error('Err:maxpulselength','Maximum pulse duration exceeded for more than one coil! Change initial velocity and/or phi0.')
    end
end

if fromfile == 0
    vzfin = vz; % final velocity of synchronous particle
    strr = strcat('final velocity:',num2str(round(vzfin*1000)),' m/s');
    disp(strr)
end

% include incoupling time:
coil(:,2:3) = coil(:,2:3)+incpltime;

% round coil timings to multiples of 10 ns to prevent pulseblaster from
% ignoring times shorter than that
coil(:,2:3) = round(coil(:,2:3)*100)/100;
coil(:,4) = coil(:,3) - coil(:,2);


%% Write results to text file:
%

% make new folder \output_initialvelocity_phasangle_mode
% use phase angle for coil 12 in file name
if fromfile == 0
    newdir = sprintf('output_%s_%s_%s',num2str(vz0dec*1000),num2str(amount2decel),num2str(Oxsim));
else
    newdir = sprintf('output_ext');
end

tf = isdir(newdir);
if isunix == 0
    if tf ~= 1
        mkdir(newdir);
    else selection2 = questdlg('Output directory already exists. Keep existing directory?', 'Data Queue','Yes','No','Yes');
        switch selection2,
        case 'No'
            rmdir(newdir,'s'); % remove folder and its contents
            mkdir(newdir); % create empty folder with same name
        case 'Yes'
            break
        end
        drawnow; % force update!
    end
else % on a LINUX machine, the old folder (if existing) will be deleted and a new folder will be created
    if tf == 1
        rmdir(newdir,'s'); % remove folder and its contents
    end
	mkdir(newdir); 
end

% copy input files into output directory
copyfile('input.txt',newdir)
copyfile('deceleration_param.txt',newdir)

% change to new folder and write results to this directory
cd(newdir)

dlmwrite('coil.txt', coil, 'delimiter', '\t', 'precision',8);

if fromfile == 0
    dlmwrite('phasecheck.txt', phasecheck, 'delimiter', '\t', 'precision',8);
    phasecheck
end

cd(defaultfolder)

copyfile('inputfor3D.txt',newdir)
copyfile('input_Halbach.txt',newdir)
copyfile('generateZeemandecel_N.m',newdir)
if fromfile == 1
    copyfile(strcat('extfolder/',file2use),newdir)
    
    configfile2use = strcat(file2use(1,1:end-8),'config.info');
        
    if exist(strcat('extfolder/',configfile2use),'file') ~= 0
        copyfile(strcat('extfolder/',configfile2use),newdir)
    end
end
% change back to old directory
cd(defaultfolder)

% used for H atom experiments:
% % good agreement using:
% % Tz = 2.0 K
% % vz0 = 450 m/s
% % zshiftdec = -15.57; % (103.9 mm - 88.33 mm)!
% % t0 = 31.14;
% % t0 = zshiftdec/vz0dec, here: t0 = 15.57 mm /0.5 mm/mus = 31.14 mus
% 
% % zshiftdec = -15.57;
% % zshiftbunch = 5;
% % t0 = 31+0;
% 
% t0 = 31;
% zshiftdec = -t0*vz0dec;
% zshiftbunch = 0; % shift bunch of particles back and forth, ~= z0!!
% % later on, I need to do it the other way around and determine t0, 
% % because I basically know what zshiftdec is now!
% 
% 
% coil(:,1) = coil(:,1) + zshiftdec;

t0 = 0;

coil


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Generate random initial positions and velocities
%

% always use groups of 500.000 particles to simulate the initial conditions
% (in order to have the same initial conditions also when you use 200.000
% or 2.000.000 particles)
% if no_particles > 500.000 split into several runs with 500.000 particles each
no_ppstream = 5E5;
numStreams = ceil(no_particles/no_ppstream);

arr_particles = zeros(numStreams,2); 
% column 1 = seed for rng
% column 2 = number of particles to simulate
arr_particles(1,1) = 1;
arr_particles(:,2) = no_ppstream;
for i = 2:numStreams
    arr_particles(i,1) = arr_particles(i-1,1) + no_ppstream; 
end

if sum(arr_particles(:,2)) > no_particles
    arr_particles(end,2) = no_ppstream-(sum(arr_particles(:,2))-no_particles);
end

keepnums = zeros(numStreams,2); % statistics for skimmer
det = zeros(numStreams,1); % statistics for gas pulse data
saves1 = zeros(numStreams,1); % statistics for Zeeman data, state 1
saves2 = zeros(numStreams,1); % statistics for Zeeman data, state 2 
saves3 = zeros(numStreams,1); % statistics for Zeeman data, state 3 
saves4 = zeros(numStreams,1); % statistics for Zeeman data, state 4 
if ZeemanN == 52
    saves5 = zeros(numStreams,1); % statistics for Zeeman data, state 5
    saves6 = zeros(numStreams,1); % statistics for Zeeman data, state 6
end

for ivi = 1:numStreams

% progress report (for each loop):
    tmp11 = sprintf('Loop %d out of %d\n',ivi,numStreams);
    fprintf('%s',tmp11)    
    
% init_cond_all = matrix of initial positions and velocities with
% [x0 y0 z0 vx0 vy0 vz0]:
% x0,y0,z0 = cartesian positions (in mm)
% vx0 vy0 vz0 = "cartesian" velocities (in mm/microsec)

% command for seeding (also used for randi to estimate spin flips):
% generator algorithm: Mersenne twister (Matlab standard)
    if seed == 1
        RandStream.setGlobalStream(RandStream('mt19937ar','Seed', arr_particles(ivi,1)))
        stream = RandStream.getGlobalStream
    end

confname = strcat('init_cond_',num2str(ivi),'.txt');

if ivi == 1
    % synchronous particle (set by default)
    % the synchronous particle will only appear in the first numStream
    init_cond_all = [0 0 z0 0 0 vz0dec];
	no_particles2sim = no_ppstream-1; % 1 = synchronous particle
    no_particles_tot = 1; % number of particles used to generate the initial conditions (1 = syn. particle)
else
    init_cond_all = [];
	no_particles2sim = no_ppstream;
    no_particles_tot = 0; % number of particles used to generate the initial conditions
end

if no_particles > 1

cc = 0;
sizM = 0;
    
% NEED TO CHANGE TO NORMAL DISTRIBUTION FOR PROPER ANALYSIS!!
warning('WarnMe:egun','Need to change particle distribution from egun for proper analysis!')

while no_ppstream - sizM > 0 
    
% (a) for positions
% let us assume that the egun beam is smaller than the cone of the supersonic
% beam emerging from the valve

% random uniform distribution within a cylinder
% r0 and phi0 span up a disk; x0 gives the height


r0_rnd = sqrt(unifrnd(0,degun/2,no_particles2sim,1))*sqrt(degun/2);
phi0_rnd = unifrnd(0,2*pi,no_particles2sim,1);

% transformation polar coordinates <--> cartesian coordinates
% [z,y] = pol2cart(phi,r)
% [z0_rnd,y0_rnd] = pol2cart(phi0_rnd,r0_rnd);
z0_rnd = r0_rnd.*cos(phi0_rnd); % + zshiftbunch;
y0_rnd = r0_rnd.*sin(phi0_rnd);

x0_rnd = unifrnd(-l0/2,l0/2,no_particles2sim,1);


% (b) for velocities
% normally distributed random numbers
% if you want to generate normally distributed vx-vy random numbers 
% that are centered at vx = 0 mm/mus and vy = 0 mm/mus, use bivar_rnd = 1
% else use bivar_rnd = 0

sigmavr0 = sqrt(kB*Tr/mass)/1000; % standard deviation vr0 component

if bivar_rnd == 0

% normally distributed random numbers NOT centered at 0 mm/mus
% DO NOT USE NORMRND FOR GENERATING RANDOM VELOCITIES VR!
% generate random values from the uniform distribution on the interval [-b,b]
b = vr0 + 10*sigmavr0;
numgen = 0;

vx0_rnd = 0;
vy0_rnd = 0;

while numgen < no_particles2sim

pp = -b + 2*b.*rand(no_particles2sim,3); % Generate N-1 3-D points

vrval = sqrt(pp(:,1).^2 + pp(:,2).^2);

% volcano function
% dep = 0.2;
% fval = (dep+(vrval./vr0).^2).*exp(-vrval.^2/(2*vr0^2));

% normpdf = Maxwell Boltzmann for a velocity component (see wiki)
fval = (1/(sigmavr0*sqrt(2*pi)))*exp(-(vrval-vr0).^2/(2*sigmavr0^2));

indk = uint32(find(abs(pp(:,3)/b) <= fval));

vx0_rnd = vertcat(vx0_rnd,pp(indk,1));
vy0_rnd = vertcat(vy0_rnd,pp(indk,2));

numgen = size(vx0_rnd,1)-1;

end

% delete excess number of particles
vx0_rnd = vx0_rnd(1:no_particles2sim,1);
vy0_rnd = vy0_rnd(1:no_particles2sim,1);

else

% normally distributed random numbers centered at 0 mm/mus
% generate bi(multi)variate Gaussian data for vx and vy
% rand_data = mvnrnd(mu,sigma,num of data)
muvr = [0 0]; % mean values centered around 0 mm/mus
% sigma1 = [1 0;  % covariance matrix, diagonals = variances of each variable,
%           0 1];                % off-diagonals = covariances between the variables
% if no correlation, then off-diagonals = 0 and Sigma can also be written as a row array           
SigmaM = [sigmavr0^2 sigmavr0^2]; 
dummy = mvnrnd(muvr,SigmaM,no_particles2sim);
vx0_rnd = dummy(:,1);
vy0_rnd = dummy(:,2);

end

% add deflection velocity (due to egun), constant value for all
vx0_rnd = vx_defl+vx0_rnd;


% vr0_rnd_pos = sqrt(vx0_rnd.^2 + vy0_rnd.^2);
% 
% theta0_rnd = acos(vx0_rnd./vr0_rnd_pos);
% or:
% theta0_rnd = asin(vy0_rnd./vr0_rnd_pos);
% phi0 ~= theta0 so that Jz =~ 0

sigmavz0 = sqrt(kB*Tz/mass)/1000; % standard deviation vz0 component 
vz0_rnd = normrnd(vz0,sigmavz0,no_particles2sim,1);


% assign a time according to the egun pulse duration (mus) to every particle
% assume a uniform gas pulse (pulse is quite flat between 380 and 420 mus,
% so this may be a good assumption)
% recalculate the initial position at t=0, then do skimmer exclusion step
% skimmer exclusion step can be done before or afterwards (same effect)
% but the change in initial positions (initial timing) will change the TOF

if egun_real == 0
    % rectangular time profile for e-gun pulse:
    t_init = vertcat(0,unifrnd(0,egun_pulsedur,no_particles2sim-1,1));
    % sync particle created at t = 0!
else
    % use experimental FC pin profile to generate the times:    
    t_init = 0; % sync particle created at t = 0!
    
    cd('eguntraces')
        % egunprofile = load(strcat('HeAr_1to3_m130_',num2str(egun_pulsedur),'mus_0mus.txt'));
        egunprofile = load(strcat('HeAr_1to3_m130_',num2str(egun_pulsedur),'mus_250mus.txt'));
    cd(defaultfolder)
    egunprofile(:,2) = smooth(egunprofile(:,2),30); % smooth the oscilloscope data
    
    tmax = egunprofile(end,1);
    smax = max(egunprofile(:,2));

    while length(t_init) < no_particles2sim
        trand = tmax.*rand(no_particles2sim,1);
        yrand = smax.*rand(no_particles2sim,1);

        inp = inpoly([trand yrand],[egunprofile(:,1) egunprofile(:,2)]);

%         hold on
%         plot(egunprofile(:,1),egunprofile(:,2),'b-')
%         plot(trand(inp==0),yrand(inp==0),'r.')
%         plot(trand(inp==1),yrand(inp==1),'k.')
%         hold off
 
        t_init = vertcat(t_init,trand(inp==1));
    end
        t_init = t_init(1:no_particles2sim); % remove excess particles created
end
% hist(t_init,500)

x0_rnd = x0_rnd-vx0_rnd.*t_init; % x0y0z0 = xyz-vxyz*t
y0_rnd = y0_rnd-vy0_rnd.*t_init; % x0y0z0 = xyz-vxyz*t
z0_rnd = z0_rnd-vz0_rnd.*t_init; % x0y0z0 = xyz-vxyz*t


% exclude all particles that won't make it through the skimmer straight
% away (linear transformation, see: linear transfer matrix)
% r = r0 + (vr0/vz0)*distance_particle_to_skimmer
% because of the theta-dependency, you cannot use r straight away
xatskimmer = x0_rnd + (vx0_rnd./vz0_rnd).*(disttoskimmer-z0_rnd);
yatskimmer = y0_rnd + (vy0_rnd./vz0_rnd).*(disttoskimmer-z0_rnd);
ratskimmer = sqrt(xatskimmer.^2 + yatskimmer.^2);
ts = uint32(find(ratskimmer<=skimmerradius));
init_cond = [x0_rnd(ts,:), y0_rnd(ts,:), z0_rnd(ts,:), ...
            vx0_rnd(ts,:), vy0_rnd(ts,:), vz0_rnd(ts,:)];
% above method needs much less (temporary) memory!
% this way, I can start with an initial number of 7 mio particles 
% (on my laptop)
% init_cond = [x0_rnd y0_rnd z0_rnd vx0_rnd vy0_rnd vz0_rnd];
% init_cond = init_cond(ts,:);

init_cond_all = vertcat(init_cond_all,init_cond);

no_particles_tot = no_particles_tot + no_particles2sim;

sizM = size(init_cond_all,1);
cc = cc + 1;
if cc == 50
    disp(sizM)
    cc = 0;
end

end

keepnums(ivi,1) = size(init_cond_all,1);
keepnums(ivi,2) = no_particles_tot;

if ivi == numStreams
    skimmerloss_no = 100*sum(keepnums(:,1))/sum(keepnums(:,2));
    realno_particles = no_particles*100/skimmerloss_no;
end

sizi = size(init_cond_all,1);
    if sizi > arr_particles(ivi,2)
        %  select rows with size(arr_particles(ivi,2)) from the init cond matrix
        init_cond_all = init_cond_all(1:arr_particles(ivi,2),:);
    end

end


%% Write initial conditions to file
%

cd(newdir)
    dlmwrite(confname, [init_cond_all(:,1:3) init_cond_all(:,4:6)*1000], 'delimiter', '\t', 'precision', fprec);
    % save velocities in m/s!
    
if ivi == numStreams
    % append results to file 'output_param.txt'
    fid1 = fopen('output_param.txt','w');
    if fromfile == 0
            if Oxsim == 2
            fprintf(fid1, ...
                'true amount to decelerate: %.2f\n', trueamount2decel);
            end
        fprintf(fid1, ...
            'final velocity (in m/s): %d\n', round(vzfin*1000));
    end
    fprintf(fid1, ...
        'particles coming out of the skimmer (in percent): %6.4f\n', skimmerloss_no);
    fclose(fid1);

    % view contents of written file
    type output_param.txt
end

cd(defaultfolder)

clear x0_rnd y0_rnd z0_rnd vx0_rnd vy0_rnd vz0_rnd ...
    xatskimmer yatskimmer ratskimmer r0_rnd phi0_rnd ts init_cond ...
    fval vrval indk pp dummy


%% Coil positions, switching times and timing
%

% DO NOT shift coil positions back to original value (incl. arb. shift?)

% for position analysis (t = const.): 
finaltime = max(coil(:,3)) + rampcoil; 
% finaltime = time at which last coil has been switched-off completely

enddecel = max(coil(:,1)) + coillength/2;

% for TOF analysis (z = const.):
% total distance to detector (in mm)
disttodetect = enddecel + addtodetect
% disttodetect = enddecel + addtodetect - zshiftdec % H atom simulation


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Simulation for the gas pulse:
% 3D trajectory simulation on the gas pulse through the Zeeman decelerator 
% WITHOUT any magnetic fields


%% Trajectory simulation
% go until particle has reached the detector or hit the wall

psget = 0;
stopinteg = 0;
s = 0; % timestep counter
time = 0; % initial time

% define initial conditions:

% p2sim = [xabs yabs zabs vx vy vz i time]
p2sim = [init_cond_all, ((ivi-1)*arr_particles(ivi,2)+1:ivi*arr_particles(ivi,2))', zeros(size(init_cond_all,1),1)];
% particle index corresponds to row number in initial conditions matrix

colinit = size(p2sim,2); % initial number of columns for p2sim


%% Initialize matrices
%  for the gas pulse:

countp_gaspulse = zeros(1,colinit);
% countp and dummy_countp are matrices for the arrival at detection 
% (2nd scheme)

tmp00 = sprintf('Please wait... gas pulse (press Ctrl-C to stop)\n');
fprintf('%s',tmp00)

while stopinteg == 0 && time <= stop_time
% iterate until all particles have been detected (none left)

s = s + 1; % begin with s = 1 since time(s=1) = 0*timestep;
time = t0 + s*timestep; % time in microsec

% progress report (every 100 timesteps):
if mod(s,100) == 0
    tmp = sprintf('Time (mus): %.2f\n', time);
    fprintf('%s',tmp)
end

%% Numerical integration of the equations of motion
% accel = 0 (no magnetic fields), hence the Velocity Verlet method
% is greatly simplified
% remember: zabs = zabs(s), vz = vz(s-1); 
% not important here since vz = const.

p2sim(:,1:3) = p2sim(:,1:3) + timestep*p2sim(:,4:6); % absxyz = absxyz + timestep*vxyz


%% Detection
%

% for "Stern-Gerlach exp."
% count particles when they have reached the detector no matter whether the laser
% is fired or not
% countp = [x y z vx vy vz i time]
% check whether particle is inside the laser beam: 
% check only z dimension, more thorough analysis in the analysis file

% detect all particles in the z detection plane (or shortly behind)
indc = uint32(find(p2sim(:,3)>=disttodetect)); 
countp_gaspulse = vertcat(countp_gaspulse,[p2sim(indc,1:7) time*ones(length(indc),1)]);

if get_phasespace == 1 && psget == 0 && time >= finaltime
    % for phase-space distribution:
    % check whether particle has reached the end of the pulse sequence
    % ps = [xabs yabs zabs vx vy vz i time]
	ps = [p2sim(:,1:7), ones(size(p2sim,1),1)*time];
    
    psget = 1;
end

%% Particle removal
% split the experiment into a number of different sections

% d) throw all particles behind the detection region
p2sim=p2sim(p2sim(:,3)<disttodetect,:);

% get r for all (remaining) particles
% transformation cartesian coordinates <--> polar coordinates
r = sqrt(p2sim(:,1).^2+p2sim(:,2).^2); % new r = sqrt(xabs.^2 + yabs.^2)

% a) for the region inside the skimmer, keep only those inside the cone of
% the skimmer
idecide1 = uint32(find(p2sim(:,3)>= disttoskimmer & p2sim(:,3)<= disttoskimmer + skimmerlength));

if isempty(idecide1) ~= 1
    % for those particles: throw particles with alpha > alphaskim
    idel1 = uint32(find(atan((r(idecide1)-skimmerradius)./(p2sim(idecide1,3)-disttoskimmer)) > alphaskim));
else
    idel1 = [];
end

% c) for the region inside the decelerator
    idecide3 = uint32(find(p2sim(:,3) >= coil(1,1)-coillength/2 & p2sim(:,3) <= enddecel));
    idel3 = uint32(find(r(idecide3) > disttowall));

% delete those particles
if fhexon == 1
    
    % b) for the region inside the hexapole (if relevant)
        idecide2 = uint32(find(p2sim(:,3) >= dist2hex & p2sim(:,3) <= dist2hex + Lhexblock));
        idel2 = uint32(find(r(idecide2) > r0hex));
    
    idelall = [idecide1(idel1);idecide2(idel2);idecide3(idel3)];
else
    idelall = [idecide1(idel1);idecide3(idel3)];
end


if isempty(idelall) ~= 1
    p2sim(idelall,:) = 0;
    p2sim(~any(p2sim,2),:) = [];
end


% no particles left
if size(p2sim,1) == 0
     stopinteg = 1;
else stopinteg = 0;
end


end

% delete zero rows from detection matrices:
countp_gaspulse = countp_gaspulse(any(countp_gaspulse,2),:);

% do some statistics:
% portion of particles that arrive at the end of the decelerator:
det(ivi,1) = size(unique(countp_gaspulse(:,7)),1);

if ivi == numStreams
    det_no = 1E6*sum(det(:,1))/realno_particles;
end

%% Write results into output folder
%

cd(newdir)

if ivi == numStreams
    % append results to file 'output_param.txt'
    fid11 = fopen('output_param.txt','a');
    fprintf(fid11, ...
        'total distance to detector (in mm): %6.4f\n', disttodetect);
    fprintf(fid11, ...
        'gaspulse particles at detection (in ppm): %6.4f\n', det_no);
    fclose(fid11);

    % view contents of written file
    type output_param.txt
end

countp_gaspulse(:,4:6) = countp_gaspulse(:,4:6)*1000; % save velocities in m/s!

dlmwrite(strcat('posatend_gaspulse_',num2str(ivi),'.txt'), countp_gaspulse, ...
    'delimiter', '\t', 'precision', fprec);

if get_phasespace == 1

    ps(:,4:6) = ps(:,4:6)*1000; % save velocities in m/s!
    
    dlmwrite(strcat('ps_gaspulse_',num2str(ivi),'.txt'), ps, ...
        'delimiter', '\t', 'precision', fprec);
    clear ps
end

cd(defaultfolder)

clear countp_gaspulse


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% 3D Simulation including magnetic fields
% 3D trajectory simulation on the Zeeman decelerator

if ivi == 1 % only load (and assign) once - will not be overwritten during the sim
    
%% B field coil
% 

% Bz_n = load('sim_files\Bz_Zurich.txt'); % contains Bz field as a grid with P(r,z) (Comsol, Zurich)
% Br_n = load('sim_files\Br_Zurich.txt'); % contains Br field as a grid with P(r,z) (Comsol, Zurich)
% raxis = load('sim_files\raxis_Zurich.txt'); % raxis as one column
% zaxis = load('sim_files\zaxis_Zurich.txt'); % zaxis as one row
cd('sim_files')
Bz_n = load('Bz_n.txt'); % contains Bz field as a grid with P(r,z) (from analytic solution)
Br_n = load('Br_n.txt'); % contains Br field as a grid with P(r,z) (from analytic solution)

raxis = load('raxis.txt'); % raxis as one column
zaxis = load('zaxis.txt'); % zaxis as one row
cd(defaultfolder)

zdist = zaxis(2) - zaxis(1); % spacing B field z axis (in mm)
rdist = raxis(2,1) - raxis(1,1); % spacing B field r axis (in mm)
bzextend = -zaxis(1); % dimension B field along decelerator z axis (in mm)
sizB = size(Bz_n,1);

if get_video == 1
    zaxh = linspace(0,disttodetect,120); % beam axis (in mm)
    vzaxh = linspace(0,vz0+0.4,30); % vz axis (in mm/mus)
    video = uint32(zeros(length(zaxh),length(vzaxh),round(stop_time/laserfrequency)));
    TOFarrival = zeros(round(stop_time/laserfrequency),2);
    [ZAXH,VZAXH] = meshgrid(zaxh,vzaxh); % important for "surf" - makes defined grid*
end

if ZeemanN == 32
    no_states = 4;
else % ZeemanN == 52
    no_states = 6;
end

% check off time only for "ordinary" deceleration (no re-pulsing)
checkofftime = zeros(sizcoil,1);
for nim = 1:sizcoil
    if nim == sizcoil
        checkofftime(nim) = coil(nim,3)+rampcoil;
    else
        checkofftime(nim) = coil(nim,3)+ramp1;
    end
end

if get_trajectories == 1
	psz = 6;   % z space
	psvz = 0.02; % vz space
	psx = 3;   % x space
	psvx = 0.01; % vx space
    
	res1 = 150;    
	res2 = 120;

	% z-vz phase space
	edges0 = [num2cell(linspace(-psz,psz,res1),2) ... % z
              num2cell(linspace(-psvz*1000,psvz*1000,res1),2)]; %vz
	[VZ,Z] = meshgrid(cell2mat(edges0(1,2)),cell2mat(edges0(1,1)));
	% x-vx phase space
    edges1 = [num2cell(linspace(-psx,psx,res2),2) ... % x
              num2cell(linspace(-psvx*1000,psvx*1000,res2),2)]; % vx
	[VX,X] = meshgrid(cell2mat(edges1(1,2)),cell2mat(edges1(1,1)));
    
    cd(newdir)
        dlmwrite('traj_Z.txt', Z,'delimiter', '\t', 'precision', fprec);
        dlmwrite('traj_VZ.txt', VZ,'delimiter', '\t', 'precision', fprec);
        dlmwrite('traj_X.txt', X,'delimiter', '\t', 'precision', fprec);
        dlmwrite('traj_VX.txt', VX,'delimiter', '\t', 'precision', fprec);
    cd(defaultfolder)
    
    traj_syn = [];

end

if get_focus == 1
    % matrices for trajectory tracing Halbach focusing, lfs1 only:
    zalongdec = linspace(0,enddecel,150);
    txmatrix = NaN(size(init_cond_all,1),length(zalongdec));
%     tymatrix = NaN(size(init_cond_all,1),length(zalongdec));
    tvxmatrix = NaN(size(init_cond_all,1),length(zalongdec));
%     tvymatrix = NaN(size(init_cond_all,1),length(zalongdec));
end

end


s1 = zeros(no_states,1);
s2 = zeros(no_states,1);
s3 = zeros(no_states,1);
s4 = zeros(no_states,1);
if ZeemanN == 52
    s5 = zeros(no_states,1);
    s6 = zeros(no_states,1);
end

for z = 1:no_states

Zeeman_effect = z;

% no spin flips, hence z = Zeeman_state
% MJ state for the calculation of the Zeeman effect
    if Zeeman_effect == 1
        MJ = 3/2;
    else if Zeeman_effect == 2
            MJ = 1/2;
        else if Zeeman_effect == 3
                MJ = -1/2;
            else if Zeeman_effect == 4
                MJ = -3/2;
                else if Zeeman_effect == 5
                MJ = -5/2;
                    else % if Zeeman_effect == 6
                MJ = 5/2;
                    end
                end
            end
        end
    end
    
tmp0 = sprintf('Please wait... Zeeman state: %d (press Ctrl-C to stop)\n', Zeeman_effect);
fprintf('%s',tmp0)


%% Trajectory simulation
%
% simulates all particles at once (matrix form - optimized for MATLAB)
% coils are checked depending on where a magnetic field has been detected 
% in the previous loop (exception: all coils turned off or particle is too 
% far off to experience the B field of a coil)
%
% B fields, gradients and accelerations are computed in polar coordinates
% r(,phi) instead of using x,y - this is possible due to the cylinder
% geometry of the coils
% the radial acceleration is then transformed to acceleration in the
% x and y direction again
% positions and velocities are calculated in cartesian coordinates x,y,z
% and vx,vy,vz
% x,y are then re-transformed into r,phi etc.
% i.e., this is a quasi-3D simulation

psget = 0;
stopinteg = 0;
s = 0; % timestep counter
time = 0; % initial time

timecon = min(coil(:,2)); % switch-on time first coil
timecoff = max(checkofftime)+2*timestep;
    % time at which last coil is completely ramped down
    % include time buffer for spin flip counting

% define initial conditions:

% p2sim = [xabs yabs zabs vx vy vz i Zeeman_effect time]
p2sim = [init_cond_all, ((ivi-1)*arr_particles(ivi,2)+1:ivi*arr_particles(ivi,2))', Zeeman_effect*ones(size(init_cond_all,1),1), zeros(size(init_cond_all,1),1)];
% particle index corresponds to row number in initial conditions matrix

colinit = size(p2sim,2); % initial number of columns for p2sim


%% Re-initialize matrices
%  for each MAGNETIC STATE:

countp = zeros(1,colinit);
% countp and dummy_countp are matrices for the arrival at detection 
% (2nd scheme)

if get_trajectories == 1
    traj = []; % initialize for taking trajectories
    checktraj = 0;
    ctr = 0;
end

if get_realtrajectories == 1
	traj_real = []; % initialize for taking trajectories
	checkrealtraj = 0;
end


while stopinteg == 0 && time <= stop_time
% iterate until all particles have been detected (none left)

s = s + 1; % begin with s = 1 since time(s=1) = 0*timestep;
time = t0 + s*timestep; % time in microsec

if mod(s,100) == 0 % progress report (every 100 timesteps)
    tmp = sprintf('Time (mus): %.2f\n', time);
    fprintf('%s',tmp)
end

sizp2sim = size(p2sim,1);

% transformation cartesian coordinates <--> polar coordinates
r = sqrt(p2sim(:,1).^2+p2sim(:,2).^2); % new r = sqrt(xabs.^2 + yabs.^2)

%% Re-initialize matrices for B fields
% for each (coil j) WHILE LOOP:
% need to calculate Br and Bz for every coil that is ramped and add up
% however, gradients must be calculated at once and CANNOT be summed up
% (add grid points instead)

% if magnetic field components are predefined before addressing each coil,
% I only need to add the magnetic fields to those particles affected

QAB_z_tot = zeros(sizp2sim,1);
QCD_z_tot = zeros(sizp2sim,1);
QAC_z_tot = zeros(sizp2sim,1);
QBD_z_tot = zeros(sizp2sim,1);

QAB_r_tot = zeros(sizp2sim,1);
QCD_r_tot = zeros(sizp2sim,1);
QAC_r_tot = zeros(sizp2sim,1);
QBD_r_tot = zeros(sizp2sim,1);

Bz_tot = zeros(sizp2sim,1);
Br_tot = zeros(sizp2sim,1);


%% DECELERATOR 
%

if decon == 1 && time >= timecon && time <= timecoff   
    
for j = 1:realnumbcoils


if j == 1
    edgemode = 1;
else if j == realnumbcoils
    edgemode = 2;
else
    edgemode = 0;
    end
end


% B field ramp 
% taking into account the finite coil switch-on/switch-off times 
% and differences in the real experimental and the simulated current
if edgemode == 1
        if time <= coil(j,2) || coil(j,2) == coil(j,3)
            rampfactor = 0;
        else if time > coil(j,2) && time < coil(j,2)+rampcoil % normal rise
            rampfactor = (current/simcurrent)*(1/rampcoil)*(time-coil(j,2));
        else if time >= coil(j,2)+rampcoil && time < coil(j,3)-timeoverlap % constant level
            rampfactor = (current/simcurrent);                
        else if time >= coil(j,3)-timeoverlap && time < coil(j,3) % overlap fall
            rampfactor = (current/simcurrent)*(m3*(time-coil(j,2))+(h2-m3*(coil(j,3)-coil(j,2))));               
        else if time >= coil(j,3) && time < coil(j,3)+ramp1 % rise 1 fall
            rampfactor = (current/simcurrent)*(m4*(time-coil(j,2))-m4*((coil(j,3)-coil(j,2))+ramp1));                
            else
            rampfactor = 0;
            end
            end
            end
            end
        end

else if edgemode == 2
        if time <= coil(j,2) || coil(j,2) == coil(j,3)
            rampfactor = 0;
        else if time > coil(j,2) && time < coil(j,2)+ramp1 % rise 1 rise
            rampfactor = (current/simcurrent)*(m1*(time-coil(j,2)));    
        else if time >= coil(j,2)+ramp1 && time < coil(j,2)+ramp1+timeoverlap % overlap rise
            rampfactor = (current/simcurrent)*(m2*(time-coil(j,2))+n2);
        else if time >= coil(j,2)+ramp1+timeoverlap && time < coil(j,3) % constant level
            rampfactor = (current/simcurrent);                
        else if time >= coil(j,3) && time < coil(j,3)+rampcoil % normal fall
            rampfactor = (current/simcurrent)*(1/rampcoil)*(coil(j,3)+rampcoil-time);
            else
            rampfactor = 0;
            end
            end
            end
            end
        end

    else if edgemode == 0
        if time <= coil(j,2) || coil(j,2) == coil(j,3)
            rampfactor = 0;
        else if time > coil(j,2) && time < coil(j,2)+ramp1 % rise 1 rise
            rampfactor = (current/simcurrent)*(m1*(time-coil(j,2)));    
        else if time >= coil(j,2)+ramp1 && time < coil(j,2)+ramp1+timeoverlap % overlap rise
            rampfactor = (current/simcurrent)*(m2*(time-coil(j,2))+n2);
        else if time >= coil(j,2)+ramp1+timeoverlap && time < coil(j,3)-timeoverlap % constant level
            rampfactor = (current/simcurrent);                
        else if time >= time < coil(j,3)-timeoverlap && time < coil(j,3) % overlap fall
            rampfactor = (current/simcurrent)*(m3*(time-coil(j,2))+(h2-m3*(coil(j,3)-coil(j,2))));               
        else if time >= coil(j,3) && time < coil(j,3)+ramp1% rise 1 fall
            rampfactor = (current/simcurrent)*(m4*(time-coil(j,2))-m4*((coil(j,3)-coil(j,2))+ramp1));                
            else
            rampfactor = 0;
            end
            end
            end
            end
            end
        end
        else % edgemode == 3
            if time <= coil(j,2) || coil(j,2) == coil(j,3)
                rampfactor = 0;
            else if time > coil(j,2) && time < coil(j,2)+rampcoil % normal rise
                rampfactor = (current/simcurrent)*(1/rampcoil)*(time-coil(j,2));
            else if time >= coil(j,2)+rampcoil && time < coil(j,3) % constant level
                rampfactor = (current/simcurrent);                
            else if time >= coil(j,3) && time < coil(j,3)+rampcoil % normal fall
                rampfactor = (current/simcurrent)*(1/rampcoil)*(coil(j,3)+rampcoil-time);
                else
                rampfactor = 0;
                end
                end
                end
            end
        end
    end
end
  
if fm == 4 % negative current after switch-off, NOT FOR LAST COIL!
           % instantaneous switch-on
    if j < realnumbcoils-1 && time >= coil(j+1,3) && time <= coil(j+2,3) % constant level
        rampfactor = (negcurrent/simcurrent);
    else if j == realnumbcoils-1 && time >= coil(j+1,3) && time <= coil(j+1,3) + rampcoil 
        rampfactor = (negcurrent/simcurrent);
        end
    end
end

%% B field gradients and acceleration
%

if rampfactor ~= 0
% include this additional criterium to make sure you get the magnetic field
% properties at the switch off even though the rampfactor is zero
% 2*timestep to prevent extensive looping
% rampfactor == 0: else do nothing

% choose particles that are inside this specific coil, mdec = 0 for all other particles
pincoil = uint32(find(p2sim(:,3)>=coil(j,1)-bzextend & p2sim(:,3)<=coil(j,1)+bzextend & r<=disttowall));
% -zdist and -rdist to account for use of "ceil" interpolation between grid
% positions z1/zp1 and r1/rp1

    % for particles inside the coil:
    if isempty(pincoil) ~= 1 % else do nothing
    
    % z particle position relative to coil centre
    % r particle position relative to coil centre
             zrel = p2sim(pincoil,3)-coil(j,1);

    % indices on grid (check by showing that zrel is between z1pos and zp1pos):        
             z1 = ceil((zrel+bzextend)/zdist);
             r1 = ceil(r(pincoil,1)/rdist);
             
             % shift particles at the boundaries of the grid back one position into the grid
             z1(z1==length(zaxis))=z1(z1==length(zaxis))-1;
             r1(r1==length(raxis))=r1(r1==length(raxis))-1;
             z1(z1==0)=z1(z1==0)+1;
             r1(r1==0)=r1(r1==0)+1;
             
             % prevent that particle positions on grid points are rounded down,
             % e.g. r = 0 should have r1 = 1 
             % Liz = ismember(zrel+bzextend,zaxis);
             % Lir = ismember(r(pincoil,1),raxis);
             % z1(Liz) = z1(Liz)+1;
             % r1(Lir) = r1(Lir)+1;
             % in the program itself, I find that this ONLY happens for
             % r=0, hence I only use the scheme used above
             % ismember is very time-intensive
            
    % positions on grid:        
             z1pos = zaxis(z1)';
             r1pos = raxis(r1);
             zp1pos = zaxis(z1+1)';
             rp1pos = raxis(r1+1);      
        
             % since Bz_n and Br_n have the same size, we can use the same
             % indexing for both
             % for information on indexing, see ref sub2ind (example 3)
             % idx1 = sub2ind(size(Bz_n),r1,z1);
             % idx2 = sub2ind(size(Bz_n),r1+1,z1);
             % idx3 = sub2ind(size(Bz_n),r1,z1+1);
             % idx4 = sub2ind(size(Bz_n),r1+1,z1+1);
             
             % this works much faster:
             % idx = rows + (cols-1)*size(M,1);
             idx1 = r1+(z1-1)*sizB;
             idx3 = r1+z1*sizB;
             
    % grid points Bz field:
             QA_z = Bz_n(idx1);
             QB_z = Bz_n(idx1+1);
             QC_z = Bz_n(idx3);
             QD_z = Bz_n(idx3+1);

    % grid points Br field:
             QA_r = Br_n(idx1);
             QB_r = Br_n(idx1+1);
             QC_r = Br_n(idx3);
             QD_r = Br_n(idx3+1);

             QAB_z = ((r(pincoil,1)-r1pos).*QB_z + (rp1pos-r(pincoil,1)).*QA_z)/rdist;
             QCD_z = ((r(pincoil,1)-r1pos).*QD_z + (rp1pos-r(pincoil,1)).*QC_z)/rdist;
             QAC_z = ((zrel-z1pos).*QC_z + (zp1pos-zrel).*QA_z)/zdist;
             QBD_z = ((zrel-z1pos).*QD_z + (zp1pos-zrel).*QB_z)/zdist;

             QAB_r = ((r(pincoil,1)-r1pos).*QB_r + (rp1pos-r(pincoil,1)).*QA_r)/rdist;
             QCD_r = ((r(pincoil,1)-r1pos).*QD_r + (rp1pos-r(pincoil,1)).*QC_r)/rdist;
             QAC_r = ((zrel-z1pos).*QC_r + (zp1pos-zrel).*QA_r)/zdist;
             QBD_r = ((zrel-z1pos).*QD_r + (zp1pos-zrel).*QB_r)/zdist;

    % Bfield components at P(r,zrel) with P = Bz,Br
             Bzval = ((r(pincoil,1)-r1pos).*QBD_z + (rp1pos-r(pincoil,1)).*QAC_z)/rdist; % Bz
             Brval = ((r(pincoil,1)-r1pos).*QBD_r + (rp1pos-r(pincoil,1)).*QAC_r)/rdist; % Br
    % or, equally : Bzval = rampfactor*((zrel-z1pos).*QCD_z + ...
    %                                     (zp1pos-zrel).*QAB_z)/zdist; % Bz
    %               Brval = rampfactor*((zrel-z1pos).*QCD_r + ...
    %                                     (zp1pos-zrel).*QAB_r)/zdist; % Br
    
             QAB_z_tot(pincoil,1) = QAB_z_tot(pincoil,1)+rampfactor*QAB_z;
             QCD_z_tot(pincoil,1) = QCD_z_tot(pincoil,1)+rampfactor*QCD_z;
             QAC_z_tot(pincoil,1) = QAC_z_tot(pincoil,1)+rampfactor*QAC_z;
             QBD_z_tot(pincoil,1) = QBD_z_tot(pincoil,1)+rampfactor*QBD_z;

             QAB_r_tot(pincoil,1) = QAB_r_tot(pincoil,1)+rampfactor*QAB_r;
             QCD_r_tot(pincoil,1) = QCD_r_tot(pincoil,1)+rampfactor*QCD_r;
             QAC_r_tot(pincoil,1) = QAC_r_tot(pincoil,1)+rampfactor*QAC_r;
             QBD_r_tot(pincoil,1) = QBD_r_tot(pincoil,1)+rampfactor*QBD_r;

    % add to total Bz and Br
             Bz_tot(pincoil,1) = Bz_tot(pincoil,1)+rampfactor*Bzval; % Bz
             Br_tot(pincoil,1) = Br_tot(pincoil,1)+rampfactor*Brval; % Br
 
    end
end

end

end


%% FRONT HALBACH HEXAPOLE ARRAY
%

% get hexapole fields and gradients:

pinfhex = [];
if fhexon == 1

Bphi_tot = zeros(sizp2sim,1);
gradBz_phi = zeros(sizp2sim,1);
gradBr_phi = zeros(sizp2sim,1);
gradBphi_z = zeros(sizp2sim,1);
gradBphi_r = zeros(sizp2sim,1);
gradBphi_phi = zeros(sizp2sim,1);
    
% phi only needed if front Halbach hexapole array or magnetic guide is on
    phi = atan2(p2sim(:,2),p2sim(:,1)); % new phi = atan2(yabs,xabs)
    
% choose particles that are inside the front Halbach hexapole array, mfhex1 = 0 and mfhex2 = 0 for all other particles
pinfhex = uint32(find(p2sim(:,3)>=dist2midfhex-bfhexextend & p2sim(:,3)<=dist2midfhex+bfhexextend));

if isempty(pinfhex) ~= 1 % else do nothing
    
    zrelhex = p2sim(pinfhex,3)-dist2midfhex; % z particle position relative to hexapole centre
    % r particle position relative to hexapole centre calculated in "Remove particles with"

    rep1 = (b3/r0hex^2).*cos(3*phi(pinfhex,1)).*r(pinfhex,1).^2;
    rep2 = (b3/r0hex^2).*sin(3*phi(pinfhex,1)).*r(pinfhex,1).^2;
    rep3 = (b15/r0hex^14).*cos(15*phi(pinfhex,1)).*r(pinfhex,1).^14;
    rep4 = (b15/r0hex^14).*sin(15*phi(pinfhex,1)).*r(pinfhex,1).^14;
    denum = (zrelhex/aGlaser).^4 + 1;

    A_hex = 1./denum.^2; % magnitude function = modified Glaser fct.
    dAdz_hex = -(8*zrelhex.^3)./(aGlaser^4.*denum.^3); 
    
% since (to date) front hexapole is the only device with Bphi
% component/gradients, these values can be used directly
    Bphi_tot(pinfhex,1) = A_hex*Brem.*(rep1+rep3); % Bphi
    gradBz_phi(pinfhex,1) = dAdz_hex*Brem.*r(pinfhex,1).*(rep1+rep3); % gradBz_phi_hex
    gradBr_phi(pinfhex,1) = A_hex*Brem*3.*(5*rep3+rep1); % gradBr_phi_hex
    gradBphi_z(pinfhex,1) = dAdz_hex*Brem.*(rep3+rep1); % gradBphi_z_hex
    gradBphi_r(pinfhex,1) = A_hex*Brem*2.*(rep1+7*rep3)./r(pinfhex,1); % gradBphi_r_hex
    gradBphi_phi(pinfhex,1) = -A_hex*Brem*3.*(5*rep4+rep2); % gradBphi_phi_hex
    
    Bz_hex = dAdz_hex*Brem.*r(pinfhex,1).*(rep2+rep4/5)/3;
    Br_hex = A_hex*Brem.*(rep2+rep4);

% must use grid point method instead of analytical solution to fit to
% decelerator simulation
    gradBz_z_hex = 24*Brem.*zrelhex.^2.*r(pinfhex,1).*(rep2/3+rep4/15)./(aGlaser^4.*denum.^3).*((4*zrelhex.^4)./(aGlaser^4.*denum)-1); % gradBz_z_hex
    gradBz_r_hex = dAdz_hex*Brem.*(rep4+rep2); % gradBz_r_hex
    gradBr_z_hex = dAdz_hex*Brem.*(rep4+rep2); % gradBr_z_hex
    gradBr_r_hex = A_hex*Brem*2.*(rep2+7*rep4)./r(pinfhex,1); % gradBr_r_hex

% use half-steps with gradients to get grid points
         QAB_z = Bz_hex-gradBz_z_hex*zdist/2;
         QCD_z = Bz_hex+gradBz_z_hex*zdist/2;
         QAC_z = Bz_hex-gradBz_r_hex*rdist/2;
         QBD_z = Bz_hex+gradBz_r_hex*rdist/2;
              
         QAB_r = Br_hex-gradBr_z_hex*zdist/2;
         QCD_r = Br_hex+gradBr_z_hex*zdist/2;
         QAC_r = Br_hex-gradBr_r_hex*rdist/2;
         QBD_r = Br_hex+gradBr_r_hex*rdist/2;

         QAB_z_tot(pinfhex,1) = QAB_z_tot(pinfhex,1)+QAB_z;
         QCD_z_tot(pinfhex,1) = QCD_z_tot(pinfhex,1)+QCD_z;
         QAC_z_tot(pinfhex,1) = QAC_z_tot(pinfhex,1)+QAC_z;
         QBD_z_tot(pinfhex,1) = QBD_z_tot(pinfhex,1)+QBD_z;

         QAB_r_tot(pinfhex,1) = QAB_r_tot(pinfhex,1)+QAB_r;
         QCD_r_tot(pinfhex,1) = QCD_r_tot(pinfhex,1)+QCD_r;
         QAC_r_tot(pinfhex,1) = QAC_r_tot(pinfhex,1)+QAC_r;
         QBD_r_tot(pinfhex,1) = QBD_r_tot(pinfhex,1)+QBD_r;
    
% add to total Bz and Br
    Bz_tot(pinfhex,1) = Bz_tot(pinfhex,1)+Bz_hex; % Bz
    Br_tot(pinfhex,1) = Br_tot(pinfhex,1)+Br_hex; % Br

end

end


% B field gradient gradBz
         gradBz_z = (QCD_z_tot-QAB_z_tot)/zdist;
         gradBz_r = (QBD_z_tot-QAC_z_tot)/rdist;

% B field gradient gradBr   
         gradBr_z = (QCD_r_tot-QAB_r_tot)/zdist;
         gradBr_r = (QBD_r_tot-QAC_r_tot)/rdist;
    

% total B field:
if fhexon == 0 || isempty(pinfhex) == 1
    Babs = sqrt(Bz_tot(:,1).^2 + Br_tot(:,1).^2);
else
    Babs = sqrt(Bz_tot(:,1).^2 + Br_tot(:,1).^2 + Bphi_tot(:,1).^2);
end


%% Determine acceleration
% use the values determined for the Zeeman effect


Bneq0 = uint32(find(Babs~=0)); % prevent division by zero
sizBneq0 = size(Bneq0,1);

if sizBneq0 ~= 0
    
    if fhexon == 0 || isempty(pinfhex) == 1

        gradBabs_z = (Bz_tot(Bneq0,1).*gradBz_z(Bneq0,1)+ Br_tot(Bneq0,1).*gradBr_z(Bneq0,1))./Babs(Bneq0,1);
        gradBabs_r = (Bz_tot(Bneq0,1).*gradBz_r(Bneq0,1)+ Br_tot(Bneq0,1).*gradBr_r(Bneq0,1))./Babs(Bneq0,1);

    else

        % 1/r for gradBabs_phi! NEW: 04/10/2014
        gradBabs_z = (Bz_tot(Bneq0,1).*gradBz_z(Bneq0,1) + Br_tot(Bneq0,1).*gradBr_z(Bneq0,1) + Bphi_tot(Bneq0,1).*gradBphi_z(Bneq0,1))./Babs(Bneq0,1);
        gradBabs_r = (Bz_tot(Bneq0,1).*gradBz_r(Bneq0,1) + Br_tot(Bneq0,1).*gradBr_r(Bneq0,1) + Bphi_tot(Bneq0,1).*gradBphi_r(Bneq0,1))./Babs(Bneq0,1);
        gradBabs_phi = 1./r(Bneq0).*(Bz_tot(Bneq0,1).*gradBz_phi(Bneq0,1) + Br_tot(Bneq0,1).*gradBr_phi(Bneq0,1) + Bphi_tot(Bneq0,1).*gradBphi_phi(Bneq0,1))./Babs(Bneq0,1);        

    end
    
   % derivative of the Zeeman effect in N:
	 dEZee = MJ*gj*muB; 


     accsum_r = -(gradBabs_r/0.001).*dEZee/mass*1E-9;
     if fhexon == 1 && isempty(pinfhex) ~= 1
        accsum_phi = -(gradBabs_phi/0.001).*dEZee/mass*1E-9;
     end
     
	% only those particles experience an acceleration that also experience a B field
    accsum_xyz = zeros(sizBneq0,3);

    % accsum_xyz = [accsum_x, accsum_y, accsum_z]
    % 
    %         accsum_z = -(gradBabs_z/0.001)*dEZee/mass*1E-9;
    %         accsum_x = (accsum_r*xabs - accsum_phi*yabs)/r;
    %         accsum_y = (accsum_r*yabs + accsum_phi*xabs)/r;

     % if r = 0, then accsum_x = accsum_y = 0 
     % (accsum_r and accsum_phi irrelevant)
     % prevent division by zero
     req0 = uint32(find(r(Bneq0) == 0));
     if isempty(req0) ~= 1
        accsum_xyz(req0,3) = -(gradBabs_z(req0)/0.001).*dEZee/mass*1E-9;
     end
     
     % if r different from 0
     rneq0 = uint32(find(r(Bneq0) ~= 0));
     if isempty(rneq0) ~= 1
         if fhexon == 0 || isempty(pinfhex) == 1
             accsum_xyz(rneq0,:) = [accsum_r(rneq0).*p2sim(Bneq0(rneq0),1)./r(Bneq0(rneq0),1), ...
                                    accsum_r(rneq0).*p2sim(Bneq0(rneq0),2)./r(Bneq0(rneq0),1), ...
                                    -(gradBabs_z(rneq0)/0.001).*dEZee/mass*1E-9];
         else
             accsum_xyz(rneq0,:) = [(accsum_r(rneq0).*p2sim(Bneq0(rneq0),1)-accsum_phi(rneq0).*p2sim(Bneq0(rneq0),2))./r(Bneq0(rneq0),1), ...
                                    (accsum_r(rneq0).*p2sim(Bneq0(rneq0),2)+accsum_phi(rneq0).*p2sim(Bneq0(rneq0),1))./r(Bneq0(rneq0),1), ...
                                    -(gradBabs_z(rneq0)/0.001).*dEZee/mass*1E-9];
         end
     end


%% Numerical integration of the equations of motion
% using the Velocity Verlet method

% no B field, no acceleration == no change in velocity
    if s > 1
        p2sim(Bneq0,4:6) = p2sim(Bneq0,4:6) + timestep*accsum_xyz; % vxyz = vhxyz + timestep*accsum_xyz;
    else % at s == 1, go half a velocity step
        p2sim(Bneq0,4:6) = p2sim(Bneq0,4:6) + 0.5*timestep*accsum_xyz; % vhxyz = vxyz + 0.5*timestep*accsum_xyz;
    end
    
end
p2sim(:,1:3) = p2sim(:,1:3) + timestep*p2sim(:,4:6); % absxyz = absxyz + timestep*vhxyz
% which is the same as:
% p2sim(:,1:3) = p2sim(:,1:3) + p2sim(:,4:6)*timestep + 0.5*accsum_xyz*timestep^2 % == "real" Velocity Verlet
%              = p2sim(:,1:3) + timestep*(p2sim(:,4:6) + 0.5*accsum_xyz*timestep)
% I tested the code vs a code with the "real" Verlet implementation (with a
% loop vVerl = 1:2) and the output was exactly the same as with this code
% hence: this code is equivalent to a "real" Verlet algorithm!
% but remember: zabs = zabs(s), vz = vz(s-1); 
% difference is negligible for small timesteps


%% Detection
%

% for "Stern-Gerlach exp."
% count particles when they have reached the detector no matter whether the laser
% is fired or not
% countp = [x y z vx vy vz i time]
% check whether particle is inside the laser beam: 
% check only z dimension, thorough analysis of detector height will be done
% in analysis file

% detect all particles in the z detection plane (or shortly behind)
indc = uint32(find(p2sim(:,3)>=disttodetect)); 
countp = vertcat(countp,[p2sim(indc,1:8) time*ones(length(indc),1)]);


if get_phasespace == 1 && psget == 0 && time >= finaltime
    % for phase-space distribution:
    % check whether particle has reached the end of the pulse sequence
    % ps = [xabs yabs zabs vx vy vz i time]
	ps = [p2sim(:,1:8), ones(sizp2sim,1)*time];
    
    psget = 1;
end

if get_trajectories == 1
    if time >= timecon && time <= timecoff % only take trajectories when the decelerator is actually turned on
        if mod(time-t0,laserfrequency) == 0 % time-t0 to really start at the beginning of the simulation
            
            if ivi == 1 && z == 1
                if p2sim(1,7) == 1 % index of synchronous particle should always remain 1
                    
                    % record trajectory of synchronous particle
                    traj_syn = vertcat(traj_syn,[p2sim(1,1:6) time Babs(1) gradBabs_z(1)]);
                    % traj_syn = [x y z vx vy vz time Babs dB/dz]
                    % for sync. particle, only dB/dz non-zero, gradBabs_z(1) = gradBz_z(1) 

                    if p2sim(1,8) ~= 1
                        error('Error:Synp','Synchronous particle not in Zeeman state 1. Check spin flip conditions.')
                    end
                else
                    error('Error:Synp','Synchronous particle not found.')
                end

                % find those particles that are inside the phase space acceptance region at this time
                % record positions and velocities relative to synchronous particle
                i4traj = uint32(find(abs(p2sim(:,3)-p2sim(1,3))<= psz & ...
                              abs(p2sim(:,6)-p2sim(1,6))<= psvz));
                traj = vertcat(traj,[p2sim(i4traj,1:6)-repmat(p2sim(1,1:6),length(i4traj),1),p2sim(i4traj,7:8),ones(length(i4traj),1)*time]);
            else
                ctr = ctr + 1;

                % find those particles that are inside the phase space acceptance region at this time
                % use the trajectory of the synchronous particle that was saved before
                % record positions and velocities relative to synchronous particle
                i4traj = uint32(find(abs(p2sim(:,3)-traj_syn(ctr,3))<= psz & ...
                              abs(p2sim(:,6)-traj_syn(ctr,6))<= psvz));
                traj = vertcat(traj,[p2sim(i4traj,1:6)-repmat(traj_syn(ctr,1:6),length(i4traj),1),p2sim(i4traj,7:8),ones(length(i4traj),1)*time]);
            end
                    
        end
    else if time >= timecoff && checktraj == 0 
            % at the end of the pulse seq., discard all particles that are lost from the simulation until then
            % (although they were within the phase space acceptance region at earlier times)
            utp = ismember(traj(:,7),p2sim(:,7)); % particles still in the simulation
            utd = ismember(traj(:,7),countp(:,7)); % particles already detected
            traj = traj(utp==1 | utd==1,:);            
            checktraj = 1;
        end
    end
end


if get_realtrajectories == 1
    if time >= timecon && time <= timecoff % only take trajectories when the decelerator is actually turned on
    % if time <= timecoff
        if mod(time,laserfrequency) == 0 % use time, not time-t0 to get output at the times that you desired
            traj_real = vertcat(traj_real,[p2sim(:,1:8),ones(size(p2sim,1),1)*time]);
        end
    else if time >= timecoff && checkrealtraj == 0
            % at the end of the pulse seq., discard all particles that are lost from the simulation until then
            utp = ismember(traj_real(:,7),p2sim(:,7)); % particles still in the simulation
            utd = ismember(traj_real(:,7),countp(:,7)); % particles already detected
            traj_real = traj_real(utp==1 | utd==1,:);
            checkrealtraj = 1;
        end
    end

end


if get_video == 1
    if mod(time-t0,laserfrequency) == 0 % time-t0 to really start at the beginning of the simulation
        coust = round((time-t0)/laserfrequency);
        video(:,:,coust) = video(:,:,coust) + uint32(hist3([p2sim(:,3),p2sim(:,6)*1000],{zaxh vzaxh})); % vz in m/s
        % standard hist3 (calculated for yours axis)
        % add for every Zeeman state at this time
        
        inda = uint32(find(abs(p2sim(indc,2)-detectorheight)<=radius_laser));
        siz3 = size(p2sim(inda,:),1); % count particles in laser volume

        TOFarrival(coust,1) = time;
        TOFarrival(coust,2) = TOFarrival(coust,2) + siz3;
    end
end


if get_focus == 1
if (ZeemanN == 52 && Zeeman_effect == 6) || (ZeemanN == 32 && Zeeman_effect == 1)
    [~,ind2bin]= histc(p2sim(:,3),zalongdec);
    fneq0 = find(ind2bin~=0);
    if isempty(fneq0) == 0
        % idx = rows + (cols-1)*size(M,1);
        idxx = p2sim(fneq0,7)+(ind2bin(fneq0)-1)*size(txmatrix,1); % convert to linear indices
        txmatrix(idxx) = p2sim(fneq0,1); % store x value
%         tymatrix(idxx) = p2sim(fneq0,2); % store y value
        tvxmatrix(idxx) = p2sim(fneq0,4); % store vx value
%         tvymatrix(idxx) = p2sim(fneq0,5); % store vy value
    end
end
end

%% Particle removal
% split the experiment into a number of different sections

% get r for all particles
% transformation cartesian coordinates <--> polar coordinates
r = sqrt(p2sim(:,1).^2+p2sim(:,2).^2); % new r = sqrt(xabs.^2 + yabs.^2)

% a) for the region inside the skimmer, keep only those inside the cone of
% the skimmer
idecide1 = uint32(find(p2sim(:,3)>= disttoskimmer & p2sim(:,3)<= disttoskimmer + skimmerlength));

if isempty(idecide1) ~= 1
    % for those particles: throw particles with alpha > alphaskim
    idel1 = uint32(find(atan((r(idecide1)-skimmerradius)./(p2sim(idecide1,3)-disttoskimmer)) > alphaskim));
else
    idel1 = [];
end

% c) for the region inside the decelerator
    idecide3 = uint32(find(p2sim(:,3) >= coil(1,1)-coillength/2 & p2sim(:,3) <= enddecel));
    idel3 = uint32(find(r(idecide3)>disttowall));

% d) delete particles behind the detection region
idel4 = uint32(find(p2sim(:,3) >= disttodetect));    

% delete those particles
if fhexon == 0
    idelall = [idecide1(idel1);idecide3(idel3);idel4];
else
    % b) for the region inside the hexapole (if relevant)
        idecide2 = uint32(find(p2sim(:,3) >= dist2hex & p2sim(:,3) <= dist2hex + Lhexblock));
        idel2 = uint32(find(r(idecide2) > r0hex));
    
    idelall = [idecide1(idel1);idecide2(idel2);idecide3(idel3);idel4];
end


if isempty(idelall) ~= 1
    p2sim(idelall,:) = 0;
    p2sim(~any(p2sim,2),:) = [];    
end


% no particles left
if size(p2sim,1) == 0
     stopinteg = 1;
else stopinteg = 0;
end


end

% delete zero rows from detection matrices:
countp = countp(any(countp,2),:);

% do some statistics:
% portion of particles that arrive at the detector (filtered):
% lab and office:
[dum1,irow,dum2] = unique(countp(:,7),'first');
% home:
% [~,irow,~] = unique(countp(:,7),'first');

% distinguish between different Zeeman states
state1 = uint32(find(countp(irow,8) == 1));
state2 = uint32(find(countp(irow,8) == 2));
state3 = uint32(find(countp(irow,8) == 3));
state4 = uint32(find(countp(irow,8) == 4));

s1(Zeeman_effect,1) = size(state1,1);
s2(Zeeman_effect,1) = size(state2,1);
s3(Zeeman_effect,1) = size(state3,1);
s4(Zeeman_effect,1) = size(state4,1);

if ZeemanN == 52
    state5 = uint32(find(countp(irow,8) == 5));
    state6 = uint32(find(countp(irow,8) == 6));
    s5(Zeeman_effect,1) = size(state5,1);
    s6(Zeeman_effect,1) = size(state6,1);
end


%% Write initial conditions and results into output folder
%

cd(newdir)

file1 = sprintf('posatend_%d_%d.txt',ivi,Zeeman_effect);
countp(:,4:6) = countp(:,4:6)*1000; % save velocities in m/s
dlmwrite(file1, countp, 'delimiter', '\t', ...
    'precision', fprec);

if get_phasespace == 1
    file3 = sprintf('ps_%d_%d.txt',ivi,Zeeman_effect);
    ps(:,4:6) = ps(:,4:6)*1000; % save velocities in m/s
    dlmwrite(file3, ps, 'delimiter', '\t', ...
        'precision', fprec);
    clear ps
end

if get_trajectories == 1
    
    if ivi == 1 && z == 1
        dlmwrite('traj_syn.txt', traj_syn, 'delimiter', '\t', ...
            'precision', fprec);    
    end
    
    file4 = sprintf('trajzvz_%d_%d.txt',ivi,Zeeman_effect);
    file5 = sprintf('trajxvx_%d_%d.txt',ivi,Zeeman_effect);
    file6 = sprintf('traji_%d_%d.txt',ivi,Zeeman_effect);
    
	Nzvz = zeros(size(VZ));
	Nxvx = zeros(size(VX));
    
    if isempty(traj) ~= 1
        
        traj(:,4:6) = traj(:,4:6)*1000; % velocities in m/s
        
        Nzvz = hist3([traj(:,3) traj(:,6)],'Edges',edges0);
        Nxvx = hist3([traj(:,1) traj(:,4)],'Edges',edges1);
        
    end
    dlmwrite(file4, Nzvz, 'delimiter', '\t', ...
        'precision', fprec);    
    dlmwrite(file5, Nxvx, 'delimiter', '\t', ...
        'precision', fprec);    
    dlmwrite(file6, traj(:,7:8), 'delimiter', '\t', ...
        'precision', fprec); 
    % remember the identity of all particles whose trajectories can be recorded
    % until the end

end

if get_realtrajectories == 1
    
	traj_real(:,4:6) = traj_real(:,4:6)*1000; % velocities in m/s

    file7 = sprintf('trajreal_%d_%d.txt',ivi,Zeeman_effect);
    dlmwrite(file7, traj_real, 'delimiter', '\t', ...
    'precision', fprec);    
end

clear countp irow traj Nzvz Nxvx traj traj_real
clear state1 state2 state3 state4

% go back to parent directory
cd(defaultfolder)

end

saves1(ivi,1) = sum(s1,1);  
saves2(ivi,1) = sum(s2,1);  
saves3(ivi,1) = sum(s3,1);  
saves4(ivi,1) = sum(s4,1);  
if ZeemanN == 52
    saves5(ivi,1) = sum(s5,1);  
    saves6(ivi,1) = sum(s6,1);  
end


if ivi == numStreams

s1all = sum(saves1,1)*1E6/(no_states*realno_particles);
s2all = sum(saves2,1)*1E6/(no_states*realno_particles);
s3all = sum(saves3,1)*1E6/(no_states*realno_particles);
s4all = sum(saves4,1)*1E6/(no_states*realno_particles);
if ZeemanN == 52
    s5all = sum(saves5,1)*1E6/(no_states*realno_particles);
    s6all = sum(saves6,1)*1E6/(no_states*realno_particles);
    allparticles = s1all+s2all+s3all+s4all+s5all+s6all;
else
    allparticles = s1all+s2all+s3all+s4all;
end

cd(newdir)

% write file for video data

if get_video == 1
    save('videofiles.mat', 'video', 'TOFarrival','ZAXH','VZAXH')
    clear video TOFarrival ZAXH VZAXH
end

if get_focus == 1
    zalongdec = zalongdec(1:end-1)-dist2midfhex;
    txmatrix = txmatrix(:,1:end-1);
%     tymatrix = tymatrix(:,1:end-1);
    tvxmatrix = tvxmatrix(:,1:end-1);
%     tvymatrix = tvymatrix(:,1:end-1);
    
    xrange = linspace(-disttowall,disttowall,50);
    vxrange = linspace(-25,25,50);

% 	save('focusfiles.mat','zalongdec','txmatrix','tymatrix','tvxmatrix','tvymatrix','xrange','vxrange')
%     clear zalongdec txmatrix tymatrix tvxmatrix tvymatrix xrange vxrange
	save('focusfiles.mat','zalongdec','txmatrix','tvxmatrix','xrange','vxrange')
    clear zalongdec txmatrix tvxmatrix xrange vxrange
end

% append results to file 'output_param.txt'
fid1 = fopen('output_param.txt','a');
fprintf(fid1, ...
    'particles of magnetic state 1 at detection (in ppm): %6.4f\n', s1all);
fprintf(fid1, ...
    'particles of magnetic state 2 at detection (in ppm): %6.4f\n', s2all);
fprintf(fid1, ...
    'particles of magnetic state 3 at detection (in ppm): %6.4f\n', s3all);
fprintf(fid1, ...
    'particles of magnetic state 4 at detection (in ppm): %6.4f\n', s4all);
if ZeemanN == 52
    fprintf(fid1, ...
        'particles of magnetic state 5 at detection (in ppm): %6.4f\n', s5all);
    fprintf(fid1, ...
        'particles of magnetic state 6 at detection (in ppm): %6.4f\n', s6all);
end
fprintf(fid1, ...
    'total number of particles at detection (in ppm): %6.4f\n', allparticles);
fprintf(fid1, ...
    'decon: %d\n', decon);
fprintf(fid1, ...
    'current: %d\n', current);
fprintf(fid1, ...
    'focusmode: %d\n', fm);
fprintf(fid1, ...
    'fhexon: %d\n', fhexon);
fclose(fid1);
% view contents of written file
type output_param.txt

% go back to parent directory
cd(defaultfolder)

end

end


%% Merge output from all loops:
% I know that this looks horrible but I should not leave the old output in 
% the workspace (too much use of memory)

cd(newdir)

% for initial conditions and gas pulse data

namecell = {'init_cond';'posatend_gaspulse';'ps_gaspulse'};

for ii = 1:size(namecell,1)
    
m2merge = [];
if ii == 3
    m2merge = 0;
end

for ivi = 1:numStreams

	file1 = char(strcat(namecell(ii),'_',num2str(ivi),'.txt'));
    if exist(file1,'file') == 2
        fsign = 1;
        FileInfo = dir(file1);
        FileSize = FileInfo.bytes;
        if FileSize > 0 % do not try to read empty files
            m2 = dlmread(file1,'\t');
            m2merge = vertcat(m2merge,m2);
        end
        delete(file1)
    else 
        fsign = 0; % do not write unnecessary files
    end
    
end

if fsign == 1
    nfile1 = char(strcat(namecell(ii),'.txt'));
    dlmwrite(nfile1,m2merge,'delimiter','\t','precision', fprec);
end

end


% for Zeeman data

namecell = {'posatend_';'ps_';'trajzvz_';'trajxvx_';'traji_';'trajreal_'};

for ii = 1:size(namecell,1)

for z = 1:no_states

m2merge = [];

if get_trajectories == 1
	if ii == 3
        	m2merge = zeros(size(VZ));
    else if ii == 4
            m2merge = zeros(size(VX));
        end
	end
end 

for ivi = 1:numStreams

	file1 = char(strcat(namecell(ii),num2str(ivi),'_',num2str(z),'.txt'));
    if exist(file1,'file') == 2
        fsign = 1;
        FileInfo = dir(file1);
        FileSize = FileInfo.bytes;
        if FileSize > 0 % do not try to read empty files
            m2 = dlmread(file1,'\t');
            if ii ~= 3 && ii ~= 4
                 m2merge = vertcat(m2merge,m2);
            else
                 m2merge = m2merge + m2; % for trajectories
            end
        end
        delete(file1)
    else 
        fsign = 0;
    end

end

if fsign == 1
    nfile1 = char(strcat(namecell(ii),num2str(z),'.txt'));
    dlmwrite(nfile1,m2merge,'delimiter','\t','precision', fprec);
end

end

end


cd(defaultfolder)


timerEnd = toc(timerStart);
fprintf('Time elapsed: %d minutes and %d seconds\n',floor(timerEnd/60),round(rem(timerEnd,60)));

if isunix == 1
    exit; 
end