function ternErrors(xlsfile,fullformat)
% function ternErrors(xlsfile,fullformat)
% generate errorbars (in a csv file) for x,y data generated for a ternary diagram based on
% the number of counts for that cluster. Uses data generated by IVAS that
% has been run through 'clusterDataReader' to decompose the cluster counts
% and produce an xls file
% the FULLFORMAT flag chooses between outputing all of the data (1) or just
% the x, xerr, y, yerr and radius of the clusters (0).
% This function assumes that rg is in the last column
comp1 = 'Y'; % component 1
comp2 = 'Ti';% component 2

% read xls file
[ndata, text] = xlsread(xlsfile);

rg = ndata(:,end); % separate rg column
ndata = ndata(:,1:end-1);
text = text(1:end-1); % remove rg from the headers
% generate totals
tot = sum(ndata(:,3:5),2);

% found x column
pos = find(strcmp(text,'x'));
x = ndata(:,pos(1));

% found y column
pos = find(strcmp(text,'y'));
y = ndata(:,pos(1));


pos = find(strcmp(text,comp1));
if isempty(pos)
    disp(strcat('Component:',comp1,' not found'));
else
    col1 = ndata(:,pos(1));% found counts of Yttrium
end

pos = find(strcmp(text,comp2));
if isempty(pos)
    disp(strcat('Component:',comp2,' not found'));
else
    col2 = ndata(:,pos(1));% found counts of Titanium
end

% calculate errors
col1err = (col1./tot).*sqrt((1./col1) + (1./tot)); % actual error in Y fraction
col2err = (col2./tot).*sqrt((1./col2) + (1./tot)); % actual error in Ti fraction

col1err(isnan(col1err)) = 0; % check for clusters with 0 counts
col2err(isnan(col2err)) = 0;

xerr = sqrt(col1err.^2 + (col2err/2).^2);
yerr = (col2err*0.866);

% write file
filename = regexprep(xlsfile,'.xls','_errs.csv');
disp(strcat('Writing file:',filename));
fid = fopen(filename, 'w');

if(fullformat)
    % headers
    for s = 1:length(text)
        fprintf(fid,'%s,',text{s});
    end
    fprintf(fid,'xerr,yerr,rg\n');% end of line for headers
    
    for row=1:length(ndata(:,1))
        for col = 1:length(ndata(1,:))
            fprintf(fid, '%f,',ndata(row,col));
        end
        fprintf(fid, '%f,%f,%f\n',xerr(row),yerr(row),rg(row));
    end
else
    % use reduced format
    % Just print x,xerr,y,yerr,rg
    % headers
    fprintf(fid,'x,xerr,y,yerr,rg\n');
    for row = 1:length(ndata(:,1))
        fprintf(fid,'%f,%f,%f,%f,%f\n',x(row),xerr(row),y(row),yerr(row),rg(row));
    end
end
fclose(fid);
%end