function result = rangeCorrector3(rangeFile,posFile,level)
% result = rangeCorrector(rangeFile,posFile,level)
% writes a new "*_level.rrng" range file that has been optimised
% result is in the format:
% 1          2        3   4      5    6    7    8     9         10        11
% rangeStart rangeEnd vol colour newR newC newL count noisePre  noisePost gausWidth[p(2)] asym[p(3)] yscale[p(4)] noiseFloor[p(5)] element1 ... elementN
% rrng file format
% [Ions]
% Number=9
% Ion1=Fe
% Ion2=C
% Ion3=N
% ...
% Ion9=W
% [Ranges]
% Number=59
% Range1=27.9475 27.9913 Vol:0.01177 Fe:1 Color:FF00FF
% Range2=55.8950 55.9738 Vol:0.01177 Fe:1 Color:FF00FF
% ...
% Range24=69.9125 69.9738 Vol:0.04060 Fe:1 O:1 Color:FF0000
% Range59=23.9750 24.0363 Vol:0.01757 C:2 Color:660066
% If it fails when peak finding, increase peakwidth (pw) [default=0.3]
% You could also increase the minium signal2noise ratio for fancy fitting
% [default = 3]
% You can also pass {fileName, element_num, range_num, elements, ranges} as returned
% by rangeReader if you want to modify the range file first (see
% rangeCheck for example).

pausing = 0; % halt at errors? 0/1 = no/yes
plotting = 1;% plot histogram? 0/1 = no/yes
noiseWinidow = 1; % the number of range-widths away the noise window is
noiseWinSym = 3; % how many times larger is the post peak window? 2 = -/\--
noiseWinMax = 0.5; % max width peak fitting window
sig2noise = 0.5; % minimum signal to noise ratio
minRangeCounts = 10; % minimum range counts to attempt fitting/quantification
minCeiling = 3000; % instead of sig2noise can use a absolute counts value
pw = 5; % peak width for fitting - which is multiplied by an approximate
asymGuess = 0.03;
peakBroadening = 0.012; % Scale factor to widen trail peaks for fitting
trailPeakHeight = 130; % Scale factor for trial peak height
trailingEdgeScale = 1; % the width of a range when the trailing edge is not found
% guess for the gaussian width of the peak (which is proportional to ...
% the mass/z value).
num_bins = 100000; % mass resolution = massSpecMax/num_bins
massSpecMax = 250; % maximum mass spectrum value for small pos files

% check input
if ~exist('level','var') % level suplied?
    level = 0.5;
elseif level > 1 % level > 1?
    disp(strcat('level set > 1, did you mean:',num2str(level/100),'?'));
    level = level/100;
elseif level < 0 || ~isnumeric(level) % other errors
    level = 0.5;
end

% Read rrng file
if ischar(rangeFile)
    [element_num, range_num, elements, ranges] = rangeReader (rangeFile);
elseif length(rangeFile) == 5
    element_num = rangeFile{2};
    range_num   = rangeFile{3};
    elements    = rangeFile{4};
    ranges      = rangeFile{5};
    rangeFile   = rangeFile{1}; % set rangeFile = fileName
else
    [element_num, range_num, elements, ranges] = rangeReader (rangeFile);
end

% Check for file errors:
if sum(sum(cell2mat(ranges(:,5:end)),2)==0)
    zeroRanges = sum(cell2mat(ranges(:,5:end)),2)==0;
    zeroElements = sum(cell2mat(ranges(:,5:end)),1)==0;
    missingRanges = strcat(cellfun(@(x) {num2str(x)}, num2cell(find(zeroRanges))),', ');
    warning(['Ranges:' horzcat(missingRanges{:}) ' are missing *elements*']);
    disp('Removing them...');
    disp(elements(zeroElements));
    ranges = ranges(~zeroRanges,logical([1 1 1 1 ~zeroElements]));
    range_num = size(ranges,1);
    elements = elements(~zeroElements);
    element_num = length(elements);
end

ranges = sortrows(ranges,1);
% estimate the centre of the ranges (using the asymmetry value)
rangeCenters = [0; (cell2mat(ranges(:,1))*noiseWinSym+cell2mat(ranges(:,2)))./(noiseWinSym+1); massSpecMax];
% The boundaries for fitting of each of the ranges
% this is used for detecting overlaps
%rangeBounds = zeros(range_num,2);
a=min(diff(rangeCenters),noiseWinMax);
rangeBounds = [rangeCenters(2:end-1)-a(1:end-1)*(1-(noiseWinSym/(noiseWinSym+1))) rangeCenters(2:end-1)+a(2:end)*(noiseWinSym/(noiseWinSym+1))];
clear a
% Make ranges bigger to store the extra information
extraCols = 14;
newRanges = cell(range_num,element_num+extraCols); % make an array to store the range details
% 1          2        3   4      5    6    7    8        9         10    11
% rangeStart rangeEnd vol colour newR newC newL noisePre noisePost count gausWidth[p(2)] asym[p(3)] yscale[p(4)] noiseFloor[p(5)] element1 ... elementN
% Copy original info
newRanges(:,1:4) = ranges(:,1:4);
% Copy ion info
newRanges(:,(extraCols+1):(extraCols+element_num)) = ranges(:,5:end);
% Assign new ranges var
ranges = newRanges;

if plotting
    % Plot histogram
    [h, heights, centres]=rangePlotter(rangeFile,posFile,num_bins);
    %set(gca,'yscale','log'); % **use log scale**
else
    % pos file mass cache
    masses = loadMasses(posFile);
    % bin the data
    % hack for clsuter pos files that have little data
    masses(end) = 200;
    masses(end-1) = 0;
    % / hack
    [heights, centres] = hist(masses,num_bins); % this plots a histogram
end
% measure bin width
binwidth = centres(2)-centres(1);

% cycle through each range
for r = 1:range_num
    % for counting
    rangeWidth = round(1+(ranges{r,2})/binwidth)-(round((ranges{r,1})/binwidth));
    thisHits = 0;
    preNoiseHits = 0;
    postNoiseHits = 0;
    for i = (1+round((ranges{r,1})/binwidth)):round((ranges{r,2})/binwidth)
        % while we're here, count the hits in this range:
        % and the hits for the pre & post noise windows
        thisHits = thisHits + heights(i);
        preNoiseHits = preNoiseHits + heights(i-rangeWidth*noiseWinidow);
        postNoiseHits = postNoiseHits + heights(i+rangeWidth*noiseWinidow);
    end
    ranges{r,8} = thisHits; % total hits in this range
    ranges{r,9} = preNoiseHits; % total hits in this range
    ranges{r,10} = postNoiseHits; % total hits in this range
end


% fit model peak
% correct ranges
% write new ranges
% loop through ranges
failCount = 0; % number of ranges with sig:noise < sig2noise (default 3)
for r = 1:range_num
    % range width in number of bins
    rangeWidth = round(1+(ranges{r,2})/binwidth)-(round((ranges{r,1})/binwidth));
    % find peaks
    if ( round(1+ranges{r,2}/binwidth)-round(ranges{r,1}/binwidth) ) > 3
        [pks, locs] = findpeaks(heights((round(ranges{r,1}/binwidth)):round(1+ranges{r,2}/binwidth)),'minpeakdistance',rangeWidth-1,'minpeakheight',heights(round(ranges{r,1}/binwidth))-1);
    else
        % too few bins to find a peak, so expand the original range width
        disp('**Too few bins for good fitting!**');
        [pks, locs] = max( heights((round(ranges{r,1}/binwidth)-1):round(2+ranges{r,2}/binwidth)) );
    end
    if length(locs)~=1
        % use a dumb maximum instead, this behaviour is expected when the
        % range doesn't have a peak in it though!
        [pks,locs]=max(heights((round(ranges{r,1}/binwidth)):round(1+ranges{r,2}/binwidth)));
        %error(strcat('no peak found:',num2str(r),'; try reranging original file'));
    end
    %ranges(r,5) % right
    %ranges(r,6) % centre
    %ranges(r,7) % left
    %centre of range as defined originally in bin-space
    cor = round(((ranges{r,1}+ranges{r,2})/2)/binwidth)+1; % in bins
    % for small peaks the centre of the orginial ranges is probably better*
    % *citation needed
    xbin = locs+round(ranges{r,1}/binwidth)-1; % bin number of range centre
    ranges{r,6}=centres(xbin); % make a note of range centre for later
    % if new peak isn't much bigger than the average choose centre
    if (ranges{r,8} >= (sig2noise*(ranges{r,9}+ranges{r,10})/2))||(heights(xbin)>minCeiling) && ranges{r,8}>minRangeCounts
        % In this case the new peak is better (high sig2noise and enough
        % counts)
        if plotting
            % plot markers
            hold on
            plot(centres(xbin),pks,'k^','markerfacecolor',[0 1 0]);
            hold off
        end
        % impirical guess of guassian width
        %gausWidth = (sqrt(ranges{r,1})*0.014 + 0.1)*peakBroadening;
        gausWidth = sqrt(ranges{r,1})*peakBroadening;
        pwg = pw*gausWidth; % Peak-width guess based on gaussian width
        pwb = round(pwg/binwidth); % peak width in bins
        pwb2 = round(pwg*.4/binwidth); % use less-than half width for the start in bins
        
        % detect collisions:
        if r>1
            if ranges{r-1,2}>centres(xbin-pwb2)
                pwb2 = round((centres(xbin)-ranges{r-1,2})/binwidth);
            end
        end
        if r<range_num
            if ranges{r+1,1}<centres(xbin+pwb)
                pwb = round((ranges{r+1,1}-centres(xbin))/binwidth);
            end
        end
        
        % remove local background
        % fit pre window noise using exp1
        % note the currently fixed window width
        preNoiseWindowStart = (ranges{r,2}-ranges{r,1})*1.5; %0.3; % relative to range start, ie 0.3 before
        if preNoiseWindowStart < 0.1
            preNoiseWindowStart = 0.1;
        end
        
        preNoiseWindowEnd = (ranges{r,2}-ranges{r,1})*0.5; %0.08; % relative to range start
        if preNoiseWindowEnd < 0.05
            preNoiseWindowEnd = 0.05;
        end
        
        % select the x y data to fit the noise curve to
        preNoiseX = centres( (1+round((ranges{r,1}-preNoiseWindowStart)/binwidth)):round((ranges{r,1}-preNoiseWindowEnd)/binwidth) )';
        preNoiseY = heights( (1+round((ranges{r,1}-preNoiseWindowStart)/binwidth)):round((ranges{r,1}-preNoiseWindowEnd)/binwidth) )';
        rangeX = centres( (1+round((ranges{r,1})/binwidth)):round((ranges{r,2})/binwidth) );
        
        % Fit to an exp in sqrt space:
        [expNoiseFit,~,~] = noiseFit(double(preNoiseX),double(preNoiseY),rangeX);
        
        fitx = rangeBounds(r,1):.001:rangeBounds(r,2);
        %plot(fitx,expNoiseFit(fitx.^2),'r-.'); % noise fit plot
        correctedHeights = heights(xbin-pwb2:xbin+pwb) - expNoiseFit(centres(xbin-pwb2:xbin+pwb).^2)';
        maxHeight = max(correctedHeights);
        correctedHeights = correctedHeights/maxHeight;
        % use nlinfit to fit a peak shape
        % initial guess = beta0
        noiseEst = 0; %min(heights(xbin-(0:pwb2))); % or heights(xbin-pwb2)
        %beta0 = [centres(xbin) gausWidth .02 (pks(1)-noiseEst)*100 noiseEst];
        %beta0 = [centres(xbin)-gausWidth gausWidth .02 (pks(1)-noiseEst)*trailPeakHeight];
        % removed centre off-set
        %beta0 = [centres(xbin) gausWidth .02 (pks(1)-noiseEst)*trailPeakHeight];
        beta0 = [0 gausWidth asymGuess 1];
        %disp(ranges{r,1}); % display current range for debugging
        %construct an error message incase it fails:
        ME = MException('rngcorrector:nlinfit', strcat('failed at range:',num2str(ranges{r,1})));
        try % fitting with asyGauss
            % plot initial guess
            %plot(fitx,asyGauss(beta0(1:5),fitx),'r-.');
            % fitting
            %beta1 = nlinfit(centres(xbin-pwb2:xbin+pwb), correctedHeights, @asyGauss, beta0,statset('DerivStep',[0.00005 eps^(1/3) eps^(1/3) 1 0.5],'Display','off'));
            %  lastwarn('blank');
            beta1 = nlinfit(centres(xbin-pwb2:xbin+pwb)-centres(xbin), correctedHeights, @asyGaussNoNoise, beta0,statset('DerivStep',[0.00005 eps^(1/3) eps^(1/3) 1],'Display','off'));
            %  lastwarnCheckErr('blank',['Fitting failed for range:' num2str(ranges{r,1})]);
            
            if plotting
                % single asymGaus fitting:
                hold on
                plot(fitx,asyGaussNoNoise(beta1(1:4),fitx-centres(xbin))*maxHeight+expNoiseFit(fitx.^2)','g-');
                hold off
            end
            % save fitting results
            ranges{r,11} = beta1(2);
            ranges{r,12} = beta1(3);
            ranges{r,13} = beta1(4);
            ranges{r,14} = 0;%beta1(5);
            % find new range edges
            markers = fwhm2(fitx,asyGaussNoNoise(beta1,fitx),level);
        catch ME % fitting with gauss
            % fit gaussian instead
            % beta1 = [amplitude centre standardDev];
            gaussFit = fit(double((centres(xbin-pwb2:xbin+pwb).^2)'), double(correctedHeights'), 'gauss1', 'lower',[0 centres(xbin).^2-10 0],'upper',[100 centres(xbin).^2+10 100],'startPoint',[1 centres(xbin).^2 0.3]);
            % assert gaussFit.a1>0
            beta1 = [sqrt(gaussFit.b1) sqrt(gaussFit.c1) gaussFit.a1 NaN];
            if plotting
                % single asymGaus fitting:
                hold on
                plot(fitx,gaussFit(fitx.^2)*maxHeight,'g-.');
                hold off
            end
            % save fitting results
            ranges{r,11} = beta1(2);
            ranges{r,12} = beta1(3);
            ranges{r,13} = beta1(4);
            ranges{r,14} = 0;%beta1(5);
            % find new range edges
            disp(r);
            if r==72
            disp(r);
            end
            markers = fwhm2(fitx,gaussFit(fitx.^2),level);
        end
        % set new ranges
        % new right, left and centre positions
        if(isfinite(markers(1)))
            ranges=rangeAssign(ranges,r,markers(1),centres(xbin)); % left
        else
            %error('rngCor:leadingEdge',strcat('Failed to find leading edge for peak:',num2str(centres(xbin))));
            warning(strcat('Failed to find leading edge for peak:',num2str(centres(xbin))));
            disp('Using original value instead, this is incorrect behaviour');
            ranges=rangeAssign(ranges,r,ranges{r,1},centres(xbin));
        end
        
        if(isfinite(beta1(1)))
            ranges{r,6} = beta1(1); % centre
        else
            error('rngCor:centre',strcat('Failed to find centre for peak:',num2str(centres(xbin))));
        end
        
        if(isfinite(markers(2)))
            ranges=rangeAssign(ranges,r,markers(2),centres(xbin)); % right
        else
            warning(strcat('Using original range right for range:',num2str(r),':',num2str(ranges{r,6})));
            ranges=rangeAssign(ranges,r,ranges{r,2}); % use original value
        end
    else
        % middle position is better
        if plotting
            hold on
            plot(centres(cor),heights(cor)*1.1,'k^','markerfacecolor',[0 0 1]);
            % mark the found peak for lolz
            plot(centres(xbin),pks*1,'k^','markerfacecolor',[1 0 0]);
            hold off
        end
        ranges{r,6} = centres(xbin); % save wrong centre position for reference
        failCount = failCount+1;
    end
end
% range width estimation
% count the number of missing rows in ranges
notEmtyRanges = logical(~(cell2mat(cellfun(@isempty,ranges(:,5),'uni',0)) | cell2mat(cellfun(@isempty,ranges(:,7),'uni',0))));
count = sum(notEmtyRanges); % number of 'full' ranges - ie not empty
if sum(~notEmtyRanges) % if there are missing rows
    % get the pre and post widths from known ranges
    disp('pre and post');
    ypre = cell2double(ranges(notEmtyRanges,6))-cell2double(ranges(notEmtyRanges,5));
    ypost = cell2double(ranges(notEmtyRanges,5))-cell2double(ranges(notEmtyRanges,7));
    x = cell2double(ranges(notEmtyRanges,6)); % range centres
    % convert to matrix and fit pre & post widths
    lastwarn('blank');
    if length(ypre)==1
        % make sure there at at least 2 data points
        ypre = [0 ypre];
        ypost = [0 ypost];
        x = [0 x];
    end
    fpre = fit(sqrt(x), ypre, 'poly1','Robust', 'LAR');
    fpost = fit(sqrt(x), ypost, 'poly1','Robust', 'LAR');
    lastwarnCheck('blank','Fitting failed for peaks for with poor signal-to-noise');
    % go back through and fit approximate ranges to the poorly defined ranges
    % the centres should stay in the same place, so average pre and post widths
    % The pre/post distinction is still there if you want to use it though
    for r=1:range_num
        if ~notEmtyRanges(r)
            centre = ranges{r,6};%(ranges{r,1}+ranges{r,2})/2;
            pre = sqrt(centre)*fpre.p1+fpre.p2;
            post = sqrt(centre)*fpost.p1+fpost.p2;
            halfWidth = (pre+post)/2;
            % should the new range contain the old range centre?
            %disp([num2str(centre-halfWidth) ':' num2str(centre) ':' num2str(centre+halfWidth)]);
            ranges=rangeAssign(ranges,r,centre-halfWidth,centre);
            ranges=rangeAssign(ranges,r,centre+halfWidth,centre);
        end
    end
end
if plotting
    hold on
    % plot new ranges on histogram
    % create array for range bounds
    lines2 = ones(range_num*6,2)*.1; % fill in all values 0.1
    % cycle through each range
    
    for r = 1:range_num
        % create lines
        %disp(ranges{r,5}); % DEBUG
        %disp(ranges{r,7}); % DEBUG
        lines2((r-1)*6+1,1) = ranges{r,5}-binwidth/5;
        lines2((r-1)*6+2,1) = ranges{r,5};
        lines2((r-1)*6+3,1) = ranges{r,5}+binwidth/5;
        lines2((r-1)*6+4,1) = ranges{r,7}-binwidth/5;
        lines2((r-1)*6+5,1) = ranges{r,7};
        lines2((r-1)*6+6,1) = ranges{r,7}+binwidth/5;
        lines2((r-1)*6+2,2) = heights(1+round(ranges{r,5}/binwidth));
        lines2((r-1)*6+5,2) = heights(round(ranges{r,7}/binwidth));
    end
    % sort lines
    sortedlines2 = sortrows(lines2);
    %hold on;
    % plot black lines where the edges of the current ranges are
    plot(sortedlines2(:,1),sortedlines2(:,2),'k');
    hold off;
end
if(failCount > 1)
    disp(strcat(num2str(failCount),' ranges had a signal:noise<',num2str(sig2noise),'or had too few counts for quantification'))
    result = failCount;
elseif (failCount == 1)
    disp(strcat(num2str(failCount),' range had a signal:noise<',num2str(sig2noise),' or had too few counts for quantification'))
    result = failCount;
else
    result = failCount;
end

% optional sort rows:
ranges = sortrows(ranges,1);

% check each range contains the ion it says it does
[ionicRangeTab,ionStr,ions]=elemental2ionicRangeTab(cell2mat(ranges(:,(end-element_num+1):end)),elements,ranges(:,5));
for r=1:size(ranges,1)
    possIons = find(ionicRangeTab(r,:));
    peaks = peakHeights(ions(possIons));
    % check for at least one of peaks in this range
    check = peaks(:,1)>cell2mat(ranges(r,5)) & peaks(:,1)<cell2mat(ranges(r,7));
    %disp(ionStr(possIons));
    if sum(check)==0
        warning(['missing theoretical ion position in range:' num2str(r)]);
        diffs = zeros(size(peaks,1),2);
        for p=1:size(peaks,1)
            diffs(p,:) = [ranges{r,5} ranges{r,7}]-peaks(p,1);
        end
        [~,ii]=min(abs(diffs(:))); % find which isotope and range bound
        shift = diffs(ii);
        [row,col]=ind2sub(size(diffs),ii);
        % shift by an extra 10% to be on the save side
        if col==1 % lower range
            newRange = ranges{r,5}-shift*1.1;
        else % upper range
            newRange = ranges{r,7}-shift*1.1;
        end
        ranges=rangeAssign(ranges,r,newRange,ranges{r,6});
    end
end
% return result
result = ranges; % set returned result to ranges



if(1) % write new (rrng) range file
    % get file name, without extension
    newName = regexprep(rangeFile,'\.rrng','','ignorecase');
    newName = strcat(newName,'_',num2str(level*100));
    exportRanges = cell(length(ranges(:,1)),(4+element_num));
    exportRanges(:,1) = ranges(:,5); % range start
    exportRanges(:,2) = ranges(:,7); % range end
    exportRanges(:,3:4) = ranges(:,3:4); % volume and colour
    % element-ion matrix
    
    %disp(strcat('5:',num2str(5+element_num),',',num2str(extraCols+1),':',num2str(size(ranges))));
    %disp(elements);
    %disp(ranges(1,:));
    %disp('export ranges');
    %disp(size(exportRanges));
    %disp('ranges');
    %disp(size(ranges));
    exportRanges(:,5:(4+element_num)) = ranges(:,(end-element_num+1):end);
    % if there are miss matching ions and elements the above will fail
    %disp(exportRanges);
    newRrng = rangeWriter(exportRanges,elements,newName);
    disp(strcat('Wrote new rrng file:',newRrng));
end
end