function result = rangeCorrector2(rangeFile,posFile,level)
% result = rangeCorrector(rangeFile,posFile,level)
% writes a new "*_level.rrng" range file that has been optimised
% result is in the format:
% 1          2        3   4      5    6    7    8        9         10    11
% rangeStart rangeEnd vol colour newR newC newL noisePre noisePost count gausWidth[p(2)] asym[p(3)] yscale[p(4)] noiseFloor[p(5)] element1 ... elementN
% rrng file format
% [Ions]
% Number=9
% Ion1=Fe
% Ion2=C
% Ion3=N
% ...
% Ion9=W
% [Ranges]
% Number=59
% Range1=27.9475 27.9913 Vol:0.01177 Fe:1 Color:FF00FF
% Range2=55.8950 55.9738 Vol:0.01177 Fe:1 Color:FF00FF
% ...
% Range24=69.9125 69.9738 Vol:0.04060 Fe:1 O:1 Color:FF0000
% Range59=23.9750 24.0363 Vol:0.01757 C:2 Color:660066
% If it fails when peak finding, increase peakwidth (pw) [default=0.3]
% You could also increase the minium signal2noise ratio for fancy fitting
% [default = 3]
% You can also pass {fileName, element_num, range_num, elements, ranges} as returned
% by rangeReader if you want to modify the range file first (see
% rangeCheck for example).

pausing = 1; % halt at errors? 0/1 = no/yes
plotting = 1;% plot histogram? 0/1 = no/yes
noiseWinidow = 1; % the number of range-widths away the noise window is
sig2noise = 1.5; % minimum signal to noise ratio
minCeiling = 3000; % instead of sig2noise can use a absolute counts value
pw = 5; % peak width for fitting - which is multiplied by an approximate
peakBroadening = 0.01; % Scale factor to widen trail peaks for fitting
trailPeakHeight = 130; % Scale factor for trial peak height
trailingEdgeScale = 1; % the width of a range when the trailing edge is not found
% guess for the gaussian width of the peak (which is proportional to ...
% the mass/z value).
num_bins = 200000; % mass resolution = massSpecMax/num_bins
massSpecMax = 200; % maximum mass spectrum value for small pos files

% check input
if ~exist('level','var') % level suplied?
    level = 0.5;
elseif level > 1 % level > 1?
    disp(strcat('level set > 1, did you mean:',num2str(level/100),'?'));
    level = level/100;
elseif level < 0 || ~isnumeric(level) % other errors
    level = 0.5;
end

% Read rrng file
if ischar(rangeFile)
    [element_num, range_num, elements, ranges] = rangeReader (rangeFile);
elseif length(rangeFile) == 5
    element_num = rangeFile{2};
    range_num   = rangeFile{3};
    elements    = rangeFile{4};
    ranges      = rangeFile{5};
    rangeFile   = rangeFile{1}; % set rangeFile = fileName
else
    [element_num, range_num, elements, ranges] = rangeReader (rangeFile);
end
% Make ranges bigger to store the extra information
extraCols = 14;
newRanges = cell(range_num,element_num+extraCols); % make an array to store the range details
% 1          2        3   4      5    6    7    8        9         10    11
% rangeStart rangeEnd vol colour newR newC newL noisePre noisePost count gausWidth[p(2)] asym[p(3)] yscale[p(4)] noiseFloor[p(5)] element1 ... elementN
% Copy original info
newRanges(:,1:4) = ranges(:,1:4);
% Copy ion info
newRanges(:,(extraCols+1):(extraCols+element_num)) = ranges(:,5:end);
% Assign new ranges var
ranges = newRanges;
% The boundaries for fitting of each of the ranges
% this is used for detecting overlaps
rangeBounds = zeros(range_num,2);

% pos file mass cache
[masses] = loadMasses(posFile);

disp('Plot histogram');
% hack for clsuter pos files that have little data
masses(end) = massSpecMax;
masses(end-1) = 0;
% bin the data
[heights, centres] = hist(masses,num_bins); % this plots a histogram
if plotting
    h = bar(centres,heights,'BarWidth',1); % this plots the bar chart
    hold on;
end
%set(gca,'yscale','log'); % **use log scale**

% measure bin width
binwidth = centres(2)-centres(1);

% To colour the ranges:
% The patches have face-vertex syntax. First, get a handle for the
% children, and then obtain the vertices for the bars and
% the vertex color data:
if plotting
    ch = get(h,'Children');
    %fvd = get(ch,'Faces'); % probably don't need this line
    fvcd = get(ch,'FaceVertexCData');
    fvcd2=ones(length(fvcd),3)*0.71; % make the whole thing light grey
    % create array for range bounds
    lines = ones(range_num*6,2)*.1; % fill in all values 0.1
end
% cycle through each range
for r = 1:range_num
    if plotting
        colour = ranges{r,4}; % set colour string (from file)
        col(1) = hex2dec(strcat(colour(1),colour(2)))/255; % red
        col(2) = hex2dec(strcat(colour(3),colour(4)))/255; % green
        col(3) = hex2dec(strcat(colour(5),colour(6)))/255; % blue
    end
    % for counting
    rangeWidth = round(1+(ranges{r,2})/binwidth)-(round((ranges{r,1})/binwidth));
    thisHits = 0;
    preNoiseHits = 0;
    postNoiseHits = 0;
    %disp(red);
    % now assign the colour to each bar in that range
    % need to use centres and ranges(r,1) and ranges(r,2) to define the
    % width of the range
    % now find the bars in this range:
    % round((ranges{r,1}-centres(1))/binwidth) and with ranges{r,2}
    % now loop through them and assign the colour
    for i = (1+round((ranges{r,1})/binwidth)):round((ranges{r,2})/binwidth)
        if plotting
            for c=1:3
                for vert=1:5
                    fvcd2(vert+5*(i-1),c) = col(c); % assign each colour r,g,b
                end
            end
        end
        % while we're here, count the hits in this range:
        % and the hits for the pre & post noise windows
        thisHits = thisHits + heights(i);
        preNoiseHits = preNoiseHits + heights(i-rangeWidth*noiseWinidow);
        postNoiseHits = postNoiseHits + heights(i+rangeWidth*noiseWinidow);
    end
    ranges{r,8} = thisHits; % total hits in this range
    ranges{r,9} = preNoiseHits; % total hits in this range
    ranges{r,10} = postNoiseHits; % total hits in this range
    if plotting
        % create lines
        lines((r-1)*6+1,1) = ranges{r,1}-binwidth/5;
        lines((r-1)*6+2,1) = ranges{r,1};
        lines((r-1)*6+3,1) = ranges{r,1}+binwidth/5;
        lines((r-1)*6+4,1) = ranges{r,2}-binwidth/5;
        lines((r-1)*6+5,1) = ranges{r,2};
        lines((r-1)*6+6,1) = ranges{r,2}+binwidth/5;
        lines((r-1)*6+2,2) = heights(1+round(ranges{r,1}/binwidth));
        lines((r-1)*6+5,2) = heights(round(ranges{r,2}/binwidth));
    end
end
if plotting
    set(ch,'FaceVertexCData',fvcd2); % set the new colour data on the plot
    % sort lines
    sortedlines = sortrows(lines);
    %hold on;
    % plot black lines where the edges of the current ranges are
    plot(sortedlines(:,1),sortedlines(:,2));
    %hold off;
end
% find peaks (1D only)
% fit model peak
% correct ranges
% write new ranges
% loop through ranges
failCount = 0; % number of ranges with sig:noise < sig2noise (default 3)
for r = 1:range_num
    
    % range width in number of bins
    rangeWidth = round(1+(ranges{r,2})/binwidth)-(round((ranges{r,1})/binwidth));
    if ( round(1+ranges{r,2}/binwidth)-round(ranges{r,1}/binwidth) ) > 3
        [pks, locs] = findpeaks(heights((round(ranges{r,1}/binwidth)):round(1+ranges{r,2}/binwidth)),'minpeakdistance',rangeWidth-1,'minpeakheight',heights(round(ranges{r,1}/binwidth))-1);
    else
        % too few bins to find a peak, so expand the original range width
        error('Too few bins for good fitting!');
        [pks, locs] = findpeaks(heights((round(ranges{r,1}/binwidth)-1):round(2+ranges{r,2}/binwidth)),'minpeakdistance',rangeWidth-1,'minpeakheight',heights(round(ranges{r,1}/binwidth))-1);
    end
    if length(locs)~=1
        % use a dumb maximum instead, this behaviour is expected when the
        % range doesn't have a peak in it though!
        [pks,locs]=max(heights((round(ranges{r,1}/binwidth)):round(1+ranges{r,2}/binwidth)));
        %error(strcat('no peak found:',num2str(r),'; try reranging original file'));
    end
    %ranges(r,5) % right
    %ranges(r,6) % centre
    %ranges(r,7) % left
    %centre of range as defined originally in bin-space
    cor = round(((ranges{r,1}+ranges{r,2})/2)/binwidth)+1; % in bins
    % for small peaks the centre of the orginial ranges is probably better*
    % *citation needed
    % average range height = ranges{r,8}/rangeWidth
    %if(heights(cor)>=ranges{r,8}/rangeWidth)
    % need to change this 18/10/12 1pm
    %if(heights(locs+round(ranges{r,1}/binwidth)-1)>=(ranges{r,8}/rangeWidth)*1.2)
    xbin = locs+round(ranges{r,1}/binwidth)-1; % bin number of range centre
    ranges{r,6}=centres(xbin); % make a note of range centre for later
    % if counts < 3*(pre+post)/2 go with range centre
    if(ranges{r,8} >= (sig2noise*(ranges{r,9}+ranges{r,10})/2))||(heights(xbin)>minCeiling)
        % if new peak isn't much bigger than the average choose centre
        % new peak is better
        if plotting
        % plot markers
        plot(centres(xbin),pks,'k^','markerfacecolor',[0 1 0]);
        end
        % impirical guess of guassian width
        %gausWidth = (sqrt(ranges{r,1})*0.014 + 0.1)*peakBroadening;
        gausWidth = sqrt(ranges{r,1})*peakBroadening;
        pwg = pw*gausWidth; % Peak-width guess based on gaussian width
        %pw = 0.4; % peak width defined above
        %pw - TODO should be optimised some how
        pwb = round(pwg/binwidth); % peak width in bins
        pwb2 = round(pwg*.4/binwidth); % use less-than half width for the start in bins
        
        
        % remove local background
        % fit pre window noise using exp1
        % note the currently fixed window width
        preNoiseWindowStart = (ranges{r,2}-ranges{r,1})*2; %0.3; % relative to range start, ie 0.3 before
        if preNoiseWindowStart < 0.1
            preNoiseWindowStart = 0.1;
        end
        
        preNoiseWindowEnd = (ranges{r,2}-ranges{r,1})*0.5; %0.08; % relative to range start
        if preNoiseWindowEnd < 0.05
            preNoiseWindowEnd = 0.05;
        end
        
        % select the x y data to fit the noise curve to
        preNoiseX = centres( (1+round((ranges{r,1}-preNoiseWindowStart)/binwidth)):round((ranges{r,1}-preNoiseWindowEnd)/binwidth) )';
        preNoiseY = heights( (1+round((ranges{r,1}-preNoiseWindowStart)/binwidth)):round((ranges{r,1}-preNoiseWindowEnd)/binwidth) )';
        rangeX = centres( (1+round((ranges{r,1})/binwidth)):round((ranges{r,2})/binwidth) );
        
        % Fit to an exp in sqrt space:
        [expNoiseFit,~,~] = noiseFit(preNoiseX,preNoiseY,rangeX);
        
        % For plotting
        %fitx = centres(xbin-pwb2):.001:centres(xbin+pwb);
        rangeBounds(r,1) = centres(xbin-pwb2);
        rangeBounds(r,2) = centres(xbin+pwb);
        % Check for over laps
        for rb = 1:range_num
            if rb ~= r
                if rangeBounds(r,1)>rangeBounds(rb,1) && rangeBounds(r,1)<rangeBounds(rb,2)
                    % left range overlap
                    rangeBounds(r,1) = rangeBounds(rb,2);
                    %disp([rangeBounds(r,1) rangeBounds(rb,1) rangeBounds(rb,2)]);
                    disp(strcat('Overlap dectected at range:',num2str(ranges{r,5})));
                end
                if rangeBounds(r,2)>rangeBounds(rb,1) && rangeBounds(r,1)<rangeBounds(rb,2)
                    % right range overlap
                    rangeBounds(r,2) = rangeBounds(rb,1);
                    disp(strcat('Overlap dectected at range:',num2str(ranges{r,7})));
                end
            end
        end
        fitx = rangeBounds(r,1):.001:rangeBounds(r,2);
        %plot(fitx,expNoiseFit(fitx.^2),'r-.'); % noise fit plot
        correctedHeights = heights(xbin-pwb2:xbin+pwb) - expNoiseFit(centres(xbin-pwb2:xbin+pwb).^2)';
        
        % use nlinfit to fit a peak shape
        % initial guess = beta0
        noiseEst = 0; %min(heights(xbin-(0:pwb2))); % or heights(xbin-pwb2)
        %beta0 = [centres(xbin) gausWidth .02 (pks(1)-noiseEst)*100 noiseEst];
        %beta0 = [centres(xbin)-gausWidth gausWidth .02 (pks(1)-noiseEst)*trailPeakHeight];
        % removed centre off-set
        beta0 = [centres(xbin) gausWidth .02 (pks(1)-noiseEst)*trailPeakHeight];
        disp(ranges{r,1}); % display current range for debugging
        %construct an error message incase it fails:
        ME = MException('rngcorrector:nlinfit', strcat('failed at range:',num2str(ranges{r,1})));
        try
            % plot initial guess
            %plot(fitx,asyGauss(beta0(1:5),fitx),'r-.');
            % fitting
            %beta1 = nlinfit(centres(xbin-pwb2:xbin+pwb), correctedHeights, @asyGauss, beta0,statset('DerivStep',[0.00005 eps^(1/3) eps^(1/3) 1 0.5],'Display','off'));
            beta1 = nlinfit(centres(xbin-pwb2:xbin+pwb), correctedHeights, @asyGaussNoNoise, beta0,statset('DerivStep',[0.00005 eps^(1/3) eps^(1/3) 1],'Display','off'));
        catch ME
            disp(strcat('failed at range:',num2str(ranges{r,1})));
            if plotting
                % plot initial guess
                plot(fitx,asyGaussNoNoise(beta0(1:4),fitx)+expNoiseFit(fitx.^2)','r-.');
                disp('See plot for the inital fit (red dashed line)');
            end
            ME
        end
        if plotting
            % single asymGaus fitting:
            plot(fitx,asyGaussNoNoise(beta1(1:4),fitx)+expNoiseFit(fitx.^2)','r-');
        end
        % use asygau2 if 'large' signal to noise
        if(ranges{r,8} >= ( 30*(ranges{r,9}+ranges{r,10})/2) )
            %if(beta1(6)>beta1(5)*2) % try asygau2 based on pre/post noise level
            % fit again with asyGauss2
            beta2 = [beta1(1) beta1(2) beta1(3) beta1(4) heights(xbin-pwb2) centres(xbin)+0.035 beta1(2) beta1(3)/2 pks*10];
            % initial guess:
            if plotting
                plot(fitx,asyGauss2(beta2,fitx),'r-.');
            end
            try
                disp('trying fancy double peak fitting');
                beta3 = nlinfit(centres(xbin-pwb2:xbin+pwb), heights(xbin-pwb2:xbin+pwb), @asyGauss2, beta2,statset('Display','off'));
                if plotting
                    plot(fitx,asyGauss2(beta3,fitx),'g-');
                end
                ranges{r,11} = beta3(2);
                ranges{r,12} = beta3(3);
                ranges{r,13} = beta3(4);
                ranges{r,14} = beta3(5);
                
                % set new ranges
                %level = 0.5; % fw half max
                % new right, left and centre positions
                markers = fwhm(fitx,asyGauss2(beta3,fitx),level);
                if(isfinite(markers(1)))
                    ranges=rangeAssign(ranges,r,markers(1),centres(xbin)); % left
                    %ranges{r,5} = markers(1); % left
                else
                    error('rngCor:leadingEdge',strcat('Failed to find leading edge for peak:',num2str(centres(xbin))));
                end
                if(isfinite(beta3(1)))
                    ranges{r,6} = beta3(1); % centre
                else
                    error('rngCor:centre',strcat('Failed to find centre for peak:',num2str(centres(xbin))));
                end
                if(isfinite(markers(2)))
                    distl = centres(xbin)-markers(1);
                    distr = markers(2)-centres(xbin);
                    % if the new edge is not too far from the centre
                    if(distr<2*distl)
                        ranges=rangeAssign(ranges,r,markers(2),centres(xbin)); % right
                        %ranges{r,7} = markers(2); % right
                    else
                        disp(strcat('New edge(',num2str(markers(2)),') is too far from the centre (',num2str(centres(xbin)),'), setting it to:',num2str(centres(xbin)+distl)))
                        ranges=rangeAssign(ranges,r,centres(xbin)+distl,centres(xbin)); % right
                        %ranges{r,7} = centres(xbin)+distl;
                    end
                else
                    %error('rngCor:trailingEdge',strcat('Failed to find trailing edge for peak:',num2str(centres(xbin))));
                    disp(strcat('Failed to find trailing edge for peak:',num2str(centres(xbin))));
                    disp('Using twice the left range width');
                    disp('Press any key to continue, but I suggest you check the range');
                    % the right position should not be more than
                    % trailingEdgeScale times the distance of the left
                    % position from the centre
                    distl = centres(xbin)-markers(1);
                    % OLD ranges{r,7} = centres(xbin)+distl*trailingEdgeScale;
                    % NEW:
                    ranges=rangeAssign(ranges,r,centres(xbin)+distl*trailingEdgeScale,centres(xbin));
                    if plotting
                        rangeNzoom(ranges{r,5},ranges{r,7},centres(xbin),pks(1));
                    end
                    if pausing
                        pause % zoom to section
                    end
                    xlim('auto'); % unzoom
                    ylim('auto');
                end
            catch ME
                disp(strcat('failed fancy fitting at range:',num2str(ranges{r,1})));
                ME
            end
        else
            % plot the results
            if plotting
                plot(fitx,asyGaussNoNoise(beta1(1:4),fitx)+expNoiseFit(fitx.^2)','g-');
            end
            ranges{r,11} = beta1(2);
            ranges{r,12} = beta1(3);
            ranges{r,13} = beta1(4);
            ranges{r,14} = 0;%beta1(5);
            
            % set new ranges
            % new right, left and centre positions
            disp(strcat('Range:',num2str(r)));
            markers = fwhm(fitx,asyGaussNoNoise(beta1,fitx),level);
            if(isfinite(markers(1)))
                ranges=rangeAssign(ranges,r,markers(1),centres(xbin)); % left
                %ranges{r,5} = markers(1); % left
                % the right position should not be more than 2x the
                % distance of the left position from the centre
            else
                %error('rngCor:leadingEdge',strcat('Failed to find leading edge for peak:',num2str(centres(xbin))));
                disp(strcat('**WARNING** Failed to find leading edge for peak:',num2str(centres(xbin))));
                disp('Using original value instead, this is incorrect behaviour');
                disp('I advise you to examine the peak in the mass spectrum and adjust the input parameters,');
                disp('Press any key to continue');
                if plotting
                    plot(fitx,asyGaussNoNoise(beta1,fitx),'r-.');
                    rangeNzoom(ranges{r,1},ranges{r,2},centres(xbin),pks(1));
                end
                if pausing
                    pause % zoom to section
                end
                xlim('auto'); % unzoom
                ylim('auto');
                %ranges{r,5} = ranges{r,1};
                ranges=rangeAssign(ranges,r,ranges{r,1},centres(xbin));
            end
            if(isfinite(beta1(1)))
                ranges{r,6} = beta1(1); % centre
            else
                error('rngCor:centre',strcat('Failed to find centre for peak:',num2str(centres(xbin))));
            end
            if(isfinite(markers(2)))
                ranges=rangeAssign(ranges,r,markers(2),centres(xbin)); % right
                %ranges{r,7} = markers(2); % right
            else
                if ranges{r,5} ~= ranges{r,1}
                    %error('rngCor:trailingEdge',strcat('Failed to find trailing edge for peak:',num2str(centres(xbin))));
                    disp(strcat('**WARNING** Failed to find trailing edge for peak:',num2str(centres(xbin))));
                    %disp('Using original value instead, this is incorrect behaviour');
                    disp('I advise you to examine the peak in the mass spectrum and adjust the input parameters (eg reduce level value),');
                    
                    if plotting
                        plot(fitx,expNoiseFit(fitx.^2)','r-.');
                        plot(fitx,asyGaussNoNoise(beta1(1:4),fitx),'k-');
                    end
                    if pausing
                        disp('Press any key to continue');
                        pause
                    end
                    disp(strcat('Failed to find trailing edge for peak:',num2str(centres(xbin))));
                    %disp('Using twice the left range width');
                    disp('Press any key to continue, but I suggest you check the range');
                    % the right position should not be more than
                    % trailingEdgeScale times the distance of the left
                    % position from the centre
                    %distl = centres(xbin)-markers(1);
                    %ranges{r,7} = centres(xbin)+distl*trailingEdgeScale;
                    % use original value
                    ranges=rangeAssign(ranges,r,ranges{r,1});
                    ranges=rangeAssign(ranges,r,ranges{r,2});
                    %ranges{r,7} = ranges{r,2}; % OLD
                    %ranges{r,5} = ranges{r,1};
                    if plotting
                        rangeNzoom(ranges{r,5},ranges{r,7},centres(xbin),pks(1));
                    end
                    if pausing
                        pause % zoom to section
                    end
                    xlim('auto'); % unzoom
                    ylim('auto');
                else
                    disp(strcat('Using original range for range:',num2str(r),':',num2str(ranges{r,6})));
                    % use original value
                    ranges=rangeAssign(ranges,r,ranges{r,2});
                    %ranges{r,7} = ranges{r,2};
                end
            end
        end
    else
        if plotting
            % middle position is better
            plot(centres(cor),heights(cor)*1.1,'k^','markerfacecolor',[0 0 1]);
            % mark the found peak for lolz
            plot(centres(xbin),pks*1,'k^','markerfacecolor',[1 0 0]);
        end
        ranges{r,6} = centres(xbin); % save wrong centre position for reference
        failCount = failCount+1;
    end
end
% range width estimation
% count the number of missing rows in ranges
count = 0;
for i=1:range_num
    %disp(strcat(num2str(i),',',num2str(ranges{i,5})));
    if(~isempty(ranges{i,5}))
        count = count+1;
    end
end
if count % if there are missing rows
    % allocate arrays to store the results
    ypre = zeros(1,count-1);
    ypost = zeros(1,count-1);
    x = zeros(1,count-1);
    % get the pre and post widths from known ranges
    disp('pre and post');
    count = 1;
    %disp(ranges(:,5:7)); % DEBUG display current ranges
    for i=1:range_num
        if(~isempty(ranges{i,5}))
            ypre(count) = ranges{i,6}-ranges{i,5}; %pre  }
            ypost(count) = ranges{i,7}-ranges{i,6}; %post } widths
            x(count) = ranges{i,1};
            count = count+1;
        end
    end
    % convert to matrix and fit pre & post widths
    fpre = fit(sqrt(x)', ypre', 'poly1','Robust', 'LAR');
    fpost = fit(sqrt(x)', ypost', 'poly1','Robust', 'LAR');
    % go back through and fit approximate ranges to the poorly defined ranges
    % the centres should stay in the same place, so average pre and post widths
    % The pre/post distinction is still there if you want to use it though
    for r=1:range_num
        if isempty(ranges{r,5})
            centre = ranges{r,6};%(ranges{r,1}+ranges{r,2})/2;
            pre = sqrt(centre)*fpre.p1+fpre.p2;
            post = sqrt(centre)*fpost.p1+fpost.p2;
            halfWidth = (pre+post)/2;
            %disp([num2str(centre-halfWidth) ':' num2str(centre) ':' num2str(centre+halfWidth)]);
            ranges=rangeAssign(ranges,r,centre-halfWidth,centre);
            ranges=rangeAssign(ranges,r,centre+halfWidth,centre);
            %ranges{r,5} = centre-halfWidth;
            %ranges{r,7} = centre+halfWidth;
        end
    end
end
if plotting
    % plot new ranges on histogram
    % create array for range bounds
    lines2 = ones(range_num*6,2)*.1; % fill in all values 0.1
    % cycle through each range
    
    for r = 1:range_num
        % create lines
        %disp(ranges{r,5}); % DEBUG
        %disp(ranges{r,7}); % DEBUG
        lines2((r-1)*6+1,1) = ranges{r,5}-binwidth/5;
        lines2((r-1)*6+2,1) = ranges{r,5};
        lines2((r-1)*6+3,1) = ranges{r,5}+binwidth/5;
        lines2((r-1)*6+4,1) = ranges{r,7}-binwidth/5;
        lines2((r-1)*6+5,1) = ranges{r,7};
        lines2((r-1)*6+6,1) = ranges{r,7}+binwidth/5;
        lines2((r-1)*6+2,2) = heights(1+round(ranges{r,5}/binwidth));
        lines2((r-1)*6+5,2) = heights(round(ranges{r,7}/binwidth));
    end
    % sort lines
    sortedlines2 = sortrows(lines2);
    %hold on;
    % plot black lines where the edges of the current ranges are
    plot(sortedlines2(:,1),sortedlines2(:,2),'k');
    hold off;
end
if(failCount > 1)
    disp(strcat(num2str(failCount),' ranges had a signal:noise<',num2str(sig2noise)))
    result = failCount;
elseif (failCount == 1)
    disp(strcat(num2str(failCount),' range had a signal:noise<',num2str(sig2noise)))
    result = failCount;
else
    result = failCount;
end

% optional sort rows:
ranges = sortrows(ranges,1);
result = ranges; % set returned result to ranges
%result = centres;
%result(2,:) = heights;
if(1) % write new (rrng) range file
    % get file name, without extension
    newName = regexprep(rangeFile,'\.rrng','','ignorecase');
    newName = strcat(newName,'_',num2str(level*100));
    exportRanges = cell(length(ranges(:,1)),(4+element_num));
    exportRanges(:,1) = ranges(:,5); % range start
    exportRanges(:,2) = ranges(:,7); % range end
    exportRanges(:,3:4) = ranges(:,3:4); % volume and colour
    % element-ion matrix
    
    %disp(strcat('5:',num2str(5+element_num),',',num2str(extraCols+1),':',num2str(size(ranges))));
    %disp(elements);
    %disp(ranges(1,:));
    %disp('export ranges');
    %disp(size(exportRanges));
    %disp('ranges');
    %disp(size(ranges));
    exportRanges(:,5:(4+element_num)) = ranges(:,(end-element_num+1):end);
    % if there are miss matching ions and elements the above will fail
    %disp(exportRanges);
    newRrng = rangeWriter(exportRanges,elements,newName);
    disp(strcat('Wrote new rrng file:',newRrng));
end
% count new composition?
% WARNING: This is very slow!
% it is probably faster (but less accurate) to use the histogram
if(0)
    rough = 500; % number of initial samples
    newCounts = zeros(range_num,1);
    % count new ranges:
    for m = 1:round(length(masses)/rough):length(masses)
        if masses(m) <= ranges{range_num,7}
            for r = 1:range_num
                % if m is in range r then count it
                if (masses(m)<=ranges{r,7})
                    if (masses(m)>=ranges{r,5})
                        newCounts(r,1) = newCounts(r,1) + 1;
                    end
                end
            end
            % if ((m - rough*floor(m/rough))==0)
            %     disp(100*m/length(masses));
            % end
        end
    end
    % store rough counts in 8th column of ranges
    disp(newCounts);
    ranges(:,8) = num2cell(newCounts);
    % sort ranges to have the 'most popular' on top (for speed in the next
    % step)
    ranges = sortrows(ranges,-8);
    tic;
    % precise
    rough = 2000000; % number of new number of samples
    if rough>length(masses)
        rough = length(masses);
    end
    newCounts = zeros(range_num,1);
    % count new ranges:
    for m = 1:round(length(masses)/rough):length(masses)
        if masses(m) <= ranges{range_num,7}
            for r = 1:range_num
                % if m is in range r then count it
                if (masses(m)<=ranges{r,7})
                    if (masses(m)>=ranges{r,5})
                        newCounts(r,1) = newCounts(r,1) + 1;
                        break;
                    end
                end
            end
            % this bit of code is a progress display, but slows the process
            % down by 150% ...
            %if ((m - round(1+length(masses)/rough)*floor(m/round(1+length(masses)/rough)))==0)
            %    sprintf('\b\b\b\b\b\b\b\b\b%7.4f',100*m/length(masses))
            %   % disp();
            %end
        end
    end
    disp(toc);
    % multiply newCounts by the element matrix (counts per element)
    % then multiply the transpose by a column of 1s to get the decomp counts
    decompCounts = (transpose(cell2mat(ranges(:,extraCols+1:extraCols+element_num)))*newCounts);
    result = cell(element_num,2);
    result(:,1) = elements;
    result(:,2) = num2cell(decompCounts);
end
end