% cluster composition from a mass spectrum
% This script looks at each cluster in a cluster data file (exported from
% IVAS), the accompanying pos file and rrng file and produces a mass spec
% which can be used for decomposition of the peaks within.

%% Input files, these could be lists
clusterFile = {'R14_15868-v01_YYOTiO_80.csv','R14_17343-v02_YTiOYOO_oxcrop2.csv',...
    'R14_16915-v02_YYOTiOO.csv'};
clusterRoot = 'C:\Users\Andy\APT\clusterAnalysis\';
% rrng files
rrngFiles = {'15868.rrng','17343.rrng','16915_80.rrng'};
rrngFiles = strcat('C:\Users\Andy\APT\',rrngFiles);
% epos cached files
eposRoot = 'C:\Users\Andy\APT\epos\';
% get file list
eposFile = {'R14_15868-v01.epos','R14_17343-v02.epos','R14_16915-v02.epos'};
eposFile = eposFile';
% remove extension:
eposFile = regexprep(eposFile,'.epos','');

%% Initialise vars
f = 1; % this can be a loop later
zs = 1; % The number of rg's to go along XYZ for the cube ROI
% For peak 32 deconvolution
isotopes = [.08, .073, .738, .055, .054];
% TiO++ peaks are:
idealMasses = [30.973774 31.47334 31.971431 32.471393 32.9698505]; % ideal centres of the ranges to be relabelled
idealMSpace = min(diff(idealMasses))*0.9; % reduced to avoid errors, assumes uniform spacing
targetElements = {'Ti' 'O'}; % Elements in the target ion
ion1 = [1 1]; % decomposed ion1,
ion2 = [0 2]; % second ion in the convolved peak
% Cluster file
cf = strcat(clusterRoot,clusterFile{f});

%% Read rrng file
[element_num, range_num, rangeElements, ranges] = rangeReader(rrngFiles{f});
% for the mass list counting:
rangeTable = cell2mat(ranges(:,[1:2 5:end])); % convert to matrix

%% identify target element positions
% TODO: Generalise to include multiple decompositions...
% position of the target elements in 'elements' array
targetEpos = zeros(1,length(targetElements));
targetRanges = zeros(1,length(idealMasses));
for t = 1:length(targetElements)
    idxC = strfind(rangeElements,targetElements{t});
    targetEpos(t) =  find(not(cellfun('isempty', idxC)));
end
%% Identify the range positions corresponding to the ideal masses
for m = 1:length(idealMasses)
    a=find(rangeTable(:,2+targetEpos(1))==1 & rangeTable(:,2+targetEpos(2))==1 & rangeTable(:,1)>(idealMasses(m)-idealMSpace)& rangeTable(:,2)<(idealMasses(m)+idealMSpace),1);
    if isempty(a)
        % range not found
        disp(strcat('No range for mass:',num2str(idealMasses(m)),' found'));
    else
        targetRanges(m)=a;
    end
end

%% Read cluster data file for all it's worth!
[elements,clusterCountsDecomp,clusterCounts,position,ionType,radii,nTable,dmax,dbulk,derode,nmin,core,count] = clusterDataReaderDetails(cf);

%% Read cached epos file pos+voltage+detector information
%eposData=[1 x, 2 y, 3 z, 4 m, 5 nm, 6 vdc,7 xd,8 yd, 9 dP, 10 tof]

[ed] = readCachedEpos(strcat(eposRoot,eposFile{f}),2);
pulses = cumsum(ed(:,9));

voltage = zeros(count,1); % mean voltage per cluster
cubeCounts = zeros(count,length(rangeElements)); % per cluster store decomp'd counts
cubeRangeCounts = zeros(count,length(ranges(:,1))); % per cluster per range counts
peak32 = zeros(count,6); % TiO++% and Absolute error
smallClusterM = [];
% add new range for secondary ion in convolved peak
rangeTable = [rangeTable;rangeTable(targetRanges(3),:)];
rangeTable(end,3:end) = 0; % set to 0
rangeTable(end,2+targetEpos(1)) = ion2(1);  % element 1 in ion 2
rangeTable(end,2+targetEpos(2)) = ion2(2);  % element 2 in ion 2
cubeRangeCounts(:,end+1) = 0;               % add extra column
%% loop through all clusters
for c = 1:count
    disp(strcat(num2str(c),'/',num2str(count))); % progress bar
    cx = position(c,1); % cluster x,y,z
    cy = position(c,2);
    cz = position(c,3);
    cr = radii(c); % current cluster radius
    p = ed(:,1)>(cx-zs*cr) & ed(:,1)<(cx+zs*cr);% longer than the cluster by zs
    p = p & ed(:,2)>(cy-zs*cr) & ed(:,2)<(cy+zs*cr);
    p = p & ed(:,3)>(cz-zs*cr) & ed(:,3)<(cz+zs*cr);
    [cubeRangeCounts(c,:),cubeCounts(c,:)] = massQuant(ed(p,4),rangeTable);
    
    % decomposition
    % TiO counts in the 32 peak:
    tio = isotopes(3)*mean(cubeRangeCounts(c,targetRanges([1 2 4]))./isotopes([1 2 4]));
    pk32 = cubeRangeCounts(c,targetRanges(3));
    o2 = pk32-tio; % O2+ in the 32 peak
    tioP = (pk32-tio)/pk32;
    % errors
    tioRerr = 1./sqrt(sum(cubeRangeCounts(c,targetRanges([1 2 4]))));
    tioPerr = tioP*sqrt((1/pk32)+((pk32+tio)/(pk32-tio)^2));
    o2Rerr = 1./sqrt(cubeRangeCounts(c,targetRanges(3)));
    if(o2<0)
        disp(strcat('100% ',num2str(tioRerr*100,'%3.1f'),'%  TiO++'));
        peak32(c,1) = 1;
        peak32(c,2) = tioRerr;
        peak32(c,3:end) = cubeRangeCounts(c,targetRanges([1 2 3 4]));
        % store mass spectrum
        smallClusterM = [smallClusterM; ed(p,4)];
        % Modify the range count table
        cubeRangeCounts(c,end) = 0;
    else
        disp(strcat(num2str(100*tio/(o2+tio),'%3.1f'),'% ',num2str(tioRerr*100,'%3.1f'),'% TiO++'));
        peak32(c,1) = tio/(o2+tio);
        peak32(c,2) = tioPerr;
        % Modify the range count table
        cubeRangeCounts(c,end) = o2;
        cubeRangeCounts(c,targetRanges(3)) = tio;
    end
end

% do another decomposition and generate the ternary data
cubeCounts2 = cubeRangeCounts*rangeTable(:,3:end);
terXY1 = [];
terXY2 = [];
t = sum(clusterCountsDecomp(:,[find(strcmp(elements,'Y')) find(strcmp(elements,'Ti')) find(strcmp(elements,'O'))]),2);
terXY1(:,1) = clusterCountsDecomp(:,strcmp(elements,'Y'))./t + 0.5*clusterCountsDecomp(:,strcmp(elements,'Ti'))./t;
terXY1(:,2) = 0.866*clusterCountsDecomp(:,strcmp(elements,'Ti'))./t;
subplot(1,2,1);
terplot(6);
hold on
title('Original data, 32 peak = TiO^{++}');
scatter(terXY1(:,1),terXY1(:,2),(radii*2).^2,clusterColour(clusterCountsDecomp(:,[find(strcmp(elements,'Y')) find(strcmp(elements,'O')) find(strcmp(elements,'Ti'))]))); % original data
hold off

subplot(1,2,2);
terplot(6);
hold on
title('Decomposed Composition');
t = sum(cubeCounts2(:,[find(strcmp(rangeElements,'Y')) find(strcmp(rangeElements,'Ti')) find(strcmp(rangeElements,'O'))]),2);
terXY2(:,1) = cubeCounts2(:,strcmp(rangeElements,'Y'))./t + 0.5*cubeCounts2(:,strcmp(rangeElements,'Ti'))./t;
terXY2(:,2) = 0.866*cubeCounts2(:,strcmp(rangeElements,'Ti'))./t;
scatter(terXY2(:,1),terXY2(:,2),(radii*2).^2,clusterColour(cubeCounts2(:,[find(strcmp(rangeElements,'Y')) find(strcmp(rangeElements,'O')) find(strcmp(rangeElements,'Ti'))])),'filled'); % new data
spacing = 0.05;
text(0.5,-spacing,'Y fraction','Rotation',0,'HorizontalAlignment','center');
text((.75)+spacing,(.866/2),'Ti fraction','Rotation',-60,'HorizontalAlignment','center');
text((.25)-spacing,(.866/2),'O fraction','Rotation',60,'HorizontalAlignment','center');
hold off