#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Tue Feb 12 13:02:15 2019

@author: aurelien
"""

import numpy as np

from scipy.optimize import curve_fit
import matplotlib.pyplot as plt
import pickle

from pandas import ExcelWriter,DataFrame
import pkg_resources

# Calibration cat2, oil objective
path = 'data/fit_dictionay_extrapolated.json'  # always use slash
filepath = pkg_resources.resource_stream(__name__, path)
fit_dictionary = pickle.load(filepath)

# Calibration cat3, water objective
path_w = 'data/cat3_fit_dictionay_extrapolated.json'  # always use slash
filepath_w = pkg_resources.resource_stream(__name__, path_w)
fit_dictionary_w = pickle.load(filepath_w)

fit_dictionaries = {'cat2':fit_dictionary,
                    'cat3':fit_dictionary_w}

AVAILABLE_SYSTEMS = ['cat2','cat3']
#----------Curve models------------------------------------

def Gt(tau,T,tauT):
    """Models tiplet states.
    Parameters:
        tau: float or array, correlation time.s
        T: float, in [0,1]. Amplitude of triplet fluctuations
        tauT: float, triplet correlation time
    Returns: 
        gt: float or array, correlations due to triplet states"""
    return (1+T/(1-T)*np.exp(-tau/tauT))

def Gc(tau,N,tauD,K,alpha):
    """Correlation function based on gaussian PSF model.
    Parameters:
        tau: float or array, correlation time.s
        N: float, number of molecules
        tauD: float, lateral transit time
        K: float, aspect ratio (K = Wz/Wxy, Wz= axial resolutionm Wxy =
                                lateral resolution)
        alpha: float, generally in [0,2]. Characterises non-brownian motion
    Returns:
        gc: float or array, correlation caused by molecular motion
        """
    factor = (1+(tau/tauD)**alpha )*np.sqrt(1+ (1/K**2)*(tau/tauD)**alpha )  
    return 1/(N*factor)


def Gc_z(tau,N,tauD,tauZ,alpha):
    """Correlation function based on gaussian PSF model. Uses axial transit
    time instead of aspect ratio
    Parameters:
        tau: float or array, correlation time.s
        N: float, number of molecules
        tauD: float, lateral transit time
        K: float, aspect ratio (K = Wz/Wxy, Wz= axial resolutionm Wxy =
                                lateral resolution)
        alpha: float, generally in [0,2]. Characterises non-brownian motion
    Returns:
        gc: float or array, correlation caused by molecular motion
        """
    factor = (1+(tau/tauD)**alpha )*np.sqrt(1+ (tau/tauZ)**alpha )  
    return 1/(N*factor)

def Gc2D(tau,N,tauD,alpha):
    """Correlation function based on gaussian PSF model for 2D diffusion.
    PArameters:
        tau: float or array, correlation time.s
        N: float, number of molecules
        tauD: float, lateral transit time
        K: float, aspect ratio (K = Wz/Wxy, Wz= axial resolutionm Wxy =
                                lateral resolution)
        alpha: float, generally in [0,2]. Characterises non-brownian motion
    Returns:
        gc: float or array, correlation caused by molecular motion
        """
    factor = (1+(tau/tauD)**alpha )
    return 1/(N*factor)

def linear_fitter(x,a):
    """Method used to create the CH fitters. a is a modality-specific parameter
    and x is the ratio lateral resolution/lateral resolution confocal
    """
    return a*(x-1)+1

def exp_fitter(x,a,b,c):
        return a*np.exp(b*x)+c
    
fitters_dict = {'lin':linear_fitter,
           'exp':exp_fitter}

def make_res_functions(fitname,k_confocal,system = 'cat2'):
    """Generates the resolution variation functions of a given depletion 
    pattern with normalised resolution
    Parameters:
        fitname: string or float,key to correct entry in fit_dictionary
        k_confocal: float, aspect ratio in confocal mode
        system: string, key to system used (aspect ratios vary with systems)
    Returns:
        f1: function, returns evolution of z-resolution with x-resolution
        fk: function, returns evolution of aspect ratio with x-resolution
        fv: function, returns evolution of focal volume with x-resolution"""
    assert(system in AVAILABLE_SYSTEMS)
    
    fd = fit_dictionaries[system]
    pars,fn = fd[fitname]
    fitter_fun = fitters_dict[fn]
    #x is normalised lateral resolution, not transit time 
    f1 = lambda x: fitter_fun(x,*pars)
    fk = lambda x: f1(x) * fd["Kconf"]/x
    fv = lambda x: x**3*fk(x)/k_confocal
    return f1,fk,fv


class Curve(object):
    """class for an FCS curve. Used to store metadata like STED power and the
    corresponding fitter"""
    def __init__(self,corr = None,power = None,fitter = None,bounds = None,
                 first_n = 3):
        """Parameters:
            corr: nx2 numpy array, shape [npts,axis] with n the number of 
                points in the ACF and axis is x (0) or y(1)
            power: int or float, STED power
            fitter: function, to be fitted using curve_fit
            bounds: tuple, bounds to be passed as parameter of curve_fit
            first_n: optional, first number of points to be removed from ACFs"""
        if corr is None or power is None or fitter is None or bounds is None:
            raise KeyError("Please enter all the parameters")
        self.x = corr[first_n:,0]
        self.y = corr[first_n:,1]
        self.power = power
        self.fitter = fitter
        self.bounds = bounds
        
        self.popt = None
        
        #Where to find N and tau in fitter outputs
        self.n_index = 0
        self.tau_index=1
        
    def fit(self):
        self.popt,_ = curve_fit(self.fitter,self.x,self.y,bounds=self.bounds)
        self.yhat = self.fitter(self.x, *self.popt)
        
    @property
    def tau(self):
        if self.popt is None:
            self.fit()
        return self.popt[self.tau_index]
    
    @property
    def N(self):
        if self.popt is None:
            self.fit()
        return self.popt[self.n_index]
        
    def plot(self,axes=None,normalise=True,offset = 0,label=None,
             offset_residuals_only=False, color = None):
        """Plots an ACF along with its fit, and potentially the residuals.
        Parameters:
            axes: list or mpl axis. if list, has to contain two elements,
            ax1 to plot ACF and ax2 to plot residuals. If mpl axis, handle to 
            axis where to plot the residuals
            normalise: bool, if True normalises the ACFs
            offset: float, offset to display multiple ACFs
            label: string, to put as legend
            offset_residuals_only: bool, if True offsets only residuals"""
        if axes is None:
            fig,axes = plt.subplots(1,2)
        try:
            iter(axes)
            ax1,ax2 = axes
            # ax1.set_title("FCS curves")
            ax2.set_title("Residuals")
            ax2.set_xlabel(r"$\tau$")
            ax1.set_xlabel(r"$\tau$")
        except:
            ax1 = axes
            # ax1.set_title("FCS curves")
            ax2 = None
            ax1.set_xlabel(r"$\tau$")
        fit = self.fitter(self.x,*self.popt)
        if normalise:
            y = self.y*self.N
            fit = fit*self.N
        else:
            y = self.y
        if offset_residuals_only:
            ax1.semilogx(self.x,y,label=label, color=color)
            ax1.semilogx(self.x,fit,color='black',linestyle='--')
        else:
            ax1.semilogx(self.x,y+offset,label=label, color=color)
            ax1.semilogx(self.x,fit+offset,color='black',linestyle='--')
        
        if ax2 is not None:
            ax2.semilogx(self.x,y-fit+offset)
            
    def __eq__(self,new):
        return (self.x == new.x).all() and (self.y == new.y).all()
    
class FcsExperiment(object):
    """Class used to process an FCS experiment."""
    def __init__(self,fcsFitter = None,files_dictionary = None,alpha = None,
                 loadfun = None, tau_t = None, tau_confocal = None,sted_maker = None):
        """
        Parameters:
            sted_type: string or float, key for the fitters dictionary. Example
            '3D','2D' or 0.75 (for CH with radius 0.75)
            fcsFitter: object of the class FcsFitter.
            files_dictionary: dict. Keys are STED powers, each element is a 
                list of files corresponding to these file powers
            alpha: float, alpha value for curves fitting
            loadfun: function, loads data
            tau_t: float, triplet correlation time
            tau_confocal: float (optional), transit time in confocal mode
            sted_maker: function (optional), if None uses the default STED function
                maker"""
            
        self.tau_confocal = None
        self.fcsFitter = fcsFitter
        if self.fcsFitter is not None:
            self.sted_type = self.fcsFitter.sted_type
            self.system = self.fcsFitter.system
        else:
            raise ValueError("You must specify an fcsFitter")
        if files_dictionary is None:
            raise ValueError("You must specify a files_dictionary")
            
        if 0 not in files_dictionary.keys() and tau_confocal is None:
            raise KeyError("Confocal data must be present")
        
        if alpha is None or loadfun is None or tau_t is None:
            raise KeyError("Please specify a value for each parameter")
            
        self.tau_t = tau_t
        self.alpha = alpha
        self.files_dictionary = files_dictionary
        if sted_maker is None:
            self.sted_maker = self.default_sted_maker
        else:
            self.sted_maker = sted_maker
            print("!! CAUTION !! You are about to use a custom STED fitting function.")
            print("Estimation of focal volumes and aspect ratios may be improper")
        #FCS curves are stored in a dictionary where keys are the STED power.
        #Each item is a list of Curve
        self.fcs_curves = {}
        #load function
        self.loadfun = loadfun
        #self.process()
        
    def process(self):
        """Processes the batch of FCS curves. Assumes a constant value of alpha
        across experiments"""
        alpha = self.alpha
        k_confocal = self.fcsFitter.k_confocal
        self.sted_powers = sorted(self.files_dictionary.keys())
        
        #Fitting confocal data
        gc = self.fcsFitter.confocal_fitter
        bds_confocal = self.fcsFitter.bounds_confocal
        
        confocal_files = self.files_dictionary[0]
        self.confocal_curves = [Curve(corr=self.loadfun(w),power = 0,fitter = gc,
                                      bounds = bds_confocal,first_n = 3)
                                for w in confocal_files]
        
        self.fcs_curves[0] = self.confocal_curves
        self.tau_confocal = np.mean([w.tau for w in self.confocal_curves])
        
        #Preparing methods for STED fitting
        self.xy_fun = lambda tau: np.sqrt( (tau/self.tau_confocal)**alpha )
        fz,fk,fv = make_res_functions(self.sted_type,k_confocal,
                                      system=self.system)
        self.k_function = lambda tau: fk( self.xy_fun(tau))
        self.v_function = lambda tau: fv(self.xy_fun(tau))
        self.z_function = lambda tau: self.xy_fun(tau) * fk(self.xy_fun(tau))/k_confocal
        
        sted_fitter = self.sted_maker(self.tau_confocal,fk)
        bds_sted = self.fcsFitter.bounds_sted
        #Assumption here: first sted_power is always 0 (verified in generalte_npy)
        for power in self.files_dictionary.keys():
            if power ==0:
                continue
            files = self.files_dictionary[power]
            sted_curves = [Curve(corr=self.loadfun(w),power = power,fitter = sted_fitter,
                                      bounds = bds_sted,first_n = 3)
                            for w in files]
            [w.fit() for w in sted_curves]
            self.fcs_curves[power] = sted_curves


    def default_sted_maker(self,tau_confocal,fk):
        def f(tau,N,tauD,T):
            r= (tauD/tau_confocal)**(self.alpha/2)
            return Gc(tau,N,tauD,fk(r),self.alpha) * Gt(tau,T,self.tau_t)
        return f

    def volumes(self,raw=False):
        """Computes focal volumes and returns their mean and std
        Returns:
            sted_powers: 1D numpy array, different sted_powers
            mean_volumes: 1D numpy array, average value of normalised focal volume
                at each STED power
            std_volume: 1D numpy array, standard deviation of focal volume 
            distribution at each power"""
        if self.tau_confocal is None:
            self.process()
        
        mean_volumes=[]
        std_volumes = []
        raw_dict = {}
        for power in self.sted_powers:
            curves = self.fcs_curves[power]
            taus = np.array([w.tau for w in curves])
            if power==0:
                volumes = self.xy_fun(taus)**3
            else:
                volumes = self.v_function(taus)
            mean_volumes.append(np.mean(volumes))
            std_volumes.append(np.std(volumes))
            raw_dict[power] = volumes
        mean_volumes = np.asarray(mean_volumes)
        std_volumes = np.asarray(std_volumes)
        if raw:
            return raw_dict
        else:
            return self.sted_powers,mean_volumes,std_volumes
    
    def get_parameter(self,index,index_confocal = None):
        """Returns a dictionary containing the value of a certain parameter, 
        for each curve at each power.
        Parameters:
            index: int, index of the parameter of interest in the array
                returned by curve_fit
            index_confocal: int, optional. Set only if the parameter of interest
                has a different position in confocal than in STED"""
        if self.tau_confocal is None:
            self.process()
        
        par_dict = {}
        
        for power in self.sted_powers:
            curves = self.fcs_curves[power]
            ind = index
            if power==0 and index_confocal is not None:
                ind = index_confocal
            par_dict[power] = [w.popt[ind] for w in curves ]
        return par_dict
    
    def number_of_molecules(self,raw=False):
        if self.tau_confocal is None:
            self.process()
                
        mean_Ns=[]
        std_Ns = []
        N0 = np.mean( [w.N for w in self.fcs_curves[0] ])
        raw_dict = {}
        for power in self.sted_powers:
            curves = self.fcs_curves[power]
            Ns = np.array([w.N for w in curves])/N0
            raw_dict[power] = np.array([w.N for w in curves])
            mean_Ns.append(np.mean(Ns))
            std_Ns.append(np.std(Ns))
        mean_Ns = np.asarray(mean_Ns)
        std_Ns = np.asarray(std_Ns)
        
        if raw:
            return raw_dict
        else:
            return self.sted_powers,mean_Ns,std_Ns
    
    def transit_times(self,raw=False):
        if self.tau_confocal is None:
            self.process()
                
        mean_taus=[]
        std_taus = []
        raw_dict = {}
        for power in self.sted_powers:
            curves = self.fcs_curves[power]
            taus = np.array([w.tau for w in curves])
            raw_dict[power] = taus
            mean_taus.append(np.mean(taus))
            std_taus.append(np.std(taus))
        mean_taus = np.asarray(mean_taus)
        std_taus = np.asarray(std_taus)
        if raw:
            return raw_dict
        else:
            return self.sted_powers,mean_taus,std_taus
    
    def axial_resolutions(self,raw=False):
        """Parameter:
            raw: if True, returns the raw data"""
        if self.tau_confocal is None:
            self.process()
                
        mean_zs=[]
        std_zs = []
        raw_dict = {}
        for power in self.sted_powers:
            curves = self.fcs_curves[power]
            taus = np.array([w.tau for w in curves])
            if power==0:
                zs = self.xy_fun(taus)
            else:
                zs = self.z_function(taus) 
            
            raw_dict[power] = zs
            mean_zs.append(np.mean(zs))
            std_zs.append(np.std(zs))
        mean_zs = np.asarray(mean_zs)
        std_zs = np.asarray(std_zs)
        if raw:
            return raw_dict
        else:
            return self.sted_powers,mean_zs,std_zs

    
    def axial_transit_times(self,raw=False):
        """Parameter:
            raw: if True, returns the raw data"""
        if self.tau_confocal is None:
            self.process()
        mean_tauzs=[]
        std_tauzs = []
        raw_dict = {}
        for power in self.sted_powers:
            curves = self.fcs_curves[power]
            taus = np.array([w.tau for w in curves])
            if power==0:
                k_confocal = self.fcsFitter.k_confocal
                tauzs = taus * k_confocal**(2/self.alpha)
            else:
                ks = self.k_function(taus)
                tauzs = taus * ks**(2/self.alpha)
            
            raw_dict[power] = tauzs
            mean_tauzs.append(np.mean(tauzs))
            std_tauzs.append(np.std(tauzs))
        mean_tauzs = np.asarray(mean_tauzs)
        std_tauzs = np.asarray(std_tauzs)
        if raw:
            return raw_dict
        else:
            return self.sted_powers,mean_tauzs,std_tauzs

    def lateral_resolutions(self,raw=False):
        """Parameter:
            raw: if True, returns the raw data"""
        if self.tau_confocal is None:
            self.process()
                
        mean_rs=[]
        std_rs = []
        raw_dict = {}
        for power in self.sted_powers:
            curves = self.fcs_curves[power]
            taus = np.array([w.tau for w in curves])
            rs = self.xy_fun(taus)
            raw_dict[power] = rs
            mean_rs.append(np.mean(rs))
            std_rs.append(np.std(rs))
        mean_rs = np.asarray(mean_rs)
        std_rs = np.asarray(std_rs)
        if raw:
            return raw_dict
        else:
            return self.sted_powers,mean_rs,std_rs
        
    def export_xls(self,fname):
        """Exports the results of an experiment as xls to be read in excel
        Parameters: 
            fname: str"""
        if not fname.endswith(".xls"):
            fname+=".xls"
        axial_resolutions = self.axial_resolutions(raw=True)
        volumes = self.volumes(raw=True)
        transit_times = self.transit_times(raw = True)
        number_of_molecules = self.number_of_molecules(raw=True)
        
        with ExcelWriter(fname) as writer:  # doctest: +SKIP
            for kk in self.files_dictionary.keys():
                vs = volumes[kk]
                axres = axial_resolutions[kk]
                fnames = self.files_dictionary[kk]
                tt = transit_times[kk]
                ns = number_of_molecules[kk]
                data = {"Name":fnames,
                        "N":ns,
                        "tau":tt,
                        "r/rconfocal":(np.array(tt)/self.tau_confocal)**(self.alpha/2),
                        "Axial resolution":axres,
                        "V/Vconfocal":vs}
                DataFrame(data).to_excel(writer, sheet_name="STED power "+str(kk))
            parameters_dict = {"K confocal":self.fcsFitter.k_confocal,
                               "alpha":self.alpha}
            print(parameters_dict)
            DataFrame(parameters_dict,index=[0]).T.to_excel(writer, 
                     sheet_name='Parameters')
        
    def plot(self,title=None):
        """Plots fitting results as well as ACFs(?) and residuals(?)
        Parameters:
            title: string, optional. If not None, gives a title to a batch of curves"""
        p1,v1,v1s = self.volumes()
        p2,n1,n1s = self.number_of_molecules()
        p3,t1,t1s = self.transit_times()
        
        plt.figure()
        plt.subplot(131)
        plt.errorbar(p1,v1,yerr=v1s,marker='o',capsize=5)
        plt.xlabel("STED power")
        plt.ylabel("Focal volume (normalised)")
        
        plt.subplot(132)
        plt.errorbar(p3,t1,yerr=t1s,marker='o',capsize=5)
        plt.xlabel("STED power")
        plt.ylabel("Transit times")
        
        plt.subplot(133)
        plt.errorbar(p2,n1,yerr=n1s,marker='o',capsize=5)
        plt.xlabel("STED power")
        plt.ylabel("Number of molecules")
        
        plt.suptitle(title)
        
    def plot_acfs(self,title=None):
        n1 = len(self.sted_powers)
        fig,axes = plt.subplots(2,n1+1)
        axes = axes.T
        
        
        for j in range(n1):
            pw = self.sted_powers[j]
            curves = self.fcs_curves[pw]
            
            offset = 0
            offset_iter=0.2
            for curve in curves:
                curve.plot(axes=axes[j],normalise=True,offset = offset)
                offset+=offset_iter
            axes[j,0].set_title("STED power "+str(pw))
            
            xvals = np.array([c.x for c in curves])
            x0 = xvals[0]
            assert(np.all([xv == x0 for xv in xvals]))
            yvals = np.median(np.array([c.y for c in curves]),axis=0)
            yvals_norm = np.median( np.array([c.y * c.N for c in curves]),
                                     axis= 0 )
            axes[n1,0].semilogx(x0,yvals_norm,label = str(pw))
            axes[n1,1].semilogx(x0,yvals,label = str(pw))
            
        axes[n1,0].set_title("Median of normalised ACFs")
        axes[n1,0].legend()
        axes[n1,1].set_title("Median of ACFs")
        axes[n1,1].legend()
        fig.suptitle(title)

    def __eq__(self,other):
        if (self.fcsFitter != other.fcsFitter or self.alpha != other.alpha or 
            self.tau_t != other.tau_t):
            return False
        else:
            return True
        
    def __add__(self,other):
        raise NotImplementedError("Addition of fcsExperiments still ill defined")
        if (self.fcsFitter != other.fcsFitter or self.alpha != other.alpha or 
            self.tau_t != other.tau_t):
            raise TypeError("can only sum FcsExperiments with simillar parameters")
        
        new = FcsExperiment(fcsFitter = self.fcsFitter,
                            files_dictionary = None,alpha = None,
                 loadfun = None, tau_t = None, tau_confocal = None,
                 sted_maker = None)
        return new
    
class FcsFitter(object):
    """Class handling FCS fitters, with attributes for both confocal and STED
    fitting."""
    def __init__(self,sted_type,confocal_fitter,bounds_confocal,
                 bounds_sted,k_confocal = 4,system='cat2'):
        self.sted_type = sted_type
        self.confocal_fitter = confocal_fitter
        self.bounds_confocal = bounds_confocal
        self.bounds_sted = bounds_sted
        self.k_confocal = k_confocal
        self.system = system
        
    def __eq__(self, other):
        if isinstance(other,FcsFitter):
            if (self.sted_type==other.sted_type and
                self.confocal_fitter == other.confocal_fitter
                and self.bounds_confocal == other.bounds_confocal
                and self.bounds_sted == other.bounds_sted
                and self.k_confocal == other.k_confocal
                and self.system == other.system):
                return True
        else:
            print("FCS fitters are different")
            return False
    
def export_experiments(exp_list,fname):
    """exports the result of a series of experiments as xls"""
    if not fname.endswith(".xls"):
        fname+=".xls"
        
    axial_resolutions = {}
    volumes = {}
    transit_times = {}
    number_of_molecules = {}
    files_dictionary = {}
    lat_sizes = {}
    axial_transit_times = {}
    def myMerge(old,new):
        for k in new.keys():
            if k not in old:
                old[k] = new[k]
            else:
                if type(old[k])==list:
                    old[k].extend(new[k])
                else:
                    old[k] = np.concatenate((old[k],new[k]))
                    
    exp1 = exp_list[0]
    for exp in exp_list:
        if exp != exp1:
            raise TypeError("Can only save experiments with same parameters")
        myMerge(axial_resolutions, exp.axial_resolutions(raw=True))
        myMerge(volumes, exp.volumes(raw=True))
        myMerge(transit_times, exp.transit_times(raw = True))
        myMerge(number_of_molecules, exp.number_of_molecules(raw=True))
        myMerge(files_dictionary,exp.files_dictionary)
        myMerge(lat_sizes, exp.lateral_resolutions(raw = True))
        myMerge(axial_transit_times, exp.axial_transit_times(raw = True))
    
    #files_dictionary has STED powers as keys
    with ExcelWriter(fname) as writer:  # doctest: +SKIP
        for kk in files_dictionary.keys():
            vs = volumes[kk]
            axres = axial_resolutions[kk]
            fnames = files_dictionary[kk]
            tt = transit_times[kk]
            rs = lat_sizes[kk]
            ns = number_of_molecules[kk]
            tz = axial_transit_times[kk]
            data = {"Name":fnames,
                    "N":ns,
                    "tau":tt,
                    "r/rconfocal":rs,
                    "Axial resolution":axres,
                    "V/Vconfocal":vs,
                    "tau z":tz
                    }
            DataFrame(data).to_excel(writer, sheet_name="STED power "+str(kk))
        parameters_dict = {"K confocal":exp.fcsFitter.k_confocal,
                           "alpha":exp.alpha}
        DataFrame(parameters_dict,index=[0]).T.to_excel(writer, 
                 sheet_name='Parameters')

def export_curves(curves_list,list_names,fname,optionals=None):
    """exports the result of a series of experiments as xls
    Parameters:
        curves_list: list, each element is a list of Curve instances
        list_names: list, should have the same length as curve_list. Contains 
            names to be associated with each Curve sub-list
        fname: string, output filename
        optionals: dict, optional list of parameters to be included, i.e filenames"""
        
    assert(len(curves_list)==len(list_names))
    if not fname.endswith(".xls"):
        fname+=".xls"
        
    transit_times = {}
    number_of_molecules = {}
    
    for curves,name in zip(curves_list,list_names):
        
        transit_times[name] = [w.tau for w in curves]
        number_of_molecules[name] = [w.N for w in curves]
        
        
        
    with ExcelWriter(fname) as writer:  # doctest: +SKIP
        for j,name in enumerate(list_names):
            data = {}
            taus = transit_times[name]
            ns = number_of_molecules[name]
            if optionals is not None:
                for opkey in optionals.keys():
                    data[opkey] = optionals[opkey][j]
            data["tau"] = taus
            data["N"]=ns
            DataFrame(data).to_excel(writer, sheet_name=name)

def read_notes(folder):
    out = "404 notes not found"
    with open(folder+"notes0.txt") as f:
        out = f.read()
    return out
    