function ForceDisplacement

%% We aim to calculate the force law as a function of indentation depth in a way that makes the non-dimensionalization work as tau->0
% Here, however, we are allowing finite tau to try and understand what the
% effect of a small value is

% y(1)=w
% y(2)=w'
% y(3)=w''
% y(4)=w'''
% y(5)=psi-tau*r
% y(6)=r*psi'-tau*r

clear all;
close all;

nu=0.5;

tau=1; % Values of tau can be changed here to examine the dependency of the 
% force-displacement law on tau

L=1000;   %This is the length of the system in units of the capillary length

x0=0.0001;                      % This is a cutof value to avoid a singularity

Num=2500; %This gives the number of points used by the simulation. A bigger 
%number increases the precision
sint = linspace(x0,L,Num);
dx=(L-x0)/Num;

cols='rgbmyckrgbmyck';

%for j=1:length(ds)
    w0=0.00001;
    testcount=0;

    Nits=2e3;
    results=zeros(Nits,6);

    it=1;
    itbigw=1;
    trialw=[0.0005 0.00075 0.00125 0.002 0.003 0.005 0.0075 0.0125 0.0175 0.0265 0.04 0.06 0.09 0.135 0.125 0.25 0.5 0.75 1 1.5 2 2.2  2.5 3 4 5 7.5 10 12.5 15 17.5 20 22.5 26.67 32.93 38.26 43.4 65.1];
    maxw=65+2;
    wplot=5;
    itplot=0;

    membrane=fopen(strcat('ForceLawTau=',num2str(tau),'.txt'), 'w');

    %% Here is the starter loop

    tic

    solinit = bvpinit(sint,@mat4init);   
    options=bvpset('RelTol',1e-6,'NMax',2e4);
    sol = bvp4c(@mat4ode,@mat4bc,solinit,options);
    fprintf('Step1\n');

    toc

    r=sol.x;
    w=sol.y(1,:);
    wp=sol.y(2,:);
    w2p=sol.y(3,:);
    w3p=sol.y(4,:);
    psi=tau*r+sol.y(5,:);

    for i=1:3 
        sol = bvp4c(@mat4ode,@mat4bc,sol,options);
    end;
    Ssint = deval(sol,sint);
    fprintf('Step2\n');
    Cut=1000;       %Cut and Cut1 give to samples for the system
    Cut1=100;
    
    %% Here is the main loop

    while w0<maxw
        sol = bvp4c(@mat4ode,@mat4bc,sol,options);
        Ssint = deval(sol,sint);
        %Here we evaluate the amplitude at two distances from the center
        Amp=trapz(sint(1:Cut),Ssint(1,1:Cut).*sint(1:Cut));
        q0=Ssint(2,Cut)*sint(Cut)+Ssint(2,Cut)*Ssint(3,Cut)-Amp;
        Amp1=trapz(sint(1:Cut1),Ssint(1,1:Cut1).*sint(1:Cut1));
        q1=Ssint(2,Cut1)*sint(Cut1)+Ssint(2,Cut1)*Ssint(3,Cut1)-Amp1;
        %Here the relation between amplitude and force is obtained
    
        results(it,1)=w0;   
        results(it,2)=q0;
        results(it,3)=q1;   
    
        if it>2
            r=sol.x;
            w=sol.y(1,:);
            wp=sol.y(2,:);
            w2p=sol.y(3,:);
            w3p=sol.y(4,:);
            psi=tau*r+sol.y(5,:);
            psip=sol.y(6,:)./r+tau;
            chi=sol.y(5,:);
            
            fce=real(2*pi*(psi.*wp-(r.*w3p+w2p-wp./r)));
            Nfit=10;

            p=polyfit(r(1:Nfit),fce(1:Nfit),1);
            figure(18)
            loglog(p(2),w0,'ro')
            hold on

            results(it,4)=p(2);
    
            fprintf(membrane,'%.12g\t',w0);
            fprintf(membrane,'%.12g\t',p(2));
            fprintf(membrane,'\n');   
                
        end   

        if w0<min(trialw)
            w0=1.5*(w0+0.0001)
            it=it+1;
        elseif w0<max(trialw)
            w0=trialw(itbigw)
            itbigw=itbigw+1;
            it=itbigw;
        else
            w0=1.5*w0
            itbigw=itbigw+1;
            it=itbigw
        end
    end

    status=fclose(membrane);

    
    figure(18)
    x=logspace(-4,1,100);
    k1=4*pi*taufun(tau/2); % The definition of tau used in the Interface paper is half that used here (so that the quadratic is lam^2-2*tau*lam+1=0)
    loglog(k1*x,x,'r--')
%x=logspace(0,log10(maxw),100);
%loglog(x,x.^2,'k:')
ylabel('$\zeta_0/h$','interpreter','latex')
xlabel('$F/(h(Bg)^{1/2}$','interpreter','latex')
axis([1e-1 pi*maxw^2 1e-2 maxw])
hold off



%% Here are the functions that we use
  % -----------------------------------------------------------------------

  function dydx = mat4ode(x,y)
  
  % Derivative function. q is provided by the outer function.
   
  % Here are solving the equations including the value of psi at infinity
  
  dydx = [      y(2)
                y(3)
                y(4)
                -2*y(4)./x+y(3)./x.^2-y(2)./x.^3-y(1)+y(3).*(y(5)./x+tau)+y(2).*(y(6)./x.^2+tau./x)
                y(6)./x
                y(5)./x-6*(1-nu^2)*y(2).^2];   
  end
  % -----------------------------------------------------------------------

  function res = mat4bc(ya,yb)
  % Boundary conditions. This is the case for clamped boundary conditions. 
    res = [  ya(1)+w0 % w(0)=-w0
             ya(2) % w'(0)=0
             ya(6)-nu*ya(5)
             yb(1)
             yb(2)
             yb(5)
             ];
  end

  
  % -----------------------------------------------------------------------
      
    
% -------------------------------------------------------------------------
% Auxiliary function -- initial guess for the solution

function y = mat4init(x)

lp=0.5*(tau+sqrt(tau^2-4));
lm=0.5*(tau-sqrt(tau^2-4));
ypred=-2/log(lm/lp)*(besselk(0,lp*x)-besselk(0,lm*x));
%ypred=0*ypred;
 
    y = [  ypred
           0
           0
           0
           0
           0
            ];
    
end

end

% -------------------------------------------------------------------------
