function Wrinkle_Onset

%% Here we try and cast the NT wrinkling problem of a poked floating sheet as an eigenvalue problem.
% The strategy is to first solve the axisymmetric problem for a given value
% of the tension tau and given indentation depths then to perturb these
% solutions and find the eigenvalue lambda=n^2 with n the number of
% wrinkles

% It seems that the problem is a quadratic eigenvalue problem, so that
% (lam^2A+lam B+C)X=0 where X=(w_0,w_1,...,\chi_0,\chi_1,...) is the
% solution vector

% Here we just focus on trying to determine the solutions of det(lam^2A+lam
% B+C)=0 numerically, since this will give us the number of wrinkles

% Here we integrate up to a maximum value at which we hope it will be
% possible to solve the eigenvalue problem and then march back down again
% to find the smallest value at which a solution exists

% y(1)=w
% y(2)=w'
% y(3)=w''
% y(4)=w'''
% y(5)=psi-tau*r
% y(6)=r*psi'-tau*r

clear all;
close all;

tt=1;
for n=0

    tau=10^n;      
% In this example, we solve for tau = 1, and assume a priori that the 
% number of wrinkles is in the range 6-9, and that the critical indentation
% is in the range 15-24.
% To change the range of wrinkles numbers to be tested, nvals, and the
% range of critical indentation depths to be tested, w0vals, please edit
% lines 141-143 of the code.

membrane=fopen(strcat('ForceLawTau=',num2str(tau),'.txt'), 'w');

L=1000;   % This is the length of the system in units of l_g
nu=0.5;   % The poisson ratio is needed for the BC

x0=0.0001; % This is a cutoff value to avoid a singularity at R=0

Num=2500; %This gives the number of points used by the simulation. A bigger number increases the precision
sint = linspace(x0,L,Num);

reachpert=0; % The variable that is used ot determine whether we have already reached the indentation depth about which we wish to perturb

cols='rgbmyckrgbmyck';

w0=0.00001;
testcount=0;

Nits=2e3;
results=zeros(Nits,6);

it=1;
itbigw=1;
trialw=[1 1.5 2 2.2  2.5 3 4 5 7.5 10 12.5 15 17.5 20 22.5 26.67 32.93 38.26 43.4];
wmax=10; 
itplot=0;

    %% Here is the starter loop

tic

solinit = bvpinit(sint,@mat4init);   
options=bvpset('RelTol',1e-6,'NMax',2e4);
sol = bvp4c(@mat4ode,@mat4bc,solinit,options);
fprintf('Step1\n');

toc

r=sol.x;
w=sol.y(1,:);
wp=sol.y(2,:);
w2p=sol.y(3,:);
w3p=sol.y(4,:);
psi=tau*r+sol.y(5,:);

for i=1:3 
    sol = bvp4c(@mat4ode,@mat4bc,sol,options);
end;
fprintf('Step2\n');

    %% Here is the main loop

    while w0<wmax
        w0
        
        sol = bvp4c(@mat4ode,@mat4bc,sol,options);
        
        if it>2
            
            
            r=sol.x;
            w=sol.y(1,:);
            wp=sol.y(2,:);
            w2p=sol.y(3,:);
            w3p=sol.y(4,:);
            psi=tau*r+sol.y(5,:);
            psip=sol.y(6,:)./r+tau;
            
            fce=real(2*pi*(psi.*wp-(r.*w3p+w2p-wp./r)));
            Nfit=10;

            p=polyfit(r(1:Nfit),fce(1:Nfit),1);
%             figure(18)
%             loglog(w0,p(2),'ro')
%             hold on

            results(it,4)=p(2);
    
            fprintf(membrane,'%.12g\t',w0);
            fprintf(membrane,'%.12g\t',p(2));
            fprintf(membrane,'\n');   
    
        end   
        
       
        if w0<10
            w0=1.5*(w0+0.0001);
            it=it+1;
        else
            w0=w0+1;
        end
    end

    status=fclose(membrane);

% Now going to try and solve the perturbation problem

% We test different integers numbers of wrinkles for various indentations
% until we find the det which crosses 0 first, i.e the most stable mode, and
% then interrogate at which indentation depth this occurs. 

nvals=6:1:8;       % wrinkle numbers to be tested 
Nw0vals=400;         % number of indentation depths to be tested
w0vals=linspace(15,24,Nw0vals);     % wrinkle numbers tested at indentation depths in this range

detvals=zeros(length(nvals),Nw0vals);

for i=1:Nw0vals
    w0=w0vals(i);
    sol = bvp4c(@mat4ode,@mat4bc,sol,options);
    r=sol.x;
    w=sol.y(1,:);
    wp=sol.y(2,:);
    w2p=sol.y(3,:);
    w3p=sol.y(4,:);
    psi=tau*r+sol.y(5,:);
    psip=sol.y(6,:)./r+tau;
    
    detvals(:,i)=CalcDets(nu,r,w,wp,w2p,w3p,psi,psip,nvals);
end

%figure(27)
%cm=colormap(jet(length(nvals)));
%hold all
for j=1:length(nvals)
    %plot(w0vals,detvals(j,:)/max((detvals(j,:))),'color',cm(j,:),'linewidth',2)
    idx = find(detvals(j,:)<0,1,'last');
    root_win(j,1) = w0vals(idx);
    if idx ~= length(detvals(j,:))
       root_win(j,2) = w0vals(idx+1);
    end
end
 
[min_val min_pos] = min(root_win(:,1));
wrinkle_number = min_pos+(nvals(1)-1);
critical_indentation = mean(root_win(min_pos,1:2));
details(tt,1) = tau; details(tt,2) = wrinkle_number; details(tt,3) = critical_indentation;

tt=tt+1;
end

tau
wrinkle_number
critical_indentation

%% Here are the functions that we use
  % -----------------------------------------------------------------------

  function dydx = mat4ode(x,y)
  
  % Derivative function. q is provided by the outer function.
   
  % Here are solving the equations including the value of psi at infinity
  
  dydx = [      y(2)
                y(3)
                y(4)
                -2*y(4)./x+y(3)./x.^2-y(2)./x.^3-y(1)+y(3).*(y(5)./x+tau)+y(2).*(y(6)./x.^2+tau./x)
                y(6)./x
                y(5)./x-6*(1-nu^2)*y(2).^2];   
  end
  % -----------------------------------------------------------------------

  function res = mat4bc(ya,yb)
  % Boundary conditions. This is the case for clamped boundary conditions. 
    res = [  ya(1)+w0 % w(0)=-w0
             ya(2) % w'(0)=0
             ya(6)-nu*ya(5)
             yb(1)
             yb(2)
             yb(5)
             ];
  end

  
  % -----------------------------------------------------------------------
      
    
% -------------------------------------------------------------------------
% Auxiliary function -- initial guess for the solution

function y = mat4init(x)

lp=0.5*(tau+sqrt(tau^2-4));
lm=0.5*(tau-sqrt(tau^2-4));
ypred=-2/log(lm/lp)*(besselk(0,lp*x)-besselk(0,lm*x));
%ypred=0*ypred;
 
    y = [  ypred
           0
           0
           0
           0
           0
            ];
    
end

end

% -------------------------------------------------------------------------
