%% Single Vibrational Mode + Superohmic Bath - Stability Diagram Simulation
% adapted from by Dr. Sowa paper 

clear all; clc; close all;

%% Parameters
k        = 8.617 * 10^-5;   % Boltzman constant
Tph      = 25e-3;           % Phononic temperature (K)
Te       = 25e-3;           % Electronic temperature (leads)
w0       = 0.007/2;         % Vibrational frequency (eV)
g0       = 0.0085/2;        % Vibrational coupling strength (eV)
eps_0    = -0.4875;         % Position of the molecular level at zero bias (eV)
wc       = 0.02;            % Bath cut-off frequency(eV)
lambda   = 0.15;            % Bath reorganisation energy (eV)

Cs       = 0.54;            % Capacitive coupling to the source
Cg       = 0.025;           % Capactivie coupling to the gate
yL       = 0.0000012;       % Left molecule-lead coupling (eV)
yR       = 0.0005;          % Right molecule-lead coupling (eV)
Y        = (yL + yR)/2;     % Lifetime broadening
betaph   = 1/(k*Tph);       % Inverse temperature


Vsd = -0.05:0.25e-3:0.05; % Bias voltage range
Vg = -19.76:4e-3:-18.4;     % Gate voltage range


%% Bath spectral density (superohmic)
w = 1e-8:1e-4:wc*20;                      % frequency range for the B(t) integral
J = lambda/2 * (w/wc).^3 .* exp(-w/wc);

% Plot 
figure(11)
plot(w,J./w); hold on;
scatter(w0,g0.^2/w0.^2); hold off;

%% Phononic correlation function
time=0:0.5:40000; % time range for the one-sided Fourier transform
B=zeros(1,numel(time));

% Loop over and compute the B integral. 
for i=1:numel(time)
   
   t=time(i);
   D1 = (g0/w0)^2 * ( (cos(w0.*t) - 1) .* coth(betaph.*w0./2) - 1i*sin(w0.*t)); % single mode part
   D2 = trapz(w,(J./w.^2) .* ( (cos(w.*t) - 1) .* coth(betaph.*w./2) - 1i*sin(w.*t) ));
   B(i) =  exp(D1) * exp(D2); %Phononic correlation function

end

%% Molecular density of states 
% To evaluate it only once we make use of the shift property of the Fourier
% transform

x = -0.4:5e-5:0.4;
i1 = zeros(1,numel(x)); % allocation

% compute the DOS (it is an integral)
for j=1:numel(x)
   i1(j) = trapz(time,(exp(1i * time * x(j)) .* B .* exp(- Y *time)));
end

% take the real value and flop the vector left to right
i1=real(i1);
i3=fliplr(i1);

figure(5)
plot(x,i1)  % this plots the molecular DOS (should be smooth and integrate to ~pi)
trapz(x,i1)

%% Compute the current based on the rate equation model 

I=zeros(numel(Vg),numel(Vsd)); % allocation

for kkk=1:numel(Vg)

   gate = Vg(kkk); % get a single Vg
   I_ = zeros(1,numel(Vsd)); % allocate current vector
   
   for kk=1:numel(Vsd)
      
      bias = Vsd(kk);
      muL = bias;                            % left chemical potential; potential applied only to one electrode
      muR = 0;                               % right chemical potential
      eps_M = eps_0 + Cs * bias - Cg * gate; % we assume that the molecular level shifts with bias


      fL=1./(exp((x+eps_M - muL)/(k*Te))+1); % compute the FD distribution for the left lead
      fR=1./(exp((x+eps_M - muR)/(k*Te))+1); % same for the right

      % The prefactors (1 or 2) here are what you need to change depending on
      % whether the transport is of N->N+-1 or N+-1-> N type (see the NComms by James and Jakub).

      y1L = 2*real(yL/pi * trapz(x,fL.*i1));
      y1R = 2*real(yR/pi * trapz(x,fR.*i1));
      y3L = 1*real(yL/pi * trapz(x,(1 - fL).*i3));
      y3R = 1*real(yR/pi * trapz(x,(1 - fR).*i3));


      current = (y1L * y3R - y3L * y1R)/(y1L + y1R + y3L + y3R); % rate equation current as result of redox reaction
      I_(kk) = current;

   end

I(kkk,:) = I_;
end

%% SD Plot
I=I' * 1.602e-3/6.582; % into Amps

G = diff(I,1,1)./(Vsd(2)-Vsd(1)) ./ 7.74809173e-5;  % G/G0; G = dI/dVsd

% figure
% hold on
% pcolor(Vg,Vsd,I)
% colormap inferno
% shading flat
% box on
% xlabel('Gate Voltage (V)')
% ylabel('Bias VOltage (V)')
% colorbar
% hold off

figure
pcolor(Vg,Vsd(2:end),G)
colormap viridis
shading flat
box on
xlabel('Gate Voltage (V)')
ylabel('Bias VOltage (V)')
colorbar
caxis([-0.5,8]*1e-5)
caxis([-0.5,2.5]*1e-5)

