% This file contains script to get information from oblique-angle project.
% The information includes:
% * 'wigner': see wigner transform of the beginning file (to set ROI and filter parameters)
% * 'frequency': get the frequency profile of the probe pulse (write files to the specified directory)
% * 'intensity': get the intensity pictures of the probe pulse (write files to the specified directory)
% We need to choose what information we would like to see (not saved automatically)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% BEGINNING OF THE PARAMETERS SECTION %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% File information for electric field and electron density distribution
fdir = 'C:\Users\kasim\Desktop\Simulation\oblique-angle\05-poc-05deg-lower-freq\h5-e3\';
fformat = 'e3-%06d.h5';
datasetname = '/e3';
dirOut = 'C:\Users\kasim\Desktop\Simulation\oblique-angle\05-poc-05deg-lower-freq\frequency\';

information = 'frequency';

startFileNum = 0; % inclusive
finishFileNum = 60; % exclusive

% region of interest, [xmin, xmax, kmin, kmax] in normalised values (all inclusive)
% xmin and xmax are relative to the moving window (in normalised values as well)
% this value is taken by looking at the wigner transform for the start and final files
ROIx = [-1, 999999];
ROIk = [4, 16];
% ROIk = [-9999999999, 9999999999];

% where we want to take data1D (expressed in ratio of full window, 0 for the top and 1 for the bottom)
yFrac = 0.50;

% scale down factor of the data right after extracting it from the h5 file
scalex = 2;
scaley = 1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% END OF THE PARAMETERS SECTION %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

nAve = [];
for i = [startFileNum:finishFileNum-1]
    % getting the complete filename first
    fname = sprintf(fformat, i);
    filename = strcat(fdir, fname);
    
    % read the attributes
    time = h5readatt(filename, '/', 'TIME');
    posMin = h5readatt(filename, '/', 'XMIN');
    posMax = h5readatt(filename, '/', 'XMAX');
    
    % get the E data
    data = h5read(filename, datasetname);
    data = data';
    [Ny Nx] = size(data);
    
    % scale down the data
    data = data(1:scaley:end, 1:scalex:end);
    Nx = Nx / scalex; Ny = Ny / scaley;
    
    % set the axis for wigner transform
    ex = linspace(posMin(1), posMax(1), Nx);
    ey = linspace(posMin(2), posMax(2), Ny);
    
    % get the ROI of position
    ROIxmin = find(ex-posMin(1) >= ROIx(1)); ROIxmin = ROIxmin(1);
    ROIxmax = find(ex-posMin(1) <= ROIx(2)); ROIxmax = ROIxmax(end);
    exROI = ex(ROIxmin:ROIxmax);
    NxROI = length(exROI);
    
    % take 1D data (horizontally)
    NyIdx = round(yFrac * Ny);
    if (NyIdx == 0) NyIdx = 1; end;
    data1D = data(NyIdx, ROIxmin:ROIxmax);
    wigner = tfrwv(data1D');
    
    % get the ROI boundaries
    ek = linspace(0, (Nx-1)/(posMax(1) - posMin(1)) * pi, NxROI);
    ROIkmin = find(ek >= ROIk(1)); ROIkmin = ROIkmin(1);
    ROIkmax = find(ek <= ROIk(2)); ROIkmax = ROIkmax(end);
    ekROI = ek(ROIkmin:ROIkmax);
    [eXROI, eKROI] = meshgrid(exROI, ekROI);
    wignerROI = wigner(ROIkmin:ROIkmax,:);
    
    % get the intensity and frequency profile
    s2 = sum(wignerROI, 1);
    ws2 = sum(eKROI .* wignerROI, 1);
    
    % get the initial frequency and the frequency change
    if (i == startFileNum)
        wInit = (ws2 ./ s2);
        initWindow = (s2 > 0.005 * max(s2));
    end
    dww0 = (ws2 ./ s2 - wInit) ./ wInit;
    dww0 = dww0 .* initWindow; % apply windowing
    
    % we now have s2, ws2, dww0, and nAve, so now start comparing the data here
    if (strcmp(information, 'wigner'))
        imagesc(exROI, ekROI, wignerROI);
        Nfilter = NxROI
        Nperiod = 2*pi/(posMax(1) - posMin(1)) * Nx * scalex / 2 * posMin(1) / time
        break;
        
    elseif (strcmp(information, 'frequency'))
        % set the plot
        h = figure;
        plot(exROI, dww0, 'b-', 'LineWidth', 2.5);
        title2 = get(gca, 'title');
        set(title2,'string','\Delta\omega/\omega_0 vs position','fontsize',18);
        xlhand = get(gca,'xlabel');
        set(xlhand,'string','z (normalized unit)','fontsize',18);
        ylhand = get(gca,'ylabel');
        set(ylhand,'string','\Delta\omega/\omega_0','fontsize',18);
        
        % title('Density profile measurement');
        % xlabel('z (normalized unit)');
        % ylabel('n/n_0 (arbitrary unit)');
        % xlim([exROI(1) exROI(end)]);
        
        % save the file
        fnameFig = sprintf('freq-%06d.png', i);
        filenameFig = strcat(dirOut, fnameFig);
        saveas(h, filenameFig, 'png');
        close(h);
    
    elseif (strcmp(information, 'intensity'))
        % set the plot
        h = figure;
        plot(exROI, s2, 'LineWidth', 2.5);
        
        % set the title and axis
        title2 = get(gca, 'title');
        set(title2,'string','Intensity profile','fontsize',18);
        xlhand = get(gca,'xlabel');
        set(xlhand,'string','z (normalized unit)','fontsize',18);
        ylhand = get(gca,'ylabel');
        set(ylhand,'string','Intensity (arbitrary unit)','fontsize',18);
        % title('Intensity profile');
        % xlabel('z (normalized unit)');
        % ylabel('Intensity (arbitrary unit)');
        
        % save the file
        fnameFig = sprintf('intensity-%06d.png', i);
        filenameFig = strcat(dirOut, fnameFig);
        saveas(h, filenameFig, 'png');
        close(h);
    end
    
end
