% This file is to check whether the spectrometer specification is sufficient to retrieve the phase modulation
% Author: M. F. Kasim (University of Oxford)
% Date: 11 February 2015

% experimental parameters for:
% * TA2 experiment:
%   * alpha = 7.69e25;
%   * tau0 = 1e-12;
%   * tau = .5e-3/c;
%   * n = 3e18*1e6;
%   * 
%   * lmbdaRange = 160e-9;
%   * lmbdaCentre = 800e-9;
%   * lmbdaRes = 0.1e-9;
%   * 
% * AWAKE experiment:
%   * alpha = 4.8077e23;
%   * tau0 = 160e-12;
%   * tau = .5e-3/c;
%   * n = 7e14*1e6;
%   * 
%   * lmbdaRange = 160e-9;
%   * lmbdaCentre = 800e-9;
%   * lmbdaRes = 0.1e-9;
%   * 

addpath('\\ppfs6.physics.ox.ac.uk\Particle\kasim\Projects\scripts\basic');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% PARAMETERS SECTION %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% fundamental variables
c = 3e8;
e = 1.6e-19;
m = 9.1e-31;
e0 = 8.85e-12;

% experimental variables
TA2 = 1;
AWAKE = 0;
AWAKE_NUV = 0;
if (TA2)
    tau0 = 140e-15; % duration of the pulses [ exp(-t^2/2/tau0^2) ]
    alpha = 7.69e25*1e-12/tau0; % linear chirp coefficient
    tau = 1e-3/c; % separation of the centre of the pulses
    n = 1e18*1e6*10; % plasma density in m^-3
    phi0 = 0.32/1.5*10; % amplitude of the phase modulation
    tOverTpMin = 3; % amount of filtering out in the spectrometer pattern
    snr = 10; % signal to noise ratio
elseif (AWAKE)
    tau0 = 160e-12/3;
    alpha = 2.4063e23;
    tau = 1e-2/c;
    n = 7e14*1e6;
    phi0 = 0.2*.2;
    tOverTpMin = 1;
    snr = Inf;
elseif (AWAKE_NUV)
    tau0 = 80e-12/3;
    alpha = 2.4063e23*4;
    tau = 1e-2/c;
    n = 7e14*1e6;
    phi0 = 0.2*0.2/1.2;
    tOverTpMin = 1;
    snr = 20;
end

% diagnostic variables
if (TA2)
    lmbdaRange = 160e-9; % total range of the measurement
    lmbdaCentre = 800e-9; % centre wavelength
    lmbdaRes = 0.1e-9; % wavelength resolution
    cameraBits = 16; % bit depth of one pixel
elseif (AWAKE)
    lmbdaRange = 30e-9;
    lmbdaCentre = 780e-9;
    lmbdaRes = 0.01e-9;
    cameraBits = 16;
elseif (AWAKE_NUV)
    lmbdaRange = 10e-9;
    lmbdaCentre = 390e-9;
    lmbdaRes = 0.01e-9;
    cameraBits = 16;
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% END OF PARAMETERS SECTION %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% the x-axes
lmbda = [lmbdaCentre-lmbdaRange/2:lmbdaRes:lmbdaCentre+lmbdaRange/2];
w = 2*pi*c./lmbda;
t = linspace(-1, 1, length(lmbda)) * lmbdaCentre^2/c/lmbdaRes / 2;

% constructing the phase spectrum of the signal
w0 = 2*pi*c/lmbdaCentre;
wp = (n*e*e/m/e0)^.5;
tp = 2*pi/wp;
Erw = tau0/(1+1i*alpha*tau0^2).^5 .* exp(-(w-w0).^2*tau0^2/2/(1+1i*alpha*tau0^2));
dphiw = phi0 * sin((w-w0)/alpha * wp); % phase modulation

% modulus of the phase spectrum of the signal
Iw = 2*abs(Erw).^2 .* (1 + cos(dphiw + w*tau));
Iw0 = 2*abs(Erw).^2 .* (1 + cos(w*tau));
Iw1 = 2*abs(Erw).^2 .* (1 + cos(w*0));

% add the noise into the signal
NIw = size(Iw);
Iwtot = sum(Iw);
Iw  = Iw  + 2*Iwtot * rand(NIw) * 10^(-snr/10) / length(Iw);
Iw0 = Iw0 + 2*Iwtot * rand(NIw) * 10^(-snr/10) / length(Iw); %(rand(NIw)) * max(Iw0) * 10^(-snr/20);
Iw1 = Iw1 + 2*Iwtot * rand(NIw) * 10^(-snr/10) / length(Iw); %(rand(NIw)) * max(Iw1) * 10^(-snr/20);

% e1 = max(Iw0) * 10^(-snr/20);
% e2 = Iwtot / length(Iw) * 10^(-snr/10);

% camera bits resolution
maxCamValue = (2^cameraBits - 1);
normalisedValue = 1 / max([Iw Iw0]) * maxCamValue;
Iw  = (Iw  * normalisedValue);
Iw0 = (Iw0 * normalisedValue);
Iw1 = (Iw1 * normalisedValue);
Iw  = Iw  .* (Iw  <= maxCamValue) + maxCamValue * (Iw  > maxCamValue);
Iw0 = Iw0 .* (Iw0 <= maxCamValue) + maxCamValue * (Iw0 > maxCamValue);
Iw1 = Iw1 .* (Iw1 <= maxCamValue) + maxCamValue * (Iw1 > maxCamValue);
Iw  = int64(Iw );
Iw0 = int64(Iw0);
Iw1 = int64(Iw1);

% processing in the computer in double precision
Iw  = double(Iw );
Iw0 = double(Iw0);
Iw1 = double(Iw1);

% filter out the non-oscillating part (can be improved to provide better results)
Iwfilter = fftshift(fft(Iw)) .* (t > tOverTpMin*tp);
Iw0filter = fftshift(fft(Iw0)) .* (t > tOverTpMin*tp);
% Iwfilter  = Iw  - Iw1 / 2;
% Iw0filter = Iw0 - Iw1 / 2;

dphiMeas = angle(ifft(ifftshift(Iwfilter)));
dphiMeas0 = angle(ifft(ifftshift(Iw0filter)));

% normalise the phase so it can have range outside (-pi pi]
maximumPhaseDiff = single((2*pi) * 0.6);
dphiMeas = normalisePhase(dphiMeas, maximumPhaseDiff);
dphiMeas0 = normalisePhase(dphiMeas0, maximumPhaseDiff);

dphiMeas2 = dphiMeas - dphiMeas0;

% plot the retrieved phase
tMeas = (w - w0) / alpha;
filterArray = boolean((tMeas >= -5*tp) .* (tMeas <= 5*tp));
data1d = dphiMeas2(filterArray)-mean(dphiMeas2(filterArray));
plot(tMeas(filterArray)/tp, data1d, 'b-');
xlabel('t / plasma period');
ylabel('Retrieved \Delta\phi / rad');
title('Phase modulation measurement');
% plot(t/tp, fftshift(abs(fft(Iw))));
% plot(abs(fft(data1d)));
