%% Code for retrospective optimization of CS undersampling parameters
% Used for "Optimization of Undersampling Parameters for 3D Intracranial
% Compressed Sensing MR Angiography at 7 Tesla"
% by Matthijs H.S. de Buck, Peter Jezzard, and Aaron T. Hess (2021)
%
% The BART toolbox needs to be installed locally for this script to run:
% https://mrirecon.github.io/bart/
% BART v0.4.02 was used originally, but other versions which use the same
% (or comparable) implementations of the tools 'pics' and 'calib' might
% work as well.
%
% Also uses the vdPoisMex implementation for generating variable density 
% Poisson disc undersampling masks, by Michael Lustig:
% https://people.eecs.berkeley.edu/~mlustig/Software.html
%   (folder: ESPIRiT/uils/vdPoisMex.m)
% 
% Depending on the matrix size of the original data, reasonably high memory
% is required to run this code. It's not optimized for parallelization.
%
% We acknowledge Joseph G. Woods for providing the code for generating MRA
% vessel-masks (createVesselMask.m)
% 
% Contact: matthijs.debuck@ndcn.ox.ac.uk / thijsdebuck@live.nl

%% 1) Extract and reorder full k-space
% As a starting point / input, provide the original (fully-sampled) k-space
% data, k_original, as a 3D, single-slab, multi-channel array of dimensions
% [nx, ny, nz, n_channels], with nx the readout direction. This step is not 
% included in this script since it's different for different 
% vendors/acquisitions.

%% 2) Compute required sensitivity maps
% Calculate the sensitivity maps (using BART) for different amounts of
% calibration data (10x10 up to 32x32 here, but can be changed). Since
% they're calculated once here, they don't need to be calculated again for
% each individual reconstruction

disp("Calculating sensitivity maps based on full k-space. This can take a while.")
[calib, ~] = bart('ecalib -m1 -k4 -r10', k_original);
sens10 = bart('slice 4 0', calib);
[calib, ~] = bart('ecalib -m1 -k4 -r12', k_original);
sens12 = bart('slice 4 0', calib);
[calib, ~] = bart('ecalib -m1 -k4 -r16', k_original);
sens16 = bart('slice 4 0', calib);
[calib, ~] = bart('ecalib -m1 -k4 -r20', k_original);
sens20 = bart('slice 4 0', calib);
[calib, ~] = bart('ecalib -m1 -k4 -r24', k_original);
sens24 = bart('slice 4 0', calib);
[calib, ~] = bart('ecalib -m1 -k4 -r28', k_original);
sens28 = bart('slice 4 0', calib);
[calib, ~] = bart('ecalib -m1 -k4 -r32', k_original);
sens32 = bart('slice 4 0', calib);

disp("All sensitivity maps calculated")
clear calib

%% 3) Compute fully-sampled (reference) image, using 32-calibs sensitivity map
disp('Starting recon for fully-sampled image')
img3D_fullysampled = Sk2img(sens32, k_original);
mip_fullysampled = mip(img3D_fullysampled, 3, 0.2); %max. intensity projection

clear img3D_fullysampled

% Show MIP of reconstruction
complot(mip_fullysampled), axis off

%% 4) Draw brain mask (/remove skull)
% The easiest way to generate a brain mask is to manually draw it based on
% the fully-sampled dataset, and then apply that brainmask to all later
% reconstructions from retrospectively undersampled data from the same
% dataset. This step is required for accurate vessel segmentation later on,
% when calculating the SSIM.

% Draw manually (on MIP)
complot(mip_fullysampled), colormap hot, axis equal
brainmask = drawfreehand
brainmask = brainmask.createMask;
 
mip_ref_mask = mip_fullysampled .* brainmask;

%% 5) Initialize simulation parameters  
% Set an array with the acceleration factor, polynomial order, and
% calibration region size for each undersampled mask
% 2D array: 3xn_tests; first dimension = [acc, pp, calib]

clear test_params
counter = 0;
for n_calibs = [10, 12:4:32]   % calibration region sizes
    for pp = 0:0.4:4.4         % polynomial orders
        for acc = 5:2:15       % (effective) acceleration factors
            counter = counter + 1;
            test_params(:, counter) = [acc, pp, n_calibs];
        end
    end
end

% test_params(:,counter+1:counter*2) = test_params;  %Optional econd average
clear n_calibs pp acc counter

%% 6) Iteratively undersample fully-sampled k-space and reconstruct
n_tests = size(test_params, 2);
tic

for test = 1:n_tests
    %select undersampling parameters
    acc = test_params(1, test); pp = test_params(2, test); n_calibs = test_params(3, test);
    
    %calculate and apply undersampling mask accordingly
    [k_under, us_factor] = undersample_effacc(k_original, acc, pp, n_calibs, 1.1);  
    disp('Undersampling completed for test '+string(test)+' on '+string(datetime()))
    
    % Select appropriate sensitivity maps
    switch n_calibs
        case 10, sens = sens10;
        case 12, sens = sens12;
        case 16, sens = sens16;
        case 20, sens = sens20;
        case 24, sens = sens24;
        case 28, sens = sens28;
        case 32, sens = sens32;
        otherwise, error('No suitable sensitivity maps available');
    end

    % Perform CS recon using sensitivities
    disp('Starting recon')
    pics_recon = bart(char('pics -i20 -R W:7:0:0.007'), k_under, sens);
    pics_recon = squeeze(pics_recon);
    
    disp('Reconstruction completed for test '+string(test))
    
    % Store effective undersampling factor (on rare occasions, the
    % undersampling factor doesn't converge to the desired value (and turns
    % out at ~30 instead). Check whether the values in 
    % effective_undersampling agree with the desired acceleration factors
    % at the end to be sure.
    effective_undersampling(test, 1) = us_factor; 
    
    % Store MIP of reconstruction
    mips_tests(:,:,test) = mip(pics_recon, 3, 0.2);
    
    clear pics_recon sens k_under
end

toc

% Mask all MIPs
mips_masked = zeros(size(mips_tests));
for test = 1:n_tests
    mips_masked(:,:,test) = mips_tests(:,:,test).*brainmask;
end

%% 7) Quantify results (in this case, using the vessel-masked SSIM)
clear SSIM_tests

rel_peakprominence = 0.15;   %Minimum required prominence of vessels to be included in peak_count

for test = 1:n_tests
    % SSIM_tests includes the vessel-masked SSIM values, which is the only
    % metric used in this code. Alternatively metrics that can be used
    % include the overall SSIM, RMSE, slice-wise RMSE, or the number of
    % detected peaks (Meixner, 2019). Here, the vessel-masked SSIM is used
    % for MRA data based on the findings by Akasaka et al., 2016
    
    [~, ~, SSIM_tests(test)] = img_compare(mip_ref_mask, squeeze(mips_masked(:,:,test)), 0);
end

%% 8) Plot results
% Prepare plots
clear SSIM_matrix
accs = unique(test_params(1,:));
pps = unique(test_params(2,:));
calibs = unique(test_params(3,:));

for i = 1:numel(calibs)
    n_calibs = calibs(i);
    SSIM_matrix(:,:,i) = us_results_plot_prep(SSIM_tests, test_params, n_calibs);
end

% Plot for each acceleration factor (assuming 6 acc factors)
diff_colorbar = true; %use different color bar scalings yes/no

figure('Renderer', 'painters', 'Position', [100 50 1000 600])
sgtitle('Reconstruction accuracy for various accelerations')
    
for index = 1:numel(accs)
    acc = accs(index);
    subplot(3,2,index)
    matrix = squeeze(SSIM_matrix(find(accs==acc), :, :))';
    
    imagesc(matrix)
    set(gca, 'XTick', 1:numel(pps), 'XTickLabel', pps)
    set(gca, 'YTick', 1:numel(calibs), 'YTickLabel', calibs)
    if index == 5 || index == 6, xlabel('Polynomial order'), end
    if mod(index,2) == 1, ylabel('#calibration lines'), end
    
    title('Acceleration = '+string(acc)) 
    colorbar()
    if ~diff_colorbar, caxis([0.75 1]*max(SSIM_matrix(:)));
    else caxis([0.95 1]*max(matrix(:))), end
end
