function [array_us, factor_us_real] = undersample_effacc(array, acc_eff, pp, calib, pct_acc)
% [array_us, factor_us_real] = undersample_effacc(array, acc_eff, pp, calib, pct_acc)
% Finds an undersampling mask where the effective acceleration differs from
% the input acceleration (acc_eff) by less than pct_acc percent by repeatedly
% calculating new masks based on adjusted acceleration factors.
% Undersampling mask is applied to 2nd and 3rd dimension of 'array'
%
% Uses the vdPoisMex implementation for generating variable density Poisson
% disc undersampling masks, by Michael Lustig:
% https://people.eecs.berkeley.edu/~mlustig/Software.html
%   (folder: ESPIRiT/uils/vdPoisMex.m)
%
% Might require to first run "mex -R2018a vdPoisMex.c"
% 
%   matthijs.debuck@ndcn.ox.ac.uk

[sx, sy, sz, ~] = size(array);

% acc_eff = target acceleration; acc = current estimate for vdPoisMex input
acc = 1/(1/acc_eff - 1/(sy*sz/calib/calib)); %very rough correction of initial guess

% calculate mask for first estimate
% note: voxel size of 0.31mm isotropic used here
[mask_2D] = vdPoisMex(sy,sz,sy*0.31,sz*0.31,sqrt(acc),sqrt(min(acc,sz)),0,0,pp);

% add square calibration region
mask_2D(floor(end/2-calib/2)+1:floor(end/2+calib/2),floor(end/2-calib/2)+1:floor(end/2+calib/2)) = 1;

%calculate "actual" undersampling factor
factor_us_real = numel(mask_2D)/nnz(mask_2D);

% Iteratively generate an undersampling mask and check if its effective
% acceleration is within pct_acc from the target (acc_eff). If not, try
% again with an adjusted input acceleration factor. 
n_tries = 0; n_tries_max = 1500;
while ~(isequal(check_acc(acc_eff,factor_us_real,pct_acc), 'correct')) && (n_tries < n_tries_max)
    
    %in rare cases (esp. with extremely high polynomial orders at high acc
    %factors), the accelerations fails to converge - so if it still hasn't
    %converged over n_tries_max, allow a larger error 
    n_tries = n_tries + 1; 
    if n_tries == ceil(n_tries_max/2), pct_acc = pct_acc*2; end
    
    %adjust acc in an attempt to converge towards acc_eff
    if isequal(check_acc(acc_eff, factor_us_real, pct_acc), 'lower')
        acc = acc * (1+pct_acc/100);
    else
        acc = acc * (1-pct_acc/100);
    end
    
    %calculate mask using adjusted acc
    [mask_2D] = vdPoisMex(sy,sz,sy*0.31,sz*0.31,sqrt(acc), sqrt(acc),0,0,pp);
    
    %overlay a square calibration region, needed for ecalib
    mask_2D(floor(end/2-calib/2)+1:floor(end/2+calib/2),floor(end/2-calib/2)+1:floor(end/2+calib/2)) = 1;
    factor_us_real = numel(mask_2D)/nnz(mask_2D);
end

% Correct the outputs from vdPoisMex (bc of squared calibration region)
disp('REAL Actual acceleration (incl. full calib area) = '+string(factor_us_real)+', found after '+string(n_tries)+' iterations.')

% Apply the mask to our data (along dims 2 and 3)
array_perm = permute(array,[2 3 4 1]);
array_us = array_perm.*mask_2D;
array_us = permute(array_us,[4 1 2 3]);
end

%helper function to check acceleration 
function [label] = check_acc(acc_eff,factor_us_real,pct_acc)
    if (factor_us_real <= acc_eff * (1-pct_acc/100) )
        label = 'lower';
    elseif (factor_us_real >= acc_eff * (1+pct_acc/100) )
        label = 'higher';
    else
        label = 'correct';
    end
end
