function [RMSE, SSIM_nomask, SSIM_vesselmask] = img_compare(img1, img2, plotBool)
%Quantifies the similarity of two 2D-images (made for MRA-mips, but can
%be used for different data) based on their RMSE and SSIM, and the SSIM
%after applying a vessel-mask. Use:
%[RMSE, SSIM_nomask, SSIM_vesselmask] = img_compare(img1, img2, plotBool)
%   img1     =   reference image
%   img2     =   reconstructed image (same dimensions as img1)
%   plotBool =   display output-report yes/no
%
%   matthijs.debuck@ndcn.ox.ac.uk

if ~exist('plotBool','var')
    plotBool = false;
end

img1 = squeeze(img1); img2 = squeeze(img2);

%Adjust data format and rescale to 99th percentile
percentile_norm = 99;
img1 = single(abs(img1)); img1(isnan(img1)) = 0;
img1 = img1 / prctile(abs(img1(:)), percentile_norm);
img2 = single(abs(img2)); img2(isnan(img2)) = 0;
img2 = img2 / prctile(abs(img2(:)), percentile_norm);

%Calculate vessel- and brain-mask (assuming img1 is already brain-masked)
%createVesselMask: Written by Joseph G. Woods, FMRIB, Oxford, June 2019
vesselmask = createVesselMask(img1,[], 10, 1, 0);
brainmask = ~(img1 == 0);

%Compute RMSE
diffmap = img1 - img2;
RMSE = sqrt(nanmean(diffmap(brainmask(:)).^2));

%Calculate SSIM-maps
[~, SSIMMAP] = ssim(img1, img2, 'radius', 2);  %Discard the default SSIM-value, since it includes the irrelevant (masked-out) background 
SSIM_vesselmask = nanmean(SSIMMAP(vesselmask(:)));
SSIM_nomask = nanmean(SSIMMAP(brainmask(:)));

if plotBool     %Display output report
    grayNess = 0.;  %Change for pptx presentations with black background
    
    figure('Renderer', 'painters', 'Position', [100 100 900 500])
    subplot_tight(2,3,1);
    imagesc(img1)
%     caxis([0. 1.])
    colorbar('Color',[1 1 1]*grayNess);
    axis equal off
    title('Reference image','Color',[1 1 1]*grayNess)
    
    subplot_tight(2,3,2)
    imagesc(img2)
%     caxis([0. 1.])
    colorbar('Color',[1 1 1]*grayNess)
    axis equal off
    title('Reconstructed image','Color',[1 1 1]*grayNess)
    
    subplot_tight(2,3,3)
    imagesc(abs(diffmap))
    caxis([0 max(abs(diffmap(:)))])
%     caxis([0 0.1])
    colorbar('Color',[1 1 1]*grayNess)
    axis equal off
%     title({'Abs. difference','(RMSE = '+string(round(RMSE,3))+')'})
    title('Abs. difference (RMSE = '+string(round(RMSE,3))+')','Color',[1 1 1]*grayNess)
    
    subplot_tight(2,3,4);
    imagesc(vesselmask)
    caxis([0 1])
    colorbar('Color',[1 1 1]*grayNess);
    axis equal off
    title('Vessel mask','Color',[1 1 1]*grayNess)
    
    subplot_tight(2,3,5);
    imagesc(SSIMMAP.*brainmask)
    caxis([max(0,nnz(SSIM_vesselmask-1)*(2*SSIM_vesselmask-1)) 1])  %nnz(SSIM-1) is a workaround for SSIM=1
    colorbar('Color',[1 1 1]*grayNess)
    axis equal off
%     title({'SSIM','(mean = '+string(round(SSIM_nomask,3))+')'})
    title('SSIM (mean = '+string(round(SSIM_nomask,3))+')','Color',[1 1 1]*grayNess)
    
    subplot_tight(2,3,6)
    imagesc(SSIMMAP.*vesselmask)
    caxis([max(0,nnz(SSIM_vesselmask-1)*(2*SSIM_vesselmask-1)) 1])  %nnz(SSIM-1) is a workaround for SSIM=1
    colorbar('Color',[1 1 1]*grayNess)
    axis equal off
%     title({'Masked SSIM','(mean = '+string(round(SSIM_vesselmask,3))+')'})
    title('Masked SSIM (mean = '+string(round(SSIM_vesselmask,3))+')','Color',[1 1 1]*grayNess)
    
    colormap hot
end
end