clear all

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This script expects to find the following files in the same folder:
%
% CT-for-REG_2020-10-12_12h48_ungated-fast.nii
% DCE-MRI_image001.nii
% DCE.timestamp
%
%
% All nifti input and output is done using "Tools for NIfTI and ANALYZE image" by Jimmy Shen 
% https://www.mathworks.com/matlabcentral/fileexchange/8797-tools-for-nifti-and-analyze-image
%
% Non-ridid registration is done using the MIND algorithm by Mattias Heinrich 
% http://www.mpheinrich.de
% To run this code you need to compile (mex) the following files:
% SymGN/trilinearSingle.cpp
% SymGN/pointsor3d2.c
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



% Choose data folder
[DataPath] = pda_uigetdir('DCE_CT_Data_Path','Select data folder');


% Load DCE image sequence
[DCE,mr_voxdim,mr_hdr] = pda_load_nii(fullfile(DataPath,'DCE-MRI_image001.nii'));

% Load timestamp file
TimeName = fullfile(DataPath,'DCE.timestamp');
[T,UT] = pda_Read_K0_Times2(TimeName);
T = T(1:30);
T = T - T(1);
savename = fullfile(DataPath,'T.mat');
save(savename,'T');

% Create mean and iAUC90 image volumes from DCE sequence
TimeLimit = 90;
GroupTimeLimit = 0;
DCE_Params.DCE_InjectionPoint = 11;
DCE_Name = 'DCE-MRI_image001.nii';
pda_DCEMRI_CreateAverageVol(DataPath,DCE_Name,DCE_Params);
pda_DCEMRI_CreateBasicParameterMaps(DataPath,DCE_Name,TimeLimit,GroupTimeLimit,DCE_Params);

% Base registration on mean of the DCE sequence 
MR_Volume = mean(DCE,4);



% Load CT image
[CT_Volume,ct_voxdim,ct_hdr] = pda_load_nii(fullfile(DataPath,'CT-for-REG_2020-10-12_12h48_ungated-fast.nii'));

% Perform initial rigid registration
Crop = false;
[MR_reg,reg_data] = pda_Register_MR_2_MILabs(MR_Volume,mr_voxdim,CT_Volume,ct_voxdim,Crop);



% Select common reference point - in case the initial registration has failed in the Z direction
h = figure('name', 'Identify Common Feature: create point on each image and press return');
hold on;
image = squeeze(CT_Volume(:,round(size(CT_Volume,2)/2),:)); 
image = pda_imnorm(image');
subplot(1,2,1), imagesc(image,[0 0.7]); colormap('gray');


image = squeeze(MR_reg(:,round(size(CT_Volume,2)/2),:)); 
image = pda_imnorm(image');
subplot(1,2,2), imagesc(image,[0 0.5]); colormap('gray');

[x,y] = getpts();
close(h);
YOffSet = y(2) - y(1);


if YOffSet ~= 0
    if YOffSet < 0
        YOffSet = size(CT_Volume,3) - (-YOffSet);
    end
    reg_data.YOffSet = YOffSet;
    MR_reg = pda_ShiftVolume(MR_reg,3,round(YOffSet));
else
    reg_data.YOffSet=0;
end


% Perform non-rigid registration
[MR_reg,reg_data] = pda_Register_MR_2_MILabs_MIND(MR_reg,CT_Volume,reg_data);
pda_save_nii(MR_reg,ct_voxdim,fullfile(DataPath,'Mean_DCE_Registered.nii'),8,ct_hdr.hist);


% Load iAUC90 image
[iAUC90,mr_voxdim,mr_hdr] = pda_load_nii(fullfile(DataPath,'iAUC90.nii'));

% Apply rigid transformation to iAUC90
[iAUC90_Reg] = pda_Apply_Register_MR_2_MILabs(iAUC90,reg_data);
iAUC90_Reg = pda_ShiftVolume(iAUC90_Reg,3,round(YOffSet));

% Apply non-rigid transformation to iAUC90
[iAUC90_Reg] = pda_Apply_Register_MR_2_MILabs_MIND(iAUC90_Reg,reg_data);

% Save registered iAUC90 image volume
pda_save_nii(iAUC90_Reg,ct_voxdim,fullfile(DataPath,'iAUC90_Registered.nii'),8,ct_hdr.hist);