********************************************************************************
/*
Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
Global Multidimensional Poverty Index 2.0 - Togo MICS 2017 [STATA do-file]. 
Available from OPHI website: http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk
*/
********************************************************************************

clear all 
set more off
set maxvar 10000



global path_in "../rdta/Togo MICS 2017"      // folder path
global path_out "cdta"
global path_ado "ado"


********************************************************************************
*** Togo MICS 2017 ***
********************************************************************************


********************************************************************************
*** Step 1: Data preparation 
********************************************************************************
	


********************************************************************************
*** Step 1.1 CH - Children under 5 years
********************************************************************************


use "$path_in/ch.dta", clear 

rename _all, lower	


gen double ind_id = hh1*10000 + hh2*100 + ln 
format ind_id %20.0g


duplicates report ind_id		// 5,030 children. Matches report (p.i)


gen child_CH=1 



*** Variable: SEX ***
codebook hl4, tab (9) 
	//"1" for male ;"2" for female 
clonevar gender = hl4
label define lab_gender 1"male" 2"female", replace
label values gender lab_gender
tab gender


*** Variable: AGE ***
desc cage caged
tab cage, miss
	//Age in months: information missing for 88 children 
tab caged, miss
	/*Age in days: information missing for 88 children. We will use age in days 
	as it result in more accurate anthropometric measures. */
sum caged if cage>9000
count if caged < 0 	
clonevar age_days = caged
replace age_days = trunc(cage*(365/12)) if age_days>=9000 & cage<9000
sum age_days 
	//4,492 children measured. Matches the country report on page i
gen str6 ageunit = "days"
lab var ageunit "Days"


*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook an8, tab (9999) 
clonevar weight = an8	
replace weight = . if an8>=99 
	//All missing values or out of range are replaced as "."
tab	uf17 an8 if an8>=99 | an8==., miss   
sum weight 	


*** Variable: HEIGHT (CENTIMETERS)
codebook an11, tab (9999) 
clonevar height = an11
replace height = . if an11>=999 
	//All missing values or out of range are replaced as "."
tab	uf17 an11 if an11>=999 | an11==., miss
sum height 

	
*** Variable: MEASURED STANDING/LYING DOWN	
codebook an12  
gen measure = "l" if an12==1 
	//Child measured lying down
replace measure = "h" if an12==2 
	//Child measured standing up
replace measure = " " if an12==9 | an12==0 | an12==. 
	//Replace with " " if unknown
tab measure 

		
*** Variable: OEDEMA ***
lookfor oedema edema
gen str1 oedema = "n"  
	//This variable assumes no one has oedema


*** Variable: SAMPLING WEIGHT ***
	/* We don't require individual weight to compute the z-scores of a child. 
	So we assume all children in the sample have the same weight */
gen sw = 1	
sum sw



*** Indicate to STATA where the igrowup_restricted.ado file is stored:
	***Source of ado file: http://www.who.int/childgrowth/software/en/
adopath + "$path_ado/igrowup_stata"

*** We will now proceed to create three nutritional variables: 
	*** weight-for-age (underweight),  
	*** weight-for-height (wasting) 
	*** height-for-age (stunting)

*** We specify the first three parameters we need in order to use the ado file:
	*** reflib, 
	*** datalib, 
	*** datalab

/* We use 'reflib' to specify the package directory where the .dta files 
containing the WHO Child Growth Standards are stored. */	
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"


/* We use datalib to specify the working directory where the input STATA 
dataset containing the anthropometric measurement is stored. */
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"


/* We use datalab to specify the name that will prefix the output files that 
will be produced from using this ado file (datalab_z_r_rc and datalab_prev_rc)*/
gen str30 datalab = "children_nutri_tgo"
lab var datalab "Working file"

	
/*We now run the command to calculate the z-scores with the adofile */
igrowup_restricted reflib datalib datalab gender age_days ageunit weight ///
height measure oedema sw


/*We now turn to using the dta file that was created and that contains 
the calculated z-scores to create the child nutrition variables following WHO 
standards */
use "$path_out/children_nutri_tgo_z_rc.dta", clear 

		
*** Standard MPI indicator ***
	//Takes value 1 if the child is under 2 stdev below the median & 0 otherwise
	
gen	underweight = (_zwei < -2.0) 
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "Child is undernourished (weight-for-age) 2sd - WHO"
tab underweight, miss 


gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "Child is stunted (length/height-for-age) 2sd - WHO"
tab stunting, miss 


gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "Child is wasted (weight-for-length/height) 2sd - WHO"
tab wasting, miss 


*** Destitution indicator  ***
	//Takes value 1 if the child is under 3 stdev below the median & 0 otherwise
	
gen	underweight_u = (_zwei < -3.0) 
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "Child is undernourished (weight-for-age) 3sd - WHO"


gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "Child is stunted (length/height-for-age) 3sd - WHO"


gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "Child is wasted (weight-for-length/height) 3sd - WHO"

 
count if _fwei==1 | _flen==1 
	/*Togo MICS 2017: 12 children were replaced as missing because 
	they have extreme z-scores which are biologically implausible. */
	
count  
	/*Togo MICS 2017: the number of eligible children is 5,030 as in 
	the country report (p.i). */		
	
clonevar weight_ch = chweight
	
	//Retain relevant variables:
keep  ind_id child_CH ln weight_ch underweight* stunting* wasting*  
order ind_id child_CH ln weight_ch underweight* stunting* wasting*
sort  ind_id
save "$path_out/TGO17_CH.dta", replace
	
	
	//Erase files from folder:
erase "$path_out/children_nutri_tgo_z_rc.xls"
erase "$path_out/children_nutri_tgo_prev_rc.xls"
erase "$path_out/children_nutri_tgo_z_rc.dta"


********************************************************************************
*** Step 1.2  BH - BIRTH RECODE 
*** (All females 15-49 years who ever gave birth)  
********************************************************************************


use "$path_in/bh.dta", clear

rename _all, lower	
		
*** Generate individual unique key variable required for data merging using:
	*** hh1=cluster number: 3-digit long; 
	*** hh2=household number: 2-digit long; 
	*** wm3=women's line number: 2-digit long
	*In total: 7-digit long
gen double ind_id = hh1*10000 + hh2*100 + wm3
format ind_id %20.0g
label var ind_id "Individual ID"


		
desc bh4c bh9c	
gen date_death = bh4c + bh9c	
	//Date of death = date of birth (bh4c) + age at death (bh9c)	
gen mdead_survey = wdoi-date_death	
	//Months dead from survey = Date of interview (wdoi) - date of death	
replace mdead_survey = . if (bh9c==0 | bh9c==.) & bh5==1	
	/*Replace children who are alive as '.' to distinguish them from children 
	who died at 0 months */ 
gen ydead_survey = mdead_survey/12

	//Years dead from survey	
gen age_death = bh9c if bh5==2
label var age_death "Age at death in months"	
tab age_death, miss
			
	
codebook bh5, tab (10)	
gen child_died = 1 if bh5==2
replace child_died = 0 if bh5==1
replace child_died = . if bh5==.
label define lab_died 0"child is alive" 1"child has died"
label values child_died lab_died
tab bh5 child_died, miss 
	
	
bysort ind_id: egen tot_child_died = sum(child_died) 
	//For each woman, sum the number of children who died
		
	
	//Identify child under 18 mortality in the last 5 years
gen child18_died = child_died 
replace child18_died=0 if age_death>=216 & age_death<.
label define lab_u18died 1 "child u18 has died" 0 "child is alive/died but older"
label values child18_died lab_u18died
tab child18_died, miss	
	
bysort ind_id: egen tot_child18_died_5y=sum(child18_died) if ydead_survey<=5
	/*Total number of children under 18 who died in the past 5 years 
	prior to the interview date */	
	
replace tot_child18_died_5y=0 if tot_child18_died_5y==. & tot_child_died>=0 & tot_child_died<.
	/*All children who are alive or who died longer than 5 years from the 
	interview date are replaced as '0'*/
	
replace tot_child18_died_5y=. if child18_died==1 & ydead_survey==.
	//Replace as '.' if there is no information on when the child died  

tab tot_child_died tot_child18_died_5y, miss

bysort ind_id: egen childu18_died_per_wom_5y = max(tot_child18_died_5y)
lab var childu18_died_per_wom_5y "Total child under 18 death for each women in the last 5 years (birth recode)"
	

	//Keep one observation per women
bysort ind_id: gen id=1 if _n==1
keep if id==1
drop id
duplicates report ind_id 


gen women_BH = 1 
	//Identification variable for observations in BH recode

	
	//Retain relevant variables
keep ind_id women_BH childu18_died_per_wom_5y 
order ind_id women_BH childu18_died_per_wom_5y
sort ind_id
save "$path_out/TGO17_BH.dta", replace	


********************************************************************************
*** Step 1.3  WM - WOMEN's RECODE  
*** (Eligible females 15-49 years in the household)
********************************************************************************

use "$path_in/wm.dta", clear 
	
rename _all, lower	

	
*** Generate individual unique key variable required for data merging using:
	*** hh1=cluster number: 3-digit long; 
	*** hh2=household number 2-digit long; 
	*** wm3=women's line number: 2-digit long
	*In total: 7-digit long
gen double ind_id = hh1*10000 + hh2*100 + wm3
format ind_id %20.0g
label var ind_id "Individual ID"

duplicates report ind_id 
	//7,657 women eligible for interview. Matches report (p. i). 

gen women_WM =1 
	//Identification variable for observations in WM recode

		
	
	//Retain relevant variables:
gen religion_wom = .
lab var religion_wom "Women's religion"	

gen ethnic_wom = .
lab var ethnic_wom "Women's ethnicity"	

lookfor insurance assurance
clonevar insurance_wom = wb18 
label var insurance_wom "Women have health insurance"	

codebook *_wom, tab (99)

codebook mstatus ma6, tab (10)
tab mstatus ma6, miss 
gen marital = 1 if mstatus == 3 & ma6==.
	//1: Never married
replace marital = 2 if mstatus == 1 & ma6==.
	//2: Currently married
replace marital = 3 if mstatus == 2 & ma6==1
	//3: Widowed	
replace marital = 4 if mstatus == 2 & ma6==2
	//4: Divorced	
replace marital = 5 if mstatus == 2 & ma6==3
	//5: Separated/not living together
replace marital = 5 if mstatus == 2 & ma6==.
	//5: Separated/not living together	
replace mstatus = . if mstatus==9	
label define lab_mar 1"never married" 2"currently married" 3"widowed" ///
4"divorced" 5"not living together"
label values marital lab_mar	
label var marital "Marital status of household member"
tab marital, miss
tab ma6 marital, miss
tab mstatus marital, miss
rename marital marital_wom	

	
keep wm7* cm1 cm8 cm9 cm10 ind_id women_WM *_wom
order wm7* cm1 cm8 cm9 cm10 ind_id women_WM *_wom
sort ind_id
save "$path_out/TGO17_WM.dta", replace


********************************************************************************
*** Step 1.4  MN - MEN'S RECODE 
***(Eligible man: 15-49 years in the household) 
********************************************************************************


use "$path_in/mn.dta", clear 

rename _all, lower

*** Generate individual unique key variable required for data merging using:
	*** hh1=cluster number; 
	*** hh2=household number; 
	*** mwm3=men's line number.  
gen double ind_id = hh1*10000 + hh2*100 + mwm3 
format ind_id %20.0g
label var ind_id "Individual ID"	
	 
	 
duplicates report ind_id 
	//2,456 men eligible for interview 


gen men_MN=1 	
	//Identification variable for observations in MR recode




	//Retain relevant variables:
gen religion_men = . 
lab var religion_men "Men's religion"	

gen ethnic_men = .
lab var ethnic_men "Men's ethnicity"

clonevar insurance_men = mwb18
label var insurance_men "Men have health insurance"	


codebook mmstatus mma6, tab (10)
tab mmstatus mma6, miss 
gen marital = 1 if mmstatus == 3 & mma6==.
	//1: Never married
replace marital = 2 if mmstatus == 1 & mma6==.
	//2: Currently married
replace marital = 3 if mmstatus == 2 & mma6==1
	//3: Widowed	
replace marital = 4 if mmstatus == 2 & mma6==2
	//4: Divorced	
replace marital = 5 if mmstatus == 2 & mma6==3
	//5: Separated/not living together	
replace marital = 5 if mmstatus == 2 & mma6==.
	//5: Separated/not living together		
label define lab_mar 1"never married" 2"currently married" 3"widowed" ///
4"divorced" 5"not living together"
label values marital lab_mar	
label var marital "Marital status of household member"
tab marital, miss
tab mma6 marital, miss
tab mmstatus marital, miss
rename marital marital_men

	
keep mcm1 mcm8 mcm9 mcm10 ind_id men_MN *_men 
order mcm1 mcm8 mcm9 mcm10 ind_id men_MN *_men 
sort ind_id
save "$path_out/TGO17_MN.dta", replace


********************************************************************************
*** Step 1.5 HH - HOUSEHOLD RECODE 
***(All households interviewed) 
********************************************************************************

use "$path_in/hh.dta", clear 
	
rename _all, lower	

*** Generate individual unique key variable required for data merging
	*** hh1=cluster number;  
	*** hh2=household number 
gen	double hh_id = hh1*100 + hh2 
format	hh_id %20.0g
lab var hh_id "Household ID"

duplicates report hh_id 
	//8,404 households sampled. Matches report (p.xi).

save "$path_out/TGO17_HH.dta", replace
	

********************************************************************************
*** Step 1.6 HL - HOUSEHOLD MEMBER  
********************************************************************************

use "$path_in/hl.dta", clear 

rename _all, lower

	
*** Generate a household unique key variable at the household level using: 
	***hh1=cluster number 
	***hh2=household number
gen double hh_id = hh1*100 + hh2 
format hh_id %20.0g
label var hh_id "Household ID"


*** Generate individual unique key variable required for data merging using:
	*** hh1=cluster number; 
	*** hh2=household number; 
	*** hl1=respondent's line number.
gen double ind_id = hh1*10000 + hh2*100 + hl1
format ind_id %20.0g
label var ind_id "Individual ID"

duplicates report ind_id 


sort ind_id

	
********************************************************************************
*** Step 1.7 DATA MERGING 
******************************************************************************** 
 
 
*** Merging BR Recode 
*****************************************
merge 1:1 ind_id using "$path_out/TGO17_BH.dta" 
drop _merge
erase "$path_out/TGO17_BH.dta" 
 
 
*** Merging WM Recode 
*****************************************
merge 1:1 ind_id using "$path_out/TGO17_WM.dta" 
drop _merge	
erase "$path_out/TGO17_WM.dta"


*** Merging HH Recode 
*****************************************
merge m:1 hh_id using "$path_out/TGO17_HH.dta" 
tab hh46 if _m==2 
	//488 households that were not interviewed 
drop  if _merge==2
	//Drop households that were not interviewed 
drop _merge
erase "$path_out/TGO17_HH.dta"


*** Merging MN Recode 
*****************************************
merge 1:1 ind_id using "$path_out/TGO17_MN.dta" 
drop _merge
erase "$path_out/TGO17_MN.dta"


*** Merging CH Recode 
*****************************************
merge 1:1 ind_id using "$path_out/TGO17_CH.dta" 
drop _merge
erase "$path_out/TGO17_CH.dta"


sort ind_id

********************************************************************************
*** Step 1.8 CONTROL VARIABLES
********************************************************************************
/*Households are identified as having 'no eligible' members if there are no 
applicable population, that is, children 0-5 years, adult women or men. These 
households will not have information on relevant indicators of health. As such, 
these households are considered as non-deprived in those relevant indicators. */


*** No eligible women 15-49 years 
*** for child mortality indicator
*****************************************
count if women_WM==1
count if hl8>0 & hl8!=.
	//Eligibility based on WM datafile (women_WM) and HL datafile (hl8) matches
gen	fem_eligible = (women_WM==1)
bys	hh_id: egen hh_n_fem_eligible = sum(fem_eligible) 	
	//Number of eligible women for interview in the hh
gen	no_fem_eligible = (hh_n_fem_eligible==0) 									
	//Takes value 1 if the household had no eligible females for an interview
lab var no_fem_eligible "Household has no eligible women"
drop hh_n_fem_eligible 
tab no_fem_eligible, miss 


*** No eligible men 15-49 years
*** for child mortality indicator (if relevant)
*****************************************
count if men_MN==1
count if hl9>0 & hl9!=.
	//Eligibility based on MN datafile (men_MN) and HL datafile (hl9) matches
gen	male_eligible = (men_MN==1)
bysort	hh_id: egen hh_n_male_eligible = sum(male_eligible)  
	//Number of eligible men for interview in the hh
gen	no_male_eligible = (hh_n_male_eligible==0) 	
	//Takes value 1 if the household had no eligible men for an interview
lab var no_male_eligible "Household has no eligible man for interview"
drop hh_n_male_eligible
tab no_male_eligible, miss 

	
*** No eligible children under 5
*** for child nutrition indicator
*****************************************
count if child_CH==1
count if hl10>0 & hl10!=.
	//Eligibility based on CH datafile (child_CH) and HL datafile (hl10) matches
gen	child_eligible = (child_CH==1) 
bysort	hh_id: egen hh_n_children_eligible = sum(child_eligible)  
	//Number of eligible children for anthropometrics
gen	no_child_eligible = (hh_n_children_eligible==0) 
	//Takes value 1 if there were no eligible children for anthropometrics
lab var no_child_eligible "Household has no children eligible for anthropometric"
drop hh_n_children_eligible
tab no_child_eligible, miss 

sort hh_id


********************************************************************************
*** Step 1.9 RENAMING DEMOGRAPHIC VARIABLES ***
********************************************************************************

//Sample weight
clonevar weight = hhweight 
label var weight "Sample weight"


//Area: urban or rural		
desc hh6	
codebook hh6, tab (5)	
clonevar area = hh6  
replace area=0 if area==2  
label define lab_area 1 "urban" 0 "rural"
label values area lab_area
label var area "Area: urban-rural"


//Relationship to the head of household
desc hl3
clonevar relationship = hl3 
codebook relationship, tab (20)
recode relationship (1=1)(2=2)(3 13=3)(4/12=4)(96=5)(14=6) 
label define lab_rel 1"head" 2"spouse" 3"child" 4"extended family" ///
5"not related" 6"maid"
label values relationship lab_rel
label var relationship "Relationship to the head of household"
tab hl3 relationship, miss	


//Sex of household member
codebook hl4
clonevar sex = hl4 
label define lab_sex 1"male" 2"female", replace
label values sex lab_sex
label var sex "Sex of household member"
tab sex


//Household headship
bys	hh_id: egen missing_hhead = min(relationship)
tab missing_hhead,m 
gen household_head=.
replace household_head=1 if relationship==1 & sex==1 
replace household_head=2 if relationship==1 & sex==2
bysort hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
label define head 1"male-headed" 2"female-headed"
label values headship head
label var headship "Household headship"
tab headship, miss


//Age of household member
codebook hl6, tab (999)
clonevar age = hl6  
replace age = . if age>=98
label var age "Age of household member"


//Age group (for global MPI estimation)
recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+") , gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"


//Total number of de jure hh members in the household
gen member = 1
bysort hh_id: egen hhsize = sum(member)
label var hhsize "Household size"
tab hhsize, miss
compare hhsize hh48 
drop member


//Religion of the household head
lookfor religion
clonevar religion_hh = hc1a
label var religion_hh "Religion of household head"
codebook religion_hh

//Ethnicity of the household head
lookfor ethnique
clonevar ethnic_hh = hc2
label var ethnic_hh "Ethnicity of household head"
codebook ethnic_hh


//Subnational region
	/* The sample for Togo MICS 2017 was designed to provide 
	estimates for a large number of indicators on the situation of children 
	and women at the national level, for urban and rural areas, and for the 
	7 economic regions of the country (p. 4). In the survey report, child 
	nutrition status (p.257) & mortality rates (p.79) are estimated for 
	the 7 regions. */   
	
codebook hh7, tab (99) 
decode hh7, gen(temp)
replace temp =  proper(temp)
encode temp, gen(region)
drop temp
codebook region, tab (99)
label define region_lab 1"Centrale" 2"Golfe Urbain" 3"Kara" ///
						4"Lomé Commune" ///
						5 "Maritime" 6 "Plateaux" 7 "Savanes"
label values region region_lab
lab var region "Region for subnational decomposition"

	
//Harmonised over time

recode hh7   (1=4 "Maritime")(2=5 "Plateaux") ///
             (3=1 "Centrale")(4=2 "Kara") ///
		     (5=6 "Savanes")(6 7=3 "Lomé"), gen(region_c)		  
lab var region_c "Region for subnational decomposition"
codebook region_c, tab (9)


********************************************************************************
***  Step 2 Data preparation  ***
***  Standardization of the 10 Global MPI indicators 
***  Identification of non-deprived & deprived individuals  
********************************************************************************

********************************************************************************
*** Step 2.1 Years of Schooling ***
********************************************************************************

	/*In Togo, children enter primary school at age 6, lower secondary at 
	age 12 and upper secondary school at age 16. There are 6 grades in primary 
	school and 4 + 3 grades in secondary school. (p.273 of country survey report). */


codebook ed5a, tab (99)
clonevar edulevel = ed5a 
	//Highest educational level attended
replace edulevel = . if ed5a==. | ed5a==7 | ed5a==8
	//All missing values or out of range are replaced as "."
replace edulevel = 0 if ed4==2 
	//Those who never attended school are replaced as '0'
label var edulevel "Highest level of education attended"


codebook ed5b, tab (99) 
	// 0-8; 98 "missing"; 99 "no response" 
clonevar eduhighyear = ed5b 
	//Highest grade attended at that level
replace eduhighyear = .  if ed5b==. | ed5b==98 | ed5b==99 
	//All missing values or out of range are replaced as "."
replace eduhighyear = 0  if ed4==2 
	//Those who never attended school are replaced as '0'
lab var eduhighyear "Highest grade attended for each level of edu"


*** Cleaning inconsistencies
replace edulevel = 0 if age<10  
replace eduhighyear = 0 if age<10 
	/*At this point, we disregard the years of education of household members 
	younger than 10 years by replacing the relevant variables with '0 years' 
	since they are too young to have completed 6 years of schooling. */  
replace eduhighyear = 0 if edulevel<1
	//Early childhood education has no grade


*** Now we create the years of schooling
tab eduhighyear edulevel, miss

gen	eduyears = eduhighyear
replace eduyears = eduhighyear + 6 if edulevel==2   
	/*There are 6 grades in primary school; followed by 4 grades in lower
	secondary school (LSS). As such we add 6 years to each of the grades 
	completed at the LSS level. */	
replace eduyears = eduhighyear + 10 if edulevel==3  
	/*There are 6 grades in primary school; followed by 4 grades in lower 
	secondary school (LSS); and 3 grades in Upper Secondary School (USS). 
	This means, individuals would have completed 10 years of schooling before 
	reaching USS. As such we add 10 years to each of the grades completed at the 
	USS level.*/	
replace eduyears = eduhighyear + 13 if edulevel==4
	/*Individuals would have completed 13 years of schooling before reaching 
	post secondary education, or university education or vocational training.
	As such we add 13 years to each of the grades completed at the higher 
	education and vocation level.*/
	
replace eduyears =  6 if edulevel==2 & eduhighyear==.
	/*We may assume that 9 individuals who have attended lower secondary
	but have missing grade have completed primary level. */
replace eduyears =  10 if edulevel==3 & eduhighyear==.
	/*We may assume that 3 individuals who have attended upper secondary
	but have missing grade have completed lower secondary level. */	
replace eduyears =  13 if edulevel==4 & eduhighyear==.
	/*We may assume that 4 individuals who have attended higher education 
	but have missing grade have completed upper secondary level. */
	
tab eduyears,miss


*** Checking for further inconsistencies 
replace eduyears = eduyears - 1 if ed6==2 & eduyears>=1 & eduyears<. 
	/*Through ed6 variable, individuals confirm whether they have completed the 
	highest grade they have attended. For individuals who responded that they 
	did not complete the highest grade attended, we re-assign them to the next  
	lower grade that they would have completed. */
replace eduyears = . if age<=eduyears & age>0 
	/*There are cases in which the years of schooling are greater than the 
	age of the individual. This is clearly a mistake in the data.*/
replace eduyears = 0 if age< 10
replace eduyears = 0 if (age==10 | age==11) & eduyears < 6
	/*The variable "eduyears" was replaced with a '0' for ineligible household 
	members, i.e.: those who have not completed 6 years of schooling following 
	their starting school age */													   
tab ed4 if edulevel==. & eduhighyear==.,miss	
replace eduyears = . if edulevel==. & eduhighyear==. & (ed4==1 | ed4==9)
	/*Replaced as missing value when level of education & grade is missing for 
	those who have attended school or lack information on whether they attended 
	school. */
lab var eduyears "Total number of years of education accomplished"
tab eduyears, miss


	/*A control variable is created on whether there is information on 
	years of education for at least 2/3 of the eligible household members*/	
gen temp = 1 if eduyears!=. & age>=12 & age!=.
replace temp = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp = 1 if age==11 & eduyears>=6 & eduyears<.
bysort	hh_id: egen no_missing_edu = sum(temp)	
	//Total eligible household members with no missing years of education
gen temp2 = 1 if age>=12 & age!=.
replace temp2 = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp2 = 1 if age==11 & eduyears>=6 & eduyears<.
bysort hh_id: egen hhs = sum(temp2)
	/*Total number of eligible household members who should have information 
	on years of education */
replace no_missing_edu = no_missing_edu/hhs
replace no_missing_edu = (no_missing_edu>=2/3)
	/*Identify whether there is information on years of education for at 
	least 2/3 of the eligible household members */
tab no_missing_edu, miss
	//The value for 0 (missing) is 0.33% 
label var no_missing_edu "No missing edu for at least 2/3 of the eligible HH members"
drop temp temp2 hhs


*** Standard MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed SIX years of schooling. */
******************************************************************* 
gen	 years_edu6 = (eduyears>=6)
replace years_edu6 = . if eduyears==.
bysort hh_id: egen hh_years_edu6_1 = max(years_edu6)
gen	hh_years_edu6 = (hh_years_edu6_1==1)
replace hh_years_edu6 = . if hh_years_edu6_1==.
replace hh_years_edu6 = . if hh_years_edu6==0 & no_missing_edu==0 
lab var hh_years_edu6 "Household has at least one member with 6 years of edu"
tab hh_years_edu6, miss 

	
*** Destitution MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed at least one year of schooling. */
*******************************************************************  
gen	years_edu1 = (eduyears>=1)
replace years_edu1 = . if eduyears==.
bysort	hh_id: egen hh_years_edu_u = max(years_edu1)
replace hh_years_edu_u = . if hh_years_edu_u==0 & no_missing_edu==0
lab var hh_years_edu_u "Household has at least one member with 1 year of edu"
tab hh_years_edu_u, miss 


********************************************************************************
*** Step 2.2 Child School Attendance ***
********************************************************************************
	
codebook ed4 ed9, tab (99)

gen	attendance = .
replace attendance = 1 if ed9==1 
	//Replace attendance with '1' if currently attending school	
replace attendance = 0 if ed9==2 
	//Replace attendance with '0' if currently not attending school	
replace attendance = 0 if ed4==2 
	//Replace attendance with '0' if never ever attended school	
tab age ed9, miss	
	//Check individuals who are not of school age	
replace attendance = 0 if age<5 | age>24 
	//Replace attendance with '0' for individuals who are not of school age	
	
label define lab_attend 1 "currently attending" 0 "not currently attending"
label values attendance lab_attend
label var attendance "Attended school during current school year"	
tab attendance, miss


*** Standard MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 8. */ 
******************************************************************* 
gen	child_schoolage = (schage>=6 & schage<=14)
	/*In Togo, the official school entrance age to primary school is 
	6 years. So, age range is 6-14 (=6+8) 
	Source 1: See page 273 of the country report for primary school entry age. 
	Source 2: "http://data.uis.unesco.org/?ReportId=163"
	Go to Education>Education>System>Official entrance age to primary education. 
	Look at the starting age and add 8. 
	*/
	
	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the school age children */
count if child_schoolage==1 & attendance==.
	//How many eligible school aged children with missing school attendance?: 0 children
gen temp = 1 if child_schoolage==1 & attendance!=.
	/*Generate a variable that captures the number of eligible school aged 
	children who are attending school */
bysort hh_id: egen no_missing_atten = sum(temp)	
	/*Total school age children with no missing information on school 
	attendance */
gen temp2 = 1 if child_schoolage==1	
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are of school age
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)
	/*Identify whether there is missing information on school attendance for 
	more than 2/3 of the school age children */			
tab no_missing_atten, miss
	//The value for 0 (missing) is 0.00% 
label var no_missing_atten "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs

	
bysort hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
	//It takes value 1 if the household has children in school age
lab var hh_children_schoolage "Household has children in school age"


gen	child_not_atten = (attendance==0) if child_schoolage==1
replace child_not_atten = . if attendance==. & child_schoolage==1
bysort	hh_id: egen any_child_not_atten = max(child_not_atten)
gen	hh_child_atten = (any_child_not_atten==0) 
replace hh_child_atten = . if any_child_not_atten==.
replace hh_child_atten = 1 if hh_children_schoolage==0
replace hh_child_atten = . if hh_child_atten==1 & no_missing_atten==0 
lab var hh_child_atten "Household has all school age children up to class 8 in school"
tab hh_child_atten, miss 


*** Destitution MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 6. */ 
******************************************************************* 
gen	child_schoolage_6 = (schage>=6 & schage<=12) 
	/*In Togo, the official school entrance age is 6 years.  
	  So, age range for destitution measure is 6-12 (=6+6) */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the children attending school up to 
	class 6 */	
count if child_schoolage_6==1 & attendance==.	
gen temp = 1 if child_schoolage_6==1 & attendance!=.
bysort hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_6==1	
bysort hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)			
tab no_missing_atten_u, miss
	/*Togo DR MICS 2017: 0.00% missing values */
label var no_missing_atten_u "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs		
		
bysort	hh_id: egen hh_children_schoolage_6 = sum(child_schoolage_6)
replace hh_children_schoolage_6 = (hh_children_schoolage_6>0) 
lab var hh_children_schoolage_6 "Household has children in school age (6 years of school)"

gen	child_atten_6 = (attendance==1) if child_schoolage_6==1
replace child_atten_6 = . if attendance==. & child_schoolage_6==1
bysort	hh_id: egen any_child_atten_6 = max(child_atten_6)
gen	hh_child_atten_u = (any_child_atten_6==1) 
replace hh_child_atten_u = . if any_child_atten_6==.
replace hh_child_atten_u = 1 if hh_children_schoolage_6==0
replace hh_child_atten_u = . if hh_child_atten_u==0 & no_missing_atten_u==0 
lab var hh_child_atten_u "Household has at least one school age children up to class 6 in school"
tab hh_child_atten_u, miss


********************************************************************************
*** Step 2.3 Nutrition ***
********************************************************************************
 
********************************************************************************
*** Step 2.3a Child Nutrition ***
********************************************************************************


*** Child Underweight Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(underweight)
gen	hh_no_underweight = (temp==0) 
	//Takes value 1 if no child in the hh is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_underweight "Household has no child underweight - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(underweight_u)
gen	hh_no_underweight_u = (temp==0) 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
lab var hh_no_underweight_u "Destitute: Household has no child underweight"
drop temp


*** Child Stunting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(stunting)
gen	hh_no_stunting = (temp==0) 
	//Takes value 1 if no child in the hh is stunted
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_stunting "Household has no child stunted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(stunting_u)
gen	hh_no_stunting_u = (temp==0) 
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
lab var hh_no_stunting_u "Destitute: Household has no child stunted"
drop temp


*** Child Wasting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(wasting)
gen	hh_no_wasting = (temp==0) 
	//Takes value 1 if no child in the hh is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_wasting "Household has no child wasted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(wasting_u)
gen	hh_no_wasting_u = (temp==0) 
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
lab var hh_no_wasting_u "Destitute: Household has no child wasted"
drop temp


*** Child Either Underweight or Stunted Indicator ***
************************************************************************

*** Standard MPI ***
gen hh_no_uw_st = 1 if hh_no_stunting==1 & hh_no_underweight==1
replace hh_no_uw_st = 0 if hh_no_stunting==0 | hh_no_underweight==0
	//Takes value 0 if child in the hh is stunted or underweight 
replace hh_no_uw_st = . if hh_no_stunting==. & hh_no_underweight==.
replace hh_no_uw_st = 1 if no_child_eligible==1
	//Households with no eligible children will receive a value of 1 
lab var hh_no_uw_st "Household has no child underweight or stunted"


*** Destitution MPI  ***
gen hh_no_uw_st_u = 1 if hh_no_stunting_u==1 & hh_no_underweight_u==1
replace hh_no_uw_st_u = 0 if hh_no_stunting_u==0 | hh_no_underweight_u==0
replace hh_no_uw_st_u = . if hh_no_stunting_u==. & hh_no_underweight_u==.
replace hh_no_uw_st_u = 1 if no_child_eligible==1 
lab var hh_no_uw_st_u "Destitute: Household has no child underweight or stunted"


*** Quality check ***
/* Compare the proportion of children under 5 who is stunted between the survey 
report and this dataset. */ 
*****************************************************************************
tab stunting [aw = weight] if child_CH==1
	/* Togo MICS 2017: The country survey report indicates that 23.8% of 
	children are stunted (p.257), while it is 23.86% in the data.*/

	
********************************************************************************
*** Step 2.3b Household Nutrition Indicator ***
********************************************************************************

*** Standard MPI ***
/* The indicator takes value 1 if the household has no child under 5 who 
has either height-for-age or weight-for-age that is under two standard 
deviation below the median. It also takes value 1 for the households that 
have no eligible children. The indicator takes a value of missing only if 
all eligible children have missing information in their respective 
nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st = 1
replace hh_nutrition_uw_st = 0 if hh_no_uw_st==0
replace hh_nutrition_uw_st = . if hh_no_uw_st==.
replace hh_nutrition_uw_st = 1 if no_child_eligible==1   		
lab var hh_nutrition_uw_st "Household has no individuals malnourished"
tab hh_nutrition_uw_st, miss


*** Destitution MPI ***
/* The indicator takes value 1 if the household has no child under 5 who 
has either height-for-age or weight-for-age that is under three standard 
deviation below the median. It also takes value 1 for the households that 
have no eligible children. The indicator takes a value of missing only if 
all eligible children have missing information in their respective 
nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st_u = 1
replace hh_nutrition_uw_st_u = 0 if hh_no_uw_st_u==0
replace hh_nutrition_uw_st_u = . if hh_no_uw_st_u==.
replace hh_nutrition_uw_st_u = 1 if no_child_eligible==1   		
lab var hh_nutrition_uw_st_u "Household has no individuals malnourished (destitution)"
tab hh_nutrition_uw_st_u, miss 


********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************

codebook cm9 cm10 mcm9 mcm10
	/*cm9 or mcm9: number of sons who have died 
	  cm10 or mcm10: number of daughters who have died */
	  
egen temp_f = rowtotal(cm9 cm10), missing
	//Total child mortality reported by eligible women
replace temp_f = 0 if cm1==1 & cm8==2 | cm1==2 
	/*Assign a value of "0" for:
	- all eligible women who have ever gave birth but reported no child death 
	- all eligible women who never ever gave birth */
replace temp_f = 0 if no_fem_eligible==1	
	/*Assign a value of "0" for:
	- individuals living in households that have non-eligible women */
bysort hh_id: egen child_mortality_f = sum(temp_f), missing
lab var child_mortality_f "Occurrence of child mortality reported by women"
tab child_mortality_f, miss
drop temp_f	


egen temp_m = rowtotal(mcm9 mcm10), missing
	//Total child mortality reported by eligible men	
replace temp_m = 0 if mcm1==1 & mcm8==2 | mcm1==2 
	/*Assign a value of "0" for:
	- all eligible men who ever fathered children but reported no child death 
	- all eligible men who never fathered children */
replace temp_m = 0 if no_male_eligible==1	
	/*Assign a value of "0" for:
	- individuals living in households that have non-eligible women */
bysort	hh_id: egen child_mortality_m = sum(temp_m), missing	
lab var child_mortality_m "Occurrence of child mortality reported by men"
tab child_mortality_m, miss
drop temp_m


egen child_mortality = rowmax(child_mortality_f child_mortality_m)
lab var child_mortality "Total child mortality within household"
tab child_mortality, miss

	
*** Standard MPI *** 
/* Members of the household are considered deprived if women in 
the household reported mortality among children under 18 in the 
last 5 years from the survey year.*/
************************************************************************

tab childu18_died_per_wom_5y, miss
		
replace childu18_died_per_wom_5y = 0 if cm1==2 															   
	/*Assign a value of "0" for:
	- all eligible women who never ever gave birth */
replace childu18_died_per_wom_5y = 0 if no_fem_eligible==1	
	/*Assign a value of "0" for:
	- individuals living in households that have non-eligible women */
	
bysort hh_id: egen childu18_mortality_5y = sum(childu18_died_per_wom_5y), missing
replace childu18_mortality_5y = 0 if childu18_mortality_5y==. & child_mortality==0
label var childu18_mortality_5y "Under 18 child mortality within household past 5 years reported by women"
tab childu18_mortality_5y, miss		
	
gen hh_mortality_u18_5y = (childu18_mortality_5y==0)
replace hh_mortality_u18_5y = . if childu18_mortality_5y==.
lab var hh_mortality_u18_5y "Household had no under 18 child mortality in the last 5 years"
tab hh_mortality_u18_5y, miss 


*** Destitution MPI *** 
*** (same as standard MPI) ***
************************************************************************
clonevar hh_mortality_u = hh_mortality_u18_5y	


*** Harmonised MPI *** 
	/*In the earlier survey, there is no birth history data. This means, 
	there is no information on the date of death of children who have died. 
	As such, we are not able to construct the indicator on child mortality 
	under 18 that occurred in the last 5 years for this survey. Instead, 
	we identify individuals as deprived if any children died in the household. 
	As such, for harmonisation purpose, we construct the same indicator in 
	this survey.*/
************************************************************************	
gen	hh_mortality_c = (child_mortality==0)
replace hh_mortality_c = . if child_mortality==.
replace hh_mortality_c = 1 if no_fem_eligible==1 
lab var hh_mortality_c "COT: HH had no child mortality"
tab hh_mortality_c, miss


clonevar hh_mortality_u_c = hh_mortality_c
lab var hh_mortality_u_c "COT-DST: HH had no child mortality"

			
********************************************************************************
*** Step 2.5 Electricity ***
********************************************************************************


*** Standard MPI ***
/*Members of the household are considered 
deprived if the household has no electricity */
****************************************
clonevar electricity = hc8 
codebook electricity, tab (9)
replace electricity = 1 if electricity==2
replace electricity = 0 if electricity==3
replace electricity = . if electricity==9 
label define lab_elec 1 "have electricity" 0 "no electricity"
label values electricity lab_elec		
label var electricity "Household has electricity"
tab electricity, miss


*** Destitution MPI  ***
*** (same as standard MPI) ***
****************************************
gen electricity_u = electricity
label var electricity_u "Household has electricity"


*** Quality check ***
/* We compare the proportion of households with 
electricity obtained from our work and as reported in 
the country survey report. */
*********************************************************
bysort hh_id: gen id = _n
tab electricity [aw = weight] if id==1,miss
	/*In the report, Table SR.2.1 (p.27) indicates that 54.9% of household 
	have access to electricity. The results obtained from our work is 52.91% 
	which matches the report.*/

	
********************************************************************************
*** Step 2.6 Sanitation ***
********************************************************************************

/*
Improved sanitation facilities include flush or pour flush toilets to sewer 
systems, septic tanks or pit latrines, ventilated improved pit latrines, pit 
latrines with a slab, and composting toilets. These facilities are only 
considered improved if it is private, that is, it is not shared with other 
households.
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-02-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.

*/

   
clonevar toilet = ws11  	
clonevar shared_toilet = ws15 
codebook shared_toilet, tab(99)  
recode shared_toilet (2=0)
replace shared_toilet=. if shared_toilet==9

			
*** Standard MPI ***
/*Members of the household are considered deprived if the household's 
sanitation facility is not improved (according to the SDG guideline) 
or it is improved but shared with other households*/
********************************************************************
codebook toilet, tab(99) 

gen	toilet_mdg     =     (toilet<=13 | toilet==18 | toilet==21 | ///
						  toilet==22 | toilet==31) & shared_toilet!=1 				
replace toilet_mdg = . if toilet==99 | toilet==.
	
replace toilet_mdg = 0 if shared_toilet==1
	/*18 individuals who did not response on the type of toilet, but 
	they indicated that they share their toilet facilities. We replaced these 
	individuals as deprived following the information on shared toilet.*/	
lab var toilet_mdg "Household has improved sanitation"
tab toilet toilet_mdg, miss
	

	
*** Destitution MPI ***
/*Members of the household are considered deprived if household practises 
open defecation or uses other unidentifiable sanitation practises */
********************************************************************
gen	toilet_u = .
replace toilet_u = 0 if toilet==95 | toilet==96 
replace toilet_u = 1 if toilet!=95 & toilet!=96 & toilet!=. & toilet!=99
lab var toilet_u "Household does not practise open defecation or others"
tab toilet toilet_u, miss


*** Quality check ***
/* We compare the proportion of household members with 
improved sanitation obtained from our work and as 
reported in the country survey report. */
*********************************************************
tab toilet_mdg [aw = weight],miss
	/*In the report, Table WS.3.2 (p.354) indicates that 19.1% of 
	household members have improved sanitation facilities that are not shared. 
	The result obtained from our work is 19.12% which closely matches the 
	country report. */

********************************************************************************
*** Step 2.7 Drinking Water  ***
********************************************************************************

/*
Improved drinking water sources include the following: piped water into 
dwelling, yard or plot; public taps or standpipes; boreholes or tubewells; 
protected dug wells; protected springs; packaged water; delivered water and 
rainwater which is located on premises or is less than a 30-minute walk from 
home roundtrip. 
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-01-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

clonevar water = ws1  
clonevar timetowater = ws4  
clonevar ndwater = ws2  	
 	

*** Standard MPI ***
/* Members of the household are considered deprived if the household 
does not have access to improved drinking water (according to the SDG 
guideline) or safe drinking water is at least a 30-minute walk from 
home, roundtrip */
********************************************************************
codebook water, tab(99)
						
gen	water_mdg     = 1 if water<=31 | water==41 | water==51 | ///
					     water==61 | water==71 | water==91 | water==92 	
replace water_mdg = 0 if water==32 | water==42 | water==81 | water==96
	
replace water_mdg = . if water==99 | water==.							 
lab var water_mdg "Household has safe drinking water on premises"
tab water water_mdg, miss	

	
*** Quality check ***
/* We compare the proportion of household members with 
improved access to safe drinking water as obtained from 
our work and as reported in the country survey report. */
*********************************************************
tab water_mdg [aw = weight],miss
	/*In the report, Table WS.1.1 (p.338), 74.6% of household members 
	have improved or safe drinking facilities. The results obtained from our 
	work is 74.63% which matches the report. */	 	
	

*** Time to water ***	
********************************************************* 
codebook timetowater, tab(999)
	
replace water_mdg = 0 if water_mdg==1 & timetowater >= 30 & timetowater!=. & ///
						 timetowater!=998 & timetowater!=999
	/*Deprived if water is at more than 30 minutes' walk (roundtrip). Please 
	exclude missing observations. */
	  	
tab timetowater if water==99 | water==., miss	
replace water_mdg = 0 if (water==99 | water==.) & water_mdg==. & ///
						  timetowater >= 30 & timetowater!=. & ///
						  timetowater!=998 & timetowater!=999 
	/*Deprived if water source is 30 minutes or more from home, roundtrip
	and the source of water is missing */ 
tab water_mdg, miss


*** Destitution MPI ***
/* Members of the household is identified as destitute if household 
does not have access to safe drinking water, or safe water is more 
than 45 minute walk from home, round trip.*/
********************************************************************
gen	water_u = .
replace	water_u = 1 if water<=31 | water==41 | water==51 | ///
					   water==91 | water==92 | water==61 | water==62 | ///
					   water==71 
					   
replace water_u = 0 if water==32 | water==42 | water==81 | water==96	
					   
replace water_u = 0 if water_u==1 & timetowater> 45 & timetowater!=. ///
					   & timetowater!=998 & timetowater!=999 	
					   
replace water_u = . if water==99 | water==.	

replace water_u = 0 if (water==99 | water==.) & water_u==. & ///
						timetowater > 45 & timetowater!=. & ///
						timetowater!=998 & timetowater!=999 
					
lab var water_u "Household has safe drinking water (considering distance)"
tab water water_u, miss


********************************************************************************
*** Step 2.8 Housing ***
********************************************************************************

/* Members of the household are considered deprived if the household 
has a dirt, sand, dung or other floor */

clonevar floor = hc4
codebook floor, tab(99)
gen	floor_imp = 1
replace floor_imp = 0 if floor<=12 | floor==96 	
replace floor_imp = . if floor==. | floor==99
lab var floor_imp "Household has floor that it is not earth/sand/dung"
tab floor floor_imp, miss	


/* Members of the household are considered deprived if the household has walls 
made of natural or rudimentary materials. We followed the report's definitions
of natural or rudimentary materials. */
clonevar wall = hc6
codebook wall, tab(99)			 
gen	wall_imp = 1 
replace wall_imp = 0 if wall<=27 | wall==96 	
replace wall_imp = . if wall==. | wall==99	
lab var wall_imp "Household has wall that it is not of low quality materials"
tab wall wall_imp, miss	

		
/* Members of the household are considered deprived if the household has roof 
made of natural or rudimentary materials. We followed the report's definitions
of natural and rudimentary materials. */
clonevar roof = hc5
codebook roof, tab(99)
gen	roof_imp = 1 
replace roof_imp = 0 if roof<=24 | roof==96
replace roof_imp = . if roof==.  | roof==99	
lab var roof_imp "Household has roof that it is not of low quality materials"
tab roof roof_imp, miss


*** Standard MPI ***
/* Members of the household is deprived in housing if the roof, 
floor OR walls are constructed from low quality materials.*/
**************************************************************
gen housing_1 = 1
replace housing_1 = 0 if floor_imp==0 | wall_imp==0 | roof_imp==0
replace housing_1 = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_1 "Household has roof, floor & walls that it is not low quality material"
tab housing_1, miss


*** Destitution MPI ***
/* Members of the household is deprived in housing if two out 
of three components (roof and walls; OR floor and walls; OR 
roof and floor) the are constructed from low quality materials. */
**************************************************************
gen housing_u = 1
replace housing_u = 0 if (floor_imp==0 & wall_imp==0 & roof_imp==1) | ///
						 (floor_imp==0 & wall_imp==1 & roof_imp==0) | ///
						 (floor_imp==1 & wall_imp==0 & roof_imp==0) | ///
						 (floor_imp==0 & wall_imp==0 & roof_imp==0)
replace housing_u = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_u "Household has one of three aspects(either roof,floor/walls) that is not low quality material"
tab housing_u, miss


*** Quality check ***
/* We compare the proportion of households with 
improved floor, walls and roof from our work and as 
reported in the country survey report. */
*********************************************************
tab floor_imp [aw = weight] if id==1,miss
	/*In the report, Table SR.2.1 (p.27) indicate that 88.3% of 
	households have improved floor (built using rudimentary or finished material). 
	The results obtained from our work is 88.37% which matches the report.*/	

tab wall_imp [aw = weight] if id==1,miss
	/*In the report, Table SR.2.1 (p.27) indicate that 67.1% of 
	households have improved floor (built using finished material). 
	The results obtained from our work is 67.10% which matches the report.*/	 	
	
tab roof_imp [aw = weight] if id==1,miss
	/*In the report, Table SR.2.1 (p.27) indicate that 89.9% of 
	households have improved floor (built using finished material). 
	The results obtained from our work is 89.92% which matches the report.*/	

	
********************************************************************************
*** Step 2.9 Cooking Fuel ***
********************************************************************************

/*
Solid fuel are solid materials burned as fuels, which includes coal as well as 
solid biomass fuels (wood, animal dung, crop wastes and charcoal). 

Source: 
https://apps.who.int/iris/bitstream/handle/10665/141496/9789241548885_eng.pdf
*/


clonevar cookingfuel = eu4 

	
*** Standard MPI ***
/* Members of the household are considered deprived if the 
household uses solid fuels and solid biomass fuels for cooking. */
*****************************************************************
codebook eu1 cookingfuel, tab(99)

tab eu1 cookingfuel, miss
gen	cooking_mdg = 1
replace cooking_mdg = 0 if cookingfuel>=4 & cookingfuel<=9						
replace cooking_mdg = 0 if eu1==96	& cookingfuel==96												 
	/*Togo MICS 2017: 26 individuals reported using other type of cookstove, 
	25 of them have used solid fuel on the cookstove, while 1 of them reported 
	using other type of fuel. It is very likely that this individual also 
	used some form of fuel that is not clean.*/
replace cooking_mdg = 0 if eu1==8 & cookingfuel==96	
	/*Togo MICS 2017: 4 individuals reported using other type of cooking fuel
	on a cookstove that is designed for solid fuel. It is very likely that 
	these individuals used some form of fuel that is not clean. */		
replace cooking_mdg = . if cookingfuel==. & eu1==.	
lab var cooking_mdg "Household cooks with clean fuels"	
tab cookingfuel cooking_mdg, miss
tab eu1 cooking_mdg, miss
tab cooking_mdg, miss


*** Destitution MPI ***
*** (same as standard MPI) ***
****************************************	
gen	cooking_u = cooking_mdg
lab var cooking_u "Household cooks with clean fuels"


*** Quality check ***
/* We compare the proportion of households using 
clean fuels & technologies for cooking obtained from 
our work and as reported in the country survey report. */
*********************************************************
tab cooking_mdg [aw = weight],miss
	/*In the report, Table TC.4.1 (p. 199) indicate that 7.0% of household 
	members use clean fuels & technologies for cooking. The result obtained 
	from our work is 7.85%.  */	

	
********************************************************************************
*** Step 2.10 Assets ownership ***
********************************************************************************

*** Television/LCD TV/plasma TV/color TV/black & white tv
lookfor tv television plasma lcd tÃ©lÃ©
codebook hc9a
clonevar television = hc9a 
lab var television "Household has television"
tab television, miss
	//49.17% missing value 
tab electricity television, miss
	/*This is because these households do not have electricity.*/
replace television=0 if electricity==0 & television==.	
	/*We make an assumption that there is no television in these households 
	given that there is no electricity. */

		
***	Radio/walkman/stereo/kindle
lookfor radio walkman stereo 
codebook hc7b
clonevar radio = hc7b 
lab var radio "Household has radio"	



***	Handphone/telephone/iphone/mobilephone/ipod
lookfor telephone mobilephone ipod portable
codebook hc7a hc12
	//No one owns landline
clonevar telephone =  hc7a
replace telephone=1 if telephone!=1 & hc12==1	
	//hc12=mobilephone. Combine information on telephone and mobilephone.	
tab hc7a hc12 if telephone==1,miss
lab var telephone "Household has telephone (landline/mobilephone)"	

	
***	Refrigerator/icebox/fridge
lookfor refrigerator 
codebook hc9b
clonevar refrigerator = hc9b 
lab var refrigerator "Household has refrigerator"
tab refrigerator, miss
	//49.17% missing value 
tab electricity refrigerator, miss
	//This is because these households do not have electricity.
replace refrigerator=0 if electricity==0 & refrigerator==.	
	/*We make an assumption that there is no refrigerator in these households 
	given that there is no electricity.*/


***	Car/van/lorry/truck
lookfor car voiture truck van
codebook hc10e
clonevar car = hc10e  
lab var car "Household has car"		

	
***	Bicycle/cycle rickshaw
lookfor bicycle bicyclette
codebook hc10b
clonevar bicycle = hc10b 
lab var bicycle "Household has bicycle"	
	
	
***	Motorbike/motorized bike/autorickshaw
lookfor motorbike moto
codebook hc10c	
clonevar motorbike = hc10c
lab var motorbike "Household has motorbike"

	
***	Computer/laptop/tablet
lookfor computer ordinateur laptop ipad tablet
codebook hc11
clonevar computer = hc11
lab var computer "Household has computer"


***	Animal cart
lookfor brouette charrette cart
codebook hc10d
gen animal_cart = hc10d
lab var animal_cart "Household has animal cart"	
 
 
foreach var in television radio telephone refrigerator car ///
			   bicycle motorbike computer animal_cart {
replace `var' = 0 if `var'==2 
label define lab_`var' 0"No" 1"Yes"
label values `var' lab_`var'			   
replace `var' = . if `var'==9 | `var'==99 | `var'==8 | `var'==98 
}
	//Labels defined and missing values replaced	
	

*** Quality check ***
/* We compare the proportion of households owning  
each assets obtained from our work and as reported 
in the country survey report. */
*********************************************************
	
tab radio [aw = weight] if id==1,miss
	/*Table SR.9.2 (p.57) indicates that 41.6% of households own radio. 
	The results obtained from our work is 41.58% which matches the report.*/	

tab television [aw = weight] if id==1,miss
	/*Table SR.2.2(p.29) indicates that 39.1% of households own 
	television. The results obtained from our work is 39.11% which matches the 
	report.*/	
	
tab telephone [aw = weight] if id==1,miss
	/*Table SR.9.2 (p.57) indicate that 84.6% of households own either 
	fixed line or mobile phone. The results obtained from our work is 82.46%. */

tab computer [aw = weight] if id==1,miss
	/*Table SR.9.2 (p.57) indicates that 9.5% of households own computer. 
	The results obtained from our work is 9.52% which matches the report.*/
	
tab refrigerator [aw = weight] if id==1,miss
	/*Table SR.2.2 (p.29) indicate that 8.7% of households own 
	refrigerator. The results obtained from our work is 8.68% which matches 
	the report.*/

tab bicycle [aw = weight] if id==1,miss
	/*Table SR.2.2 (p.29) indicate that 27.5% of households own bicycle. 
	The results obtained from our work is 27.54% which matches the report. */

tab animal_cart [aw = weight] if id==1,miss
	/*Table SR.2.2 (p.29) indicate that 1.9% of households own animal 
	cart. The results obtained from our work is 1.87% which matches the report.*/

tab motorbike [aw = weight]if id==1,miss
	/*Table SR.2.2 (p.29) indicate that 35.2% of households own bicycle. 
	The results obtained from our work is 35.16% which matches the report.*/

tab car [aw = weight] if id==1,miss
	/*Table SR.2.2 (p.29) indicate that 5.3% of households own car. 
	The results obtained from our work is 5.33% which matches the report.*/

	
*** Standard MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own more than one of: radio, TV, telephone, bike, motorbike, 
refrigerator, computer or animal cart and does not own a car or truck.*/
*****************************************************************************
egen n_small_assets2 = rowtotal(television radio telephone refrigerator bicycle motorbike computer animal_cart), missing
lab var n_small_assets2 "Household Number of Small Assets Owned" 
   
gen hh_assets2 = (car==1 | n_small_assets2 > 1) 
replace hh_assets2 = . if car==. & n_small_assets2==.
lab var hh_assets2 "Household Asset Ownership: HH has car or more than 1 small assets incl computer & animal cart"
tab hh_assets2, miss


*** Destitution MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own any assets.*/
*****************************************************************************	
gen	hh_assets2_u = (car==1 | n_small_assets2>0)
replace hh_assets2_u = . if car==. & n_small_assets2==.
lab var hh_assets2_u "Household Asset Ownership: HH has car or at least 1 small assets incl computer & animal cart"


*** Harmonised MPI *** 
	/*The earlier survey lacked information on computer. 
	As such, the harmonised work excludes computer.*/
************************************************************************
gen computer_c = .
lab var computer_c "COT: Household has computer"

egen n_small_assets2_c = rowtotal(television radio telephone refrigerator bicycle motorbike computer_c animal_cart), missing
lab var n_small_assets2_c "COT: HH Number of Small Assets Owned" 
    
gen hh_assets2_c = (car==1 | n_small_assets2_c > 1) 
replace hh_assets2_c = . if car==. & n_small_assets2_c==.
lab var hh_assets2_c "COT: HH has car or more than 1 small assets"

gen	hh_assets2_u_c = (car==1 | n_small_assets2_c>0)
replace hh_assets2_u_c = . if car==. & n_small_assets2_c==.
lab var hh_assets2_u_c "COT-DST: HH has car or at least 1 small assets"	
	

********************************************************************************
*** Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************
	

	//Retain data on sampling design: 
*gen psu = hh1
clonevar strata = stratum
label var psu "Primary sampling unit"
label var strata "Sample strata"


	//Retain year, month & date of interview:
desc hh5y hh5m hh5d 
clonevar year_interview = hh5y 	
clonevar month_interview = hh5m 
clonevar date_interview = hh5d 


	//Generate presence of subsample
gen subsample = .
 
	
*** Keep main variables require for MPI calculation ***
keep hh_id ind_id subsample strata psu weight weight_ch sex age hhsize ///
area agec7 agec4 agec2 region headship marital_wom marital_men relationship ///
no_fem_eligible no_male_eligible child_eligible no_child_eligible ///
religion_wom religion_men religion_hh ethnic_wom ethnic_men ethnic_hh ///
insurance_wom insurance_men year_interview month_interview date_interview /// 
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet toilet_mdg shared_toilet toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
hh_mortality_c hh_mortality_u_c ///
hh_assets2_c hh_assets2_u_c	region_c

	 
*** Order file	***
order hh_id ind_id subsample strata psu weight weight_ch sex age hhsize ///
area agec7 agec4 agec2 region headship marital_wom marital_men relationship ///
no_fem_eligible no_male_eligible child_eligible no_child_eligible ///
religion_wom religion_men religion_hh ethnic_wom ethnic_men ethnic_hh ///
insurance_wom insurance_men year_interview month_interview date_interview /// 
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet toilet_mdg shared_toilet toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
hh_mortality_c hh_mortality_u_c ///
hh_assets2_c hh_assets2_u_c	region_c


*** Rename key global MPI indicators for estimation ***
recode hh_mortality_u18_5y  (0=1)(1=0) , gen(d_cm)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ)
recode electricity 			(0=1)(1=0) , gen(d_elct)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani)
recode housing_1 			(0=1)(1=0) , gen(d_hsg)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl)
recode hh_assets2 			(0=1)(1=0) , gen(d_asst)
 

*** Rename key global MPI indicators for destitution estimation ***
recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ)
recode electricity_u		(0=1)(1=0) , gen(dst_elct)
recode water_u 				(0=1)(1=0) , gen(dst_wtr)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl)
recode hh_assets2_u 		(0=1)(1=0) , gen(dst_asst) 


*** Rename indicators for changes over time estimation ***	
recode hh_mortality_c       (0=1)(1=0) , gen(d_cm_01)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr_01)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt_01)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ_01)
recode electricity 			(0=1)(1=0) , gen(d_elct_01)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr_01)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani_01)
recode housing_1 			(0=1)(1=0) , gen(d_hsg_01)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl_01)
recode hh_assets2_c 		(0=1)(1=0) , gen(d_asst_01)	
	

recode hh_mortality_u_c     (0=1)(1=0) , gen(dst_cm_01)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr_01)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt_01)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ_01)
recode electricity_u		(0=1)(1=0) , gen(dst_elct_01)
recode water_u	 			(0=1)(1=0) , gen(dst_wtr_01)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani_01)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg_01)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl_01)
recode hh_assets2_u_c 		(0=1)(1=0) , gen(dst_asst_01)


	/*In this survey, the region is 
	harmonised over time.*/	
clonevar region_01 = region_c 
 
 
*** Generate coutry and survey details for estimation ***
char _dta[cty] "Togo"
char _dta[ccty] "TGO"
char _dta[year] "2017" 	
char _dta[survey] "MICS"
char _dta[ccnum] "768"
char _dta[type] "micro"

	
	
*** Sort, compress and save data for estimation ***
sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/tgo_mics17.dta", replace 
	
	
gen educ_elig = 1 
replace educ_elig = 0 if age < 10 
replace educ_elig = 0 if (age==10 | age==11) & eduyears < 6 
label define lab_educ_elig 0"ineligible" 1"eligible"  
label values educ_elig lab_educ_elig
lab var educ_elig "Individual is eligible for educ indicator"
tab eduyears educ_elig,m	
	