
** Global Multidimensional Poverty Index - Philippines DHS 2022 **

clear all 
set more off
set maxvar 10000


global path_in "../rdta/Philippines DHS 2022"    		// folder path
global path_out "cdta"
global path_ado "ado"



**# step 1: data cleaning


**## birth history

use "$path_in/PHBR81FL.dta", clear

gen double ind_id = v001*1000000 + v002*100 + v003 
format ind_id %20.0g


gen bh_id = 1
lab var bh_id "female birth recode" 


desc b3 b5 b7 v008		// dob cmc; child is alive; age at dth mth; doi cmc

codebook b5 			// qc: 0=not alive; 1=alive

sum b7  				// qc: age at dth for all not alive
	
	
recode b5 (1=0 "alive") (0=1 "not alive"), gen (chdth)	// total child deceased 
		

bys ind_id: egen fchdth = sum(chdth)  	
egen fchdth_2 = rsum(v206 v207) 		// v206-7: sons/daughters not alive
compare fchdth fchdth_2   				// qc: figures are identical


recode chdth (1=1 "childu18 death") (0=0 "older/no death"), gen (ch18dth)	
replace ch18dth=0 if b7>=216 & b7!=.
ta ch18dth,m  							// total child under 18 deceased 



gen ydth_svy = (v008 - (b3 + b7)) / 12					// dod = b3 + b7
lab var ydth_svy "years deceased from svy year"			// doi - dod
su ydth_svy


bys ind_id: egen fch18dth=sum(ch18dth) if ydth_svy<=5

replace fch18dth=0 if fch18dth==. & fchdth>=0 & fchdth!=.
	/* note: all children who are alive or who died longer 
	than 5 years from the interview date are replaced as '0' */
	
replace fch18dth=. if ch18dth==1 & ydth_svy==.  // missing data on death year

ta fchdth fch18dth, m


bys ind_id: egen bh_cm = max(fch18dth)
lab var bh_cm "total cm reported by female"	
ta bh_cm,m


bys ind_id: gen id=1 if _n==1
keep if id==1								// keep one observation per female
drop id
duplicates report ind_id 					


	
keep *_id bh_cm		
order *_id bh_cm
sort ind_id
save "$path_out/PHL22_BH.dta", replace	



**## eligible female

use "$path_in/PHIR81FL.dta", clear


gen double ind_id = v001*1000000 + v002*100 + v003 
format ind_id %20.0g
duplicates report ind_id


gen fm_id = 1 
lab var fm_id "female recode file"


ta v012, m 						 		 // qc: age 15-49 years

codebook v201 v206 v207,ta (999)  		// qc: check for missing values


keep *_id v201 v206 v207 
order *_id v201 v206 v207 
sort ind_id
save "$path_out/PHL22_FM.dta", replace

	
		
**## hh members

use "$path_in/PHPR81FL.dta", clear 


gen double hh_id = hv001*10000 + hv002 
format hh_id %20.0g
lab var hh_id "household id"


gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g
lab var ind_id "individual id"

duplicates report ind_id 


sort hh_id ind_id


bys hh_id: gen id = _n
count if id==1						// qc: 30,372 hh - matches report, p.5.



**## data merging
 
merge 1:1 ind_id using "$path_out/PHL22_BH.dta"  			 // bh recode
drop _merge
erase "$path_out/PHL22_BH.dta"


merge 1:1 ind_id using "$path_out/PHL22_FM.dta" 			 // fem recode
drop _merge
erase "$path_out/PHL22_FM.dta"

	
sort ind_id



**## usual resident

codebook hv102, ta (9) 

drop if hv102!=1   				// keep only usual residents

	

**## eligibility variables

													// elg for cm
codebook hv117 										

bys	hh_id: egen no_f_elg = min(hv117==0) 
lab var no_f_elg "hh has no elg female"



**## relevant variables 

desc hv005 hv021 hv022

gen weight = hv005/1000000 							 // sample weight 
lab var weight "sample weight"


gen psu = hv021										// sample design
lab var psu "primary sampling unit"


gen strata = hv022
lab var strata "sample strata"

svyset psu [pw=weight] , strata(strata)	 singleunit(centered)




desc hv007 hv006 hv008 						// interview dates

clonevar intvw_y = hv007 	
clonevar intvw_m = hv006
 
clonevar intvw_dt = hv008 



codebook hv025								// urban-rural	

recode hv025 (1=1 "urban") (2=0 "rural"), gen (area)			
lab var area "area"
ta hv025 area, m

	
	
codebook hv101, ta (99)								// relationship

recode hv101 (1=1 "head")(2=2 "spouse")(3 11=3 "child") ///
(4/10=4 "extended family")(12/15=5 "not related")(98=.), ///
gen (rship)
lab var rship "relationship"
ta hv101 rship, m	



codebook hv104 									// sex

recode hv104  (1=1 "male") (2=2 "female"), gen (sex)			
lab var sex "sex"




bys	hh_id: egen xhhead = min(rship) 			// headship
ta xhhead,m

gen hhead = (rship==1 & sex==1) | (xhhead==2 & rship==2 & sex==1)  
replace hhead = 2 if (rship==1 & sex==2) | (xhhead==2 & rship==2 & sex==2)

bys hh_id: egen hship = sum(hhead)

replace hship = . if xhhead > 2 | hship==0  	// hh, no head
lab def la_hs 1"male-headed" 2"female-headed"
lab val hship la_hs
lab var hship "gender of hh head"
ta hship, m



codebook hv105, ta (999)					// age; age group

gen age = hv105 if hv105<98 
lab var age "age"


recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+"), gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"


 
codebook hv115, ta (9) 								// marital status

recode hv115 (0=1 "never married") ///
(1=2 "currently married") (3=3 "widowed") ///
(4=4 "divorced"), gen (marital)	
lab var marital "marital status"
ta hv115 marital, m


gen member = 1 										// hh size
bys hh_id: egen hhsize = sum(member)
lab var hhsize "household size"
ta hhsize, m


		
codebook hv024, ta (99)							// region

recode hv024 (15=1 "ARMM") (5=2 "Bicol") ///
(2=3 "Cagayan Valley") (4=4 "Calabarzon") (16=5 "Caraga") ///
(3=6 "Central Luzon") (7=7 "Central Visayas") ///
(14=8 "Cordillera Admin") (11=9 "Davao") ///
(8=10 "Eastern Visayas") (1=11 "Ilocos") (17=12 "Mimaropa") ///
(13=13 "National Capital") (10=14 "Northern Mindanao") ///
(12=15 "Soccsksargen") (6=16 "Western Visayas") ///
(9=17 "Zamboanga Peninsula"), gen(region)

lab var region "subnational region"
codebook region, ta(99)



						// subnational region, harmonised over time

clonevar region_01 = region
lab var region_01 "hot: subnational region"



**# step 2: indicator construction


**## school attendance

codebook hv121, ta (99)
recode hv121 (2=1 "in sch") (0=0 "out of sch") (9=.), gen (att)
lab var att "sch att status"	
ta att, m 										// qc: missing replaced as '.'

	
gen	scheg_a = (age>=6 & age<=14)   				// qc: mpi - age range of elg
lab var scheg_a "mpi: elg for school att"


gen	scheg_u = (age>=6 & age<=12)    			// qc: dst - age range of elg
lab var scheg_u "dst: elg for school att"	


								// policy: >=2/3 elg pop have non-missing data 
								
foreach v in a u {										// a=mpi; u=dst	 
	 	
	gen tmp_`v' = 1 if scheg_`v'==1 & att!=. 		
	bys hh_id: egen hsattdt_`v' = sum(tmp_`v')	 	// elg & no missing data
	drop tmp_`v'
	
	bys hh_id: egen hscheg_`v' = sum(scheg_`v')  		// elg population

	replace hsattdt_`v' = hsattdt_`v'/hscheg_`v' 
	replace hsattdt_`v' = (hsattdt_`v'>=2/3)	 		
	ta hsattdt_`v', m							     	// qc: missing < 0.5% 	
}



foreach i in a u {							// school att indc (a=mpi; u=dst)

	gen tmp_`i' = (att==1) if scheg_`i'==1 & att!=.     		
	bys	hh_id: egen satt_`i' = min(tmp_`i') 
	replace satt_`i' = . if hsattdt_`i'==0	& satt_`i'==1			// policy
	replace satt_`i' = 1 if hscheg_`i'==0  						   // non-elg 
	drop tmp_`i'

	ta satt_`i', m
							
}

lab var satt_a "mpi: not deprived in satt"					
lab var satt_u "dst: not deprived in satt"



**## years of schooling 

codebook hv108 hv106, ta(99)

ta hv108 hv106,m
ta ha67 ha66 if hv108==98,m
ta hv109 if hv108==98,m


clonevar eduyr = hv108 if hv108 < 98      		// qc: check missing 
lab var eduyr "total years of schooling"     
  
  
											// qc: replace missing correctly  
replace eduyr = 6  if hv108==98 & hv106==2        
replace eduyr = 12 if hv108==98 & hv106==3 
replace eduyr = 0 if hv108==98 & hv109==1 	// incomplete primary

replace eduyr = . if age<=eduyr & age>0 	 // missing if eduyr >= age
	
ta eduyr, m



	/* note: all those who should have completed 6 years 
	of schooling from their starting sch age are counted 
	as eligible, as well as early starters (from age of 10)
	who uniqely have up to six years of schooling are eligible. */
	
											// formal sch age is from 6 years
gen edueg = (age>=12 & age!=.) | ((age==10 | age==11) & ///
			(eduyr>=6 & eduyr!=.)) | (age==. & eduyr>=6 & eduyr!=.)	
							 
lab var edueg "elg for years of schooling"
ta edueg,m


								// policy: >=2/3 elg pop have non-missing data 	
gen tmp = 1 if eduyr!=. & edueg==1
bys	hh_id: egen hedudt = sum(tmp)				// elg pop with no missing data
drop tmp				
	
bys hh_id: egen hedueg = sum(edueg==1) 			// elg population

replace hedudt = hedudt/hedueg
replace hedudt = (hedudt>=2/3)		
ta hedudt, m									// qc: missing < 0.5% 	
lab var hedudt "non-missing edu" 
							
									
									// mpi: completed six years of schooling
gen	tmp_a = (eduyr>=6 & eduyr!=. & edueg==1)   	
replace tmp_a = . if eduyr==. & edueg==1

									// dst: completed one year of schooling
gen	tmp_u = (eduyr>=1 & eduyr!=. & edueg==1)		
replace tmp_u = . if eduyr==. & edueg==1								
								
								
foreach i in a u {							
	bys hh_id: egen educ_`i' = max(tmp_`i')
	replace educ_`i' = . if educ_`i'==0 & hedudt==0 				// policy
	drop tmp_`i'
	
	ta educ_`i', m							
}
																
lab var educ_a "mpi: not deprived in yos"
lab var educ_u "dst: not deprived in yos"



**## nutrition

gen nutr9 = .
lab var nutr9 "no nutrition data"


gen chelg = .
lab var chelg "child eligible for measurement" 


gen chswgt = .
lab var chswgt "sample weight child u5" 



**## child mortality
	
ta bh_cm,m   									// generated in bh recode
ta bh_cm if bh_id==1, m	


gen tmp = bh_cm
replace tmp = 0 if v201==0 | no_f_elg==1 	// never gave birth | non-elg fem	
	
bys hh_id: egen cm = sum(tmp),m


recode cm (1/max=0) (0=1) if cm!=.
lab var cm "not deprived in cm"
ta cm, m 											// mpi = dst

drop tmp



**## electricity 

codebook hv206, ta (9)

recode hv206 (0=0 "no") (1=1 "yes") (9=.), gen (elct)			// mpi = dst
lab var elct "not deprived in elct"


svy: prop elct 										// qc: matches the report 



**## sanitation

desc hv205 hv225 

codebook hv205, ta(99) 	
	
recode hv205 (11/13 15/22 41 = 1 "yes") ///
(14 23 31 42/96 = 0 "no") (99=.), gen(sani_a)    					 // mpi


codebook hv225

replace sani_a = 0 if hv225==1				
lab var sani_a "mpi: not deprived in sani"


svy: prop sani_a 							// qc: matches the report (p.348)


recode hv205 (11/23 41/43 = 1 "yes") ///
(31 96 = 0 "no") (99=.), gen(sani_u)   							// dst

lab var sani_u "dst: not deprived in sani"
ta hv205 sani_u, m



**## drinking water 

codebook hv201, ta(99)

recode hv201 (11/31 41 51/72 = 1 "yes") ///
(32 42 43 96 = 0 "no") (99=.), gen(wtr_sdg) 
 							
lab var wtr_sdg "improved source of dwtr sdg"

ta hv201 wtr_sdg,m


svy: prop wtr_sdg						// qc: matches the report (p.342)



codebook hv204, ta(99) 				// time to water

	
clonevar wtr_a = wtr_sdg	

replace wtr_a = 0 if hv204 >=30 & hv204 <=900		// mpi >=30 mins			 
lab var wtr_a "mpi: not deprived in dwtr"
ta wtr_a,m


	
clonevar wtr_u = wtr_sdg	
														   
replace wtr_u = 0 if hv204 >45 & hv204 <=900		// dst	> 45 mins			  
lab var wtr_u "dst: not deprived in dwtr"
ta wtr_u,m 



**## housing

desc hv213 hv214 hv215


codebook hv213, ta (99)			// improved = rudimentary & finished floor
 
recode hv213 (21/36 = 1 "yes") (11 96 = 0 "no") (99=.), gen(floor)
lab var floor "hh has improved floor"
ta hv213 floor,m


codebook hv214, ta (99)					// improved = finished walls 

recode hv214 (31/37 = 1 "yes") (11/28 96 = 0 "no") (99=.), gen(wall)
lab var wall "hh has improved wall"
ta hv214 wall,m


codebook hv215, ta (99)					// improved = finished roofing 

recode hv215 (31/37 = 1 "yes") (11/24 96 = 0 "no") (99=.), gen(roof)
lab var roof "hh has improved roof"
ta hv215 roof,m	


gen hsg_a = 1  														// mpi
replace hsg_a = 0 if floor==0 | wall==0 | roof==0
replace hsg_a = . if floor==. & wall==. & roof==.
lab var hsg_a "mpi: not deprived in hsg"
ta hsg_a, m



gen hsg_u = 1 														// dst
replace hsg_u = 0 if ///
(floor==0 & wall==0 & roof==1) | ///
(floor==0 & wall==1 & roof==0) | ///
(floor==1 & wall==0 & roof==0) | ///
(floor==0 & wall==0 & roof==0)

replace hsg_u = . if floor==. & wall==. & roof==.
lab var hsg_u "dst: not deprived in hsg"
ta hsg_u, m



**## cooking fuel 

codebook hv226, ta (99)


recode hv226 (1/8 95 96 = 1 "yes") ///
(9/17 = 0 "no") (99=.), gen(ckfl) 										// mpi

ta hv222 if hv226==96,m  // other fuel; other stove

lab var ckfl "not deprived in ckfl"	
ta hv226 ckfl,m 

svy: prop ckfl 						// qc: matches the report (p.19)



**## assets

lookfor car van truck
codebook hv212	
clonevar car = hv212  
lab var car "hh has car"



lookfor telephone téléphone mobilephone ipod telefone tele celular
ta hv221 hv243a,m nol
gen phone = (hv221==1 | hv243a==1)
lab var phone "hh has telephone"			
ta hv221 hv243a if phone==1,m


lookfor radio walkman stereo stéréo
codebook hv207
clonevar radio = hv207
lab var radio "hh has radio"


lookfor tv television plasma lcd télé tele
codebook hv208
clonevar tv = hv208 
lab var tv "hh has television"


lookfor refrigerator réfrigérateur refri freezer
codebook hv209
clonevar fridge = hv209
lab var fridge "hh has refrigerator"


lookfor bicycle bicyclette bicicleta
codebook hv210
clonevar bike = hv210 
lab var bike "hh has bicycle"	


lookfor motorbike moto
codebook hv211	
clonevar mbike = hv211
lab var mbike "hh has motorbike"


lookfor brouette cart carro carreta
codebook hv243c
clonevar acart = hv243c
lab var acart "hh has animal cart"	


lookfor computer ordinateur laptop ipad tablet 
codebook hv243e
clonevar comp = hv243e
lab var comp "hh has computer"



lab def la_at 0"no" 1"yes"
foreach var in car phone radio tv fridge bike mbike acart comp {
lab val `var' la_at	
replace `var' = . if `var'==9 | `var'==99 | `var'==8 | `var'==98	   	
}

		

egen n_bast = rowtotal (phone radio tv fridge bike mbike acart comp), m
lab var n_bast "basic assets owned by hh" 
 
 
gen asst_a = (car==1 | n_bast > 1)   								// mpi
replace asst_a = . if car==. & n_bast==.
lab var asst_a "mpi: not deprived in assets"



gen	asst_u = (car==1 | n_bast > 0)  								// dst
replace asst_u = . if car==. & n_bast==.
lab var asst_u "dst: not deprived in assets"



**# step 3: spec & flav

recode cm   	(0=1)(1=0) , gen(d_cm)						// MPI est 
recode nutr9 	(0=1)(1=0) , gen(d_nutr)
recode satt_a 	(0=1)(1=0) , gen(d_satt)
recode educ_a 	(0=1)(1=0) , gen(d_educ)
recode elct		(0=1)(1=0) , gen(d_elct)
recode wtr_a 	(0=1)(1=0) , gen(d_wtr)
recode sani_a	(0=1)(1=0) , gen(d_sani)
recode hsg_a 	(0=1)(1=0) , gen(d_hsg)
recode ckfl 	(0=1)(1=0) , gen(d_ckfl)
recode asst_a 	(0=1)(1=0) , gen(d_asst)


recode cm   	(0=1)(1=0) , gen(d_cm_01)	 			   // hot MPI est 
recode nutr9 	(0=1)(1=0) , gen(d_nutr_01)
recode satt_a 	(0=1)(1=0) , gen(d_satt_01)
recode educ_a 	(0=1)(1=0) , gen(d_educ_01)
recode elct		(0=1)(1=0) , gen(d_elct_01)
recode wtr_a 	(0=1)(1=0) , gen(d_wtr_01)
recode sani_a	(0=1)(1=0) , gen(d_sani_01)
recode hsg_a 	(0=1)(1=0) , gen(d_hsg_01)
recode ckfl 	(0=1)(1=0) , gen(d_ckfl_01)
recode asst_a 	(0=1)(1=0) , gen(d_asst_01)




recode cm   	(0=1)(1=0) , gen(dst_cm)					// dst est
recode nutr9 	(0=1)(1=0) , gen(dst_nutr)
recode satt_u 	(0=1)(1=0) , gen(dst_satt)
recode educ_u 	(0=1)(1=0) , gen(dst_educ)
recode elct 	(0=1)(1=0) , gen(dst_elct)
recode wtr_u  	(0=1)(1=0) , gen(dst_wtr)
recode sani_u 	(0=1)(1=0) , gen(dst_sani)
recode hsg_u  	(0=1)(1=0) , gen(dst_hsg)
recode ckfl 	(0=1)(1=0) , gen(dst_ckfl)
recode asst_u 	(0=1)(1=0) , gen(dst_asst) 


recode cm   	(0=1)(1=0) , gen(dst_cm_01)				   	// hot dst est
recode nutr9	(0=1)(1=0) , gen(dst_nutr_01)
recode satt_u 	(0=1)(1=0) , gen(dst_satt_01)
recode educ_u 	(0=1)(1=0) , gen(dst_educ_01)
recode elct 	(0=1)(1=0) , gen(dst_elct_01)
recode wtr_u  	(0=1)(1=0) , gen(dst_wtr_01)
recode sani_u 	(0=1)(1=0) , gen(dst_sani_01)
recode hsg_u  	(0=1)(1=0) , gen(dst_hsg_01)
recode ckfl 	(0=1)(1=0) , gen(dst_ckfl_01)
recode asst_u 	(0=1)(1=0) , gen(dst_asst_01) 



lab def la_dp 1"yes" 0"no"
foreach var in d_* dst_* d_*_* dst_*_* {
lab val `var' la_dp
}

foreach var in cm nutr satt educ elct wtr sani hsg ckfl asst {
lab var d_`var' "deprived in `var'"
lab var d_`var'_01 "hot: deprived in `var'"
lab var dst_`var' "dst: deprived in `var'"
lab var dst_`var'_01 "hot-dst: deprived in `var'"
}


keep hh_id ind_id strata psu weight chswgt ///
sex age agec* area region* hship intvw_* ///
scheg_a att edueg eduyr chelg  ///
d_* dst_* wtr_sdg hv204

	 
order hh_id ind_id strata psu weight chswgt ///
sex age agec* area region* hship intvw_* ///
scheg_a att edueg eduyr chelg  ///
d_* dst_* wtr_sdg hv204


mdesc psu strata area age hship region* d_* dst_*


char _dta[cty] "Philippines"
char _dta[ccty] "PHL"
char _dta[year] "2022" 	
char _dta[survey] "DHS"
char _dta[ccnum] "608"
char _dta[type] "micro"


sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/phl_dhs22.dta", replace 
