********************************************************************************
/*
Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
Global Multidimensional Poverty Index - Nigeria MICS 2016-17 
[STATA do-file]. Available from OPHI website: http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk
*/
********************************************************************************

clear all 
set more off
set maxvar 10000


global path_in "../rdta/Nigeria MICS 2016-17"  		 // folder path
global path_out "cdta"
global path_ado "ado"


********************************************************************************
**# Step 1: Data preparation 
********************************************************************************


********************************************************************************
**# Step 1.1 Children under 5 years
********************************************************************************

use "$path_in/ch.dta", clear 

rename _all, lower	


gen double ind_id = hh1*100000 + hh2*100 + ln 
format ind_id %20.0g
duplicates report ind_id
	//No duplicates 

	
gen child_id=1
lab var child_id "child under 5" 
count			// 28,578 children; matches report (p.iii)
	
	
codebook hl4, tab (9) 	//sex
clonevar gender = hl4



desc cage caged 				// qc: use age in days if available 
ta cage if caged>9000 			// qc: check age in months if days missing
clonevar age_days = caged
replace age_days = trunc(cage*(365/12)) if caged>=9000 & cage<9000
su age_days 					

	
gen str6 ageunit = "days"
lab var ageunit "days"



codebook an3, ta (999) 					// qc: body weight in kilogram (kg)
clonevar weight = an3 if an3<99 		// qc: check out of range value 
ta an2 an3 if an3>=99 | an3==., m    	// qc: why missing	
*hist weight, freq normal



codebook an4, ta (999) 					// qc: height in centimeters (cm)
clonevar height = an4 if an4<900  		// qc: check out of range value
ta an2 an4 if an4>=999 | an4==., m		// qc: why missing
*hist height, freq normal

	
codebook an4a  										// how child was measured
gen measure = "l" if an4a==1 							// lying down
replace measure = "h" if an4a==2 						// standing up
replace measure = " " if an4a==9 | an4a==0 | an4a==.	// " " if unknown
ta measure 


gen str1 oedema = "n"  			// oedema


gen sw = 1						// sampling weight

clonevar weight_ch = chweight
lab var weight_ch "sample weight child under 5"



adopath + "$path_ado/igrowup_stata"			     // compute z-score
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "children_nutri_nga"
lab var datalab "Working file"

igrowup_restricted reflib datalib datalab gender ///
age_days ageunit weight height measure oedema sw


use "$path_out/children_nutri_nga_z_rc.dta", clear 


gen	underweight = (_zwei < -2.0) 
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "child is underweight"
ta underweight [aw=weight_ch] 		 


gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "child is stunted"
ta stunting [aw=weight_ch]			


gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "child is wasted"
ta wasting [aw=weight_ch]


gen	underweight_u = (_zwei < -3.0) 
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "child is severely underweight"


gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "child is severely stunted"


gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "child is severely wasted"

 
count if _fwei==1 | _flen==1  	// z-scores which are biologically implausible 
 


keep ind_id child_id weight_ch underweight* stunting* wasting* 
order ind_id child_id weight_ch underweight* stunting* wasting*
sort ind_id
save "$path_out/NGA16-17_CH.dta", replace


erase "$path_out/children_nutri_nga_z_rc.xls"  				 // erase files 
erase "$path_out/children_nutri_nga_prev_rc.xls"
erase "$path_out/children_nutri_nga_z_rc.dta"


********************************************************************************
**# Step 1.2  BIRTH RECODE 
*** (female 15-49 years who ever gave birth)  
********************************************************************************

use "$path_in/bh.dta", clear


rename _all, lower	

		
gen double ind_id = hh1*100000 + hh2*100 + ln
format ind_id %20.0g


gen fem_bh = 1
lab var fem_bh "female with birth history" 



desc bh4c bh9c	
gen date_death = bh4c + bh9c		//  (date of birth) + (age at death)	
gen mdead_svy = wdoi-date_death		//	(date of interview) - (date of death)	
replace mdead_svy = . if (bh9c==0 | bh9c==.) & bh5==1
/* note: replaced children who are alive as '.' to 
distinguish them from children who died at 0 months */ 

		
gen ydead_svy = mdead_svy/12
lab var ydead_svy "total years dead from svy year"


gen age_death = bh9c if bh5==2
lab var age_death "age at death (months)"	
ta age_death, m


codebook bh5, tab (10)	
gen child_died = 1 if bh5==2
replace child_died = 0 if bh5==1
replace child_died = . if bh5==.
lab define lab_died 0"alive" 1"died"
lab values child_died lab_died
lab var child_died "child alive/dead"
ta bh5 child_died, m 
	
	
bys ind_id: egen tot_child_died = sum(child_died)
// note: for each woman, sum the number of children who died


gen child18_died = child_died // child under 18 mortality in the last 5 years
replace child18_died=0 if age_death>=216 & age_death!=.
lab def lab_u18died 1 "child u18 has died" 0 "child is alive/died but older"
lab val child18_died lab_u18died
ta child18_died, m	

bys ind_id: egen tot_child18_died_5y=sum(child18_died) if ydead_svy<=5

replace tot_child18_died_5y=0 if tot_child18_died_5y==. & tot_child_died>=0 & tot_child_died!=.
/* note: all children who are alive or who died longer than 
5 years from the interview date are replaced as '0' */

replace tot_child18_died_5y=. if child18_died==1 & ydead_svy==.
// note: replaced as '.' if there is no data on when the child died  

ta tot_child_died tot_child18_died_5y, m

bys ind_id: egen childu18_died_per_wom_5y = max(tot_child18_died_5y)
lab var childu18_died_per_wom_5y "total under 18 death for each women in the last 5 years"


bys ind_id: gen id=1 if _n==1
keep if id==1								// keep one observation per women
drop id
duplicates report ind_id 


keep ind_id fem_bh childu18_died_per_wom_5y		// keep relevant variables
order ind_id fem_bh childu18_died_per_wom_5y
sort ind_id
save "$path_out/NGA16-17_BH.dta", replace	



********************************************************************************
**# Step 1.3  WOMEN's RECODE  
*** (eligible female 15-49 years)
********************************************************************************

use "$path_in/wm.dta", clear 
	
rename _all, lower	

	
gen double ind_id = hh1*100000 + hh2*100 + ln
format ind_id %20.0g
duplicates report ind_id


gen wom_id = 1 
lab var wom_id "female 15-49 recode" 
count


ta wb2, m
	// qc: women 15-49 years interviewed	

ta cm1 cm8, m
	/* note: women who reported child mortality but identified 
	 no birth; we use this data in section 2.4. */
	 
ta mstatus, m
ta mstatus if cm1==. & cm8==.,m 
	// qc: check whether birth data is '.' for those never married

	
codebook mstatus ma6, tab (10)
ta mstatus ma6, m 
gen marital_wom = 1 if mstatus == 3 & ma6==.
	//1: never married
replace marital_wom = 2 if mstatus == 1 & ma6==.
	//2: currently married
replace marital_wom = 3 if mstatus == 2 & ma6==1
	//3: widowed	
replace marital_wom = 4 if mstatus == 2 & ma6==2
	//4: divorced	
replace marital_wom = 5 if mstatus == 2 & ma6==3
replace marital_wom = 5 if (mstatus == 2 & ma6==9) | (mstatus == 2 & ma6==.)
	//5: separated/not living together	
lab def lab_mar 1"never married" 2"currently married" ///
3"widowed" 4"divorced" 5"not living together"
lab val marital_wom lab_mar	
lab var marital_wom "marital status of women"
ta marital_wom, m
ta ma6 marital_wom, m
ta mstatus marital_wom, m


	
keep ind_id wom_id cm1 cm8 cm9a cm9b *_wom
order ind_id wom_id cm1 cm8 cm9a cm9b *_wom
sort ind_id
save "$path_out/NGA16-17_WM.dta", replace



********************************************************************************
**# Step 1.4  MEN'S RECODE 
********************************************************************************

use "$path_in/mn.dta", clear 

rename _all, lower

	
gen double ind_id = hh1*100000 + hh2*100 + ln
format ind_id %20.0g
duplicates report ind_id


gen men_id = 1 	
lab var men_id "man 15-49 recode" 
count


ta mcm1 mcm8, m
/* note: men who reported child mortality but identified 
as never fathered a child; we use this data in section 2.4. */



codebook mmstatus mma6, tab (10)
ta mmstatus mma6, m
gen marital_men = 1 if mmstatus == 3 & mma6==.
	//1: never married
replace marital_men = 2 if mmstatus == 1 & mma6==.
	//2: currently married
replace marital_men = 3 if mmstatus == 2 & mma6==1
	//3: widowed	
replace marital_men = 4 if mmstatus == 2 & mma6==2
	//4: divorced	
replace marital_men = 5 if mmstatus == 2 & mma6==3
replace marital_men = 5 if (mmstatus == 2 & mma6==9) | (mmstatus == 2 & mma6==.)
	//5: separated/not living together	
lab def lab_mar 1"never married" 2"currently married" ///
3"widowed" 4"divorced" 5"not living together"
lab val marital_men lab_mar	
lab var marital_men "marital status of men"
ta marital_men, m
ta mma6 marital_men, m
ta mmstatus marital_men, m
	
		
keep ind_id men_id mcm1 mcm8 mcm9a mcm9b *_men
order ind_id men_id mcm1 mcm8 mcm9a mcm9b *_men
sort ind_id
save "$path_out/NGA16-17_MN.dta", replace


********************************************************************************
**# Step 1.5 HOUSEHOLD RECODE 
********************************************************************************

use "$path_in/hh.dta", clear 

	
rename _all, lower	


gen	double hh_id = hh1*100 + hh2 
format	hh_id %20.0g
duplicates report hh_id 


save "$path_out/NGA16-17_HH.dta", replace


********************************************************************************
**# Step 1.6 HL - HOUSEHOLD MEMBER  
********************************************************************************

use "$path_in/hl.dta", clear 
	
rename _all, lower


gen double hh_id = hh1*100 + hh2 
format hh_id %20.0g
lab var hh_id "household id"


gen double ind_id = hh1*100000 + hh2*100 + hl1 
format ind_id %20.0g
lab var ind_id "individual id"
duplicates report ind_id 


sort ind_id

bys hh_id: gen id = _n
count if id==1
	// 33,901 hh - matches report, p. iii.

	
********************************************************************************
**# Step 1.7 DATA MERGING 
********************************************************************************  
 
merge 1:1 ind_id using "$path_out/NGA16-17_BH.dta"       // bh recode
drop _merge
erase "$path_out/NGA16-17_BH.dta" 
 
 

merge 1:1 ind_id using "$path_out/NGA16-17_WM.dta"  	 // wm recode
drop _merge
erase "$path_out/NGA16-17_WM.dta"



merge m:1 hh_id using "$path_out/NGA16-17_HH.dta"		 // hh recode
tab hh9 if _m==2
drop  if _merge==2   					// hh not interviewed, excluded 
drop _merge
erase "$path_out/NGA16-17_HH.dta"



merge 1:1 ind_id using "$path_out/NGA16-17_MN.dta"   	// mn recode
drop _merge
erase "$path_out/NGA16-17_MN.dta"



merge 1:1 ind_id using "$path_out/NGA16-17_CH.dta"		// ch recode
drop _merge
erase "$path_out/NGA16-17_CH.dta"



sort ind_id


********************************************************************************
**# Step 1.8 CONTROL VARIABLES
********************************************************************************

count if wom_id==1					// no eligible women 15-49 years for cm
count if hl7>0 & hl7!=.				// qc: match between WM & HL datafiles 
gen	fem_eligible = (wom_id==1)  	// no eligible women for cm
bys	hh_id: egen hh_n_fem_eligible = sum(fem_eligible) 	
gen	no_fem_eligible = (hh_n_fem_eligible==0) 									
lab var no_fem_eligible "hh has no elig woman"
ta no_fem_eligible, m



count if men_id==1 					// no eligible men for cm
count if hl7a>0 & hl7a!=.			// qc: match between MN & HL datafiles
gen	male_eligible = (men_id==1)	// no eligible men for cm
bys	hh_id: egen hh_n_male_eligible = sum(male_eligible)  
gen	no_male_eligible = (hh_n_male_eligible==0) 	
lab var no_male_eligible "hh has no elig man for interview"
ta no_male_eligible, m


	
count if child_id==1			// no eligible child for anthropometrics
count if hl7b>0 & hl7b!=.		// qc: match between CH & HL datafiles		
gen	child_eligible = (child_id==1) 
bys	hh_id: egen hh_n_children_eligible = sum(child_eligible)  
gen	no_child_eligible = (hh_n_children_eligible==0) 
lab var no_child_eligible "hh has no elig children for anthropometric"
ta no_child_eligible, m
	
	

sort hh_id


********************************************************************************
**# Step 1.9 RENAMING DEMOGRAPHIC VARIABLES 
********************************************************************************


clonevar weight = hhweight  							 //sample weight 
lab var weight "sample weight"


*gen psu = hh1 											//sample design
lab var psu "primary sampling unit"

clonevar strata = stratum
lab var strata "sample strata"

svyset psu [pw=weight] , strata(strata)	 singleunit(centered)



codebook hh6, ta (9)									// area: urban-rural		
recode hh6 (1=1 "urban") (2=0 "rural"), gen (area)						 
lab var area "area: urban-rural"
ta hh6 area, m


codebook hl3, ta (99)									     // relationship
recode hl3 (1=1 "head")(2=2 "spouse")(3 13=3 "child") ///
(4/12=4 "extended family")(96=5 "not related")(14=6 "maid") ///
(98=.), gen (relationship)
lab var relationship "relationship to the head of hh"
ta hl3 relationship, m	



codebook hl4 											// sex
clonevar sex = hl4 
lab var sex "sex of household member"


bys	hh_id: egen missing_hhead = min(relation) 			// headship
ta missing_hhead,m 
gen household_head=.
replace household_head=1 if relation==1 & sex==1 
replace household_head=2 if relation==1 & sex==2
bys hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
lab def head 1"male-headed" 2"female-headed"
lab val headship head
lab var headship "household headship"
ta headship, m


codebook hl6, tab (999) 								// age; age group
clonevar age = hl6 if hl6<98  
lab var age "age of household member"


recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+") , gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"



gen member = 1 										// hh size
bys hh_id: egen hhsize = sum(member)
lab var hhsize "household size"
ta hhsize, m
compare hhsize hh11 			// hh48 from datafile matches generated hhsize
drop member



codebook hh7, tab (99)			 				// 37 states
decode hh7, gen(temp)
replace temp =  proper(temp)
encode temp, gen(region)
lab var region "subnational region"
*ta hh7 region, m
drop temp
lab def lab_reg ///
1 "Abia" 2 "Adamawa" 3 "Akwa Ibom" 4 "Anambra" 5 "Bauchi" 6 "Bayelsa" ///
7 "Benue" 8 "Borno" 9 "Cross River" 10 "Delta" 11 "Ebonyi" 12 "Edo" ///
13 "Ekiti" 14 "Enugu" 15 "FCT" 16 "Gombe" 17 "Imo" 18 "Jigawa" 19 "Kaduna" ///
20 "Kano" 21 "Katsina" 22 "Kebbi" 23 "Kogi" 24 "Kwara" 25 "Lagos" ///
26 "Nasarawa" 27 "Niger" 28 "Ogun" 29 "Ondo" 30 "Osun" 31 "Oyo" ///
32 "Plateau" 33 "Rivers" 34 "Sokoto" 35 "Taraba" 36 "Yobe" 37 "Zamfara"
lab val region lab_reg


clonevar region_01 = region 
lab var region_01 "hot: subnational region"


********************************************************************************
**#  Step 2 Data preparation  
***  Standardisation of the global MPI indicators 
********************************************************************************

********************************************************************************
**# Step 2.1 Years of Schooling 
********************************************************************************

codebook ed4a ed4b ed3, tab (99)
	// qc: check the numeric of missing values & early educ (ece)
	

clonevar edulevel = ed4a
replace edulevel = . if ed4a==. | ed4a==8 |ed4a==98 | ed4a==99  
	// note: missing values or out of range are replaced as "."
replace edulevel = 0 if ed3==2 | ed4a==4 
	/*  note: those who never attended school are replaced as '0'
	    those who attended non-formal education. */	
lab var edulevel "highest level of educ"



gen eduhighyear = ed4b - 9 
	// qc: recode
replace eduhighyear = 0  if ed3==2 | ed4b<=3 | ed4a==4 
	// qc: preschool, never attended school, informal - replaced as '0'
lab var eduhighyear "highest grade"


ta eduhighyear edulevel, m

gen	eduyears = eduhighyear

replace eduyears = 0 if age<10  		
replace eduyears = 0 if (age==10 | age==11) & eduyears < 6 
			// non-eligible age group		
lab var eduyears "total years of educ"
ta eduyears, m
ta eduyears edulevel, m


gen educ_elig = 1  							// eligibility for educ indicator 
replace educ_elig = 0 if age < 10  
replace educ_elig = 0 if (age==10 | age==11) & eduyears < 6 
lab def lab_educ_elig 0"not eligible" 1"eligible"  
lab val educ_elig lab_educ_elig
lab var educ_elig "eligibility for educ indicator"
ta eduyears educ_elig, m


/* control variable: information on years of educ 
	present for at least 2/3 of the eligible householders */		
gen temp = 1 if eduyears!=. & educ_elig == 1
bys	hh_id: egen no_mis_edu = sum(temp)		// elig with educ data 
	
bysort hh_id: egen hhs = sum(educ_elig == 1) 		// all eligible for educ

replace no_mis_edu = no_mis_edu/hhs
replace no_mis_edu = (no_mis_edu>=2/3)		
ta no_mis_edu, m							// qc: missing (0) is < 0.5% 
lab var no_mis_edu "no missing yos"
drop temp hhs


*** MPI ***
/* Householders are considered not deprived if at least 
one eligible householder has six or more years of education. */
******************************************************************* 
gen	educ6 = (eduyears>=6 & eduyears!=.)
replace educ6 = . if eduyears==.

bys hh_id: egen educ = max(educ6)
replace educ = . if educ==0 & no_mis_edu==0
lab var educ "not deprived in education"
ta educ, m


*** Destitution ***
/* Householders are considered not deprived if at least 
one eligible householder has one or more years of education. */
******************************************************************* 
gen	educ1 = (eduyears>=1 & eduyears!=.)
replace educ1 = . if eduyears==.

bys	hh_id: egen educ_u = max(educ1)
replace educ_u = . if educ_u==0 & no_mis_edu==0
lab var educ_u "dst: not deprived in education"
ta educ_u, m



********************************************************************************
**# Step 2.2 Child School Attendance ***
********************************************************************************

codebook ed5 ed3, tab (99)

recode ed5 (1=1 "attending") (2=0 "not attending") (9=.), gen (attendance)
replace attendance = 0 if ed3==2 			// never ever attended school	
replace attendance = 0 if age<5 | age>24 	// not of school age	
lab var attendance "current school year"	
ta ed5 attendance, m


*** MPI ***
/* Householders are considered not deprived if all 
school-aged children are attending up to class 8. */ 
******************************************************************* 
gen	child_schoolage = (schage>=6 & schage<=14)
lab var child_schoolage "eligible for school attendance"		
	// qc: official school entrance age to primary school: 6 years
	// qc: age range 6-14 (=6+8) 

	
	/* control variable: school attendance data is 
	missing for at least 2/3 of the school-aged children */
count if child_schoolage==1 & attendance==.			// qc: missing satt
gen temp = 1 if child_schoolage==1 & attendance!=. 	// elig children in school 
bys hh_id: egen no_missing_atten = sum(temp)		
gen temp2 = 1 if child_schoolage==1					//elig children
bys hh_id: egen hhs = sum(temp2)					
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)	 		
ta no_missing_atten, m							// qc: missing (0) is < 0.5% 
lab var no_missing_atten "no missing satt"		
drop temp temp2 hhs
		
bys hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
lab var hh_children_schoolage "hh has elig child"

gen	not_atten = (attendance==0) if child_schoolage==1
replace not_atten = . if attendance==. & child_schoolage==1

bys	hh_id: egen any_not_atten = max(not_atten)

gen	satt = (any_not_atten==0) 
replace satt = . if any_not_atten==.
replace satt = 1 if hh_children_schoolage==0
replace satt = . if satt==1 & no_missing_atten==0 
lab var satt "non-deprivation in school attendance"
ta satt, m

	
*** Destitution ***
/* Householders are considered not deprived if all 
school-aged children are attending up to class 6. */ 
******************************************************************* 
gen	child_schoolage_u = (schage>=6 & schage<=12) 
lab var child_schoolage_u "dst: eligible for school attendance"	

	
	/* control variable: school attendance data is 
	missing for at least 2/3 of the school-aged children */	
count if child_schoolage_u==1 & attendance==.	
gen temp = 1 if child_schoolage_u==1 & attendance!=.
bys hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_u==1	
bys hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)			
ta no_missing_atten_u, m			// qc: missing (0) is < 0.5% 
lab var no_missing_atten_u "no missing satt"		
drop temp temp2 hhs
		
bys	hh_id: egen hh_children_schoolage_u = sum(child_schoolage_u)
replace hh_children_schoolage_u = (hh_children_schoolage_u>0) 
lab var hh_children_schoolage_u "hh has elig child"

gen	atten_6 = (attendance==1) if child_schoolage_u==1
replace atten_6 = . if attendance==. & child_schoolage_u==1

bys	hh_id: egen any_atten_6 = max(atten_6)

gen	satt_u = (any_atten_6==1) 
replace satt_u = . if any_atten_6==.
replace satt_u = 1 if hh_children_schoolage_u==0
replace satt_u = . if satt_u==0 & no_missing_atten_u==0 
lab var satt_u "dst: non-deprivation in school attendance"
ta satt_u, m


********************************************************************************
**# Step 2.3 Nutrition 
********************************************************************************

********************************************************************************
**# Step 2.3a underweight, stunting & wasting
********************************************************************************

bys hh_id: egen temp = max(underweight)
gen	hh_no_underweight = (temp==0)			//no child in the hh is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1	// no eligible child 
lab var hh_no_underweight "hh has no child underweight"
drop temp

bys hh_id: egen temp = max(underweight_u) 				//severely underweight
gen	hh_no_underweight_u = (temp==0) 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
lab var hh_no_underweight_u "dst: hh has no child underweight"
drop temp


bys hh_id: egen temp = max(stunting)
gen	hh_no_stunting = (temp==0)					//no child in the hh is stunted 
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1	// no eligible child 	
lab var hh_no_stunting "hh has no child stunted"
drop temp

bys hh_id: egen temp = max(stunting_u) 				//severely stunted
gen	hh_no_stunting_u = (temp==0) 
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
lab var hh_no_stunting_u "dst: hh has no child stunted"
drop temp


bys hh_id: egen temp = max(wasting)
gen	hh_no_wasting = (temp==0) 					// no child in the hh is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1		// no eligible child
lab var hh_no_wasting "hh has no child wasted"
drop temp

bys hh_id: egen temp = max(wasting_u) 					// severely wasted
gen	hh_no_wasting_u = (temp==0) 
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
lab var hh_no_wasting_u "dst: hh has no child wasted"
drop temp


gen uw_st = 1 if stunting==1 | underweight==1		// underweight or stunted 
replace uw_st = 0 if stunting==0 & underweight==0
replace uw_st = . if stunting==. & underweight==.
lab var uw_st "child is underweight or stunted"

bys hh_id: egen temp = max(uw_st)
gen	hh_no_uw_st = (temp==0)		// no child in the hh is underweight or stunted
replace hh_no_uw_st = . if temp==.
replace hh_no_uw_st = 1 if no_child_eligible==1		// no elig child
lab var hh_no_uw_st "hh has no child underweight or stunted"
drop temp


gen uw_st_u = 1 if stunting_u==1 | underweight_u==1 		// destitution
replace uw_st_u = 0 if stunting_u==0 & underweight_u==0
replace uw_st_u = . if stunting_u==. & underweight_u==.
lab var uw_st_u "dst: child is underweight or stunted"

bys hh_id: egen temp = max(uw_st_u)
gen	hh_no_uw_st_u = (temp==0) 
replace hh_no_uw_st_u = . if temp==.
replace hh_no_uw_st_u = 1 if no_child_eligible==1 
lab var hh_no_uw_st_u "dst: hh has no child underweight or stunted"
drop temp


********************************************************************************
**# Step 2.3b nutrition indicator
********************************************************************************

*** MPI ***
/* Householders are not deprived if all eligible 
person with anthropometric measurement in the household 
are nourished; or household has no eligible person. */
************************************************************************
gen	nutr_0 = 1  											
replace nutr_0 = 0 if hh_no_uw_st==0
replace nutr_0 = . if hh_no_uw_st==.
replace nutr_0 = 1 if no_child_eligible==1   		
lab var nutr_0 "non-deprivation in nutr (under 5)"
tab nutr_0, m


gen nutr_99 = .  								// no nutr data
lab var nutr_99 "no nutrition data"



*** Destitution ***
/* Householders are not deprived if all eligible person with
anthropometric measurement in the household are not severely 
undernourished; or household has no eligible person. */
************************************************************************
gen	nutr_0_u = 1  											
replace nutr_0_u = 0 if hh_no_uw_st_u==0
replace nutr_0_u = . if hh_no_uw_st_u==.
replace nutr_0_u = 1 if no_child_eligible==1   	
lab var nutr_0_u "dst: non-deprivation in nutr (under 5)"
tab nutr_0_u, m


gen nutr_99_u = .								// no nutr data
lab var nutr_99_u "dst: no nutrition data"


********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************
	
codebook cm9a cm9b mcm9a mcm9b
	/*cm9 or mcm9: number of sons who have died 
	  cm10 or mcm10: number of daughters who have died */	
	

ta cm1 cm8, m
ta childu18_died_per_wom_5y if cm8==1 & cm1==2,m
	/* women who reported child mortality but identified 
	 no birth; we use this data. */
			  
egen temp_f = rowtotal(cm9a cm9b), missing
replace temp_f = 0 if (cm1==1 & cm8==2) | (cm1==2 & cm8!=1) 
	// all elig woman who reported no child death
bys	hh_id: egen child_mortality_f = sum(temp_f), missing
lab var child_mortality_f "child mortality reported by woman"
ta child_mortality_f, m
drop temp_f
	
	
ta mcm1 mcm8, m

egen temp_m = rowtotal(mcm9a mcm9b), missing	
replace temp_m = 0 if (mcm1==1 & mcm8==2) | (mcm1==2 & mcm8!=1) 
	// all elig man who reported no child death
bys	hh_id: egen child_mortality_m = sum(temp_m), missing	
lab var child_mortality_m "child mortality reported by man"
ta child_mortality_m, m
drop temp_m


egen child_mortality = rowmax(child_mortality_f child_mortality_m)
replace child_mortality = 0 if child_mortality==. & no_fem_eligible==1 & no_male_eligible==1
lab var child_mortality "total child mortality in a hh"
ta child_mortality, m

	
*** MPI *** 
/* Householders are not deprived if all eligible women 
in the household reported zero mortality among children 
under 18 in the last 5 years from the survey year.*/
************************************************************************
ta childu18_died_per_wom_5y, m	
	
replace childu18_died_per_wom_5y = 0 if cm1==2 & cm8==2 
	// elig woman who never ever gave birth 
	
replace childu18_died_per_wom_5y = 0 if no_fem_eligible==1	
	// no elig woman
	
bys hh_id: egen childu18_mortality_5y = sum(childu18_died_per_wom_5y), missing
replace childu18_mortality_5y = 0 if childu18_mortality_5y==. & child_mortality==0
	// women has missing value and men reported no death 
lab var childu18_mortality_5y "total u18 child mortality last 5 years"
ta childu18_mortality_5y, m		
	
gen cm_0 = (childu18_mortality_5y==0)
replace cm_0 = . if childu18_mortality_5y==.
lab var cm_0 "non-deprivation in cm"
ta cm_0, m


*** Destitution *** 
*** (same as MPI) ***
************************************************************************
clonevar cm_0_u = cm_0	
lab var cm_0_u "dst: non-deprivation in cm"	


********************************************************************************
**# Step 2.5 Electricity 
********************************************************************************

*** MPI ***
/* Householders are considered not deprived 
if the household has electricity */
****************************************
codebook hc8a, tab (9)
recode hc8a  (1 = 1 "yes") (2 = 0 "no") (9=.), gen(elct)	 
lab var elct "non-deprivation in electricity"
ta hc8a elct, m

svy: prop elct if id==1
	// qc: matches the report (p.19)

	
*** Destitution ***
*** (same as MPI) ***
****************************************
gen elct_u = elct
lab var elct_u "dst: non-deprivation in electricity"


********************************************************************************
**# Step 2.6 Sanitation 
********************************************************************************


*** MPI ***
/* Householders are not deprived if the household has improved 
sanitation facilities that are not shared with other households. */
********************************************************************

desc ws8 ws9 

codebook ws8, ta(99) 	
	
recode ws8 (11/13 15/22 31 = 1 "yes") ///
(14 23 41/96 = 0 "no") (99=.), gen(sani)


codebook ws9

replace sani = 0 if ws9==1				
lab var sani "non-deprivation in sanitation"
ta ws8 sani, m


*** Destitution ***
/* Householders are not deprived if the 
household has sanitation facilities. */
********************************************************************
recode ws8 (11/51 = 1 "yes") ///
(95 96 = 0 "no") (99=.), gen(sani_u)

lab var sani_u "dst: non-deprivation in sanitation"
ta ws8 sani_u, m


********************************************************************************
**# Step 2.7 Drinking Water 
********************************************************************************


*** MPI ***
/* Householders are not deprived if household have access to safe 
drinking water and is under 30 minutes walk from home, round trip.*/
********************************************************************
codebook ws1, ta(99)

recode ws1 (11/31 41 51/71 91/92 = 1 "yes") ///
(32 42 81 96 = 0 "no") (99=.), gen(dwtr_src)
lab var dwtr_src "improved main source of drinking water"
ta ws1 dwtr_src , m


codebook ws4a, ta(999)

clonevar wtr = dwtr_src
replace wtr = 0 if ws4a >=30 & ws4a <=990		// time to water	
lab var wtr "non-deprivation in drinking water"
ta dwtr_src wtr, m



*** Destitution ***
/* Householders are not deprived if household have access to safe 
drinking water and is 45 minutes walk or less from home, round trip.*/
********************************************************************
clonevar wtr_u = dwtr_src						   
replace wtr_u = 0 if ws4a >45 & ws4a <=990				  
lab var wtr_u "dst: non-deprivation in drinking water"
ta dwtr_src wtr_u,m



********************************************************************************
**# Step 2.8 Housing
********************************************************************************

desc hc3 hc4 hc5

codebook hc3, tab (99)			// improved = rudimentary & finished floor 
recode hc3 (21/36 = 1 "yes") (11/12 96 = 0 "no") (99=.), gen(floor)
lab var floor "hh has improved floor"
ta hc3 floor, m
svy: prop floor if id==1	
	// qc: matches the report (p.19)


codebook hc4, tab (99)				// improved = finished roofing 
recode hc4 (31/36 = 1 "yes") (11/24 96 = 0 "no") (99=.), gen(roof)
lab var roof "hh has improved roof"
ta hc4 roof, m 	
svy: prop roof if id==1
	// qc: matches the report (p.19)
	
	
	
codebook hc5, tab (99)				// improved = finished walls 
recode hc5 (31/36 = 1 "yes") (11/26 96 = 0 "no") (99=.), gen(wall)
lab var wall "hh has improved wall"
ta hc5 wall, m
svy: prop wall if id==1
	// qc: matches the report (p.19)	

	
	
*** MPI ***
/* Householders are not deprived in housing if the roof, 
floor and walls are constructed from quality materials.*/
**************************************************************
gen hsg = 1
replace hsg = 0 if floor==0 | wall==0 | roof==0
replace hsg = . if floor==. & wall==. & roof==.
lab var hsg "non-deprivation in housing"
ta hsg, m


*** Destitution ***
/* Householders are not deprived in housing if at least two of the three 
components (roof/floor/walls) are constructed from quality materials. */
**************************************************************
gen hsg_u = 1

replace hsg_u = 0 if ///
(floor==0 & wall==0 & roof==1) | ///
(floor==0 & wall==1 & roof==0) | ///
(floor==1 & wall==0 & roof==0) | ///
(floor==0 & wall==0 & roof==0)

replace hsg_u = . if floor==. & wall==. & roof==.
lab var hsg_u "dst: non-deprivation in housing"
ta hsg_u, m


********************************************************************************
**# Step 2.9 Cooking Fuel 
********************************************************************************

/*
Solid fuel are solid materials burned as fuels, which includes coal as well as 
solid biomass fuels (wood, animal dung, crop wastes and charcoal). 

Source: 
https://apps.who.int/iris/bitstream/handle/10665/141496/9789241548885_eng.pdf
*/

desc hc6 


** MPI ***
/* Householders are considered not deprived if the 
household uses non-solid fuels for cooking. */
*****************************************************************

codebook hc6, tab (99)

recode hc6  (1/5 95 96 = 1 "yes") ///
(6/11 = 0 "no") (99=.), gen(ckfl)

ta hc6 ckfl,m 
lab var ckfl "non-deprivation in cooking fuel"	



*** Destitution  ***
*** (same as MPI) ***
****************************************	
clonevar ckfl_u = ckfl 
lab var ckfl_u "dst: non-deprivation in cooking fuel"	


********************************************************************************
**# Step 2.10 Assets ownership 
********************************************************************************

	// handphone/telephone/iphone/mobilephone/ipod
lookfor mobile
codebook hc8d hc9b 
gen telephone = (hc8d==1 | hc9b==1)
lab var telephone "hh has telephone"			
ta hc8d hc9b if telephone==1



	// radio/stereo/kindle
lookfor radio stereo stéréo
codebook hc8b
clonevar radio = hc8b 
lab var radio "hh has radio"



	// television/lcd tv/plasma tv/color tv/black & white tv
lookfor tv television plasma lcd télé tele
codebook hc8c
clonevar television = hc8c 
lab var television "hh has television"


	// refrigerator/icebox/fridge
lookfor refrigerator réfrigérateur refri
codebook hc8e
clonevar refrigerator = hc8e
lab var refrigerator "hh has refrigerator"
 

	// bicycle/cycle rickshaw
lookfor bicycle bicyclette bicicleta
codebook hc9c
clonevar bicycle = hc9c
lab var bicycle "hh has bicycle"		
	
	
	// motorbike/motorized bike/autorickshaw
lookfor motorbike moto
codebook hc9d	
clonevar motorbike = hc9d
lab var motorbike "hh has motorbike"
	
	
	// animal cart
lookfor brouette cart carro carreta
codebook hc9e
clonevar animal_cart = hc9e
lab var animal_cart "hh has animal cart"	


	// car/van/lorry/truck
lookfor car van truck
codebook hc9f	
clonevar car = hc9f 
lab var car "hh has car"		
	
	
	// computer/laptop/tablet
lookfor computer ordinateur laptop ipad tablet 
codebook hc8j
clonevar computer = hc8j
lab var computer "hh has computer"
	

	
lab def lab_asst 0"no" 1"yes"
foreach var in television radio telephone refrigerator car ///
			   bicycle motorbike computer animal_cart {
replace `var' = 0 if `var'==2 
lab val `var' lab_asst				   
replace `var' = . if `var'==9 
}


*** MPI ***
/* Householders are considered not deprived in assets 
if the household own more than one of the assets. */
*****************************************************************************
egen n_small_asst = rowtotal (television radio telephone refrigerator ///
bicycle motorbike computer animal_cart), miss
lab var n_small_asst "small assets owned by hh" 
   
gen asst = (car==1 | n_small_asst > 1) 
replace asst = . if car==. & n_small_asst==.
lab var asst "non-deprivation in assets"


*** Destitution ***
/* Householders are considered not deprived in assets 
if the household own at least one asset. */
*****************************************************************************	
gen	asst_u = (car==1 | n_small_asst>0)
replace asst_u = . if car==. & n_small_asst==.
lab var asst_u "dst: non-deprivation in assets" 
 
 	
********************************************************************************
*** Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************
	

desc hh5y hh5m hh5d    								 // interview dates
clonevar intvw_y = hh5y 	
clonevar intvw_m = hh5m 
clonevar intvw_d = hh5d 


recode cm_0   	(0=1)(1=0) , gen(d_cm)					// MPI for est 
recode nutr_0 	(0=1)(1=0) , gen(d_nutr)
recode satt 	(0=1)(1=0) , gen(d_satt)
recode educ 	(0=1)(1=0) , gen(d_educ)
recode elct		(0=1)(1=0) , gen(d_elct)
recode wtr 		(0=1)(1=0) , gen(d_wtr)
recode sani		(0=1)(1=0) , gen(d_sani)
recode hsg 		(0=1)(1=0) , gen(d_hsg)
recode ckfl 	(0=1)(1=0) , gen(d_ckfl)
recode asst 	(0=1)(1=0) , gen(d_asst)
 

recode cm_0_u   	(0=1)(1=0) , gen(dst_cm)			// dst for est
recode nutr_0_u		(0=1)(1=0) , gen(dst_nutr)
recode satt_u 		(0=1)(1=0) , gen(dst_satt)
recode educ_u 		(0=1)(1=0) , gen(dst_educ)
recode elct_u 		(0=1)(1=0) , gen(dst_elct)
recode wtr_u  		(0=1)(1=0) , gen(dst_wtr)
recode sani_u 		(0=1)(1=0) , gen(dst_sani)
recode hsg_u  		(0=1)(1=0) , gen(dst_hsg)
recode ckfl_u 		(0=1)(1=0) , gen(dst_ckfl)
recode asst_u 		(0=1)(1=0) , gen(dst_asst) 



recode cm_0   	(0=1)(1=0) , gen(d_cm_01)				// hot MPI for est 
recode nutr_99 	(0=1)(1=0) , gen(d_nutr_01)
recode satt 	(0=1)(1=0) , gen(d_satt_01)
recode educ 	(0=1)(1=0) , gen(d_educ_01)
recode elct		(0=1)(1=0) , gen(d_elct_01)
recode wtr 		(0=1)(1=0) , gen(d_wtr_01)
recode sani		(0=1)(1=0) , gen(d_sani_01)
recode hsg 		(0=1)(1=0) , gen(d_hsg_01)
recode ckfl 	(0=1)(1=0) , gen(d_ckfl_01)
recode asst 	(0=1)(1=0) , gen(d_asst_01)
 

recode cm_0_u   	(0=1)(1=0) , gen(dst_cm_01)			// hot dst for est
recode nutr_99_u	(0=1)(1=0) , gen(dst_nutr_01)
recode satt_u 		(0=1)(1=0) , gen(dst_satt_01)
recode educ_u 		(0=1)(1=0) , gen(dst_educ_01)
recode elct_u 		(0=1)(1=0) , gen(dst_elct_01)
recode wtr_u  		(0=1)(1=0) , gen(dst_wtr_01)
recode sani_u 		(0=1)(1=0) , gen(dst_sani_01)
recode hsg_u  		(0=1)(1=0) , gen(dst_hsg_01)
recode ckfl_u 		(0=1)(1=0) , gen(dst_ckfl_01)
recode asst_u 		(0=1)(1=0) , gen(dst_asst_01) 



lab def lab_dp 1"yes" 0"no"
foreach var in d_* dst_* d_*_* dst_*_* {
lab val `var' lab_dp
}

foreach var in cm nutr satt educ elct wtr sani hsg ckfl asst {
lab var d_`var' "deprived in `var'"
lab var dst_`var' "deprived in `var' (dst)"
lab var d_`var'_01 "deprived in `var' (hot)"
lab var dst_`var'_01 "deprived in `var' (dst-hot)"
}



keep hh_id ind_id strata psu weight sex age area ///
agec7 agec4 agec2 region region_* headship d_* dst_* 

	 
order hh_id ind_id strata psu weight sex age area ///
agec7 agec4 agec2 region region_* headship d_* dst_* 


mdesc psu strata area age headship region region_* d_* dst_*



char _dta[cty] "Nigeria"
char _dta[ccty] "NGA"
char _dta[year] "2016-2017" 	
char _dta[survey] "MICS"
char _dta[ccnum] "566"
char _dta[type] "micro"


sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/nga_mics16-17.dta", replace 

