********************************************************************************
/* Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
Global Multidimensional Poverty Index - Madagascar MICS 2018
[STATA do-file]. Available from http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk */
********************************************************************************

clear all 
set more off
set maxvar 10000



global path_in "../rdta/Madagascar MICS 2018"     // folder path
global path_out "cdta"
global path_ado "ado"



********************************************************************************
**# Step 1: Data preparation 
********************************************************************************
	

********************************************************************************
**# Step 1.1 Children under 5 years
********************************************************************************

use "$path_in/ch.dta", clear 

rename _all, lower	

gen double ind_id = hh1*1000000 + hh2*100 + ln 
format ind_id %20.0g


duplicates report ind_id


gen child_id=1
lab var child_id "child under 5" 
count 

	
codebook hl4, tab (9) 						// sex
clonevar gender = hl4



desc cage caged 				// qc: use age in days if available 
ta cage if caged>9000 			// qc: check age in months if days missing


codebook caged,ta(9999)

clonevar age_days = caged
replace age_days = trunc(cage*(365/12)) if age_days>=9000 & cage<9000
su age_days 					// qc: check min & max value


	
gen str6 ageunit = "days"
lab var ageunit "days"




codebook an8, ta (999) 					// qc: body weight in kilogram (kg)
clonevar weight = an8 if an8<99 		// qc: check out of range value 
ta uf17 an8 if an8>=99 | an8==., m  	// qc: why missing	
*hist weight, freq normal


	
codebook an11, ta (999) 				// qc: height in centimeters (cm)
clonevar height = an11 if an11<900  	// qc: check out of range value
ta uf17 an11 if an11>=999 | an11==., m	// qc: why missing
*hist height, freq normal



codebook an12  										// how child was measured
gen measure = "l" if an12==1 							// lying down
replace measure = "h" if an12==2 						// standing up
replace measure = " " if an12==9 | an12==0 | an12==.	// " " if unknown
ta measure 

			
gen str1 oedema = "n"  			// oedema


gen sw = 1						// sampling weight


clonevar weight_ch = chweight
lab var weight_ch "sample weight child under 5"



adopath + "$path_ado/igrowup_stata"
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "children_nutri_mdg"
lab var datalab "Working file"

igrowup_restricted reflib datalib datalab gender age_days ageunit weight ///
height measure oedema sw


		
use "$path_out/children_nutri_mdg_z_rc.dta", clear 


	
gen	underweight = (_zwei < -2.0) 
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "child is underweight"
ta underweight [aw=weight_ch] 			


gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "child is stunted"
ta stunting [aw=weight_ch]			


gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "child is wasted"
ta wasting [aw=weight_ch]


gen	underweight_u = (_zwei < -3.0) 
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "child is severely underweight"


gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "child is severely stunted"


gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "child is severely wasted"
 
 
count if _fwei==1 | _flen==1  	// z-scores which are biologically implausible
 
 
keep ind_id weight_ch child_id underweight* stunting* wasting* 
order ind_id weight_ch child_id underweight* stunting* wasting* 
sort ind_id
save "$path_out/MDG18_CH.dta", replace

	
	
erase "$path_out/children_nutri_mdg_z_rc.xls"    		 // erase files
erase "$path_out/children_nutri_mdg_prev_rc.xls"
erase "$path_out/children_nutri_mdg_z_rc.dta"

	
	
********************************************************************************
**# Step 1.2  BIRTH HISTORY
********************************************************************************

use "$path_in/bh.dta", clear

rename _all, lower	

		
gen double ind_id = hh1*1000000 + hh2*100 + wm3 
format ind_id %20.0g


gen fem_bh = 1
lab var fem_bh "female with birth history" 

	
	
desc bh4c bh9c	
gen date_death = bh4c + bh9c		//  (date of birth) + (age at death)	
gen mdead_svy = wdoi-date_death		//	(date of interview) - (date of death)	
replace mdead_svy = . if (bh9c==0 | bh9c==.) & bh5==1	
/* note: replaced children who are alive as '.' to 
distinguish them from children who died at 0 months */ 


gen ydead_svy = mdead_svy/12
lab var ydead_svy "total years dead from svy year"	


gen age_death = bh9c if bh5==2
lab var age_death "age at death (months)"	
ta age_death, m	
	
	
codebook bh5, tab (10)	
gen child_died = 1 if bh5==2
replace child_died = 0 if bh5==1
replace child_died = . if bh5==.
lab def lab_died 0"alive" 1"died"
lab val child_died lab_died
lab var child_died "child alive/dead"
ta bh5 child_died, m
	
	
bys ind_id: egen tot_child_died = sum(child_died)
 // note: number of children who died for each women
 
 
gen child18_died = child_died // child under 18 mortality in the last 5 years
replace child18_died=0 if age_death>=216 & age_death!=.
lab def lab_u18died 1 "child u18 has died" 0 "child is alive/died but older"
lab val child18_died lab_u18died
ta child18_died, m	
	
bys ind_id: egen tot_child18_died_5y=sum(child18_died) if ydead_svy<=5

replace tot_child18_died_5y=0 if tot_child18_died_5y==. & ///
								 tot_child_died>=0 & tot_child_died!=.
/* note: all children who are alive or who died longer than 
5 years from the interview date are replaced as '0' */
	
replace tot_child18_died_5y=. if child18_died==1 & ydead_svy==.
// note: replaced as '.' if there is no data on when the child died  

ta tot_child_died tot_child18_died_5y, m

bys ind_id: egen childu18_died_per_wom_5y = max(tot_child18_died_5y)
lab var childu18_died_per_wom_5y "total under 18 death in the last 5 years"
	

	
bys ind_id: gen id=1 if _n==1
keep if id==1								// keep one observation per women
drop id
duplicates report ind_id 
	

	
keep ind_id fem_bh childu18_died_per_wom_5y		// keep relevant variables
order ind_id fem_bh childu18_died_per_wom_5y
sort ind_id
save "$path_out/MDG18_BH.dta", replace	



********************************************************************************
**# Step 1.3  INDIVIDUAL (WOMEN) 
********************************************************************************

use "$path_in/wm.dta", clear 
	
rename _all, lower	

	 
gen double ind_id = hh1*1000000 + hh2*100 + wm3 
format ind_id %20.0g


duplicates report ind_id



gen wom_id = 1 
lab var wom_id "female 15-49 recode" 
count



ta wb4 wm17, m	// qc: 17,161 women interviewed; matches report. 	
	
ta cm1 cm8, m	
	/* note: women who reported child mortality but identified 
	 no birth; we use this data in section 2.4. */


	 	 
codebook mstatus ma6, ta (9)
ta mstatus ma6, m 
gen marital_wom = 1 if mstatus == 3 & ma6==.
	//1: never married
replace marital_wom = 2 if mstatus == 1 & ma6==.
	//2: currently married
replace marital_wom = 3 if mstatus == 2 & ma6==1
	//3: widowed	
replace marital_wom = 4 if mstatus == 2 & ma6==2
	//4: divorced	
replace marital_wom = 5 if mstatus == 2 & ma6==3
replace marital_wom = 5 if (mstatus == 2 & ma6==9) | (mstatus == 2 & ma6==.)
	//5: separated/not living together	
lab def lab_mar 1"never married" 2"currently married" ///
3"widowed" 4"divorced" 5"not living together"
lab val marital_wom lab_mar	
lab var marital_wom "marital status of women"
ta marital_wom, m
ta ma6 marital_wom, m
ta mstatus marital_wom, m	 
	 
	 
	
keep ind_id wom_id cm1 cm8 cm9 cm10 *_wom
order ind_id wom_id cm1 cm8 cm9 cm10 *_wom
sort ind_id
save "$path_out/MDG18_WM.dta", replace

 

********************************************************************************
**# Step 1.4  INDIVIDUAL (MEN)  
********************************************************************************

use "$path_in/mn.dta", clear 

rename _all, lower

 
gen double ind_id = hh1*1000000 + hh2*100 + ln 
format ind_id %20.0g

	
duplicates report ind_id 


gen men_id = 1 	
lab var men_id "man 15-49 recode" 
count		// 8,980 men 15-49 years 

		
	
ta mwb4 mwm17, m	// qc: 7,623 men interviewed; matches report.



codebook mmstatus mma6, ta (9)
ta mmstatus mma6, m
gen marital_men = 1 if mmstatus == 3 & mma6==.
	//1: never married
replace marital_men = 2 if mmstatus == 1 & mma6==.
	//2: currently married
replace marital_men = 3 if mmstatus == 2 & mma6==1
	//3: widowed	
replace marital_men = 4 if mmstatus == 2 & mma6==2
	//4: divorced	
replace marital_men = 5 if mmstatus == 2 & mma6==3
replace marital_men = 5 if (mmstatus == 2 & mma6==9) | (mmstatus == 2 & mma6==.)
	//5: separated/not living together	
lab define lab_mar 1"never married" 2"currently married" ///
3"widowed" 4"divorced" 5"not living together"
lab val marital_men lab_mar	
lab var marital_men "marital status of men"
ta marital_men, m
ta mma6 marital_men, m
ta mmstatus marital_men, m


 
keep ind_id men_id *_men 
order ind_id men_id *_men
sort ind_id
save "$path_out/MDG18_MN.dta", replace



********************************************************************************
**# Step 1.5 HOUSEHOLDS
********************************************************************************

use "$path_in/hh.dta", clear 
	
	
rename _all, lower	


gen	double hh_id = hh1*1000 + hh2 
format	hh_id %20.0g


duplicates report hh_id 	

	
save "$path_out/MDG18_HH.dta", replace

	
	
********************************************************************************
**# Step 1.6  HOUSEHOLD MEMBERS 
********************************************************************************

use "$path_in/hl.dta", clear 

rename _all, lower

	

gen double hh_id = hh1*1000 + hh2 
format hh_id %20.0g
lab var hh_id "household id"



gen double ind_id = hh1*1000000 + hh2*100 + hl1 
format ind_id %20.0g
lab var ind_id "individual id"


duplicates report ind_id 



sort ind_id


	
********************************************************************************
**# Step 1.7 DATA MERGING 
******************************************************************************** 
 
merge 1:1 ind_id using "$path_out/MDG18_BH.dta" 				// bh recode
drop _merge
erase "$path_out/MDG18_BH.dta"  
 
 
 
merge 1:1 ind_id using "$path_out/MDG18_WM.dta"					// wm recode
count if hl8>0 					   // qc: 18,812 women 15-49; matches report
drop _merge	
erase "$path_out/MDG18_WM.dta"



merge m:1 hh_id using "$path_out/MDG18_HH.dta" 					// hh recode
ta hh46 if _m==2 
drop if _merge==2								//hh not interviewed excluded 
drop _merge


bys hh_id: gen id = _n
count if id==1				      // qc: 17,870 hh interviewed; matches report
erase "$path_out/MDG18_HH.dta"




merge 1:1 ind_id using "$path_out/MDG18_MN.dta" 				// mn recode
drop _merge
count if hl9>0 & hl9!=.			       // qc: 8,980 men 15-49; matches report
erase "$path_out/MDG18_MN.dta"




merge 1:1 ind_id using "$path_out/MDG18_CH.dta" 				// ch recode
drop _merge
count if hl10>0				        // qc: 13,355 child under 5; matches report
erase "$path_out/MDG18_CH.dta"



sort ind_id



********************************************************************************
**# Step 1.8 CONTROL VARIABLES
********************************************************************************


count if wom_id==1					// eligible women for cm
count if hl8>0 & hl8!=.				// qc: match between WM & HL datafiles 
gen	fem_eligible = (wom_id==1)
bys	hh_id: egen hh_n_fem_eligible = sum(fem_eligible) 	
gen	no_fem_eligible = (hh_n_fem_eligible==0) 									 
lab var no_fem_eligible "hh has no eligible woman"
ta no_fem_eligible, m



count if men_id==1 					// eligible men for cm
count if hl9>0 & hl9!=.				// qc: match between MN & HL datafiles
gen	male_eligible = (men_id==1)
bys	hh_id: egen hh_n_male_eligible = sum(male_eligible)  
gen	no_male_eligible = (hh_n_male_eligible==0) 	 
lab var no_male_eligible "hh has no man eligible for interview"
ta no_male_eligible, m


	

count if child_id==1			// eligible child for nutr
count if hl10>0 & hl10!=.		// qc: match between CH & HL datafiles
gen	child_eligible = (child_id==1) 
bys	hh_id: egen hh_n_children_eligible = sum(child_eligible)  
gen	no_child_eligible = (hh_n_children_eligible==0) 
lab var no_child_eligible "hh has no elig child for anthro"
ta no_child_eligible, m



gen no_eligibles = (no_fem_eligible==1 & no_child_eligible==1)  
lab var no_eligibles "hh has no elig member for nutr"
ta no_eligibles, m


sort hh_id



********************************************************************************
**# 1.9 RELEVANT VARIABLES ***
********************************************************************************

clonevar weight = hhweight 								// sample weight 
lab var weight "sample weight"

*gen psu = hh1 							  				// sample design	
lab var psu "primary sampling unit"

clonevar strata = stratum
lab var strata "sample strata"

svyset psu [pw=weight] , strata(strata)	 singleunit(centered)




codebook hh6, ta (9)									// area: urban-rural		
recode hh6 (1=1 "urban") (2=0 "rural"), gen (area)						 
lab var area "area: urban-rural"
ta hh6 area, m


codebook hl3, ta (99)									     // relationship
recode hl3 (1=1 "head")(2=2 "spouse")(3 13=3 "child") ///
(4/12=4 "extended family")(96=5 "not related")(14=6 "maid") ///
(98=.), gen (relationship)
lab var relationship "relationship to the head of hh"
ta hl3 relationship, m	



codebook hl4 												// sex
recode hl4 (1=1 "male")	(2=2 "female"), gen (sex)						
lab var sex "sex of household member"
ta hl4 sex,m



bys	hh_id: egen missing_hhead = min(relation) 			// headship
ta missing_hhead,m 
gen household_head=.
replace household_head=1 if relation==1 & sex==1 
replace household_head=2 if relation==1 & sex==2
bys hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
lab def head 1"male-headed" 2"female-headed"
lab val headship head
lab var headship "household headship"
ta headship, m



codebook hl6, ta (999) 								// age; age group
clonevar age = hl6  if hl6<98
lab var age "age of household member"


recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+") , gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"



gen member = 1 										// hh size
bys hh_id: egen hhsize = sum(member)
lab var hhsize "household size"
ta hhsize, m
compare hhsize hh48  			// hh48 from datafile matches generated hhsize  
drop member




codebook hh7, ta (99) 							// subnational regions 


recode hh7 (33=1 "Alaotra Mangoro") (22=2 "Amoron'i Mania") ///
(11=3 "Analamanga") (32=4 "Analanjirofo") (52=5 "Androy") (53=6 "Anosy") ///
(51=7 "Atsimo Andrefana") (25=8 "Atsimo Atsinanana") (31=9 "Atsinanana") ///
(43=10 "Betsiboka") (41=11 "Boeny") (14=12 "Bongolava") (61=13 "Diana") ///
(21=14 "Haute Matsiatra") (24=15 "Ihorombe") (13=16 "Itasy") (44=17 "Melaky") ///
(54=18 "Menabe") (62=19 "Sava") (42=20 "Sofia") (12=21 "Vakinankaratra") ///
(23=22 "Vatovavy Fitovinany"), gen (region)


lab var region "subnational region"
codebook region, tab (99)



clonevar region_01 = region 
lab var region_01 "subnational region (hot)"



********************************************************************************
**#  Step 2 Data preparation  ***
***  Standardization of the global MPI indicators 
********************************************************************************


********************************************************************************
**# Step 2.1 Years of Schooling ***
********************************************************************************


codebook ed5a, ta (99)

clonevar edulevel = ed5a 

replace edulevel = . if ed5a==. | ed5a==8 | ed5a==9  
	// note: missing values or out of range are replaced as "."
replace edulevel = 0 if ed4==2 
	// note: those who never attended school are replaced as '0'
replace edulevel = 0 if age<10  
	// non-eligible age group	
lab var edulevel "highest level of educ"


codebook ed5b, ta (99)   	 // qc: check the numeric of missing values 

clonevar eduhighyear = ed5b 
replace eduhighyear = .  if ed5b==. | ed5b==97 | ed5b==98 | ed5b==99 
	// qc: missing values or out of range are replaced as "."
replace eduhighyear = 0 if ed4==2 
replace eduhighyear = 0 if edulevel<1
	// qc: preschool & never attended school - replaced as '0'
replace eduhighyear = 0 if age<10 
	// non-eligible age group
lab var eduhighyear "highest grade"			
			

			
ta eduhighyear edulevel, m

gen	eduyears = eduhighyear

replace eduyears = eduhighyear + 5 if edulevel==2   
	/* note: there are 5 grades in primary school; followed by 4 grades in 
	junior secondary school (JSS). As such we add 5 years to each of 
	the grades completed at the JSS level. */
	
replace eduyears = eduhighyear + 9 if edulevel==3   
	/* note: there are 5 grades in primary school; followed by 4 grades in 
	junior secondary school; and 3 grades in Senior Secondary School (SSS). 
	This means, individuals would have completed 9 years of schooling before 
	reaching SSS. As such we add 9 years to each of the grades completed at 
	the SSS level. */
	
replace eduyears = eduhighyear + 12 if edulevel==4  
	/* note: individuals would have completed 12 years of schooling before 
	reaching post secondary education, or university education. As such we 
	add 12 years to each of the grades completed at the tertiary level. */

replace eduyears = 0 if eduhighyear==. & edulevel==1
	/* note: we assume that an individual has not completed 6 years of 
	education if he or she has attended primary, but no information on grade. 
	This is because in Madagascar, there is only 5 grades in primary level. 
	So even if we assume these individuals have completed primary then they 
	are still under the 6 year cutoff. */ 
	
replace eduyears = 9 if eduhighyear==. & edulevel==3
	/* note: we assume that an individual has completed lower secondary 
	education if he or she has attended upper secondary education, but 
	no information on grade. */ 
	
replace eduyears = 12  if eduhighyear==. & edulevel==4
	/* note: we assume that an individual has completed upper secondary 
	education if he or she has attended tertiery education, but no 
	information on grade. */ 	
	
	
replace eduyears = eduyears - 1 if ed6==2 & eduyears>=1 & eduyears<. 
	/* note: those who did not complete the highest grade attended, 
	we re-assign them to the next lower grade that they would have completed */
	
replace eduyears = . if age<=eduyears & age>0 
	// note: if years of schooling is greater than the age of the individual
	
replace eduyears = 0 if age< 10
replace eduyears = 0 if (age==10 | age==11) & eduyears < 6 
			// non-eligible age group		
lab var eduyears "total years of educ"
ta eduyears, m



gen educ_elig = 1  							// eligibility for educ indicator 
replace educ_elig = 0 if age < 10  
replace educ_elig = 0 if (age==10 | age==11) & eduyears < 6 
lab def lab_educ_elig 0"not eligible" 1"eligible"  
lab val educ_elig lab_educ_elig
lab var educ_elig "eligibility for educ indicator"
ta eduyears educ_elig,m


	/* control variable: information on years of educ 
	present for at least 2/3 of the eligible householders */		
gen temp = 1 if eduyears!=. & educ_elig == 1
bys	hh_id: egen no_mis_edu = sum(temp)		// elig with educ data 
	
bys hh_id: egen hhs = sum(educ_elig == 1) 		// all eligible for educ

replace no_mis_edu = no_mis_edu/hhs
replace no_mis_edu = (no_mis_edu>=2/3)		
ta no_mis_edu, m							// qc: missing (0) is < 0.5% 
lab var no_mis_edu "no missing yos"
drop temp hhs


*** MPI ***
/* Householders are considered not deprived if at least 
one eligible householder has six or more years of education. */
******************************************************************* 
gen	educ6 = (eduyears>=6 & eduyears!=.)
replace educ6 = . if eduyears==.

bys hh_id: egen educ = max(educ6)
replace educ = . if educ==0 & no_mis_edu==0
lab var educ "non-deprivation in education"
ta educ, m 

	
*** Destitution ***
/* Householders are considered not deprived if at least 
one eligible householder has one or more years of education. */
******************************************************************* 
gen	educ1 = (eduyears>=1 & eduyears!=.)
replace educ1 = . if eduyears==.

bys	hh_id: egen educ_u = max(educ1)
replace educ_u = . if educ_u==0 & no_mis_edu==0
lab var educ_u "dst: non-deprivation in education"
ta educ_u,m


********************************************************************************
**# Step 2.2 School Attendance ***
********************************************************************************
	
codebook ed4 ed9, ta (99)


recode ed9 (1=1 "attending") (2=0 "not attending") (9=.), gen (attendance)
replace attendance = 0 if ed4==2 			// never ever attended school	
replace attendance = 0 if age<5 | age>24 	// not of school age	
lab var attendance "current school year"	
ta attendance, m



*** MPI ***
/* Householders are considered not deprived if all 
school-aged children are attending up to class 8. */ 
******************************************************************* 
gen	child_schoolage = (schage>=6 & schage<=14)
lab var child_schoolage "eligible for school attendance"		
	// qc: official school entrance age to primary school: 6 years
	// qc: age range 6-14 (=6+8) 

	
	/* control variable: school attendance data is 
	missing for at least 2/3 of the school-aged children */
count if child_schoolage==1 & attendance==.			// qc: missing satt
gen temp = 1 if child_schoolage==1 & attendance!=. 	// elig children in school 
bys hh_id: egen no_missing_atten = sum(temp)		
gen temp2 = 1 if child_schoolage==1					//elig children
bys hh_id: egen hhs = sum(temp2)					
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)	 		
ta no_missing_atten, m							     // qc: missing < 0.5% 
lab var no_missing_atten "no missing satt"		
drop temp temp2 hhs
		
bys hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
lab var hh_children_schoolage "hh has elig child"

gen	not_atten = (attendance==0) if child_schoolage==1
replace not_atten = . if attendance==. & child_schoolage==1

bysort	hh_id: egen any_not_atten = max(not_atten)

gen	satt = (any_not_atten==0) 
replace satt = . if any_not_atten==.
replace satt = 1 if hh_children_schoolage==0
replace satt = . if satt==1 & no_missing_atten==0 
lab var satt "non-deprivation in school attendance"
ta satt, m

	
*** Destitution ***
/* Householders are considered not deprived if all 
school-aged children are attending up to class 6. */ 
******************************************************************* 
gen	child_schoolage_u = (schage>=6 & schage<=12) 
lab var child_schoolage_u "dst: eligible for school attendance"	

	
	/* control variable: school attendance data is 
	missing for at least 2/3 of the school-aged children */	
count if child_schoolage_u==1 & attendance==.	
gen temp = 1 if child_schoolage_u==1 & attendance!=.
bys hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_u==1	
bys hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)			
ta no_missing_atten_u, m			// qc: missing (0) is < 0.5% 
lab var no_missing_atten_u "no missing satt"		
drop temp temp2 hhs
		
bys	hh_id: egen hh_children_schoolage_u = sum(child_schoolage_u)
replace hh_children_schoolage_u = (hh_children_schoolage_u>0) 
lab var hh_children_schoolage_u "hh has elig child"

gen	atten_6 = (attendance==1) if child_schoolage_u==1
replace atten_6 = . if attendance==. & child_schoolage_u==1

bys	hh_id: egen any_atten_6 = max(atten_6)

gen	satt_u = (any_atten_6==1) 
replace satt_u = . if any_atten_6==.
replace satt_u = 1 if hh_children_schoolage_u==0
replace satt_u = . if satt_u==0 & no_missing_atten_u==0 
lab var satt_u "dst: non-deprivation in school attendance"
ta satt_u, m


********************************************************************************
*** Step 2.3 Nutrition ***
********************************************************************************
 
bys hh_id: egen temp = max(underweight)
gen	hh_no_underweight = (temp==0)		// no child is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1	
lab var hh_no_underweight "hh has no child underweight"
drop temp

bys hh_id: egen temp = max(underweight_u) 			
gen	hh_no_underweight_u = (temp==0) 	// no child is severely underweight 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
lab var hh_no_underweight_u "dst: hh has no child underweight"
drop temp



bys hh_id: egen temp = max(stunting)
gen	hh_no_stunting = (temp==0)					// no child is stunted 
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1	
lab var hh_no_stunting "hh has no child stunted"
drop temp

bys hh_id: egen temp = max(stunting_u) 				
gen	hh_no_stunting_u = (temp==0) 				// no child is severely stunted 
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
lab var hh_no_stunting_u "dst: hh has no child stunted"
drop temp



bys hh_id: egen temp = max(wasting)
gen	hh_no_wasting = (temp==0) 						// no child is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1		
lab var hh_no_wasting "hh has no child wasted"
drop temp

bys hh_id: egen temp = max(wasting_u) 			
gen	hh_no_wasting_u = (temp==0) 				// no child is severely wasted
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
lab var hh_no_wasting_u "dst: hh has no child wasted"
drop temp



gen uw_st = 1 if stunting==1 | underweight==1		// underweight or stunted 
replace uw_st = 0 if stunting==0 & underweight==0
replace uw_st = . if stunting==. & underweight==.
lab var uw_st "child is underweight or stunted"

bys hh_id: egen temp = max(uw_st)
gen	hh_no_uw_st = (temp==0)		
replace hh_no_uw_st = . if temp==.
replace hh_no_uw_st = 1 if no_child_eligible==1	
lab var hh_no_uw_st "hh has no child underweight or stunted"
drop temp

gen uw_st_u = 1 if stunting_u==1 | underweight_u==1 
replace uw_st_u = 0 if stunting_u==0 & underweight_u==0
replace uw_st_u = . if stunting_u==. & underweight_u==.
lab var uw_st_u "dst: child is underweight or stunted"

bys hh_id: egen temp = max(uw_st_u)
gen	hh_no_uw_st_u = (temp==0) 
replace hh_no_uw_st_u = . if temp==.
replace hh_no_uw_st_u = 1 if no_child_eligible==1 
lab var hh_no_uw_st_u "dst: hh has no child underweight or stunted"
drop temp



*** MPI ***
/* Householders are not deprived if all eligible 
person with anthropometric measurement in the household 
are nourished; or household has no eligible person. */
************************************************************************

gen	nutr_0 = 1  						// child under 5
replace nutr_0 = 0 if hh_no_uw_st==0
replace nutr_0 = . if hh_no_uw_st==.
replace nutr_0 = 1 if no_child_eligible==1   		
lab var nutr_0 "non-deprivation in nutr (under 5)"
ta nutr_0, m



gen	nutr_70 = 1  						// child under 5, only stunting
replace nutr_70 = 0 if hh_no_stunting==0
replace nutr_70 = . if hh_no_stunting==.
replace nutr_70 = 1 if no_child_eligible==1   		
lab var nutr_70 "non-deprivation in stunting (under 5)"
ta nutr_70, m



*** Destitution ***
/* Householders are not deprived if all eligible person with
anthropometric measurement in the household are not severely 
undernourished; or household has no eligible person. */
************************************************************************

gen	nutr_0_u = 1  								// child under 5
replace nutr_0_u = 0 if hh_no_uw_st_u==0
replace nutr_0_u = . if hh_no_uw_st_u==.
replace nutr_0_u = 1 if no_child_eligible==1   	
lab var nutr_0_u "dst: non-deprivation in nutr (under 5)"
ta nutr_0_u, m



gen	nutr_70_u = 1  							// child under 5, only stunting
replace nutr_70_u = 0 if hh_no_stunting_u==0
replace nutr_70_u = . if hh_no_stunting_u==.
replace nutr_70_u = 1 if no_child_eligible==1   	
lab var nutr_70_u "dst: non-deprivation in stunting (under 5)"
ta nutr_70_u, m



********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************

codebook cm9 cm10
	/*cm9: number of sons who have died 
	  cm10: number of daughters who have died */

	  	  
ta cm1 cm8, m
ta childu18_died_per_wom_5y if cm8==1 & cm1==2,m
	/* women who reported child mortality but identified 
	 no birth; we use this data. */
			  
egen temp_f = rowtotal(cm9 cm10), missing
replace temp_f = 0 if (cm1==1 & cm8==2) | (cm1==2 & cm8!=1) // never given birth
bys	hh_id: egen child_mortality_f = sum(temp_f), missing
lab var child_mortality_f "child mortality reported by women"
ta child_mortality_f, m
drop temp_f	  
	  
	  
egen child_mortality = rowmax(child_mortality_f)
replace child_mortality = 0 if child_mortality==. & no_fem_eligible==1 
lab var child_mortality "total child mortality in a hh"
ta child_mortality, m	 


*** MPI *** 
/* Householders are not deprived if all eligible women 
in the household reported zero mortality among children 
under 18 in the last 5 years from the survey year.*/
************************************************************************
ta childu18_died_per_wom_5y, m	
	
replace childu18_died_per_wom_5y = 0 if cm1==2 & cm8==2 
	// elig woman who never ever gave birth 
	
replace childu18_died_per_wom_5y = 0 if no_fem_eligible==1	
	// no elig woman
	
bys hh_id: egen childu18_mortality_5y = sum(childu18_died_per_wom_5y), missing
replace childu18_mortality_5y = 0 if childu18_mortality_5y==. & child_mortality==0
	// women has missing value and men reported no death 
lab var childu18_mortality_5y "total u18 child mortality last 5 years"
ta childu18_mortality_5y, m		
	
gen cm_0 = (childu18_mortality_5y==0)
replace cm_0 = . if childu18_mortality_5y==.
lab var cm_0 "non-deprivation in cm"
ta cm_0, m


*** Destitution *** 
*** (same as MPI) ***
************************************************************************
clonevar cm_0_u = cm_0	
lab var cm_0_u "dst: non-deprivation in cm"	


				
********************************************************************************
**# Step 2.5 Electricity 
********************************************************************************

*** MPI ***
/* Householders are considered not deprived 
if the household has electricity */
****************************************
codebook hc8, ta (9)
recode hc8  (1/2 = 1 "yes") (3 = 0 "no") (9=.), gen(elct)	 
lab var elct "non-deprivation in electricity"
ta hc8 elct,m

svy: prop elct if id==1				// qc: matches the report 


*** Destitution ***
*** (same as MPI) ***
****************************************
gen elct_u = elct
lab var elct_u "dst: non-deprivation in electricity"


	
********************************************************************************
**# Step 2.6 Sanitation 
********************************************************************************

	
*** MPI ***
/* Householders are not deprived if the household has improved 
sanitation facilities that are not shared with other households. */
********************************************************************
desc ws11 ws15

codebook ws11, ta(99) 	
	
recode ws11 (11/13 18/22 31 = 1 "yes") ///
(14 23 24 41/96 = 0 "no") (99=.), gen(sani)


codebook ws15

replace sani = 0 if ws15==1				
lab var sani "non-deprivation in sanitation"
ta ws11 sani, m


svy: prop sani 				
	
	
	
*** Destitution ***
/* Householders are not deprived if the 
household has sanitation facilities. */
********************************************************************
recode ws11 (11/51 = 1 "yes") ///
(95 96 = 0 "no") (99=.), gen(sani_u)

lab var sani_u "dst: non-deprivation in sanitation"
ta ws11 sani_u, m

	
	
********************************************************************************
**# Step 2.7 Drinking Water 
********************************************************************************


*** MPI ***
/* Householders are not deprived if household have access to safe 
drinking water and is under 30 minutes walk from home, round trip.*/
********************************************************************
desc ws1 ws4 

codebook ws1, ta (99)


recode ws1 (11/31 41 51/71 91 = 1 "yes") ///
(32 42 81 96 = 0 "no") (99=.), gen(dwtr_src)
lab var dwtr_src "improved main source of drinking water"
ta ws1 dwtr_src



codebook ws4, ta (999)							// time to water

clonevar wtr = dwtr_src
replace wtr = 0 if ws4 >=30 & ws4 <900							 
lab var wtr "non-deprivation in drinking water"
ta dwtr_src wtr,m

svy: prop wtr

	
*** Destitution ***
/* Householders are not deprived if household have access to safe 
drinking water and is 45 minutes walk or less from home, round trip.*/
********************************************************************
clonevar wtr_u = dwtr_src						   
replace wtr_u = 0 if ws4 >45 & ws4 <900				  
lab var wtr_u "dst: non-deprivation in drinking water"
ta dwtr_src wtr_u,m



********************************************************************************
**# Step 2.8 Housing
********************************************************************************

desc hc4 hc6 hc5

codebook hc4, ta (99)				// improved = rudimentary & finished floor 
recode hc4 (21/35 = 1 "yes") (11 12 96 = 0 "no") (99=.), gen(floor)
lab var floor "hh has improved floor"
ta hc4 floor,m
svy: prop floor if id==1						

	
codebook hc6, ta (99)							// improved = finished walls 
recode hc6 (31/38 = 1 "yes") (11/27 96 = 0 "no") (99=.), gen(wall)
lab var wall "hh has improved wall"
ta hc6 wall,m
svy: prop wall if id==1			 				

	
codebook hc5, ta (99)							// improved = finished roofing 
recode hc5 (31/36 = 1 "yes") (11/25 96 = 0 "no") (99=.), gen(roof)
lab var roof "hh has improved roof"
ta hc5 roof,m	
svy: prop roof if id==1							
	

	
*** MPI ***
/* Householders are not deprived in housing if the roof, 
floor and walls are constructed from quality materials.*/
**************************************************************
gen hsg = 1
replace hsg = 0 if floor==0 | wall==0 | roof==0
replace hsg = . if floor==. & wall==. & roof==.
lab var hsg "non-deprivation in housing"
ta hsg, m


*** Destitution ***
/* Householders are not deprived in housing if at least two of the three 
components (roof/floor/walls) are constructed from quality materials. */
**************************************************************
gen hsg_u = 1

replace hsg_u = 0 if ///
(floor==0 & wall==0 & roof==1) | ///
(floor==0 & wall==1 & roof==0) | ///
(floor==1 & wall==0 & roof==0) | ///
(floor==0 & wall==0 & roof==0)

replace hsg_u = . if floor==. & wall==. & roof==.
lab var hsg_u "dst: non-deprivation in housing"
ta hsg_u, m
	

	
********************************************************************************
**# Step 2.9 Cooking Fuel 
********************************************************************************

	
*** MPI ***
/* Householders are considered not deprived if the 
household uses non-solid fuels for cooking. */
*****************************************************************

codebook eu1 eu4, ta(99)

ta eu1 eu4, m	

gen ckfl = 1	// solid fuel: coal, charcoal, wood, straw/grass/crop, dung 
	
replace ckfl = 0 if eu4>=4 & eu4<=11	
replace ckfl = 0 if eu1==96 & eu4==96	
replace ckfl = 0 if (eu1==7 & eu4==96) | (eu1==9 & eu4==96)
replace ckfl = . if eu4==99 & eu1==99

lab var ckfl "non-deprivation in cooking fuel"	

svy: prop ckfl


*** Destitution  ***
*** (same as MPI) ***
****************************************	
clonevar ckfl_u = ckfl 
lab var ckfl_u "dst: non-deprivation in cooking fuel"



********************************************************************************
**# Step 2.10 Assets
********************************************************************************


	// handphone/telephone/iphone/mobilephone/ipod
lookfor telephone téléphone mobilephone ipod telefone tele celular
codebook hc7a hc12
gen telephone = (hc7a==1 | hc12==1)
lab var telephone "hh has telephone"			
ta hc7a hc12 if telephone==1,m



		// radio/walkman/stereo/kindle
lookfor radio walkman stereo stéréo
codebook hc7b
clonevar radio = hc7b 
lab var radio "hh has radio"


	// television/lcd tv/plasma tv/color tv/black & white tv
lookfor tv television plasma lcd télé tele
codebook hc9a
clonevar television = hc9a 
lab var television "hh has television"
ta elct television, m		// assume no television if no electricity
replace television=0 if elct==0 & television==.	




	// refrigerator/icebox/fridge
lookfor refrigerator réfrigérateur refri
codebook hc9b
clonevar refrigerator = hc9b
lab var refrigerator "hh has refrigerator"
ta elct refrigerator, m				// assume no television if no electricity
replace refrigerator=0 if elct==0 & refrigerator==.	



	// bicycle/cycle rickshaw
lookfor bicycle bicyclette bicicleta
codebook hc10b
clonevar bicycle = hc10b
lab var bicycle "hh has bicycle"		
	
	
	
	// motorbike/motorized bike/autorickshaw
lookfor motorbike moto
codebook hc10c	
clonevar motorbike = hc10c
lab var motorbike "hh has motorbike"
	
	
	
	// animal cart
lookfor brouette cart carro carreta
codebook hc10d
clonevar animal_cart = hc10d
lab var animal_cart "hh has animal cart"	




	// car/van/lorry/truck
lookfor car van truck
codebook hc10e	
clonevar car = hc10e  
lab var car "hh has car"		
	
	
	
	// computer/laptop/tablet
lookfor computer ordinateur laptop ipad tablet 
codebook hc11
clonevar computer = hc11
lab var computer "hh has computer"


 
lab def lab_asst 0"no" 1"yes"
foreach var in television radio telephone refrigerator car ///
			   bicycle motorbike computer animal_cart {
replace `var' = 0 if `var'==2 
lab val `var' lab_asst			   
replace `var' = . if `var'==9 | `var'==99 | `var'==8 | `var'==98 
}

		
*** MPI ***
/* Householders are considered not deprived in assets 
if the household own more than one of the assets. */
*****************************************************************************
egen n_small_asst = rowtotal (television radio telephone refrigerator ///
bicycle motorbike computer animal_cart), m
lab var n_small_asst "small assets owned by hh" 
 
 
gen asst = (car==1 | n_small_asst > 1) 
replace asst = . if car==. & n_small_asst==.
lab var asst "non-deprivation in assets"


												// harmonised asset, exc. comp
egen n_small_asst_70 = rowtotal (television radio telephone ///
refrigerator bicycle motorbike animal_cart), m
lab var n_small_asst_70 "small assets owned by hh (x comp)" 
   
   
gen asst_70 = (car==1 | n_small_asst_70 > 1) 
replace asst_70 = . if car==. & n_small_asst_70==.
lab var asst_70 "non-deprivation in assets (x comp)"



*** Destitution ***
/* Householders are considered not deprived in assets 
if the household own at least one asset. */
*****************************************************************************	
gen	asst_u = (car==1 | n_small_asst>0)
replace asst_u = . if car==. & n_small_asst==.
lab var asst_u "dst: non-deprivation in assets"



gen	asst_70_u = (car==1 | n_small_asst_70>0)  
replace asst_70_u = . if car==. & n_small_asst_70==.
lab var asst_70_u "dst: non-deprivation in assets (x comp)"


********************************************************************************
**# Step 2.11 MPI indicators
********************************************************************************

desc hh5y hh5m hh5d 							// interview dates
clonevar year_interview = hh5y 	
clonevar month_interview = hh5m 
clonevar date_interview = hh5d 



recode cm_0   		(0=1)(1=0) , gen(d_cm)					// for MPI est 
recode nutr_0 		(0=1)(1=0) , gen(d_nutr)
recode satt 		(0=1)(1=0) , gen(d_satt)
recode educ 		(0=1)(1=0) , gen(d_educ)
recode elct			(0=1)(1=0) , gen(d_elct)
recode wtr 			(0=1)(1=0) , gen(d_wtr)
recode sani			(0=1)(1=0) , gen(d_sani)
recode hsg 			(0=1)(1=0) , gen(d_hsg)
recode ckfl 		(0=1)(1=0) , gen(d_ckfl)
recode asst 		(0=1)(1=0) , gen(d_asst)


recode cm_0_u   	(0=1)(1=0) , gen(dst_cm)				// for dst est
recode nutr_0_u 	(0=1)(1=0) , gen(dst_nutr)
recode satt_u 		(0=1)(1=0) , gen(dst_satt)
recode educ_u 		(0=1)(1=0) , gen(dst_educ)
recode elct_u 		(0=1)(1=0) , gen(dst_elct)
recode wtr_u  		(0=1)(1=0) , gen(dst_wtr)
recode sani_u 		(0=1)(1=0) , gen(dst_sani)
recode hsg_u  		(0=1)(1=0) , gen(dst_hsg)
recode ckfl_u 		(0=1)(1=0) , gen(dst_ckfl)
recode asst_u 		(0=1)(1=0) , gen(dst_asst) 


recode cm_0   		(0=1)(1=0) , gen(d_cm_01)	 		// for hot MPI est 
recode nutr_70 		(0=1)(1=0) , gen(d_nutr_01)
recode satt 		(0=1)(1=0) , gen(d_satt_01)
recode educ 		(0=1)(1=0) , gen(d_educ_01)
recode elct			(0=1)(1=0) , gen(d_elct_01)
recode wtr 			(0=1)(1=0) , gen(d_wtr_01)
recode sani			(0=1)(1=0) , gen(d_sani_01)
recode hsg 			(0=1)(1=0) , gen(d_hsg_01)
recode ckfl 		(0=1)(1=0) , gen(d_ckfl_01)
recode asst_70 		(0=1)(1=0) , gen(d_asst_01)
 

recode cm_0_u   	(0=1)(1=0) , gen(dst_cm_01)			// for hot dst est
recode nutr_70_u	(0=1)(1=0) , gen(dst_nutr_01)
recode satt_u 		(0=1)(1=0) , gen(dst_satt_01)
recode educ_u 		(0=1)(1=0) , gen(dst_educ_01)
recode elct_u 		(0=1)(1=0) , gen(dst_elct_01)
recode wtr_u  		(0=1)(1=0) , gen(dst_wtr_01)
recode sani_u 		(0=1)(1=0) , gen(dst_sani_01)
recode hsg_u  		(0=1)(1=0) , gen(dst_hsg_01)
recode ckfl_u 		(0=1)(1=0) , gen(dst_ckfl_01)
recode asst_70_u 	(0=1)(1=0) , gen(dst_asst_01) 



lab def lab_dp 1"yes" 0"no"
foreach var in d_* dst_* d_*_* dst_*_* {
lab val `var' lab_dp
}

foreach var in cm nutr satt educ elct wtr sani hsg ckfl asst {
lab var d_`var' "deprived in `var'"
lab var dst_`var' "deprived in `var' (dst)"
lab var d_`var'_01 "deprived in `var' (hot)"
lab var dst_`var'_01 "deprived in `var' (dst-hot)"
}


keep hh_id ind_id strata psu weight sex age area ///
agec7 agec4 agec2 region region_* headship d_* dst_* 

	 
order hh_id ind_id strata psu weight sex age area ///
agec7 agec4 agec2 region region_* headship d_* dst_* 


mdesc psu strata area age headship region region_* d_* dst_*




char _dta[cty] "Madagascar"
char _dta[ccty] "MDG"
char _dta[year] "2018" 	
char _dta[survey] "MICS"
char _dta[ccnum] "450"
char _dta[type] "micro"


sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/mdg_mics18.dta", replace 


