********************************************************************************
/*
Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
Global Multidimensional Poverty Index - Morocco PAPFAM 2017-2018 
[STATA do-file]. Available from OPHI website: http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk
*/
********************************************************************************
 
clear all 
set more off
set maxvar 10000


*** Working Folder Path ***
global path_in "../rdta/Morocco PAPFAM 2017-18" 
global path_out "cdta"
global path_ado "ado"

	
********************************************************************************
*** MOROCCO PAPFAM 2017-2018 ***
********************************************************************************


********************************************************************************
*** Step 1: Data preparation 
********************************************************************************

	/*
	Note on microdata:
	The Moroccan National Survey on Population and Family Health (PAPFAM) 2017-
	2018 microdata is not available on the public platform. We obtained the 
	microdata with permission from the Haut Commissariat au Plan – National 
	Office of Statistic in Morocco and the Ministry of Health in Morocco 
	solely for computing and publishing the global MPI aggregates.
	
	Note on nutrition:
	Anthropometric data was collected from children under 5 years.  
	*/

********************************************************************************
*** Step 1.1 CHILDREN's RECODE 
********************************************************************************

use "$path_in/CH.dta", clear 

rename _all, lower	
	
			
*** Generate individual unique key variable required for data merging
*** district=Census district number; 
*** hh_n=household number; 
*** q501=line number of eligible child
gen double ind_id = district*1000000 + hh_n*100 + q501

duplicates  report ind_id
	//2 duplicate observations
sort ind_id
quietly by ind_id: gen dup = cond(_N==1,0,_n)
tab dup
*br district hh_n q501 milieu respondent ind_id q503 q504 q505 resultat q507 if dup>0
drop ind_id

clonevar ln = q501
replace ln=11 if district==717 & hh_n==3 & q501==10 & q504==4.2

gen double ind_id = district*1000000 + hh_n*100 + ln
format ind_id %20.0g
label var  ind_id "Individual ID"

duplicates report ind_id
	

gen child_CH=1 
	//Generate identification variable for observations in child recode

tab q507, miss
count
	//6,668 children aged 0-60 months

*** Check the variables that WHO ado needs to calculate the z-scores:
*** sex, age, weight, height, measurement, oedema & child sampling weight


*** Variable: SEX ***
codebook q102,tab(30)	
	//"1" for male ;"2" for female
tab q102, nol
clonevar gender = q102
desc gender
tab gender


*** Variable: AGE ***
gen idate = ym(yy1, mm1)
format idate %tm

gen bdate = ym(annee, mois)
format bdate %tm

sum idate bdate

gen age_months = (idate-bdate)
sum age_months

gen  str6 ageunit = "months" 
lab var ageunit "Months"


*** Variable: BODY WEIGHT (KILOGRAMS) ***
desc q504
codebook q504, tab (999)   
clonevar weight = q504	
tab	q507 q504 if q504==., miss 
	//q507: Result of child measurement 
sum weight


*** Variable: HEIGHT (CENTIMETERS) 
desc q505
codebook q505, tab (999)  
clonevar height = q505 
tab	q507 q505 if q505==., miss
sum height


*** Variable: MEASURED STANDING/LYING DOWN ***
desc q506
codebook q506, tab (9)
gen measure = "l" if q506==1 
	//Child measured lying down
replace measure = "h" if q506==2 
	//Child measured standing up
replace measure = " " if q506==. 
	//Replace with " " if unknown
tab measure
	
	
*** Variable: OEDEMA ***
gen  oedema = "n"  
	//It assumes no-one has oedema
	

*** Variable: SAMPLING WEIGHT ***
	/* We don't require individual weight to compute the z-scores of a child. 
	So we assume all children in the sample have the same weight. */
gen sw = 1	
sum sw


*** Indicate to STATA where the igrowup_restricted.ado file is stored:
	***Source of ado file: http://www.who.int/childgrowth/software/en/
adopath + "$path_ado/igrowup_stata"

*** We will now proceed to create three nutritional variables: 
	*** weight-for-age (underweight),  
	*** weight-for-height (wasting) 
	*** height-for-age (stunting)

*** We specify the first three parameters we need in order to use the ado file:
	*** reflib, 
	*** datalib, 
	*** datalab

/* We use 'reflib' to specify the package directory where the .dta files 
containing the WHO Child Growth Standards are stored. */	
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"


/* We use datalib to specify the working directory where the input STATA 
dataset containing the anthropometric measurement is stored. */
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"


/* We use datalab to specify the name that will prefix the output files that 
will be produced from using this ado file (datalab_z_r_rc and datalab_prev_rc)*/
gen str30 datalab = "children_nutri_mar"
lab var datalab "Working file"

	
/*We now run the command to calculate the z-scores with the adofile */
igrowup_restricted reflib datalib datalab gender age_months ageunit weight ///
height measure oedema sw


/*We now turn to using the dta file that was created and that contains 
the calculated z-scores to create the child nutrition variables following WHO 
standards */
use "$path_out/children_nutri_mar_z_rc.dta", clear 

	
*** Standard MPI indicator ***
	//Takes value 1 if the child is under 2 stdev below the median & 0 otherwise	
gen	underweight = (_zwei < -2.0) 
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "Child is undernourished (weight-for-age) 2sd - WHO"
tab underweight [aw=normalizedhhweight], miss  


gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "Child is stunted (length/height-for-age) 2sd - WHO"
tab stunting [aw=normalizedhhweight], miss 


gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "Child is wasted (weight-for-length/height) 2sd - WHO"
tab wasting [aw=normalizedhhweight], miss 


count if _fwei==1 | _flen==1
	/*Morocco PAPFAM 2017-2018: 53 children are replaced as '.' because 
	they have extreme z-scores which are biologically implausible. */

	
*** Destitution indicator  ***
	//Takes value 1 if the child is under 3 stdev below the median & 0 otherwise	
gen	underweight_u = (_zwei < -3.0) 
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "Child is undernourished (weight-for-age) 3sd - WHO"


gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "Child is stunted (length/height-for-age) 3sd - WHO"


gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "Child is wasted (weight-for-length/height) 3sd - WHO"
 

clonevar weight_ch = normalizedhhweight
label var weight_ch "sample weight child under 5"


rename resultat iwrslt_CH
	
 
	//Retain relevant variables:
keep  ind_id child_CH weight_ch underweight* stunting* wasting* iwrslt_CH  
order ind_id child_CH weight_ch underweight* stunting* wasting* iwrslt_CH  
sort  ind_id
save "$path_out/MAR17-18_CH.dta", replace


	//Erase files from folder:
erase "$path_out/children_nutri_mar_z_rc.xls"
erase "$path_out/children_nutri_mar_prev_rc.xls"
erase "$path_out/children_nutri_mar_z_rc.dta"


********************************************************************************
*** Step 1.2 BIRTH RECODE 
*** (All females 193-599 age in months who ever gave birth)  
********************************************************************************
/*The purpose of step 1.2 is to identify children of any age who died in 
the last 5 years prior to the survey date.*/


use "$path_in/BH.dta", clear

rename _all, lower	

		
*** Generate individual unique key variable required for data merging
*** district=Census district number;  
*** hh_n=household number; 
*** wmn_n=respondent's line number
gen double ind_id = district*1000000 + hh_n*100 + wmn_n 
format ind_id %20.0g
label var ind_id "Individual ID"

gen w_bd = ym(w102_y, w102_m)
format w_bd %tm
sum w_bd

gen wdoi = ym(w_yy, w_mm)
format wdoi %tm
sum wdoi

gen w_age_months = (wdoi-w_bd)
sum w_age_months
	//6,330 women 16-50 years (193-599 age in months) provided birth history data 
		
gen bh4c = ym(w216_y, w216_m)
format bh4c %tm
sum bh4c 
	

codebook w221u, tab (9)
tab w221n w221u if w217==2,m
	//125 children who died and with age at death
gen bh9c = w221n if w221u==2
	//dates rpeorted in months are retained
replace bh9c = w221n*12 if w221u==3
	//dates reported in years are converted to months
replace bh9c = 0 if w221u==1
	//replace 0 months for children who died within 30 days
sum bh9c
	//Age in months at death
	
 
gen date_death = bh4c + bh9c	
	//Date of death = date of birth (bh4c) + age at death (bh9c)
gen mdead_survey = wdoi-date_death	
	//Months dead from survey = Date of interview (wdoi) - date of death	
gen ydead_survey = mdead_survey/12
	//Years dead from survey
sum ydead_survey
	

gen age_death = bh9c if w217==2
label var age_death "Age at death in months"	
tab age_death, miss	
	

codebook w217, tab (9)	
gen child_died = 1 if w217==2
replace child_died = 0 if w217==1
replace child_died = . if w217==.
label define lab_died 0"child is alive" 1"child has died"
label values child_died lab_died
tab w217 child_died, miss

	
bysort ind_id: egen tot_child_died = sum(child_died) 
	//For each woman, sum the number of children who died
		
	
	//Identify child under 18 mortality in the last 5 years
gen child18_died = child_died 
replace child18_died=0 if age_death>=216 & age_death<.
label define lab_u18died 1 "child u18 has died" 0 "child is alive/died but older"
label values child18_died lab_u18died
tab child18_died, miss	
	
bysort ind_id: egen tot_child18_died_5y=sum(child18_died) if ydead_survey<=5
	/*Total number of children under 18 who died in the past 5 years 
	prior to the interview date */	
	
replace tot_child18_died_5y=0 if tot_child18_died_5y==. & tot_child_died>=0 & tot_child_died<.
	/*All children who are alive or who died longer than 5 years from the 
	interview date are replaced as '0'*/
	
replace tot_child18_died_5y=. if child18_died==1 & ydead_survey==.
	//Replace as '.' if there is no information on when the child died  

tab tot_child_died tot_child18_died_5y, miss

bysort ind_id: egen childu18_died_per_wom_5y = max(tot_child18_died_5y)
lab var childu18_died_per_wom_5y "Total child under 18 death for each women in the last 5 years (birth recode)"
	

	//Keep one observation per women
bysort ind_id: gen id=1 if _n==1
keep if id==1
drop id
duplicates report ind_id 


gen women_BH = 1 
	//Identification variable for observations in BH recode


rename resultat iwrslt_BH
	
	
	//Retain relevant variables
keep ind_id women_BH childu18_died_per_wom_5y iwrslt_BH
order ind_id women_BH childu18_died_per_wom_5y iwrslt_BH		
sort ind_id
save "$path_out/MAR17-18_BH.dta", replace		
	

********************************************************************************
*** Step 1.3  WOMEN's RECODE  
*** (All eligible ever-married females in the household)
********************************************************************************
	/*The usual variables retained through the WOM data file is available 
	in the HR recode. */
	
use "$path_in/WOM.dta", clear

rename _all, lower	

		
*** Generate individual unique key variable required for data merging
*** district=Census district number;  
*** hh_n=household number; 
*** wmn_n=respondent's line number
gen double ind_id = district*1000000 + hh_n*100 + wmn_n 
format ind_id %20.0g
label var ind_id "Individual ID"

duplicates report ind_id 

gen women_WM=1 
	//Identification variable for observations in WM recode
	
	
gen w_bd = ym(w102_y, w102_m)
format w_bd %tm
sum w_bd

gen wdoi = ym(w_yy, w_mm)
format wdoi %tm
sum wdoi

gen w_age_months = (wdoi-w_bd)
sum w_age_months
	//9,967 women 16-50 years (189-600 age in months)  	

tab w118 w_rslt, miss
	/*Morocco PAPFAM 2017-2018: 9,967 ever married women 16-50 years were 
	interviewed.*/
	
tab w201 w206, miss	
	/*Morocco PAPFAM 2017-2018: 6 women report never having given birth but has 
	given birth to a boy or girl who was born alive but later died. We use the 
	child mortality information provided by these individuals in section 2.4.*/

tab w118, miss	
	//Child mortality information collected only from ever married women.

//Retain relevant variables:		
gen religion_wom = .
lab var religion_wom "Women's religion"	

gen ethnic_wom = .
lab var ethnic_wom "Women's ethnicity"	

gen insurance_wom = .
label var insurance_wom "Women have health insurance"		
	
	
keep w118 w201 w202 w204 w206 w207_1 w207_2 w222_b ind_id women_WM *_wom
order w118 w201 w202 w204 w206 w207_1 w207_2 w222_b ind_id women_WM *_wom
sort ind_id
save "$path_out/MAR17-18_WM.dta", replace	
	

********************************************************************************
*** Step 1.4  MN - MEN'S RECODE 
***(Eligible man: 15-59 years in the household) 
********************************************************************************
	//Morocco PAPFAM 2017-2018: No male recode datafile available.
	
	
********************************************************************************
*** Step 1.5 HH - Household's recode ***
********************************************************************************

use "$path_in/HH.dta", clear 


rename _all, lower


*** Generate individual unique key variable required for data merging
*** cluster=cluster number;  
*** hhnum=household number; 
gen	double hh_id = district*100 + hh_n 
format	hh_id %20.0g
lab var hh_id "Household ID"


save "$path_out/MAR17-18_HH.dta", replace


********************************************************************************
*** Step 1.6 HR - Household Member's recode ****
********************************************************************************

use "$path_in/HR.dta", clear 


rename _all, lower


*** Generate a household unique key variable at the household level using: 
	***cluster=cluster number 
	***hhnum=household number
gen double hh_id = district*100 + hh_n
format hh_id %20.0g
label var hh_id "Household ID"


*** Generate individual unique key variable required for data merging using:
	*** cluster=cluster number; 
	*** hhnum=household number; 
	***	hln=respondent's line number.
gen double ind_id = district*1000000 + hh_n*100 + q100 
format ind_id %20.0g
label var ind_id "Individual ID"


********************************************************************************
*** Step 1.7 DATA MERGING 
******************************************************************************** 


*** Merging BR Recode 
*****************************************
merge 1:1 ind_id using "$path_out/MAR17-18_BH.dta"
tab iwrslt_BH _merge,m
tab childu18_died_per_wom_5y _merge,m
drop if _merge==2
	//7 observations dropped
drop _merge
erase "$path_out/MAR17-18_BH.dta" 


*** Merging WM Recode 
*****************************************
merge 1:1 ind_id using "$path_out/MAR17-18_WM.dta"
drop if _merge==2
	//13 observations dropped
drop _merge
erase "$path_out/MAR17-18_WM.dta" 


*** Merging HH Recode 
*****************************************
merge m:1 hh_id using "$path_out/MAR17-18_HH.dta"
drop _merge
erase "$path_out/MAR17-18_HH.dta"


*** Merging MR Recode 
*****************************************
	//Morocco PAPFAM 2018 has no male recode file

gen religion_men = . 
lab var religion_men "Men's religion"	

gen ethnic_men = .
lab var ethnic_men "Men's ethnicity"

gen insurance_men = .
label var insurance_men "Men have health insurance"	


*** Merging children 0-60 months
*****************************************
merge 1:m ind_id using "$path_out/MAR17-18_CH.dta"	
tab iwrslt_CH _merge,m
tab stunting _merge,m
drop if _merge==2
drop _merge
erase "$path_out/MAR17-18_CH.dta"


sort ind_id


********************************************************************************
*** Step 1.8 CONTROL VARIABLES
********************************************************************************

/*Households are identified as having 'no eligible' members if there are no 
applicable population, that is, children under 5 years, adult women or men. These 
households will not have information on relevant indicators of health. As such, 
these households are considered as non-deprived in those relevant indicators.*/

*** No eligible women 15-49 years 
*** for child mortality indicator
*****************************************
count if q102==2 & q105>=15 & q105<=49 & q106>=2 & q106<=5
	//Ever-married women 15-49 years: 10,412
count if q107!=.
tab w206 if women_BH==. & women_WM==1,m
tab w201 if women_BH==. & women_WM==1,m
gen	fem_eligible = (q107!=.) 
bys	hh_id: egen hh_n_fem_eligible = sum(fem_eligible) 	
	//Number of eligible women for interview in the hh
gen	no_fem_eligible = (hh_n_fem_eligible==0) 									
	//Takes value 1 if the household had no eligible females for an interview
tab no_fem_eligible, miss	
lab var no_fem_eligible "Household has no eligible women"


*** No Eligible Men 
*****************************************
	//Morocco PAPFAM 2017-18 have no male recode file 
gen male_eligible = .	
gen no_male_eligible = . 
lab var no_male_eligible "Household has no eligible man"
tab no_male_eligible, miss

	
*** No eligible children under 5
*** for child nutrition indicator
*****************************************
count if q108<. & q105<5
count if child_CH==1
gen	child_eligible = (q108<. & q105<5) | child_CH==1 
bysort hh_id: egen hh_n_children_eligible = sum(child_eligible)  
	//Number of eligible children for anthropometrics
gen	no_child_eligible = (hh_n_children_eligible==0) 
	//Takes value 1 if there were no eligible children for anthropometrics
tab no_child_eligible,miss	
lab var no_child_eligible "Household has no children eligible"	


sort hh_id ind_id


********************************************************************************
*** Step 1.10 RENAMING DEMOGRAPHIC VARIABLES ***
********************************************************************************


//Sample weight
clonevar weight = weighthh
label var weight "Sample weight"


//Area: urban or rural	
codebook milieu, tab (5)		
clonevar area = milieu 
replace area=0 if area==2  
label define lab_area 1 "urban" 0 "rural"
label values area lab_area
label var area "Area: urban-rural"


//Relationship to the head of household 
clonevar relationship = q103 
codebook relationship, tab (20)
recode relationship (1=1)(2=2)(3 13=3)(4/12=4)(15=5)(14=6)(98=.)
label define lab_rel 1"head" 2"spouse" 3"child" 4"extended family" 5"not related" 6"maid"
label values relationship lab_rel
label var relationship "Relationship to the head of household"
tab q103 relationship, miss


//Sex of household member
codebook q102 
clonevar sex = q102 
label var sex "Sex of household member"


//Household headship
bys	hh_id: egen missing_hhead = min(relation)
tab missing_hhead,m 
gen household_head=.
replace household_head=1 if relation==1 & sex==1 
replace household_head=2 if relation==1 & sex==2
bysort hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
replace headship = . if headship>2
label define head 1"male-headed" 2"female-headed"
label values headship head
label var headship "Household headship"
tab headship, miss


//Age of household member
codebook q105, tab (999)  
clonevar age = q105  
label var age "Age of household member"


//Age group 
recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+"), gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"


//Marital status of household member
desc q106
clonevar marital = q106
codebook marital, tab (10)
label define lab_mar 1"never married" 2"currently married" 3"widowed" ///
4"divorced" 5"not living together"
label values marital lab_mar	
label var marital "Marital status of household member"
tab q106 marital, miss


//Total number of de jure hh members in the household
gen member = 1
bysort hh_id: egen hhsize = sum(member)
label var hhsize "Household size"
tab hhsize, miss
drop member


//Religion of the household head
gen religion_hh = .
label var religion_hh "Religion of household head"


//Ethnicity of the household head
gen ethnic_hh = .
label var ethnic_hh "Ethnicity of household head"


//Subnational region
codebook region , tab (99)
decode region, gen(temp)
replace temp =  proper(temp)
*encode temp, gen(region)
lab var region "Region for subnational decomposition"
codebook region, tab (100)
tab region, miss 
drop temp
label define lab_reg ///
1 "Tanger-Tetouan-Al Hoceima" ///
2 "Oriental" ///
3 "Fès-Meknès" ///
4 "Rabat-Salé-Kénitra" ///
5 "Béni Mellal-Khénifra" ///
6 "Casablanca- Settat" ///
7 "Marrakech-Safi" ///
8 "Drâa-Tafilalet" ///
9 "Souss-Massa" ///
10 "Guelmim-Oued Noun" ///
11 "Laâyoune-Sakia El Hamra" ///
12 "Ed Dakhla-Oued ed Dahab"
label values region lab_reg

	
********************************************************************************
***  Step 2 Data preparation  ***
***  Standardization of the 10 Global MPI indicators 
***  Identification of non-deprived & deprived individuals  
********************************************************************************


********************************************************************************
*** Step 2.1 Years of Schooling ***
********************************************************************************

	/*PAPFAM does not provide the number of years of education so we need to 
	construct that variable from the edulevel and eduhighyear variables */ 
	
codebook q112 q113, tab (999)
tab q113 q112, m
gen	edulevel = q112 
	//Highest educational level attended
replace edulevel = 0 if q111 == 3  
	//Never attended school
label define lab_edulevel 0 "None" 1 "Primary"  2 "College" ///
						  3 "Secondary" 4"Superior" 
label values edulevel lab_edulevel
label var  edulevel "Highest educational level attended"


clonevar eduhighyear = q113
	//Highest grade attended at that level
replace eduhighyear = . if q113>30
	//Recode any unreasonable years of highest education as missing value
replace eduhighyear = 0  if q111==3
	//Those who never attended school are replaced as '0'
lab var eduhighyear "Highest grade attended for each level of edu"



*** Cleaning inconsistencies
replace edulevel = 0 if age<10  
replace eduhighyear = 0 if age<10 
	/*At this point, we disregard the years of education of household members 
	younger than 10 years by replacing the relevant variables with '0 years' 
	since they are too young to have completed 6 years of schooling.*/ 
replace eduhighyear = 0 if edulevel<1
	//No education

	
*** Now we create the years of schooling
tab eduhighyear edulevel, miss
gen	eduyears = eduhighyear 
replace eduyears = 6 if (eduhighyear==7 & edulevel==1) | (eduhighyear==11 & edulevel==1)
replace eduyears = eduhighyear + 6 if edulevel==2 
replace eduyears = 10 if (eduhighyear>=4 & eduhighyear<=8) & edulevel==2
	/*There are 6 grades in primary school. As such we add 6 years to each 
	of the grades completed at the colllege level of middle secondary level.*/
replace eduyears = eduhighyear + 9 if edulevel==3 
replace eduyears = 12 if (eduhighyear>=3 & eduhighyear<=6) & edulevel==3
	/*Individuals would have completed 9 years of schooling before reaching 
	secondary education. This includes 6 year primary and 3 year college or 
	middle secondary.*/
replace eduyears = eduhighyear + 12 if edulevel==4
replace eduyears = 17 if (eduhighyear>=6 & eduhighyear<=24) & edulevel==4
	/*Individuals would have completed 12 years of schooling before reaching 
	higher education.*/	


*** Checking for further inconsistencies 	
replace eduyears = . if eduyears>=age & age>0
	/*The variable "eduyears" was replaced with a '.' if total years of 
	education was more than individual's age */
replace eduyears = 0 if age < 10 
replace eduyears = 0 if (age==10 | age==11) & eduyears < 6 
	/*The variable "eduyears" was replaced with a '0' for ineligible household 
	members, i.e.: those who have not completed 6 years of schooling following 
	their starting school age */
tab eduyears, miss
	
	/*A control variable is created on whether there is information on 
	years of education for at least 2/3 of the eligible household members*/	
gen temp = 1 if eduyears!=. & age>=12 & age!=.
replace temp = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp = 1 if age==11 & eduyears>=6 & eduyears<.
bysort	hh_id: egen no_missing_edu = sum(temp)
	//Total eligible household members with no missing years of education
gen temp2 = 1 if age>=12 & age!=.
replace temp2 = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp2 = 1 if age==11 & eduyears>=6 & eduyears<.
bysort hh_id: egen hhs = sum(temp2)
	//Total number of eligible household members  
replace no_missing_edu = no_missing_edu/hhs
replace no_missing_edu = (no_missing_edu>=2/3)
	/*Identify whether there is information on years of education for at 
	least 2/3 of the eligible household members */
tab no_missing_edu, miss
	//The values for 0 (missing) is 0.01%
label var no_missing_edu "No missing edu for at least 2/3 of the eligible HH members"		
drop temp temp2 hhs


*** Standard MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed SIX years of schooling. */
******************************************************************* 
gen	 years_edu6 = (eduyears>=6)
replace years_edu6 = . if eduyears==.
bysort hh_id: egen hh_years_edu6_1 = max(years_edu6)
gen	hh_years_edu6 = (hh_years_edu6_1==1)
replace hh_years_edu6 = . if hh_years_edu6_1==.
replace hh_years_edu6 = . if hh_years_edu6==0 & no_missing_edu==0 
lab var hh_years_edu6 "Household has at least one member with 6 years of edu"
tab hh_years_edu6,m

	
*** Destitution MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed at least one year of schooling. */
******************************************************************* 
gen	years_edu1 = (eduyears>=1)
replace years_edu1 = . if eduyears==.
bysort	hh_id: egen hh_years_edu_u = max(years_edu1)
replace hh_years_edu_u = . if hh_years_edu_u==0 & no_missing_edu==0
lab var hh_years_edu_u "Household has at least one member with 1 year of edu"
tab hh_years_edu_u, miss




********************************************************************************
*** Step 2.2 Child School Attendance ***
********************************************************************************

codebook q111, tab (10)
clonevar attendance = q111 
recode attendance (2 3=0) 
	//2='attended in the past'; 3='never attended'
tab attendance, miss


*** Standard MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 8. */ 
******************************************************************* 	
gen	child_schoolage = (age>=6 & age<=14)
	/* In Morocco, the official school entrance age is 6 years.  
	So, age range is 6-14 (=6+8) 
	Source: "http://data.uis.unesco.org/?ReportId=163"
	Go to "National monitoring > Official entrance age and theoretical duration 
	by level of education > Official entrance age > Official entrance age to 
	primary education. Look at the starting age and add 8. */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the school age children */
count if child_schoolage==1 & attendance==.
	/*How many eligible school aged children with missing school 
	attendance?: 1 child */
gen temp = 1 if child_schoolage==1 & attendance!=.
	/*Generate a variable that captures the number of eligible school aged 
	children who are attending school */
bysort hh_id: egen no_missing_atten = sum(temp)	
	/*Total school age children with no missing information on school 
	attendance */
gen temp2 = 1 if child_schoolage==1	
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are of school age
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)
	/*Identify whether there is missing information on school attendance for 
	more than 2/3 of the school age children */			
tab no_missing_atten, miss
	//The value for 0 (missing) is 0.01% 
label var no_missing_atten "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs
	
	
bysort hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
	//It takes value 1 if the household has children in school age
lab var hh_children_schoolage "Household has children in school age"


gen	child_not_atten = (attendance==0) if child_schoolage==1
replace child_not_atten = . if attendance==. & child_schoolage==1
bysort	hh_id: egen any_child_not_atten = max(child_not_atten)
gen	hh_child_atten = (any_child_not_atten==0) 
replace hh_child_atten = . if any_child_not_atten==.
replace hh_child_atten = 1 if hh_children_schoolage==0
replace hh_child_atten = . if hh_child_atten==1 & no_missing_atten==0 
	/*If the household has been intially identified as non-deprived, but has 
	missing school attendance for at least 2/3 of the school aged children, then 
	we replace this household with a value of '.' because there is insufficient 
	information to conclusively conclude that the household is not deprived */
lab var hh_child_atten "Household has all school age children up to class 8 in school"
tab hh_child_atten, miss

/*Note: The indicator takes value 1 if ALL children in school age are attending 
school and 0 if there is at least one child not attending. Households with no 
children receive a value of 1 as non-deprived. The indicator has a missing value 
only when there are all missing values on children attendance in households that 
have children in school age. */

	
*** Destitution MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 6. */ 
******************************************************************* 
gen	child_schoolage_6 = (age>=6 & age<=12) 
	/*In Morocco, the official school entrance age is 6 years.  
	  So, age range for destitution measure is 6-12 (=6+6) */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the children attending school up to 
	class 6 */	
count if child_schoolage_6==1 & attendance==.	
gen temp = 1 if child_schoolage_6==1 & attendance!=.
bysort hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_6==1	
bysort hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)			
tab no_missing_atten_u, miss
label var no_missing_atten_u "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs		
		
bysort	hh_id: egen hh_children_schoolage_6 = sum(child_schoolage_6)
replace hh_children_schoolage_6 = (hh_children_schoolage_6>0) 
lab var hh_children_schoolage_6 "Household has children in school age (6 years of school)"

gen	child_atten_6 = (attendance==1) if child_schoolage_6==1
replace child_atten_6 = . if attendance==. & child_schoolage_6==1
bysort	hh_id: egen any_child_atten_6 = max(child_atten_6)
gen	hh_child_atten_u = (any_child_atten_6==1) 
replace hh_child_atten_u = . if any_child_atten_6==.
replace hh_child_atten_u = 1 if hh_children_schoolage_6==0
replace hh_child_atten_u = . if hh_child_atten_u==0 & no_missing_atten_u==0 
lab var hh_child_atten_u "Household has at least one school age children up to class 6 in school"
tab hh_child_atten_u, miss


********************************************************************************
*** Step 2.3 Nutrition ***
********************************************************************************

********************************************************************************
*** Step 2.3a Child 0-60 months Nutrition ***
********************************************************************************


*** Child Underweight Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(underweight)
gen	hh_no_underweight = (temp==0) 
	//Takes value 1 if no child in the hh is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_underweight "Household has no child underweight - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(underweight_u)
gen	hh_no_underweight_u = (temp==0) 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
lab var hh_no_underweight_u "Destitute: Household has no child underweight"
drop temp


*** Child Stunting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(stunting)
gen	hh_no_stunting = (temp==0) 
	//Takes value 1 if no child in the hh is stunted
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_stunting "Household has no child stunted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(stunting_u)
gen	hh_no_stunting_u = (temp==0) 
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
lab var hh_no_stunting_u "Destitute: Household has no child stunted"
drop temp


*** Child Wasting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(wasting)
gen	hh_no_wasting = (temp==0) 
	//Takes value 1 if no child in the hh is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_wasting "Household has no child wasted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(wasting_u)
gen	hh_no_wasting_u = (temp==0) 
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
lab var hh_no_wasting_u "Destitute: Household has no child wasted"
drop temp


*** Child Either Underweight or Stunted Indicator ***
************************************************************************

*** Standard MPI ***
gen uw_st = 1 if stunting==1 | underweight==1
replace uw_st = 0 if stunting==0 & underweight==0
replace uw_st = . if stunting==. & underweight==.
bysort hh_id: egen temp = max(uw_st)
gen	hh_no_uw_st = (temp==0) 
	//Takes value 1 if no child in the hh is underweight or stunted
replace hh_no_uw_st = . if temp==.
replace hh_no_uw_st = 1 if no_child_eligible==1
	//Households with no eligible children will receive a value of 1 
lab var hh_no_uw_st "Household has no child underweight or stunted"
drop temp


*** Destitution MPI  ***
gen uw_st_u = 1 if stunting_u==1 | underweight_u==1
replace uw_st_u = 0 if stunting_u==0 & underweight_u==0
replace uw_st_u = . if stunting_u==. & underweight_u==.
bysort hh_id: egen temp = max(uw_st_u)
gen	hh_no_uw_st_u = (temp==0) 
	//Takes value 1 if no child in the hh is underweight or stunted
replace hh_no_uw_st_u = . if temp==.
replace hh_no_uw_st_u = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1 
lab var hh_no_uw_st_u "Destitute: Household has no child underweight or stunted"
drop temp


********************************************************************************
*** Step 2.3b Household Nutrition Indicator ***
********************************************************************************

*** Standard MPI ***
/* The indicator takes value 1 if the household has no child under 5 who 
has either height-for-age or weight-for-age that is under two standard 
deviation below the median. It also takes value 1 for the households that 
have no eligible children. The indicator takes a value of missing only if 
all eligible children have missing information in their respective 
nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st = 1
replace hh_nutrition_uw_st = 0 if hh_no_uw_st==0
replace hh_nutrition_uw_st = . if hh_no_uw_st==.
replace hh_nutrition_uw_st = 1 if no_child_eligible==1   		
lab var hh_nutrition_uw_st "Household has no individuals malnourished"
tab hh_nutrition_uw_st, miss


*** Destitution MPI ***
/* The indicator takes value 1 if the household has no child under 5 who 
has either height-for-age or weight-for-age that is under three standard 
deviation below the median. It also takes value 1 for the households that 
have no eligible children. The indicator takes a value of missing only if 
all eligible children have missing information in their respective 
nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st_u = 1
replace hh_nutrition_uw_st_u = 0 if hh_no_uw_st_u==0
replace hh_nutrition_uw_st_u = . if hh_no_uw_st_u==.
replace hh_nutrition_uw_st_u = 1 if no_child_eligible==1   		
lab var hh_nutrition_uw_st_u "Household has no individuals malnourished (destitution)"
tab hh_nutrition_uw_st_u, miss


********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************


codebook w201 w206 w207_1 w207_2
	/* w206: Had children who died 
	   w207_1: number of sons who have died 
	   w207_2: number of daughters who have died */
	
egen temp_f = rowtotal(w207_1 w207_2), missing
	//Total child mortality reported by eligible women
replace temp_f = 0 if (w201==1 & w206==2) | (w201==2 & w206==2) 
replace temp_f = 0 if marital==0 & sex==2 & age>=15 & age<=49
	/*Assign a value of "0" for:
	- all eligible women who have ever gave birth but reported no child death. 
	- all eligible women who never ever gave birth and reported no child death.
	- all eligible women who never married */
bysort	hh_id: egen child_mortality_f = sum(temp_f), missing
lab var child_mortality_f "Occurrence of child mortality reported by women"
tab child_mortality_f, miss
drop temp_f		
	
	
egen child_mortality = rowmax(child_mortality_f)
replace child_mortality = 0 if child_mortality==. & no_fem_eligible==1 
lab var child_mortality "Total child mortality within household"
tab child_mortality, miss


*** Standard MPI *** 
/* The standard MPI indicator takes a value of "0" if women in the household 
reported mortality among children under 18 in the last 5 years from the survey 
year. The indicator takes a value of "1" if eligible women within the household 
reported (i) no child mortality or (ii) if any child died longer than 5 years 
from the survey year or (iii) if any child 18 years and older died in the last 
5 years. Households were replaced with a value of "1" if eligible 
men within the household reported no child mortality in the absence of 
information from women. The indicator takes a missing value if there was 
missing information on reported death from eligible individuals. */
************************************************************************
tab childu18_died_per_wom_5y, miss		
replace childu18_died_per_wom_5y = 0 if w201==2 & w206==2 
replace childu18_died_per_wom_5y = 0 if marital==0 & sex==2 & age>=15 & age<=49
replace childu18_died_per_wom_5y = 0 if w201==1 & w206==2 & childu18_died_per_wom_5y==.															   
	/*Assign a value of "0" for:
	- all eligible women who never ever gave birth 
	- all eligible women who never married */
replace childu18_died_per_wom_5y = 0 if no_fem_eligible==1	
	/*Assign a value of "0" for:
	- individuals living in households that have non-eligible women */
	
bysort hh_id: egen childu18_mortality_5y = sum(childu18_died_per_wom_5y), missing
replace childu18_mortality_5y = 0 if childu18_mortality_5y==. & child_mortality==0
	/*Replace all households as 0 death if women has missing value and men 
	reported no death in those households */
label var childu18_mortality_5y "Under 18 child mortality within household past 5 years reported by women"
tab childu18_mortality_5y, miss		
	
gen hh_mortality_u18_5y = (childu18_mortality_5y==0)
replace hh_mortality_u18_5y = . if childu18_mortality_5y==.
lab var hh_mortality_u18_5y "Household had no under 18 child mortality in the last 5 years"
tab hh_mortality_u18_5y, miss 


*** Destitution MPI *** 
*** (same as standard MPI) ***
************************************************************************
clonevar hh_mortality_u = hh_mortality_u18_5y		


*** Harmonised MPI: PAPFAM 2017/18 - PAPFAM 2011 *** 
************************************************************************
	/*In the earlier survey, there is no birth history data. This means, 
	there is no information on the date of death of children who have died. 
	As such, we are not able to construct the indicator on child mortality 
	under 18 that occurred in the last 5 years for this survey. Instead, we 
	identify individuals as deprived if any children died in the household. 
	As such, for harmonisation purpose, we construct the same indicator 
	in this survey.*/

gen	hh_mortality_c = (child_mortality==0)
replace hh_mortality_c = . if child_mortality==.
replace hh_mortality_c = 1 if no_fem_eligible==1 
lab var hh_mortality_c "COT: HH had no child mortality"
tab hh_mortality_c, miss


clonevar hh_mortality_u_c = hh_mortality_c
lab var hh_mortality_c "COT: HH had no child mortality (dst)"


********************************************************************************
*** Step 2.5 Electricity ***
********************************************************************************
	/*Morocco PAPFAM 2017-18 has no direct question on whether household has 
	electricity or not. As the best alternative, the electricity indicator for 
	this survey was drawn from the q416 variable: Main type of lighting. The 
	categories are: Electricity; Oil lamps/candles; Solar energy; Other. As 
	such, the category 'Electricity' and 'Solar energy' is recoded as 'have 
	electricity' and all other categories are recoded as 'no electricity'.*/


*** Standard MPI ***
/*Members of the household are considered 
deprived if the household has no electricity */
***************************************************
codebook q416, tab (9)
	//q416 - Main type of lighting	
gen electricity = q416 
recode electricity (2/3=0)(4=1) (6=0)

label define lab_elec 1 "have electricity" 0 "no electricity"
label values electricity lab_elec		
label var electricity "Household has electricity"
tab electricity, miss

*** Destitution MPI  ***
*** (same as standard MPI) ***
***************************************************
gen electricity_u = electricity
label var electricity_u "Household has electricity"



********************************************************************************
*** Step 2.6 Sanitation ***
********************************************************************************
/*
Improved sanitation facilities include flush or pour flush toilets to sewer 
systems, septic tanks or pit latrines, ventilated improved pit latrines, pit 
latrines with a slab, and composting toilets. These facilities are only 
considered improved if it is private, that is, it is not shared with other 
households.
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-02-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/
desc q413 q414
clonevar toilet = q413  
codebook q414, tab(9)  
recode q414(2=0), gen(shared_toilet)  
tab shared_toilet, miss



*** Standard MPI ***
/*Members of the household are considered deprived if the household's 
sanitation facility is not improved (according to the SDG guideline) 
or it is improved but shared with other households*/
********************************************************************
codebook toilet, tab (999)

/*The toilet categories for Morocco PAPFAM 2017-18 are different from the 
standardised version found in DHS and MICS. The categories are: 

1 toilet with sifon connected to the sewer		IMPROVED
2 Toilet with sifon not connected to sewer		IMPROVED
3 Toilet without sifon connected to sewer		IMPROVED
4 Toilet latrine								IMPROVED
5 In nature										NON-IMPROVED
6 Other 										NON-IMPROVED
*/


gen	toilet_mdg = (toilet==1 & shared_toilet!=1) | ///
				 (toilet==2 & shared_toilet!=1) | ///
				 (toilet==3 & shared_toilet!=1)	| ///
				 (toilet==4 & shared_toilet!=1)
	/*Household is assigned a value of '1' if it uses improved sanitation and 
	does not share toilet with other households  */	
lab var toilet_mdg "Household has improved sanitation with MDG Standards"
tab toilet toilet_mdg, miss


*** Destitution MPI ***
/*Members of the household are considered deprived if household practises 
open defecation or uses other unidentifiable sanitation practises */
********************************************************************
gen	toilet_u = 1
replace toilet_u = 0 if toilet==5 | toilet==6  
	/*Household is assigned a value of '0' if it practises open defecation or 
	others */
lab var toilet_u "Household does not practise open defecation or others"
tab toilet toilet_u, miss


********************************************************************************
*** Step 2.7 Drinking Water  ***
********************************************************************************

/*
Improved drinking water sources include the following: piped water into 
dwelling, yard or plot; public taps or standpipes; boreholes or tubewells; 
protected dug wells; protected springs; packaged water; delivered water and 
rainwater which is located on premises or is less than a 30-minute walk from 
home roundtrip. 
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-01-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

desc q408 q410
clonevar water = q408  
clonevar timetowater = q410   
gen ndwater = . 
	//No data on the use of water for non-drinking activities


 
*** Standard MPI ***
/* Members of the household are considered deprived if the household 
does not have access to improved drinking water (according to the SDG 
guideline) or safe drinking water is at least a 30-minute walk from 
home, roundtrip */
********************************************************************
codebook water, tab(99) 

/* The sources of drinking water categories for Morocco PAPFAM 2018 are: 
		
	 1 public network							improved
	 2 public tap (Fountain)					improved 
	 3 Well equipped with a pump				improved 
	 4 Well not equipped with a pump			non-improved 
	 5 Source protected							improved 
	 6 Source not protected						non-improved 
	 7 River / stream							non-improved 
	 8 Lake										non-improved 		
	 9 Rainwater								improved 		
	 10 Water tanker					 		non-improved 		
	 11 Bottled mineral water					non-improved 		
	 96 Other									non-improved */

gen	water_mdg = 1 if water==1 | water==2 | water==3  | ///
					 water==5 | water==9 
	
replace water_mdg = 0 if water==4  | water==6  | water==7 | water==8 | ///
						 water==10 | water==11 | water==96 
						 
lab var water_mdg "Household has safe drinking water on premises"
tab water water_mdg, miss	
tab water_mdg, miss


*** Time to water ***	
********************************************************* 
codebook timetowater, tab(999)
						 
replace water_mdg = 0 if water_mdg==1 & timetowater >= 30 & timetowater!=. & ///
						 timetowater!=998 
	//Deprived if water is at more than 30 minutes' walk (roundtrip) 
		


*** Destitution MPI ***
/* Members of the household is identified as destitute if household 
does not have access to safe drinking water, or safe water is more 
than 45 minute walk from home, round trip.*/
********************************************************************
gen	water_u = 1 if   water==1 | water==2 | water==3  | ///
					 water==5 | water==9 
	
replace water_u = 0 if   water==4 | water==6  | water==7  | ///
						 water==8 | water==10 | water==11 | water==96 
						   
replace water_u = 0 if water_u==1 & timetowater>45 & timetowater!=. & ///					
					   timetowater!=998 
						   					  
lab var water_u "Household has drinking water with MDG standards (45 minutes distance)"
tab water_u, miss


********************************************************************************
*** Step 2.8 Housing ***
********************************************************************************

/*Members of the household are considered deprived if the household 
has a dirt, sand or dung floor. 

1 Mosaic / Marble				improved
2 Natural clay					improved
3 Mixture of cement and sand	improved
6 Other 						improved */

desc q401 q403
tab q401 q403,m
clonevar floor = q403
codebook floor, tab(99)
tab q401 if q403==6,m
	/*We consider those who reported having other floor as having 
	improved floor since they live in apartment of modern house.*/ 
gen	floor_imp = 1 
replace floor_imp = . if floor==. 		
lab var floor_imp "Household has floor that it is not earth/sand/dung"
replace floor_imp = 0 if floor==. & q401>=5 	
	/*Specific to Morocco PAPFAM 2017-18: Deprived if type of dwelling is 
	traditional, tent, temporary shelter or other.*/
tab floor floor_imp, miss


/* Members of the household are considered deprived if the household has wall 
made of natural or rudimentary materials
1 Polished walls	improved
2 Natural walls		non-improved
3 Primitive walls	non-improved
6 Other				non-improved
*/
desc q405
clonevar wall = q405
codebook wall, tab(99)
gen	wall_imp = 1
replace wall_imp = 0 if wall==2 | wall==3 | wall==6    
replace wall_imp = . if wall==.
lab var wall_imp "Household has wall that it is not of low quality materials"
replace wall_imp = 0 if wall==. & q401>=5  
	/*Specific to Morocco PAPFAM 2017-18: Deprived if type of dwelling is 
	traditional, tent, temporary shelter or other.*/
tab wall wall_imp, miss	



/* Members of the household are considered deprived if the household has roof 
made of natural or rudimentary materials 
1 concrete 									improved	
2 wood shingles								improved
3 cane/grass covered in mud					non-improved
4 tin roof 									improved
6 other										non-improved
8 there is no ceiling						non-improved
*/
desc q404 	
clonevar roof = q404
codebook roof, tab(99)		
gen	roof_imp = 1 
replace roof_imp = 0 if roof==3 | roof==6 | roof==8 
replace roof_imp = . if roof==. 
lab var roof_imp "Household has roof that it is not of low quality materials"
replace roof_imp = 0 if wall==. & q401>=5  
	/*Specific to Morocco PAPFAM 2017-18: Deprived if type of dwelling is 
	traditional, tent, temporary shelter or other.*/
tab roof roof_imp, miss


*** Standard MPI ***
/* Members of the household is deprived in housing if the roof, 
floor OR walls are constructed from low quality materials.*/
**************************************************************
gen housing_1 = 1
replace housing_1 = 0 if floor_imp==0 | wall_imp==0 | roof_imp==0
replace housing_1 = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_1 "Household has roof, floor & walls that it is not low quality material"
tab housing_1, miss


*** Destitution MPI ***
/* Members of the household is deprived in housing if two out 
of three components (roof and walls; OR floor and walls; OR 
roof and floor) the are constructed from low quality materials. */
**************************************************************
gen housing_u = 1
replace housing_u = 0 if (floor_imp==0 & wall_imp==0 & roof_imp==1) | ///
						 (floor_imp==0 & wall_imp==1 & roof_imp==0) | ///
						 (floor_imp==1 & wall_imp==0 & roof_imp==0) | ///
						 (floor_imp==0 & wall_imp==0 & roof_imp==0)
replace housing_u = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_u "Household has one of three aspects(either roof,floor/walls) that is not low quality material"
tab housing_u, miss


*** Harmonised MPI *** 
************************************************************************
	/*In the case of Morocco PAPFAM 2011, there is no data on wall. The final
	housing indicator was constructed using information on floor and roof only. 
	For the purpose of harmonisation, we construct a comparable indicator in 
	this survey.*/
gen housing_1_c = 1
replace housing_1_c = 0 if floor_imp==0 | roof_imp==0
replace housing_1_c = . if floor_imp==. & roof_imp==.
lab var housing_1_c "Household has roof & floor that it is not low quality material"
tab housing_1_c, miss

gen housing_u_c = 1
replace housing_u_c = 0 if (floor_imp==0 & roof_imp==1) | ///
						 (floor_imp==0 & roof_imp==0) | ///
						 (floor_imp==1 & roof_imp==0) 
replace housing_u_c = . if floor_imp==. & roof_imp==.
lab var housing_u_c "Household has one of three aspects(either roof,floor/walls) that it is not low quality material"
tab housing_u_c, miss	
	


********************************************************************************
*** Step 2.9 Cooking Fuel ***
********************************************************************************
/*
Solid fuel are solid materials burned as fuels, which includes coal as well as 
solid biomass fuels (wood, animal dung, crop wastes and charcoal). 

Source: 
https://apps.who.int/iris/bitstream/handle/10665/141496/9789241548885_eng.pdf
*/
desc q418
clonevar cookingfuel = q418  

*** Standard MPI ***
/* Members of the household are considered deprived if the 
household uses solid fuels and solid biomass fuels for cooking. */
*****************************************************************
codebook cookingfuel, tab(99)
gen	cooking_mdg = 1
replace cooking_mdg = 0 if cookingfuel>=4 & cookingfuel<6  
replace cooking_mdg = . if cookingfuel==. 
lab var cooking_mdg "Household has cooking fuel by SDG standards"
	//Deprived if: coal; wood 
tab cookingfuel cooking_mdg, miss


*** Destitution MPI ***
*** (same as standard MPI) ***
****************************************
gen	cooking_u = cooking_mdg
lab var cooking_u "Household uses clean fuels for cooking"


********************************************************************************
*** Step 2.10 Assets ownership ***
********************************************************************************

desc q419_1 q419_2 q419_10 q419_11 q419_5 q420_2 q420_1 ///
q419_15 q419_16 q420_9 q420_5
 
 
*** Television/LCD TV/plasma TV/color TV/black & white tv
clonevar television = q419_1 
lab var television "Household has television"
tab television, miss


***	Radio/walkman/stereo/kindle
clonevar radio = q419_2 
lab var radio "Household has radio"	
tab radio, miss


***	Handphone/telephone/iphone/mobilephone/ipod
clonevar telephone = q419_10
replace telephone=1 if telephone!=1 & q419_11==1	
	//q419_11=mobilephone. Combine information on telephone and mobilephone.	
tab q419_10 q419_11 if telephone==1,miss
lab var telephone "Household has telephone (landline/mobilephone)"	
tab telephone, miss

***	Refrigerator/icebox/fridge
clonevar refrigerator = q419_5
lab var refrigerator "Household has refrigerator"
tab refrigerator, miss


***	Car/van/lorry/truck
clonevar car = q420_2
lab var car "Household has car"
tab car, miss


***	Bicycle/cycle rickshaw
gen bicycle = .
lab var bicycle "Household has bicycle"
	//Morocco PAPFAM 2017-18 has no data on ownership of bicycle
	
	
***	Motorbike/motorized bike/autorickshaw	
gen motorbike = q420_1
lab var motorbike "Household has motorbike"
tab motorbike, miss


***	Computer/laptop/tablet
clonevar computer = q419_15
replace computer=1 if computer!=1 & q419_16==1	
	//q419_16=tablet. Combine information on computer and tablet.	
tab q419_15 q419_16 if computer==1,miss
lab var computer "Household has computer"
tab computer, miss


***	Animal cart
clonevar animal_cart = q420_9
lab var animal_cart "Household has animal cart"	
tab animal_cart, miss


foreach var in television radio telephone refrigerator car ///
			   bicycle motorbike computer animal_cart {
replace `var' = 0 if `var'==2 
label define lab_`var' 0"No" 1"Yes"
label values `var' lab_`var'			   
replace `var' = . if `var'==9 | `var'==99 | `var'==8 | `var'==98 
}
	//Labels defined and missing values replaced	
	
	

*** Standard MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own more than one of: radio, TV, telephone, bike, motorbike, 
refrigerator, computer or animal cart and does not own a car or truck.*/
*****************************************************************************
egen n_small_assets2 = rowtotal(television radio telephone refrigerator bicycle motorbike computer animal_cart), missing
lab var n_small_assets2 "Household Number of Small Assets Owned" 
   
gen hh_assets2 = (car==1 | n_small_assets2 > 1) 
replace hh_assets2 = . if car==. & n_small_assets2==.
lab var hh_assets2 "Household Asset Ownership: HH has car or more than 1 small assets incl computer & animal cart"
tab hh_assets2, miss


*** Destitution MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own any assets.*/
*****************************************************************************	
gen	hh_assets2_u = (car==1 | n_small_assets2>0)
replace hh_assets2_u = . if car==. & n_small_assets2==.
lab var hh_assets2_u "Household Asset Ownership: HH has car or at least 1 small assets incl computer & animal cart"



*** Harmonised MPI: PAPFAM 2017/18 - PAPFAM 2011 *** 
	/*The earlier survey lacked information on animal cart. As 
	such, the harmonised work excludes the use of animal cart.
	But this made no difference because the use of animal cart 
	in this island nation is almost 0%.*/
************************************************************************
gen animal_cart_c = .
lab var animal_cart_c "COT: Household has animal cart"

egen n_small_assets2_c = rowtotal(television radio telephone refrigerator bicycle motorbike computer animal_cart_c), missing
lab var n_small_assets2_c "COT: HH Number of Small Assets Owned" 
    
gen hh_assets2_c = (car==1 | n_small_assets2_c > 1) 
replace hh_assets2_c = . if car==. & n_small_assets2_c==.
lab var hh_assets2_c "COT: HH has car or more than 1 small assets"

gen	hh_assets2_u_c = (car==1 | n_small_assets2_c>0)
replace hh_assets2_u_c = . if car==. & n_small_assets2_c==.
lab var hh_assets2_u_c "COT-DST: HH has car or at least 1 small assets"


********************************************************************************
*** Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************

	//Retain data on sampling design: 
clonevar strata = region  
clonevar psu = district
label var psu "Primary sampling unit"
label var strata "Sample strata"


	//Retain year, month & date of interview:
clonevar year_interview = yy1 	
clonevar month_interview = mm1 
clonevar date_interview = dd1


	//Generate presence of subsample
gen subsample = .

	
*** Keep main variables require for MPI calculation ***
keep hh_id ind_id subsample strata psu weight weight_ch sex age hhsize ///
area agec7 agec4 agec2 region headship marital relationship ///
no_fem_eligible no_male_eligible child_eligible no_child_eligible ///
religion_wom religion_men religion_hh ethnic_wom ethnic_men ethnic_hh ///
insurance_wom insurance_men year_interview month_interview date_interview /// 
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet toilet_mdg shared_toilet toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
hh_mortality_c hh_mortality_u_c housing_1_c housing_u_c ///
hh_assets2_c hh_assets2_u_c

	 
*** Order file	***
order hh_id ind_id subsample strata psu weight weight_ch sex age hhsize ///
area agec7 agec4 agec2 region headship marital relationship ///
no_fem_eligible no_male_eligible child_eligible no_child_eligible ///
religion_wom religion_men religion_hh ethnic_wom ethnic_men ethnic_hh ///
insurance_wom insurance_men year_interview month_interview date_interview /// 
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet toilet_mdg shared_toilet toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
hh_mortality_c hh_mortality_u_c housing_1_c housing_u_c ///
hh_assets2_c hh_assets2_u_c


*** Rename key global MPI indicators for estimation ***
recode hh_mortality_u18_5y  (0=1)(1=0) , gen(d_cm)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ)
recode electricity 			(0=1)(1=0) , gen(d_elct)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani)
recode housing_1 			(0=1)(1=0) , gen(d_hsg)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl)
recode hh_assets2 			(0=1)(1=0) , gen(d_asst)
 

*** Rename key global MPI indicators for destitution estimation ***
recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ)
recode electricity_u		(0=1)(1=0) , gen(dst_elct)
recode water_u 				(0=1)(1=0) , gen(dst_wtr)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl)
recode hh_assets2_u 		(0=1)(1=0) , gen(dst_asst) 


*** Rename indicators for changes over time estimation ***	
recode hh_mortality_c		(0=1)(1=0) , gen(d_cm_01)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr_01)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt_01)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ_01)
recode electricity 			(0=1)(1=0) , gen(d_elct_01)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr_01)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani_01)
recode housing_1_c 			(0=1)(1=0) , gen(d_hsg_01)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl_01)
recode hh_assets2_c 		(0=1)(1=0) , gen(d_asst_01)	
	

recode hh_mortality_u_c     (0=1)(1=0) , gen(dst_cm_01)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr_01)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt_01)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ_01)
recode electricity_u		(0=1)(1=0) , gen(dst_elct_01)
recode water_u	 			(0=1)(1=0) , gen(dst_wtr_01)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani_01)
recode housing_u_c 			(0=1)(1=0) , gen(dst_hsg_01)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl_01)
recode hh_assets2_u_c 		(0=1)(1=0) , gen(dst_asst_01) 


	/*In this survey, the harmonised 'region_01' variable requires 
	check.*/	
gen region_01 = .



*** Total number of missing values  for each variable *** 
mdesc psu strata area age headship region ///
d_cm d_nutr d_satt d_educ d_elct d_wtr d_sani d_hsg d_ckfl d_asst



mdesc psu strata area age headship region_01 ///
d_cm_01 d_nutr_01 d_satt_01 d_educ_01 ///
d_elct_01 d_wtr_01 d_sani_01 d_hsg_01 d_ckfl_01 d_asst_01



gen N = d_cm!=. & d_nutr!=. & d_satt!=. & d_educ!=. & d_elct!=. & ///
d_wtr!=. & d_sani!=. & d_hsg!=. & d_ckfl!=. & d_asst!=.
tab N, miss
tab N [aw = weight]
tab region N [aw = weight], row
drop N
	/*Final sample retained for estimation is 62,240 individuals 
	(92.33 % unweighted; 91.87% weighted) */


*** Generate coutry and survey details for estimation ***
char _dta[cty] "Morocco"
char _dta[ccty] "MAR"
char _dta[year] "2017-2018" 	
char _dta[survey] "PAPFAM"
char _dta[ccnum] "504"
char _dta[type] "micro"


*** Sort, compress and save data for estimation ***
sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/mar_papfam17-18.dta", replace 


gen educ_elig = 1 
replace educ_elig = 0 if age < 10 
replace educ_elig = 0 if (age==10 | age==11) & eduyears < 6 
	/*The variable "educ_eligible" was replaced with a '0' for ineligible 
	individuals, i.e.: those who have not completed 6 years of schooling 
	following their starting school age.*/
label define lab_educ_elig 0"ineligible" 1"eligible"  
label values educ_elig lab_educ_elig
lab var educ_elig "Individual is eligible for educ indicator"
tab eduyears educ_elig,m

