********************************************************************************
/*
Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
2021 Global Multidimensional Poverty Index - Jamaica JSLC 2010 [STATA do-file]. 
Available from OPHI website: http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk
*/
********************************************************************************

	/*FOR JAMAICA: Raw dta files that we will be using in JSLC 2010:
	anthro Section C  - Children anthropometrics
	rec001	 Cover Data - gives you the weights
	rec003	 Section R  - Roster : list all members of each household, 
	and relation with hh head.
	rec006	- Health		
	rec007,rec09, rec010 - Education (educ)
	rec026 - Housing and Related Expenses
	rec027 - Inventory of Durable Goods
	Household Population Data 
	
	Its important to review the Survey Questionnaire, the dataset description, 
	dictionary and each raw data set. This is because several variables and 
	coding of those variables could change between year.*/


clear all 
set more off
set maxvar 10000
set mem 500m
	
	
*** Working Folder Path ***
global path_in "../rdta/Jamaica JSLC 2010" 	  
global path_out "cdta"
global path_ado "ado"
		

********************************************************************************
*** Step 1: Data preparation  
********************************************************************************
	

********************************************************************************
*** Step 1.1 CHILDREN UNDER 5 RECODE 
********************************************************************************

use "$path_in/anthro.dta", clear 

codebook ind serial
destring serial, replace
destring ind, replace
codebook ind serial 

*** Generate individual unique key variable required for data merging
*** serial=household number; 
*** ind=child's line number in household; 
gen double ind_id = serial*100 + ind 
format ind_id %20.0g
label var  ind_id "Individual ID"


duplicates report ind_id
	//No duplicates: 419 observations
	//Report tables indicate 411 children (p.VII). 

gen child_KR=1 
	//Generate identification variable for observations in KR recode


*** Next, indicate to STATA where the igrowup_restricted.ado file is stored:
	***Source of ado file: http://www.who.int/childgrowth/software/en/
adopath + "$path_ado/igrowup_stata"


*** We will now proceed to create three nutritional variables: 
	*** weight-for-age (underweight),  
	*** weight-for-height (wasting) 
	*** height-for-age (stunting)

	
/* We use 'reflib' to specify the package directory where the .dta files 
containing the WHO Child Growth Standards are stored. Note that we use 
strX to specify the length of the path in string. If the path is long, 
you may specify str55 or more, so it will run. */	
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"


/* We use datalib to specify the working directory where the input STATA 
dataset containing the anthropometric measurement is stored. */
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"


/* We use datalab to specify the name that will prefix the output files that 
will be produced from using this ado file (datalab_z_r_rc and datalab_prev_rc)*/
gen str30 datalab = "children_nutri_jam" 
lab var datalab "Working file"


*** Next check the variables that WHO ado needs to calculate the z-scores:
*** sex, age, weight, height, measurement, oedema & child sampling weight


*** Variable: SEX ***
tab sex, miss 
	//"1" for male ;"2" for female
clonevar gender = sex
desc gender
tab gender


*** Variable: AGE ***
lookfor age month day
codebook c6_y c6_m
tab c6_y c6_m
gen agey = c6_y*12
gen age_months = agey + c6_m
tab age_months, m
tab age_months c6_y, m

gen  str6 ageunit = "months" 
lab var ageunit "Months"

	
*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook c10, tab (999) 
gen	weight = c10
tab c10 ,m nol  


*** Variable: HEIGHT (CENTIMETERS)
codebook c11, tab (999)
gen	height = c11
replace height = . if c11>=990
tab c11,m nol  


*** Variable: MEASURED STANDING/LYING DOWN		
codebook c12
	// 1"lying down" 2 "standing up" according to questionnaire (p.B7)
destring c12, replace
gen measure = "l" if c12==1 
	//Child measured lying down
replace measure = "h" if c12==2 
	//Child measured standing up
replace measure = " " if c12==. | c12==97
	//Replace with " " if unknown


*** Variable: OEDEMA ***
gen  oedema = "n"  
	//It assumes no-one has oedema



*** Variable: INDIVIDUAL CHILD SAMPLING WEIGHT ***
gen  sw = 1



/*We now run the command to calculate the z-scores with the adofile */
igrowup_restricted reflib datalib datalab gender ///
age_months ageunit weight height measure oedema sw


/*We now turn to using the dta file that was created and that contains 
the calculated z-scores */
use "$path_out/children_nutri_jam_z_rc.dta", clear 

	
*** Standard MPI indicator ***
	//Takes value 1 if the child is under 2 stdev below the median & 0 otherwise	
gen	underweight = (_zwei < -2.0) 
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "Child is undernourished (weight-for-age) 2sd - WHO"
tab underweight, miss


gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "Child is stunted (length/height-for-age) 2sd - WHO"
tab stunting, miss


gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "Child is wasted (weight-for-length/height) 2sd - WHO"
tab wasting, miss


*** Destitution indicator  ***
	//Takes value 1 if the child is under 3 stdev below the median & 0 otherwise	
gen	underweight_u = (_zwei < -3.0) 
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "Child is undernourished (weight-for-age) 3sd - WHO"


gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "Child is stunted (length/height-for-age) 3sd - WHO"


gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "Child is wasted (weight-for-length/height) 3sd - WHO"


clonevar weight_ch = finwght
label var weight_ch "sample weight child under 5"


	//Retain relevant variables:
keep ind_id child_KR weight_ch underweight* stunting* wasting*  
order ind_id child_KR weight_ch underweight* stunting* wasting*
sort ind_id
save "$path_out/JAM10_PR.dta", replace


	//Erase files from folder:
erase "$path_out/children_nutri_jam_z_rc.xls"
erase "$path_out/children_nutri_jam_prev_rc.xls"
erase "$path_out/children_nutri_jam_z_rc.dta"
	
	
********************************************************************************
*** Step 1.2 HOUSEHOLD MEMBER'S RECODE 
********************************************************************************

use "$path_in/rec003.dta", clear


destring serial, replace
gen double hh_id = serial 
format hh_id %20.0g
label var hh_id "Household ID"
codebook hh_id  

destring ind, replace
gen double ind_id = serial*100+ ind  
format ind_id %20.0g
label var ind_id "Individual ID"
codebook ind_id

save "$path_out/rec003.dta", replace
	
********************************************************************************
*** 1.3a DATA MERGING
*********************************************************************************


	// Cover data at hh level
use "$path_in/rec001.dta", clear
destring serial, replace
save "$path_out/rec001.dta", replace

	// Housing and related expenses at hh level
use "$path_in/rec026.dta", clear
destring serial, replace
save "$path_out/rec026.dta", replace

	// Inventory of durable goods at hh level
use "$path_in/rec027.dta", clear
destring serial, replace
save "$path_out/rec027.dta", replace

	// Household size at hh level
use "$path_in/hhsizes.dta", clear
destring serial, replace
save "$path_out/hhsizes.dta", replace

	// Education at individual level
use "$path_in/rec007.dta", clear
destring serial, replace
destring ind, replace
save "$path_out/rec007.dta", replace

use "$path_in/rec009.dta", clear
destring serial, replace
destring ind, replace
save "$path_out/rec009.dta", replace

use "$path_in/rec010.dta", clear
destring serial, replace
destring ind, replace
save "$path_out/rec010.dta", replace

	// Health at individual level
use "$path_in/rec006.dta", clear
destring serial, replace
destring ind, replace
save "$path_out/rec006.dta", replace


********************************************************************************
*** 1.3b DATA MERGING
*********************************************************************************

use "$path_out/rec003.dta", clear

merge m:1 serial using "$path_out/rec001.dta", nogen
	//Data at hh level
	
merge m:1 serial using "$path_out/rec026.dta", nogen
	//Housing and related expenses at hh level
	
merge m:1 serial using "$path_out/rec027.dta", nogen
	//Inventory of durable good at hh level
	
merge m:1 serial using "$path_out/hhsizes.dta", nogen
	//Household size at hh level
	
merge 1:1 serial ind using "$path_out/rec007.dta", nogen
merge 1:1 serial ind using "$path_out/rec009.dta", nogen
merge 1:1 serial ind using "$path_out/rec010.dta", nogen
	//Education at individual level
	
merge 1:1 serial ind using "$path_out/rec006.dta", nogen
	//Health at individual level
	
merge 1:1 ind_id using "$path_out/JAM10_PR.dta", nogen
	//Measured children data at individual level
erase "$path_out/JAM10_PR.dta"	


gen religion_wom = .
lab var religion_wom "Women's religion"	

gen ethnic_wom = .
lab var ethnic_wom "Women's ethnicity"	

gen insurance_wom = .
label var insurance_wom "Women have health insurance"	

gen marital_wom = .

gen religion_men = . 
lab var religion_men "Men's religion"	

gen ethnic_men = .
lab var ethnic_men "Men's ethnicity"

gen insurance_men = .
label var insurance_men "Men have health insurance"	

gen marital_men = .

sort hh_id ind_id


********************************************************************************
*** Step 1.4 KEEPING ONLY DE JURE HOUSEHOLD MEMBERS ***
********************************************************************************

desc hhmember
codebook hhmember, tab (10)
	/* Questionnaire, Household roster, q.10
		01 = Still a member
		02 = No longer a member
		03 = New member */
destring hhmember, replace
count if hhmember!=2 
	// 5534
	
clonevar resident = hhmember
replace resident = 1 if hhmember==3
label var resident "Permanent (de jure) household member"


drop if resident==2
tab resident, miss
	/*Jamaica JSLC 2010: 195 (3.4%) individuals who were 
	non-usual residents were dropped from the sample*/

	
********************************************************************************
*** Step 1.5 CONTROL VARIABLES
********************************************************************************


*** No Eligible Women 15-49 years
*****************************************
gen	no_fem_eligible = .
lab var no_fem_eligible "Household has no eligible women"
tab no_fem_eligible, miss


*** No Eligible Men 
*****************************************
gen no_male_eligible = . 
lab var no_male_eligible "Household has no eligible man"
tab no_male_eligible, miss


*** No Eligible Children 0-5 years
*****************************************
gen	child_eligible = (child_KR==1) 
bysort	hh_id: egen hh_n_children_eligible = sum(child_eligible)  
	//Number of eligible children for anthropometrics
gen	no_child_eligible = (hh_n_children_eligible==0) 
	//Takes value 1 if there were no eligible children for anthropometrics
lab var no_child_eligible "Household has no children eligible"
tab no_child_eligible, miss


sort hh_id


********************************************************************************
*** 1.6 RENAMING DEMOGRAPHIC VARIABLES ***
********************************************************************************

//Sample weight
desc finwght
clonevar weight = finwght 


//Area: urban or rural	
desc area
codebook area, tab (5)
	/* Area codes for JSLC 2010 are the following: (p.8)
		01 = KMA (Kingston metropolitan area)
		02 = Other towns
		03 = Rural areas  */
destring area, replace
rename area oldarea	
clonevar area = oldarea
recode area (3=0) (2=1)		
label define lab_area 1 "urban" 0 "rural"
label values area lab_area
label var area "Area: urban-rural"
tab area, miss
tab area oldarea, m


//Relationship to the head of household 
destring relat, replace
clonevar relationship = relat 
codebook relationship, tab (20)
	/* The questionnaire has the following categories for "relationship to 
	the head of the household":
		1 = Head
		2 = Spouse
		3 = Child of head/spouse
		4 = Spouse of child
		5 = Grandchild
		6 = Parent of head/spous
		7 = Other relative
		8 = Helper/Domestic
		9 = Other not relative
	This also matches the coding for the JSLC 2014 in the GMPI 2018.*/
recode relationship (1=1)(2=2)(3=3)(4/7=4)(9=5)(8=6)
label define lab_rel 1"head" 2"spouse" 3"child" 4"extended family" 5"not related" 6"maid"
label values relationship lab_rel
label var relationship "Relationship to the head of household"
tab relat relationship, miss
	/* Note that there is a large difference between numbers for head of 
	household (1682) and spouse (617). This result from the large proportion of
	female-headed household in Jamaica (see p.II and 1.7 of report). */

	
//Sex of household member	
codebook sex, tab (5)
destring sex, replace
recode sex (2=0)  
label define lab_sex 1 "male" 0 "female"
label values sex lab_sex


//Household headship
bys	hh_id: egen missing_hhead = min(relation)
tab missing_hhead,m 
gen household_head=.
replace household_head=1 if relation==1 & sex==1 
replace household_head=2 if relation==1 & sex==2
bysort hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
label define head 1"male-headed" 2"female-headed"
label values headship head
label var headship "Household headship"
tab headship, miss

//Age of household member
codebook age, tab (999)
destring age, replace


//Age group 
recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+"), gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"


//Marital status of household member
clonevar marital = marital_stat 
destring marital, replace
codebook marital, tab (20)
	/* Questionnaire coding (Household roster, q.17)
		1 = Married
		2 = Never married
		3 = Divorced
		4 = Separated
		5 = Widowed */
recode marital (2=1)(1=2)(5=3)(3=4)(4=5)(97 99=.)
label define lab_mar 1"never married" 2"currently married" ///
					 3"widowed" 4"divorced" 5"not living together"
label values marital lab_mar	
label var marital "Marital status of household member"
tab marital_stat marital, miss


//Total number of hh members in the household
gen member = 1
bysort hh_id: egen hhsize = sum(member)
label var hhsize "Household size"
tab hhsize, miss
drop member


//Religion of the household head
gen religion_hh = .
label var religion_hh "Religion of household head"


//Ethnicity of the household head
gen ethnic_hh = .
label var ethnic_hh "Ethnicity of household head"


//Subnational region
	/* The JSLC 2014 uses the parish variable for subnational regions. 
	However, the JSLC 2010 has a smaller sample and hence the sample is 
	representative at the region level: Kingston Metropolitan 
	Area (KMA), Other Towns and Rural Areas.*/	
gen region = oldarea
codebook region, tab (99)
lab define lab_region 1 "Kingston Metropolitan Area" ///
					  2 "Other Towns" ///
					  3 "Rural Areas"
lab values region lab_region
lab var region "Region for subnational decomposition"


********************************************************************************
***  Step 2 Data preparation  ***
***  Standardization of the 10 Global MPI indicators 
***  Identification of non-deprived & deprived individuals  
********************************************************************************

********************************************************************************
*** Step 2.1 Years of Schooling ***
********************************************************************************

codebook b21, tab (100)
gen edulevel = b21  
	/* Highest educational level attended, values based on questionnaire (p.B3 q.21) */  
destring edulevel, replace
destring b21, replace
replace edulevel = . if b21 == . | b21 >= 97 
	// Missing values
replace edulevel = 0 if b21 == 20  
	// None
label define lab_edulevel 0 "None" 1"Basic/infant/Kindergarten" ///
						  2 "Primary" 3 "Preparatory" ///
						  4 "All age school(grades1-6)"  ///
						  5 "All age school (grade 7-9)" ///
						  6 "Primary/junior high (grades1-6)" ///
						  7 "Primary/junior high (grades 7-9)" ///
						  8 "Junior high (grades 7-9)" 9 "New secondary" ///
						  10 "Comprehensive" 11 "Secondary high" ///
						  12 "Technical" 13 "Vocat/Agri" 14"University" ///
						  15 "Other tertiary (public)" 16 "Other tertiary (private)" ///
						  17 "Adult literacy classes" 18 "Adult education/night" ///
						  19 "Special school", modify 
label values edulevel lab_edulevel
tab edulevel, m


codebook b22, tab(30)
destring b22, replace
clonevar  eduhighyear = b22   
	//Total number of years of education
replace eduhighyear = . if b22 >=97 | b22 ==.
	//Recode any unreasonable years of highest education as missing value

*** Cleaning inconsistencies 
replace edulevel = 0 if age<10  
replace eduhighyear = 0 if age<10 
	/*The variables edulevel and eduhighyear was replaced with a '0' given that 
	the criteria for this indicator is household member aged 10 years or older */ 
replace eduhighyear = 0 if edulevel<1
	//Early childhood education has no grade
	
tab eduhighyear edulevel, m
	
clonevar eduyears = eduhighyear	
replace eduyears = 14 if edulevel >=14 & edulevel <=16  
	//Assumes 14 years of education for all tertiary education 
replace eduyears = 0 if edulevel >=17 & edulevel <=19  
	//Assumes that adult education and special schools is not measured as formal schooling
replace eduyears = 0 if edulevel==1  & eduyears==.  
	//Assumes kindergarden represents 0 years
replace eduyears = 1 if (edulevel == 2 | edulevel == 4) & eduyears== .
replace eduyears = 7 if (edulevel == 5 | edulevel == 7 | edulevel == 8) & eduyears== .
replace eduyears = 3 if edulevel == 6 & eduyears== .
replace eduyears = 7 if (edulevel >= 9 & edulevel <= 11) & eduyears== .
replace eduyears = 9 if (edulevel >= 12 & edulevel <= 13 | edulevel==16 ) & eduyears== .

codebook b4
gen lastgrade = b4
destring lastgrade, replace

codebook b1, tab (100)
	// "What type of school is (name) attendting this academic year?"
	// See detailed list of values in next section
destring b1, replace

replace eduyears = 0 if edulevel == 0 | b1 == 1 
replace eduyears = lastgrade - 1 if eduyears==. & edulevel ==. & b1>= 3 & b1<= 11 & lastgrade!=.
replace eduyears = 8 if eduyears==. & edulevel ==. & b1==12 & lastgrade!=.
replace eduyears = 8 if eduyears==. & edulevel ==. &  b1>=10 & b1<=12 & lastgrade==.
replace eduyears = 12 if eduyears==. & edulevel ==. &  b1>=13 & b1<=15 & lastgrade==.
replace eduyears = 0 if eduyears==. & edulevel ==. & (b1<=2 | b1>=16 & b1<=99 ) 
lab var eduyears "Highest year of education completed"

*** Checking for further inconsistencies 
replace eduyears = . if age<=eduyears & age>0 
	/*There are cases in which the years of schooling are greater than the 
	age of the individual. This is clearly a mistake in the data. Please check 
	whether this is the case and correct when necessary */
replace eduyears = 0 if age< 10 
	/*The variable "eduyears" was replaced with a '0' given that the criteria 
	for this indicator is household member aged 10 years or older */
lab var eduyears "Total number of years of education accomplished"
tab eduyears, miss

tab eduyears edulevel, m
	
	
	/*A control variable is created on whether there is information on 
	years of education for at least 2/3 of the household members aged 10 years 
	and older */	
gen temp = 1 if eduyears!=. & age>=10 & age!=.
bysort	hh_id: egen no_missing_edu = sum(temp)
	/*Total household members who are 10 years and older with no missing 
	years of education */
gen temp2 = 1 if age>=10 & age!=.
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are 10 years and older 
replace no_missing_edu = no_missing_edu/hhs
replace no_missing_edu = (no_missing_edu>=2/3)
	/*Identify whether there is information on years of education for at 
	least 2/3 of the household members aged 10 years and older */
tab no_missing_edu, miss
label var no_missing_edu "No missing edu for at least 2/3 of the HH members aged 10 years & older"		
drop temp temp2 hhs


*** Standard MPI ***
/*The entire household is considered deprived if no household member aged 
10 years or older has completed SIX years of schooling. */
******************************************************************* 
gen	 years_edu6 = (eduyears>=6)
replace years_edu6 = . if eduyears==.
bysort hh_id: egen hh_years_edu6_1 = max(years_edu6)
gen	hh_years_edu6 = (hh_years_edu6_1==1)
replace hh_years_edu6 = . if hh_years_edu6_1==.
replace hh_years_edu6 = . if hh_years_edu6==0 & no_missing_edu==0 
lab var hh_years_edu6 "Household has at least one member with 6 years of edu"
tab hh_years_edu6, miss

	
*** Destitution MPI ***
/*The entire household is considered deprived if no household member 
aged 10 years or older has completed at least one year of schooling. */
******************************************************************* 
gen	years_edu1 = (eduyears>=1)
replace years_edu1 = . if eduyears==.
bysort	hh_id: egen hh_years_edu_u = max(years_edu1)
replace hh_years_edu_u = . if hh_years_edu_u==0 & no_missing_edu==0
lab var hh_years_edu_u "Household has at least one member with 1 year of edu"


********************************************************************************
*** Step 2.2 Child School Attendance ***
********************************************************************************

/*For Jamaica JSLC 2010 b1 is the type of school that individual is attending 
the current year:
        1   NURSERY/DAYCARE(INCLUDE NEWBORN BABIES)
        2   BASIC/INFANT/KINDERGARTEN
        3   PRIMARY
        4   PREPARATORY
        5   ALL AGE SCHOOL(GRADES 1-6)
		6   ALL AGE SCHOOL(GRADES 7-9) 
        7   PRIMARY AND JUNIOR HIGH (GRADES 1-6)
        8   PRIMARY AND JUNIOR HIGH (GRADES 7-9)
		9   JUNIOR HIGH (GRADES 7 - 9)
        10  SECONDARY HIGH
        11  TECHNICAL
        12  VOCAT./AGR.
        13  UNIVERSITY
        14  OTHER TERTIARY (PUBLIC)
        15  OTHER TERTIARY (PRIVATE)
        16  ADULT LITERACY CLASSES
        17  ADULT EDUCATION/NIGHT
        18  SPECIAL SCHOOL
        19  NONE
        97  INDIVIDUAL NON-RESPONSE*/
		
gen	attendance = .
replace attendance = 1 if b1>=3 & b1<=18 
	//Currently attending formal education//
replace attendance = 0 if b1<=2  | b1==19
	//Not attending school or attending below formal (primary) education//	
replace attendance = 0 if age<5 | age>24 
	/*Replace attendance with '0' for individuals who are not of school age */
tab attendance, miss
label define lab_attend 1 "currently attending" 0 "not currently attending"
label values attendance lab_attend
label var attendance "Attended school during current school year"
tab b1 attendance, m


*** Standard MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 8. */ 
******************************************************************* 
gen	child_schoolage = (age>=6 & age<=14)
	/*Note: In Jamaica, the official school entrance age is 6 years.  
	  So, age range is 6-14 (=6+8)  
	  Source: http://data.uis.unesco.org/?ReportId=163 */
	

	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the school age children */
count if child_schoolage==1 & attendance==.
	//Understand how many eligible school aged children are not attending school 	
gen temp = 1 if child_schoolage==1 & attendance!=.
bysort hh_id: egen no_missing_atten = sum(temp)	
	/*Total school age children with no missing information on school 
	attendance */
gen temp2 = 1 if child_schoolage==1	
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are of school age
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)
	/*Identify whether there is missing information on school attendance for 
	more than 2/3 of the school age children */		
tab no_missing_atten, miss
label var no_missing_atten "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs
		
bysort	hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 

lab var hh_children_schoolage "Household has children in school age"


gen	child_not_atten = (attendance==0) if child_schoolage==1
replace child_not_atten = . if attendance==. & child_schoolage==1
bysort	hh_id: egen any_child_not_atten = max(child_not_atten)
gen	hh_child_atten = (any_child_not_atten==0) 
replace hh_child_atten = . if any_child_not_atten==.
replace hh_child_atten = 1 if hh_children_schoolage==0
replace hh_child_atten = . if hh_child_atten==1 & no_missing_atten==0 
lab var hh_child_atten "Household has all school age children up to class 8 in school"
tab hh_child_atten, miss


*** Destitution MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 6. */ 
******************************************************************* 
gen	child_schoolage_6 = (age>=6 & age<=12) 
	/*In Jamaica, the official school entrance age is 6 years.  
	  So, age range for destitution measure is 6-12 (=6+6) */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the children attending school up to 
	class 6 */	
count if child_schoolage_6==1 & attendance==.	
gen temp = 1 if child_schoolage_6==1 & attendance!=.
bysort hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_6==1	
bysort hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)			
tab no_missing_atten_u, miss
label var no_missing_atten_u "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs		
		
bysort	hh_id: egen hh_children_schoolage_6 = sum(child_schoolage_6)
replace hh_children_schoolage_6 = (hh_children_schoolage_6>0) 
lab var hh_children_schoolage_6 "Household has children in school age (6 years of school)"

gen	child_atten_6 = (attendance==1) if child_schoolage_6==1
replace child_atten_6 = . if attendance==. & child_schoolage_6==1
bysort	hh_id: egen any_child_atten_6 = max(child_atten_6)
gen	hh_child_atten_u = (any_child_atten_6==1) 
replace hh_child_atten_u = . if any_child_atten_6==.
replace hh_child_atten_u = 1 if hh_children_schoolage_6==0
replace hh_child_atten_u = . if hh_child_atten_u==0 & no_missing_atten_u==0 
lab var hh_child_atten_u "Household has at least one school age children up to class 6 in school"
tab hh_child_atten_u, miss


********************************************************************************
*** Step 2.3 Nutrition ***
********************************************************************************

********************************************************************************
*** Step 2.3a Child Nutrition ***
********************************************************************************

*** Child Underweight Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(underweight)
gen	hh_no_underweight = (temp==0) 
	//Takes value 1 if no child in the hh is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_underweight "Household has no child underweight - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(underweight_u)
gen	hh_no_underweight_u = (temp==0) 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
lab var hh_no_underweight_u "Destitute: Household has no child underweight"
drop temp


*** Child Stunting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(stunting)
gen	hh_no_stunting = (temp==0) 
	//Takes value 1 if no child in the hh is stunted
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_stunting "Household has no child stunted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(stunting_u)
gen	hh_no_stunting_u = (temp==0) 
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
lab var hh_no_stunting_u "Destitute: Household has no child stunted"
drop temp


*** Child Wasting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(wasting)
gen	hh_no_wasting = (temp==0) 
	//Takes value 1 if no child in the hh is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_wasting "Household has no child wasted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(wasting_u)
gen	hh_no_wasting_u = (temp==0) 
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
lab var hh_no_wasting_u "Destitute: Household has no child wasted"
drop temp


*** Child Either Underweight or Stunted Indicator ***
************************************************************************

*** Standard MPI ***
gen hh_no_uw_st = 1 if hh_no_stunting==1 & hh_no_underweight==1
replace hh_no_uw_st = 0 if hh_no_stunting==0 | hh_no_underweight==0
	//Takes value 0 if child in the hh is stunted or underweight 
replace hh_no_uw_st = . if hh_no_stunting==. & hh_no_underweight==.
replace hh_no_uw_st = 1 if no_child_eligible==1
	//Households with no eligible children will receive a value of 1 
lab var hh_no_uw_st "Household has no child underweight or stunted"


*** Destitution MPI  ***
gen hh_no_uw_st_u = 1 if hh_no_stunting_u==1 & hh_no_underweight_u==1
replace hh_no_uw_st_u = 0 if hh_no_stunting_u==0 | hh_no_underweight_u==0
replace hh_no_uw_st_u = . if hh_no_stunting_u==. & hh_no_underweight_u==.
replace hh_no_uw_st_u = 1 if no_child_eligible==1 
lab var hh_no_uw_st_u "Destitute: Household has no child underweight or stunted"


********************************************************************************
*** Step 2.3b Household Nutrition Indicator ***
********************************************************************************

*** Standard MPI ***
/* The indicator takes value 1 if the household has no child under 5 who 
has either height-for-age or weight-for-age that is under 2 stdev below 
the median. It also takes value 1 for the households that have no eligible 
children. The indicator takes a value of missing only if all eligible 
children have missing information in their respective nutrition variable. */
************************************************************************

gen	hh_nutrition_uw_st = 1
replace hh_nutrition_uw_st = 0 if hh_no_uw_st==0
replace hh_nutrition_uw_st = . if hh_no_uw_st==.
replace hh_nutrition_uw_st = 1 if no_child_eligible==1   
 	/*We replace households that do not have the applicable population, that is, 
	children 0-5, as non-deprived in nutrition*/		
lab var hh_nutrition_uw_st "Household has no individuals malnourished"
tab hh_nutrition_uw_st, miss


*** Destitution MPI ***
/* The indicator takes value 1 if the household has no child under 5 who 
has either height-for-age or weight-for-age that is under 2 stdev below 
the median. It also takes value 1 for the households that have no eligible 
children. The indicator takes a value of missing only if all eligible 
children have missing information in their respective nutrition variable. */
************************************************************************

gen	hh_nutrition_uw_st_u = 1
replace hh_nutrition_uw_st_u = 0 if hh_no_uw_st_u==0
replace hh_nutrition_uw_st_u = . if hh_no_uw_st_u==.
replace hh_nutrition_uw_st_u = 1 if no_child_eligible==1   
 	/*We replace households that do not have the applicable population, that is, 
	children 0-5, as non-deprived in nutrition*/		
lab var hh_nutrition_uw_st_u "Household has no individuals malnourished (destitution)"
tab hh_nutrition_uw_st_u, miss


********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************
	/*NOTE: Jamaica JSLC 2010 has no data on child mortality. As such the 
	indicator under this section is assigned with missing value */
			
gen child_mortality = .
			
gen hh_mortality_u18_5y = .	
lab var hh_mortality_u18_5y "Household had no under 18 child mortality in the last 5 years"

gen hh_mortality_u = .
lab var hh_mortality_u "Household had no child mortality"


********************************************************************************
*** Step 2.5 Electricity ***
********************************************************************************
lookfor light elect
codebook i27 i28, tab(100)

count if i28==0
	//320 observations
	
count if i27 !="01"
	//370 observations, with 10 missing info.

codebook i27, tab (99)
/* Q19. "What is the main source of lighting for this dwelling?"
		1 = Electricity
		2 = Kerosene
		3 = Other
		4 = None */

*** Standard MPI ***
/*Members of the household are considered deprived 
if the household has no electricity */
****************************************		
destring i27, replace
gen electricity = (i27==1) 
	/*Main source of lightening assumes that they have no electricity when they 
	use kerosene or other */
codebook electricity, tab (10)
replace electricity = . if i27 >= 97 | i27== .  
label var electricity "Electricity"
tab electricity,m

*** Destitution MPI  ***
*** (same as standard MPI) ***
****************************************

gen electricity_u = electricity
label var electricity_u "Household has electricity"


********************************************************************************
*** Step 2.6 Sanitation ***
********************************************************************************

/*
Improved sanitation facilities include flush or pour flush toilets to sewer 
systems, septic tanks or pit latrines, ventilated improved pit latrines, pit 
latrines with a slab, and composting toilets. These facilities are only 
considered improved if it is private, that is, it is not shared with other 
households.
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-02-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

destring i5, replace
codebook i5, tab (99)
	/* Q5. "What kind of toilet facilities are used by your household?"
				1 = W.C linked to sewer system
				2 = W.C not linked
				3 = Pit
				4 = Other
				5 = None */

	//Type of toilet 
gen toilet = i5  
replace toilet=. if toilet == 97
lab define lab_t 1"W.C linked to sewer system" 2"W.C not linked" 3"Pit" ///
				 4"Other" 5"None"
lab val toilet lab_t
gen toilet_facilities = (i5 != 5)
	//i5: do not have toilet facility	
destring i6, replace
codebook i6
	// 1 "Exclusive use", 2 "Shared"
gen shared_toilet = (i6 == 2)
	//1=shared toilet
replace shared_toilet = . if i6== .  
tab i6 shared_toilet, m	

		
*** Standard MPI ***
/*Members of the household are considered deprived if the household's 
sanitation facility is not improved (according to the SDG guideline) 
or it is improved but shared with other households*/
********************************************************************
codebook toilet, tab(99) 
	
gen toilet_mdg = 0 if toilet_facilities==0 | shared_toilet==1 | toilet==3
	//Do not have toilet facility or is a pit; if there is toilet facility, it is shared 
	
replace toilet_mdg = 1 if (toilet<= 2 & shared_toilet!=1)
	//Have water closet toilets that are linked / not linked to the sewer
	
replace toilet_mdg = . if toilet==. & shared_toilet==. & toilet_facilities!=0
	/* Note: There are 0 individuals who reported that the toilet facilities 
	are inside their dwelling, but did not provide information on the type of 
	toilet facility or whether it is shared or non-shared facility. For the 
	purpose of the global MPI, we have coded these individuals as missing. */

lab var toilet_mdg "Household has improved sanitation"
tab toilet_facilities toilet_mdg, miss
tab toilet toilet_mdg, m


*** Destitution MPI ***
/*Members of the household are considered deprived if household practises 
open defecation or uses other unidentifiable sanitation practises */
********************************************************************
gen	toilet_u = 1

replace toilet_u = 0 if toilet==4 | toilet==5 | toilet_facilities==0  
	//31 open defecation/no toilet facility, 96 Other
replace toilet_u = . if toilet==. & shared_toilet==. & toilet_facilities!=0

lab var toilet_u "Household does not practise open defecation or others"
tab toilet toilet_u, miss


********************************************************************************
*** Step 2.7 Drinking Water  ***
********************************************************************************

/*
Improved drinking water sources include the following: piped water into 
dwelling, yard or plot; public taps or standpipes; boreholes or tubewells; 
protected dug wells; protected springs; packaged water; delivered water and 
rainwater which is located on premises or is less than a 30-minute walk from 
home roundtrip. 
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-01-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report. */

lookfor water distance
destring i20, replace
destring i26_1, replace
destring i26_2, replace

clonevar water = i20  

gen ndwater = .

clonevar timetowater = i26_1  
replace timetowater = . if timetowater > 999
replace timetowater = timetowater * 1000 if i26_2 == 1 
	//convert from kilometers to meters
codebook timetowater, tab (999)

	/* In the case of Jamaica JSLC 2010, distance to water is provided in 
	kilometers/meters. For the purpose of the global MPI, we have converted 
	these distances to meters. Every 1000 metres represents 30 minutes 
	roundtrip to the water source.

	In Jamaica, the National Water Commission (NWC) and the Ministry of 
	Health restrict the definition of imporved drinking water sources to only 
	treated water, that is, household connection (piped into yard/dwelling) and 
	public standpipe. The country report (p.74 of pdf(5.6)) states that "While 
	uncontaminated water, which may include untreated sources (rivers, spring, 
	rainwater, tanks, trucked water (non-NWC), bottled and other) may be safe, 
	the National Water Commission and the Ministry of Health restrict the local
	definition of “Safe Water” to only treated water.
	
	The Global MPI 2018 using the JSLC 2014 survey considered trucked 
	water (NWC) and bottled water as improved sources based on the report.*/
	
*** Standard MPI ***
/* Members of the household are considered deprived if the household 
does not have access to improved drinking water (according to the SDG 
guideline) or safe drinking water is at least a 30-minute walk from 
home, roundtrip */
********************************************************************
codebook water, tab(99)
	/* Q.20 "What is the main source of drinking water for your household?"
				1 = Indoor tap/pipe
				2 = Outside private pipe/tap
				3 = Public standpipe
				4 = Well
				5 = River, lake, spring,pond
				6 = Rainwater(tank) piped into dwelling (PID)
				7 = Rainwater(tank) not piped into dwelling (NPID)
				8 = Trucked water(NWC) PID
				9 = Trucked water(NWC) NPID
				10 = Trucked water(Priv.) PID
				11 = Trucked water(Priv.) NPID
				12 = Bottled water
				13 = Other */
				
				
gen water_mdg = 1 if (water <=3)| water==8 | water==9 | water==12 
	/*Non deprived if water is indoor tap/pipe; outside private tap/pipe; 
	  public standpipe; trucked water(NWC); bottled water */
	  
replace water_mdg = 0 if water==4  | water==5  | water==6 | water==7 | ///
						 water==10 | water==11 | water==13 
	/*Deprived if it is well; river/lake/spring/pond; rainwater (tank);
	  trucked water (private companies)*/

replace water_mdg = 0 if water_mdg==1 & timetowater>=1000 & timetowater!=.  
	//Deprived if water is at more than 1000 meters (30 minutes walk, roundtrip)
	
replace water_mdg = . if water >= 97 

lab var water_mdg "Household has drinking water with MDG standards (considering distance)"
tab water water_mdg, miss


*** Destitution MPI ***
/* Members of the household is identified as destitute if household 
does not have access to safe drinking water, or safe water is more 
than 45 minute walk from home, round trip.*/
********************************************************************
gen water_u = 1 if (water <=3)
	/*Non deprived if water is indoor tap/pipe; outside private tap/pipe; 
	  public standpipe */
	  
replace water_u = 0 if water>=4 
	/*Deprived if it is well; river/lake/spring/pond; rainwater (tank);
	  trucked water (private companies), trucked water(NWC); bottled water*/

replace water_u = 0 if water_u==1 & timetowater>1500 & timetowater!=.  
	//Deprived if water is at more than 1500 meters (45 minutes walk, roundtrip)
	
replace water_u = . if water >= 97 

lab var water_u "Household has safe drinking water (considering distance)"
tab water water_u, miss


********************************************************************************
*** Step 2.8 Housing ***
********************************************************************************
/*Note: In Jamaica, housing was only constructed using information on wall, as 
the survey did not collect data on floor and roof. The country report included 
only walls made of concrete block & steel in the housing quality index. The 
report states that this type of walls has durability for withstanding the 
elements of weather and for providing occupants with a greater level of 
security (p.69 (5.1). Following the country report, we identify only walls made 
of concrete block & steel as non-deprived. All other wall materials including 
wood, stone, brick, cocrete nog, wattle/adobe and others as non-improved. */


/* Members of the household are considered deprived if the household 
has a dirt, sand or dung floor. 
Jamaica JSLC 2010 has no data on floor. */
gen floor = .
gen floor_imp = .
lab var floor_imp "Household has floor that it is not earth/sand/dung"


/* Members of the household are considered deprived if the household has wall 
made of natural & low quality materials. */
lookfor wall
destring i2, replace
clonevar wall = i2
label def walls 1 "Wood" 2 "Stone" 3 "Brick" 4"Concrete nog" ///
				5 "Concrete block & steel" 6 "Wattle/Adobe" 7"Other"
label values wall walls
codebook wall, tab(99)	
gen	wall_imp = 1 
replace wall_imp = 0 if wall==1 | wall==2 | wall==3 | wall==4 | ///
						wall==6 | wall==7  
replace wall_imp = . if wall==97
lab var wall_imp "Household has wall that it is not of low quality materials"
tab wall wall_imp, miss	
	

/* Members of the household are considered deprived if the household has roof 
made of natural & low quality materials.
Jamaica JSLC 2014 has no data on roof.*/	
gen roof = .	
gen	roof_imp = . 
lab var roof_imp "Household has roof that it is not of low quality materials"


*** Standard MPI ***
/* Members of the household is deprived in housing if the roof, 
floor OR walls are constructed from low quality materials.*/
**************************************************************
gen housing_1 = 1
replace housing_1 = 0 if floor_imp==0 | wall_imp==0 | roof_imp==0
replace housing_1 = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_1 "Household has roof, floor & walls that it is not low quality material"
tab housing_1, miss


*** Destitution MPI ***
/* Members of the household is deprived in housing if two out 
of three components (roof and walls; OR floor and walls; OR 
roof and floor) the are constructed from low quality materials. */
**************************************************************
gen housing_u = 1
replace housing_u = 0 if (floor_imp==0 & wall_imp==0 & roof_imp==1) | ///
						 (floor_imp==0 & wall_imp==1 & roof_imp==0) | ///
						 (floor_imp==1 & wall_imp==0 & roof_imp==0) | ///
						 (floor_imp==0 & wall_imp==0 & roof_imp==0)
replace housing_u = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_u "Household has one of three aspects(either roof,floor/walls) that is not low quality material"
tab housing_u, miss


********************************************************************************
*** Step 2.9 Cooking Fuel ***
********************************************************************************

/*
Solid fuel are solid materials burned as fuels, which includes coal as well as 
solid biomass fuels (wood, animal dung, crop wastes and charcoal). 

Source: 
https://apps.who.int/iris/bitstream/handle/10665/141496/9789241548885_eng.pdf
*/

destring i39, replace
clonevar cookingfuel = i39  
label def cookingfuel 1 "Gas" 2 "Electricity" 3 "Wood" 4"Kerosene" ///
				      5 "Charcoal" 6 "Biogas" 7 "Solar" 97 "Other" 9 "Other"
label values cookingfuel cookingfuel
codebook cookingfuel, tab(99)

*** Standard MPI ***
/* Members of the household are considered deprived if the 
household uses solid fuels and solid biomass fuels for cooking. */
*****************************************************************
gen	cooking_mdg = 1
replace cooking_mdg = 0 if cookingfuel == 3  | cookingfuel == 5 |  ///
						   cookingfuel == 97 | cookingfuel == 9 
	//3=wood; 5=charcoal
replace cooking_mdg = . if cookingfuel==.
lab var cooking_mdg "Household cooks with clean fuels"
tab cookingfuel cooking_mdg, miss
	//Jamaica JSLC 2010 codes "other" as deprived.


*** Destitution MPI ***
*** (same as standard MPI) ***
****************************************	
gen	cooking_u = cooking_mdg
lab var cooking_u "Household cooks with clean fuels"


********************************************************************************
*** Step 2.10 Assets ownership ***
********************************************************************************

/* Members of the household are considered deprived if the household does not 
own more than one of: radio, TV, telephone, bike, motorbike or refrigerator and 
does not own a car or truck. */

foreach var in j610 j607 i30_1 i30_2 i30_3 j604 j617 j615 j616 i33 i34  {
destring `var', replace
}
//
codebook j610 j607 i30_1 i30_2 i30_3 j604 j617 j615 j616 i33 i34


*** Television/LCD TV/plasma TV/color TV/black & white tv
clonevar television = j610
lab var television "Household has television"


***	Radio/walkman/stereo/kindle
clonevar radio = j607
lab var radio "Household has radio"


***	Handphone/telephone/iphone/mobilephone/ipod
clonevar telephone =  i30_1 
replace telephone=1 if telephone!=1 & i30_2==1	
replace telephone=1 if telephone!=1 & i30_3==1	
lab var telephone "Household has telephone (landline/mobilephone)"


***	Refrigerator/icebox/fridge
clonevar refrigerator = j604
lab var refrigerator "Household has refrigerator"


***	Car/van/lorry/truck
clonevar car = j617 
lab var car "Household has car"	


***	Bicycle/cycle rickshaw
clonevar bicycle = j615 
lab var bicycle "Household has bicycle"	


***	Motorbike/motorized bike/autorickshaw
clonevar motorbike = j616 
lab var motorbike "Household has motorbike"


***	Computer/laptop/tablet
clonevar computer = i33
replace computer = 1 if i33<=3
replace computer = 0 if i33==4
lab var computer "Household has computer"


*** Animal cart
gen animal_cart = .
lab var animal_cart "Household has animal cart"	


foreach var in television radio telephone refrigerator car ///
			   bicycle motorbike computer animal_cart  {
replace `var' = 0 if `var'==2 
	//0=no; 1=yes
replace `var' = . if `var'==9 | `var'==97
}



*** Standard MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own more than one of: radio, TV, telephone, bike, motorbike, 
refrigerator, computer or animal cart and does not own a car or truck.*/
*****************************************************************************
egen n_small_assets2 = rowtotal(television radio telephone refrigerator bicycle motorbike computer animal_cart), missing
lab var n_small_assets2 "Household Number of Small Assets Owned" 
   
gen hh_assets2 = (car==1 | n_small_assets2 > 1) 
replace hh_assets2 = . if car==. & n_small_assets2==.
lab var hh_assets2 "Household Asset Ownership: HH has car or more than 1 small assets incl computer & animal cart"
tab hh_assets2, m


*** Destitution MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own any assets.*/
*****************************************************************************	
gen	hh_assets2_u = (car==1 | n_small_assets2>0)
replace hh_assets2_u = . if car==. & n_small_assets2==.
lab var hh_assets2_u "Household Asset Ownership: HH has car or at least 1 small assets incl computer & animal cart"


********************************************************************************
*** Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************
	
	
	//Retain data on sampling design: 	
clonevar strata = region
clonevar psu = district
destring psu, replace
label var psu "Primary sampling unit"
label var strata "Sample strata"



	//Retain year, month & date of interview:
clonevar year_interview = int_day 	
clonevar month_interview = int_mth 
clonevar date_interview = int_yer
destring year_interview, replace
destring month_interview, replace
destring date_interview, replace

 
	//Generate presence of subsample
gen subsample = . 
 
		
*** Keep main variables require for MPI calculation ***
keep hh_id ind_id subsample strata psu weight weight_ch relationship ///
area age agec7 agec4 agec2 headship region sex marital hhsize ///
child_eligible no_child_eligible /// 
religion_wom religion_men religion_hh ethnic_wom ethnic_men ethnic_hh ///
insurance_wom insurance_men year_interview month_interview date_interview /// 
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
hh_no_underweight hh_no_stunting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet toilet_mdg toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u

 
*** Order file	***
order hh_id ind_id subsample strata psu weight weight_ch relationship ///
area age agec7 agec4 agec2 headship region sex marital hhsize ///
child_eligible no_child_eligible /// 
religion_wom religion_men religion_hh ethnic_wom ethnic_men ethnic_hh ///
insurance_wom insurance_men year_interview month_interview date_interview /// 
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
hh_no_underweight hh_no_stunting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet toilet_mdg toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u



*** Rename key global MPI indicators for estimation ***
recode hh_mortality_u18_5y  (0=1)(1=0) , gen(d_cm)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ)
recode electricity 			(0=1)(1=0) , gen(d_elct)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani)
recode housing_1 			(0=1)(1=0) , gen(d_hsg)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl)
recode hh_assets2 			(0=1)(1=0) , gen(d_asst)
 
 
*** Rename key global MPI indicators for destitution estimation ***
recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ)
recode electricity_u		(0=1)(1=0) , gen(dst_elct)
recode water_u 				(0=1)(1=0) , gen(dst_wtr)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl)
recode hh_assets2_u 		(0=1)(1=0) , gen(dst_asst) 



*** Rename indicators for changes over time estimation ***	
recode hh_mortality_u18_5y  (0=1)(1=0) , gen(d_cm_01)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr_01)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt_01)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ_01)
recode electricity 			(0=1)(1=0) , gen(d_elct_01)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr_01)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani_01)
recode housing_1 			(0=1)(1=0) , gen(d_hsg_01)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl_01)
recode hh_assets2 			(0=1)(1=0) , gen(d_asst_01)	
	

recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm_01)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr_01)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt_01)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ_01)
recode electricity_u		(0=1)(1=0) , gen(dst_elct_01)
recode water_u	 			(0=1)(1=0) , gen(dst_wtr_01)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani_01)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg_01)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl_01)
recode hh_assets2_u 		(0=1)(1=0) , gen(dst_asst_01) 


	/*In this survey, the harmonised 'region_01' variable is the 
	same as the standardised 'region' variable.*/	
clonevar region_01 = region 



*** Total number of missing values  for each variable *** 
mdesc psu strata area age headship region ///
d_cm d_nutr d_satt d_educ d_elct d_wtr d_sani d_hsg d_ckfl d_asst



mdesc psu strata area age headship region_01 ///
d_cm_01 d_nutr_01 d_satt_01 d_educ_01 ///
d_elct_01 d_wtr_01 d_sani_01 d_hsg_01 d_ckfl_01 d_asst_01



*** Generate coutry and survey details for estimation ***
char _dta[cty] "Jamaica"
char _dta[ccty] "JAM"
char _dta[year] "2010" 	
char _dta[survey] "JSLC"
char _dta[ccnum] "388"
char _dta[type] "micro"



*** Sort, compress and save data for estimation ***
sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/jam_jslc10.dta", replace

	//Erase files from folder:
erase "$path_out/rec001.dta"
erase "$path_out/rec026.dta"
erase "$path_out/rec027.dta"
erase "$path_out/hhsizes.dta"
erase "$path_out/rec007.dta"
erase "$path_out/rec009.dta"
erase "$path_out/rec010.dta"
erase "$path_out/rec006.dta"
erase "$path_out/rec003.dta"
