********************************************************************************
/*
Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
Global Multidimensional Poverty Index 2023 - India DHS 2019-2021 [STATA do-file]. 
Available from OPHI website: http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk
*/
********************************************************************************

clear all 
set more off
set maxvar 10000


global path_in "../rdta/India DHS 2019-20" 	      // folder path
global path_out "cdta"
global path_ado "ado"


********************************************************************************
*** Step 1: Data preparation 
********************************************************************************
		
/* NOTE: India DHS (NFHS) 2019-21: Height and weight were measured for all 
children age 0-59 months, women age 15-49, and a subsample of men age 15-54 who
were selected in the state module (p.4). */ 


********************************************************************************
**# Step 1.1 CHILD UNDER 5
********************************************************************************

use "$path_in/IAPR7AFL.DTA", clear 

gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g



*** Eligible children ***
tab hv120,miss	
	//224,349 children are eligible for measurement. 	
count if hc1!=.
	//224,349 children have information on age in months
tab hv105 if hc1!=.
	//All are within the 5 year age group.
tab hc13 if hv120==1, miss
	//213,902 (95.34%) of the 224,349 children have been measured.
tab hc13 if hc3<=9990, miss
tab hc13 if hc2<=9990, miss

gen child_nutr = (hv120==1)
keep if child_nutr==1
count	
	// 224,349 children under 5		
	
	
*** Variable: SEX (1=male; 2=female) ***
desc hc27 hv104
compare hc27 hv104
clonevar gender = hc27
tab gender


*** Variable: AGE ***
tab hc1, miss 
clonevar age_months = hc1  
sum age_months

gen mdate = mdy(hc18, hc17, hc19)
gen bdate = mdy(hc30, hc16, hc31) if hc16 <= 31
	//Calculate birth date in days from date of interview
replace bdate = mdy(hc30, 15, hc31) if hc16 > 31 
	//If date of birth of child has been expressed as more than 31, we use 15
gen age = (mdate-bdate)/30.4375 
	//Calculate age in months with days expressed as decimals
sum age
replace age = age_months if age<0 | age==.

gen  str6 ageunit = "months" 
lab var ageunit "months"

	
*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook hc2, tab (9999)
gen	weight = hc2/10 
tab hc2 if hc2>9990, miss nol   
replace weight = . if hc2>=9990 
tab	hc13 if hc2>=9990 | hc2==., miss 
sum weight


*** Variable: HEIGHT (CENTIMETERS)
codebook hc3, tab (9999)
gen	height = hc3/10 
tab hc3 if hc3>9990, miss nol   
replace height = . if hc3>=9990 
tab	hc13 if hc3>=9990 | hc3==., miss 
sum height


*** Variable: MEASURED STANDING/LYING DOWN ***	
codebook hc15
gen measure = "l" if hc15==1 
replace measure = "h" if hc15==2 
replace measure = " " if hc15==9 | hc15==0 | hc15==. 
tab measure

*** Variable: OEDEMA ***
gen  oedema = "n"  

*** Variable: SAMPLING WEIGHT ***
gen  sw = 1	


*** z-score ***
adopath + "$path_ado/igrowup_stata"
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "children_nutri_ind" 
lab var datalab "Working file"
	
igrowup_restricted reflib datalib datalab gender age ageunit weight height ///
measure oedema sw

use "$path_out/children_nutri_ind_z_rc.dta", clear 

	
*** Standard MPI indicator ***	
gen	underweight = (_zwei < -2.0) 
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "child is underweight"
ta underweight [aw=hv005], m


gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "child is stunted"
ta stunting [aw=hv005], m


gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "child is wasted"
ta wasting [aw=hv005], m


*** Destitution MPI indicator  ***	
gen	underweight_u = (_zwei < -3.0) 
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "dst: child is underweight"


gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "dst: child is stunted"


gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "dst: child is wasted"


count if _fwei==1 | _flen==1 	
	//Extreme z-scores which are biologically implausible replaced: 5,757 obs. 
 
 	
gen weight_ch = hv005/1000000
label var weight_ch "sample weight child under 5"

 
	//Retain relevant variables:
keep ind_id child_nutr weight_ch underweight* stunting* wasting* 
order ind_id child_nutr weight_ch underweight* stunting* wasting* 
sort ind_id
save "$path_out/IND19-21_PR_child.dta", replace
	
	
	//Erase files from folder:
erase "$path_out/children_nutri_ind_z_rc.xls"
erase "$path_out/children_nutri_ind_prev_rc.xls"
erase "$path_out/children_nutri_ind_z_rc.dta"


********************************************************************************
**# Step 1.2  BIRTH RECODE 
*** (female 15-49 years who ever gave birth)  
********************************************************************************

use "$path_in/IABR7AFL.DTA", clear
	
gen double ind_id = v001*1000000 + v002*100 + v003 
format ind_id %20.0g
label var ind_id "individual id"


gen women_bh = 1 
lab var women_bh "female with birth history" 


desc b3 b7 v008	
gen date_death = b3 + b7
	//Date of death = date of birth (b3) + age at death (b7)
gen mdead_svy = v008 - date_death
	//Months dead from survey = Date of interview (v008) - date of death
gen ydead_svy = mdead_svy/12
lab var ydead_svy "total years dead from svy year"	

	
gen age_death = b7	
label var age_death "age at death (months)"
tab age_death, miss

	
codebook b5, tab (9)	
gen child_died = 1 if b5==0
	//1=child dead; 0=child alive
replace child_died = 0 if b5==1
replace child_died = . if b5==.
lab define lab_died 1 "child has died" 0 "child is alive"
lab values child_died lab_died
tab b5 child_died, miss


	/* For each woman, sum the number of children who died and compare to 
	the number of sons/daughters whom they reported have died. */
bysort ind_id: egen tot_child_died = sum(child_died) 
egen tot_child_died_2 = rsum(v206 v207)
	//v206: sons who have died; v207: daughters who have died	
compare tot_child_died tot_child_died_2
	//India DHS 2019-21: the figures are identical. 

			
	//Identify child under 18 mortality in the last 5 years
gen child18_died = child_died 
replace child18_died=0 if age_death>=216 & age_death!=.
label define lab_u18died 1 "child u18 has died" 0 "child is alive/died but older"
label values child18_died lab_u18died
ta child18_died, m	
			
bysort ind_id: egen tot_child18_died_5y=sum(child18_died) if ydead_svy<=5
	/*Total number of children under 18 who died in the past 5 years 
	prior to the interview date */	
	
replace tot_child18_died_5y=0 if tot_child18_died_5y==. & tot_child_died>=0 & tot_child_died!=.
	/*All children who are alive or who died longer than 5 years from the 
	interview date are replaced as '0'*/
	
replace tot_child18_died_5y=. if child18_died==1 & ydead_svy==.
	//Replace as '.' if there is no information on when the child died  

tab tot_child_died tot_child18_died_5y, miss

bysort ind_id: egen childu18_died_per_wom_5y = max(tot_child18_died_5y)
lab var childu18_died_per_wom_5y "total under 18 death for each women in the last 5 years"
	

	//Keep one observation per women
bysort ind_id: gen id=1 if _n==1
keep if id==1
drop id
duplicates report ind_id 

	
	//Retain relevant variables
keep ind_id women_bh childu18_died_per_wom_5y 
order ind_id women_bh childu18_died_per_wom_5y
sort ind_id
save "$path_out/IND19-21_BR.dta", replace	


********************************************************************************
**# Step 1.3  WOMEN's RECODE  
*** (eligible female 15-49 years)
********************************************************************************

use "$path_in/IAIR7AFL.DTA", clear


gen double ind_id = v001*1000000 + v002*100 + v003 
format ind_id %20.0g
label var ind_id "Individual ID"

duplicates report ind_id

tab v012, miss
codebook v201 v206 v207,tab (999)
	//Mortality question was collected from women 15-49 years.

gen women_IR=1 
	//Id variable for eligible women 15-49 years: 724,115 
	

keep ind_id women_IR v003 v005 v012 v201 v206 v207  
order ind_id women_IR v003 v005 v012 v201 v206 v207 
sort ind_id
save "$path_out/IND19-21_IR.dta", replace	


********************************************************************************
**# Step 1.4  BMI-FOR-AGE
*** (girls 15-19 years)
********************************************************************************

use "$path_in/IAPR7AFL.DTA", clear
		
gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g


*** Identify anthropometric sample for girls
count if hv105>=15 & hv105<=19 & hv104==2 
tab shbsel if hv105>=15 & hv105<=19 & hv104==2, m
keep if hv105>=15 & hv105<=19 & hv104==2 
count
	//Total girls 15-19 years: 129,817


*** Variable: SEX ***
codebook hv104, tab (9)
clonevar gender = hv104
	//2:female 


*** Variable: AGE ***
lookfor hv807c hv008 ha32
gen age_month = hv807c - ha32
lab var age_month "Age in months, girls 15-19 years"
sum age_month
count if age_month > 228
	/* For 23,353 girls, we find that the age in months is 
	beyond 228 months. In this secton, while calculating the 
	z-scores, these cases will be excluded. However, in section 2.3, 
	we will take the BMI information of these girls. */

	
*** Variable: AGE UNIT ***
gen str6 ageunit = "months" 
lab var ageunit "Months"

			
*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook ha2, tab (9999)
count if ha2>9990 
tab ha13 if ha2>9990, miss
gen weight = ha2/10 if ha2<9990
sum weight


*** Variable: HEIGHT (CENTIMETERS)	
codebook ha3, tab (9999)
count if ha3>9990 
tab ha13 if ha3>9990, miss
gen height = ha3/10 if ha3<9990
sum height


*** Variable: OEDEMA
gen oedema = "n"  


*** Variable: SAMPLING WEIGHT ***
gen sw = 1


*** z-score ***
adopath + "$path_ado/who2007_stata"		
gen str100 reflib = "$path_ado/who2007_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "girl_nutri_ind" 
lab var datalab "Working file"
	
who2007 reflib datalib datalab gender age_month ageunit weight height oedema sw

use "$path_out/girl_nutri_ind_z.dta", clear 
	
gen	z_bmi = _zbfa
replace z_bmi = . if _fbfa==1 
lab var z_bmi "z-score bmi-for-age"


*** Standard MPI indicator ***
gen	low_bmiage = (z_bmi < -2.0) 
replace low_bmiage = . if z_bmi==.
lab var low_bmiage "Adolescent girls have low bmi"


*** Destitution indicator ***
gen	low_bmiage_u = (z_bmi < -3.0)
replace low_bmiage_u = . if z_bmi==.
lab var low_bmiage_u "DST: Adolescent girls have low bmi"

tab low_bmiage, miss
tab low_bmiage_u, miss


	//Retain relevant variables:
gen girl_PR=1 	
keep ind_id girl_PR age_month low_bmiage*
order ind_id girl_PR age_month low_bmiage*
sort ind_id
save "$path_out/IND19-21_PR_girls.dta", replace
 	

	//Erase files from folder:
erase "$path_out/girl_nutri_ind_z.xls"
erase "$path_out/girl_nutri_ind_prev.xls"
erase "$path_out/girl_nutri_ind_z.dta"


********************************************************************************
**# Step 1.5 MEN'S RECODE  
***(eligible man 15-54 years) 
********************************************************************************

use "$path_in/IAMR7AFL.DTA", clear 
	
gen double ind_id = mv001*1000000 + mv002*100 + mv003 	
format ind_id %20.0g
label var ind_id "Individual ID"

duplicates report ind_id

tab mv012, miss
codebook mv201 mv206 mv207,tab (999)
	/* India DHS 2019-21: Fertility and mortality question 
	was collected from men 15-54 years. */

gen men_MR=1 	
	//Identification variable for observations in MR recode: 101,839 

	
keep ind_id men_MR mv003 mv005 mv012 mv201 mv206 mv207  
order ind_id men_MR mv003 mv005 mv012 mv201 mv206 mv207 
sort ind_id
save "$path_out/IND19-21_MR.dta", replace


********************************************************************************
**# Step 1.6  BMI-FOR-AGE 
*** (boys 15-19 years)
********************************************************************************

use "$path_in/IAPR7AFL.DTA", clear

gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g
label var ind_id "Individual ID"


*** Identify anthropometric sample for boys
count if hv105>=15 & hv105<=19 & hv104==1 
count if hv105>=15 & hv105<=19 & hv104==1 & hv027==1 
tab shbsel hv027 if hv105>=15 & hv105<=19 & hv104==1, miss
keep if hv105>=15 & hv105<=19 & hv104==1 & hv027==1 
count
	//Total boys 15-19 years: 18,348

	
*** Variable: SEX ***
codebook hv104, tab (9)
clonevar gender = hv104
	//1: male; 2:female 


*** Variable: AGE ***
gen age_month_b = hv807c - hb32
lab var age_month_b "Age in months, boys 15-19 years"
sum age_month_b
count if age_month_b > 228	
	/* For 3,101 boys, we find that the age in months is beyond 
	228 months. In this secton, while calculating the z-scores, 
	these cases will be excluded. However, in section 2.3, we 
	will take the BMI information of these boys. */

	
*** Variable: AGE UNIT ***
gen str6 ageunit = "months" 
lab var ageunit "Months"

			
*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook hb2, tab (9999)
count if hb2>9990 
tab hb13 if hb2>9990, miss
gen weight = hb2/10 if hb2<9990	
sum weight


*** Variable: HEIGHT (CENTIMETERS)	
codebook hb3, tab (9999)
count if hb3>9990 
tab hb13 if hb3>9990, miss
gen height = hb3/10 if hb3<9990
sum height


*** Variable: OEDEMA
gen oedema = "n"  


*** Variable: SAMPLING WEIGHT ***
gen sw = 1


*** z-score ***	
adopath + "$path_ado/who2007_stata"		
gen str100 reflib = "$path_ado/who2007_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "boy_nutri_ind" 
lab var datalab "Working file"
	

who2007 reflib datalib datalab gender age_month_b ageunit weight height oedema sw


use "$path_out/boy_nutri_ind_z.dta", clear 
	
gen	z_bmi = _zbfa
replace z_bmi = . if _fbfa==1 
lab var z_bmi "z-score bmi-for-age"


*** Standard MPI indicator ***	
gen	low_bmiage_b = (z_bmi < -2.0) 
replace low_bmiage_b = . if z_bmi==.
lab var low_bmiage_b "Adolescent boys have low bmi"


*** Destitution indicator ***	
gen	low_bmiage_b_u = (z_bmi < -3.0)
replace low_bmiage_b_u = . if z_bmi==.
lab var low_bmiage_b_u "DST: Adolescent boys have low bmi"


tab low_bmiage_b, miss
tab low_bmiage_b_u, miss


gen boy_PR=1 
	//Identification variable for boys 15-19 years in PR recode: 20,087 


	//Retain relevant variables:	
keep ind_id boy_PR age_month_b low_bmiage*
order ind_id boy_PR age_month_b low_bmiage*
sort ind_id
save "$path_out/IND19-21_PR_boys.dta", replace
 	

	//Erase files from folder:
erase "$path_out/boy_nutri_ind_z.xls"
erase "$path_out/boy_nutri_ind_prev.xls"
erase "$path_out/boy_nutri_ind_z.dta"


********************************************************************************
**# Step 1.7  PR - HOUSEHOLD MEMBER'S RECODE 
********************************************************************************

use "$path_in/IAPR7AFL.DTA", clear


gen double hh_id = hv001*10000 + hv002 
format hh_id %20.0g
label var hh_id "household id"



gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g
label var ind_id "individual id"



sort hh_id ind_id


bys hh_id: gen id = _n
count if id==1			// 636,189 hh - matches report


********************************************************************************
**# Step 1.8 DATA MERGING 
******************************************************************************** 
 

merge 1:1 ind_id using "$path_out/IND19-21_BR.dta"  	 // bh recode
drop _merge
erase "$path_out/IND19-21_BR.dta"



merge 1:1 ind_id using "$path_out/IND19-21_IR.dta"  	// wom recode
ta women_IR hv117, m col
ta ha65 if hv117==1 & women_IR ==., m
drop _merge
erase "$path_out/IND19-21_IR.dta"

count if women_bh==1   
count if v201!=0 & v201!=. & women_IR==1
/* note: qc - check if the number of women in BR recode matches the 
number of those who provided birth history information in IR recode. */

count if hv117==1
count if women_bh==1 | v201==0
count if (women_bh==1 | v201==0) & hv117==1
ta v201 if hv117==1, m
ta v201 ha65 if hv117==1, m
/* note: qc - check if the number of women in BR and IR recode who 
provided birth history information matches with the number of eligible 
women identified by hv117. */

/* note: some 3.1% eligible women did not provide information on their birth 
history as they were not interviewed. This will result in missing value 
for the child mortality indicator that we will construct later. */




merge 1:1 ind_id using "$path_out/IND19-21_PR_girls.dta"   	// girls 15-19 y
ta girl_PR shbsel if hv105>=15 & hv105<=19 & hv104==2, m col
drop _merge
erase "$path_out/IND19-21_PR_girls.dta"	
	
	

	
merge 1:1 ind_id using "$path_out/IND19-21_MR.dta"  	// male recode
ta men_MR hv118 if hv027==1, m col
ta hb65 if hv118==1 & men_MR ==., m
drop _merge
erase "$path_out/IND19-21_MR.dta"



merge 1:1 ind_id using "$path_out/IND19-21_PR_boys.dta"		// boys 15-19 y
ta boy_PR hv027 if hv105>=15 & hv105<=19 & hv104==1, m col
drop _merge
erase "$path_out/IND19-21_PR_boys.dta"	



merge 1:1 ind_id using "$path_out/IND19-21_PR_child.dta"   // child under 5
ta hv120, m
ta hc13 if hv120==1, m
drop _merge
erase "$path_out/IND19-21_PR_child.dta"



sort ind_id


********************************************************************************
**# Step 1.9 DE JURE HOUSEHOLD MEMBERS ***
********************************************************************************

codebook hv102
drop if hv102!=1 
	/* India DHS 2019-21: 48,023 (1.69%) individuals who were 
	non-usual residents were dropped from the sample. */


********************************************************************************
**# Step 1.10 ANTHROPOMETRIC SUBSAMPLE ***
********************************************************************************

	// no subsample

		
********************************************************************************
**# Step 1.11 CONTROL VARIABLES
********************************************************************************


count if hv104==2 & hv105>=15 & hv105<=49   		// fem eligible for nutr
tab shbsel if hv105>=15 & hv105<=49 & hv104==2, m	
count if ha13!=.
	/* note: the ha13 variable (results of measurement) indicate that 1,041,650 
	women eligible for measurement. This is because women aged 50 and older 
	have been captured under this variable (under the category other). */ 	 
gen fem_nutri_eligible = (hv105>=15 & hv105<=49 & hv104==2)
bys hh_id: egen hh_n_fem_nutri_eligible = sum(fem_nutri_eligible) 	
gen	no_fem_nutri_eligible = (hh_n_fem_nutri_eligible==0)
lab var no_fem_nutri_eligible "hh has no eligible woman for nutr"
drop hh_n_fem_nutri_eligible
ta no_fem_nutri_eligible, m



gen	fem_eligible = (hv117==1) 						// fem eligible for cm
bys	hh_id: egen hh_n_fem_eligible = sum(fem_eligible) 	
gen	no_fem_eligible = (hh_n_fem_eligible==0) 									
lab var no_fem_eligible "hh has no eligible woman"
drop hh_n_fem_eligible 
ta no_fem_eligible, m



count if hv105>=15 & hv105<=54 & hv104==1 & hv027==1  // male eligible for nutr
ta hb13 shbsel if hv105>=15 & hv105<=54 & hv104==1 & hv027==1, m
count if hb13!=.
	/* The hb13 variable (results of measurement) indicate that 349,674 men 
	eligible for measurement. This is because men aged 50 and older have been 
	captured under this variable (under the category other). */ 	
gen male_nutri_eligible = (hv105>=15 & hv105<=54 & hv104==1 & hv027==1)
bys hh_id: egen hh_n_male_nutri_eligible = sum(male_nutri_eligible)  
gen	no_male_nutri_eligible = (hh_n_male_nutri_eligible==0)
lab var no_male_nutri_eligible "hh has no eligible man for nutr"	
drop hh_n_male_nutri_eligible
ta no_male_nutri_eligible, m



gen	male_eligible = (hv118==1)					 // male eligible for cm
bys	hh_id: egen hh_n_male_eligible = sum(male_eligible)  
gen	no_male_eligible = (hh_n_male_eligible==0) 	
lab var no_male_eligible "hh has no eligible man"
drop hh_n_male_eligible
ta no_male_eligible, m



gen	child_eligible = (hv120==1)   				// child eligible for nutr
bys	hh_id: egen hh_n_children_eligible = sum(child_eligible)  
gen	no_child_eligible = (hh_n_children_eligible==0) 
lab var no_child_eligible "hh has no children eligible for anthro"
drop hh_n_children_eligible
ta no_child_eligible, m



gen	no_adults_eligible = (no_fem_nutri_eligible==1 & no_male_nutri_eligible==1) 
lab var no_adults_eligible "hh has no elig adult for nutr"
lab values no_adults_eligible lab_yn
ta no_adults_eligible, m



gen no_eligibles = (no_fem_nutri_eligible==1 & ///
					no_male_nutri_eligible==1 & no_child_eligible==1)
lab var no_eligibles "hh has no elig member for nutr"
lab values no_eligibles lab_yn
ta no_eligibles, m


sort hh_id ind_id


********************************************************************************
**# Step 1.12 RENAMING DEMOGRAPHIC VARIABLES ***
********************************************************************************

desc hv005 											// sample weight
clonevar weight = hv005
replace weight = weight/1000000 
lab var weight "sample weight"


clonevar psu = hv021
lab var psu "primary sampling unit"

clonevar strata = hv022
lab var strata "sample strata"

svyset psu [pw=weight] , strata(strata)	 singleunit(centered)



codebook hv025											// area: urban-rural	
recode hv025 (1=1 "urban") (2=0 "rural"), gen (area)			
lab var area "area: urban-rural"
ta hv025 area, m



codebook hv101, ta (99)										// relationship
recode hv101 (1=1 "head")(2=2 "spouse")(3 11=3 "child") ///
(4/10 15/16=4 "extended family")(12=5 "not related")(17=6 "maid") (98=.), ///
gen (relationship)
lab var relationship "relationship to the head of hh"
ta hv101 relationship, m


	
codebook hv104 											 // sex
clonevar sex = hv104 
replace sex = . if hv104==3
lab var sex "sex of household member"
	/* note: the category transgender is very small in number, to allow for
	   meaningful analysis. Hence, we recoded this category as missing gender 
	   information. */

	   

bys	hh_id: egen missing_hhead = min(relation) 	 		// headship
ta missing_hhead,m 
gen household_head=.
replace household_head=1 if relation==1 & sex==1 
replace household_head=2 if relation==1 & sex==2
bys hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
lab def head 1"male-headed" 2"female-headed"
lab val headship head
lab var headship "household headship"
ta headship, m
replace headship=. if headship==0
	/* 55 individuals living across 15 household have reported their gender
	identity as transgender. This means, these households will have missing 
	value for the headship variable. */

	

codebook hv105, tab (999) 						//age; age group
clonevar age = hv105  
replace age = . if age>=98
lab var age "age of hh member"


recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+"), gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"



codebook hv115, ta (99) 								// marital status
recode hv115 (0=1 "never married")(1=2 "currently married") ///
(3=3 "widowed")(4=4 "divorced")(5=5 "not living together")(8=.), gen (marital)	
lab var marital "marital status of household member"
ta hv115 marital, m



gen member = 1 										// hh size
bys hh_id: egen hhsize = sum(member)
lab var hhsize "household size"
ta hhsize, m
drop member



codebook sh47, tab (99)								//   religion of hh head
recode sh47 (1=1 "Hindu") (2=2 "Muslim") (3=3 "Christian") ///
(4/96=4 "Other"), gen (religion_hh)
lab var religion_hh "religion of hh head"
ta sh47 religion_hh,m
ta religion_hh,m


						
desc sh49											// caste of hh head
codebook sh49, tab (99)
recode sh49 (1=1 "Scheduled Caste") (2=2 "Scheduled Tribe") ///
(3=3 "Other Backward Classes") (4/.=4 "Other"), gen (caste_hh)
lab var caste_hh "caste of hh head"
ta sh49 caste_hh,m		
ta caste_hh,m	
//'other' incl those who reported do not know which caste/tribe & no caste. 

	
	
codebook hv024, ta (99)  // subnational; 28 states & 8 union territories (p.2)
decode hv024, gen(temp)
replace temp =  proper(temp)
encode temp, gen(region)
drop temp
lab var region "region for subnational decomposition"
codebook region, ta (99)
lab def lab_reg ///
1  "Andaman & Nicobar Islands" 2  "Andhra Pradesh" ///
3  "Arunachal Pradesh" 4  "Assam" 5  "Bihar" 6  "Chandigarh" ///
7  "Chhattisgarh" 8  "Dadra & Nagar Haveli and Daman & Diu" ///
9  "Goa" 10 "Gujarat" 11 "Haryana" 12 "Himachal Pradesh" ///
13 "Jammu & Kashmir" 14 "Jharkhand" 15 "Karnataka" 16 "Kerala" ///
17 "Ladakh" 18 "Lakshadweep" 19 "Madhya Pradesh" 20 "Maharashtra" ///
21 "Manipur" 22 "Meghalaya" 23 "Mizoram" 24 "Nagaland" 25 "Delhi" ///
26 "Odisha" 27 "Puducherry" 28 "Punjab" 29 "Rajasthan" 30 "Sikkim" ///
31 "Tamil Nadu" 32 "Telangana" 33 "Tripura" 34 "Uttar Pradesh" ///
35 "Uttarakhand" 36 "West Bengal"  
lab val region lab_reg


clonevar district = shdist  				// 707 districts (p.2)
lab var district "district"


/*
Changes over time:
1) Chandigarh does not exist in 2005-06 survey, but exist in the 2015-16 & 
2019-21 survey data. We merged Chandigarh into Punjab in the later survey 
years, so it is comparable to the 2005-06 survey.   

2) Telangana does not exist in 2005-06 survey, but exist in the 2015-16 & 
2019-21 survey data. We merged Telangana into Andhra Pradesh in the later 
survey years, so it is comparable to the 2005-06 survey.

3) Ladakh was established as a union territory of India on 31 October 2019, 
following the passage of the Jammu and Kashmir Reorganisation Act. Prior to 
that, it was part of the Jammu and Kashmir state. We merged Ladakh into Jammu 
and Kashmir in the 2019-21 survey data, so it is comparable to the 2005-06 
and 2015-16 surveys.

4) We also have 5 Union Territories where these areas were not sampled in 2005-06 
but sampled in 2015-16 and 2019-21. These were dropped in 2015-16 & 2019-21, 
so it is comparable to 2005-06.   
*/

  
recode region (1 8 18 27 =.) (2 32=2 "Andhra Pradesh") /// 
(3=3 "Arunachal Pradesh") (4=4 "Assam") (5=5 "Bihar") (7=7 "Chhattisgarh") ///
(9=10 "Goa") (10=11 "Gujarat") (11=12 "Haryana") (12=13 "Himachal Pradesh") ///
(13 17=14 "Jammu & Kashmir") (14=15 "Jharkhand") (15=16 "Karnataka") ///
(16=17 "Kerala") (19=19 "Madhya Pradesh") (20=20 "Maharashtra") ///
(21=21 "Manipur") (22=22 "Meghalaya") (23=23 "Mizoram") (24=24 "Nagaland") ///
(25=25 "Delhi") (26=26 "Odisha") (6 28=28 "Punjab") (29=29 "Rajasthan") ///
(30=30 "Sikkim")(31=31 "Tamil Nadu")(33=32 "Tripura")(34=33 "Uttar Pradesh") ///
(35=34 "Uttarakhand") (36=35 "West Bengal"), gen(region_01) 
lab var region_01 "COT: Region for subnational decomposition"
codebook region_01, ta (99)


********************************************************************************
**#  Step 2 Data preparation  ***
***  Standardization of the global MPI indicators  
********************************************************************************


********************************************************************************
**# Step 2.1 Years of Schooling ***
********************************************************************************

codebook hv108, ta (99)

clonevar  eduyears = hv108   					// years of education
replace eduyears = . if eduyears>30				// missing value
ta hv108 hv106,m		
replace eduyears = . if eduyears>=age & age>0	// qc: eduyears not exceed age 
replace eduyears = 0 if age < 10 
replace eduyears = 0 if (age==10 | age==11) & eduyears < 6 
			// non-eligible age group	
lab var eduyears "total years of educ"



gen educ_elig = 1  							// eligibility for educ indicator 
replace educ_elig = 0 if age < 10  
replace educ_elig = 0 if (age==10 | age==11) & eduyears < 6 
lab def lab_educ_elig 0"not eligible" 1"eligible"  
lab val educ_elig lab_educ_elig
lab var educ_elig "eligibility for educ indicator"
ta eduyears educ_elig,m

	
	/* control variable: information on years of educ 
	present for at least 2/3 of the eligible householders */		
gen temp = 1 if eduyears!=. & educ_elig == 1
bys	hh_id: egen no_mis_edu = sum(temp)		// elig with educ data 
	
bys hh_id: egen hhs = sum(educ_elig == 1) 		// all eligible for educ

replace no_mis_edu = no_mis_edu/hhs
replace no_mis_edu = (no_mis_edu>=2/3)		
ta no_mis_edu, m							// qc: missing (0) is > 0.5% 
lab var no_mis_edu "no missing yos"
drop temp hhs


*** MPI ***
/* Householders are considered not deprived if at least 
one eligible householder has six or more years of education. */
******************************************************************* 
gen	educ6 = (eduyears>=6 & eduyears!=.)
replace educ6 = . if eduyears==.

bys hh_id: egen educ = max(educ6)
replace educ = . if educ==0 & no_mis_edu==0
lab var educ "non-deprivation in education"
ta educ, m 

	
*** Destitution ***
/* Householders are considered not deprived if at least 
one eligible householder has one or more years of education. */
******************************************************************* 
gen	educ1 = (eduyears>=1 & eduyears!=.)
replace educ1 = . if eduyears==.

bys	hh_id: egen educ_u = max(educ1)
replace educ_u = . if educ_u==0 & no_mis_edu==0
lab var educ_u "dst: non-deprivation in education"
ta educ_u,m



********************************************************************************
**# Step 2.2 Child School Attendance ***
********************************************************************************

codebook hv121, ta (99)
recode hv121 (2=1 "attending") (0=0 "not attending") (8=.), gen (attendance)
lab var attendance "current school year"	
ta attendance, m

	
	
*** MPI ***
/* Householders are considered not deprived if all 
school-aged children are attending up to class 8. */ 
******************************************************************* 
gen	child_schoolage = (age>=6 & age<=14)
lab var child_schoolage "eligible for school attendance"		
	// qc: official school entrance age to primary school: 6 years
	// qc: age range 6-14 (=6+8) 

	
	/* control variable: school attendance data is 
	missing for at least 2/3 of the school-aged children */
count if child_schoolage==1 & attendance==.			// qc: missing satt
gen temp = 1 if child_schoolage==1 & attendance!=. 	// elig children in school 
bys hh_id: egen no_missing_atten = sum(temp)		
gen temp2 = 1 if child_schoolage==1					//elig children
bys hh_id: egen hhs = sum(temp2)					
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)	 		
ta no_missing_atten, m							     // qc: missing < 0.5% 
lab var no_missing_atten "no missing satt"		
drop temp temp2 hhs
		
bys hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
lab var hh_children_schoolage "hh has elig child"

gen	not_atten = (attendance==0) if child_schoolage==1
replace not_atten = . if attendance==. & child_schoolage==1

bysort	hh_id: egen any_not_atten = max(not_atten)

gen	satt = (any_not_atten==0) 
replace satt = . if any_not_atten==.
replace satt = 1 if hh_children_schoolage==0
replace satt = . if satt==1 & no_missing_atten==0 
lab var satt "non-deprivation in school attendance"
ta satt, m

	
*** Destitution ***
/* Householders are considered not deprived if all 
school-aged children are attending up to class 6. */ 
******************************************************************* 
gen	child_schoolage_u = (age>=6 & age<=12) 
lab var child_schoolage_u "dst: eligible for school attendance"	

	
	/* control variable: school attendance data is 
	missing for at least 2/3 of the school-aged children */	
count if child_schoolage_u==1 & attendance==.	
gen temp = 1 if child_schoolage_u==1 & attendance!=.
bys hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_u==1	
bys hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)			
ta no_missing_atten_u, m			// qc: missing (0) is < 0.5% 
lab var no_missing_atten_u "no missing satt"		
drop temp temp2 hhs
		
bys	hh_id: egen hh_children_schoolage_u = sum(child_schoolage_u)
replace hh_children_schoolage_u = (hh_children_schoolage_u>0) 
lab var hh_children_schoolage_u "hh has elig child"

gen	atten_6 = (attendance==1) if child_schoolage_u==1
replace atten_6 = . if attendance==. & child_schoolage_u==1

bys	hh_id: egen any_atten_6 = max(atten_6)

gen	satt_u = (any_atten_6==1) 
replace satt_u = . if any_atten_6==.
replace satt_u = 1 if hh_children_schoolage_u==0
replace satt_u = . if satt_u==0 & no_missing_atten_u==0 
lab var satt_u "dst: non-deprivation in school attendance"
ta satt_u, m



********************************************************************************
**# Step 2.3 Nutrition 
********************************************************************************

********************************************************************************
**# Step 2.3a bmi & bmi-for-age ***
********************************************************************************

foreach var in ha40 hb40 {
			 gen inf_`var' = 1 if `var'!=.
			 bysort sex: tab age inf_`var' 
			  // qc: women 15-49 & men 15-54 measured
			 drop inf_`var'
			 }

gen	f_bmi = ha40/100 if hv105>=15 & hv105<=49 & hv104==2  	// bmi woman
lab var f_bmi "women's bmi"
gen	f_low_bmi = (f_bmi<18.5)
replace f_low_bmi = . if f_bmi==. | f_bmi>=99.97
lab var f_low_bmi "bmi of women < 18.5"
lab val f_low_bmi lab_yn
ta f_low_bmi if fem_nutri_eligible==1,m

gen	f_low_bmi_u = (f_bmi<17) if hv105>=15 & hv105<=49 & hv104==2
replace f_low_bmi_u = . if f_bmi==. | f_bmi>=99.97
lab var f_low_bmi_u "bmi of women <17"
lab val f_low_bmi_u lab_yn
ta f_low_bmi_u if fem_nutri_eligible==1,m



gen m_bmi = hb40/100 if hv105>=15 & hv105<=54 & hv104==1 & hv027==1	 // bmi man
lab var m_bmi "male's bmi"
gen m_low_bmi = (m_bmi<18.5)
replace m_low_bmi = . if m_bmi==. | m_bmi>=99.97 
lab var m_low_bmi "bmi of male < 18.5"
lab val m_low_bmi lab_yn
ta m_low_bmi if male_nutri_eligible==1,m

gen	m_low_bmi_u = (m_bmi<17) if hv105>=15 & hv105<=54 & hv104==1 & hv027==1	
replace m_low_bmi_u = . if m_bmi==. | m_bmi>=99.97
lab var m_low_bmi_u "bmi of male <17"
lab val m_low_bmi_u lab_yn
ta m_low_bmi_u if male_nutri_eligible==1,m



gen low_bmi_byage = 0								// MPI: bmi-for-age and bmi
lab var low_bmi_byage "low bmi or bmi-for-age"
replace low_bmi_byage = 1 if f_low_bmi==1					   // low bmi woman	
replace low_bmi_byage = 1 if low_bmi_byage==0 & m_low_bmi==1  // low bmi man
replace low_bmi_byage = 1 if low_bmiage==1 & age_month!=.  	 // bmi-for-age fem
replace low_bmi_byage = 0 if low_bmiage==0 & age_month!=.
replace low_bmi_byage = 1 if low_bmiage_b==1 & age_month_b!=. //bmi-for-age male
replace low_bmi_byage = 0 if low_bmiage_b==0 & age_month_b!=.	
replace low_bmi_byage = . if f_low_bmi==. & m_low_bmi==. & ///
							 low_bmiage==. & low_bmiage_b==.        // missing
		
bys hh_id: egen low_bmi = max(low_bmi_byage)

gen	hh_no_low_bmiage = (low_bmi==0)			// 1=normal bmi or bmi-for-age 
replace hh_no_low_bmiage = . if low_bmi==.				// missing
replace hh_no_low_bmiage = 1 if no_adults_eligible==1	// non-eligible 
drop low_bmi
lab var hh_no_low_bmiage "no adult with low bmi or bmi-for-age"
ta hh_no_low_bmiage, m	

	

gen low_bmi_byage_u = 0						// dst: bmi-for-age and bmi
lab var low_bmi_byage_u "dst: low bmi or bmi-for-age"
replace low_bmi_byage_u = 1 if f_low_bmi_u==1
replace low_bmi_byage_u = 1 if low_bmi_byage_u==0 & m_low_bmi_u==1 
replace low_bmi_byage_u = 1 if low_bmiage_u==1 & age_month!=.
replace low_bmi_byage_u = 0 if low_bmiage_u==0 & age_month!=.
replace low_bmi_byage_u = 1 if low_bmiage_b_u==1 & age_month_b!=.
replace low_bmi_byage_u = 0 if low_bmiage_b_u==0 & age_month_b!=.	
replace low_bmi_byage_u = . if f_low_bmi_u==. & low_bmiage_u==. & ///
								m_low_bmi_u==. & low_bmiage_b_u==. 
		
bys hh_id: egen low_bmi = max(low_bmi_byage_u)

gen	hh_no_low_bmiage_u = (low_bmi==0)
replace hh_no_low_bmiage_u = . if low_bmi==.
replace hh_no_low_bmiage_u = 1 if no_adults_eligible==1	
drop low_bmi
lab var hh_no_low_bmiage_u "dst: no adult with low bmi or bmi-for-age"
ta hh_no_low_bmiage_u, m		


********************************************************************************
**# Step 2.3b underweight, stunting & wasting
********************************************************************************

bys hh_id: egen temp = max(underweight)
gen	hh_no_underweight = (temp==0)		// no child is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1	
lab var hh_no_underweight "hh has no child underweight"
drop temp

bys hh_id: egen temp = max(underweight_u) 			
gen	hh_no_underweight_u = (temp==0) 	// no child is severely underweight 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
lab var hh_no_underweight_u "dst: hh has no child underweight"
drop temp



bys hh_id: egen temp = max(stunting)
gen	hh_no_stunting = (temp==0)					// no child is stunted 
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1	
lab var hh_no_stunting "hh has no child stunted"
drop temp

bys hh_id: egen temp = max(stunting_u) 				
gen	hh_no_stunting_u = (temp==0) 				// no child is severely stunted 
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
lab var hh_no_stunting_u "dst: hh has no child stunted"
drop temp



bys hh_id: egen temp = max(wasting)
gen	hh_no_wasting = (temp==0) 						// no child is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1		
lab var hh_no_wasting "hh has no child wasted"
drop temp

bys hh_id: egen temp = max(wasting_u) 			
gen	hh_no_wasting_u = (temp==0) 				// no child is severely wasted
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
lab var hh_no_wasting_u "dst: hh has no child wasted"
drop temp



gen uw_st = 1 if stunting==1 | underweight==1		// underweight or stunted 
replace uw_st = 0 if stunting==0 & underweight==0
replace uw_st = . if stunting==. & underweight==.
lab var uw_st "child is underweight or stunted"

bys hh_id: egen temp = max(uw_st)
gen	hh_no_uw_st = (temp==0)		
replace hh_no_uw_st = . if temp==.
replace hh_no_uw_st = 1 if no_child_eligible==1	
lab var hh_no_uw_st "hh has no child underweight or stunted"
drop temp

gen uw_st_u = 1 if stunting_u==1 | underweight_u==1 
replace uw_st_u = 0 if stunting_u==0 & underweight_u==0
replace uw_st_u = . if stunting_u==. & underweight_u==.
lab var uw_st_u "dst: child is underweight or stunted"

bys hh_id: egen temp = max(uw_st_u)
gen	hh_no_uw_st_u = (temp==0) 
replace hh_no_uw_st_u = . if temp==.
replace hh_no_uw_st_u = 1 if no_child_eligible==1 
lab var hh_no_uw_st_u "dst: hh has no child underweight or stunted"
drop temp


********************************************************************************
**# Step 2.3c nutrition indicator
********************************************************************************

*** MPI ***
/* Householders are not deprived if all eligible 
person with anthropometric measurement in the household 
are nourished; or household has no eligible person. */
************************************************************************
gen	nutr_3 = 1
replace nutr_3 = 0 if hh_no_low_bmiage==0 | hh_no_uw_st==0
replace nutr_3 = . if hh_no_low_bmiage==. & hh_no_uw_st==.  // missing

replace nutr_3 = . if hh_no_low_bmiage==. & ///
					  hh_no_uw_st==1 & no_child_eligible==1
	// elig adult has missing data; no elig child 
	
replace nutr_3 = . if hh_no_uw_st==. & ///
					  hh_no_low_bmiage==1 & no_adults_eligible==1
	// elig child has missing data; no elig adult 
	
replace nutr_3 = 1 if no_eligibles==1   // non-applicable population		
lab var nutr_3 "non-deprivation in nutr (under 5 & adults)"
ta nutr_3, m


gen nutr_0 = 1 if hh_no_stunting==1 & hh_no_underweight==1  // child under 5
replace nutr_0 = 0 if hh_no_stunting==0 | hh_no_underweight==0
replace nutr_0 = . if hh_no_stunting==. & hh_no_underweight==.
replace nutr_0 = 1 if no_child_eligible==1
lab var nutr_0 "non-deprivation in nutr (under 5)"


*** Destitution ***
/* Householders are not deprived if all eligible person with
anthropometric measurement in the household are not severely 
undernourished; or household has no eligible person. */
************************************************************************
gen	nutr_3_u = 1
replace nutr_3_u = 0 if hh_no_low_bmiage_u==0 | hh_no_uw_st_u==0
							 							  
replace nutr_3_u = . if (hh_no_low_bmiage_u==. & hh_no_uw_st_u==.)  | ///
(hh_no_low_bmiage_u==. & hh_no_uw_st_u==1 & no_child_eligible==1) | ///
(hh_no_uw_st_u==. & hh_no_low_bmiage_u==1 & no_adults_eligible==1)
							  
replace nutr_3_u = 1 if no_eligibles==1   	
lab var nutr_3_u "dst: non-deprivation in nutr (under 5 & adults)"
ta nutr_3_u, m


gen nutr_0_u = 1 if hh_no_stunting_u==1 & hh_no_underweight_u==1
replace nutr_0_u = 0 if hh_no_stunting_u==0 | hh_no_underweight_u==0
replace nutr_0_u = . if hh_no_stunting_u==. & hh_no_underweight_u==.
replace nutr_0_u = 1 if no_child_eligible==1 
lab var nutr_0_u "dst: non-deprivation in nutr (under 5)"


********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************

codebook v206 v207 mv206 mv207
	/*v206 or mv206: number of sons who have died 
	  v207 or mv207: number of daughters who have died*/
	

egen temp_f = rowtotal(v206 v207), missing
replace temp_f = 0 if v201==0 	 // have never given birth
bys	hh_id: egen child_mortality_f = sum(temp_f), m
lab var child_mortality_f "child mortality reported by woman"
ta child_mortality_f, m
drop temp_f

	
egen temp_m = rowtotal(mv206 mv207), missing
replace temp_m = 0 if mv201==0
bys	hh_id: egen child_mortality_m = sum(temp_m), m
lab var child_mortality_m "child mortality reported by man"
ta child_mortality_m, m
drop temp_m


egen child_mortality = rowmax(child_mortality_f child_mortality_m)
lab var child_mortality "child mortality in a hh"
ta child_mortality, m	
	
	
*** MPI *** 
/* Householders are not deprived if all eligible women 
in the household reported zero mortality among children 
under 18 in the last 5 years from the survey year. */
************************************************************************
ta childu18_died_per_wom_5y, m	

replace childu18_died_per_wom_5y = 0 if v201==0 
	// elig woman who never ever gave birth 
	
replace childu18_died_per_wom_5y = 0 if no_fem_eligible==1	
	// no elig woman
	
bys hh_id: egen childu18_mortality_5y = sum(childu18_died_per_wom_5y), m
replace childu18_mortality_5y = 0 if childu18_mortality_5y==. & child_mortality==0
	// women has missing value and men reported no death 
lab var childu18_mortality_5y "total u18 child mortality last 5 years"
ta childu18_mortality_5y, m		
	
gen cm_0 = (childu18_mortality_5y==0)
replace cm_0 = . if childu18_mortality_5y==.
lab var cm_0 "non-deprivation in cm, with bh"
ta cm_0, m	


*** Destitution *** 
*** (same as MPI) ***
************************************************************************
clonevar cm_0_u = cm_0	
lab var cm_0_u "dst: non-deprivation in cm, with bh"	


********************************************************************************
**# Step 2.5 Electricity 
********************************************************************************

*** MPI ***
/* Householders are considered not deprived 
if the household has electricity */
****************************************
codebook hv206, ta (9)
clonevar elct = hv206
lab var elct "non-deprivation in electricity"

svy: prop elct
	// qc: matches the report (p.33)

	
*** Destitution ***
*** (same as MPI) ***
****************************************
gen elct_u = elct
lab var elct_u "dst: non-deprivation in electricity"


********************************************************************************
**# Step 2.6 Sanitation ***
********************************************************************************
	
*** MPI ***
/* Householders are not deprived if the household has improved 
sanitation facilities that are not shared with other households. */
********************************************************************
desc hv205 hv225 

codebook hv205, ta(99) 	
	
recode hv205 (11/13 15/22 41 = 1 "yes") ///
(14 23 31 42/96 = 0 "no") (99=.), gen(sani)


codebook hv225

replace sani = 0 if hv225==1				
lab var sani "non-deprivation in sanitation"
ta hv205 sani, m

svy: prop sani 
	// qc: matches the report (p.69 pdf)


*** Destitution ***
/* Householders are not deprived if the 
household has sanitation facilities. */
********************************************************************

recode hv205 (11/23 41/44 = 1 "yes") ///
(31 96 = 0 "no") (99=.), gen(sani_u)

lab var sani_u "dst: non-deprivation in sanitation"
ta hv205 sani_u, m



********************************************************************************
**# Step 2.7 Drinking Water 
********************************************************************************


*** MPI ***
/* Householders are not deprived if household have access to safe 
drinking water and is under 30 minutes walk from home, round trip.*/
********************************************************************
codebook hv201, ta(99)

recode hv201 (11/31 41 51/92 = 1 "yes") ///
(32 42 43 96 = 0 "no") (99=.), gen(dwtr_src)
lab var dwtr_src "improved main source of drinking water"
ta hv201 dwtr_src,m

svy: prop dwtr_src
	// qc: matches the report (p.67, pdf)

	
codebook hv204, ta(99)				// time to water

clonevar wtr = dwtr_src
replace wtr = 0 if hv204 >=30 & hv204 <=900							 
lab var wtr "non-deprivation in drinking water"
ta dwtr_src wtr,m

	
*** Destitution ***
/* Householders are not deprived if household have access to safe 
drinking water and is 45 minutes walk or less from home, round trip.*/
********************************************************************
clonevar wtr_u = dwtr_src						   
replace wtr_u = 0 if hv204 >45 & hv204 <=900				  
lab var wtr_u "dst: non-deprivation in drinking water"
ta dwtr_src wtr_u,m 


********************************************************************************
**# Step 2.8 Housing ***
********************************************************************************

desc hv213 hv214 hv215


codebook hv213, ta (99)			// improved = rudimentary & finished floor 
recode hv213 (21/36 = 1 "yes") (11/13 96 = 0 "no") (99=.), gen(floor)
lab var floor "hh has improved floor"
ta hv213 floor, m	


codebook hv214, ta (99)				// improved = finished walls 
recode hv214 (31/36 = 1 "yes") (11/26 96 = 0 "no") (99=.), gen(wall)
lab var wall "hh has improved wall"
ta hv214 wall, m	


codebook hv215, ta (99)				// improved = finished roofing 
recode hv215 (31/39 = 1 "yes") (11/25 96 = 0 "no") (99=.), gen(roof)
lab var roof "hh has improved roof"
ta hv215 roof, m	

	
	
*** MPI ***
/* Householders are not deprived in housing if the roof, 
floor and walls are constructed from quality materials.*/
**************************************************************
gen hsg = 1
replace hsg = 0 if floor==0 | wall==0 | roof==0
replace hsg = . if floor==. & wall==. & roof==.
lab var hsg "non-deprivation in housing"
ta hsg, m


*** Destitution ***
/* Householders are not deprived in housing if at least two of the three 
components (roof/floor/walls) are constructed from quality materials. */
**************************************************************
gen hsg_u = 1

replace hsg_u = 0 if ///
(floor==0 & wall==0 & roof==1) | ///
(floor==0 & wall==1 & roof==0) | ///
(floor==1 & wall==0 & roof==0) | ///
(floor==0 & wall==0 & roof==0)

replace hsg_u = . if floor==. & wall==. & roof==.
lab var hsg_u "dst: non-deprivation in housing"
ta hsg_u, m


	
********************************************************************************
**# Step 2.9 Cooking Fuel 
********************************************************************************
	
	
*** MPI ***
/* Householders are considered not deprived if the 
household uses non-solid fuels for cooking. */
*****************************************************************

codebook hv226, ta (99)

recode hv226 (1/5 95 96 = 1 "yes") ///
(6/11 = 0 "no") (99=.), gen(ckfl)

ta hv226 ckfl,m 
lab var ckfl "non-deprivation in cooking fuel"	

svy: prop ckfl			// qc: match the report (p.74, pdf)	
	
	
*** Destitution  ***
*** (same as MPI) ***
****************************************	
clonevar ckfl_u = ckfl 
lab var ckfl_u "dst: non-deprivation in cooking fuel"	
	


********************************************************************************
**# Step 2.10 Assets ***
********************************************************************************


* radio/walkman/stereo/kindle
lookfor radio walkman stereo
codebook hv207
clonevar radio = hv207 
lab var radio "hh has radio"	


* television/LCD TV/plasma TV/color TV/black & white tv
lookfor tv television plasma lcd	
codebook hv208
clonevar television = hv208 
lab var television "hh has television"


* refrigerator/icebox/fridge
lookfor refrigerator réfrigérateur
codebook hv209
clonevar refrigerator = hv209 
lab var refrigerator "hh has refrigerator"


* bicycle/cycle rickshaw
lookfor bicycle bicyclette
codebook hv210	
clonevar bicycle = hv210 
lab var bicycle "hh has bicycle"	


* motorbike/motorized bike/autorickshaw
lookfor motorbike moto
codebook hv211	
clonevar motorbike = hv211 
lab var motorbike "hh has motorbike"


* car/van/lorry/truck
lookfor car voiture truck van
codebook hv212
clonevar car = hv212  
lab var car "hh has car"	

	
* handphone/telephone/iphone/mobilephone/ipod
lookfor telephone téléphone mobilephone ipod
codebook hv221 hv243a
clonevar telephone = hv221
replace telephone=1 if telephone!=1 & hv243a==1	
ta hv243a hv221 if telephone==1,m
lab var telephone "hh has telephone"
	

* animal cart
lookfor cart 
codebook hv243c
clonevar animal_cart = hv243c
lab var animal_cart "hh has animal cart"	

		
* computer/laptop/tablet
lookfor computer ordinateur laptop ipad tablet
codebook hv243e
clonevar computer = hv243e
lab var computer "hh has computer"
	
	
lab def lab_asst 0"No" 1"Yes"
foreach var in television radio telephone refrigerator car ///
			   bicycle motorbike computer animal_cart {
lab val `var' lab_asst	
replace `var' = . if `var'==9 | `var'==99 | `var'==8 | `var'==98	   	
}

		
	
*** MPI ***
/* Householders are considered not deprived in assets 
if the household own more than one of the assets. */
*****************************************************************************
egen n_small_asst = rowtotal (television radio telephone refrigerator ///
bicycle motorbike computer animal_cart), m
lab var n_small_asst "small assets owned by hh" 
   
gen asst = (car==1 | n_small_asst > 1) 
replace asst = . if car==. & n_small_asst==.
lab var asst "non-deprivation in assets"



*** Destitution ***
/* Householders are considered not deprived in assets 
if the household own at least one asset. */
*****************************************************************************	
gen	asst_u = (car==1 | n_small_asst>0)
replace asst_u = . if car==. & n_small_asst==.
lab var asst_u "dst: non-deprivation in assets"


 
********************************************************************************
**# Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************

desc hv007 hv006 hv008
clonevar intvw_y = hv007 	
clonevar intvw_m = hv006 
clonevar intvw_d = hv008
 


recode cm_0   		(0=1)(1=0) , gen(d_cm)		// MPI indicators recoded for est 
recode nutr_3 		(0=1)(1=0) , gen(d_nutr)
recode satt 		(0=1)(1=0) , gen(d_satt)
recode educ 		(0=1)(1=0) , gen(d_educ)
recode elct			(0=1)(1=0) , gen(d_elct)
recode wtr 			(0=1)(1=0) , gen(d_wtr)
recode sani			(0=1)(1=0) , gen(d_sani)
recode hsg 			(0=1)(1=0) , gen(d_hsg)
recode ckfl 		(0=1)(1=0) , gen(d_ckfl)
recode asst 		(0=1)(1=0) , gen(d_asst)
 

recode cm_0_u   	(0=1)(1=0) , gen(dst_cm)	// dst indicators recoded for est
recode nutr_3_u 	(0=1)(1=0) , gen(dst_nutr)
recode satt_u 		(0=1)(1=0) , gen(dst_satt)
recode educ_u 		(0=1)(1=0) , gen(dst_educ)
recode elct_u 		(0=1)(1=0) , gen(dst_elct)
recode wtr_u  		(0=1)(1=0) , gen(dst_wtr)
recode sani_u 		(0=1)(1=0) , gen(dst_sani)
recode hsg_u  		(0=1)(1=0) , gen(dst_hsg)
recode ckfl_u 		(0=1)(1=0) , gen(dst_ckfl)
recode asst_u 		(0=1)(1=0) , gen(dst_asst) 
 
 

recode cm_0   	(0=1)(1=0) , gen(d_cm_01)				// hot MPI for est 
recode nutr_3 	(0=1)(1=0) , gen(d_nutr_01)
recode satt 	(0=1)(1=0) , gen(d_satt_01)
recode educ 	(0=1)(1=0) , gen(d_educ_01)
recode elct		(0=1)(1=0) , gen(d_elct_01)
recode wtr 		(0=1)(1=0) , gen(d_wtr_01)
recode sani		(0=1)(1=0) , gen(d_sani_01)
recode hsg 		(0=1)(1=0) , gen(d_hsg_01)
recode ckfl 	(0=1)(1=0) , gen(d_ckfl_01)
recode asst 	(0=1)(1=0) , gen(d_asst_01)
 

recode cm_0_u   	(0=1)(1=0) , gen(dst_cm_01)			// hot dst for est
recode nutr_3_u		(0=1)(1=0) , gen(dst_nutr_01)
recode satt_u 		(0=1)(1=0) , gen(dst_satt_01)
recode educ_u 		(0=1)(1=0) , gen(dst_educ_01)
recode elct_u 		(0=1)(1=0) , gen(dst_elct_01)
recode wtr_u  		(0=1)(1=0) , gen(dst_wtr_01)
recode sani_u 		(0=1)(1=0) , gen(dst_sani_01)
recode hsg_u  		(0=1)(1=0) , gen(dst_hsg_01)
recode ckfl_u 		(0=1)(1=0) , gen(dst_ckfl_01)
recode asst_u 		(0=1)(1=0) , gen(dst_asst_01) 
	 
	 

lab def lab_dp 1"deprived" 0"non-deprived"
foreach var in d_* dst_* d_*_* dst_*_* {
lab val `var' lab_dp
} 	 
	 
	 
keep hh_id ind_id strata psu weight sex age ///
area agec7 agec4 agec2 region headship d_* dst_* 

	 
order hh_id ind_id strata psu weight sex age ///
area agec7 agec4 agec2 region headship d_* dst_* 


mdesc psu strata area age headship d_* dst_*


char _dta[cty] "India"
char _dta[ccty] "IND"
char _dta[year] "2019-2021" 	
char _dta[survey] "DHS"
char _dta[ccnum] "356"
char _dta[type] "micro"


sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/ind_dhs19-21.dta", replace 
