********************************************************************************
/*
Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
Global Multidimensional Poverty Index - India DHS 2005-06 
[STATA do-file]. Available from OPHI website: http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk
*/
********************************************************************************

clear all 
set more off
set maxvar 10000


global path_in "../rdta/India DHS 2005-06"  	// folder path
global path_out "cdta"
global path_ado "ado"



********************************************************************************
*** Step 1: Data preparation 
********************************************************************************
		
/* NOTE:
India DHS (NFHS-3) 2005-06: Height and weight were measured for all 
children age 0-59 months, women age 15-49, and a subsample of men age 
15-54 (p.8).
 
India DHS (NFHS) 2015-16: Height and weight were measured for all 
children age 0-59 months, women age 15-49, and a subsample of men age 15-54 who
were selected in the state module (p.4).

As such, nutrition indicator over time will be constructed using information 
from children, women and men across time. It should be noted that in OPHI's 
published 2005-06 MPI figures, men were excluded from the nutrition indicator.*/ 	

	
********************************************************************************
*** Step 1.1 PR - Children under 5 years
********************************************************************************

use "$path_in/IAPR52FL.dta", clear 


gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g


tab hv120,miss	
count if hv120>=1 
	/*66,012 children under 5 are eligible for anthropometric 
	measurement. Of this 55,240 are child of interviewed mother.
	Weight and height data were collected from these children. */	
count if hc1!=.
	//55,240 children under 5 have information on age in months
tab hv105 if hc1!=.
	/*A cross check with the age in years reveal that all are within the 5 year 
	age group */
tab hc13 hv120, miss	
tab hc13 if hv120==1, miss
	//46,347 (83.9%) of the 55,240 children have been measured
tab hc13 if hc3<=9990, miss
tab hc13 if hc2<=9990, miss
	/*For example, height and weight data is available for all 46,347 children 
	who have been successfully measured */

	/*Following the checks carried out above, we keep only eligible children in
	this section since the interest is to generate measures for children under 
	5*/
keep if hv120==1
count	
	//55,240 children under 5	


*** Variable: SEX ***
desc hc27 hv104
	/*hc27=sex of the child from biomarker questionnaire;
	hv104=sex from household roaster */
compare hc27 hv104
	//hc27 should match with hv104
tab hc27, miss 
	//"1" for male ;"2" for female 
tab hc27, nol 
clonevar gender = hc27
desc gender
tab gender


*** Variable: AGE ***
tab hc1, miss  
codebook hc1 
clonevar age_months = hc1  
desc age_months
sum age_months

gen mdate = mdy(hc18, hc17, hc19)
gen bdate = mdy(hc30, hc16, hc31) if hc16 <= 31
	//Calculate birth date in days from date of interview
replace bdate = mdy(hc30, 15, hc31) if hc16 > 31 
	//If date of birth of child has been expressed as more than 31, we use 15
gen age = (mdate-bdate)/30.4375 
	//Calculate age in months with days expressed as decimals
	
gen  str6 ageunit = "months" 
lab var ageunit "Months"


*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook hc2, tab (9999)
gen	weight = hc2/10 
	//We divide it by 10 in order to express it in kilograms 
tab hc2 if hc2>990, miss nol   
replace weight = . if hc2>=990 
	//All missing values or out of range are replaced as "."
tab	hc13 hc2 if hc2>=990 | hc2==., miss 
	//hw13: result of the measurement 
sum weight


*** Variable: HEIGHT (CENTIMETERS)
codebook hc3, tab (9999)
gen	height = hc3/10 
	//We divide it by 10 in order to express it in centimeters
tab hc3 if hc3>9990, miss nol   
	//Missing values are 9994 to 9996
replace height = . if hc3>=9990 
	//All missing values or out of range are replaced as "."
tab	hc13 hc3   if hc3>=9990 | hc3==., miss 
sum height


*** Variable: MEASURED STANDING/LYING DOWN ***	
codebook hc15
gen measure = "l" if hc15==1 
	//Child measured lying down
replace measure = "h" if hc15==2 
	//Child measured standing up
replace measure = " " if hc15==9 | hc15==0 | hc15==. 
	//Replace with " " if unknown
tab measure


*** Variable: OEDEMA ***
lookfor oedema
gen  oedema = "n"  
	//It assumes no-one has oedema
tab oedema	


*** Variable: SAMPLING WEIGHT ***
	/* We don't require individual weight to compute the z-scores of a child. 
	So we assume all children in the sample have the same sample weight */
gen  sw = 1	
sum sw


*** Indicate to STATA where the igrowup_restricted.ado file is stored:
	***Source of ado file: http://www.who.int/childgrowth/software/en/
adopath + "$path_ado/igrowup_stata"

*** We will now proceed to create three nutritional variables: 
	*** weight-for-age (underweight),  
	*** weight-for-height (wasting) 
	*** height-for-age (stunting)

/* We use 'reflib' to specify the package directory where the .dta files 
containing the WHO Child Growth Standards are stored.*/	
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"


/* We use datalib to specify the working directory where the input STATA 
dataset containing the anthropometric measurement is stored. */
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"


/* We use datalab to specify the name that will prefix the output files that 
will be produced from using this ado file (datalab_z_r_rc and datalab_prev_rc)*/
gen str30 datalab = "children_nutri_ind" 
lab var datalab "Working file"
	
	
/*We now run the command to calculate the z-scores with the adofile */
igrowup_restricted reflib datalib datalab gender age ageunit weight height ///
measure oedema sw


/*We now turn to using the dta file that was created and that contains 
the calculated z-scores to create the child nutrition variables following WHO 
standards */
use "$path_out/children_nutri_ind_z_rc.dta", clear 

	
*** Standard MPI indicator ***	
gen	underweight = (_zwei < -2.0) 
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "Child is underweight"
tab underweight [aw=hv005], miss


gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "Child is stunted"
tab stunting [aw=hv005], miss


gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "Child is wasted"
tab wasting [aw=hv005], miss


*** Destitution MPI indicator  ***	
gen	underweight_u = (_zwei < -3.0) 
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "DST: Child is underweight"

gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "DST: Child is stunted"

gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "DST: Child is wasted"

  
count if _fwei==1 | _flen==1 	
	/* 1,488 children were replaced as missing because they have extreme 
	z-scores which are biologically implausible. */
 
 
gen child_nutr=1 
	//Identification variable for children under 5 in PR recode 
	
	
gen weight_ch = hv005/1000000
label var weight_ch "sample weight child under 5" 
	
 
	//Retain relevant variables:
keep ind_id child_nutr weight_ch underweight* stunting* wasting* 
order ind_id child_nutr weight_ch underweight* stunting* wasting* 
sort ind_id
save "$path_out/IND05-06_PR_child.dta", replace
	
	
	//Erase files from folder:
erase "$path_out/children_nutri_ind_z_rc.xls"
erase "$path_out/children_nutri_ind_prev_rc.xls"
erase "$path_out/children_nutri_ind_z_rc.dta"



********************************************************************************
*** Step 1.2  BR - BIRTH RECODE 
*** (All females 15-49 years who ever gave birth)  
********************************************************************************
/*The purpose of step 1.2 is to identify children of any age who died in 
the last 5 years prior to the survey date.*/

use "$path_in/IABR52FL.dta", clear

		
*** Generate individual unique key variable required for data merging
*** v001=cluster number;  
*** v002=household number; 
*** v003=respondent's line number
gen double ind_id = v001*1000000 + v002*100 + v003 
format ind_id %20.0g
label var ind_id "Individual ID"


desc b3 b7 v008	
gen date_death = b3 + b7
	//Date of death = date of birth (b3) + age at death (b7)
gen mdead_survey = v008 - date_death
	//Months dead from survey = Date of interview (v008) - date of death
gen ydead_survey = mdead_survey/12
	//Years dead from survey

	
gen age_death = b7	
label var age_death "Age at death in months"
tab age_death, miss
	//Check whether the age is in months

	
codebook b5, tab (10)	
gen child_died = 1 if b5==0
	//Redefine the coding and labels (1=child dead; 0=child alive)
replace child_died = 0 if b5==1
replace child_died = . if b5==.
label define lab_died 1 "child has died" 0 "child is alive"
label values child_died lab_died
tab b5 child_died, miss
	

	/*NOTE: For each woman, sum the number of children who died and compare to 
	the number of sons/daughters whom they reported have died */
bysort ind_id: egen tot_child_died = sum(child_died) 
egen tot_child_died_2 = rsum(v206 v207)
	//v206: sons who have died; v207: daughters who have died
compare tot_child_died tot_child_died_2
	//India DHS 2005-06: the figures are identical 

	
	//Identify child under 18 mortality in the last 5 years
gen child18_died = child_died 
replace child18_died=0 if age_death>=216 & age_death<.
label values child18_died lab_died
tab child18_died, miss	
			
bysort ind_id: egen tot_child18_died_5y=sum(child18_died) if ydead_survey<=5
	/*Total number of children under 18 who died in the past 5 years 
	prior to the interview date */	
	
replace tot_child18_died_5y=0 if tot_child18_died_5y==. & tot_child_died>=0 & tot_child_died<.
	/*All children who are alive or who died longer than 5 years from the 
	interview date are replaced as '0'*/
	
replace tot_child18_died_5y=. if child18_died==1 & ydead_survey==.
	//Replace as '.' if there is no information on when the child died  

tab tot_child_died tot_child18_died_5y, miss

bysort ind_id: egen childu18_died_per_wom_5y = max(tot_child18_died_5y)
lab var childu18_died_per_wom_5y "Total child under 18 death for each women in the last 5 years (birth recode)"
	

	//Keep one observation per women
bysort ind_id: gen id=1 if _n==1
keep if id==1
drop id
duplicates report ind_id 

gen women_BR = 1 
	//Identification variable for observations in BR recode

	
	//Retain relevant variables
keep ind_id women_BR childu18_died_per_wom_5y 
order ind_id women_BR childu18_died_per_wom_5y
sort ind_id
save "$path_out/IND05-06_BR.dta", replace	


********************************************************************************
*** Step 1.3  IR - WOMEN's RECODE  
*** (All eligible females 15-49 years in the household)
********************************************************************************

use "$path_in/IAIR52FL.dta", clear


codebook v012, tab (999)
	//Age of individual women: 15-45 years 
	
*** Generate individual unique key variable required for data merging
*** v001=cluster number;  
*** v002=household number; 
*** v003=respondent's line number
gen double ind_id = v001*1000000 + v002*100 + v003 
format ind_id %20.0g
label var ind_id "Individual ID"

duplicates report ind_id

gen women_IR=1 
	//Identification variable for observations in IR recode
		
 
	//Retain relevant variables:
keep ind_id women_IR v003 v005 v012 v201 v206 v207  
order ind_id women_IR v003 v005 v012 v201 v206 v207 
sort ind_id
save "$path_out/IND05-06_IR.dta", replace


********************************************************************************
*** Step 1.4  PR - INDIVIDUAL RECODE  
*** (Girls 15-19 years in the household)
********************************************************************************
/*The purpose of step 1.4 is to compute bmi-for-age for girls 15-19 years. */

use "$path_in/IAPR52FL.dta", clear

	
*** Generate individual unique key variable required for data merging using:
gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g
label var ind_id "Individual ID"
codebook ind_id

	
*** Identify anthropometric sample for girls
count if hv105>=15 & hv105<=19 & hv104==2 
tab ha13 if hv105>=15 & hv105<=19 & hv104==2, miss
	/*Total number of women 15-19 years who have anthropometric data and 
	were selected for the biomarker questionnaire: 26,238 */
tab ha13 hv117 if hv105>=15 & hv105<=19 & hv104==2, miss
tab ha13 hv103 if hv105>=15 & hv105<=19 & hv104==2, miss
	/*1,443 of the 26,238 women 15-19 years are identified as non-eligible
	for the female interview as they did not sleep the night before in the 
	household. Hence they will not have data on child mortality but they have 
	anthropometric information as they were covered in the biomarker 
	questionnaire. */
	
		
*** Keep relevant sample	
keep if hv105>=15 & hv105<=19 & hv104==2 
count
	//Total girls 15-19 years: 26,238	

	
***Variables required to calculate the z-scores to produce BMI-for-age:

*** Variable: SEX ***
codebook hv104, tab (9)
clonevar gender = hv104
	//2:female 


*** Variable: AGE ***
desc hv008 ha32
gen age_month = hv008 - ha32
lab var age_month "Age in months, girls 15-19 years"
tab age_month, miss	
	/* For a couple of observations, we find that the age in months is 
	beyond 228 months. In this secton, while calculating the z-scores, these 
	cases will be excluded. However, in section 2.3, we will take the BMI 
	information of these girls. */

	
*** Variable: AGE UNIT ***
gen str6 ageunit = "months" 
lab var ageunit "Months"

			
*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook ha2, tab (9999)
count if ha2>9990 
tab ha13 if ha2>9990, miss
gen weight = ha2/10 if ha2<9990
	/*Weight information from girls. We divide it by 10 in order to express 
	it in kilograms. Missing values or out of range are identified as "." */	
sum weight


*** Variable: HEIGHT (CENTIMETERS)	
codebook ha3, tab (9999)
count if ha3>9990 
tab ha13 if ha3>9990, miss
gen height = ha3/10 if ha3<9990
	/*Height information from girls. We divide it by 10 in order to express 
	it in centimeters. Missing values or out of range are identified as "." */
sum height


*** Variable: OEDEMA
gen oedema = "n"  


*** Variable: SAMPLING WEIGHT ***
gen sw = 1


***z-scores***	
adopath + "$path_ado/who2007_stata"		
gen str100 reflib = "$path_ado/who2007_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "girl_nutri_ind" 
lab var datalab "Working file"
	

who2007 reflib datalib datalab gender age_month ageunit weight height oedema sw

use "$path_out/girl_nutri_ind_z.dta", clear 
	
gen	z_bmi = _zbfa
replace z_bmi = . if _fbfa==1 
lab var z_bmi "z-score bmi-for-age"


*** Standard MPI indicator ***	
gen	low_bmiage = (z_bmi < -2.0) 
replace low_bmiage = . if z_bmi==.
lab var low_bmiage "Adolescent girls have low bmi"


*** Destitution indicator ***	
gen	low_bmiage_u = (z_bmi < -3.0)
replace low_bmiage_u = . if z_bmi==.
lab var low_bmiage_u "DST: Adolescent girls have low bmi"

tab low_bmiage, miss
tab low_bmiage_u, miss


gen girl_PR=1 
	//Identification variable for girls 15-19 years in PR recode 


	//Retain relevant variables:	
keep ind_id girl_PR age_month low_bmiage*
order ind_id girl_PR age_month low_bmiage*
sort ind_id
save "$path_out/IND05-06_PR_girls.dta", replace
 	

	//Erase files from folder:
erase "$path_out/girl_nutri_ind_z.xls"
erase "$path_out/girl_nutri_ind_prev.xls"
erase "$path_out/girl_nutri_ind_z.dta"


********************************************************************************
*** Step 1.5  MR - MEN'S RECODE  
***(Eligible man 15-54 years in the household) 
********************************************************************************

use "$path_in/IAMR52FL.dta", clear 


gen double ind_id = mv001*1000000 + mv002*100 + mv003 	
format ind_id %20.0g
label var ind_id "Individual ID"

duplicates report ind_id

tab mv012, miss
	//Individual men: 15-54 years


gen men_MR=1 	
	//Identification variable for observations in MR recode
	


	//Retain relevant variables:	
keep ind_id men_MR mv003 mv005 mv012 mv201 mv206 mv207  
order ind_id men_MR mv003 mv005 mv012 mv201 mv206 mv207 
sort ind_id
save "$path_out/IND05-06_MR.dta", replace


********************************************************************************
*** Step 1.6  PR - INDIVIDUAL RECODE  
*** (Boys 15-19 years in the household)
********************************************************************************

use "$path_in/IAPR52FL.dta", clear 

	
gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g
label var ind_id "Individual ID"


*** Identify anthropometric sample for boys
count if hv105>=15 & hv105<=19 & hv104==1 
count if hv105>=15 & hv105<=19 & hv104==1 & hv027==1 
tab hb13 hv027 if hv105>=15 & hv105<=19 & hv104==1, miss
	/*Total number of men 15-19 years who have anthropometric data and 
	were selected for the biomarker questionnaire: 15,410 */
tab hb13 hv118 if hv105>=15 & hv105<=19 & hv104==1 & hv027==1, miss
tab hb13 hv103 if hv105>=15 & hv105<=19 & hv104==1 & hv027==1, miss
	/* 648 of the 15,410 men 15-19 years are identified as non-eligible
	for the male interview as they did not sleep the night before in the 
	household. Hence they will not have data on child mortality but they have 
	anthropometric information as they were covered in the biomarker 
	questionnaire. */

*** Keep relevant sample		
keep if hv105>=15 & hv105<=19 & hv104==1 & hv027==1 
count
	//Total boys 15-19 years: 15,410
	

***Variables required to calculate the z-scores to produce BMI-for-age:

*** Variable: SEX ***
codebook hv104, tab (9)
clonevar gender = hv104
	//1: male; 2:female 


*** Variable: AGE ***
desc hv008 hb32
gen age_month_b = hv008 - hb32
lab var age_month_b "Age in months, boys 15-19 years"
tab age_month_b, miss	
	/* For a couple of observations, we find that the age in months is 
	beyond 228 months. In this secton, while calculating the z-scores, these 
	cases will be excluded. However, in section 2.3, we will take the BMI 
	information of these boys. */

	
*** Variable: AGE UNIT ***
gen str6 ageunit = "months" 
lab var ageunit "Months"

			
*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook hb2, tab (9999)
count if hb2>9990 
tab hb13 if hb2>9990, miss
gen weight = hb2/10 if hb2<9990	
sum weight


*** Variable: HEIGHT (CENTIMETERS)	
codebook hb3, tab (9999)
count if hb3>9990 
tab hb13 if hb3>9990, miss
gen height = hb3/10 if hb3<9990
sum height


*** Variable: OEDEMA
gen oedema = "n"  
	

*** Variable: SAMPLING WEIGHT ***
gen sw = 1


***z-score***	
adopath + "$path_ado/who2007_stata"		
gen str100 reflib = "$path_ado/who2007_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "boy_nutri_ind" 
lab var datalab "Working file"
	
who2007 reflib datalib datalab gender age_month_b ageunit weight height oedema sw

use "$path_out/boy_nutri_ind_z.dta", clear 

	
gen	z_bmi = _zbfa
replace z_bmi = . if _fbfa==1 
lab var z_bmi "z-score bmi-for-age"


*** Standard MPI indicator ***	
gen	low_bmiage_b = (z_bmi < -2.0) 
replace low_bmiage_b = . if z_bmi==.
lab var low_bmiage_b "Adolescent boys have low bmi"


*** Destitution indicator ***	
gen	low_bmiage_b_u = (z_bmi < -3.0)
replace low_bmiage_b_u = . if z_bmi==.
lab var low_bmiage_b_u "DST: Adolescent boys have low bmi"

tab low_bmiage_b, miss
tab low_bmiage_b_u, miss


gen boy_PR=1 
	//Identification variable for boys 15-19 years in PR recode 


	//Retain relevant variables:	
keep ind_id boy_PR age_month_b low_bmiage*
order ind_id boy_PR age_month_b low_bmiage*
sort ind_id
save "$path_out/IND05-06_PR_boys.dta", replace
 	

	//Erase files from folder:
erase "$path_out/boy_nutri_ind_z.xls"
erase "$path_out/boy_nutri_ind_prev.xls"
erase "$path_out/boy_nutri_ind_z.dta"



********************************************************************************
*** Step 1.7  PR - HOUSEHOLD MEMBER'S RECODE 
********************************************************************************

use "$path_in/IAPR52FL.DTA", clear
	

gen double hh_id = hv001*10000 + hv002 
format hh_id %20.0g
label var hh_id "Household ID"
codebook hh_id  


gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g
label var ind_id "Individual ID"
codebook ind_id

	
sort hh_id ind_id


********************************************************************************
*** Step 1.8 DATA MERGING 
******************************************************************************** 
 
 
*** Merging BR Recode 
*****************************************
merge 1:1 ind_id using "$path_out/IND05-06_BR.dta"
drop _merge
*erase "$path_out/IND05-06_BR.dta"



*** Merging IR Recode 
*****************************************
merge 1:1 ind_id using "$path_out/IND05-06_IR.dta"
tab women_IR hv117, miss col
tab ha65 if hv117==1 & women_IR ==., miss 
	//Total number of eligible women not interviewed
drop _merge
*erase "$path_out/IND05-06_IR.dta"


/*Check if the number of women in BR recode matches the number of those
who provided birth history information in IR recode. */
count if women_BR==1
count if v201!=0 & v201!=. & women_IR==1


/*Check if the number of women in BR and IR recode who provided birth history 
information matches with the number of eligible women identified by hv117. */
count if hv117==1
count if women_BR==1 | v201==0
count if (women_BR==1 | v201==0) & hv117==1
tab v201 if hv117==1, miss
tab v201 ha65 if hv117==1, miss
	/* Some 5.48% eligible women did not provide information on their birth 
	history as they were not interviewed. This will result in missing value 
	for the child mortality indicator that we will construct later */


*** Merging 15-19 years: girls 
*****************************************
merge 1:1 ind_id using "$path_out/IND05-06_PR_girls.dta"
tab ha13 girl_PR if hv105>=15 & hv105<=19 & hv104==2, miss col
drop _merge
*erase "$path_out/IND05-06_PR_girls.dta"



*** Merging MR Recode 
*****************************************
merge 1:1 ind_id using "$path_out/IND05-06_MR.dta"
tab men_MR hv118 if hv027==1, miss col
tab hb65 if hv118==1 & men_MR ==. 
	//Total of eligible men not interviewed
drop _merge
*erase "$path_out/IND05-06_MR.dta"


*** Merging 15-19 years: boys 
*****************************************
merge 1:1 ind_id using "$path_out/IND05-06_PR_boys.dta"
tab boy_PR hv027 if hv105>=15 & hv105<=19 & hv104==1, miss col
drop _merge
*erase "$path_out/IND05-06_PR_boys.dta"	


*** Merging child under 5 
*****************************************
merge 1:1 ind_id using "$path_out/IND05-06_PR_child.dta"
tab hv120, miss  
tab hc13 if hv120==1, miss
drop _merge
*erase "$path_out/IND05-06_PR_child.dta"


sort ind_id


********************************************************************************
*** Step 1.9 KEEPING ONLY DE JURE HOUSEHOLD MEMBERS ***
********************************************************************************

clonevar resident = hv102
codebook resident, tab (9) 
lab var resident "Permanent (de jure) household member"
label define lab_yn 1 "yes" 0 "no"
lab values resident lab_yn
tab resident, miss
drop if resident!=1
	/* The global MPI is based on de jure (permanent) household members 
	only. As such, non-usual residents will be excluded from the sample. 
	
	In the context of India DHS 2005-06, 17,910 (3.35%) individuals who were 
	non-usual residents were dropped from the sample. */

	
********************************************************************************
*** Step 1.10 SUBSAMPLE VARIABLE ***
********************************************************************************

/*In the context of India DHS 2005-06, height and weight measurements was 
collected from all women age 15-49 and children 0-5, as well as a subsample of 
men 15-59. Hemoglobin data was collected from all women 15-49, children 0-5 and 
men 15-59 that were sampled. Hence in this case, there is no subsample 
selection. */

gen subsample = .
lab var subsample "HH selected as part of nutr subsample" 
tab subsample, miss
	

********************************************************************************
*** Step 1.11 CONTROL VARIABLES
********************************************************************************

/* Households are identified as having 'no eligible' members if there are no 
applicable population, that is, children 0-5 years, adult women 15-49 years or 
men 15-54 years. These households will not have information on relevant 
indicators of health. As such, these households are considered as non-deprived 
in those relevant indicators. */


*** No eligible women 15-49 years 
*** for adult nutrition indicator
***********************************************
tab ha13 if hv105>=15 & hv105<=49 & hv104==2, miss
gen fem_nutri_eligible = (ha13!=.)
lab var fem_nutri_eligible "Women eligible for nutrition ind"	
lab values fem_nutri_eligible lab_yn
tab fem_nutri_eligible, miss

bysort hh_id: egen hh_n_fem_nutri_eligible = sum(fem_nutri_eligible) 	
gen	no_fem_nutri_eligible = (hh_n_fem_nutri_eligible==0)
	//Takes value 1 if the household had no eligible women for anthropometrics
lab var no_fem_nutri_eligible "HH has no eligible women for nutrition ind"
lab values no_fem_nutri_eligible lab_yn	
drop hh_n_fem_nutri_eligible
tab no_fem_nutri_eligible, miss


*** No eligible women 15-49 years 
*** for child mortality indicator
*****************************************
gen	fem_eligible = (hv117==1)
lab var fem_eligible "Women eligible for child mortality ind"
lab values fem_eligible lab_yn
tab fem_eligible,m

bysort	hh_id: egen hh_n_fem_eligible = sum(fem_eligible) 	
	//Number of eligible women for interview in the hh
gen	no_fem_eligible = (hh_n_fem_eligible==0) 									
	//Takes value 1 if the household had no eligible women for an interview
lab var no_fem_eligible "HH has no eligible women for child mortality ind"
lab values no_fem_eligible lab_yn	
drop hh_n_fem_eligible 
tab no_fem_eligible, miss
	
		
*** No eligible men 15-54 years 
*** for adult nutrition indicator (if relevant)
***********************************************
tab hb13 if hv105>=15 & hv105<=54 & hv104==1 & hv027==1, miss
gen male_nutri_eligible = (hb13!=.)
lab var male_nutri_eligible "Men eligible for nutrition ind"	
lab values male_nutri_eligible lab_yn
tab male_nutri_eligible,miss

bysort hh_id: egen hh_n_male_nutri_eligible = sum(male_nutri_eligible)  
gen	no_male_nutri_eligible = (hh_n_male_nutri_eligible==0)
	//Takes value 1 if the household had no eligible men for anthropometrics
lab var no_male_nutri_eligible "HH has no eligible men for nutrition ind"
lab values no_male_nutri_eligible lab_yn		
drop hh_n_male_nutri_eligible
tab no_male_nutri_eligible, miss


*** No eligible men 15-54 years
*** for child mortality indicator (if relevant)
*****************************************
gen	male_eligible = (hv118==1)
lab var male_eligible "Men eligible for child mortality ind"
lab values male_eligible lab_yn
tab male_eligible,m

bysort	hh_id: egen hh_n_male_eligible = sum(male_eligible)  
	//Number of eligible men for interview in the hh
gen	no_male_eligible = (hh_n_male_eligible==0) 	
	//Takes value 1 if the household had no eligible men for an interview
lab var no_male_eligible "HH has no eligible men for child mortality ind"
lab values no_male_eligible lab_yn
drop hh_n_male_eligible
tab no_male_eligible, miss


*** No eligible children under 5
*** for child nutrition indicator
*****************************************
count if child_nutr==1 
count if hv120==1

gen	child_eligible = (hv120==1) 
lab var child_eligible "Child under 5 eligible for nutrition ind"	
lab values child_eligible lab_yn
tab child_eligible,miss

bysort	hh_id: egen hh_n_children_eligible = sum(child_eligible)  
	//Number of eligible children for anthropometrics
gen	no_child_eligible = (hh_n_children_eligible==0) 
	//Takes value 1 if there were no eligible children for anthropometrics
lab var no_child_eligible "HH has no eligible child under 5 for nutrition ind"
lab values no_child_eligible lab_yn
drop hh_n_children_eligible
tab no_child_eligible, miss

	/*hv120 has 3 categories: 0-not eligible; 
							  1-eligible,child of interviewed mother
							  2-eligible: defacto in household
	Children were not measured if their mother was not interviewed. As such 
	in constructing the child eligibility variable, we only consider children
	of interviewed mother as eligible.*/


*** No eligible women and men 
*** for adult nutrition indicator
***********************************************
gen	no_adults_eligible = (no_fem_nutri_eligible==1 & no_male_nutri_eligible==1) 
lab var no_adults_eligible "HH has no eligible women/men for nutrition ind"
lab values no_adults_eligible lab_yn
tab no_adults_eligible, miss 


*** No Eligible Children and Women
*** for child and women nutrition indicator 
***********************************************
gen	no_child_fem_eligible = (no_child_eligible==1 & no_fem_nutri_eligible==1)
lab var no_child_fem_eligible "HH has no eligible child/women for nutrition ind"
lab values no_child_fem_eligible lab_yn
tab no_child_fem_eligible, miss 


*** No Eligible Women, Men or Children 
*** for nutrition indicator 
***********************************************
gen no_eligibles = (no_fem_nutri_eligible==1 & ///
					no_male_nutri_eligible==1 & no_child_eligible==1)
lab var no_eligibles "HH has no eligible member for nutrition ind"
lab values no_eligibles lab_yn
tab no_eligibles, miss


sort hh_id ind_id
	

********************************************************************************
*** Step 1.12 RENAMING DEMOGRAPHIC VARIABLES ***
********************************************************************************

//Sample weight
desc hv005
clonevar weight = hv005
replace weight = weight/1000000 
label var weight "Sample weight"


//Area: urban or rural	
desc hv025
codebook hv025, tab (5)		
clonevar area = hv025  
replace area=0 if area==2  
label define lab_area 1 "urban" 0 "rural"
label values area lab_area
label var area "Area: urban-rural"


 
//Relationship to the head of household 
clonevar relationship = hv101 
codebook relationship, tab (20)
recode relationship (1=1)(2=2)(3=3)(11=3)(4/10=4)(15/16=4)(12=5)(17=6)(98=.)
label define lab_rel 1"head" 2"spouse" 3"child" 4"extended family" 5"not related" 6"maid"
label values relationship lab_rel
label var relationship "Relationship to the head of household"
tab hv101 relationship, miss


//Sex of household member	
codebook hv104
clonevar sex = hv104 
label var sex "Sex of household member"


//Household headship
bys	hh_id: egen missing_hhead = min(relation)
tab missing_hhead,m 
gen household_head=.
replace household_head=1 if relation==1 & sex==1 
replace household_head=2 if relation==1 & sex==2
bysort hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
label define head 1"male-headed" 2"female-headed"
label values headship head
label var headship "Household headship"
tab headship, miss


//Age of household member
codebook hv105, tab (100)
clonevar age = hv105  
replace age = . if age>=98
label var age "Age of household member"


//Age group 
recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+"), gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"
 
 
//Marital status of household member
clonevar marital = hv115 
codebook marital, tab (10)
recode marital (0=1)(1=2)(8=.)
	
	/*Note: In the context of India DHS 2005-06, an additional category was 
	introduced, that is, 'married but gauna not performed'. This category 
	is associated with the custom of child marriage. The gauna ceremony takes 
	place several years after marriage. Until the ceremony, couples stay apart.
    As such the category is recoded as 5"not living together". */
replace marital=5 if sh08==2	
label define lab_mar 1"never married" 2"currently married" 3"widowed" ///
4"divorced" 5"not living together"
label values marital lab_mar	
label var marital "Marital status of household member"
tab hv115 marital, miss


//Total number of de jure hh members in the household
gen member = 1
bysort hh_id: egen hhsize = sum(member)
label var hhsize "Household size"
tab hhsize, miss
drop member
 

	//Religion of the household head
lookfor religion
codebook sh44, tab (99)
recode sh44 (1=1 "Hindu") (2=2 "Muslim") (3=3 "Christian") ///
(4/96=4 "Other"), gen (religion_hh)
label var religion_hh "Religion of household head"
tab sh44 religion_hh,m
tab religion_hh,m



	//Caste of the household head
lookfor caste
tab sh46 sh45,m
codebook sh46, tab (99)
recode sh46 (1=1 "Scheduled Caste") (2=2 "Scheduled Tribe") ///
(3=3 "Other Backward Classes") (4/.=4 "Other"), gen (caste_hh)
replace caste_hh=. if sh46==. & sh45==.
label var caste_hh "Caste of household head"
	/* The recoded category 'other' includes those who reported being part 
		of a caste or tribe but do not know which caste/tribe they belong to. 
		In addition, this category includes those who reported no caste. The 
		number of observation for each of these categories are small, hence 
		we grouped them together for analysis purposes. */
tab sh46 caste_hh,m		
tab caste_hh,m


/*Subnational representation
	The sample for the India DHS 2005-06 was designed to provide 
	estimates of key indicators for the country as a whole, for urban and rural 
	areas separately, and for each of the 29 states. */
codebook hv024, tab (99)	   
clonevar region = hv024
lab var region "Region for subnational decomposition"
recode region   (1=14)(2=13)(3=28)(5=34)(6=12)(7=25)(8=29)(9=33)(10=5) ///
				(11=30)(12=3)(13=24)(14=21)(15=23)(16=32)(17=22)(18=4) ///
				(19=35)(20=15)(21=26)(22=7)(23=19)(24=11) ///
				(27=20)(28=2)(29=16)(30=10)(32=17)(33=31)
label define lab_reg 1"Andaman & Nicobar Islands" 2"Andhra Pradesh" ///
				 3"Arunachal Pradesh" 4"Assam" 5"Bihar" 6"Chandigarh" ///
				 7"Chhattisgarh" 8"Dadra and Nagar Haveli" 9"Daman and Diu" ///
				 10"Goa" 11"Gujarat" 12"Haryana" 13"Himachal Pradesh" ///
				 14"Jammu & Kashmir" 15"Jharkhand" 16"Karnataka" ///
				 17"Kerala" 18"Lakshadweep" 19"Madhya Pradesh" ///
				 20"Maharashtra" 21"Manipur" 22"Meghalaya" 23"Mizoram" ///
				 24"Nagaland" 25"Delhi" 26"Odisha" 27"Puducherry" 28"Punjab" ///
				 29"Rajasthan" 30"Sikkim" 31"Tamil Nadu" 32"Tripura" ///
				 33"Uttar Pradesh" 34"Uttarakhand" 35"West Bengal" ///
				 36"Telangana"				 								 
label values region lab_reg	
lab var region "Region for subnational decomposition"
tab hv024 region, miss

/* Changes over time:
1) Chandigarh does not exist in this 2005-06 survey, but exist in later 
surveys. We merged Chandigarh into Punjab for later surveys, so it is 
comparable to 2005-06.   

2) Telangana does not exist in this 2005-06 survey, but exist in later
surveys. We merged Telangana into Andhra Pradesh for later surveys, so it is comparable to 2005-06.

3) We also have 5 Union Territories where these areas were not sampled in 
this 2005/06 survey but sampled in later surveys. These were excluded from subnational analysis of later surveys, so it is comparable to 2005-06. */

clonevar region_01 = region
lab var region_01 "COT: Region for subnational decomposition"


********************************************************************************
*** Step 2.1 Years of Schooling ***
********************************************************************************

codebook hv108, tab(30)
clonevar  eduyears = hv108   
	//Total number of years of education
replace eduyears = . if eduyears>30
	//Recode any unreasonable years of highest education as missing value
tab hv108 hv106,miss	
replace eduyears = . if eduyears>=age & age>0
	/*The variable "eduyears" was replaced with a '.' if total years of 
	education was more than individual's age */
replace eduyears = 0 if age < 10 
replace eduyears = 0 if (age==10 | age==11) & eduyears < 6 
	/*The variable "eduyears" was replaced with a '0' for ineligible household 
	members, i.e.: those who have not completed 6 years of schooling following 
	their starting school age */
lab var eduyears "Total number of years of education accomplished"

	
	/*A control variable is created on whether there is information on 
	years of education for at least 2/3 of the eligible household members*/	
gen temp = 1 if eduyears!=. & age>=12 & age!=.
replace temp = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp = 1 if age==11 & eduyears>=6 & eduyears<.
bysort	hh_id: egen no_missing_edu = sum(temp)
	//Total eligible household members with no missing years of education
gen temp2 = 1 if age>=12 & age!=.
replace temp2 = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp2 = 1 if age==11 & eduyears>=6 & eduyears<.
bysort hh_id: egen hhs = sum(temp2)
	//Total number of eligible household members  
replace no_missing_edu = no_missing_edu/hhs
replace no_missing_edu = (no_missing_edu>=2/3)
	/* Identify whether there is information on years of education for at 
	least 2/3 of the eligible household members. */
tab no_missing_edu, miss
	//The values for 0 (missing) is 0.11%
label var no_missing_edu "No missing edu for at least 2/3 of the members"		
drop temp temp2 hhs


*** Standard MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed six years of schooling. */
*******************************************************************  
gen	 years_edu6 = (eduyears>=6)
replace years_edu6 = . if eduyears==.
bysort hh_id: egen hh_years_edu6_1 = max(years_edu6)
gen	hh_years_edu = (hh_years_edu6_1==1)
replace hh_years_edu = . if hh_years_edu6_1==.
replace hh_years_edu = . if hh_years_edu==0 & no_missing_edu==0 
lab var hh_years_edu "HH has at least one member with 6 years of edu"
tab hh_years_edu, miss

	
*** Destitution MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed at least one year of schooling. */
*******************************************************************
gen	years_edu1 = (eduyears>=1)
replace years_edu1 = . if eduyears==.
bysort	hh_id: egen hh_years_edu_u = max(years_edu1)
replace hh_years_edu_u = . if hh_years_edu_u==0 & no_missing_edu==0
lab var hh_years_edu_u "DST: HH has at least one member with 1 year of edu"
tab hh_years_edu_u, miss


********************************************************************************
*** Step 2.2 Child School Attendance ***
********************************************************************************

codebook hv121, tab (99)
clonevar attendance = hv121 
recode attendance (2=1) (8=.)
label define lab_attend 1 "attending" 0 "not attending"
label values attendance lab_attend
label var attendance "Attended school during current school year"
codebook attendance, tab (99)	
	

*** Standard MPI ***
/*The entire household is considered deprived if any 
school-aged child is not attending school up to class 8. */ 
******************************************************************* 
gen	child_schoolage = (age>=6 & age<=14)
lab var child_schoolage "Child eligible for sch att ind"
label values child_schoolage lab_yn
tab child_schoolage,m
	/* In India, the official school entrance age to primary school is 
	6 years. So, age range is 6-14 (=6+8) 
	Source: "http://data.uis.unesco.org/?ReportId=163"*/


	
	/* A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the school age children. */
count if child_schoolage==1 & attendance==.
	//How many eligible school aged children are not attending school: 181 children 
gen temp = 1 if child_schoolage==1 & attendance!=.
	/*Generate a variable that captures the number of eligible school aged 
	children who are attending school. */
bysort hh_id: egen no_missing_atten = sum(temp)	
	/*Total school age children with no missing information on school 
	attendance. */
gen temp2 = 1 if child_schoolage==1	
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are of school age
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)
	/*Identify whether there is missing information on school attendance for 
	more than 2/3 of the school age children. */			
tab no_missing_atten, miss
	//The values for 0 (missing) is 0.12%
label var no_missing_atten "No missing for at least 2/3 of the children"		
drop temp temp2 hhs
		
bysort hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
lab var hh_children_schoolage "HH has school aged children"

gen	not_atten = (attendance==0) if child_schoolage==1
replace not_atten = . if attendance==. & child_schoolage==1
bysort	hh_id: egen any_not_atten = max(not_atten)
gen	hh_sch_atten = (any_not_atten==0) 
replace hh_sch_atten = . if any_not_atten==.
replace hh_sch_atten = 1 if hh_children_schoolage==0
replace hh_sch_atten = . if hh_sch_atten==1 & no_missing_atten==0 
lab var hh_sch_atten "HH has all children up to class 8 in school"
label values hh_sch_atten lab_yn
tab hh_sch_atten, miss

	
*** Destitution MPI ***
/*The entire household is considered deprived if any 
school-aged child is not attending school up to class 6. */ 
******************************************************************* 
gen	child_schoolage_u = (age>=6 & age<=12)
lab var child_schoolage_u "DST: Child eligible for sch att ind"
label values child_schoolage_u lab_yn
tab child_schoolage_u,m 
	/* In India, the official school entrance age is 6 years.  
	  So, age range for destitution measure is 6-12 (=6+6). */

	
	/* A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the children attending school up to 
	class 6. */	
count if child_schoolage_u==1 & attendance==.	
gen temp = 1 if child_schoolage_u==1 & attendance!=.
bysort hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_u==1	
bysort hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)			
tab no_missing_atten_u, miss
label var no_missing_atten_u "DST: No missing for at least 2/3 of the children"		
drop temp temp2 hhs		
		
bysort	hh_id: egen hh_children_schoolage_u = sum(child_schoolage_u)
replace hh_children_schoolage_u = (hh_children_schoolage_u>0) 
lab var hh_children_schoolage_u "DST: HH has school aged children"

gen	atten_6 = (attendance==1) if child_schoolage_u==1
replace atten_6 = . if attendance==. & child_schoolage_u==1
bysort	hh_id: egen any_atten_6 = max(atten_6)
gen	hh_sch_atten_u = (any_atten_6==1) 
replace hh_sch_atten_u = . if any_atten_6==.
replace hh_sch_atten_u = 1 if hh_children_schoolage_u==0
replace hh_sch_atten_u = . if hh_sch_atten_u==0 & no_missing_atten_u==0 
lab var hh_sch_atten_u "DST: HH has all children up to class 6 in school"
label values hh_sch_atten_u lab_yn
tab hh_sch_atten_u, miss

	
********************************************************************************
*** Step 2.3 Nutrition ***
********************************************************************************


********************************************************************************
*** Step 2.3a Adult and Adolescent Nutrition ***
********************************************************************************


codebook ha40 hb40 
foreach var in ha40 hb40 {
			 gen inf_`var' = 1 if `var'!=.
			 bysort sex: tab age inf_`var' 
			 // women 15-49; men 15-54 
			 drop inf_`var'
			 }
***


*** BMI Indicator for Women 15-49 years ***
******************************************************************* 
gen	f_bmi = ha40/100
lab var f_bmi "Women's BMI"
gen	f_low_bmi = (f_bmi<18.5)
replace f_low_bmi = . if f_bmi==. | f_bmi>=99.97
lab var f_low_bmi "BMI of women < 18.5"
label values f_low_bmi lab_yn
tab f_low_bmi if fem_nutri_eligible==1,m


gen	f_low_bmi_u = (f_bmi<17)
replace f_low_bmi_u = . if f_bmi==. | f_bmi>=99.97
lab var f_low_bmi_u "BMI of women <17"
label values f_low_bmi_u lab_yn
tab f_low_bmi_u if fem_nutri_eligible==1,m


*** BMI Indicator for Men 15-54 years ***
******************************************************************* 
gen m_bmi = hb40/100
lab var m_bmi "Male's BMI"
gen m_low_bmi = (m_bmi<18.5)
replace m_low_bmi = . if m_bmi==. | m_bmi>=99.97 
lab var m_low_bmi "BMI of male < 18.5"
label values m_low_bmi lab_yn
tab m_low_bmi if male_nutri_eligible==1,m

gen	m_low_bmi_u = (m_bmi<17)
replace m_low_bmi_u = . if m_bmi==. | m_bmi>=99.97
lab var m_low_bmi_u "BMI of male <17"
label values m_low_bmi_u lab_yn
tab m_low_bmi_u if male_nutri_eligible==1,m


*** Standard MPI: BMI-for-age for individuals 15-19 years 
*** 				  and BMI for individuals 20-54 years ***
******************************************************************* 
gen low_bmi_byage = 0
lab var low_bmi_byage "Individuals with low BMI or BMI-for-age"
replace low_bmi_byage = 1 if f_low_bmi==1
	//Replace variable "low_bmi_byage = 1" if eligible women have low BMI	
replace low_bmi_byage = 1 if low_bmi_byage==0 & m_low_bmi==1 
	//Replace variable "low_bmi_byage = 1" if eligible men have low BMI. 

/* Note: The following command replaces BMI with BMI-for-age for those between 
the age group of 15-19 by their age in months where information is available */
	//Replacement for girls: 
replace low_bmi_byage = 1 if low_bmiage==1 & age_month!=.
replace low_bmi_byage = 0 if low_bmiage==0 & age_month!=.
	/*Replacements for boys: */
replace low_bmi_byage = 1 if low_bmiage_b==1 & age_month_b!=.
replace low_bmi_byage = 0 if low_bmiage_b==0 & age_month_b!=.
	
	
/*Note: The following control variable is applied when there is BMI information 
for adults and BMI-for-age for adolescents.*/	
replace low_bmi_byage = . if f_low_bmi==. & m_low_bmi==. & ///
							 low_bmiage==. & low_bmiage_b==. 
		
bysort hh_id: egen low_bmi = max(low_bmi_byage)
gen	hh_no_low_bmiage = (low_bmi==0)
	/*Households take a value of '1' if all eligible adults and adolescents in 
	the household has normal bmi or bmi-for-age */	
replace hh_no_low_bmiage = . if low_bmi==.
	/*Households take a value of '.' if there is no information from eligible 
	individuals in the household */
replace hh_no_low_bmiage = 1 if no_adults_eligible==1	
	//Households take a value of '1' if there is no eligible adult population.
drop low_bmi
lab var hh_no_low_bmiage "HH has no adult with low BMI or BMI-for-age"
tab hh_no_low_bmiage, miss	

	/* The 'hh_no_low_bmiage' var takes value 1 if: (a) no any eligible 
	individuals in the household has (observed) low BMI or (b) there are no 
	eligible individuals in the household. The variable takes values 0 for 
	those households that have at least one adult with observed low BMI. 
	The variable has a missing value only when there is missing info on BMI 
	for ALL eligible adults in the household. */

	
*** Destitution MPI: BMI-for-age for individuals 15-19 years 
*** 			     and BMI for individuals 20-54 years ***
********************************************************************************
gen low_bmi_byage_u = 0
lab var low_bmi_byage_u "DST: Individuals with low BMI or BMI-for-age"
replace low_bmi_byage_u = 1 if f_low_bmi_u==1
	/*Replace variable "low_bmi_byage_u = 1" if eligible women have low 
	BMI (destitute cutoff)*/	
replace low_bmi_byage_u = 1 if low_bmi_byage_u==0 & m_low_bmi_u==1 
	/*Replace variable "low_bmi_byage_u = 1" if eligible men have low 
	BMI (destitute cutoff).*/

/*Note: The following command replaces BMI with BMI-for-age for those between 
the age group of 15-19 by their age in months where information is available */
	//Replacement for girls: 
replace low_bmi_byage_u = 1 if low_bmiage_u==1 & age_month!=.
replace low_bmi_byage_u = 0 if low_bmiage_u==0 & age_month!=.
	/*Replacements for boys - if there is no male anthropometric data for boys, 
	then 0 changes are made: */
replace low_bmi_byage_u = 1 if low_bmiage_b_u==1 & age_month_b!=.
replace low_bmi_byage_u = 0 if low_bmiage_b_u==0 & age_month_b!=.
	
	
/*Note: The following control variable is applied when there is BMI 
information for adults and BMI-for-age for adolescents. */
replace low_bmi_byage_u = . if f_low_bmi_u==. & low_bmiage_u==. & ///
								m_low_bmi_u==. & low_bmiage_b_u==. 
	
	
bysort hh_id: egen low_bmi = max(low_bmi_byage_u)
gen	hh_no_low_bmiage_u = (low_bmi==0)
	/*Households take a value of '1' if all eligible adults and adolescents in
	the household has normal bmi or bmi-for-age (destitution cutoff). */
replace hh_no_low_bmiage_u = . if low_bmi==.
	/*Households take a value of '.' if there is no information from eligible 
	individuals in the household. */
replace hh_no_low_bmiage_u = 1 if no_adults_eligible==1	
	//Households take a value of '1' if there is no eligible adult population.
drop low_bmi
lab var hh_no_low_bmiage_u "DST: HH has no adult with low BMI or BMI-for-age"
tab hh_no_low_bmiage_u, miss		


********************************************************************************
*** Step 2.3b Child Nutrition ***
********************************************************************************

*** Child Underweight Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(underweight)
gen	hh_no_underweight = (temp==0) 
	//Takes value 1 if no child in the hh is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_underweight "HH has no child underweight"
label values underweight hh_no_underweight lab_yn
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(underweight_u)
gen	hh_no_underweight_u = (temp==0) 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
lab var hh_no_underweight_u "DST: HH has no child underweight"
label values underweight_u hh_no_underweight_u lab_yn
drop temp


*** Child Stunting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(stunting)
gen	hh_no_stunting = (temp==0) 
	//Takes value 1 if no child in the hh is stunted
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_stunting "HH has no child stunted"
label values stunting hh_no_stunting lab_yn
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(stunting_u)
gen	hh_no_stunting_u = (temp==0) 
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
lab var hh_no_stunting_u "DST: HH has no child stunted"
label values stunting_u hh_no_stunting_u lab_yn
drop temp


*** Child Wasting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(wasting)
gen	hh_no_wasting = (temp==0) 
	//Takes value 1 if no child in the hh is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_wasting "HH has no child wasted"
label values wasting hh_no_wasting lab_yn
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(wasting_u)
gen	hh_no_wasting_u = (temp==0) 
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
lab var hh_no_wasting_u "DST: HH has no child wasted"
label values wasting_u hh_no_wasting_u lab_yn
drop temp


*** Child Either Underweight or Stunted Indicator ***
************************************************************************

*** Standard MPI ***
gen hh_no_uw_st = 1 if hh_no_stunting==1 & hh_no_underweight==1
replace hh_no_uw_st = 0 if hh_no_stunting==0 | hh_no_underweight==0
	//Takes value 0 if child in the hh is stunted or underweight 
replace hh_no_uw_st = . if hh_no_stunting==. & hh_no_underweight==.
replace hh_no_uw_st = 1 if no_child_eligible==1
	//Households with no eligible children will receive a value of 1 
lab var hh_no_uw_st "HH has no child underweight or stunted"
label values hh_no_uw_st lab_yn


*** Destitution MPI  ***
gen hh_no_uw_st_u = 1 if hh_no_stunting_u==1 & hh_no_underweight_u==1
replace hh_no_uw_st_u = 0 if hh_no_stunting_u==0 | hh_no_underweight_u==0
replace hh_no_uw_st_u = . if hh_no_stunting_u==. & hh_no_underweight_u==.
replace hh_no_uw_st_u = 1 if no_child_eligible==1 
lab var hh_no_uw_st_u "DST: HH has no child underweight or stunted"
label values hh_no_uw_st_u lab_yn


********************************************************************************
*** Step 2.3c Household Nutrition Indicator ***
********************************************************************************

*** Standard MPI ***
/* Members of the household are considered deprived if any person under 
70 years of age for whom there is nutritional information is undernourished. Households that have no eligible person are considered non-deprived. The 
indicator takes a value of missing  if all eligible person have missing
information. */
************************************************************************
gen	hh_nutrition = 1
replace hh_nutrition = 0 if hh_no_low_bmiage==0 | hh_no_uw_st==0
replace hh_nutrition = . if hh_no_low_bmiage==. & hh_no_uw_st==.
	/*Replace indicator as missing if household has eligible adult and child 
	with missing nutrition information */
replace hh_nutrition = . if hh_no_low_bmiage==. & ///
						    hh_no_uw_st==1 & no_child_eligible==1
	/*Replace indicator as missing if household has eligible adult with missing 
	nutrition information and no eligible child for anthropometric measures */ 
replace hh_nutrition = . if hh_no_uw_st==. & ///
							hh_no_low_bmiage==1 & no_adults_eligible==1
	/*Replace indicator as missing if household has eligible child with missing 
	nutrition information and no eligible adult for anthropometric measures */ 
replace hh_nutrition = 1 if no_eligibles==1  
 	/*We replace households that do not have the applicable population, that is, 
	women 15-49 & children 0-5, as non-deprived in nutrition*/	
lab var hh_nutrition "HH has no individuals undernourished"
label values hh_nutrition lab_yn
tab hh_nutrition, miss


*** Destitution MPI ***
/* Members of the household are considered deprived if any person under 
70 years of age for whom there is nutritional information is severely undernourished. Households that have no eligible person are considered non-deprived. The indicator takes a value of missing  if all eligible 
person have missing information. */
************************************************************************
gen	hh_nutrition_u = 1
replace hh_nutrition_u = 0 if hh_no_low_bmiage_u==0 | hh_no_uw_st_u==0
replace hh_nutrition_u = . if hh_no_low_bmiage_u==. & hh_no_uw_st_u==.
	/*Replace indicator as missing if household has eligible adult and child 
	with missing nutrition information */
replace hh_nutrition_u = . if hh_no_low_bmiage_u==. & ///
							  hh_no_uw_st_u==1 & no_child_eligible==1
	/*Replace indicator as missing if household has eligible adult with missing 
	nutrition information and no eligible child for anthropometric measures */ 
replace hh_nutrition_u = . if hh_no_uw_st_u==. & ///
							  hh_no_low_bmiage_u==1 & no_adults_eligible==1
	/*Replace indicator as missing if household has eligible child with missing 
	nutrition information and no eligible adult for anthropometric measures */ 
replace hh_nutrition_u = 1 if no_eligibles==1   
 	/*We replace households that do not have the applicable population, that is, 
	women 15-49 & children 0-5, as non-deprived in nutrition*/	
lab var hh_nutrition_u "DST: HH has no individuals undernourished"
label values hh_nutrition_u lab_yn
tab hh_nutrition_u, miss
	

********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************
	
codebook v206 v207 mv206 mv207
	//v206 or mv206: number of sons who have died 
	//v207 or mv207: number of daughters who have died
	

egen temp_f = rowtotal(v206 v207), missing
	//Total child mortality reported by eligible women
replace temp_f = 0 if v201==0
	//This line replaces women who have never given birth
bysort	hh_id: egen child_mortality_f = sum(temp_f), missing
lab var child_mortality_f "Occurrence of child mortality reported by women"
tab child_mortality_f, miss
drop temp_f
	
	//Total child mortality reported by eligible men	
egen temp_m = rowtotal(mv206 mv207), missing
replace temp_m = 0 if mv201==0
bysort	hh_id: egen child_mortality_m = sum(temp_m), missing
lab var child_mortality_m "Occurrence of child mortality reported by men"
tab child_mortality_m, miss
drop temp_m

egen child_mortality = rowmax(child_mortality_f child_mortality_m)
lab var child_mortality "Total child mortality within HH"
tab child_mortality, miss	
	
	
*** Standard MPI *** 
/* Members of the household are considered deprived if a child under 18 has 
died in the household in the five-year period preceding the survey. Members 
of the household is considered non-deprived if household reported (i) no child mortality or (ii) if any child died longer than 5 years from the survey year or (iii) if any child 18 years and older died in the last 5 years. In adddition, members of the household were identified as non-deprived if eligible men 
within the household reported no child mortality in the absence of information 
from women. Households that have no eligible individuals are considered
non-deprived. The indicator takes a missing value if eligible individuals
have missing information. */
************************************************************************
tab childu18_died_per_wom_5y, miss
replace childu18_died_per_wom_5y = 0 if v201==0 
	/*Assign a value of "0" for:
	- all eligible women who never ever gave birth */	
replace childu18_died_per_wom_5y = 0 if no_fem_eligible==1 
	/*Assign a value of "0" for:
	- individuals living in households that have non-eligible women */	
	
bysort hh_id: egen childu18_mortality_5y = sum(childu18_died_per_wom_5y), missing
replace childu18_mortality_5y = 0 if childu18_mortality_5y==. & child_mortality==0
	/*Replace all households as 0 death if women has missing value and men 
	reported no death in those households */
label var childu18_mortality_5y "Total child under 18 mortality last 5 years"
tab childu18_mortality_5y, miss		
	
gen hh_child_mort = (childu18_mortality_5y==0)
replace hh_child_mort = . if childu18_mortality_5y==.
lab var hh_child_mort "HH had no child mortality"
label values hh_child_mort lab_yn
tab hh_child_mort, miss 


*** Destitution MPI *** 
*** (same as standard MPI) ***
************************************************************************
clonevar hh_child_mort_u = hh_child_mort	
lab var hh_child_mort_u "DST: HH had no child mortality"	


********************************************************************************
*** Step 2.5 Electricity ***
********************************************************************************

desc hv206
codebook hv206, tab (9)

*** Standard MPI ***
/* Members of the household are deprived 
if the household has no electricity. */
***************************************************
clonevar electricity = hv206 
label values electricity lab_yn
label var electricity "HH has electricity"
tab hv206 electricity,m


*** Destitution MPI  ***
*** (same as standard MPI) ***
***************************************************
clonevar electricity_u = electricity
label var electricity_u "DST: HH has electricity"


********************************************************************************
*** Step 2.6 Sanitation ***
********************************************************************************
/*
Improved sanitation facilities include flush or pour flush toilets to sewer 
systems, septic tanks or pit latrines, ventilated improved pit latrines, pit 
latrines with a slab, and composting toilets. These facilities are only 
considered improved if it is private, that is, it is not shared with other 
households.
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-02-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/


desc hv205 hv225
clonevar toilet_fac = hv205 
lab var toilet_fac "Type of toilet facility"

codebook hv225, tab(9)  
clonevar toilet_shr = hv225 
lab var toilet_shr "Shared toilet facility"	
	//0=no; 1=yes; .=missing
	

*** Standard MPI ***
/* Members of the household are deprived if the household has unimproved or 
no sanitation facility or it is improved but shared with other households. */
********************************************************************
codebook toilet_fac, tab(99) 
	
gen	toilet = (toilet_fac<23 | toilet_fac==41) 	
replace toilet = 0 if toilet_fac==14		
replace toilet = . if toilet_fac==. 
replace toilet = 0 if toilet_shr==1	
lab var toilet "HH has improved sanitation"
label values toilet lab_yn
ta toilet_fac toilet, m


*** Destitution MPI ***
/* Members of the household are deprived if the household 
   practices open defecation or use other sanitation facility. */
********************************************************************
gen	toilet_u = .

replace toilet_u = 0 if toilet_fac==31 | toilet_fac==96 	
replace toilet_u = 1 if toilet_fac!=31 & toilet_fac!=96 & ///
						toilet_fac!=. & toilet_fac!=99	
replace toilet_u = 0 if toilet_u==. & toilet_shr==1		
lab var toilet_u "DST: HH has improved sanitation"
label values toilet_u lab_yn
tab toilet_fac toilet_u, miss

	
********************************************************************************
*** Step 2.7 Drinking Water  ***
********************************************************************************
/*
Improved drinking water sources include the following: piped water into 
dwelling, yard or plot; public taps or standpipes; boreholes or tubewells; 
protected dug wells; protected springs; packaged water; delivered water and 
rainwater which is located on premises or is less than a 30-minute walk from 
home roundtrip. 
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-01-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/


desc hv201 hv204 hv202
clonevar water_src = hv201
lab var water_src "Source of drinking water"

clonevar timetowater = hv204  
lab var timetowater "Time to get to water source"


clonevar ndwater = hv202  
lab var ndwater "Source of non-drinking water"

tab ndwater if water==71 	
	/* Because the quality of bottled water is not known, households 
	using bottled water for drinking are classified as using an improved 
	or unimproved source according to their water source for non-drinking
	activities such as cooking and hand washing. */	


*** Standard MPI ***
/* Members of the household are deprived if the household's 
source of drinking water is not safe or safe drinking water 
is a 30-minute or longer walk from home, roundtrip. */
********************************************************************
codebook water_src, tab(99)
	
gen	water_grp = 1 if water_src<=31 | water_src==41 |  water_src==51  

replace water_grp = 0 if water_src==32 | water_src==42 | water_src==43 | ///
					     (water_src>=61 & water_src<=71) | water_src==96
			
replace water_grp = . if water_src==99 | water_src==.
label values water_grp lab_yn
lab var water_grp "HH has safe drinking water"
tab water_src water_grp, miss 	


codebook timetowater, tab(999)
	
clonevar water = water_grp	  
replace water = 0 if  timetowater>=30 & timetowater<995						 
lab var water "HH has safe drinking water, under 30 mins distance"
tab water, miss 

	
*** Destitution MPI ***
/* Members of the household is identified as destitute if household 
does not have access to safe drinking water, or safe water is more 
than 45 minute walk from home, round trip.*/
********************************************************************
clonevar water_u = water_grp						   
replace water_u = 0 if timetowater>45 & timetowater<995	 				   			  
lab var water_u "DST: HH has safe drinking water, 0-45 mins distance"
tab water_src water_u
	

*** Harmonised MPI: 2019/21 - 2015/16 - 2005/06 *** 
	/* Bottled water, water delivered via tanker truck and cart are 
	harmonised as improved following the later definitions of the survey. */
************************************************************************
codebook water_src, tab (99)

gen	water_grp_01 = 1 if water_src<=31 | water_src==41 | ///
					    (water_src>=51 & water_src<=72) 
						
replace water_grp_01 = 0 if water_src==32 | water_src==42 | ///
						    water_src==43 | water_src==96
							
replace water_grp_01 = . if water_src==99 | water_src==.							
label values water_grp_01 lab_yn
lab var water_grp_01 "COT(2): HH has safe drinking water"
tab water_src water_grp_01, miss


clonevar water_01 = water_grp_01						   
replace water_01 = 0 if timetowater>=30 & timetowater<995
lab var water_01 "COT(2): HH has safe drinking water, under 30 mins distance"	
tab water_src water_01, miss
tab water_01,m


clonevar water_u_01 = water_grp_01						 
replace water_u_01 = 0 if timetowater>45 & timetowater<995
lab var water_u_01 "COT-DST(2): HH has safe drinking water, 0-45 mins distance"
tab water_src water_u_01, miss
tab water_u_01,m


********************************************************************************
*** Step 2.8 Housing ***
********************************************************************************

desc hv213 hv214 hv215

/* Members of the household are deprived if 
the household has a dirt, sand or dung floor. */
clonevar floor = hv213 
lab var floor "Main floor material"
codebook floor, tab(99)
gen	floor_imp = 1
replace floor_imp = 0 if floor<=13 | floor==96  
replace floor_imp = . if floor==.	
lab var floor_imp "HH has improved floor"
label values floor_imp lab_yn
tab floor floor_imp, miss


/* Members of the household are deprived if the household has walls 
made of natural or rudimentary materials. We followed the report's 
definitions of natural or rudimentary materials. */
clonevar wall = hv214 
lab var wall "Main wall material"
codebook wall, tab(99)	
gen	wall_imp = 1 
replace wall_imp = 0 if wall<=26 | wall==96  	
replace wall_imp = . if wall==.	
lab var wall_imp "HH has improved wall"
label values wall_imp lab_yn
tab wall wall_imp, miss	


/* Members of the household are deprived if the household has roof 
made of natural or rudimentary materials. We followed the report's 
definitions of natural and rudimentary materials. */
clonevar roof = hv215
lab var roof "Main roof material"
codebook roof, tab(99)		
gen	roof_imp = 1  
replace roof_imp = 0 if roof<=25 | roof==96 
replace roof_imp = . if roof==.	
lab var roof_imp "HH has improved roof"
label values wall_imp lab_yn
tab roof roof_imp, miss


*** Standard MPI ***
/* Members of the household is deprived in housing if the roof, 
floor OR walls are constructed from low quality materials.*/
**************************************************************
gen housing = 1
replace housing = 0 if floor_imp==0 | wall_imp==0 | roof_imp==0
replace housing = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing "HH is built from improved materials"
label values housing lab_yn
tab housing, miss


*** Destitution MPI ***
/* Members of the household is deprived in housing if two out 
of three components (roof and walls; OR floor and walls; OR 
roof and floor) the are constructed from low quality materials. */
**************************************************************
gen housing_u = 1
replace housing_u = 0 if (floor_imp==0 & wall_imp==0 & roof_imp==1) | ///
						 (floor_imp==0 & wall_imp==1 & roof_imp==0) | ///
						 (floor_imp==1 & wall_imp==0 & roof_imp==0) | ///
						 (floor_imp==0 & wall_imp==0 & roof_imp==0)
replace housing_u = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_u "DST: HH is built from improved materials"
label values housing_u lab_yn
tab housing_u, miss


********************************************************************************
*** Step 2.9 Cooking Fuel ***
********************************************************************************
/*
Solid fuel are solid materials burned as fuels, which includes coal as well as 
solid biomass fuels (wood, animal dung, crop wastes and charcoal). 

Source: 
https://apps.who.int/iris/bitstream/handle/10665/141496/9789241548885_eng.pdf
*/

desc hv226
clonevar cookingfuel = hv226
lab var cookingfuel "Type of cooking fuel"


*** Standard MPI ***
/* Members of the household are deprived if the household 
uses solid fuels and solid biomass fuels for cooking. */
*****************************************************************
codebook cookingfuel, tab(99)

gen	cooking = 1
replace cooking = 0 if cookingfuel>5 & cookingfuel<95 
replace cooking = . if cookingfuel==. | cookingfuel==99
	/* Deprived if: coal/lignite, charcoal, wood, straw/shrubs/grass, 
					agricultural crop, animal dung. */	
					
lab var cooking "HH uses clean cooking fuel"
label values cooking lab_yn			 
tab cookingfuel cooking, miss
tab cooking, miss
	
	
*** Destitution MPI ***
*** (same as standard MPI) ***
****************************************
gen	cooking_u = cooking
lab var cooking_u "DST: HH uses clean cooking fuel"
label values cooking_u lab_yn	
tab cookingfuel cooking_u, miss


********************************************************************************
*** Step 2.10 Assets ownership ***
********************************************************************************

clonevar television = hv208 
clonevar bw_television = sh47i 
lab var television "HH has television"

clonevar radio = hv207
lab var radio "HH has radio"
 
clonevar telephone =  hv221 
clonevar mobiletelephone = hv243a 
lab var telephone "HH has telephone (landline/mobilephone)"	

clonevar refrigerator = hv209 
lab var refrigerator "HH has refrigerator"

clonevar car = hv212 
lab var car "HH has car/truck/van"	
 
clonevar bicycle = hv210 
lab var bicycle "HH has bicycle"

clonevar motorbike = hv211 
lab var motorbike "HH has motorbike"
	
clonevar computer = sh47n
lab var computer "HH has computer"
	
clonevar animal_cart = hv243c
lab var animal_cart "HH has animal cart"	


foreach var in television radio telephone mobiletelephone refrigerator ///
			   car bicycle motorbike computer animal_cart {
replace `var' = . if `var'==9 | `var'==99 | `var'==8 | `var'==98 
label values `var' lab_yn
}

	//Merge telephone
replace telephone=1 if telephone==0 & mobiletelephone==1
replace telephone=1 if telephone==. & mobiletelephone==1

	//Merge television
replace television=1 if television==0 & bw_television==1
replace television=1 if television==. & bw_television==1



*** Standard MPI ***
/* Members of the household are deprived if the household does not own 
more than one of these assets: radio, TV, telephone, bike, motorbike, 
refrigerator, computer or animal cart and does not own a car or truck.*/
*****************************************************************************
egen n_small_assets = rowtotal(television radio telephone refrigerator bicycle motorbike computer animal_cart), m
lab var n_small_assets "Number of small assets owned by HH" 
   
gen assets = (car==1 | n_small_assets > 1) 
replace assets = . if car==. & n_small_assets==.
lab var assets "HH has assets"
label values assets lab_yn
tab n_small_assets assets, miss
tab assets, miss


*** Destitution MPI ***
/* Members of the household are considered deprived if 
the household does not own any of these assets. */
*****************************************************************************	
gen	assets_u = (car==1 | n_small_assets>0)
replace assets_u = . if car==. & n_small_assets==.
lab var assets_u "DST: HH has assets"
label values assets_u lab_yn
tab n_small_assets assets_u, miss
tab assets_u, miss

	
	
********************************************************************************
*** Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************

	//Retain data on sampling design: 
codebook hv022 hv021
	/* No observation for hv022 & hv021. So need 
	to construct the strata & psu variable. */
		
replace shcity=0 if shcity==.
replace shslumc=0 if shslumc==.
gen strata=hv024*1000+shcity*100+shslumc*10+hv025
codebook strata
	
clonevar psu = hv001
label var psu "Primary sampling unit"
label var strata "Sample strata"
	
	
	//Retain year, month & date of interview:
desc hv007 hv006 hv008
clonevar year_intvw = hv007 	
clonevar month_intvw = hv006 
clonevar date_intvw = hv008


*** Keep main variables require for MPI calculation ***
keep hh_id ind_id subsample strata psu weight weight_ch area ///
region region 01 age agec7 agec4 agec2 sex relationship headship ///
hhsize marital religion_hh caste_hh year_intvw month_intvw date_intvw ///
fem_nutri_eligible no_fem_nutri_eligible ///
male_nutri_eligible no_male_nutri_eligible ///
child_eligible no_child_eligible ///
fem_eligible no_fem_eligible male_eligible no_male_eligible ///
eduyears no_missing_edu hh_years_edu hh_years_edu_u ///
attendance child_schoolage child_schoolage_u ///
hh_children_schoolage hh_children_schoolage_u ///
no_missing_atten no_missing_atten_u hh_sch_atten hh_sch_atten_u ///
f_bmi f_low_bmi f_low_bmi_u m_bmi m_low_bmi m_low_bmi_u ///
age_month age_month_b low_bmiage low_bmiage_u low_bmiage_b low_bmiage_b_u ///
low_bmi_byage low_bmi_byage_u hh_no_low_bmiage hh_no_low_bmiage_u ///
underweight underweight_u hh_no_underweight hh_no_underweight_u ///
stunting stunting_u hh_no_stunting hh_no_stunting_u ///
wasting wasting_u hh_no_wasting hh_no_wasting_u ///
hh_no_uw_st hh_no_uw_st_u hh_nutrition hh_nutrition_u ///
child_mortality_f child_mortality childu18_mortality_5y ///
hh_child_mort hh_child_mort_u ///
electricity electricity_u toilet_fac toilet_shr toilet toilet_u /// 
water_src timetowater water water_u ///
floor wall roof floor_imp wall_imp roof_imp housing housing_u ///
cookingfuel cooking cooking_u ///
television radio telephone refrigerator bicycle motorbike computer ///
animal_cart car n_small_assets assets assets_u  water_01 water_u_01 


*** Order file	***
order hh_id ind_id subsample strata psu weight weight_ch area ///
region region 01 age agec7 agec4 agec2 sex relationship headship ///
hhsize marital religion_hh caste_hh year_intvw month_intvw date_intvw ///
fem_nutri_eligible no_fem_nutri_eligible ///
male_nutri_eligible no_male_nutri_eligible ///
child_eligible no_child_eligible ///
fem_eligible no_fem_eligible male_eligible no_male_eligible ///
eduyears no_missing_edu hh_years_edu hh_years_edu_u ///
attendance child_schoolage child_schoolage_u ///
hh_children_schoolage hh_children_schoolage_u ///
no_missing_atten no_missing_atten_u hh_sch_atten hh_sch_atten_u ///
f_bmi f_low_bmi f_low_bmi_u m_bmi m_low_bmi m_low_bmi_u ///
age_month age_month_b low_bmiage low_bmiage_u low_bmiage_b low_bmiage_b_u ///
low_bmi_byage low_bmi_byage_u hh_no_low_bmiage hh_no_low_bmiage_u ///
underweight underweight_u hh_no_underweight hh_no_underweight_u ///
stunting stunting_u hh_no_stunting hh_no_stunting_u ///
wasting wasting_u hh_no_wasting hh_no_wasting_u ///
hh_no_uw_st hh_no_uw_st_u hh_nutrition hh_nutrition_u ///
child_mortality_f child_mortality childu18_mortality_5y ///
hh_child_mort hh_child_mort_u ///
electricity electricity_u toilet_fac toilet_shr toilet toilet_u /// 
water_src timetowater water water_u ///
floor wall roof floor_imp wall_imp roof_imp housing housing_u ///
cookingfuel cooking cooking_u ///
television radio telephone refrigerator bicycle motorbike computer ///
animal_cart car n_small_assets assets assets_u  water_01 water_u_01 
 
	 
*** Rename key global MPI indicators for estimation ***
recode hh_child_mort  (0=1)(1=0) , gen(d_cm)
recode hh_nutrition	  (0=1)(1=0) , gen(d_nutr)
recode hh_sch_atten   (0=1)(1=0) , gen(d_satt)
recode hh_years_edu   (0=1)(1=0) , gen(d_educ)
recode electricity 	  (0=1)(1=0) , gen(d_elct)
recode water	 	  (0=1)(1=0) , gen(d_wtr)
recode toilet	 	  (0=1)(1=0) , gen(d_sani)
recode housing	 	  (0=1)(1=0) , gen(d_hsg)
recode cooking	 	  (0=1)(1=0) , gen(d_ckfl)
recode assets 		  (0=1)(1=0) , gen(d_asst)
 

*** Rename key global MPI indicators for destitution estimation ***
recode hh_child_mort_u  (0=1)(1=0) , gen(dst_cm)
recode hh_nutrition_u 	(0=1)(1=0) , gen(dst_nutr)
recode hh_sch_atten_u 	(0=1)(1=0) , gen(dst_satt)
recode hh_years_edu_u 	(0=1)(1=0) , gen(dst_educ)
recode electricity_u	(0=1)(1=0) , gen(dst_elct)
recode water_u 			(0=1)(1=0) , gen(dst_wtr)
recode toilet_u 		(0=1)(1=0) , gen(dst_sani)
recode housing_u 		(0=1)(1=0) , gen(dst_hsg)
recode cooking_u		(0=1)(1=0) , gen(dst_ckfl)
recode assets_u 		(0=1)(1=0) , gen(dst_asst) 
 

*** Rename indicators for changes over time estimation ***
recode hh_child_mort  	(0=1)(1=0) , gen(d_cm_01)
recode hh_nutrition 	(0=1)(1=0) , gen(d_nutr_01)
recode hh_sch_atten 	(0=1)(1=0) , gen(d_satt_01)
recode hh_years_edu 	(0=1)(1=0) , gen(d_educ_01)
recode electricity 		(0=1)(1=0) , gen(d_elct_01)
recode water_01	 		(0=1)(1=0) , gen(d_wtr_01)
recode toilet 		 	(0=1)(1=0) , gen(d_sani_01)
recode housing  		(0=1)(1=0) , gen(d_hsg_01)
recode cooking	 		(0=1)(1=0) , gen(d_ckfl_01)
recode assets 			(0=1)(1=0) , gen(d_asst_01)	
	

recode hh_child_mort_u   (0=1)(1=0) , gen(dst_cm_01)
recode hh_nutrition_u    (0=1)(1=0) , gen(dst_nutr_01)
recode hh_sch_atten_u 	 (0=1)(1=0) , gen(dst_satt_01)
recode hh_years_edu_u 	 (0=1)(1=0) , gen(dst_educ_01)
recode electricity_u	 (0=1)(1=0) , gen(dst_elct_01)
recode water_u_01  		 (0=1)(1=0) , gen(dst_wtr_01)
recode toilet_u 		 (0=1)(1=0) , gen(dst_sani_01)
recode housing_u 		 (0=1)(1=0) , gen(dst_hsg_01)
recode cooking_u		 (0=1)(1=0) , gen(dst_ckfl_01)
recode assets_u 		 (0=1)(1=0) , gen(dst_asst_01) 	


label define lab_dp 1 "deprived" 0 "non-deprived"
foreach var in d_* dst_* d_*_* dst_*_* {
label values `var' lab_dp
} 
	

	
*** Generate coutry and survey details for estimation ***
char _dta[cty] "India"
char _dta[ccty] "IND"
char _dta[year] "2005-2006" 	
char _dta[survey] "DHS"
char _dta[ccnum] "356"
char _dta[type] "micro"


*** Sort, compress and save data for estimation ***
sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/ind_dhs05-06.dta", replace 	
