********************************************************************************
/*
Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
Global Multidimensional Poverty Index - Indonesia DHS 2017 
[STATA do-file]. Available from OPHI website: http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk
*/
********************************************************************************

clear all 
set more off
set maxvar 10000


global path_in "../rdta/Indonesia DHS 2017" 	// folder path
global path_out "cdta"
global path_ado "ado"


********************************************************************************
*** Step 1: Data preparation 
********************************************************************************


********************************************************************************
*** Step 1.1 PR - Children under 5 years
********************************************************************************

	//No anthropometric data

********************************************************************************
*** Step 1.2  BR - BIRTH RECODE 
*** (All females 15-49 years who ever gave birth)  
********************************************************************************

use "$path_in/IDBR71FL.dta", clear

	
gen double ind_id = v001*1000000 + v002*100 + v003 
format ind_id %20.0g


desc b3 b7	
gen date_death = b3 + b7
	//Date of death = date of birth (b3) + age at death (b7)
gen mdead_survey = v008 - date_death
	//Months dead from survey = Date of interview (v008) - date of death
gen ydead_survey = mdead_survey/12
	//Years dead from survey
	
gen age_death = b7	
label var age_death "Age at death in months"
tab age_death, miss
	//Age is in months		
	
codebook b5, tab (10)	
gen child_died = 1 if b5==0
replace child_died = 0 if b5==1
replace child_died = . if b5==.
label define lab_died 1 "child has died" 0 "child is alive"
label values child_died lab_died
tab b5 child_died, miss
	

	/*NOTE: For each woman, sum the number of children who died and compare to 
	the number of sons/daughters whom they reported have died */
bysort ind_id: egen tot_child_died = sum(child_died) 
egen tot_child_died_2 = rsum(v206 v207)
	//v206: sons who have died; v207: daughters who have died
compare tot_child_died tot_child_died_2
	//Indonesia DHS 2017: these figures are identical.
	
	
	//Identify child under 18 mortality in the last 5 years
gen child18_died = child_died 
replace child18_died=0 if age_death>=216 & age_death<.
label define lab_u18died 1 "child u18 has died" 0 "child is alive/died but older"
label values child18_died lab_u18died
tab child18_died, miss	
			
bysort ind_id: egen tot_child18_died_5y=sum(child18_died) if ydead_survey<=5
	/*Total number of children under 18 who died in the past 5 years 
	prior to the interview date */	
	
replace tot_child18_died_5y=0 if tot_child18_died_5y==. & tot_child_died>=0 & tot_child_died<.
	/*All children who are alive or who died longer than 5 years from the 
	interview date are replaced as '0'*/
	
replace tot_child18_died_5y=. if child18_died==1 & ydead_survey==.
	//Replace as '.' if there is no information on when the child died  

tab tot_child_died tot_child18_died_5y, miss

bysort ind_id: egen childu18_died_per_wom_5y = max(tot_child18_died_5y)
lab var childu18_died_per_wom_5y "Total child under 18 death for each women in the last 5 years (birth recode)"
	

	//Keep one observation per women
bysort ind_id: gen id=1 if _n==1
keep if id==1
drop id
duplicates report ind_id 

gen women_BR = 1 
	//Identification variable for observations in BR recode: 34,199

	
	//Retain relevant variables
keep ind_id women_BR childu18_died_per_wom_5y 
order ind_id women_BR childu18_died_per_wom_5y
sort ind_id
save "$path_out/IDN17_BR.dta", replace	

	
********************************************************************************
*** Step 1.3  IR - WOMEN's RECODE  
*** (Eligible female 15-49 years in the household)
********************************************************************************
/*The purpose of step 1.3 is to identify all deaths that are reported by 
eligible women.*/

use "$path_in/IDIR71FL.dta", clear

	
*** Generate individual unique key variable required for data merging
*** Hv001=cluster number;  
*** Hv002=household number; 
*** v003=respondent's line number
gen double ind_id = v001*1000000 + v002*100 + v003 
format ind_id %20.0g
label var ind_id "Individual ID"

duplicates report ind_id 

tab v012, miss
codebook v201 v206 v207,tab (999)	
	/*Indonesia DHS 2017: Fertility and mortality question was only 
	collected from women 15-49 years (p. 2).*/
	
gen women_IR=1 
	//Identification variable for observations in IR recode

	

	//Retain relevant variables:
keep ind_id women_IR v003 v005 v012 v201 v206 v207 
order ind_id women_IR v003 v005 v012 v201 v206 v207 
sort ind_id
save "$path_out/IDN17_IR.dta", replace

	
********************************************************************************
*** Step 1.4  PR - INDIVIDUAL RECODE  
*** (Girls 15-19 years in the household)
********************************************************************************
/*The purpose of step 1.4 is to compute bmi-for-age for girls 15-19 years. */

/*Indonesia DHS 2017: Anthropometric data was not collected. 
So there is no anthropometric information for girls 15-19 years. Hence the 
command lines under this section have been removed.*/


********************************************************************************
*** Step 1.5  MR - MEN'S RECODE  
***(Eligible man 15-54 years in the household) 
********************************************************************************
/*The purpose of step 1.5 is to identify all deaths that are reported by 
eligible men.*/

use "$path_in/IDMR71FL.dta", clear 

	
*** Generate individual unique key variable required for data merging
	*** mv001=cluster number; 
	*** mv002=household number;
	*** mv003=respondent's line number
gen double ind_id = mv001*1000000 + mv002*100 + mv003 	
format ind_id %20.0g
label var ind_id "Individual ID"

duplicates report ind_id 

tab mv012, miss
codebook mv201 mv206 mv207,tab (999)
	/*Indonesia DHS 2017: Mortality question was collected using the Married 
	Men's Questionnaire. The questionnaire was administered on all married men
	15-54 years living in 8 of the 25 households in the sampled census block 
	(p.3). However, the number of men 15-19 years who were interviewed are very 
	low in number (see footnote in p.248-9).*/
	
gen men_MR=1 	
	//Identification variable for observations in MR recode

	
keep ind_id men_MR mv003 mv005 mv012 mv201 mv206 mv207  
order ind_id men_MR mv003 mv005 mv012 mv201 mv206 mv207 
sort ind_id
save "$path_out/IDN17_MR.dta", replace

	
********************************************************************************
*** Step 1.6  PR - INDIVIDUAL RECODE  
*** (Boys 15-19 years in the household)
********************************************************************************
/*The purpose of step 1.6 is to compute bmi-for-age for boys 15-19 years. */

/*Indonesia DHS 2017: Anthropometric data was not collected. 
So there is no anthropometric information for boys 15-19 years. Hence the 
command lines under this section have been removed.*/
	
	
********************************************************************************
*** Step 1.7  PR - HOUSEHOLD MEMBER'S RECODE 
********************************************************************************

use "$path_in/IDPR71FL.dta", clear

	
*** Generate a household unique key variable at the household level using: 
	***hv001=cluster number 
	***hv002=household number
gen double hh_id = hv001*10000 + hv002 
format hh_id %20.0g
label var hh_id "Household ID"
codebook hh_id  


*** Generate individual unique key variable required for data merging using:
	*** hv001=cluster number; 
	*** hv002=household number; 
	*** hvidx=respondent's line number.
gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g
label var ind_id "Individual ID"
codebook ind_id


sort hh_id ind_id

	
********************************************************************************
*** Step 1.8 DATA MERGING 
******************************************************************************** 
 
 
*** Merging BR Recode 
*****************************************
merge 1:1 ind_id using "$path_out/IDN17_BR.dta"
drop _merge
erase "$path_out/IDN17_BR.dta"


*** Merging IR Recode 
*****************************************
merge 1:1 ind_id using "$path_out/IDN17_IR.dta"
tab women_IR hv117, miss col 
tab ha65 if hv117==1 & women_IR ==., miss 
	//Total number of eligible women not interviewed: 1,103 (Table 1, p. 5) 
drop _merge
erase "$path_out/IDN17_IR.dta"


/*Check if the number of women in BR recode matches the number of those
who provided birth history information in IR recode. */
count if women_BR==1 
count if v201!=0 & v201!=. & women_IR==1 


/*Check if the number of women in BR and IR recode who provided birth history 
information matches with the number of eligible women identified by hv117. */
count if hv117==1 
count if women_BR==1 | v201==0 
count if (women_BR==1 | v201==0) & hv117==1 
tab v201 if hv117==1, miss  
tab v201 ha65 if hv117==1, miss 
	/*Note: 2.17% eligible women did not provide information on their birth 
	history. This will result in missing value for the child mortality 
	indicator that we will construct later */

	
*** Merging MR Recode 
*****************************************
merge 1:1 ind_id using "$path_out/IDN17_MR.dta"
tab men_MR hv118 if hv027==1, miss col 									  
drop _merge
erase "$path_out/IDN17_MR.dta"	



sort ind_id


********************************************************************************
*** Step 1.9 KEEP ONLY DE JURE HOUSEHOLD MEMBERS ***
********************************************************************************
/*The Global MPI is based on de jure (permanent) household members only. As 
such, non-usual residents will be excluded from the sample. */

clonevar resident = hv102 
tab resident, miss 
label var resident "Permanent (de jure) household member"

drop if resident!=1 
tab resident, miss
	/* Indonesia DHS 2017: 4,319 (2.18%) individuals who were non-usual 
	residents were dropped from the sample. */

********************************************************************************
*** Step 1.10 KEEP HOUSEHOLDS SELECTED FOR ANTHROPOMETRIC SUBSAMPLE ***
*** if relevant
********************************************************************************

/*
In the context of Indonesia DHS 2017, anthropometric data was not collected 
as part of the Indonesian DHS 2017 dataset. As such there was no subsample 
selection as part of the survey. Given this, the variable 'subsample' is 
generated with missing observations. 
*/

gen subsample=.
label var subsample "Households selected as part of nutrition subsample"

	
********************************************************************************
*** Step 1.11 CONTROL VARIABLES
********************************************************************************
/* Households are identified as having 'no eligible' members if there are no 
applicable population, that is, children 0-5 years, adult women 15-49 years or 
men 15-54 years. These households will not have information on relevant 
indicators of health. As such, these households are considered as non-deprived 
in those relevant indicators. */


*** No eligible women 15-49 years 
*** for adult nutrition indicator
***********************************************
//Indonesia DHS 2017 has no female anthropometric data.
gen	fem_nutri_eligible = .
gen	no_fem_nutri_eligible = .
lab var no_fem_nutri_eligible "Household has no eligible women for anthropometric"	


*** No eligible women 15-49 years 
*** for child mortality indicator
*****************************************
gen	fem_eligible = (hv117==1)
bysort	hh_id: egen hh_n_fem_eligible = sum(fem_eligible) 	
	//Number of eligible women for interview in the hh
gen	no_fem_eligible = (hh_n_fem_eligible==0) 									
	//Takes value 1 if the household had no eligible women for an interview
lab var no_fem_eligible "Household has no eligible women for interview"
drop fem_eligible hh_n_fem_eligible 
tab no_fem_eligible, miss


*** No eligible men 15-54 years 
*** for adult nutrition indicator (if relevant)
***********************************************
	//Indonesia DHS 2017 has no male anthropometric data.
gen male_nutri_eligible = .	
gen	no_male_nutri_eligible = .
lab var no_male_nutri_eligible "Household has no eligible men for anthropometric"	


*** No eligible men 15-54 years
*** for child mortality indicator (if relevant)
*****************************************
gen	male_eligible = (hv118==1)
bysort	hh_id: egen hh_n_male_eligible = sum(male_eligible)  
	//Number of eligible men for interview in the hh
gen	no_male_eligible = (hh_n_male_eligible==0) 	
	//Takes value 1 if the household had no eligible men for an interview
lab var no_male_eligible "Household has no eligible man for interview"
drop male_eligible hh_n_male_eligible
tab no_male_eligible, miss


*** No eligible children under 5
*** for child nutrition indicator
*****************************************
	//Indonesia DHS 2017 has no anthropometric data.
gen	child_eligible = .	
gen	no_child_eligible = .
lab var no_child_eligible "Household has no children eligible for anthropometric"


*** No eligible women and men 
*** for adult nutrition indicator
***********************************************
	//Indonesia DHS 2017 has no anthropometric data.
gen no_adults_eligible = . 
lab var no_adults_eligible "Household has no eligible women or men for anthropometrics"


*** No Eligible Children and Women
*** for child and women nutrition indicator 
***********************************************
	//Indonesia DHS 2017 has no anthropometric data.
gen	no_child_fem_eligible = .
lab var no_child_fem_eligible "Household has no children or women eligible for anthropometric"


*** No Eligible Women, Men or Children 
*** for nutrition indicator 
***********************************************
	//Indonesia DHS 2017 has no anthropometric data.
gen no_eligibles = .
lab var no_eligibles "Household has no eligible women, men, or children"

sort hh_id ind_id


********************************************************************************
*** Step 1.12 RENAMING DEMOGRAPHIC VARIABLES ***
********************************************************************************

//Sample weight
	/*Note: DHS sample weight are calculated to six decimals but are presented 
	in the standard recode files without the decimal point. As such, all DHS 
	weights should be divided by 1,000,000 before applying the weights to 
	calculation or analysis. */
desc hv005
clonevar weight = hv005
replace weight = weight/1000000 
label var weight "Sample weight"


//Area: urban or rural	
desc hv025
codebook hv025, tab (5)		
clonevar area = hv025  
replace area=0 if area==2  
label define lab_area 1 "urban" 0 "rural"
label values area lab_area
label var area "Area: urban-rural"


//Relationship to the head of household 
clonevar relationship = hv101 
codebook relationship, tab (20)
recode relationship (1=1)(2=2)(3=3)(11=3)(4/10=4)(15/16=4)(12=5)(17=6)(98=.)
label define lab_rel 1"head" 2"spouse" 3"child" 4"extended family" 5"not related" 6"maid"
label values relationship lab_rel
label var relationship "Relationship to the head of household"
tab hv101 relationship, miss


//Sex of household member	
codebook hv104
clonevar sex = hv104 
label var sex "Sex of household member"


//Household headship
bys	hh_id: egen missing_hhead = min(relationship)
tab missing_hhead,m 
gen household_head=.
replace household_head=1 if relationship==1 & sex==1 
replace household_head=2 if relationship==1 & sex==2
bysort hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
label define head 1"male-headed" 2"female-headed"
label values headship head
label var headship "Household headship"
tab headship, miss


//Age of household member
codebook hv105, tab (999)
clonevar age = hv105  
replace age = . if age>=98
label var age "Age of household member"


//Age group 
recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+"), gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"


//Marital status of household member
clonevar marital = hv115 
codebook marital, tab (10)
recode marital (0=1)(1=2)(8=.)
label define lab_mar 1"never married" 2"currently married" 3"widowed" ///
4"divorced" 5"not living together"
label values marital lab_mar	
label var marital "Marital status of household member"
tab hv115 marital, miss


//Total number of de jure hh members in the household
gen member = 1
bysort hh_id: egen hhsize = sum(member)
label var hhsize "Household size"
tab hhsize, miss
drop member


//Subnational region
	/*The sample for the Indonesia DHS 2017 was designed to provide 
	reliable estimates at national and 34 provincial level (p.384 of 623).*/   
codebook hv024, tab (99)
decode hv024, gen(temp)
replace temp =  proper(temp)
encode temp, gen(region)
drop temp
lab var region "Region for subnational decomposition"
codebook region, tab (99)


	/*Harmonisation: North Kalimantan is regrouped into East Kalimantan. 
	This is because in earlier survey, North was part of East Kalimantan.*/
gen region_01 = region
recode region_01 (17=10)(18=17)(19=18)(20=19)(21=20)(22=21)(23=22) ///
				 (24=23)(25=24)(26=25)(27=26)(28=27)(29=28)(30=29) ///
				 (31=30)(32=31)(33=32)(34=33)
label define lab_reg ///
1"Aceh" 2"Bali" 3"Bangka Belitung" 4"Banten" 5"Bengkulu" 6"Central Java" ///
7"Central Kalimantan" 8"Central Sulawesi" 9"East Java" 10"East Kalimantan" ///
11"East Nusa Tenggara" 12"Gorontalo" 13"Jakarta" 14"Jambi" 15"Lampung" ///
16"Maluku" 17"North Maluku" 18"North Sulawesi" 19"North Sumatera" ///
20"Papua" 21"Riau" 22"Riau Islands" 23"South Kalimantan" 24"South Sulawesi" ///
25"South Sumatera" 26"Southeast Sulawesi" 27"West Java" 28"West Kalimantan" ///
29"West Nusa Tenggara" 30"West Papua" 31"West Sulawesi" 32"West Sumatera" ///
33"Yogyakarta"
label values region_01 lab_reg
codebook region_01, tab (99)


********************************************************************************
***  Step 2 Data preparation  ***
***  Standardization of the 10 Global MPI indicators 
***  Identification of non-deprived & deprived individuals  
********************************************************************************


********************************************************************************
*** Step 2.1 Years of Schooling ***
********************************************************************************

codebook hv108, tab(30)
clonevar  eduyears = hv108   
	//Total number of years of education
replace eduyears = . if eduyears>30
	//Recode any unreasonable years of highest education as missing value	
codebook hv106, tab (99)
tab hv108 hv106,m
count if eduyears==. & (hv106==2 | hv106==3)	
replace eduyears = 6 if eduyears==. & (hv106==2 | hv106==3)
	/* There are individuals with missing hv108 but according to hv106 we 
	know they are in secondary or higher and so have completed 6 years in
	primary level. As such we assume they have completed at least 6 years of 
	schooling.*/	
replace eduyears = . if eduyears>=age & age>0
	/*The variable "eduyears" was replaced with a '.' if total years of 
	education was more than individual's age */
replace eduyears = 0 if age < 10 
replace eduyears = 0 if (age==10 | age==11 | age==12) & eduyears < 6 
	/*The variable "eduyears" was replaced with a '0' for ineligible household 
	members, i.e.: those who have not completed 6 years of schooling following 
	their starting school age */

	
	/*A control variable is created on whether there is information on 
	years of education for at least 2/3 of the eligible household members*/	
gen temp = 1 if eduyears!=. & age>=13 & age!=.
replace temp = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp = 1 if age==11 & eduyears>=6 & eduyears<.
replace temp = 1 if age==12 & eduyears>=6 & eduyears<.
bysort	hh_id: egen no_missing_edu = sum(temp)
	//Total eligible household members with no missing years of education
gen temp2 = 1 if age>=13 & age!=.
replace temp2 = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp2 = 1 if age==11 & eduyears>=6 & eduyears<.
replace temp2 = 1 if age==12 & eduyears>=6 & eduyears<.
bysort hh_id: egen hhs = sum(temp2)
	/*Total number of eligible household members who should have information 
	on years of education */
replace no_missing_edu = no_missing_edu/hhs
replace no_missing_edu = (no_missing_edu>=2/3)
	/*Identify whether there is information on years of education for at 
	least 2/3 of the eligible household members */
tab no_missing_edu, miss
	//The value for 0 (missing) is 0.10% 
label var no_missing_edu "No missing edu for at least 2/3 of the HH members aged 13 years & older"		
drop temp temp2 hhs


*** Standard MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed SIX years of schooling. */
*******************************************************************  
gen	 years_edu6 = (eduyears>=6)
replace years_edu6 = . if eduyears==.
bysort hh_id: egen hh_years_edu6_1 = max(years_edu6)
gen	hh_years_edu6 = (hh_years_edu6_1==1)
replace hh_years_edu6 = . if hh_years_edu6_1==.
replace hh_years_edu6 = . if hh_years_edu6==0 & no_missing_edu==0 
lab var hh_years_edu6 "Household has at least one member with 6 years of edu"
tab hh_years_edu6, miss

	
*** Destitution MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed at least one year of schooling. */
*******************************************************************
gen	years_edu1 = (eduyears>=1)
replace years_edu1 = . if eduyears==.
bysort	hh_id: egen hh_years_edu_u = max(years_edu1)
replace hh_years_edu_u = . if hh_years_edu_u==0 & no_missing_edu==0
lab var hh_years_edu_u "Household has at least one member with 1 year of edu"
tab hh_years_edu_u, miss


*** Quality check ***
/* We compare the proportion of women and men with less than 6 years of 
education obtained from our work with the proportion of women and men with 
incomplete primary or no education as reported in the country survey report.*/
*****************************************************************************

**Women
tab eduyears [aw = weight] if hv104==2 & hv105>5 & hv105<. & hv103==1,m
tab hv109 [aw = weight] if hv104==2 & hv105>5 & hv105<. & hv103==1,m	
	/*Using the hv108 variable, which is of interest to the global MPI, we find
	that 29.93% of de facto women have less than 6 years of education. Using the 
	hv109 variable, we find that 29.95% of women have incomplete primary 
	education or no education. On the other hand, the country survey report 
	indicate that a total 30.3% of women have some primary education or no 
	education (p.56 of 623). We may conclude that the figure based on this 
	dataset closely corresponds to the figure reported in the survey report.*/

**Men	
tab eduyears [aw = weight] if hv104==1 & hv105>5 & hv105<. & hv103==1,m
tab hv109 [aw = weight] if hv104==1 & hv105>5 & hv105<. & hv103==1,m	
	/*Using the hv108 variable, which is of interest to the global MPI, we find
	that 26.69% of men have less than 6 years of education. Using the hv109 
	variable, we find that 26.72% of men have incomplete primary education or no 
	education. On the other hand, the country survey report indicate that a 
	total 27.3% of men have some primary education or no education (p.57 of 623). 
	We may conclude that the figure based on this dataset closely corresponds 
	to the figure reported in the survey report.*/



********************************************************************************
*** Step 2.2 Child School Attendance ***
********************************************************************************

codebook hv121, tab (99)
clonevar attendance = hv121 
recode attendance (2=1) 
label define lab_attend 1 "currently attending" 0 "not currently attending"
label values attendance lab_attend
label var attendance "Attended school during current school year"
codebook attendance, tab (99)
replace attendance = 0 if (attendance==9 | attendance==.) & hv109==0 
replace attendance = . if  attendance==9 & hv109!=0

	

*** Standard MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 8. */ 
******************************************************************* 
gen	child_schoolage = (age>=7 & age<=15)
	/*In Indonesia, the official school entrance age is 7 years. The report 
	 (p. 13) indicate that the sample used to calculate the net and gross 
	  attendance ratios for primary school is children age 6-12, although it is 
	  not stated that 6 is the official school age entrance. The official age 
	  of entry is seven, but many pupils enter at the age of six. The 
	  statistical agency of Indonesia (BPS) has confirmed on their website that 
	  the  entry is seven years, we follow BPS. In addition, we also note that 
	  some 22.88% of children 6 years are not attending school, but this 
	  is very likely because they have not reached the official age to join 
	  school. Therefore, using 6 years as starting age will be misleading. 
	  So age range for Indonesia is 7-15 (=7+8) 
	  Source: "https://www.bps.go.id/subject/28/pendidikan.html#subjekViewTab3"
			  "http://data.uis.unesco.org/?ReportId=163" */
	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the school age children */
count if child_schoolage==1 & attendance==.
	//How many eligible school aged children are not attending school?: 132 
gen temp = 1 if child_schoolage==1 & attendance!=.
	/*Generate a variable that captures the number of eligible school aged 
	children who are attending school */
bysort hh_id: egen no_missing_atten = sum(temp)	
	/*Total school age children with no missing information on school 
	attendance */
gen temp2 = 1 if child_schoolage==1	
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are of school age
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)
	/*Identify whether there is missing information on school attendance for 
	more than 2/3 of the school age children */			
tab no_missing_atten, miss
label var no_missing_atten "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs
		
bysort hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
	//It takes value 1 if the household has children in school age
lab var hh_children_schoolage "Household has children in school age"

gen	child_not_atten = (attendance==0) if child_schoolage==1
replace child_not_atten = . if attendance==. & child_schoolage==1
bysort	hh_id: egen any_child_not_atten = max(child_not_atten)
gen	hh_child_atten = (any_child_not_atten==0) 
replace hh_child_atten = . if any_child_not_atten==.
replace hh_child_atten = 1 if hh_children_schoolage==0
replace hh_child_atten = . if hh_child_atten==1 & no_missing_atten==0 
	/*If the household has been intially identified as non-deprived, but has 
	missing school attendance for at least 2/3 of the school aged children, then 
	we replace this household with a value of '.' because there is insufficient 
	information to conclusively conclude that the household is not deprived */
lab var hh_child_atten "Household has all school age children up to class 8 in school"
tab hh_child_atten, miss

/*Note: The indicator takes value 1 if ALL children in school age are attending 
school and 0 if there is at least one child not attending. Households with no 
children receive a value of 1 as non-deprived. The indicator has a missing value 
only when there are all missing values on children attendance in households that 
have children in school age. */


	
*** Destitution MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 6. */ 
******************************************************************* 
gen	child_schoolage_6 = (age>=7 & age<=13) 
	/*Note: In Indonesia, the official school entrance age is 7 years.  
	  So, age range for destitution measure is 7-13 (=7+6) */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the children attending school up to 
	class 6 */	
count if child_schoolage_6==1 & attendance==.	
gen temp = 1 if child_schoolage_6==1 & attendance!=.
bysort hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_6==1	
bysort hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)			
tab no_missing_atten_u, miss
label var no_missing_atten_u "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs		
		
bysort	hh_id: egen hh_children_schoolage_6 = sum(child_schoolage_6)
replace hh_children_schoolage_6 = (hh_children_schoolage_6>0) 
lab var hh_children_schoolage_6 "Household has children in school age (6 years of school)"

gen	child_atten_6 = (attendance==1) if child_schoolage_6==1
replace child_atten_6 = . if attendance==. & child_schoolage_6==1
bysort	hh_id: egen any_child_atten_6 = max(child_atten_6)
gen	hh_child_atten_u = (any_child_atten_6==1) 
replace hh_child_atten_u = . if any_child_atten_6==.
replace hh_child_atten_u = 1 if hh_children_schoolage_6==0
replace hh_child_atten_u = . if hh_child_atten_u==0 & no_missing_atten_u==0 
lab var hh_child_atten_u "Household has at least one school age children up to class 6 in school"
tab hh_child_atten_u, miss


********************************************************************************
*** Step 2.3 Nutrition ***
********************************************************************************
	// Indonesia DHS 2017 has no information on nutrition. 

gen	hh_nutrition_uw_st = .
lab var hh_nutrition_uw_st "Household has no child underweight/stunted or adult deprived by BMI/BMI-for-age"

gen	hh_nutrition_uw_st_u = .
lab var hh_nutrition_uw_st_u "Household has no child underweight/stunted or adult deprived by BMI/BMI-for-age (destitute)"
	
gen weight_ch = .
label var weight_ch "sample weight child under 5" 

	
********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************
	
codebook v206 v207 mv206 mv207
	/*v206 or mv206: number of sons who have died 
	  v207 or mv207: number of daughters who have died */
	
egen temp_f = rowtotal(v206 v207), missing
	//Total child mortality reported by eligible women
replace temp_f = 0 if v201==0
	//This line replaces women who have never given birth
bysort	hh_id: egen child_mortality_f = sum(temp_f), missing
lab var child_mortality_f "Occurrence of child mortality reported by women"
tab child_mortality_f, miss
drop temp_f
		
egen temp_m = rowtotal(mv206 mv207), missing
	//Total child mortality reported by eligible men
replace temp_m = 0 if mv201==0

	
bysort	hh_id: egen child_mortality_m = sum(temp_m), missing
lab var child_mortality_m "Occurrence of child mortality reported by men"
tab child_mortality_m, miss
drop temp_m

egen child_mortality = rowmax(child_mortality_f child_mortality_m)
lab var child_mortality "Total child mortality within household"
tab child_mortality, miss	
		
*** Standard MPI *** 
/* Members of the household are considered deprived if women in the household 
reported mortality among children under 18 in the last 5 years from the survey 
year. Members of the household is considered non-deprived if eligible women 
within the household reported (i) no child mortality or (ii) if any child died 
longer than 5 years from the survey year or (iii) if any child 18 years and 
older died in the last 5 years. In adddition, members of the household were 
identified as non-deprived if eligible men within the household reported no 
child mortality in the absence of information from women. Households that have 
no eligible women or adult are considered non-deprived. The indicator takes 
a missing value if there was missing information on reported death from 
eligible individuals. */
************************************************************************

tab childu18_died_per_wom_5y, miss
replace childu18_died_per_wom_5y = 0 if v201==0 
	/*Assign a value of "0" for:
	- all eligible women who never ever gave birth */		
replace childu18_died_per_wom_5y = 0 if no_fem_eligible==1 
	/*Assign a value of "0" for:
	- individuals living in households that have non-eligible women */	
	
bysort hh_id: egen childu18_mortality_5y = sum(childu18_died_per_wom_5y), missing
replace childu18_mortality_5y = 0 if childu18_mortality_5y==. & child_mortality==0
	/*Replace all households as 0 death if women has missing value and men 
	reported no death in those households */
label var childu18_mortality_5y "Under 18 child mortality within household past 5 years reported by women"
tab childu18_mortality_5y, miss		
	
gen hh_mortality_u18_5y = (childu18_mortality_5y==0)
replace hh_mortality_u18_5y = . if childu18_mortality_5y==.
lab var hh_mortality_u18_5y "Household had no under 18 child mortality in the last 5 years"
tab hh_mortality_u18_5y, miss 
*br hh_id ind_id v206 v207 mv206 mv207 child_mortality no_fem_eligible no_male_eligible if hh_mortality_u18_5y==.


*** Destitution MPI *** 
*** (same as standard MPI) ***
************************************************************************
gen hh_mortality_u = hh_mortality_u18_5y	
lab var hh_mortality_u "Household had no under 18 child mortality in the last 5 years"	


********************************************************************************
*** Step 2.5 Electricity ***
********************************************************************************

*** Standard MPI ***
/*Members of the household are considered 
deprived if the household has no electricity */
***************************************************
clonevar electricity = hv206 
codebook electricity, tab (9)
label var electricity "Household has electricity"
tab electricity, miss


*** Destitution MPI  ***
*** (same as standard MPI) ***
***************************************************
gen electricity_u = electricity
label var electricity_u "Household has electricity"


*** Quality check ***
/* We compare the proportion of households with 
electricity obtained from our work and as reported in 
the country survey report. */
*********************************************************
bysort hh_id: gen id = _n 
tab electricity [aw = weight] if id==1,m
	/*In the report, Table 2.4 (p.17), 97.1% of households have access to 
	electricity. The results obtained from our work is 97.12% which matches the 
	report. */

	
********************************************************************************
*** Step 2.6 Sanitation ***
********************************************************************************

/*
Improved sanitation facilities include flush or pour flush toilets to sewer 
systems, septic tanks or pit latrines, ventilated improved pit latrines, pit 
latrines with a slab, and composting toilets. These facilities are only 
considered improved if it is private, that is, it is not shared with other 
households.
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-02-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

desc hv205 hv225 hv238
clonevar toilet = hv205  

codebook hv205
codebook hv238, tab (999)
gen shared_toilet = 1 if hv238>=2& hv238<=95
tab shared_toilet, miss
	/*Indonesia DHS 2017 lack information on shared toilet (hv225) but it has 
	information whether facility is shared through the hv205 variable.
	We also cosidered the hv238 (number of households sharing toilet) variable 
	but this made no difference since individuals who reported using flush 
	toilet that are shared/public, are the ones who answered hv238. */

*** Standard MPI ***
/*Members of the household are considered deprived if the household's 
sanitation facility is not improved (according to the SDG guideline) 
or it is improved but shared with other households*/
********************************************************************
codebook toilet, tab(99)
	/* Note: In Indonesia,the report considers improved toilet facilities as 
	that are used by only one household with spetic tank or non-septic tank 
	(p.8)*/

tab toilet shared_toilet, miss
 
gen toilet_mdg = (toilet==12 | toilet==16)
replace toilet_mdg = . if (toilet==.  | toilet==99) & shared_toilet==.
	//Household is assigned a value of '.' if it has missing information 		
lab var toilet_mdg "Household has improved sanitation"
tab toilet toilet_mdg, miss
tab toilet_mdg if toilet==. & shared_toilet==1,miss
tab toilet_mdg, miss


*** Destitution MPI ***
/*Members of the household are considered deprived if household practises 
open defecation or uses other unidentifiable sanitation practises */
********************************************************************

gen	toilet_u = 1
replace toilet_u = 0 if toilet==31 | toilet==96 
	/*Household is assigned a value of '0' if it practises open defecation or 
	others */
replace toilet_u = . if toilet==.  
	//Household is assigned a value of '.' if it has missing information 		
lab var toilet_u "Household does not practise open defecation or others"
tab toilet toilet_u, miss


*** Quality check ***
/* We compare the proportion of households with 
improved sanitation obtained from our work and as 
reported in the country survey report. */
*********************************************************
tab toilet_mdg [aw = weight] if id==1,m
	/*In the report, Table 2.3 (p.16), 79.8% of households have improved 
	sanitation facilities. The results obtained from our work is 79.74% which 
	matches closely with the report. */


********************************************************************************
*** Step 2.7 Drinking Water  ***
********************************************************************************
/*
Improved drinking water sources include the following: piped water into 
dwelling, yard or plot; public taps or standpipes; boreholes or tubewells; 
protected dug wells; protected springs; packaged water; delivered water and 
rainwater which is located on premises or is less than a 30-minute walk from 
home roundtrip. 
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-01-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

desc hv201 hv204 hv202
clonevar water = hv201  
clonevar timetowater = hv204  
clonevar ndwater = hv202  	

*** Standard MPI ***
/* Members of the household are considered deprived if the household 
does not have access to improved drinking water (according to the SDG 
guideline) or safe drinking water is at least a 30-minute walk from 
home, roundtrip */
********************************************************************
codebook water, tab(99)

gen	water_mdg = 1 if water<=31 | water==41 | water==51 | water==71 | water==72  
replace water_mdg = 0 if water==32 | water==42 | water==43 | ///
						 water==61 | water==62 | water==96 

replace water_mdg = . if water==. | water==99
replace water_mdg = 0 if water==71 & ///
						(ndwater==32 | ndwater==42 | ndwater==43 | /// 
						 ndwater==61 | ndwater==62 | ndwater==96)

replace water_mdg = 0 if water==72 & ///
						(ndwater==32 | ndwater==42 | ndwater==43 | /// 
						 ndwater==61 | ndwater==62 | ndwater==96)

lab var water_mdg  "HH has safe drinking water, under 30 mins dist"	  
tab water water_mdg, miss
tab water_mdg, miss


*** Quality check ***
/* We compare the proportion of households with 
improved access to safe drinking water obtained from 
our work and as reported in the country survey report. */
*********************************************************
tab water_mdg [aw = weight] if id==1,m
	/*In the report, Table 2.1 (p.15), 86.7% of households have 
	improved or safe drinking facilities. The results obtained 
	from our work is 86.75% which matches the report. */	  

	
*** Time to water ***	
*********************************************************  
codebook timetowater, tab(9999)	
replace water_mdg = 0 if water_mdg==1 & timetowater >= 30 & timetowater!=. & ///
						 timetowater!=996 & timetowater!=998 & timetowater!=999 
tab water_mdg, miss


*** Destitution MPI ***
/* Members of the household is identified as destitute if household 
does not have access to safe drinking water, or safe water is more 
than 45 minute walk from home, round trip.*/
********************************************************************
gen	water_u = .
replace water_u = 1 if water<=31 | water==41 | water==51 | ///
					   water==71 | water==72
						   
replace water_u = 0 if water==32 | water==42 | water==43 | water==61 | ///
					   water==62 | water==96
						   
replace water_u = 0 if water_u==1 & timetowater>45 & timetowater!=. & ///
					   timetowater!=995 & timetowater!=996 & ///
					   timetowater!=998 & timetowater!=999
						   
replace water_u = . if water==. | water==99

replace water_u = 0 if water==71 & ///
					   (ndwater==32 | ndwater==42 | ndwater==43 | /// 
					    ndwater==61 | ndwater==62 | ndwater==96) 

replace water_u = 0 if water==72 & ///
						(ndwater==32 | ndwater==42 | ndwater==43 | /// 
						 ndwater==61 | ndwater==62 | ndwater==96)
						  
lab var water_u "DST: HH has safe drinking water, 0-45 mins dist"
tab water water_u, miss
tab water_u, miss

*** Harmonised MPI *** 
	/*The use of bottled water for non-drinking activities are not taken
	into account in the later surveys. The harmonised work excludes the 
	use of the 'ndwater' variable.*/
************************************************************************
gen	water_mdg_c = 1 if water<=31 | water==41 | water==51 | ///
					   water==71 | water==72  
					   
replace water_mdg_c = 0 if water==32 | water==42 | water==43 | ///
						   water==61 | water==62 | water==96 
replace water_mdg_c = . if water==. | water==99
	  
replace water_mdg_c = 0 if water_mdg_c==1 & timetowater >= 30 & timetowater!=. & ///
						 timetowater!=996 & timetowater!=998 & timetowater!=999 

lab var water_mdg_c "COT: HH has safe drinking water"
tab water_mdg_c, miss


gen	water_u_c = .
replace water_u_c = 1 if water<=31 | water==41 | water==51 | ///
					     water==71 | water==72
						   
replace water_u_c = 0 if water==32 | water==42 | water==43 | water==61 | ///
					     water==62 | water==96
						   
replace water_u_c = 0 if water_u_c==1 & timetowater>45 & timetowater!=. & ///
					     timetowater!=995 & timetowater!=996 & ///
					     timetowater!=998 & timetowater!=999
						   
replace water_u_c = . if water==. | water==99					  
lab var water_u_c "COT-DST: HH has safe drinking water"
tab water water_u_c, miss
tab water_u_c, miss


********************************************************************************
*** Step 2.8 Housing ***
********************************************************************************

/* Members of the household are considered deprived if the household 
has a dirt, sand or dung floor */
lookfor floor
clonevar floor = hv213 
codebook floor, tab(99)
gen	floor_imp = 1
replace floor_imp = 0 if floor<=12 | floor==96   	
replace floor_imp = . if floor==. | floor==99 	
lab var floor_imp "Household has floor that it is not earth/sand/dung"
tab floor floor_imp, miss		


/* Members of the household are considered deprived if the household has walls 
made of natural or rudimentary materials. We followed the report's definitions
of natural or rudimentary materials.
Note: Indonesia DHS 2017: codes checked in the report questionnaire (p. 446) */
lookfor wall
clonevar wall = hv214 
codebook wall, tab(99)	
gen	wall_imp = 1 
replace wall_imp = 0 if wall<=26 | wall==96  	
replace wall_imp = . if wall==. | wall==99 
lab var wall_imp "Household has wall that it is not of low quality materials"
tab wall wall_imp, miss	
	

/* Members of the household are considered deprived if the household has roof 
made of natural or rudimentary materials. We followed the report's definitions
of natural and rudimentary materials. 
Note: Indonesia SH 2017: codes checked in the report questionnaire (p. 445) */
lookfor roof
clonevar roof = hv215
codebook roof, tab(99)		
gen	roof_imp = 1  
replace roof_imp = 0 if roof<=23 | roof==96 	
replace roof_imp = . if roof==. | roof==99 	
lab var roof_imp "Household has roof that it is not of low quality materials"
tab roof roof_imp, miss


*** Standard MPI ***
/* Members of the household is deprived in housing if the roof, 
floor OR walls are constructed from low quality materials.*/
**************************************************************
gen housing_1 = 1
replace housing_1 = 0 if floor_imp==0 | wall_imp==0 | roof_imp==0
replace housing_1 = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_1 "Household has roof, floor & walls that it is not low quality material"
tab housing_1, miss


*** Destitution MPI ***
/* Members of the household is deprived in housing if two out 
of three components (roof and walls; OR floor and walls; OR 
roof and floor) the are constructed from low quality materials. */
**************************************************************
gen housing_u = 1
replace housing_u = 0 if (floor_imp==0 & wall_imp==0 & roof_imp==1) | ///
						 (floor_imp==0 & wall_imp==1 & roof_imp==0) | ///
						 (floor_imp==1 & wall_imp==0 & roof_imp==0) | ///
						 (floor_imp==0 & wall_imp==0 & roof_imp==0)
replace housing_u = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_u "Household has one of three aspects(either roof,floor/walls) that is not low quality material"
tab housing_u, miss


*** Quality check ***
/* We compare the proportion of households with 
improved floor from  our work and as reported in the 
country survey report. */
*********************************************************
tab floor_imp [aw = weight] if id==1,m
	/*In the report, Table 2.4 (p.17) indicate that 95.1% of households have 
	improved floor (built using rudimentary or finished material). The results 
	obtained from our work is 94.89% which closely matches the report. */	


********************************************************************************
*** Step 2.9 Cooking Fuel ***
********************************************************************************
/*
Solid fuel are solid materials burned as fuels, which includes coal as well as 
solid biomass fuels (wood, animal dung, crop wastes and charcoal). 

Source: 
https://apps.who.int/iris/bitstream/handle/10665/141496/9789241548885_eng.pdf
*/

lookfor cooking combustible
clonevar cookingfuel = hv226


*** Standard MPI ***
/* Members of the household are considered deprived if the 
household uses solid fuels and solid biomass fuels for cooking. */
*****************************************************************
codebook cookingfuel, tab(99)
gen	cooking_mdg = 1
replace cooking_mdg = 0 if cookingfuel>5 & cookingfuel<95 
replace cooking_mdg = . if cookingfuel==. | cookingfuel==99
lab var cooking_mdg "Household uses clean fuels for cooking"
tab cookingfuel cooking_mdg, miss	
tab cooking_mdg, miss
	
*** Destitution MPI ***
*** (same as standard MPI) ***
****************************************
gen	cooking_u = cooking_mdg
lab var cooking_u "Household uses clean fuels for cooking"


*** Quality check ***
/* We compare the proportion of households using 
solid fuel for cooking obtained from our work and as 
reported in the country survey report. */
*********************************************************
tab cooking_mdg [aw = weight] if id==1,m
	/*In the report, Table 2.4 (p.17) indicate that 23.1% of households use 
	solid fuel for cooking. The results obtained from our work is 23.13% which 
	matches the report. */	


********************************************************************************
*** Step 2.10 Assets ownership ***
********************************************************************************

*** Television/LCD TV/plasma TV/color TV/black & white tv
lookfor tv television plasma lcd	
codebook hv208
	//1=yes; 0=no
clonevar television = hv208 
lab var television "Household has television"	
	
***	Radio/walkman/stereo/kindle
lookfor radio walkman stereo
codebook hv207
	//1=yes; 0=no
clonevar radio = hv207 
lab var radio "Household has radio"	

	
***	Handphone/telephone/iphone/mobilephone/ipod
lookfor telephone téléphone mobilephone ipod
codebook hv221 hv243a
	//1=yes; 0=no
clonevar telephone = hv221
replace telephone=1 if telephone!=1 & hv243a==1	
	//hv243a=mobilephone. Combine information on telephone and mobilephone.	
tab hv243a hv221 if telephone==1,miss
lab var telephone "Household has telephone (landline/mobilephone)"	

	
***	Refrigerator/icebox/fridge
lookfor refrigerator réfrigérateur
codebook hv209
	//1=yes; 0=no
clonevar refrigerator = hv209 
lab var refrigerator "Household has refrigerator"


***	Car/van/lorry/truck
lookfor car voiture truck van
codebook hv212
	//1=yes; 0=no
clonevar car = hv212  
lab var car "Household has car"		

		
***	Bicycle/cycle 
lookfor bicycle bicyclette
codebook hv210
	//1=yes; 0=no	
clonevar bicycle = hv210 
lab var bicycle "Household has bicycle"	


***	Motorbike/motorized bike
lookfor motorbike moto
codebook hv211
	//1=yes; 0=no	
clonevar motorbike = hv211 
lab var motorbike "Household has motorbike"	

	
***	Computer/laptop/tablet
lookfor computer ordinateur laptop ipad tablet
codebook hv243e
	//1=yes; 0=no
clonevar computer = hv243e
lab var computer "Household has computer"

***	Animal cart
lookfor cart 
codebook hv243c
	//1=yes; 0=no
clonevar animal_cart = hv243c
lab var animal_cart "Household has animal cart"	



foreach var in television radio telephone refrigerator car ///
			   bicycle motorbike computer animal_cart {
replace `var' = . if `var'==9 | `var'==99 | `var'==8 | `var'==98 
}
	//Missing values replaced


*** Quality check ***
/* We compare the proportion of households owning  
each assets obtained from our work and as reported 
in the country survey report. */
*********************************************************
	
tab radio [aw = weight] if id==1,m
	/*Table 2.5 (p.18) indicate that 22.9% of households own radio. The results 
	obtained from our work is 22.94% which closely matches the report.*/	

tab television [aw = weight] if id==1,m
	/*Table 2.5 (p.18) indicate that 88.7% of households own television. The 
	results obtained from our work is 88.72% which closely matches the report.*/	
	
tab hv243a [aw = weight] if id==1,m
	/*Table 2.5 (p.18) indicate that 89.0% of households own mobile phone. The 
	results obtained from our work is 88.99% which matches the report.*/

tab computer [aw = weight] if id==1,m
	/*Table 2.5 (p.18) indicate that 22.0% of households own computer. The 
	results obtained from our work is 22.04% which closely matches the report.*/
	
tab refrigerator [aw = weight] if id==1,m
	/*Table 2.5 (p.18) indicate that 56.9% of households own refrigerator. The 
	results obtained from our work is 56.90% which matches the report.*/

tab bicycle [aw = weight] if id==1,m
	/*Table 2.5 (p.18) indicate that 39.3% of households own bicycle. The 
	results obtained from our work is 39.27% which closely matches the report.*/

tab animal_cart [aw = weight] if id==1,m
	/*Table 2.5 (p.18) indicate that 0.2% of households own animal cart. The 
	results obtained from our work is 0.18% which closely matches the report.*/

tab motorbike [aw = weight]if id==1,m
	/*Table 2.5 (p.18) indicate that 78.7% of households own bicycle. The 
	results obtained from our work is 78.74% which closely matches the report.*/

tab car [aw = weight] if id==1,m
	/*Table 2.5 (p.18) indicate that 12.6% of households own car. The results 
	obtained from our work is 12.59% which closely matches the report.*/

	
*** Standard MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own more than one of: radio, TV, telephone, bike, motorbike, 
refrigerator, computer or animal cart and does not own a car or truck.*/
*****************************************************************************
egen n_small_assets2 = rowtotal(television radio telephone refrigerator bicycle motorbike computer animal_cart), missing
lab var n_small_assets2 "Household Number of Small Assets Owned" 
   
gen hh_assets2 = (car==1 | n_small_assets2 > 1) 
replace hh_assets2 = . if car==. & n_small_assets2==.
lab var hh_assets2 "Household Asset Ownership: HH has car or more than 1 small assets incl computer & animal cart"
tab hh_assets2, miss


*** Destitution MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own any assets.*/
*****************************************************************************	
gen	hh_assets2_u = (car==1 | n_small_assets2>0)
replace hh_assets2_u = . if car==. & n_small_assets2==.
lab var hh_assets2_u "Household Asset Ownership: HH has car or at least 1 small assets incl computer & animal cart"
tab hh_assets2_u, miss


*** Harmonised MPI *** 
	/*The earlier survey lacked information on computer. As 
	such, the harmonised work excludes computer.*/
************************************************************************
gen computer_c = .
lab var computer_c "COT: Household has computer"

egen n_small_assets2_c = rowtotal(television radio telephone refrigerator bicycle motorbike computer_c animal_cart), missing
lab var n_small_assets2_c "COT: HH Number of Small Assets Owned" 
    
gen hh_assets2_c = (car==1 | n_small_assets2_c > 1) 
replace hh_assets2_c = . if car==. & n_small_assets2_c==.
lab var hh_assets2_c "COT: HH has car or more than 1 small assets"

gen	hh_assets2_u_c = (car==1 | n_small_assets2_c>0)
replace hh_assets2_u_c = . if car==. & n_small_assets2_c==.
lab var hh_assets2_u_c "COT-DST: HH has car or at least 1 small assets"


		
********************************************************************************
*** Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************

	//Retain data on sampling design: 
desc hv022 hv021	
clonevar strata = hv022
clonevar psu = hv021
label var psu "Primary sampling unit"
label var strata "Sample strata"

	
	//Retain year, month & date of interview:
desc hv007 hv006 hv008
clonevar year_interview = hv007 	
clonevar month_interview = hv006 
clonevar date_interview = hv008
 
 		
*** Keep main variables require for MPI calculation ***
keep hh_id ind_id subsample strata psu weight weight_ch sex age hhsize ///
area agec7 agec4 agec2 headship region marital relationship ///
region region_01 year_interview month_interview date_interview /// 
fem_nutri_eligible no_fem_nutri_eligible no_fem_eligible ///
male_nutri_eligible no_male_nutri_eligible no_male_eligible ///
child_eligible no_child_eligible no_adults_eligible ///
no_child_fem_eligible no_eligibles ///
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet shared_toilet toilet_mdg toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
water_mdg_c water_u_c hh_assets2_c hh_assets2_u_c
	
	 
*** Order file	***
order hh_id ind_id subsample strata psu weight weight_ch sex age hhsize ///
area agec7 agec4 agec2 headship region marital relationship ///
region region_01 year_interview month_interview date_interview /// 
fem_nutri_eligible no_fem_nutri_eligible no_fem_eligible ///
male_nutri_eligible no_male_nutri_eligible no_male_eligible ///
child_eligible no_child_eligible no_adults_eligible ///
no_child_fem_eligible no_eligibles ///
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet shared_toilet toilet_mdg toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
water_mdg_c water_u_c hh_assets2_c hh_assets2_u_c


*** Rename key global MPI indicators for estimation ***
recode hh_mortality_u18_5y  (0=1)(1=0) , gen(d_cm)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ)
recode electricity 			(0=1)(1=0) , gen(d_elct)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani)
recode housing_1 			(0=1)(1=0) , gen(d_hsg)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl)
recode hh_assets2 			(0=1)(1=0) , gen(d_asst)
 

*** Rename key global MPI indicators for destitution estimation ***
recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ)
recode electricity_u		(0=1)(1=0) , gen(dst_elct)
recode water_u 				(0=1)(1=0) , gen(dst_wtr)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl)
recode hh_assets2_u 		(0=1)(1=0) , gen(dst_asst) 


*** Rename indicators for changes over time estimation ***	
recode hh_mortality_u18_5y  (0=1)(1=0) , gen(d_cm_01)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr_01)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt_01)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ_01)
recode electricity 			(0=1)(1=0) , gen(d_elct_01)
recode water_mdg_c 			(0=1)(1=0) , gen(d_wtr_01)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani_01)
recode housing_1 			(0=1)(1=0) , gen(d_hsg_01)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl_01)
recode hh_assets2_c 		(0=1)(1=0) , gen(d_asst_01)	
	

recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm_01)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr_01)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt_01)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ_01)
recode electricity_u		(0=1)(1=0) , gen(dst_elct_01)
recode water_u_c	 		(0=1)(1=0) , gen(dst_wtr_01)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani_01)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg_01)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl_01)
recode hh_assets2_u_c 		(0=1)(1=0) , gen(dst_asst_01) 


 
*** Eligibility for years of schooling indicator ***
gen educ_elig = 1 
replace educ_elig = 0 if age < 10 
replace educ_elig = 0 if (age==10 | age==11 | age==12) & eduyears < 6 
label define lab_educ_elig 0"ineligible" 1"eligible"  
label values educ_elig lab_educ_elig
lab var educ_elig "Individual is eligible for educ indicator"
ta eduyears educ_elig,m
 
 
*** Generate coutry and survey details for estimation ***
char _dta[cty] "Indonesia"
char _dta[ccty] "IDN"
char _dta[year] "2017" 	
char _dta[survey] "DHS"
char _dta[ccnum] "360"
char _dta[type] "micro"


	
*** Sort, compress and save data for estimation ***
sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/idn_dhs17.dta", replace 
