********************************************************************************
/*
Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
Global Multidimensional Poverty Index - Ecuador ENSANUT 2018 
[STATA do-file]. Available from OPHI website: http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk
*/
********************************************************************************

clear all 
set more off
set maxvar 10000


global path_in "../rdta/Ecuador ENSANUT 2018"  		// folder path
global path_out "cdta"
global path_ado "ado"



/* 
Ecuador ENSANUT 2018: The survey collected information of women from 10 to 49 
years old, children under 5 years old, men 12 years and over, girls and boys 5
to 11 years old, from 43311 households.
 
There are 3 database that we use to compute the Global MPI: 

1_BDD_ENS2018_f1_personas: database with information regarding the 
sociodemographic characteristics of each member of the household, 
as well as information on education, work, income, health, weight and height

2_BDD_ENS2018_f1_hogar: database with information regarding the 
characteristics of the dwellings.

4_BDD_ENS2018_f2_mef: database with information on women from 10 to 49 years 
of age regarding pregnancy history, maternal health, sexual activity, family 
planning, marriage and sexually transmitted diseases.

Anthropometric information (weight and height) was collected for all households
members, three times (p. 20)
https://www.ecuadorencifras.gob.ec/documentos/web-inec/Estadisticas_Sociales/ENSANUT/ENSANUT_2018/Evolucion%20Historica%20de%20ENSANUT%202018.pdf
*/ 

********************************************************************************
**# Step 1.1 NUTRITION DATA
********************************************************************************

use "$path_in/1_BDD_ENS2018_f1_personas.dta", clear


duplicates report id_hogar id_per
gen hh_id = id_hogar
label var hh_id "household id"

gen ind_id = id_per
label var ind_id "individual id"


duplicates report hh_id ind_id 


*** Variable: SEX ***
codebook sexo
	//"1" for male ;"2" for female 
clonevar gender = sexo 
label define label_sex 1 "male" 2 "female"
label values gender label_sex
label var gender "sex"
codebook gender


*** Variable: AGE *** 
gen bdate = mdy(f1_s2_4_2, f1_s2_4_1, f1_s2_4_3) if f1_s2_4_1 <= 31
gen mdate = mdy(f1_s7_3_2, f1_s7_3_1, f1_s7_3_3)


gen age_months = (mdate-bdate)/30.4375
lab var age_months "Age in months"
sum age_months


gen age_days = (mdate-bdate)	
lab var age_days "Age in days"
sum age_days


compare edaddias age_days
  /* 'edaddias' - age in days computed by survey provider. 
     'age_days' - age in days computed using date, month and year of birth 
     and measurement. Number of mismatch: 23,226. We use the 'age_days'
	 variable that we constructed. */ 

  
gen str6 ageunit = "days"
lab var ageunit "Days"	
	

*** Variable: BODY WEIGHT (KILOGRAMS) ***
lookfor peso 
/* Weight was measured 3 times in Ecuador ENSANUT 2018:
	f1_s7_4_1
	f1_s7_4_2
	f1_s7_4_3
*/
count if f1_s7_4_1!=. & f1_s7_4_2!=. & f1_s7_4_3==.
  // Weight measured twice: 153,065 observations

count if f1_s7_4_1!=. & f1_s7_4_2!=. & f1_s7_4_3!=.
count if f1_s7_4_3!=.
 // Weight measured thrice: 1,095 observations
 
count if f1_s7_4_1==. & f1_s7_4_2==. & f1_s7_4_3==.
  // Missing: 14,587 observations
  
gen weight = (f1_s7_4_1 + f1_s7_4_2)/2  if ///
			  f1_s7_4_1!=. & f1_s7_4_2!=. & f1_s7_4_3==.
replace weight = (f1_s7_4_1 + f1_s7_4_2 + f1_s7_4_3)/3 if f1_s7_4_3!=.
codebook f1_s7_1 
  // Interviewer, record whether you measured the weight and height - yes; no
sum weight


*** Variable: HEIGHT (CENTIMETERS)
lookfor talla longitud 
/* Height was measured 3 times in Ecuador ENSANUT 2018:
	Lenght:
	f1_s7_5_1
	f1_s7_5_2
	f1_s7_5_3
	
	Height:
	f1_s7_6_1
	f1_s7_6_2
	f1_s7_6_3
*/
 
* Length 
count if f1_s7_5_1!=. & f1_s7_5_2!=. & f1_s7_5_3==.
	//Length measured twice: 7,837 observations
count if f1_s7_5_1!=. & f1_s7_5_2!=. & f1_s7_5_3!=.
count if f1_s7_5_3!=.
	//Length measured thrice: 67 observations

* Height 
count if f1_s7_6_1!=. & f1_s7_6_2!=. & f1_s7_6_3==.
	//Height measured twice: 145,166 observations
count if f1_s7_6_1!=. & f1_s7_6_2!=. & f1_s7_6_3!=.
	//Height measured thrice: 1,090 observations

  
  
gen height = (f1_s7_5_1 + f1_s7_5_2)/2  if ///
			  f1_s7_5_1!=. & f1_s7_5_2!=. & f1_s7_5_3==.
			  
replace height = (f1_s7_5_1 + f1_s7_5_2 + f1_s7_5_3)/3 if f1_s7_5_3!=.

replace height = (f1_s7_6_1 + f1_s7_6_2)/2  if ///
			     f1_s7_6_1!=. & f1_s7_6_2!=. & f1_s7_6_3==.
								
replace height = (f1_s7_6_1 + f1_s7_6_2 + f1_s7_6_3)/3 if ///
				  f1_s7_6_1!=. & f1_s7_6_2!=. & f1_s7_6_3!=.
				
codebook f1_s7_1 
  // Interviewer, record whether you measured the weight and height - yes; no
sum height

	
*** Variable: MEASURED STANDING/LYING DOWN ***	
	// h: height - l: length 
gen measure = "h" if f1_s7_6_1!=.
replace measure = "l" if f1_s7_5_1!=.
replace measure = " " if f1_s7_6_1==. & f1_s7_5_1==. 
tab measure,m

	
*** Variable: OEDEMA ***
gen  oedema = "n"  
	

*** Variable: INDIVIDUAL SAMPLING WEIGHT ***	
gen sw = 1


	/* Save file in order to use the .dta file to compute nutrition indicators 
	individually for children under 5, young individuls, and adults. */
save "$path_out/ECU18_NU.dta", replace


********************************************************************************
**# Step 1.1a Nutrition indicators for children 0-60 months
********************************************************************************

clear
use "$path_out/ECU18_NU.dta"
sort ind_id


gen child_id=1 if age_months<61
keep if child_id==1 
count
	
*** z-scores ***
adopath + "$path_ado/igrowup_stata"
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "children_nutri_ecu" 
lab var datalab "Working file"


igrowup_restricted reflib datalib datalab gender age_days ageunit weight ///
height measure oedema sw


use "$path_out/children_nutri_ecu_z_rc.dta", clear 


*** Standard MPI indicator ***	
gen	underweight = (_zwei < -2.0) 
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "Child is underweight"
tab underweight, miss

gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "Child is stunted"
tab stunting, miss

gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "Child is wasted"
tab wasting, miss


*** Destitution indicator  ***
gen	underweight_u = (_zwei < -3.0) 
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "DST: Child is underweight"

gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "DST: Child is stunted"

gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "DST: Child is wasted"
 
 
count if _fwei==1 | _flen==1
	/*In Ecuador ENSANUT 2018, 272 children were replaced as missing 
	because they have extreme z-scores which are biologically implausible. */
 
 
gen weight_ch = fexp
label var weight_ch "sample weight child under 5"  
 
 
	//Retain relevant variables:
keep  hh_id ind_id age_months age_days child_id ///
underweight* stunting* wasting* weight_ch 
order hh_id ind_id age_months age_days child_id ///
underweight* stunting* wasting* weight_ch 
sort  hh_id ind_id 
save "$path_out/ECU18_KR.dta", replace	


	//Erase files from folder:
erase "$path_out/children_nutri_ecu_z_rc.xls"
erase "$path_out/children_nutri_ecu_prev_rc.xls"
erase "$path_out/children_nutri_ecu_z_rc.dta"

	
********************************************************************************
**# Step 1.1b BMI-for-age for individuals 61-228 months
********************************************************************************

clear 
use "$path_out/ECU18_NU.dta", clear
	
 
gen youth_id = (age_months>=61 & age_months<229)		
keep if youth_id==1 
count
	//44,692 individuals 61-228 months (5-19 years)


	
*** z-scores ***
adopath + "$path_ado/who2007_stata"		
gen str100 reflib = "$path_ado/who2007_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "youth_nutri_ecu" 
lab var datalab "Working file"


who2007 reflib datalib datalab gender age_days ageunit weight height oedema sw


use "$path_out/youth_nutri_ecu_z.dta", clear 
	
gen	z_bmi = _zbfa
replace z_bmi = . if _fbfa==1 
lab var z_bmi "z-score bmi-for-age WHO"


*** Standard MPI indicator ***
gen	low_bmiage = (z_bmi < -2.0) 
replace low_bmiage = . if z_bmi==.
lab var low_bmiage "Adolescent have low bmi"


*** Destitution indicator ***
gen	low_bmiage_u = (z_bmi < -3.0)
replace low_bmiage_u = . if z_bmi==.
lab var low_bmiage_u "DST: Adolescent have low bmi"


tab low_bmiage, miss
tab low_bmiage_u, miss


	//Retain relevant variables:	
keep  hh_id ind_id youth_id age_months age_days low_bmiage* 
order hh_id ind_id youth_id age_months age_days low_bmiage* 
 
 
	/*Append the nutrition information of children above 5 years with 
	children under 5 */	
append using "$path_out/ECU18_KR.dta"
	
	
	//Save a temp file for merging later: 
save "$path_out/ECU18_childyouth.dta", replace
count
	// 65,616 individuals (20,924 children + 44,692 youth)

	
	//Erase files from folder:
erase "$path_out/youth_nutri_ecu_z.xls"
erase "$path_out/youth_nutri_ecu_prev.xls"
erase "$path_out/youth_nutri_ecu_z.dta"	
erase "$path_out/ECU18_KR.dta"


********************************************************************************
**# Step 1.2  HOUSEHOLD RECODE
********************************************************************************

clear 
use "$path_in/2_BDD_ENS2018_f1_hogar.dta", clear

duplicates report id_hogar 
gen hh_id = id_hogar
label var hh_id "Household ID"
duplicates report hh_id 
	// 43,311 households and no duplicates

save "$path_out/ECU18_HR.dta", replace


********************************************************************************
**# Step 1.3 Information on child mortality
********************************************************************************

/* In Ecuador ENSANUT 2018 the information about mortality is in the 
   data file '4_BDD_ENS2018_f2_mef': women 10-49 years old 
   (pp. 28 of the methodological report) */

use "$path_in/4_BDD_ENS2018_f2_mef.dta", clear

duplicates report id_per 
sum edadanios
	//48,700 women 10-49 years; no duplicates
	
gen ind_id = id_per
label var ind_id "Individual ID"
	
	
gen hh_id = id_hogar
label var hh_id "Household ID"


desc f2_s2_200 f2_s2_206 f2_s2_207 
desc f2_s2_210 f2_s2_211_1 f2_s2_211_2 f2_s2_211_3
desc f2_s2_218*
	//Data in wide format

keep ind_id hh_id f2_s2_206 f2_s2_207 f2_s2_210 f2_s2_211_* ///
f2_s2_217_3 f2_s2_218* fecha_mes fecha_dia fecha_anio	
	

 forvalues i=1(1)10 {

ren f2_s2_218_`i'_1 child_`i'
ren f2_s2_218_`i'_a childsex_`i' 
ren f2_s2_218_`i'_b1 borndate_`i'
ren f2_s2_218_`i'_b2 bornmth_`i'
ren f2_s2_218_`i'_b3 bornyr_`i'
ren f2_s2_218_`i'_c childalive_`i'
ren f2_s2_218_`i'_d1 ageyr_`i'
ren f2_s2_218_`i'_d2 agemth_`i'
ren f2_s2_218_`i'_e1 diedmth_`i'
ren f2_s2_218_`i'_e2 diedyr_`i'
ren f2_s2_218_`i'_f1 agedied_days_`i'
ren f2_s2_218_`i'_f2 agedied_mths_`i'
ren f2_s2_218_`i'_f3 agedied_yrs_`i'
}


reshape long child_ childsex_ borndate_ bornmth_ bornyr_ childalive_ ///
ageyr_ agemth_ diedmth_ diedyr_ agedied_days_ agedied_mths_ agedied_yrs_, ///
i(ind_id) j(child) 



codebook childalive_, tab (9)
	//Is child alive? (1)yes (2)no; 1,812 child mortality reported
count if diedmth_!=.
	//Month of death
count if diedyr_!=.
	//Year of death
count if agedied_yrs_>=0 & agedied_yrs_<88	
count if agedied_yrs_>=0 & agedied_yrs_<18
	/*How old was child when he or she died? 
	1,498 of the 1,812 reported mortality reported age in years (0-32 yrs)
	Of this, 1,402 deaths are among those under 18 years. */


//Age of death in years	
*codebook agedied_days_ agedied_mths_ agedied_yrs_, tab (999)	
gen age_death = agedied_yrs_ if agedied_yrs_<88
	//1,498 death reported in years
tab agedied_days_ agedied_mths_ if agedied_yrs_>=88,m
	//Convert deaths reported in days or months to years
replace age_death = 0 if agedied_days_<=31 & agedied_yrs_>=88
replace age_death = 0 if agedied_mths_<12 & agedied_yrs_>=88
replace age_death = 1 if agedied_mths_==12 & agedied_yrs_>=88
sum age_death
label var age_death "Age at death in years"

gen age_death_days = age_death*365
label var age_death_days "Age at death in days"
sum age_death_days

	
//Birth date
*codebook bornmth_ borndate_ bornyr_, tab (999)
replace bornmth_=.  if bornmth_>=88
replace borndate_=. if borndate_>=88
replace bornyr_=.   if bornyr_>=8888

gen date_birth = mdy(bornmth_, borndate_, bornyr_)
replace date_birth = mdy(bornmth_, 15, bornyr_) if ///
					 date_birth==. & bornmth_!=. & bornyr_!=.
sum date_birth


//Death date
*codebook diedmth_ diedyr_, tab (999)

replace diedmth_ = 6 if diedmth_>=88
replace diedyr_ = . if diedyr_>=8888

gen date_death = mdy(diedmth_, 15, diedyr_)  
sum date_death  


//Interview date	
*codebook fecha_mes fecha_dia fecha_anio, tab (999)   
gen date_intvw = mdy(fecha_mes, fecha_dia, fecha_anio)
sum date_intvw	


gen ddead_survey = date_intvw - date_death
	//Days dead from survey = date of interview - date of death
gen ydead_survey = ddead_survey/365
	//Years dead from survey
sum ydead_survey


gen date_death2 = date_birth + age_death_days
	//Date of death = date of birth + age at death (days)
gen ddead_survey2 = date_intvw - date_death2
	//Months dead from survey = date of interview - date of death
gen ydead_survey2 = ddead_survey2/365

count if ydead_survey==. & ydead_survey2!=.
replace ydead_survey = ydead_survey2 if ydead_survey==. & ydead_survey2!=.


codebook childalive_, tab (9)	
gen child_died = 1 if childalive_==2
replace child_died = 0 if childalive_==1
replace child_died = . if childalive_==.
label define lab_died 1 "child has died" 0 "child is alive"
label values child_died lab_died
tab childalive_ child_died, miss


bysort ind_id: egen tot_child_died = sum(child_died)
	//For each woman, sum the number of children who died
	
bysort ind_id: egen child_died_per_wom = max(tot_child_died)
lab var child_died_per_wom "Total child death for each women (birth recode)"

gen missing_cm = (ydead_survey==. & childalive_==2)
tab missing_cm,m
bysort ind_id: egen hh_miss_cm = max(missing_cm)
tab hh_miss_cm,m

	
	//Identify child under 18 mortality in the last 5 years
gen child18_died = child_died 
replace child18_died=0 if age_death>=18 & age_death<. 
label values child18_died lab_died
tab child18_died, miss	

			
bysort ind_id: egen tot_child18_died_5y=sum(child18_died) if ydead_survey<=5
	/*Total number of children under 18 who died in the past 5 years 
	prior to the interview date */	
	
replace tot_child18_died_5y=0 if tot_child18_died_5y==. & tot_child_died>=0 & tot_child_died<.
	/*All children who are alive or who died longer than 5 years from the 
	interview date are replaced as '0'*/
	
replace tot_child18_died_5y=. if child18_died==1 & ydead_survey==.
	//Replace as '.' if there is no information on when the child died  
replace tot_child18_died_5y=. if hh_miss_cm==1 & tot_child18_died_5y==0

	
tab tot_child_died tot_child18_died_5y, miss


bysort ind_id: egen childu18_died_per_wom_5y = max(tot_child18_died_5y)
lab var childu18_died_per_wom_5y "Total child under 18 death for each women in the last 5 years (birth recode)"	
tab childu18_died_per_wom_5y,m


	//Keep one observation per women
bysort ind_id: gen id=1 if _n==1
keep if id==1
drop id
duplicates report ind_id 

gen women_BR = 1 
	//Identification variable for observations in BR recode

		
	//Retain relevant variables:	
keep ind_id women_BR child_died_per_wom childu18_died_per_wom_5y ///
f2_s2_206 f2_s2_207 f2_s2_210 f2_s2_211_* f2_s2_217_3
order ind_id women_BR child_died_per_wom childu18_died_per_wom_5y ///
f2_s2_206 f2_s2_207 f2_s2_210 f2_s2_211_* f2_s2_217_3
	

save "$path_out/ECU18_BR.dta", replace


********************************************************************************
**# 1.4 DATA MERGING
********************************************************************************

clear
use "$path_out/ECU18_NU.dta", clear
count
rename weight weight_kg
rename height height_cm



*** Merging women's record data
***************************************************
merge 1:1 ind_id using "$path_out/ECU18_BR.dta"  
drop _merge
erase "$path_out/ECU18_BR.dta"



*** Merging nutrition recode 
***************************************************
merge 1:1 hh_id ind_id using "$path_out/ECU18_childyouth.dta"
drop _merge	
erase "$path_out/ECU18_childyouth.dta"


*** Merging household recode 
***************************************************
merge m:1 hh_id using "$path_out/ECU18_HR.dta" 
drop _merge
erase "$path_out/ECU18_HR.dta"


sort hh_id ind_id
duplicates report hh_id ind_id 
	//168,747 individuals; no duplicates. 


	
********************************************************************************
**# 1.5 RENAMING DEMOGRAPHIC VARIABLES ***
********************************************************************************

//Sample weight
desc fexp
clonevar  weight = fexp 
label var weight "Sample weight"


//Area: urban or rural	
codebook area 
recode area (2=0)
label define lab_area 1 "urban" 0 "rural"
label values area lab_area
label var area "Area: urban-rural"
tab area, miss


//Relationship to the head of household
desc f1_s2_7 
clonevar relationship = f1_s2_7
codebook relationship, tab (99) 
recode relationship (4/7=4) (9=5) (8=6)
label define lab_rel 1"head" 2"spouse" 3"child" 4"extended family" ///
					 5"not related" 6"maid"
label values relationship lab_rel
label var relationship "Relationship to the head of household"
tab f1_s2_7 relationship, miss


//Sex of household member	
codebook sexo 
clonevar sex = sexo	
label define lab_sex 1"male" 2"female"
label values sex lab_sex	
label var sex "Sex of household member"
tab sex,m


//Household headship
bys	hh_id: egen missing_hhead = min(relationship)
tab missing_hhead,m 
gen household_head=.
replace household_head=1 if relationship==1 & sex==1 
replace household_head=2 if relationship==1 & sex==2
bysort hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
label define head 1"male-headed" 2"female-headed"
label values headship head
label var headship "Household headship"
tab headship, miss
*list hh_id ind_id relationship sex if headship==3
replace headship=. if headship==3


//Age of household member
sum edadanios
*compare f1_s2_3_1 edadanios
	// We use the 'edadanios' variable for consistency purpose
clonevar age = edadanios 
label var age "Age of household member"


//Age group 
recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+"), gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"


//Marital status of household member

codebook f1_s2_16, tab (99) 

	/* 1 = married, 2 = some form of union; 
	3 = living together but not married; 
	4 = separated from a marriage (not divorced); 
	5 = divorced; 6 = widow; 7 = single; . = individuals <=11 years old */
	
gen		marital = 1 if f1_s2_16==7  
replace marital = 2 if f1_s2_16<= 3 			
replace marital = 3 if f1_s2_16 == 6				
replace marital = 4 if f1_s2_16 == 5				
replace marital = 5 if f1_s2_16 == 4 				
label define lab_mar 1"never married" 2"currently married" 3"widowed" ///
					 4"divorced" 5"not living together"
label values marital lab_mar	
label var marital "Marital status of household member"
tab f1_s2_16 marital, miss


//Total number of de jure hh members in the household
gen member = 1
bysort hh_id: egen hhsize = sum(member)
label var hhsize "Household size"
tab hhsize, miss
drop member


//Subnational region
	/* The geographical coverage of the ENSANUT 2018 covers national, 
	regional and provincial levels (p.19 of the methodological document). */

rename region region_svy
gen region = prov
destring region, replace
label define region_lab  ///
1"Azuay" 2"Bolívar" 3"Cañar" ///
4"Carchi" 5"Cotopaxi" 6"Chimborazo" ///
7"El Oro" 8"Esmeraldas" 9"Guayas" ///
10"Imbabura" 11"Loja" 12"Los Ríos" ///
13"Manabí" 14"Morona Santiago" 15"Napo" ///
16 "Pastaza" 17 "Pichincha" 18 "Tungurahua" ///
19 "Zamora Chinchipe" 20 "Galápagos" 21 "Sucumbios" ///
22 "Orellana" 23 "Santo Domingo" ///
24 "Santa Elena" 90 "Zona No Delimitada"
label values region region_lab 
lab var region "Region for subnational decomposition"
codebook region, tab (99)


clonevar region_01 = region
replace region_01 = . if region==90
lab var region_01 "COT:Region for subnational decomposition"


********************************************************************************
***  Step 2 Data preparation  ***
***  Standardization of the 10 Global MPI indicators 
***  Identification of non-deprived & deprived individuals  
********************************************************************************

********************************************************************************
*** Step 2.1 Years of Schooling ***
********************************************************************************

/* f1_s2_19_1 = Highest educational level completed
		1: none 
		2: literacy center 
		3: pre-school 
		4: primary (basic elementary + middle basic = 6 years)
		5: basic education (10 years of compulsory education; 
		   5-14 years old - from basic preparatory to upper basic)
		6: Secondary (6 years of secondary = lower + upper)
		7: Middle education / high school (upper secondary; 15 to 17 years old)
		8: non-university higher education 
		9: university superior 
		10: graduate, doctorate, phd */

desc f1_s2_19_1 f1_s2_19_2
codebook f1_s2_19_1 f1_s2_19_2, tab(99) 
	/* f1_s2_19_1 = Highest level of education completed
	   f1_s2_19_2 = Highest year of education completed */
	   
tab f1_s2_19_1 f1_s2_19_2,m	
	
clonevar edulevel = f1_s2_19_1 
label define lab_edu 1"No education" 2"Literacy center" 3"Pre-school" ///
4"Primary" 5"Basic education" 6"Secondary" 7"Secondary/High school" ///
8"Higher non-universitary" 9"Universitary" 10"Postgraduate;doctorate;phd"
lab values edulevel lab_edu
label var edulevel "Highest level of education attended"
replace edulevel = 0 if edadmeses<60 & f1_s2_19_1==.
tab edulevel f1_s2_19_1,m

clonevar eduhighyear = f1_s2_19_2 
lab var eduhighyear "Highest grade attended for each level of edu"
replace eduhighyear = 0 if edadmeses<60 & f1_s2_19_2==.
replace eduhighyear = 0 if edulevel<=3
tab eduhighyear f1_s2_19_2,m


** Creating educational level variable
gen	eduyears = eduhighyear
replace eduyears = eduhighyear + 6  if edulevel==6
	/* We assume individuals have gone through 6 years of primary 
	before reaching secondary. */
replace eduyears = eduhighyear + 9  if edulevel==7
	/* We assume individuals have gone through 9 years of schooling 
	before reaching high school. */
replace eduyears = eduhighyear + 12 if edulevel==8 | edulevel==9
	// Tertiary education 
replace eduyears = eduhighyear + 15 if edulevel==10
	// Post-graduate
lab var eduyears "Total number of years of education accomplished"


*** Checking for further inconsistencies 
replace eduyears = . if age<=eduyears & age>0 
	/*There are cases in which the years of schooling are greater than the 
	age of the individual. This is clearly a mistake in the data.*/
replace eduyears = 0 if age< 10
replace eduyears = 0 if (age==10 | age==11) & eduyears < 6 
	/*The variable "eduyears" was replaced with a '0' for ineligible household 
	members, i.e.: those who have not completed 6 years of schooling following 
	their starting school age */
tab eduyears, miss


	/*A control variable is created on whether there is information on 
	years of education for at least 2/3 of the eligible household members*/	
gen temp = 1 if eduyears!=. & age>=12 & age!=.
replace temp = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp = 1 if age==11 & eduyears>=6 & eduyears<.
bysort	hh_id: egen no_missing_edu = sum(temp)	
	//Total eligible household members with no missing years of education
gen temp2 = 1 if age>=12 & age!=.
replace temp2 = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp2 = 1 if age==11 & eduyears>=6 & eduyears<.
bysort hh_id: egen hhs = sum(temp2)
	/*Total number of eligible household members who should have information 
	on years of education */
replace no_missing_edu = no_missing_edu/hhs
replace no_missing_edu = (no_missing_edu>=2/3)
	/*Identify whether there is information on years of education for at 
	least 2/3 of the eligible household members */
tab no_missing_edu, miss
	//The value for 0 (missing) is 0.06% 
label var no_missing_edu "No missing edu for at least 2/3 of the eligible HH members"
drop temp temp2 hhs


*** Standard MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed six years of schooling. */
*******************************************************************  
gen	 years_edu6 = (eduyears>=6)
replace years_edu6 = . if eduyears==.
bysort hh_id: egen hh_years_edu6_1 = max(years_edu6)
gen	hh_years_edu6 = (hh_years_edu6_1==1)
replace hh_years_edu6 = . if hh_years_edu6_1==.
replace hh_years_edu6 = . if hh_years_edu6==0 & no_missing_edu==0 
lab var hh_years_edu6 "Household has at least one member with 6 years of edu"
tab hh_years_edu6, miss

	
*** Destitution MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed at least one year of schooling. */
*******************************************************************
gen	years_edu1 = (eduyears>=1)
replace years_edu1 = . if eduyears==.
bysort	hh_id: egen hh_years_edu_u = max(years_edu1)
replace hh_years_edu_u = . if hh_years_edu_u==0 & no_missing_edu==0
lab var hh_years_edu_u "Household has at least one member with 1 year of edu"
tab hh_years_edu_u, miss


********************************************************************************
*** Step 2.2 Child School Attendance ***
********************************************************************************

desc f1_s2_17 
codebook f1_s2_17, tab (99)
	//Does (...) currently attend school?
tab f1_s2_17 if edadmeses<60,m 

gen	attendance = .
replace attendance = 0 if f1_s2_17==2 
	//not currently attending
replace attendance = 1 if f1_s2_17==1 
	//currently attending
replace attendance = 0 if age<5 | age>24 
	//Replace attendance with '0' for individuals who are not of school age 
label define lab_attendance 1 "currently attending" 0 "not currently attending"
label values attendance lab_attendance
lab var attendance "Currently attending school"
tab attendance, miss 

	
*** Standard MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 8. */ 
******************************************************************* 
gen	child_schoolage = (age>=6 & age<=14)
	/*In Ecuador, the official school entrance age to primary school is 
	6 years. So, age range is 6-14 (=6+8) */ 

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the school age children */
count if child_schoolage==1 & attendance==.
	//How many eligible school aged children are not attending school: 3 children 
gen temp = 1 if child_schoolage==1 & attendance!=.
	/*Generate a variable that captures the number of eligible school aged 
	children who are attending school */
bysort hh_id: egen no_missing_atten = sum(temp)	
	/*Total school age children with no missing information on school 
	attendance */
gen temp2 = 1 if child_schoolage==1	
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are of school age
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)
	/*Identify whether there is missing information on school attendance for 
	more than 2/3 of the school age children */			
tab no_missing_atten, miss
	// No missing value
label var no_missing_atten "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs
	
bysort hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
	//It takes value 1 if the household has children in school age
lab var hh_children_schoolage "Household has children in school age"


gen	child_not_atten = (attendance==0) if child_schoolage==1
replace child_not_atten = . if attendance==. & child_schoolage==1
bysort	hh_id: egen any_child_not_atten = max(child_not_atten)
gen	hh_child_atten = (any_child_not_atten==0) 
replace hh_child_atten = . if any_child_not_atten==.
replace hh_child_atten = 1 if hh_children_schoolage==0
replace hh_child_atten = . if hh_child_atten==1 & no_missing_atten==0 
lab var hh_child_atten "Household has all school age children up to class 8 in school"
tab hh_child_atten, miss

	
*** Destitution MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 6. */ 
******************************************************************* 
gen	child_schoolage_6 = (age>=6 & age<=12) 
	/*Note: In Ecuador, the official school entrance age is 6 years.  
	  So, age range for destitution measure is 6-12 (=6+6) */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the children attending school up to 
	class 6 */	
count if child_schoolage_6==1 & attendance==.	
gen temp = 1 if child_schoolage_6==1 & attendance!=.
bysort hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_6==1	
bysort hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)			
tab no_missing_atten_u, miss
label var no_missing_atten_u "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs		
		
bysort	hh_id: egen hh_children_schoolage_6 = sum(child_schoolage_6)
replace hh_children_schoolage_6 = (hh_children_schoolage_6>0) 
lab var hh_children_schoolage_6 "Household has children in school age (6 years of school)"

gen	child_atten_6 = (attendance==1) if child_schoolage_6==1
replace child_atten_6 = . if attendance==. & child_schoolage_6==1
bysort	hh_id: egen any_child_atten_6 = max(child_atten_6)
gen	hh_child_atten_u = (any_child_atten_6==1) 
replace hh_child_atten_u = . if any_child_atten_6==.
replace hh_child_atten_u = 1 if hh_children_schoolage_6==0
replace hh_child_atten_u = . if hh_child_atten_u==0 & no_missing_atten_u==0 
lab var hh_child_atten_u "Household has at least one school age children up to class 6 in school"
tab hh_child_atten_u, miss


********************************************************************************
*** Step 2.3 Nutrition ***
********************************************************************************
	
desc f1_s7_1
	/* All household members in the sample from the age of 0-99 years 
	were eligible for anthropometric measurement. The 'f1_s7_1' variable 
	identifies whether height & weight measurement was successfully 
	collected from them. */ 
tab f1_s7_1,m
	/* Height & weight measurement was successfully collected from 154,160 
	(91% of the unweighted sample). This means, some 9% of the eligible 
	population (unweighted) was not successfully measured. For the purpose
	of the global MPI, we will consider those not successfully measured 
	as missing. */
	

*** No Eligible Women, Men or Children	
*** for nutrition indicator
******************************************************	
gen nutri_eligible = age_months<=840
replace nutri_eligible = 1 if age<=70
	/* For individuals with missing age in months information, 
	   we replace with information from the age in years variable. */
bysort	hh_id: egen n_nutri_eligible = sum(nutri_eligible) 	
gen	no_eligibles = (n_nutri_eligible==0) 	
lab var no_eligibles "Household has no eligible adults, youth or children"
tab no_eligibles, miss
drop n_nutri_eligible


*** No eligible adolescent/adult 
*** for nutrition indicator
***********************************************
gen elig_ag5_70 = (age_months>=61 & age_months<=840)
replace elig_ag5_70 = 1 if age>=5 & age<=70
	/* For individuals with missing age in months information, 
	   we replace with information from the age in years variable. */
bysort hh_id: egen n_elig_ag5_70 = sum(elig_ag5_70) 	
gen	no_elig_ag5_70 = (n_elig_ag5_70==0)
	//Takes value 1 if the household had no eligible 5-70 years 
lab var no_elig_ag5_70  "HH has no eligible 5-70 years for anthropometric"	
drop n_elig_ag5_70
tab no_elig_ag5_70, miss


*** No eligible children under 5
*** for child nutrition indicator
*****************************************
gen	child_eligible = (age_months<61) 
	/* For individuals with missing age in months information, 
	   we replace with information from the age in years variable. */
bysort	hh_id: egen hh_n_children_eligible = sum(child_eligible)  
	//Number of eligible children for anthropometrics
gen	no_child_eligible = (hh_n_children_eligible==0) 
	//Takes value 1 if there were no eligible children for anthropometrics
lab var no_child_eligible "Household has no children eligible for anthropometric"
drop hh_n_children_eligible
tab no_child_eligible, miss

	
********************************************************************************
*** Step 2.3a Adult & Adolescent Nutrition ***
********************************************************************************


*** Variable: ADULT BMI ***
	// In this section we compute BMI for all individuals up to 70 years
gen bmi=(weight_kg)/((height_cm/100)^2) if age<=70
lab var bmi "BMI"

gen low_bmi = (bmi<18.5)
replace low_bmi = . if child_eligible==1
	/* Replace children under 5 with '.' since we capture them through 
	   the underweight & stunting measures. */
replace low_bmi = . if bmi==.	   
lab var low_bmi "BMI <18.5"
lab define lab_low_bmi 1 "bmi<18.5" 0 "bmi>=18.5"
lab values low_bmi lab_low_bmi
tab low_bmi,miss


gen low_bmi_u = (bmi<17) 
replace low_bmi_u = . if child_eligible==1						 
replace low_bmi_u = . if bmi==. 
lab var low_bmi_u "BMI <17"
lab define lab_low_bmi_u 1 "bmi<17" 0 "bmi>=17"
lab values low_bmi_u lab_low_bmi_u
tab low_bmi_u, miss



*** Standard MPI: BMI-for-age for individuals 5-19 years 
*** 				  and BMI for individuals 20-70 years ***
*******************************************************************  
gen low_bmi_byage = 0
lab var low_bmi_byage "Individuals with low BMI or BMI-for-age"
replace low_bmi_byage = 1 if low_bmi==1
	// Replace variable "low_bmi_byage = 1" if eligible adults have low BMI	
 
/* Note: The following command replaces BMI with BMI-for-age for those between 
the age group of 5-19 where information is available. */
replace low_bmi_byage = 1 if low_bmiage==1 & youth_id==1
replace low_bmi_byage = 0 if low_bmiage==0 & youth_id==1

		
/* Note: The following control variable is applied when there is BMI information 
for adults and BMI-for-age for young people. */	
replace low_bmi_byage = . if low_bmi==. & low_bmiage==. 
		
bysort hh_id: egen hh_low_bmi = max(low_bmi_byage)
gen	hh_no_low_bmiage = (hh_low_bmi==0)
	/* Households take a value of '1' if all eligible adults and youth 
	in the household has normal bmi or bmi-for-age */	
replace hh_no_low_bmiage = . if hh_low_bmi==.
	/* Households take a value of '.' if there is no information from 
	eligible individuals in the household */
replace hh_no_low_bmiage = 1 if no_elig_ag5_70==1	
	//Households take a value of '1' if there is no eligible 5-70 yrs.		
drop hh_low_bmi
lab var hh_no_low_bmiage "HH has no persons with low BMI or BMI-for-age"
tab hh_no_low_bmiage, miss	


*** Destitution MPI: BMI-for-age for individuals 15-19 years 
*** 			     and BMI for individuals 20-54 years ***
********************************************************************************
gen low_bmi_byage_u = 0
replace low_bmi_byage_u = 1 if low_bmi_u==1
	/* Replace variable "low_bmi_byage_u = 1" if eligible adults have low 
	BMI (destitute cutoff). */	

/* Note: The following command replaces BMI with BMI-for-age for those between 
the age group of 15-19 here information is available. */
replace low_bmi_byage_u = 1 if low_bmiage_u==1 & youth_id==1
replace low_bmi_byage_u = 0 if low_bmiage_u==0 & youth_id==1


/*Note: The following control variable is applied when there is BMI information 
for adults and BMI-for-age for teenagers. */
replace low_bmi_byage_u = . if low_bmi_u==. & low_bmiage_u==.  

		
bysort hh_id: egen hh_low_bmi_u = max(low_bmi_byage_u)
gen	hh_no_low_bmiage_u = (hh_low_bmi_u==0)
	/*Households take a value of '1' if all eligible adults and teenagers in the 
	household has normal bmi or bmi-for-age (destitution cutoff) */
replace hh_no_low_bmiage_u = . if hh_low_bmi_u==.
	/*Households take a value of '.' if there is no information from eligible 
	individuals in the household */
replace hh_no_low_bmiage_u = 1 if no_elig_ag5_70==1	
	//Households take a value of '1' if there is no eligible 5-70 yrs.	
drop hh_low_bmi
lab var hh_no_low_bmiage_u "DST: HH has no persons with low BMI or BMI-for-age"
tab hh_no_low_bmiage_u, miss

	
********************************************************************************
*** Step 2.3b Child Nutrition ***
********************************************************************************

*** Child Underweight Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(underweight)
gen	hh_no_underweight = (temp==0) 
	//Takes value 1 if no child in the hh is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_underweight "Household has no child underweight - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(underweight_u)
gen	hh_no_underweight_u = (temp==0) 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
lab var hh_no_underweight_u "Destitute: Household has no child underweight"
drop temp


*** Child Stunting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(stunting)
gen	hh_no_stunting = (temp==0) 
	//Takes value 1 if no child in the hh is stunted
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_stunting "Household has no child stunted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(stunting_u)
gen	hh_no_stunting_u = (temp==0) 
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
lab var hh_no_stunting_u "Destitute: Household has no child stunted"
drop temp


*** Child Wasting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(wasting)
gen	hh_no_wasting = (temp==0) 
	//Takes value 1 if no child in the hh is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_wasting "Household has no child wasted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(wasting_u)
gen	hh_no_wasting_u = (temp==0) 
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
lab var hh_no_wasting_u "Destitute: Household has no child wasted"
drop temp


*** Child Either Underweight or Stunted Indicator ***
************************************************************************

*** Standard MPI ***
gen uw_st = 1 if stunting==1 | underweight==1
	//Takes value 1 if child in the hh is stunted or underweight 
replace uw_st = 0 if stunting==0 & underweight==0
	//Takes value 0 if child in the hh is not stunted and not underweight 
replace uw_st = . if stunting==. & underweight==.

bysort hh_id: egen temp = max(uw_st)
gen hh_no_uw_st = (temp==0) 
replace hh_no_uw_st = . if temp==.
drop temp
replace hh_no_uw_st = 1 if no_child_eligible==1
	//Households with no eligible children will receive a value of 1 
lab var hh_no_uw_st "HH has no child underweight or stunted"


*** Destitution MPI  ***
gen uw_st_u = 1 if stunting_u==1 | underweight_u==1
replace uw_st_u = 0 if stunting_u==0 & underweight_u==0
replace uw_st_u = . if stunting_u==. & underweight_u==.

bysort hh_id: egen temp = max(uw_st_u)
gen hh_no_uw_st_u = (temp==0) 
replace hh_no_uw_st_u = . if temp==.
drop temp
replace hh_no_uw_st_u = 1 if no_child_eligible==1
lab var hh_no_uw_st_u "DST: HH has no child underweight or stunted"


********************************************************************************
*** Step 2.3c Household Nutrition Indicator ***
********************************************************************************

*** Standard MPI ***
/* Members of the household are considered deprived if the household has a 
child under 5 whose height-for-age or weight-for-age is under two standard 
deviation below the median, or has youth with BMI-for-age that is under two 
standard deviation below the median, or has adults with BMI threshold that is 
below 18.5 kg/m2. Households that have no eligible adult, youth AND no eligible 
children are considered non-deprived. The indicator takes a value of missing 
only if all eligible adults, youth and eligible children have missing 
information in their respective nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st = 1
replace hh_nutrition_uw_st = 0 if hh_no_low_bmiage==0 | hh_no_uw_st==0
replace hh_nutrition_uw_st = . if hh_no_low_bmiage==. & hh_no_uw_st==.
	/* Replace indicator as missing if household has eligible adult, 
	adolescent and child with missing nutrition information. */
replace hh_nutrition_uw_st = . if hh_no_low_bmiage==. & hh_no_uw_st==1 & no_child_eligible==1
	/* Replace indicator as missing if household has eligible 
	adolescent/adult with missing nutrition information and no eligible 
	child for anthropometric measures. */ 
replace hh_nutrition_uw_st = . if hh_no_uw_st==. & ///
								  hh_no_low_bmiage==1 & no_elig_ag5_70==1
	/* Replace indicator as missing if household has eligible child 
	with missing nutrition information and no eligible adolescent/adult 
	for anthropometric measures. */ 
replace hh_nutrition_uw_st = 1 if no_eligibles==1  
 	/* We replace households that do not have the applicable population, 
	that is, adults, adolescent & children, as non-deprived in nutrition. */ 
lab var hh_nutrition_uw_st "HH has no individuals undernourished"
tab hh_nutrition_uw_st, miss


*** Destitution MPI ***
/* Members of the household are considered deprived if the household has a 
child under 5 whose height-for-age or weight-for-age is under three standard 
deviation below the median, or has youth with BMI-for-age that is under three 
standard deviation below the median, or has adults with BMI threshold that is 
below 17.0 kg/m2. Households that have no eligible adult, youth AND no eligible 
children are considered non-deprived. The indicator takes a value of missing 
only if all eligible adults, youth and eligible children have missing 
information in their respective nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st_u = 1
replace hh_nutrition_uw_st_u = 0 if hh_no_low_bmiage_u==0 | hh_no_uw_st_u==0
replace hh_nutrition_uw_st_u = . if hh_no_low_bmiage_u==. & hh_no_uw_st_u==.
	/* Replace indicator as missing if household has eligible adult, 
	adolescent and child with missing nutrition information. */
replace hh_nutrition_uw_st_u = . if hh_no_low_bmiage_u==. & hh_no_uw_st_u==1 & no_child_eligible==1
	/* Replace indicator as missing if household has eligible 
	adolescent/adult with missing nutrition information and no eligible 
	child for anthropometric measures. */ 
replace hh_nutrition_uw_st_u = . if hh_no_uw_st_u==. & ///
									hh_no_low_bmiage_u==1 & no_elig_ag5_70==1
	/* Replace indicator as missing if household has eligible child 
	with missing nutrition information and no eligible adolescent/adult 
	for anthropometric measures. */ 
replace hh_nutrition_uw_st_u = 1 if no_eligibles==1   
 	/* We replace households that do not have the applicable population, 
	that is, adults, adolescent & children, as non-deprived in nutrition. */ 	
lab var hh_nutrition_uw_st_u "DST: HH has no individuals undernourished"
tab hh_nutrition_uw_st_u, miss


********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************


*** No eligible women 10-49 years 
*** for child mortality indicator
*****************************************
gen	fem_eligible = (age>=10 & age<=49 & sex==2)
bysort	hh_id: egen hh_n_fem_eligible = sum(fem_eligible) 	
	//Number of eligible women for interview in the hh
gen	no_fem_eligible = (hh_n_fem_eligible==0) 									
	//Takes value 1 if the household had no eligible women for an interview
lab var no_fem_eligible "Household has no eligible women 10-49 years"
drop hh_n_fem_eligible 
tab no_fem_eligible, miss


codebook f2_s2_211_1 f2_s2_211_2 f2_s2_211_3
	/* f2_s2_211_1: number of sons who have died 
	   f2_s2_211_2: number of daughters who have died
	   f2_s2_211_3: total number of children who have died */	
	
	
egen temp_f = rowtotal (f2_s2_211_1 f2_s2_211_2), missing
compare temp_f f2_s2_211_3
	//Total child mortality reported by women 10-49 years: matches
replace temp_f = 0 if f2_s2_210==1 & f2_s2_217_3==0
	//This line replaces data inconsistency	
replace temp_f = 0 if f2_s2_210==2
	//This line replaces women who reported no mortality	
replace temp_f = 0 if f2_s2_207==2 
	//This line replaces women who have never got pregnant	
replace temp_f = 0 if f2_s2_206==1 
	/*This line replaces women who are pregnant for 
	the first time, so no previous mortality. */
tab temp_f,m	
		
bysort	hh_id: egen child_mortality_f = sum(temp_f), missing
lab var child_mortality_f "Occurrence of child mortality reported by women"
tab child_mortality_f, miss

	
clonevar child_mortality = child_mortality_f 
lab var child_mortality "Total child mortality within household"
tab child_mortality, miss		


gen	hh_mortality = (child_mortality==0)
	/*Household is replaced with a value of "1" if there is no incidence of 
	child mortality*/
replace hh_mortality = . if child_mortality==.
replace hh_mortality = 1 if no_fem_eligible==1 
	/*Household is replaced with a value of "1" if there is 
	no eligible women  */
lab var hh_mortality "HH had no child mortality"
tab hh_mortality, miss


*** Standard MPI *** 
/* Members of the household are considered deprived if women in the household 
reported mortality among children under 18 in the last 5 years from the survey 
year. Members of the household is considered non-deprived if eligible women 
within the household reported (i) no child mortality or (ii) if any child died 
longer than 5 years from the survey year or (iii) if any child 18 years and 
older died in the last 5 years. In adddition, members of the household were 
identified as non-deprived if eligible men within the household reported no 
child mortality in the absence of information from women. Households that have 
no eligible women or adult are considered non-deprived. The indicator takes 
a missing value if there was missing information on reported death from 
eligible individuals. */
************************************************************************
tab childu18_died_per_wom_5y, miss

replace childu18_died_per_wom_5y = 0 if childu18_died_per_wom_5y==. & temp_f==0 
	/*Assign a value of "0" for: all eligible women who never 
	ever gave birth; never got pregnant; reported no mortality; 
	first time pregnant at time of survey. */	
replace childu18_died_per_wom_5y = 0 if no_fem_eligible==1 
	/*Assign a value of "0" for:
	- individuals living in households that have non-eligible women */	
	
replace childu18_died_per_wom_5y = . if ind_id=="08015400020110102" | ///
										ind_id=="13175090160105102" | ///
										ind_id=="23015001230112101"
	/* Assign a value of "missing" for three women who reported 
	child mortality through the f2_s2_218_2_c f2_s2_218_5_c f2_s2_218_6_c 
	and f2_s2_218_7_c variables. However, they reported no mortality across
	the f2_s2_217_3 and f2_s2_210 variables. */	
	
	
bysort hh_id: egen childu18_mortality_5y = sum(childu18_died_per_wom_5y), missing
label var childu18_mortality_5y "Under 18 child mortality within household past 5 years reported by women"
tab childu18_mortality_5y, miss		
	
gen hh_mortality_u18_5y = (childu18_mortality_5y==0)
replace hh_mortality_u18_5y = . if childu18_mortality_5y==.
lab var hh_mortality_u18_5y "Household had no under 18 child mortality in the last 5 years"
tab hh_mortality_u18_5y, miss 


*** Destitution MPI *** 
*** (same as standard MPI) ***
************************************************************************
gen hh_mortality_u = hh_mortality_u18_5y	
lab var hh_mortality_u "Household had no under 18 child mortality in the last 5 years"	


*** Harmonised MPI *** 
************************************************************************
	/*In the earlier survey, there is no birth history data. This means, 
	there is no information on the date of death of children who have died. 
	As such, we are not able to construct the indicator on child mortality 
	under 18 that occurred in the last 5 years for this survey. Instead, 
	we identify individuals as deprived if any children died in the household. 
	As such, for harmonisation purpose, we construct the same indicator in 
	this survey.*/
	
gen	hh_mortality_01 = hh_mortality
lab var hh_mortality_01 "COT: HH had no child mortality"
tab hh_mortality_01, miss


clonevar hh_mortality_u_01 = hh_mortality_01
lab var hh_mortality_u_01 "COT-DST: HH had no child mortality"


********************************************************************************
**# Step 2.5 Electricity ***
********************************************************************************

*** Standard MPI ***
/*Members of the household are considered 
deprived if the household has no electricity */
***************************************************
lookfor electricidad alumbrado
codebook f1_s1_10, tab(99)
	/* f1_s1_10: Main source of lighting:
		1. Public electricity 
		2. Private power plant
		3. Candle, lamp, lighter, gas
		4. None */
clonevar electricity = f1_s1_10
recode   electricity (1/2=1) (3/4=0)
lab define lab_yes_no 0 "no" 1 "yes"
lab values electricity lab_yes_no
label var  electricity "Electricity"
tab f1_s1_10 electricity, m
tab electricity, m


*** Destitution MPI  ***
*** (same as standard MPI) ***
***************************************************
gen electricity_u = electricity
label var electricity_u "Household has electricity"


********************************************************************************
*** Step 2.6 Sanitation ***
********************************************************************************
/*
Improved sanitation facilities include flush or pour flush toilets to sewer 
systems, septic tanks or pit latrines, ventilated improved pit latrines, pit 
latrines with a slab, and composting toilets. These facilities are only 
considered improved if it is private, that is, it is not shared with other 
households.
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-02-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/
lookfor sanitaria servicio higiénico	
	/*
	In Ecuador ENSANUT 2018:
	f1_s1_13: Type of toilet service that the household has 
		1. Toilet and sewer
		2. Toilet and septic tank
		3. Toilet and cesspool
		4. Latrine
		5. None
	f1_s1_14: How does septic tank / cesspool waste end up
		1. Any open place (river, creek, ditch, street, patio, land, open field)
		2. Do they remain in the septic tank / cesspool to be buried later
		3. Another part, provided it is not an open site
		4. Don't know	
	f1_s1_15: Type of latrine that the household has 
		1. Latrine - pit with slab (covered with some material such as cement, wood, etc.)
		2. Latrine - pit without slab / open pit (rudimentary hole in the ground)
	f1_s1_17: What do the members of the household if there is no toilet service
		1. Go to the mountains, fields, throw in the garbage in a package
		2. Use a sanitary and/or borrowed facility	

	f1_s1_19: Do you share the toilet service you have with other people who are not members of this household
		1. Yes
		2. No
		*/
	
codebook f1_s1_13 f1_s1_13 f1_s1_14 f1_s1_15 f1_s1_18 f1_s1_19, tab(99)
tab  	 f1_s1_13 f1_s1_14, m
tab  	 f1_s1_13 f1_s1_15, m

clonevar toilet = f1_s1_13 
label define lab_toilet 1"Toilet and sewer" 2"Toilet and septic tank" ///
3"Toilet and cesspool" 4"Latrine" 5"None"										
lab values toilet lab_toilet
lab var toilet "Type of toilet"
tab toilet f1_s1_13, miss


codebook f1_s1_19 
	/* 1 = yes; 2 = no */
gen shared_toilet = f1_s1_19
recode shared_toilet (2=0) 
lab define lab_toi 0 "No" 1 "Yes"
lab values shared_toilet lab_toi
tab shared_toilet, miss
lab var shared_toilet "Household has access to shared toilet"
tab f1_s1_19 shared_toilet, m


*** Standard MPI ***
/*Members of the household are considered deprived if the household's 
sanitation facility is not improved (according to the SDG guideline) 
or it is improved but shared with other households*/
********************************************************************
codebook toilet, tab(9)

gen	toilet_mdg = (toilet==1 & shared_toilet==0) | ///
				 (f1_s1_14>=2 & f1_s1_14<=4) | ///
				 (f1_s1_15==1) 
	/*Household is assigned a value of '1' if it uses improved 
	sanitation and does not share toilet with other households.  */
	
replace toilet_mdg = 0 if (toilet==1 & shared_toilet==1) | ///
						  (f1_s1_13==5) | (f1_s1_14==1)  | ///
						  (f1_s1_15==2) 
	/*Household is assigned a value of '0' if it uses improved sanitation 
	but shares toilet with other households  */	
	
	
lab var toilet_mdg "Household has improved sanitation with SDG Standards"
tab toilet toilet_mdg, miss
*tab f1_s1_14 toilet_mdg, miss
*tab f1_s1_15 toilet_mdg, miss
*tab f1_s1_19 toilet_mdg, miss
tab toilet_mdg, m


*** Destitution MPI ***
/*Members of the household are considered deprived if household practises 
open defecation or uses other unidentifiable sanitation practises */
********************************************************************
gen	toilet_u = 1
replace toilet_u = 0 if toilet==5
lab var toilet_u "Household does not practise open defecation or others"
tab toilet toilet_u, miss
tab toilet_u,m


********************************************************************************
*** Step 2.7 Drinking Water  ***
********************************************************************************

/*
Improved drinking water sources include the following: piped water into 
dwelling, yard or plot; public taps or standpipes; boreholes or tubewells; 
protected dug wells; protected springs; packaged water; delivered water and 
rainwater which is located on premises or is less than a 30-minute walk from 
home roundtrip. 
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-01-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

codebook f1_s1_25, tab (99) 
codebook f1_s1_27, tab(999)
	/* f1_s1_25: Where does the water used for drinking in this household mainly come from:	
	   f1_s1_27: Time it takes to get to the source to get water to drink */
	
gen timetowater= f1_s1_27
gen ndwater = .	
	
gen water = f1_s1_25
label define lab_water  ///
		1 "Public network" ///
		2 "Public pipe" ///
		3 "Other piped source" ///
		4 "Delivery truck/tanker" ///
		5 "Bottled/packaged water" ///
		6 "Water in case (bottled water)" ///
		7 "Closed well/protected well" ///
		8 "Unprotected well" ///
		9 "Protected spring" ///
		10 "Spring not protected" ///
		11 "River or ditch" ///
		12 "Rainwater" ///
		13 "Other"
lab values water lab_water
lab var water "Type of water"
 	
	
*** Standard MPI ***
/* Members of the household are considered deprived if the household 
does not have access to improved drinking water (according to the SDG 
guideline) or safe drinking water is at least a 30-minute walk from 
home, roundtrip */
********************************************************************
codebook water, tab(99)

gen	water_mdg = 1 if water<=7 | water==9 | water==12 	
replace water_mdg = 0 if water==8 | water==10 | water==11 | water==13 	
tab water water_mdg,m	
tab water if timetowater==888, m	
replace water_mdg = 0 if water_mdg==1 & timetowater >= 30 & timetowater!=. & ///
						 timetowater!=888 	
lab var water_mdg "HH has safe drinking water, under 30 mins dist"
tab water_mdg, m


*** Destitution MPI ***
/* Members of the household is identified as destitute if household 
does not have access to safe drinking water, or safe water is more 
than 45 minute walk from home, round trip.*/
********************************************************************
gen	water_u     = 1 if water<=7 | water==9 | water==12   
replace water_u = 0 if water==8 | water==10 | water==11 | water==13 
replace water_u = 0 if water_u==1 & timetowater > 45 & timetowater!=. & ///
					   timetowater!=888 & timetowater!=999
lab var water_u "DST: HH has safe drinking water, 0-45 mins dist"
tab water_u, m


********************************************************************************
*** Step 2.8 Housing ***
********************************************************************************

desc f1_s1_3 f1_s1_4 f1_s1_5 
codebook f1_s1_3 f1_s1_4 f1_s1_5, tab(99)

/* In Ecuador ENSANUT 2018:
	f1_s1_3: Predominant material of the roof of the dwelling:
			1. Concrete/ slab/ cement
			2. Asbestos 
			3. Zinc 
			4. Tile 
			5. Palm/ straw/ leaf - - non-imprv
			6. Other - non-imprv
	f1_s1_4: Predominant material of the floor of the dwelling:
			1. Parquet /plank/treated plank/floor
			2. Ceramic/tile/vinyl 
			3. Marble/ marmeton
			4. Cement/brick
			5. Untreated board/plank
			6. Cane 
			7. Dirt - non-imprv
			8. Other - non-imprv
	f1_s1_5: Predominant material of the walls of the dwelling:
			1. Concrete/block/brick
			2. Asbestos/cement 
			3. Adobe 
			4. Wood - non-imprv
			5. Cane, coated reed - non-imprv
			6. Cane or mat - non-imprv
			7. Other - non-imprv 
*/

/* Members of the household are considered deprived if the household 
has a dirt, sand or dung floor */
codebook f1_s1_4, tab(9) 
gen floor = f1_s1_4
label define lab_floor ///
1 "Parquet/plank/treated plank/floor" ///
2 "Ceramic/tile/vinyl" /// 
3 "Marble/marmeton" ///
4 "Cement/brick" ///
5 "Untreated board/plank" ///
6 "Cane" ///
7 "Dirt" ///
8 "Other" 
label values floor lab_floor
lab var floor "Material of floor"
tab f1_s1_4 floor, m

gen	floor_imp = 1
replace floor_imp = 0 if floor==7 | floor==8  
lab var floor_imp "Household has floor that it is not earth/sand/dung"
tab floor floor_imp, miss


/* Members of the household are considered deprived if the household has wall 
made of natural or rudimentary materials */
codebook f1_s1_5, tab(9) 
gen wall = f1_s1_5
label define lab_wall 	///
1 "Concrete/block/brick" ///
2 "Asbestos/cement" ///
3 "Adobe" ///
4 "Wood" ///
5 "Cane, coated reed" ///
6 "Cane or mat" ///
7 "Other" 
label values wall lab_wall
lab var wall "Material of walls"
tab f1_s1_5 wall, m

gen	wall_imp = 1 
replace wall_imp = 0 if wall>=5 & wall<=7 
lab var wall_imp "Household has wall that it is not of low quality materials"
tab wall wall_imp, miss	


/* Members of the household are considered deprived if the household has roof 
made of natural or rudimentary materials */
codebook f1_s1_3, tab(9)
gen roof = f1_s1_3 
label define lab_roof ///
1 "Concrete/slab/cement" ///
2 "Asbestos" ///
3 "Zinc " ///
4 "Tile" 	///
5 "Palm/ straw/ leaf" ///
6 "Other" 
label values roof lab_roof
lab var roof "Material of roof"
tab f1_s1_3 roof, m

gen	roof_imp = 1 
replace roof_imp = 0 if roof==5 | roof==6  
lab var roof_imp "Household has roof that it is not of low quality materials"
tab roof roof_imp, miss


*** Standard MPI ***
/* Members of the household is deprived in housing if the roof, 
floor OR walls are constructed from low quality materials.*/
**************************************************************
gen housing_1 = 1
replace housing_1 = 0 if floor_imp==0 | wall_imp==0 | roof_imp==0
replace housing_1 = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_1 "Household has roof, floor & walls that it is not low quality material"
tab housing_1, miss


*** Destitution MPI ***
/* Members of the household is deprived in housing if two out 
of three components (roof and walls; OR floor and walls; OR 
roof and floor) the are constructed from low quality materials. */
**************************************************************
gen housing_u = 1
replace housing_u = 0 if (floor_imp==0 & wall_imp==0 & roof_imp==1) | ///
						 (floor_imp==0 & wall_imp==1 & roof_imp==0) | ///
						 (floor_imp==1 & wall_imp==0 & roof_imp==0) | ///
						 (floor_imp==0 & wall_imp==0 & roof_imp==0)
replace housing_u = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_u "Household has one of three aspects(either roof,floor/walls) that is not low quality material"
tab housing_u, miss


********************************************************************************
*** Step 2.9 Cooking Fuel ***
********************************************************************************

/*
Solid fuel are solid materials burned as fuels, which includes coal as well as 
solid biomass fuels (wood, animal dung, crop wastes and charcoal). 

Source: 
https://apps.who.int/iris/bitstream/handle/10665/141496/9789241548885_eng.pdf
*/
desc f1_s1_9
codebook f1_s1_9, tab(9) 
	/*
	f1_s1_9: This household mainly cooks with
		1. Gas
		2. Firewood/charcoal
		3. Electricity/induction 
		5. Does not cook
	*/
	
gen cookingfuel = f1_s1_9
label define lab_cf ///
1 "gas" ///
2 "firewood/charcoal" ///
3 "electricity" ///
5 "no food cooked in the house"
label values cookingfuel lab_cf 
lab var cookingfuel "Type of cookingfuel"
tab f1_s1_9 cookingfuel, m	


*** Standard MPI ***
/* Members of the household are considered deprived if the 
household uses solid fuels and solid biomass fuels for cooking. */
*****************************************************************
gen	cooking_mdg = 1
replace cooking_mdg = 0 if cookingfuel==2 
lab var cooking_mdg "Household uses clean fuels for cooking"
	/* Deprived if: coal/lignite, charcoal, wood, straw/shrubs/grass, 
					agricultural crop, animal dung */			 
tab cookingfuel cooking_mdg, miss	
tab cooking_mdg, miss
	
	
*** Destitution MPI ***
*** (same as standard MPI) ***
****************************************
gen	cooking_u = cooking_mdg
lab var cooking_u "Household uses clean fuels for cooking"



********************************************************************************
*** Step 2.10 Assets ownership ***
********************************************************************************


	//Television 
lookfor  televisor	
codebook f1_s1_44_7
*tab f1_s1_43 f1_s1_44_7,m
gen	television = f1_s1_44_7
replace television = 0 if f1_s1_44_7==2 
label var television "The household has at least one TV"
tab television, m

	//Radio - no data
lookfor radio	
gen	radio = . 
label var radio "The household has at least one radio"


	//Telephone 
lookfor telefónica 	celular
codebook f1_s1_44_10 f1_s2_23	
gen	telephone = f1_s1_44_10
replace telephone = 0 if f1_s1_44_10==2 

gen hp = 1 if f1_s2_23==1
replace hp = 0 if f1_s2_23==2 | f1_s2_23==.
bysort hh_id: egen mobilephone = max(hp)
tab mobilephone,m

replace telephone=1 if telephone!=1 & mobilephone==1	
tab mobilephone f1_s1_44_10 if telephone==1,miss
lab var telephone "Household has telephone (landline/mobilephone)"	
tab telephone,m


	//Refrigerator
lookfor refrigeradora	
codebook f1_s1_44_1
gen	refrigerator = f1_s1_44_1
replace refrigerator = 0 if f1_s1_44_1==2
label var refrigerator "Household has refrigerator"
tab refrigerator,m

	//Car - includes car and van
lookfor automovil	
codebook f1_s1_44_11
gen	car = f1_s1_44_11
replace car = 0 if f1_s1_44_11==2
label var car "Household has car or van"
tab car, miss

	//Bicycle - no data
lookfor bicicleta	
gen	bicycle = . 
label var bicycle "Household has bicycle"


	//Motorbike - no data
gen	motorbike = . 
label var motorbike "Household has motorbike"


	//Computer
lookfor computador	
codebook f1_s1_44_2
gen	computer = f1_s1_44_2
replace computer = 0 if f1_s1_44_2==2
label var computer "Household has computer"
tab computer,miss


	//Animal cart - no data
gen	animal_cart = . 
label var animal_cart "Household has nimal cart"


label define lab_assets 1 "Yes" 0 "No"
foreach var in television radio telephone refrigerator car ///
			   bicycle motorbike computer animal_cart {
label values `var' lab_assets			   	
replace `var' = . if `var'==. 
}
	//Missing values replaced



*** Standard MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own more than one of: radio, TV, telephone, bike, motorbike, 
refrigerator, computer or animal cart and does not own a car or truck.*/
*****************************************************************************
egen n_small_assets2 = rowtotal(television radio telephone refrigerator bicycle motorbike computer animal_cart), m
lab var n_small_assets2 "Household Number of Small Assets Owned" 
   
   
gen hh_assets2 = (car==1 | n_small_assets2 > 1) 
replace hh_assets2 = . if car==. & n_small_assets2==.
lab var hh_assets2 "HH has car or more than 1 small assets"
tab hh_assets2, miss


*** Destitution MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own any assets.*/
*****************************************************************************	
gen	hh_assets2_u = (car==1 | n_small_assets2>0)
replace hh_assets2_u = . if car==. & n_small_assets2==.
lab var hh_assets2_u "HH has car or at least 1 small assets"


********************************************************************************
*** Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************
	

	//Retain data on sampling design: 
desc estrato upm	
clonevar strata = estrato
clonevar psu = upm
destring hh_id ind_id psu strata, replace
desc hh_id ind_id psu strata


	//Retain year, month & date of interview:
sum fecha_anio fecha_mes fecha_dia 
gen year_interview  = fecha_anio
gen month_interview = fecha_mes 
gen date_interview  = fecha_dia 


	//Generate presence of subsample
gen subsample = .		


*** Keep main variables require for MPI calculation ***
keep hh_id ind_id subsample strata psu weight sex age hhsize ///
area agec7 agec4 agec2 region headship marital relationship  ///
no_eligibles no_elig_ag5_70 no_child_eligible no_fem_eligible ///
year_interview month_interview date_interview /// 
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
low_bmi low_bmi_u low_bmi_byage hh_no_low_bmiage ///
low_bmi_byage_u hh_no_low_bmiage_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u child_mortality ///
hh_mortality hh_mortality_u18_5y hh_mortality_u /// 
electricity electricity_u toilet shared_toilet toilet_mdg toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
region_01 hh_mortality_01 hh_mortality_u_01

	
	 
*** Order file	***
order hh_id ind_id subsample strata psu weight sex age hhsize ///
area agec7 agec4 agec2 region headship marital relationship  ///
no_eligibles no_elig_ag5_70 no_child_eligible no_fem_eligible ///
year_interview month_interview date_interview /// 
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
low_bmi low_bmi_u low_bmi_byage hh_no_low_bmiage ///
low_bmi_byage_u hh_no_low_bmiage_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u child_mortality ///
hh_mortality hh_mortality_u18_5y hh_mortality_u  /// 
electricity electricity_u toilet shared_toilet toilet_mdg toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
region_01 hh_mortality_01 hh_mortality_u_01



*** Rename key global MPI indicators for estimation ***
recode hh_mortality_u18_5y  (0=1)(1=0) , gen(d_cm)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ)
recode electricity 			(0=1)(1=0) , gen(d_elct)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani)
recode housing_1 			(0=1)(1=0) , gen(d_hsg)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl)
recode hh_assets2 			(0=1)(1=0) , gen(d_asst)
 

*** Rename key global MPI indicators for destitution estimation ***
recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ)
recode electricity_u		(0=1)(1=0) , gen(dst_elct)
recode water_u 				(0=1)(1=0) , gen(dst_wtr)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl)
recode hh_assets2_u 		(0=1)(1=0) , gen(dst_asst) 


*** Rename indicators for changes over time estimation ***	
recode hh_mortality_01		(0=1)(1=0) , gen(d_cm_01)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr_01)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt_01)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ_01)
recode electricity 			(0=1)(1=0) , gen(d_elct_01)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr_01)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani_01)
recode housing_1 			(0=1)(1=0) , gen(d_hsg_01)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl_01)
recode hh_assets2   		(0=1)(1=0) , gen(d_asst_01)	
	

recode hh_mortality_u_01    (0=1)(1=0) , gen(dst_cm_01)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr_01)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt_01)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ_01)
recode electricity_u		(0=1)(1=0) , gen(dst_elct_01)
recode water_u	 			(0=1)(1=0) , gen(dst_wtr_01)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani_01)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg_01)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl_01)
recode hh_assets2_u  		(0=1)(1=0) , gen(dst_asst_01)


*** Eligibility for years of schooling indicator ***
gen educ_elig = 1 
replace educ_elig = 0 if age < 10 
replace educ_elig = 0 if (age==10 | age==11) & eduyears < 6 
	/*The variable "educ_eligible" was replaced with a '0' for ineligible 
	individuals, i.e.: those who have not completed 6 years of schooling 
	following their starting school age.*/
label define lab_educ_elig 0"ineligible" 1"eligible"  
label values educ_elig lab_educ_elig
lab var educ_elig "Individual is eligible for educ indicator"
tab eduyears educ_elig,m


*** Generate coutry and survey details for estimation ***
char _dta[cty] 	  "Ecuador"
char _dta[ccty]   "ECU"
char _dta[year]   "2018" 	
char _dta[survey] "ENSANUT"
char _dta[ccnum]  "218"
char _dta[type]   "micro"



*** Sort, compress and save data for estimation ***
sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."	
save "$path_out/ecu_ensanut18.dta", replace 


	//Erase files from folder:
erase "$path_out/ECU18_NU.dta"	