********************************************************************************
/*
Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
Global Multidimensional Poverty Index - Ecuador ECV 2013-14 [STATA do-file]. 
Available from OPHI website: http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk
*/
********************************************************************************

clear all 
set more off
set maxvar 10000

global path_in "../rdta/Ecuador ECV 2013-14"    	// folder path
global path_out "cdta"
global path_ado "ado"


********************************************************************************
*** Step 1: Data preparation 
********************************************************************************
	
	/*Ecuador ECV 2013-14: Anthropometric information were recorded for 
	all individuals aged 0-98 years. For the purpose of the global MPI, 
	we have used nutrition data when the data is available for all 
	individuals but up to the age of 70 years. */


********************************************************************************
**# Step 1.1a Nutrition indicators for children 0-60 months
********************************************************************************

use "$path_in/ecv6r_personas.dta", clear 


tostring persona, replace
forvalues i=1(1)9 {
replace persona="0`i'" if persona=="`i'"
}
gen ind_id = identif_hog + persona  
label var  ind_id "individual id"


duplicates report ind_id
	//109,694 individuals; No duplicates

	
**Compute age in months for all members	
foreach var in ps80b ps80c {
replace `var'=. if `var'<0
}

gen am = (ps81c - ps80c)*12
	//ps81c: measurement year; ps80c: birth year
replace am = am + (ps81b - ps80b) if ps81b>=ps80b
	//ps81b: measurement month; ps80b: birth month
replace am = (am - 12) + (ps81b - 1) + (13 - ps80b) if ps81b<ps80b 
replace am = edad*12 if ps79==1 & ps80b==. 
label var am "Age in months for all members"	
sum am


**Childen up to 60 months	
gen age_months = pd03b if edad==0
replace age_months = pd03b+12 if edad==1
replace age_months = pd03b+24 if edad==2
replace age_months = pd03b+36 if edad==3
replace age_months = pd03b+48 if edad==4
replace age_months = 60 if am==60 & age_months==.
tab age_months, miss


gen child_id=1 if age_months<61
keep if child_id==1 
count
	//11,598 children

	
*** Variable: SEX ***
tab sexo, miss 
gen gender = sexo
tab gender


*** Variable: AGE ***
sum age_months
gen str6 ageunit = "months" 
label var ageunit "Months"	


*** Variable: BODY WEIGHT (KILOGRAMS) ***
gen weight = (ps82 + ps82a)/2 if ps82b==. 
replace weight = (ps82 + ps82a + ps82b)/3 if ps82b<.
sum weight


*** Variable: HEIGHT (CENTIMETERS) ***
gen height = (ps83 + ps83a)/2 if ps83b==. & edad<2 
replace height = (ps83 + ps83a + ps83b)/3 if ps83b<. & edad<2
replace height = (ps84 + ps84a)/2 if ps84b==. & edad>=2
replace height = (ps84 + ps84a + ps84b)/3 if ps84b<. & edad>=2
ta height, miss
replace height=. if ps84<0
sum height


*** Variable: MEASURED STANDING/LYING DOWN ***
gen measure = "l" if ps83<. 
replace measure = "h" if ps84<. 
tab measure, m


*** Variable: OEDEMA ***
gen oedema=" "


*** Variable: INDIVIDUAL CHILD SAMPLING WEIGHT ***
gen sw = fexp


*** z-scores ***
adopath + "$path_ado/igrowup_stata"	
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "children_nutri_ecu" 
lab var datalab "Working file"

igrowup_restricted reflib datalib datalab gender age_months ageunit ///
weight height measure oedema sw

use "$path_out/children_nutri_ecu_z_rc.dta", clear 

gen z_scorewa = _zwei
replace z_scorewa = . if _fwei==1 
lab var z_scorewa "z-score weight-for-age WHO"


*** Standard MPI indicator ***	
gen	underweight = (_zwei < -2.0) 
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "Child is underweight"
tab underweight, miss


gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "Child is stunted"
tab stunting, miss


gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "Child is wasted"
tab wasting, miss


count if _fwei==1 | _flen==1
	/* Note: in the context of Ecuador ECV 2013-14, 65 children 
	   were replaced as '.' because they have extreme z-scores 
	   that are biologically implausible. */


*** Destitution indicator  ***	
gen	underweight_u = (_zwei < -3.0) 
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "DST: Child is underweight"


gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "DST: Child is stunted"


gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "DST: Child is wasted"

 
clonevar weight_ch = fexp
 
	//Retain relevant variables:
keep  ind_id child_id age_months weight_ch under* stunting* wasting* 
order ind_id child_id age_months weight_ch under* stunting* wasting* 
sort  ind_id
duplicates report ind_id
save "$path_out/ECU13-14_child.dta", replace


	//Erase files from folder:
erase "$path_out/children_nutri_ecu_z_rc.xls"
erase "$path_out/children_nutri_ecu_prev_rc.xls"
erase "$path_out/children_nutri_ecu_z_rc.dta"	

	
********************************************************************************
**# Step 1.1b BMI-for-age for individuals 61-228 months
********************************************************************************

use "$path_in/ecv6r_personas.dta", clear


*** Generate individual unique id variable required for data merging:
tostring persona, replace
forvalues i=1(1)9 {
replace persona="0`i'" if persona=="`i'"
}
gen ind_id = identif_hog + persona  
label var  ind_id "Individual ID"


duplicates report ind_id	
	//No duplicates	

	
**Compute age in months for all members	
foreach var in ps80b ps80c {
replace `var'=. if `var'<0
}

gen am = (ps81c - ps80c)*12
	//ps81c: measurement year; ps80c: birth year
replace am = am + (ps81b - ps80b) if ps81b>=ps80b
	//ps81b: measurement month; ps80b: birth month
replace am = (am - 12) + (ps81b - 1) + (13 - ps80b) if ps81b<ps80b 
replace am = edad*12 if ps79==1 & ps80b==.  
label var am "Age in months for all members"	


**Childen up to 60 months	
gen age_months = pd03b if edad==0
replace age_months = pd03b+12 if edad==1
replace age_months = pd03b+24 if edad==2
replace age_months = pd03b+36 if edad==3
replace age_months = pd03b+48 if edad==4
replace age_months = 60 if am==60 & age_months==.
tab age_months, miss
count if age_months <=60
	//Count children under 5 years: 11,598 children



	//Merge information from individuals 0-60 months and older than 60 months	
count if age_months ==. & am!=.	
replace age_months = am if age_months==.
label var age_months "Age in months for all members"
tab age_months, miss	
	
	
gen youth_id = (age_months>=61 & age_months<229)		
keep if youth_id==1 
count
	//32,984 individuals 61-228 age in months
	

*** Variable: SEX ***
tab sexo, miss 
clonevar sex = sexo


*** Variable: AGE IN MONTHS ***	
sum age_months
gen str6 ageunit="months"				
lab var ageunit "months"


*** Variable: BODY WEIGHT (KILOGRAMS) ***
gen weight      = (ps82+ps82a+ps82b)/3 if ps82b<.
replace weight  = (ps82+ps82a)/2 if ps82b==.
sum weight 

*** Variable: HEIGHT (CENTIMETERS)
gen height      = (ps84+ps84a+ps84b)/3 if ps84b<. 
replace height  = (ps84+ps84a)/2 if ps84b==.
replace height  = . if ps84<0 | ps84a<0 | ps84b<0
sum height


*** Variable: OEDEMA
gen oedema = "n"  


*** Variable: SAMPLING WEIGHT ***
gen sw = fexp


*** z-scores ***	
adopath + "$path_ado/who2007_stata"		
gen str100 reflib = "$path_ado/who2007_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "youth_nutri_ecu" 
lab var datalab "Working file"


who2007 reflib datalib datalab sex age_month ageunit weight height oedema sw


use "$path_out/youth_nutri_ecu_z.dta", clear 

		
gen	z_bmi = _zbfa
replace z_bmi = . if _fbfa==1 
lab var z_bmi "z-score bmi-for-age WHO"


*** Standard MPI indicator ***
gen	low_bmiage = (z_bmi < -2.0) 
replace low_bmiage = . if z_bmi==.
lab var low_bmiage "Adolescent have low bmi"


*** Destitution indicator ***
gen	low_bmiage_u = (z_bmi < -3.0)
replace low_bmiage_u = . if z_bmi==.
lab var low_bmiage_u "DST: Adolescent have low bmi"

tab low_bmiage, miss
tab low_bmiage_u, miss
	

	//Retain relevant variables:	
keep ind_id age_months youth_id low_bmiage*
order ind_id age_months youth_id low_bmiage*
 
sort ind_id


	/*Append the nutrition information of children above 5 years with 
	children under 5 */	
append using "$path_out/ECU13-14_child.dta"


	//Check appended information
tab age_months, miss
tab stunting if age_months<= 60, miss 
tab low_bmiage if age_months>60 & age_months<229, miss 

	
	//Save a temp file for merging later:
save "$path_out/ECU13-14_children.dta", replace


	//Erase files from folder:
erase "$path_out/youth_nutri_ecu_z.xls"
erase "$path_out/youth_nutri_ecu_prev.xls"
erase "$path_out/youth_nutri_ecu_z.dta"
erase "$path_out/ECU13-14_child.dta"


********************************************************************************
**# Step 1.1c BMI for all individuals
********************************************************************************
use "$path_in/ecv6r_personas.dta", clear


*** Generate individual unique id variable required for data merging:
tostring persona, replace
forvalues i=1(1)9 {
replace persona="0`i'" if persona=="`i'"
}
gen ind_id = identif_hog + persona  
label var  ind_id "Individual ID"


duplicates report ind_id	
	//No duplicates	

	
**Compute age in months for all members	
foreach var in ps80b ps80c {
replace `var'=. if `var'<0
}

gen am = (ps81c - ps80c)*12
	//ps81c: measurement year; ps80c: birth year
replace am = am + (ps81b - ps80b) if ps81b>=ps80b
	//ps81b: measurement month; ps80b: birth month
replace am = (am - 12) + (ps81b - 1) + (13 - ps80b) if ps81b<ps80b 
replace am = edad*12 if ps79==1 & ps80b==. 
label var am "Age in months for all members"	
sum am



*** Variable: HEIGHT (CENTIMETERS) ***
gen rtalla = (ps84+ps84a+ps84b)/3 if ps84b<. 
replace rtalla = (ps84+ps84a)/2 if ps84b==. 
replace rtalla = . if ps84<0 | ps84a<0 | ps84b<0


*** Variable: BODY WEIGHT (KILOGRAMS) ***
gen rpeso = (ps82+ps82a+ps82b)/3 if ps82b<. 
replace rpeso = (ps82+ps82a)/2 if ps82b==. 
  
  
*** Variable: BMI MEASURE *** 
gen bmi = rpeso/((rtalla/100)^2)
tab edad if bmi!=., miss
lab var bmi "BMI"


gen low_bmi = (bmi<18.5)
replace low_bmi=. if bmi==.
lab var low_bmi "BMI <18.5"
lab define lab_low_bmi 1 "bmi<18.5" 0 "bmi>=18.5"
lab values low_bmi lab_low_bmi
tab low_bmi, miss


gen low_bmi_u = (bmi<17)
replace low_bmi_u = . if bmi==. 
lab var low_bmi_u "BMI <17"
lab define lab_low_bmi_u 1 "bmi<17" 0 "bmi>=17"
lab values low_bmi_u lab_low_bmi_u
tab low_bmi_u, miss


	//Retain relevant variables:	
keep ind_id am bmi low_bmi*
order ind_id am bmi low_bmi*
 
sort ind_id


	//Merge nutrition information from individuals under 20 years
merge 1:1 ind_id using "$path_out/ECU13-14_children.dta"

drop _merge

	
	/*Save a temp file that contains nutrition information for all age group 
	for merging later */	
save "$path_out/ECU13-14_nutri.dta", replace

********************************************************************************
**# Step 1.2  HOUSEHOLD LEVEL INFORMATION
********************************************************************************

use "$path_in/ecv6r_equipamiento.dta", clear
keep  identif_hog eq00 eqbien  eq01
keep if eq00== 4 | eq00== 9 | eq00==10 | eq00==11 | eq00==19 | eq00==25 | ///
		eq00==26 | eq00==28 | eq00==29 | eq00==30 | eq00==33 | eq00==34 | ///
		eq00==35 

replace eqbien = "eqpsonido"     if eq00==11
replace eqbien = "telfijo"       if eq00==19
replace eqbien = "grabadora"     if eq00==25
replace eqbien = "TV_bn"         if eq00==28
replace eqbien = "TV_plas_lcd"   if eq00==29
replace eqbien = "TV_color"      if eq00==30
replace eqbien = "carro"         if eq00==33
replace eqbien = "moto"          if eq00==34
replace eqbien = "bicicleta"     if eq00==4
replace eqbien = "refrigeradora" if eq00==26
replace eqbien = "laptop"  	     if eq00==9
replace eqbien = "computer"  	 if eq00==10
replace eqbien = "terrenos"  	 if eq00==35
drop eq00
rename eq01 _
reshape wide _, i(identif_hog) j(eqbien) string
sort identif_hog


*** Dwelling database 
***********************************
merge 1:1 identif_hog using "$path_in/ecv6r_vivienda.dta"
sort identif_hog
drop _merge

save "$path_out/ECU13-14_hh1.dta", replace

 
use "$path_in/ecv6r_agro1.dta", clear
rename _all, lower
sort identif_hog

save "$path_out/ECU13-14_hh2.dta", replace


use "$path_in/ecv6r_agro_parte_e_v_f_g.dta", clear
rename _all, lower	
clonevar truck = ff0602
replace truck = 0 if ff0602==2

keep identif_hog region area_5000 regional ciudad zona sector ///
	 vivienda hogar ff01 ff0602 truck

sort identif_hog

save "$path_out/ECU13-14_hh3.dta", replace


********************************************************************************
**# Step 1.3  HOUSEHOLD MEMBER'S INFORMATION 
********************************************************************************

use "$path_in/ecv6r_personas.dta", clear


*** Generate a household unique key variable at the household level using: 
tostring persona, replace
forvalues i=1(1)9 {
replace persona="0`i'" if persona=="`i'"
}
gen ind_id = identif_hog+persona  
label var  ind_id "Individual ID"

gen hh_id = identif_hog
label var hh_id "Household ID"


sort hh_id ind_id



********************************************************************************
**# 1.4 DATA MERGING
********************************************************************************



*** Merging Nutrition Data
*************************************************
merge 1:1 ind_id using "$path_out/ECU13-14_nutri.dta"
drop _merge
erase "$path_out/ECU13-14_nutri.dta"
sort identif_hog


*** Merging Household Data  
*****************************************
merge m:1 identif_hog using "$path_out/ECU13-14_hh1.dta"
drop _merge
erase "$path_out/ECU13-14_hh1.dta"

sort identif_hog
merge m:1 identif_hog using "$path_out/ECU13-14_hh2.dta"
drop _merge
erase "$path_out/ECU13-14_hh2.dta"

sort identif_hog
merge m:1 identif_hog using "$path_out/ECU13-14_hh3.dta"
drop _merge
erase "$path_out/ECU13-14_hh3.dta"


sort ind_id


********************************************************************************
**# 1.5 RENAMING DEMOGRAPHIC VARIABLES ***
********************************************************************************

//Sample weight
desc fexp
clonevar weight = fexp


//Area: urban or rural	
codebook area_5000, tab (5)		
clonevar area = area_5000  
replace area=0 if area==2  
label define lab_area 1 "urban" 0 "rural"
label values area lab_area
label var area "Area: urban-rural"
tab area area_5000, miss


//Relationship to the head of household 
clonevar relationship = pd04 
codebook relationship, tab (20)
recode relationship (1=1)(2=2)(3=3)(4/10=4) (12/13=5) (11=6) (99=.)
label define lab_rel 1"head" 2"spouse" 3"child" 4"extended family" ///
				     5"not related" 6"maid"
label values relationship lab_rel
label var relationship "Relationship to the head of household"
tab pd04 relationship, miss


//Sex of household member	
codebook sexo, tab (5)
clonevar sex = sexo  


//Household headship
bys	hh_id: egen missing_hhead = min(relationship)
tab missing_hhead,m 
gen household_head=.
replace household_head=1 if relationship==1 & sex==1 
replace household_head=2 if relationship==1 & sex==2
bysort hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
label define head 1"male-headed" 2"female-headed"
label values headship head
label var headship "Household headship"
tab headship, miss


//Age of household member
codebook edad, tab (999)
clonevar age = edad  


//Age group 
recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+"), gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"



//Marital status of household member
gen marital = 1 if pd19==6
replace marital = 2 if pd19<=2
replace marital = 3 if pd19==5
replace marital = 4 if pd19==4
replace marital = 5 if pd19==3 	
label define lab_mar 1"never married" 2"currently married" ///
					 3"widowed" 4"divorced" 5"not living together"
label values marital lab_mar	
label var marital "Marital status of household member"
tab pd19 marital, miss


//Total number of de jure hh members in the household
gen member = 1
bysort hh_id: egen hhsize = sum(member)
label var hhsize "Household size"
tab hhsize, miss
drop member



//Subnational region
	/*The sample is representative at the national, urban and rural levels, 4
natural regions, 24 provinces, 9 planning areas and 4 cities self-represented 
(Quito, Guayaquil, Cuenca and Machala) */
rename region region_natural

codebook provincia, tab (99)
gen region = provincia
label define reg_lab  ///
1"Azuay" 2"Bolívar" 3"Cañar" ///
4"Carchi" 5"Cotopaxi" 6"Chimborazo" ///
7"El Oro" 8"Esmeraldas" 9"Guayas" ///
10"Imbabura" 11"Loja" 12"Los Ríos" ///
13"Manabí" 14"Morona Santiago" 15"Napo" ///
16 "Pastaza" 17 "Pichincha" 18 "Tungurahua" ///
19 "Zamora Chinchipe" 20 "Galápagos" 21 "Sucumbios" ///
22 "Orellana" 23 "Santo Domingo" 24 "Santa Elena" 
label values region reg_lab 
lab var region "Region for subnational decomposition"
tab provincia region, m


clonevar region_01 = region
lab var region_01 "COT:Region for subnational decomposition"


********************************************************************************
***  Step 2 Data preparation  ***
***  Standardization of the 10 Global MPI indicators 
***  Identification of non-deprived & deprived individuals  
********************************************************************************


********************************************************************************
*** Step 2.1 Years of Schooling ***
********************************************************************************

	// In Ecuador, the education system: 
	* Entrance age of primary: 6 years
	* Duration of primary: 6 years
	* Entrance age of lower secondary: 12 years
	* Durantion lower secondary: 3 years
	* Entrance age high secondary: 15 years
	* Duration high secondary: 3 years 

tab pe48 pe47, miss

clonevar edulevel = pe47
clonevar eduhighyear = pe48
tab eduhighyear edulevel,m

gen eduyears = .
replace eduyears = 0 if edulevel<=3
replace eduyears = eduhighyear - 1 if edulevel==5
replace eduyears = eduhighyear if edulevel==6
replace eduyears = eduhighyear + 9  if edulevel==7 
	//Level following basic education
replace eduyears = eduhighyear + 6 if edulevel==8 
	//Level following primary education (probably former system)
replace eduyears = eduhighyear + 12 if edulevel==9 | edulevel==10 
	//University
replace eduyears = eduhighyear + 17 if edulevel==11 
	//Post-graduation
replace eduyears = 0 if eduyears==-1
lab var eduyears "Total number of years of education accomplished"


*** Checking for further inconsistencies 
replace eduyears = . if age<=eduyears & age>0 
	/*There are cases in which the years of schooling are greater than the 
	age of the individual. This is clearly a mistake in the data.*/
replace eduyears = 0 if age< 10
replace eduyears = 0 if (age==10 | age==11) & eduyears < 6 
	/*The variable "eduyears" was replaced with a '0' for ineligible household 
	members, i.e.: those who have not completed 6 years of schooling following 
	their starting school age */
tab eduyears, miss


	/*A control variable is created on whether there is information on 
	years of education for at least 2/3 of the eligible household members*/	
gen temp = 1 if eduyears!=. & age>=12 & age!=.
replace temp = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp = 1 if age==11 & eduyears>=6 & eduyears<.
bysort	hh_id: egen no_missing_edu = sum(temp)	
	//Total eligible household members with no missing years of education
gen temp2 = 1 if age>=12 & age!=.
replace temp2 = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp2 = 1 if age==11 & eduyears>=6 & eduyears<.
bysort hh_id: egen hhs = sum(temp2)
	/*Total number of eligible household members who should have information 
	on years of education */
replace no_missing_edu = no_missing_edu/hhs
replace no_missing_edu = (no_missing_edu>=2/3)
	/*Identify whether there is information on years of education for at 
	least 2/3 of the eligible household members */
tab no_missing_edu, miss
	//The value for 0 (missing) is 0% 
label var no_missing_edu "No missing edu for at least 2/3 of the eligible HH members"
drop temp temp2 hhs


*** Standard MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed six years of schooling. */
*******************************************************************  
gen	 years_edu6 = (eduyears>=6)
replace years_edu6 = . if eduyears==.
bysort hh_id: egen hh_years_edu6_1 = max(years_edu6)
gen	hh_years_edu6 = (hh_years_edu6_1==1)
replace hh_years_edu6 = . if hh_years_edu6_1==.
replace hh_years_edu6 = . if hh_years_edu6==0 & no_missing_edu==0 
lab var hh_years_edu6 "Household has at least one member with 6 years of edu"
tab hh_years_edu6, miss

	
*** Destitution MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed at least one year of schooling. */
*******************************************************************
gen	years_edu1 = (eduyears>=1)
replace years_edu1 = . if eduyears==.
bysort	hh_id: egen hh_years_edu_u = max(years_edu1)
replace hh_years_edu_u = . if hh_years_edu_u==0 & no_missing_edu==0
lab var hh_years_edu_u "Household has at least one member with 1 year of edu"
tab hh_years_edu_u, miss


********************************************************************************
*** Step 2.2 Child School Attendance ***
********************************************************************************

codebook pe18, tab (99)
gen attendance = (pe18<8) if pe18!=.
tab pe18 attendance, miss
replace attendance = 0 if age<5 | age>24 
	//Replace attendance with '0' for individuals who are not of school age 
label define lab_attendance 1 "currently attending" 0 "not currently attending"
label values attendance lab_attendance
lab var attendance "Currently attending school"
tab attendance, miss  


*** Standard MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 8. */ 
******************************************************************* 
gen	child_schoolage = (age>=6 & age<=14)
	/*In Ecuador, the official school entrance age to primary school is 
	6 years. So, age range is 6-14 (=6+8) */ 

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the school age children */
count if child_schoolage==1 & attendance==.
	//How many eligible school aged children are not attending school: 0 children 
gen temp = 1 if child_schoolage==1 & attendance!=.
	/*Generate a variable that captures the number of eligible school aged 
	children who are attending school */
bysort hh_id: egen no_missing_atten = sum(temp)	
	/*Total school age children with no missing information on school 
	attendance */
gen temp2 = 1 if child_schoolage==1	
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are of school age
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)
	/*Identify whether there is missing information on school attendance for 
	more than 2/3 of the school age children */			
tab no_missing_atten, miss
	// No missing value
label var no_missing_atten "No missingattendance for at least 2/3 of the eligible children"		
drop temp temp2 hhs
	
bysort hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
	//It takes value 1 if the household has children in school age
lab var hh_children_schoolage "Household has children in school age"


gen	child_not_atten = (attendance==0) if child_schoolage==1
replace child_not_atten = . if attendance==. & child_schoolage==1
bysort	hh_id: egen any_child_not_atten = max(child_not_atten)
gen	hh_child_atten = (any_child_not_atten==0) 
replace hh_child_atten = . if any_child_not_atten==.
replace hh_child_atten = 1 if hh_children_schoolage==0
replace hh_child_atten = . if hh_child_atten==1 & no_missing_atten==0 
lab var hh_child_atten "Household has all school age children up to class 8 in school"
tab hh_child_atten, miss

	
*** Destitution MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 6. */ 
******************************************************************* 
gen	child_schoolage_6 = (age>=6 & age<=12) 
	/*Note: In Ecuador, the official school entrance age is 6 years.  
	  So, age range for destitution measure is 6-12 (=6+6) */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the children attending school up to 
	class 6 */	
count if child_schoolage_6==1 & attendance==.	
gen temp = 1 if child_schoolage_6==1 & attendance!=.
bysort hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_6==1	
bysort hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)			
tab no_missing_atten_u, miss
label var no_missing_atten_u "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs		
		
bysort	hh_id: egen hh_children_schoolage_6 = sum(child_schoolage_6)
replace hh_children_schoolage_6 = (hh_children_schoolage_6>0) 
lab var hh_children_schoolage_6 "Household has children in school age (6 years of school)"

gen	child_atten_6 = (attendance==1) if child_schoolage_6==1
replace child_atten_6 = . if attendance==. & child_schoolage_6==1
bysort	hh_id: egen any_child_atten_6 = max(child_atten_6)
gen	hh_child_atten_u = (any_child_atten_6==1) 
replace hh_child_atten_u = . if any_child_atten_6==.
replace hh_child_atten_u = 1 if hh_children_schoolage_6==0
replace hh_child_atten_u = . if hh_child_atten_u==0 & no_missing_atten_u==0 
lab var hh_child_atten_u "Household has at least one school age children up to class 6 in school"
tab hh_child_atten_u, miss


********************************************************************************
*** Step 2.3 Nutrition ***
********************************************************************************
	/*Note: Ecuador ECV 2013-14: Anthropometric information were recorded for 
	all individuals aged 0-98 years. This departs from the usual DHS surveys 
	that tend to collect anthropometric data only from children under 5 and 
	adults between the age group of 15-49/15-59 years. In the case of Ecuador, 
	we make use of the anthropometric data for individuals aged 0 - 70 years 
	only, even if the data is available for up to the age of 98 years. This 
	is in line with the global MPI requirement. The age cut-off is captured in 
	the final indicator through the eligibility criteria. */ 

	
***As a first step, construct the eligibility criteria	
*** No Eligible Women, Men or Children for Nutrition
***********************************************	
gen nutri_eligible = age<=70
bysort	hh_id: egen n_nutri_eligible = sum(nutri_eligible) 	
gen	no_nutri_eligible = (n_nutri_eligible==0) 	
lab var no_nutri_eligible "Household has no eligible women, men, or children"
tab no_nutri_eligible, miss	

drop n_nutri_eligible


count if age<=70
count if am<=840
count if am==. & age<=70


********************************************************************************
*** Step 2.3a Adult Nutrition ***
********************************************************************************


*** Standard MPI: BMI Indicator for Adults 20 years and older ***
******************************************************************* 
tab low_bmi, miss
count if age>=20 & age_month==.
	//62,486 adults 20 years and older	
tab low_bmi if age>=20 & age_month==., miss	
	/* 2,874 (4.60%) adults 20 years and older have 
	missing value for the low_bmi indicator. */
gen low_bmi_20 = low_bmi if age>=20 & age_month==.
	/* In the context of ECV 2013-14, we focus on BMI measure 
	for individuals aged 20 years and older because BMI-for-age 
	is applied for individuals above 5 years and under 20 years. */
	
bysort hh_id: egen temp = max(low_bmi_20)
tab temp, miss 
gen hh_no_low_bmi = (temp==0) 
	/* Under this section, households take a value of '1' if 
	no adults in the household has low bmi. */
replace hh_no_low_bmi = . if temp==.
	/* Under this section, households take a value 
	of '.' if there is no information from adults. */
drop temp
lab var hh_no_low_bmi "Household has no adult with low BMI"
tab hh_no_low_bmi, miss
	//Figures are based on information from adults aged 20 years and older.


*** Destitute MPI: BMI Indicator for Adults 20 years and older ***
******************************************************************* 
tab low_bmi_u, miss
tab low_bmi_u if age>=20 & age_month==., miss
	/* 2,874 (4.60%) adults 20 years and older have 
	missing value for the low_bmi indicator. */
gen low_bmi_20_u = low_bmi_u if age>=20 & age_month==.
	/* In the context of ECV 2013-14, we focus on BMI measure 
	for individuals aged 20 years and older because BMI-for-age 
	is applied for individuals 61-228 age in months. */
	
bysort hh_id: egen temp = max(low_bmi_20_u)
tab temp, miss 
gen hh_no_low_bmi_u = (temp==0) 
	/* Under this section, households take a value of '1' if no adults in the 
	household has destitute bmi. */
replace hh_no_low_bmi_u = . if temp==.
	/* Under this section, households take a value of '.' if there is no 
	information from adults.*/		
drop temp
lab var hh_no_low_bmi_u "Household has no adult with low BMI (<17)"	
tab hh_no_low_bmi_u, miss
	/* Destitute figures are based on information from 
	adults aged 20 years and older. */


********************************************************************************
*** Step 2.3b Nutrition for individuals 61-228 age in months ***
********************************************************************************


*** Standard MPI: BMI-for-age for individuals 61-228 age in months ***
******************************************************************* 	

count if age_month>60 & age_month<229
count if age_month>60 & age_month<229 & youth_id==1
	//32,984 individuals individuals 61-228 age in months
	
tab low_bmiage if youth==1, miss	

bysort hh_id: egen temp = max(low_bmiage)  
tab temp, miss
gen	hh_no_low_bmiage = (temp==0) 
	/*Takes value 1 if no individuals 61-228 age in months in the 
	household has low bmi-for-age*/
replace hh_no_low_bmiage = . if temp==.
drop temp

lab var hh_no_low_bmiage "Household has no adult with low BMI-for-age"
tab hh_no_low_bmiage, miss
	//Figures are based on information from individuals 61-228 age in months.
	

*** Destitute MPI: BMI-for-age for individuals 61-228 age in months ***
******************************************************************* 

tab low_bmiage_u if youth_id==1, miss	

bys	hh_id: egen temp = max(low_bmiage_u)  
tab temp, miss
gen	hh_no_low_bmiage_u = (temp==0) 
	/*The destitute measure takes a value of '1' if no 
	individuals 61-228 age in months in the household 
	has low bmi-for-age. */
replace hh_no_low_bmiage_u = . if temp==.
drop temp
lab var hh_no_low_bmiage_u "Household has no adult with low BMI-for-age(-3sd)"
tab hh_no_low_bmiage_u, miss
	/*Destitute figures are based on information 
	from individuals 61-228 age in months. */

	
********************************************************************************
*** Step 2.3c Child Nutrition ***
********************************************************************************

***As a first step, construct the eligibility criteria for children
*** No Eligible Children for Nutrition
***********************************************	
gen child_eligible = age_months<=60
bysort	hh_id: egen n_child_eligible = sum(child_eligible) 	
gen	no_child_eligible = (n_child_eligible==0) 	
lab var no_child_eligible "HH has no eligible children <=60 months"
tab no_child_eligible, miss	

drop n_child_eligible


*** Child Underweight Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(underweight)
gen	hh_no_underweight = (temp==0) 
	//Takes value 1 if no child in the hh is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_underweight "Household has no child underweight - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(underweight_u)
gen	hh_no_underweight_u = (temp==0) 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
lab var hh_no_underweight_u "Destitute: Household has no child underweight"
drop temp


*** Child Stunting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(stunting)
gen	hh_no_stunting = (temp==0) 
	//Takes value 1 if no child in the hh is stunted
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_stunting "Household has no child stunted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(stunting_u)
gen	hh_no_stunting_u = (temp==0) 
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
lab var hh_no_stunting_u "Destitute: Household has no child stunted"
drop temp


*** Child Wasting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(wasting)
gen	hh_no_wasting = (temp==0) 
	//Takes value 1 if no child in the hh is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_wasting "Household has no child wasted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(wasting_u)
gen	hh_no_wasting_u = (temp==0) 
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
lab var hh_no_wasting_u "Destitute: Household has no child wasted"
drop temp


*** Child Either Underweight or Stunted Indicator ***
************************************************************************

*** Standard MPI ***
gen uw_st = 1 if stunting==1 | underweight==1
	//Takes value 1 if child in the hh is stunted or underweight 
replace uw_st = 0 if stunting==0 & underweight==0
	//Takes value 0 if child in the hh is not stunted and not underweight 
replace uw_st = . if stunting==. & underweight==.

bysort hh_id: egen temp = max(uw_st)
gen hh_no_uw_st = (temp==0) 
replace hh_no_uw_st = . if temp==.
drop temp
replace hh_no_uw_st = 1 if no_child_eligible==1
	//Households with no eligible children will receive a value of 1 
lab var hh_no_uw_st "HH has no child underweight or stunted"


*** Destitution MPI  ***
gen uw_st_u = 1 if stunting_u==1 | underweight_u==1
replace uw_st_u = 0 if stunting_u==0 & underweight_u==0
replace uw_st_u = . if stunting_u==. & underweight_u==.

bysort hh_id: egen temp = max(uw_st_u)
gen hh_no_uw_st_u = (temp==0) 
replace hh_no_uw_st_u = . if temp==.
drop temp
replace hh_no_uw_st_u = 1 if no_child_eligible==1
lab var hh_no_uw_st_u "DST: HH has no child underweight or stunted"


********************************************************************************
*** Step 2.3d Household Nutrition Indicator ***
********************************************************************************


*** Standard MPI ***
/* Members of the household are considered deprived if the household has a 
child under 5 whose height-for-age or weight-for-age is under two standard 
deviation below the median, or has teenager with BMI-for-age that is under two 
standard deviation below the median, or has adults with BMI threshold that is 
below 18.5 kg/m2. Households that have no eligible adult AND no eligible 
children are considered non-deprived. The indicator takes a value of missing 
only if all eligible adults and eligible children have missing information 
in their respective nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st = 1
replace hh_nutrition_uw_st = 0 if hh_no_low_bmi==0 | hh_no_low_bmiage==0 | hh_no_uw_st==0
replace hh_nutrition_uw_st = . if hh_no_low_bmi==. & hh_no_low_bmiage==. & hh_no_uw_st==.	
replace hh_nutrition_uw_st = 1 if no_nutri_eligible==1
lab value hh_nutrition_uw_st lab_nutri 
lab var hh_nutrition_uw_st "HH has no individuals undernourished"
tab hh_nutrition_uw_st, miss


*** Destitution MPI ***
/* Members of the household are considered deprived if the household has a 
child under 5 whose height-for-age or weight-for-age is under three standard 
deviation below the median, or has teenager with BMI-for-age that is under three 
standard deviation below the median, or has adults with BMI threshold that is 
below 17.0 kg/m2. Households that have no eligible adult AND no eligible 
children are considered non-deprived. The indicator takes a value of missing 
only if all eligible adults and eligible children have missing information 
in their respective nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st_u = 1
replace hh_nutrition_uw_st_u = 0 if hh_no_low_bmi_u==0 | hh_no_low_bmiage_u==0 | hh_no_uw_st_u==0
replace hh_nutrition_uw_st_u = . if hh_no_low_bmi_u==. & hh_no_low_bmiage_u==. & hh_no_uw_st_u==.
replace hh_nutrition_uw_st_u = 1 if no_nutri_eligible==1	 	
lab var hh_nutrition_uw_st_u "DST: HH has no individuals undernourished"
lab value hh_nutrition_uw_st_u lab_nutri 
tab hh_nutrition_uw_st_u, miss


********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************

	/*As a first step, construct the eligibility criteria for child mortality.
	In the ECV 2013-14, child_mortality, that is, the number of sons and 
	daughters who have died was collected from all women aged 12-49 years.  */
	
*** No Eligible Women 12-49 years
*****************************************
gen fem_eligible = (pf01==1)
bys hh_id: egen hh_n_fem_eligible = sum(fem_eligible) 	
	//Number of eligible women for interview in the hh
gen no_fem_eligible = (hh_n_fem_eligible==0) 									
	//Takes value 1 if the household had no eligible females for an interview
lab var no_fem_eligible "Household has no eligible women"
tab no_fem_eligible, miss
	
gen f_child_mortality = pf15-pf17 
replace f_child_mortality = 0 if pf05==0 | pf15==0 
tab f_child_mortality, miss

bys hh_id: egen child_mortality = max(f_child_mortality)
lab var child_mortality "Total child mortality within household reported by women"
tab child_mortality, miss

gen hh_mortality = (child_mortality==0) 
replace hh_mortality = . if child_mortality==.
replace hh_mortality = 1 if no_fem_eligible==1
lab var hh_mortality "Household had no child mortality"
tab hh_mortality, miss


*** Child Mortality: Standard MPI *** 
/*Deprived if any children under 18 died in the household 
in the last 5 years from the survey year */
************************************************************************
	/*In the case of Ecuador, there is no birth history data. There is 
	information on the date of the last birth (if after 1999) and the date of 
	the death of that child. So, assuming that any child died in the 
	last 5 years would be the last child born, it would be possible to build a 
	minimum bound for the indicator hh_child_mortality_5y. However, that is 
	a very strong assumption. Thus, we are not able to construct the indicator 
	on child mortality that occurred in the last 5 years. We construct the final 
	indicator using all reported mortality */

	
gen childu18_mortality_5y = .
label var childu18_mortality_5y "Under 18 child mortality within household past 5 years reported by women"

		
gen hh_mortality_u18_5y = .	
lab var hh_mortality_u18_5y "Household had no under 18 child mortality in the last 5 years"


	
*** Destitution MPI *** 
*** (same as standard MPI - all reported death due to lack of birth history data) ***
************************************************************************
gen hh_mortality_u = hh_mortality	
lab var hh_mortality_u "Household had no child mortality"


********************************************************************************
*** Step 2.5 Electricity ***
********************************************************************************


*** Standard MPI ***
/*Members of the household are considered 
deprived if the household has no electricity */
***************************************************
gen electricity = 1 if vi26==1 | vi26==2 | vi26==3
replace electricity = 0 if vi26==4 | vi26==5
label var electricity "Electricity"
tab vi26 electricity, miss


*** Destitution MPI  ***
*** (same as standard MPI) ***
***************************************************
gen electricity_u = electricity
label var electricity "Electricity"


********************************************************************************
*** Step 2.6 Sanitation ***
********************************************************************************

/*
Improved sanitation facilities include flush or pour flush toilets to sewer 
systems, septic tanks or pit latrines, ventilated improved pit latrines, pit 
latrines with a slab, and composting toilets. These facilities are only 
considered improved if it is private, that is, it is not shared with other 
households.
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-02-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

clonevar toilet = vi14

/*
1  inodoro y alcantarillado - ND
2  inodoro y pozo s�ptico - ND
3  inodoro y pozo ciego - ND
4  letrina - D
5  no tiene - D
*/
					   
gen shared_toilet = 0 if vi15b>0 & vi15b<.
replace shared_toilet = 1 if vi15b==0


*** Standard MPI ***
/* Members of the household are considered deprived if the household's 
sanitation facility is not improved (according to the SDG guideline) 
or it is improved but shared with other households. */
********************************************************************
codebook toilet, tab (99)
	/* The ECV 2013-14 recorded 5 categories of toilet facility in the 
	data. The country report (p.24, footnote 24) specifies that toilets with 
	sewer, septic tank and other flush systems as improved. This suggest that 
	all other categories are non-improved, including the category identified as 
	latrine. Please note that vi15b represents the number of toilets that has 
	been exclusively used by the household. If the household has at least one 
	private toilet, we then consider this as non-deprived. */

gen toilet_mdg = 1 if vi14<4
	//Household is assigned a value of '1' if it uses improved sanitation 
	
replace toilet_mdg = 0 if vi14==4 | vi14==5
 	//Household is assigned a value of '0' if it uses unimproved sanitation 
	
replace toilet_mdg = 0 if vi15b==0 
 	/*Household is assigned a value of '0' if it does not have a private 
	facility */
	
lab var toilet_mdg "Household has improved sanitation with MDG Standards"
tab toilet toilet_mdg, miss


*** Destitution MPI ***
/*Members of the household are considered deprived if household practises 
open defecation or uses other unidentifiable sanitation practises */
********************************************************************
gen toilet_u = 1
replace toilet_u = 0 if vi14==5 
	/*Household is assigned a value of '0' if it practises open defecation or 
	others */
lab var toilet_u "Household does not practise open defecation or others"
tab toilet toilet_u, miss


********************************************************************************
*** Step 2.7 Drinking Water  ***
********************************************************************************

/*
Improved drinking water sources include the following: piped water into 
dwelling, yard or plot; public taps or standpipes; boreholes or tubewells; 
protected dug wells; protected springs; packaged water; delivered water and 
rainwater which is located on premises or is less than a 30-minute walk from 
home roundtrip. 
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-01-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

clonevar water = vi17

gen timeminutes= vi21a * 60  
egen timetowater=rowtotal(timeminutes vi21b), miss 
codebook timetowater, tab (999)
tab water if timetowater>=300 &  timetowater!=.	
	/*Note that 23 individuals live in households that reported more than 5 
	hours distance to obtain water. Since these households 
	are getting their drinking water from deprived sources, we did not replace 
	these potentially unreasonable values as missing.*/
	

*** Standard MPI ***
/* Members of the household are considered deprived if the household 
does not have access to improved drinking water (according to the SDG 
guideline) or safe drinking water is at least a 30-minute walk from 
home, roundtrip */
********************************************************************
codebook water, tab (99)
	/* From page 23 of the country report, it may be inferred that 
	individuals in the higher income quintile are more likely to get 
	their drinking water from piped sources (public network and other pipes). 
	This suggest that all other categories including water drawn from well 
	is considered as unimporved. We have followed this standard for the 
	global MPI. 
	
1  red p�blica	ND
2  otra fuente por tuber�a	ND
3  carro repartidor / triciclo	D
4  pozo	D
5  r�o vertiente o acequia	D
6  otro, cual	D
*/

gen	water_mdg = 1 if water==1 | water==2 | water==3 	
replace water_mdg = 0 if water==4 | water==5 | water==6 
ta water water_mdg,m
replace water_mdg = 0 if timetowater>=30 & timetowater!=.
lab var water_mdg "HH has safe drinking water, under 30 mins dist"
ta water_mdg,m


*** Destitution MPI ***
/* Members of the household is identified as destitute if household 
does not have access to safe drinking water, or safe water is more 
than 45 minute walk from home, round trip.*/
********************************************************************
gen	water_u = 1 if water==1 | water==2 | water==3 	
replace water_u = 0 if water==4 | water==5 | water==6 
replace water_u = 0 if timetowater>45 & timetowater!=.
lab var water_u "DST: HH has safe drinking water, 0-45 mins dist"
ta water_u, miss


********************************************************************************
*** Step 2.8 Housing ***
********************************************************************************

/* Members of the household are considered deprived if the household 
has a dirt, sand or dung floor */
lookfor piso
clonevar floor = vi05
codebook floor, tab(99)
gen floor_imp = 1 if vi05==1 | vi05==2 | vi05==3 | vi05==4 | vi05==5 | vi05==6
replace floor_imp = 0 if vi05==7 | vi05==8
lab var floor_imp "Household has floor that it is not earth/sand/dung"
tab floor floor_imp, miss


/* Members of the household are considered deprived if the household has walls 
made of natural or rudimentary materials.

In the case of Ecuador, we classified "adobe" and "wood" as improved, and cane/coated reed as unimproved. */
lookfor pared
clonevar wall = vi04
codebook wall, tab(99)	
gen wall_imp = 1 if wall<=5
replace wall_imp = 0 if wall>=6 & wall<=8 
lab var wall_imp "Household has wall that it is not of low quality materials"
tab wall wall_imp, miss	

	
/* Members of the household are considered deprived if the household has walls 
made of natural or rudimentary materials */
lookfor techo
clonevar roof = vi03
codebook roof, tab(99)	
gen roof_imp = 1 if roof<=5
replace roof_imp = 0 if roof>=6 & roof<=7
lab var roof_imp "Household has roof that it is not of low quality materials"
tab roof roof_imp, miss


*** Standard MPI ***
/* Members of the household is deprived in housing if the roof, 
floor OR walls are constructed from low quality materials.*/
**************************************************************
gen housing_1 = 1
replace housing_1 = 0 if floor_imp==0 | wall_imp==0 | roof_imp==0
replace housing_1 = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_1 "Household has roof, floor & walls that it is not low quality material"
tab housing_1, miss


*** Destitution MPI ***
/* Members of the household is deprived in housing if two out 
of three components (roof and walls; OR floor and walls; OR 
roof and floor) the are constructed from low quality materials. */
**************************************************************
gen housing_u = 1
replace housing_u = 0 if (floor_imp==0 & wall_imp==0 & roof_imp==1) | ///
						 (floor_imp==0 & wall_imp==1 & roof_imp==0) | ///
						 (floor_imp==1 & wall_imp==0 & roof_imp==0) | ///
						 (floor_imp==0 & wall_imp==0 & roof_imp==0)
replace housing_u = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_u "Household has one of three aspects(either roof,floor/walls) that is not low quality material"


********************************************************************************
*** Step 2.9 Cooking Fuel ***
********************************************************************************

/*
Solid fuel are solid materials burned as fuels, which includes coal as well as 
solid biomass fuels (wood, animal dung, crop wastes and charcoal). 

Source: 
https://apps.who.int/iris/bitstream/handle/10665/141496/9789241548885_eng.pdf
*/

lookfor cocina
codebook vi11 vi12 vi13
clonevar cookingfuel = vi13
replace cookingfuel = 5 if vi11==5
label define cookingfuel 1 "gas" 2 "electricity" 3 "wood/charcoal"  ///
						 4 "other" 5 "no cooking at home"
label values cookingfuel cookingfuel
tab vi13 cookingfuel, miss


*** Standard MPI ***
/* Members of the household are considered deprived if the 
household uses solid fuels and solid biomass fuels for cooking. */
*****************************************************************
gen cooking_mdg = 0 if cookingfuel==3 
replace cooking_mdg = 1 if cookingfuel==1 | cookingfuel==2 | ///
						   cookingfuel==4 | cookingfuel==5 
replace cooking_mdg = . if cookingfuel==99
lab var cooking_mdg "Househod has cooking fuel according to MDG standards"			 
tab cookingfuel cooking_mdg, miss	


*** Destitution MPI ***
*** (same as standard MPI) ***
****************************************
gen	cooking_u = cooking_mdg
lab var cooking_u "Household uses clean fuels for cooking"


********************************************************************************
*** Step 2.10 Assets ownership ***
********************************************************************************


*Television
gen television=.
replace television=1 if  _TV_bn==1 | _TV_color==1 | _TV_plas_lcd==1
replace television=0 if  _TV_bn==2 & _TV_color==2 & _TV_plas_lcd==2
replace television=. if  _TV_bn==. & _TV_color==. & _TV_plas_lcd==.
gen bw_television   = .
tab television,m


*Radio
gen radio=.
replace radio=1 if _grabadora==1 | _eqpsonido==1
replace radio=0 if _grabadora==2 & _eqpsonido==2
replace radio=. if _grabadora==. & _eqpsonido==.
tab radio,m


*Fix Telephone at home
gen	telephone = 1 if  _telfijo==1
replace telephone = 0 if _telfijo==2


*Mobile phone 
gen hp = 1 if ph09a==1
tab age if ph09a==., miss 
	//persons <12 years were not asked on cell phone ownership 
replace hp = 0 if ph09a==2 | ph09a==.
bysort hh_id: egen mobilephone = max(hp)
tab hp,m

replace telephone=1 if telephone!=1 & mobilephone==1	
tab mobilephone _telfijo if telephone==1,miss
lab var telephone "Household has telephone (landline/mobilephone)"	
tab telephone,m


*Refrigerator
gen  refrigerator     = 1 if  _refrigeradora==1
replace  refrigerator = 0 if  _refrigeradora==2
tab refrigerator,m

*Car
gen  car = 1 if     _carro==1
replace  car = 0 if _carro==2
tab car,m

*Bicycle
gen bicycle     = 1 if  _bicicleta==1
replace bicycle = 0 if  _bicicleta==2


*Motorcycle
gen  motorbike   = 1 if _moto==1
replace motorbike= 0 if _moto==2 
tab motorbike, miss


*Computer
lookfor _comput _lap
gen computer = 1 if _laptop==1 | _computer==1
replace computer=0 if _laptop==2 & _computer==2
tab computer, miss


*Animal cart
gen animal_cart = .


label define lab_assets 1 "Yes" 0 "No"
foreach var in television radio telephone refrigerator car ///
			   bicycle motorbike computer animal_cart {
label values `var' lab_assets			   	
replace `var' = . if `var'==. 
}
	//Missing values replaced

	
	//Label indicators
lab var television "Household has television"
lab var radio "Household has radio"	
lab var telephone "Household has telephone (landline/mobilephone)"	
lab var refrigerator "Household has refrigerator"
lab var car "Household has car"
lab var bicycle "Household has bicycle"	
lab var motorbike "Household has motorbike"
lab var computer "Household has computer"
lab var animal_cart "Household has animal cart"


*** Standard MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own more than one of: radio, TV, telephone, bike, motorbike, 
refrigerator, computer or animal cart and does not own a car or truck.*/
*****************************************************************************
egen n_small_assets2 = rowtotal(television radio telephone refrigerator bicycle motorbike computer animal_cart), missing
lab var n_small_assets2 "Household Number of Small Assets Owned" 
    
gen hh_assets2 = (car==1 | n_small_assets2 > 1) 
replace hh_assets2 = . if car==. & n_small_assets2==.
lab var hh_assets2 "Household Asset Ownership: HH has car or more than 1 small assets incl computer & animal cart"


*** Destitution MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own any assets.*/
*****************************************************************************		
gen	hh_assets2_u = (car==1 | n_small_assets2>0)
replace hh_assets2_u = . if car==. & n_small_assets2==.
lab var hh_assets2_u "Household Asset Ownership: HH has car or at least 1 small assets incl computer & animal cart"


*** Harmonised MPI: ECV 2013/14 - ECU 2018 *** 
	/*The later survey lacked information on radio, bicycle and motorbike. 
	Both surveys lack information on animal cart. As such, the harmonised 
	work excludes these small assets.*/
************************************************************************
gen radio_c = .
lab var radio_c "COT: HH has radio"

gen	motorbike_c = . 
label var motorbike_c "COT: HH has motorbike"

gen	bicycle_c = . 
label var bicycle_c "COT: HH has bicycle"


egen n_small_assets2_c = rowtotal(television telephone refrigerator computer animal_cart), missing
lab var n_small_assets2_c "COT: HH Number of Small Assets Owned" 
    
gen hh_assets2_c = (car==1 | n_small_assets2_c > 1) 
replace hh_assets2_c = . if car==. & n_small_assets2_c==.
lab var hh_assets2_c "COT: HH has car or more than 1 small assets"

gen	hh_assets2_u_c = (car==1 | n_small_assets2_c>0)
replace hh_assets2_u_c = . if car==. & n_small_assets2_c==.
lab var hh_assets2_u_c "COT-DST: HH has car or at least 1 small assets"


********************************************************************************
*** Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************


	//Retain data on sampling design: 
clonevar strata = dominio
clonevar psu = identif_sect 	
codebook strata psu
destring hh_id ind_id psu, replace
desc hh_id ind_id psu strata
label var psu "Primary sampling unit"
label var strata "Sample strata"


	//Retain year, month & date of interview:
desc ps81a ps81b ps81c
clonevar year_interview = ps81c 	
clonevar month_interview = ps81b 
clonevar date_interview = ps81a
 

	//Generate presence of subsample
gen subsample = .


*** Keep main variables require for MPI calculation ***
keep hh_id ind_id subsample strata psu weight sex age hhsize ///
area agec7 agec4 agec2 region headship marital relationship  ///
no_child_eligible no_fem_eligible ///
year_interview month_interview date_interview /// 
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
low_bmi low_bmi_u hh_no_low_bmiage hh_no_low_bmiage_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u child_mortality ///
hh_mortality hh_mortality_u18_5y hh_mortality_u  /// 
electricity electricity_u toilet shared_toilet toilet_mdg toilet_u ///
water timetowater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
region_01 radio_c motorbike_c bicycle_c ///
n_small_assets2_c hh_assets2_c hh_assets2_u_c 

   	 
*** Order file	***
order hh_id ind_id subsample strata psu weight sex age hhsize ///
area agec7 agec4 agec2 region headship marital relationship  ///
no_child_eligible no_fem_eligible ///
year_interview month_interview date_interview /// 
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
low_bmi low_bmi_u hh_no_low_bmiage hh_no_low_bmiage_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u child_mortality ///
hh_mortality hh_mortality_u18_5y hh_mortality_u  /// 
electricity electricity_u toilet shared_toilet toilet_mdg toilet_u ///
water timetowater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
region_01 radio_c motorbike_c bicycle_c ///
n_small_assets2_c hh_assets2_c hh_assets2_u_c    


*** Rename key Global MPI indicators for estimation ***	
recode hh_mortality         (0=1)(1=0) , gen(d_cm)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ)
recode electricity 			(0=1)(1=0) , gen(d_elct)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani)
recode housing_1 			(0=1)(1=0) , gen(d_hsg)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl)
recode hh_assets2 			(0=1)(1=0) , gen(d_asst)
 

*** Rename key global MPI indicators for destitution estimation ***
recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ)
recode electricity_u		(0=1)(1=0) , gen(dst_elct)
recode water_u 				(0=1)(1=0) , gen(dst_wtr)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl)
recode hh_assets2_u 		(0=1)(1=0) , gen(dst_asst) 


*** Rename indicators for changes over time estimation ***	
recode hh_mortality 		(0=1)(1=0) , gen(d_cm_01)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr_01)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt_01)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ_01)
recode electricity 			(0=1)(1=0) , gen(d_elct_01)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr_01)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani_01)
recode housing_1 			(0=1)(1=0) , gen(d_hsg_01)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl_01)
recode hh_assets2_c 		(0=1)(1=0) , gen(d_asst_01)	
	

recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm_01)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr_01)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt_01)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ_01)
recode electricity_u		(0=1)(1=0) , gen(dst_elct_01)
recode water_u		 		(0=1)(1=0) , gen(dst_wtr_01)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani_01)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg_01)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl_01)
recode hh_assets2_u_c 		(0=1)(1=0) , gen(dst_asst_01)



*** Eligibility for years of schooling indicator ***
gen educ_elig = 1 
replace educ_elig = 0 if age < 10 
replace educ_elig = 0 if (age==10 | age==11) & eduyears < 6 
	/*The variable "educ_eligible" was replaced with a '0' for ineligible 
	individuals, i.e.: those who have not completed 6 years of schooling 
	following their starting school age.*/
label define lab_educ_elig 0"ineligible" 1"eligible"  
label values educ_elig lab_educ_elig
lab var educ_elig "Individual is eligible for educ indicator"
tab eduyears educ_elig,m


*** Generate coutry and survey details for estimation ***
char _dta[cty] "Ecuador"
char _dta[ccty] "ECU"
char _dta[year] "2013-2014" 	
char _dta[survey] "ECV"
char _dta[ccnum] "218"
char _dta[type] "micro"


*** Sort, compress and save data for estimation ***
sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/ecu_ecv13-14.dta", replace 


	//Erase files from folder:
erase "$path_out/ECU13-14_children.dta"	
