********************************************************************************
/*
Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
Global Multidimensional Poverty Index - Dominican Republic MICS 2019 
[STATA do-file]. Available from OPHI website: http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk
*/
********************************************************************************

clear all 
set more off
set maxvar 10000


global path_in "../rdta/Dominican Republic MICS 2019" 	  	// folder path
global path_out "cdta"
global path_ado "ado"


********************************************************************************
**# Step 1 Data preparation 
********************************************************************************


********************************************************************************
**# Step 1.1 CH - CHILDREN's RECODE (under 5)
********************************************************************************
	
use "$path_in/ch.dta", clear 

rename _all, lower

		
gen double ind_id = hh1*100000 + hh2*100 + ln 
format ind_id %20.0g

duplicates report ind_id


*** Variable: SEX ***
codebook hl4, tab (9) 
	//"1" for male ;"2" for female 
clonevar gender = hl4
tab gender


*** Variable: AGE ***
desc cage caged
sum cage if caged>9000
clonevar age_days = caged
replace age_days = trunc(cage*(365/12)) if age_days>=9000 & cage<9000
sum age_days 

gen str6 ageunit = "days"
lab var ageunit "Days"


*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook an8, tab (9999) 
clonevar weight = an8	
replace weight = . if an8>=99 
tab	uf17 an8 if an8>=99 | an8==., miss   
sum weight 	


*** Variable: HEIGHT (CENTIMETERS)
codebook an11, tab (9999) 
clonevar height = an11
replace height = . if an11>=999 
tab	uf17 an11 if an11>=999 | an11==., miss
sum height 

	
*** Variable: MEASURED STANDING/LYING DOWN	
codebook an12  
gen measure = "l" if an12==1 
replace measure = "h" if an12==2 
replace measure = " " if an12==9 | an12==0 | an12==. 
tab measure 

		
*** Variable: OEDEMA ***
gen str1 oedema = "n"  


*** Variable: SAMPLING WEIGHT ***
gen sw = 1	


*** z-scores ***
adopath + "$path_ado/igrowup_stata"	
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "children_nutri_dom"
lab var datalab "Working file"

igrowup_restricted reflib datalib datalab gender age_days ageunit weight ///
height measure oedema sw

use "$path_out/children_nutri_dom_z_rc.dta", clear 

		
*** Standard MPI indicator ***
gen	underweight = (_zwei < -2.0) 
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "Child is undernourished (weight-for-age) 2sd - WHO"
tab underweight [aw=chweight], miss  


gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "Child is stunted (length/height-for-age) 2sd - WHO"
tab stunting [aw=chweight], miss 


gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "Child is wasted (weight-for-length/height) 2sd - WHO"
tab wasting [aw=chweight], miss 


*** Destitution indicator  ***	
gen	underweight_u = (_zwei < -3.0) 
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "Child is undernourished (weight-for-age) 3sd - WHO"


gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "Child is stunted (length/height-for-age) 3sd - WHO"


gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "Child is wasted (weight-for-length/height) 3sd - WHO"


count if _fwei==1 | _flen==1 
	/* 74 children were replaced as missing because they 
	have extreme z-scores which are biologically implausible. */
	
count
gen child_CH=1 
	//Number of eligible children: 8,503 as in the country report (p.7). 		
	
clonevar weight_ch = chweight
label var weight_ch "sample weight child under 5"


	//Retain relevant variables:
keep ind_id child_CH ln weight_ch underweight* stunting* wasting*  
order ind_id child_CH ln weight_ch underweight* stunting* wasting*
sort ind_id
save "$path_out/DOM19_CH.dta", replace

	
	//Erase files from folder:
erase "$path_out/children_nutri_dom_z_rc.xls"
erase "$path_out/children_nutri_dom_prev_rc.xls"
erase "$path_out/children_nutri_dom_z_rc.dta"


********************************************************************************
**# Step 1.2 BH - BIRTH RECODE 
*** (All females 15-49 years who ever gave birth)  
********************************************************************************

use "$path_in/bh.dta", clear

rename _all, lower	

	
*** Generate individual unique key variable required for data merging using:
	*** hh1=cluster number; 
	*** hh2=household number; 
	*** wm4=women's line number.   
gen double ind_id = hh1*100000 + hh2*100 + ln 
format ind_id %20.0g
label var ind_id "Individual ID"

		
desc bh4c bh9c	
gen date_death = bh4c + bh9c	
	//Date of death = date of birth (bh4c) + age at death (bh9c)	
gen mdead_survey = wdoi-date_death	
	//Months dead from survey = Date of interview (wdoi) - date of death	
replace mdead_survey = . if (bh9c==0 | bh9c==.) & bh5==1	
	/*Replace children who are alive as '.' to distinguish them from children 
	who died at 0 months */ 
gen ydead_survey = mdead_survey/12
	//Years dead from survey
	

gen age_death = bh9c if bh5==2
label var age_death "Age at death in months"	
tab age_death, miss


	
codebook bh5, tab (10)	
gen child_died = 1 if bh5==2
replace child_died = 0 if bh5==1
replace child_died = . if bh5==.
label define lab_died 0"child is alive" 1"child has died"
label values child_died lab_died
tab bh5 child_died, miss
	
	
bysort ind_id: egen tot_child_died = sum(child_died) 
	//For each woman, sum the number of children who died
		
	
	//Identify child under 18 mortality in the last 5 years
gen child18_died = child_died 
replace child18_died=0 if age_death>=216 & age_death<.
label values child18_died lab_died
tab child18_died, miss	
	
bysort ind_id: egen tot_child18_died_5y=sum(child18_died) if ydead_survey<=5
	/*Total number of children under 18 who died in the past 5 years 
	prior to the interview date */	
	
replace tot_child18_died_5y=0 if tot_child18_died_5y==. & tot_child_died>=0 & tot_child_died<.
	/*All children who are alive or who died longer than 5 years from the 
	interview date are replaced as '0'*/
	
replace tot_child18_died_5y=. if child18_died==1 & ydead_survey==.
	//Replace as '.' if there is no information on when the child died  

tab tot_child_died tot_child18_died_5y, miss

bysort ind_id: egen childu18_died_per_wom_5y = max(tot_child18_died_5y)
lab var childu18_died_per_wom_5y "Total child under 18 death for each women in the last 5 years (birth recode)"
	

	//Keep one observation per women
bysort ind_id: gen id=1 if _n==1
keep if id==1
drop id
duplicates report ind_id 


gen women_BH = 1 
	//Identification variable for observations in BH recode

	
	//Retain relevant variables
keep  ind_id women_BH childu18_died_per_wom_5y 
order ind_id women_BH childu18_died_per_wom_5y
sort  ind_id
save "$path_out/DOM19_BH.dta", replace	


********************************************************************************
**# Step 1.3 WM - WOMEN's RECODE  
*** (All eligible females 15-49 years in the household)
********************************************************************************

use "$path_in/wm.dta", clear 

	
rename _all, lower	

	
*** Generate individual unique key variable required for data merging
*** hh1=cluster number;  
*** hh2=household number; 
*** ln=respondent's line number
gen double ind_id = hh1*100000 + hh2*100 + ln
format ind_id %20.0g
label var ind_id "Individual ID"


duplicates report ind_id


gen women_WM =1 
	//Identification variable for observations in WM recode

tab wb4 wm17, miss
	/* Dominican Republic MICS 2019: 22,295 women 15-49 years were 
	successfully interviewed. Matches report (p.7). */	
	
tab cm1 cm8, miss	
	/*Dominican Republic MICS 2019: 10 women reported never having 
	given birth but has given birth to a boy or girl who was born 
	alive but later died. We use the child mortality information 
	provided by these individuals in section 2.4.*/
	
tab mstatus, miss	
tab mstatus if cm1==. & cm8==.,m
	/*In few datasets, you will note ALL never married women have missing 
	values for cm1 and cm8. This is because the questionnaire is only 
	administered to ever-married women. In such case, we will have to ensure 
	that these never married women are re-coded from missing to ineligible in 
	Section 2.4. However, this is not the case for Dominican Republic.*/
	
	
codebook mstatus ma6, tab (10)
tab mstatus ma6, miss 
gen marital = 1 if mstatus==3 & ma6==.
	//1: Never married
replace marital = 2 if mstatus==1 & ma6==.
	//2: Currently married
replace marital = 3 if mstatus==2 & ma6==1
	//3: Widowed	
replace marital = 4 if mstatus==2 & ma6==2
	//4: Divorced	
replace marital = 5 if (mstatus==2 & ma6==3) | ///
					   (mstatus==2 & ma6==.) | ///
					   (mstatus==9 & ma6==3)
	//5: Separated/not living together	
label define lab_mar 1"never married" 2"currently married" 3"widowed" ///
4"divorced" 5"not living together"
label values marital lab_mar	
label var marital "Marital status of household member"
tab marital, miss
tab ma6 marital, miss
tab mstatus marital, miss
rename marital marital_wom	


	//Retain relevant variables:	
keep cm1 cm8 cm9 cm10 ind_id women_WM *_wom
order cm1 cm8 cm9 cm10 ind_id women_WM *_wom
sort ind_id
save "$path_out/DOM19_WM.dta", replace



********************************************************************************
**# Step 1.4 MN - MEN'S RECODE 
***(All eligible man) 
********************************************************************************

	//No male recode

********************************************************************************
**# Step 1.5 HH - HOUSEHOLD RECODE 
***(All households interviewed) 
********************************************************************************

use "$path_in/hh.dta", clear 

rename _all, lower	

*** Generate individual unique key variable required for data merging
	*** hh1=cluster number;  
	*** hh2=household number; 
gen	double hh_id = hh1*100 + hh2 
format	hh_id %20.0g
lab var hh_id "Household ID"

duplicates report hh_id 

save "$path_out/DOM19_HH.dta", replace


********************************************************************************
**# Step 1.6 HL - HOUSEHOLD MEMBER  
********************************************************************************

use "$path_in/hl.dta", clear 
	
rename _all, lower


*** Generate a household unique key variable at the household level using: 
	***hh1=cluster number 
	***hh2=household number
gen double hh_id = hh1*100 + hh2 
format hh_id %20.0g
label var hh_id "Household ID"
codebook hh_id


*** Generate individual unique key variable required for data merging using:
	*** hh1=cluster number; 
	*** hh2=household number; 
	*** hl1=respondent's line number.
gen double ind_id = hh1*100000 + hh2*100 + hl1 
format ind_id %20.0g
label var ind_id "Individual ID"
codebook ind_id


sort ind_id


********************************************************************************
**# Step 1.7 DATA MERGING 
******************************************************************************** 
 
*** Merging BR Recode 
*****************************************
merge 1:1 ind_id using "$path_out/DOM19_BH.dta"
drop _merge
erase "$path_out/DOM19_BH.dta" 
  
  
*** Merging WM Recode 
*****************************************
merge 1:1 ind_id using "$path_out/DOM19_WM.dta"
count if hl8>0
	/*22,988 women 15-49 years were eligible 
	for interview. This matches the survey report 
	of Dominican Republic MICS 2019 (p.7). */
drop _merge
erase "$path_out/DOM19_WM.dta"


*** Merging HH Recode 
*****************************************
merge m:1 hh_id using "$path_out/DOM19_HH.dta"
tab hh46 if _m==2 
drop  if _merge==2
	//Drop households that were not interviewed 
drop _merge
erase "$path_out/DOM19_HH.dta"


*** Merging MN Recode 
*****************************************
//No male recode
	
gen marital_men = .
label var marital_men "Marital status of household member"


*** Merging CH Recode 
*****************************************
merge 1:1 ind_id using "$path_out/DOM19_CH.dta"
drop _merge
erase "$path_out/DOM19_CH.dta"

sort ind_id


********************************************************************************
**# Step 1.8 CONTROL VARIABLES
********************************************************************************


*** No Eligible Women 15-49 years
*****************************************
count if women_WM==1
count if hl8>0 & hl8!=.
	//Eligibility based on WM datafile (women_WM) and HL datafile (hl8) matches
gen	fem_eligible = (women_WM==1)
bys	hh_id: egen hh_n_fem_eligible = sum(fem_eligible) 	
	//Number of eligible women for interview in the hh
gen	no_fem_eligible = (hh_n_fem_eligible==0) 									
	//Takes value 1 if the household had no eligible females for an interview
lab var no_fem_eligible "Household has no eligible women"
drop fem_eligible hh_n_fem_eligible 
tab no_fem_eligible, miss


*** No Eligible Men 15-59 years
*****************************************
gen male_eligible = . 
gen no_male_eligible = . 
lab var no_male_eligible "Household has no eligible man"

	
*** No Eligible Children 0-5 years
***************************************** 
count if child_CH==1
count if hl10>0 & hl10!=.
	//Eligibility based on CH datafile (child_CH) and HL datafile (hl10) matches
gen	child_eligible = (child_CH==1) 
bysort	hh_id: egen hh_n_children_eligible = sum(child_eligible)  
	//Number of eligible children for anthropometrics
gen	no_child_eligible = (hh_n_children_eligible==0) 
	//Takes value 1 if there were no eligible children for anthropometrics
lab var no_child_eligible "Household has no children eligible for anthropometric"
drop hh_n_children_eligible
tab no_child_eligible, miss

sort hh_id


********************************************************************************
**# Step 1.9 RENAMING DEMOGRAPHIC VARIABLES 
********************************************************************************

//Sample weight
clonevar weight = hhweight 
label var weight "Sample weight"


//Area: urban or rural		
codebook hh6, tab (9)	
clonevar area = hh6  
replace area=0 if area==2  
label define lab_area 1 "urban" 0 "rural"
label values area lab_area
label var area "Area: urban-rural"


//Relationship to the head of household
desc hl3
clonevar relationship = hl3 
codebook relationship, tab (20)
recode relationship (1=1)(2=2)(3 13=3)(4/12=4)(96=5)(14=6)(98=.)(99=.)
label define lab_rel 1"head" 2"spouse" 3"child" 4"extended family" ///
5"not related" 6"maid"
label values relationship lab_rel
label var relationship "Relationship to the head of household"
tab hl3 relationship, miss	


//Sex of household member
codebook hl4
clonevar sex = hl4
replace sex=. if sex==9
label var sex "Sex of household member"


//Household headship
bys	hh_id: egen missing_hhead = min(relationship)
tab missing_hhead,m 
gen household_head=.
replace household_head=1 if relationship==1 & sex==1 
replace household_head=2 if relationship==1 & sex==2
bysort hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
label define head 1"male-headed" 2"female-headed"
label values headship head
label var headship "Household headship"
tab headship, miss


//Age of household member
codebook hl6, tab (999)
clonevar age = hl6  
replace age = . if age>=98
label var age "Age of household member"


//Age group 
recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+"), gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"


//Total number of hh members in the household
gen member = 1
bysort hh_id: egen hhsize = sum(member)
label var hhsize "Household size"
tab hhsize, miss
compare hhsize hh48
	/*hh48 comes with the datafile. We find that the hhsize variable that 
	we generated matches the original variable from the data file.*/ 
drop member


//Subnational region 
codebook hh7, tab (99)
decode hh7, gen(temp)
replace temp =  proper(temp)
encode temp, gen(region)
drop temp
lab var region "Region for subnational decomposition"
codebook region, tab (99)


//Harmonised over time

recode region (1=1 "Cibao Nordeste") (2=2 "Cibao Noroeste") ///
		     (3=3 "Ciabo Norte") (4=4 "Cibao Sur")  ///
			 (10=5 "Del Yuma") (5=6 "El Valle") ///
			 (6=7 "Enriquillo") (7=8 "Higuamo") ///
			 (8=9 "Metropolitana") (9=10 "Valdesia"), gen(region_01)			 
lab var region_01 "HOT:Region for subnational decomposition"
codebook region_01, tab (99)


********************************************************************************
**#  Step 2 Data preparation
***  Standardization of the 10 Global MPI indicators 
***  Identification of non-deprived & deprived individuals  
********************************************************************************

********************************************************************************
**# Step 2.1 Years of Schooling 
********************************************************************************

	/* Primary education takes place from age 6-14 (grades 1-8) and 
	it lasts 8 years. Secondary education takes place from age 
	14-18 (grades 9-12) and it lasts 4 years. */


codebook ed5a, tab (99)
clonevar edulevel = ed5a 
	//Highest educational level attended
replace edulevel = . if ed5a==. | ed5a==8 | ed5a==9  
	//All missing values or out of range are replaced as "."
replace edulevel = 0 if ed4==2 
	//Those who never attended school are replaced as '0'
label var edulevel "Highest level of education attended"


codebook ed5b, tab (99)
clonevar eduhighyear = ed5b 
	//Highest grade attended at that level
replace eduhighyear = .  if ed5b==. | ed5b==98 | ed5b==99 
	//All missing values or out of range are replaced as "."
replace eduhighyear = 0  if ed4==2 
	//Those who never attended school are replaced as '0'
lab var eduhighyear "Highest grade attended for each level of edu"


*** Cleaning inconsistencies
replace edulevel = 0 if age<10  
replace eduhighyear = 0 if age<10 
	/*At this point, we disregard the years of education of household members 
	younger than 10 years by replacing the relevant variables with '0 years' 
	since they are too young to have completed 6 years of schooling.*/ 
replace eduhighyear = 0 if edulevel<1
	//Early childhood education has no grade

	
*** Now we create the years of schooling
tab eduhighyear edulevel, miss
gen	eduyears = eduhighyear
replace eduyears = eduhighyear + 8 if edulevel==2 
	//Secondary (lower and higher) after 8 years of primary
	
replace eduyears = eduhighyear + 12 if edulevel==4  
	/* University education assumed to start after 12 years of general 
	education */

replace eduyears = 8 if edulevel==2 & eduhighyear==.
	/* We assume individuals would have completed at least 8 years 
	of schooling if they reported secondary as their highest level of 
	education but have missing grade. */	
	
replace eduyears = 12 if edulevel==4 & eduhighyear==.
	/* We assume individuals would have completed at least 12 years 
	of general schooling if they reported university as their highest 
	level of education but have missing year. */
	
	
	
*** Checking for further inconsistencies 
replace eduyears = eduyears - 1 if ed6==2 & eduyears>=1 & eduyears<. 
	/*Through ed6 variable, individuals confirm whether they have completed the 
	highest grade they have attended. For individuals who responded that they 
	did not complete the highest grade attended, we re-assign them to the next  
	lower grade that they would have completed. */
replace eduyears = . if age<=eduyears & age>0 
	/*There are cases in which the years of schooling are greater than the 
	age of the individual. This is clearly a mistake in the data.*/
replace eduyears = 0 if age< 10
replace eduyears = 0 if (age==10 | age==11) & eduyears < 6 
	/*The variable "eduyears" was replaced with a '0' for ineligible household 
	members, i.e.: those who have not completed 6 years of schooling following 
	their starting school age */
lab var eduyears "Total number of years of education accomplished"
tab eduyears, miss
tab eduyears edulevel, miss



	/*A control variable is created on whether there is information on 
	years of education for at least 2/3 of the eligible household members*/	
gen temp = 1 if eduyears!=. & age>=12 & age!=.
replace temp = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp = 1 if age==11 & eduyears>=6 & eduyears<.
bysort	hh_id: egen no_missing_edu = sum(temp)	
	//Total eligible household members with no missing years of education
gen temp2 = 1 if age>=12 & age!=.
replace temp2 = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp2 = 1 if age==11 & eduyears>=6 & eduyears<.
bysort hh_id: egen hhs = sum(temp2)
	/*Total number of eligible household members who should have information 
	on years of education */
replace no_missing_edu = no_missing_edu/hhs
replace no_missing_edu = (no_missing_edu>=2/3)
	/*Identify whether there is information on years of education for at 
	least 2/3 of the eligible household members */
tab no_missing_edu, miss
	//The value for 0 (missing) is 1.71% 
label var no_missing_edu "No missing edu for at least 2/3 of the eligible HH members"
drop temp temp2 hhs


*** Standard MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed six years of schooling. */
******************************************************************* 
gen	 years_edu6 = (eduyears>=6)
replace years_edu6 = . if eduyears==.
bysort hh_id: egen hh_years_edu6_1 = max(years_edu6)
gen	hh_years_edu6 = (hh_years_edu6_1==1)
replace hh_years_edu6 = . if hh_years_edu6_1==.
replace hh_years_edu6 = . if hh_years_edu6==0 & no_missing_edu==0 
lab var hh_years_edu6 "Household has at least one member with 6 years of edu"
tab hh_years_edu6, miss 

	
*** Destitution MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed at least one year of schooling. */
******************************************************************* 
gen	years_edu1 = (eduyears>=1)
replace years_edu1 = . if eduyears==.
bysort	hh_id: egen hh_years_edu_u = max(years_edu1)
replace hh_years_edu_u = . if hh_years_edu_u==0 & no_missing_edu==0
lab var hh_years_edu_u "Household has at least one member with 1 year of edu"


********************************************************************************
**# Step 2.2 Child School Attendance 
********************************************************************************

codebook ed4 ed9, tab (99)

gen	attendance = .
replace attendance = 1 if ed9==1 
	//Replace attendance with '1' if currently attending school	
replace attendance = 0 if ed9==2 
	//Replace attendance with '0' if currently not attending school	
replace attendance = 0 if ed4==2 
	//Replace attendance with '0' if never ever attended school	
replace attendance = 0 if age<5 | age>24 
	//Replace attendance with '0' for individuals who are not of school age	
label define lab_attend 1 "currently attending" 0 "not attending"
label values attendance lab_attend
label var attendance "Attended school during current school year"	
tab attendance, miss


*** Standard MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 8. */ 
******************************************************************* 
gen	child_schoolage = (schage>=6 & schage<=14)
	/*Note: In Dominican Republic, the official school entrance age is 6 years  
	 So, age range is 6-14 (=6+8). 
	Source: "http://data.uis.unesco.org/?ReportId=163"
	Education > Other policy relevant indicators >  
	Official entrance age to each ISCED level of education > 
	Official entrance age to primary education (years)
	Look at the starting age and add 8. 
	*/

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the school age children */
count if child_schoolage==1 & attendance==.
	/*How many eligible school aged children with missing school 
	attendance?: 4 children */
gen temp = 1 if child_schoolage==1 & attendance!=.
	/*Generate a variable that captures the number of eligible school aged 
	children who are attending school */
bysort hh_id: egen no_missing_atten = sum(temp)	
	/*Total school age children with no missing information on school 
	attendance */
gen temp2 = 1 if child_schoolage==1	
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are of school age
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)
	/*Identify whether there is missing information on school attendance for 
	more than 2/3 of the school age children */			
tab no_missing_atten, miss
	//The value for 0 (missing) is 0.01% 
label var no_missing_atten "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs
		
bysort hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
	//It takes value 1 if the household has children in school age
lab var hh_children_schoolage "Household has children in school age"

gen	child_not_atten = (attendance==0) if child_schoolage==1
replace child_not_atten = . if attendance==. & child_schoolage==1
bysort	hh_id: egen any_child_not_atten = max(child_not_atten)
gen	hh_child_atten = (any_child_not_atten==0) 
replace hh_child_atten = . if any_child_not_atten==.
replace hh_child_atten = 1 if hh_children_schoolage==0
replace hh_child_atten = . if hh_child_atten==1 & no_missing_atten==0 
lab var hh_child_atten "Household has all school age children up to class 8 in school"
tab hh_child_atten, miss


*** Destitution MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 6. */ 
******************************************************************* 
gen	child_schoolage_6 = (schage>=6 & schage<=12) 
	/* The official school entrance age is 6 years.  
	  So, age range for destitution measure is 6-12 (=6+6) */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the children attending school up to 
	class 6 */	
count if child_schoolage_6==1 & attendance==.	
gen temp = 1 if child_schoolage_6==1 & attendance!=.
bysort hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_6==1	
bysort hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)			
tab no_missing_atten_u, miss
label var no_missing_atten_u "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs		
		
bysort	hh_id: egen hh_children_schoolage_6 = sum(child_schoolage_6)
replace hh_children_schoolage_6 = (hh_children_schoolage_6>0) 
lab var hh_children_schoolage_6 "Household has children in school age (6 years of school)"

gen	child_atten_6 = (attendance==1) if child_schoolage_6==1
replace child_atten_6 = . if attendance==. & child_schoolage_6==1
bysort	hh_id: egen any_child_atten_6 = max(child_atten_6)
gen	hh_child_atten_u = (any_child_atten_6==1) 
replace hh_child_atten_u = . if any_child_atten_6==.
replace hh_child_atten_u = 1 if hh_children_schoolage_6==0
replace hh_child_atten_u = . if hh_child_atten_u==0 & no_missing_atten_u==0 
lab var hh_child_atten_u "Household has at least one school age children up to class 6 in school"
tab hh_child_atten_u, miss

 
********************************************************************************
**# Step 2.3 Nutrition 
********************************************************************************

********************************************************************************
**# Step 2.3a Child Nutrition 
********************************************************************************

*** Child Underweight Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(underweight)
gen	hh_no_underweight = (temp==0) 
	//Takes value 1 if no child in the hh is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_underweight "Household has no child underweight - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(underweight_u)
gen	hh_no_underweight_u = (temp==0) 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
lab var hh_no_underweight_u "Destitute: Household has no child underweight"
drop temp


*** Child Stunting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(stunting)
gen	hh_no_stunting = (temp==0) 
	//Takes value 1 if no child in the hh is stunted
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_stunting "Household has no child stunted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(stunting_u)
gen	hh_no_stunting_u = (temp==0) 
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
lab var hh_no_stunting_u "Destitute: Household has no child stunted"
drop temp


*** Child Wasting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(wasting)
gen	hh_no_wasting = (temp==0) 
	//Takes value 1 if no child in the hh is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_wasting "Household has no child wasted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(wasting_u)
gen	hh_no_wasting_u = (temp==0) 
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
lab var hh_no_wasting_u "Destitute: Household has no child wasted"
drop temp


*** Child Either Underweight or Stunted Indicator ***
************************************************************************

*** Standard MPI ***
gen uw_st = 1 if stunting==1 | underweight==1
replace uw_st = 0 if stunting==0 & underweight==0
replace uw_st = . if stunting==. & underweight==.

bysort hh_id: egen temp = max(uw_st)
gen	hh_no_uw_st = (temp==0) 
	//Takes value 1 if no child in the hh is underweight or stunted
replace hh_no_uw_st = . if temp==.
replace hh_no_uw_st = 1 if no_child_eligible==1
	//Households with no eligible children will receive a value of 1 
lab var hh_no_uw_st "Household has no child underweight or stunted"
drop temp


*** Destitution MPI  ***
gen uw_st_u = 1 if stunting_u==1 | underweight_u==1
replace uw_st_u = 0 if stunting_u==0 & underweight_u==0
replace uw_st_u = . if stunting_u==. & underweight_u==.

bysort hh_id: egen temp = max(uw_st_u)
gen	hh_no_uw_st_u = (temp==0) 
	//Takes value 1 if no child in the hh is underweight or stunted
replace hh_no_uw_st_u = . if temp==.
replace hh_no_uw_st_u = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1 
lab var hh_no_uw_st_u "Destitute: Household has no child underweight or stunted"
drop temp


*** Quality check ***
/* Compare the proportion of children under 5 who is stunted 
between the survey report and this dataset. */ 
*****************************************************************************
tab stunting [aw = weight] if child_CH==1
	/* Dominican Republic MICS 2019-20: The country survey report 
	indicate that 6.7% of children are stunted (see TC.8.1), while 
	it is 6.72% in this data which matches the report. */


********************************************************************************
**# Step 2.3b Household Nutrition Indicator 
********************************************************************************

*** Standard MPI ***
/* The indicator takes value 1 if the household has no child under 5 who 
has either height-for-age or weight-for-age that is under two standard 
deviation below the median. It also takes value 1 for the households that 
have no eligible children. The indicator takes a value of missing only if 
all eligible children have missing information in their respective 
nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st = 1
replace hh_nutrition_uw_st = 0 if hh_no_uw_st==0
replace hh_nutrition_uw_st = . if hh_no_uw_st==.
replace hh_nutrition_uw_st = 1 if no_child_eligible==1   		
lab var hh_nutrition_uw_st "Household has no individuals malnourished"
tab hh_nutrition_uw_st, miss


*** Destitution MPI ***
/* The indicator takes value 1 if the household has no child under 5 who 
has either height-for-age or weight-for-age that is under three standard 
deviation below the median. It also takes value 1 for the households that 
have no eligible children. The indicator takes a value of missing only if 
all eligible children have missing information in their respective 
nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st_u = 1
replace hh_nutrition_uw_st_u = 0 if hh_no_uw_st_u==0
replace hh_nutrition_uw_st_u = . if hh_no_uw_st_u==.
replace hh_nutrition_uw_st_u = 1 if no_child_eligible==1   		
lab var hh_nutrition_uw_st_u "Household has no individuals malnourished (destitution)"
tab hh_nutrition_uw_st_u, miss


*** Harmonised MPI ***
/* The indicator is excluded as nutrition data is not available
in the earlier survey. */
************************************************************************
gen	hh_nutr_01 = .
lab var hh_nutr_01 "HOT:HH has no individuals undernourished"


gen	hh_nutr_u_01 = .
lab var hh_nutr_u_01 "DST-HOT: HH has no individuals undernourished"


********************************************************************************
**# Step 2.4 Child Mortality 
********************************************************************************

codebook cm9 cm10 
	/*cm9: number of sons who have died 
	  cm10: number of daughters who have died */
	  
tab cm1 cm8, miss
tab childu18_died_per_wom_5y if cm8==1 & cm1==2,miss
	/* Identify the number of women who never gave birth 
	but has birth history data: 10 women.
	We make use of the birth history information from
	these women. */
		
	  
egen temp_f = rowtotal(cm9 cm10), missing
	//Total child mortality reported by eligible women
replace temp_f = 0 if (cm1==1 & cm8==2) | (cm1==2 & cm8==2) 
	/*Assign a value of "0" for:
	- all eligible women who have ever gave birth but reported no child death. 
	- all eligible women who never ever gave birth and reported no child death.*/
bysort	hh_id: egen child_mortality_f = sum(temp_f), missing
lab var child_mortality_f "Occurrence of child mortality reported by women"
tab child_mortality_f, miss
drop temp_f
	

egen child_mortality = rowmax(child_mortality_f)
replace child_mortality = 0 if child_mortality==. & no_fem_eligible==1 & no_male_eligible==1
lab var child_mortality "Total child mortality within household"
tab child_mortality, miss

	
*** Standard MPI *** 
/* The standard MPI indicator takes a value of "0" if women 
in the household reported mortality among children under 18 
in the last 5 years from the survey year.*/
************************************************************************
tab childu18_died_per_wom_5y, miss	
	
replace childu18_died_per_wom_5y = 0 if cm1==2 & cm8==2 
	/*Assign a value of "0" for:
	- all eligible women who never ever gave birth */
	
replace childu18_died_per_wom_5y = 0 if no_fem_eligible==1	
	/*Assign a value of "0" for:
	- individuals living in households that have non-eligible women */
	
bysort hh_id: egen childu18_mortality_5y = sum(childu18_died_per_wom_5y), missing
replace childu18_mortality_5y = 0 if childu18_mortality_5y==. & child_mortality==0
	/*Replace all households as 0 death if women has missing value and men 
	reported no death in those households */
	
label var childu18_mortality_5y "Under 18 child mortality within household past 5 years reported by women"
tab childu18_mortality_5y, miss		
	
gen hh_mortality_u18_5y = (childu18_mortality_5y==0)
replace hh_mortality_u18_5y = . if childu18_mortality_5y==.
lab var hh_mortality_u18_5y "Household had no under 18 child mortality in the last 5 years"
tab hh_mortality_u18_5y, miss 


*** Destitution MPI *** 
*** (same as standard MPI) ***
************************************************************************
clonevar hh_mortality_u = hh_mortality_u18_5y	


********************************************************************************
**# Step 2.5 Electricity 
********************************************************************************


*** Standard MPI ***
/*Members of the household are considered deprived 
if the household has no electricity */
****************************************
clonevar electricity = hc8 
codebook electricity, tab (9)
replace electricity = 1 if electricity==1 | electricity==2
replace electricity = 0 if electricity==3
	//0=no; 1=yes; 2 =yes
replace electricity = . if electricity==9 
	//Replace missing values 
label define lab_elec 1 "have electricity" 0 "no electricity"
label values electricity lab_elec		
label var electricity "Household has electricity"
tab electricity, miss



*** Destitution MPI  ***
*** (same as standard MPI) ***
****************************************
gen electricity_u = electricity
label var electricity_u "Household has electricity"


*** Quality check ***
/* We compare the proportion of households with 
electricity obtained from our work and as reported 
in the country survey report. */
*********************************************************
bysort hh_id: gen id = _n
tab electricity [aw = weight] if id==1,m
	/* Table SR.2.1 indicate that 98.56%  of households have access 
	to electricity. The results obtained from our work is 98.56% 
	which matches the report. */


********************************************************************************
**# Step 2.6 Sanitation 
********************************************************************************

/*
Improved sanitation facilities include flush or pour flush toilets to sewer 
systems, septic tanks or pit latrines, ventilated improved pit latrines, pit 
latrines with a slab, and composting toilets. These facilities are only 
considered improved if it is private, that is, it is not shared with other 
households. 
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-02-01.pdf

In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

desc ws11 ws15  
clonevar toilet = ws11  
clonevar shared_toilet = ws15 
codebook shared_toilet, tab(99)  
recode   shared_toilet (2=0)
replace shared_toilet=. if shared_toilet==9
 
/*  11 toilet with drain to public sewer or sewer
	12 toilet with drain to septic well
	13 a black well or hole
	14 toilet flush to drain or river
	18 toilet flush to do not know where
	21 improved pit latrine with ventilation
	22 latrine with drawer (dry composting)
	23 latrine without drawer
	41 cube or bacinilla
	95 no sanitary facility, goes to the mountain or open field
	96 other (specify)
	99 not responding
*/

				
*** Standard MPI ***
/*Members of the household are considered deprived if the household's 
sanitation facility is not improved (according to the SDG guideline) 
or it is improved but shared with other households*/
********************************************************************
codebook toilet, tab(30) 

gen toilet_mdg =  (toilet<=22 & shared_toilet!=1) 

replace toilet_mdg = 0 if (toilet==23 |toilet==41 | ///
					       toilet==95 | toilet==96) & shared_toilet==1
						   
replace toilet_mdg = . if toilet==.  | toilet==99
replace toilet_mdg = 0 if shared_toilet==1
lab var toilet_mdg "Household has improved sanitation with MDG Standards"
tab toilet toilet_mdg, miss

	
*** Destitution MPI ***
/*Members of the household are considered deprived if household practises 
open defecation or uses other unidentifiable sanitation practises */
********************************************************************
gen toilet_u = .
replace toilet_u = 0 if toilet==95 | toilet==96 
replace toilet_u = 1 if toilet!=95 & toilet!=96 & toilet!=. & toilet!=99
lab var toilet_u "Household does not practise open defecation or others"
tab toilet toilet_u, miss


*** Quality check ***
/* We compare the proportion of household members with 
improved sanitation obtained from our work and as 
reported in the country survey report. */
*********************************************************
tab toilet_mdg [aw = weight],m
	/* Table WS.3.2 indicates that 86.10% of 
	household members have improved sanitation facilities
	that are not shared. The results obtained from our 
	work is 87.69% */

	
********************************************************************************
**# Step 2.7 Drinking Water  
********************************************************************************
/*
Improved drinking water sources include the following: piped water into 
dwelling, yard or plot; public taps or standpipes; boreholes or tubewells; 
protected dug wells; protected springs; packaged water; delivered water and 
rainwater which is located on premises or is less than a 30-minute walk from 
home roundtrip. 
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-01-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

desc ws1 ws4 ws2
clonevar water = ws1  
clonevar timetowater = ws4  
clonevar ndwater = ws2  


*** Standard MPI ***
/* Members of the household are considered deprived if the household 
does not have access to improved drinking water (according to the SDG 
guideline) or safe drinking water is at least a 30-minute walk from 
home, roundtrip */
********************************************************************
codebook water, tab(99)

/* Improved water sources:
11 piped water: piped into the home
12 piped water: piped in the yard, solar or land
13 piped water: piped to a neighbor's house or yard
14 piped water: public tap
21 drilled well pipe
31 dug well: protected well
41 spring water: protected spring
51 rainwater
61 tank truck or tanker truck
71 small tank wagon
72 water station								
91 bottled water: bottled or packaged water (bottles)
92 bottled water: water bottle
93 processed truck water 

Unimproved water sources:
32 dug pit: unprotected pit
42 spring water: unprotected spring
81 river water, dam, lake, pond, stream, canal, irrigation canal
96 other (specify)
99 no answer */	

gen water_mdg = 1 if water<=31 | water==41 | ///
					 water==51 | water==61 | water==71 | ///
					 water==72 | water==91 | water==92 | water==93

replace water_mdg = 0 if water==32 | water==42 |  ///
						 water==81 | water==96 
					 
replace water_mdg = . if water==99 | water==.						 
tab water water_mdg, miss	
tab water_mdg, miss
	
	
*** Quality check ***
/* We compare the proportion of household members with 
improved access to safe drinking water as obtained from 
our work and as reported in the country survey report. */
*********************************************************
tab water_mdg [aw = weight],m
	/* Table WS.1.1 show that 98.84% of household members have 
	improved or safe drinking facilities. The results obtained 
	from our work is 98.84% which matches the report. */	 	
			
replace water_mdg = 0 if timetowater>= 30 & timetowater<995
lab var water_mdg "HH has safe drinking water, under 30 mins dist"	  	
tab water_mdg,m	
	
	
*** Destitution MPI ***
/* Members of the household is identified as destitute if household 
does not have access to safe drinking water, or safe water is more 
than 45 minute walk from home, round trip.*/
********************************************************************
gen	water_u = .
replace water_u = 1 if water<=31 | water==41 | ///
					   water==51 | water==61 | water==71 | ///
					   water==72 | water==91 | water==92 | water==93

replace water_u = 0 if water==32 | water==42 |  ///
					   water==81 | water==96 
					   
replace water_u = 0 if water_u==1 & timetowater> 45 & timetowater!=. ///
					   & timetowater!=998 & timetowater!=999 	
					   
replace water_u = . if water==99 | water==.	

replace water_u = 0 if (water==99 | water==.) & water_u==. & ///
						timetowater > 45 & timetowater!=. & ///
						timetowater!=998 & timetowater!=999 
						
lab var water_u "DST: HH has safe drinking water, 0-45 mins dist"
tab water water_u, miss
tab water_u, miss


********************************************************************************
**# Step 2.8 Housing
********************************************************************************

/* Members of the household are considered deprived if the household 
has a dirt, sand or dung floor. */

clonevar floor = hc4
codebook floor, tab(99)
gen floor_imp = 1
replace floor_imp = 0 if floor==11 | floor==96  
replace floor_imp = . if floor==.  | floor==99 
lab var floor_imp "Household has floor that it is not earth/sand/dung"
tab floor floor_imp, miss	
	
/* Members of the household are considered deprived if the household has wall 
made of natural or rudimentary materials. */ 

clonevar wall = hc6
codebook wall, tab(99)
gen wall_imp = 1 
replace wall_imp = 0 if wall<=28 | wall==96 
replace wall_imp = . if wall==.  | wall==99 
lab var wall_imp "Household has wall that it is not of low quality materials"
tab wall wall_imp, miss
		
	
/* Members of the household are considered deprived if the household has roof 
made of natural or rudimentary materials. */
clonevar roof = hc5
codebook roof, tab(99)	
gen roof_imp = 1 
replace roof_imp = 0 if roof==12 | roof==96
replace roof_imp = . if roof==.  | roof==99 
lab var roof_imp "Household has roof that it is not of low quality materials"
tab roof roof_imp, miss


*** Standard MPI ***
/* Members of the household is deprived in housing if the roof, 
floor OR walls are constructed from low quality materials.*/
**************************************************************
gen housing_1 = 1
replace housing_1 = 0 if floor_imp==0 | wall_imp==0 | roof_imp==0
replace housing_1 = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_1 "Household has roof, floor & walls that it is not low quality material"
tab housing_1, miss


*** Destitution MPI ***
/* Members of the household is deprived in housing if two out 
of three components (roof and walls; OR floor and walls; OR 
roof and floor) the are constructed from low quality materials. */
**************************************************************
gen housing_u = 1
replace housing_u = 0 if (floor_imp==0 & wall_imp==0 & roof_imp==1) | ///
						 (floor_imp==0 & wall_imp==1 & roof_imp==0) | ///
						 (floor_imp==1 & wall_imp==0 & roof_imp==0) | ///
						 (floor_imp==0 & wall_imp==0 & roof_imp==0)
replace housing_u = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_u "Household has one of three aspects(either roof,floor/walls) that is not low quality material"


*** Quality check ***
/* We compare the proportion of households with 
improved floor, walls and roof from our work and as 
reported in the country survey report. */
*********************************************************
tab floor_imp [aw = weight] if id==1,m
	/* Table SR.2.1 indicates that 97.8% of households have improved floor, 
	built using rudimentary or finished material. The result obtained 
	from our work is 97.77% which matches the report. */	

tab wall_imp [aw = weight] if id==1,m
	/* Table SR.2.1 indicates that 85.8% households have improved 
	wall (built using finished material). The result obtained from our 
	work is 85.84% which matches the report. */	
	
tab roof_imp [aw = weight] if id==1,m
	/* Table SR.2.1 indicates that 99.6% households have improved 
	roof (built using finished material). The results obtained from 
	our work is 99.63% which matches the report. */


********************************************************************************
**# Step 2.9 Cooking Fuel 
********************************************************************************
/*
Solid fuel are solid materials burned as fuels, which includes coal as well as 
solid biomass fuels (wood, animal dung, crop wastes and charcoal). 

Source: 
https://apps.who.int/iris/bitstream/handle/10665/141496/9789241548885_eng.pdf
*/

clonevar cookingfuel = eu4 
	//eu4 = type of fuel or energy source used for the cookstove
	

*** Standard MPI ***
/* Members of the household are considered deprived if the 
household uses solid fuels and solid biomass fuels for cooking. */
*****************************************************************
codebook eu1 cookingfuel, tab(99)
tab eu1 cookingfuel, miss

gen cooking_mdg = 1
replace cooking_mdg = 0 if cookingfuel>=4 & cookingfuel<=12
	/* Deprived if: coal/lignite, charcoal, wood, straw/shrubs/grass, 
					agricultural crop, animal dung, woodchips, coconut shell */	
						
replace cooking_mdg = 0 if eu1==96 | cookingfuel==96
	/*10 individuals did not respond on the type of fuel used for cooking
	but they have used other type of stove. We identified these individuals 
	as using non-clean fuel for cooking. In addition, 9 individuals used other
	type of cooking fuel on other type of stove. We identified these 
	individuals as using non-clean fuel for cooking. */

replace cooking_mdg = . if cookingfuel==99 & eu1==99
	//Missing values replaced
	
lab var cooking_mdg "Household cooks with clean fuels"	
tab cookingfuel cooking_mdg, miss
tab eu1 cooking_mdg, miss
tab cooking_mdg, miss



*** Destitution MPI ***
*** (same as standard MPI) ***
****************************************	
gen	cooking_u = cooking_mdg
lab var cooking_u "Household cooks with clean fuels"


*** Quality check ***
/*We compare the proportion of households using 
clean fuels & technologies for cooking obtained from 
our work and as reported in the country survey report.*/
*********************************************************
tab cooking_mdg [aw = weight], m
	/* Table TC.4.2 indicates that 91.1% of household members use clean 
	fuels for cooking and 2.7% do not cook at home (91.1+2.7=93.8%). The 
	results obtained from our work is 93.84%, which matches the report. */	


********************************************************************************
**# Step 2.10 Assets ownership 
********************************************************************************


*** Television/LCD TV/plasma TV/color TV/black & white tv
lookfor tv television plasma lcd télé tele
codebook hc9a
clonevar television = hc9a 
lab var television "Household has television"
tab television, miss
	//2.26% missing values
tab electricity television, miss
	//This is because these households do not have electricity.
replace television=0 if electricity==0 & television==.	
	/*We make an assumption that there is no television in these households 
	given that there is no electricity. */

		
***	Radio/walkman/stereo/kindle
lookfor radio walkman stereo stéréo
codebook hc7b
clonevar radio = hc7b 
lab var radio "Household has radio"	



***	Handphone/telephone/iphone/mobilephone/ipod
lookfor telephone téléphone mobilephone ipod telefone tele celular
codebook hc7a hc12
clonevar telephone =  hc7a
replace telephone=1 if telephone!=1 & hc12==1	
	//hc12=mobilephone. Combine information on telephone and mobilephone.	
tab hc7a hc12 if telephone==1,miss
lab var telephone "Household has telephone (landline/mobilephone)"	

	
***	Refrigerator/icebox/fridge
lookfor refrigerator réfrigérateur refri
codebook hc9b
clonevar refrigerator = hc9b 
lab var refrigerator "Household has refrigerator"
tab refrigerator, miss
	//2.26% missing values 
tab electricity refrigerator, miss
	//This is because these households do not have electricity.
replace refrigerator=0 if electricity==0 & refrigerator==.	
	/*We make an assumption that there is no refrigerator in these households 
	given that there is no electricity.*/


***	Car/van/lorry/truck
lookfor carro camión jeep
codebook hc10e hc10g hc10h
clonevar car = hc10e
replace car=1 if car!=1 & hc10h==1	
	//hc10h=truck. 
replace car=1 if car!=1 & hc10g==1	
	//hc10g=jeep. 	
lab var car "Household has car"		

	
***	Bicycle/cycle rickshaw
lookfor bicycle bicyclette bicicleta
codebook hc10b
clonevar bicycle = hc10b 
lab var bicycle "Household has bicycle"	
	
	
***	Motorbike/motorized bike/autorickshaw
lookfor motorbike moto
codebook hc10c	
clonevar motorbike = hc10c
lab var motorbike "Household has motorbike"

	
***	Computer/laptop/tablet
lookfor computer ordinateur laptop ipad tablet
codebook hc11
clonevar computer = hc11
lab var computer "Household has computer"


***	Animal cart
lookfor brouette  cart carro carreta
codebook hc10d
gen animal_cart = hc10d
lab var animal_cart "Household has animal cart"	
 
 
foreach var in television radio telephone refrigerator car ///
			   bicycle motorbike computer animal_cart {
replace `var' = 0 if `var'==2 
label define lab_`var' 0"No" 1"Yes"
label values `var' lab_`var'			   
replace `var' = . if `var'==9 | `var'==99 | `var'==8 | `var'==98 
}
	//Labels defined and missing values replaced	
	

*** Quality check ***
/* We compare the proportion of households owning  
each assets obtained from our work and as reported 
in the country survey report. */
*********************************************************
tab radio [aw = weight] if id==1,m
	/* Table SR.9.2 indicates that 47.8% of households 
	own a radio. The results obtained from our work is 44.98%. */	

tab television [aw = weight] if id==1,m
	/* Table SR.9.2 indicates that 81.9% of households 
	own television. The results obtained from our work is 
	81.91% which matches the report. */	
	
tab telephone [aw = weight] if id==1,m
	/* Table SR.9.2 indicates that 93.6% of households own either fixed 
	line or mobile phone. The results obtained from our work is 92.97%. */

tab computer [aw = weight] if id==1,m
	/* Table SR.9.2 indicates that 27.6% of households 
	own computer. The results obtained from our work is 
	27.62% which matches the report. */
	
tab refrigerator [aw = weight] if id==1,m
	/* Table SR.2.2 indicates that 80.4% of households own 
	refrigerator. The results obtained from our work is 80.38% 
	which matches the report. */

tab bicycle [aw = weight] if id==1,m
	/* Table SR.2.2 indicates that 4.2% of households own bicycle. 
	The results obtained from our work is 4.21% which matches the report.*/

tab animal_cart [aw = weight] if id==1,m
	/* Table SR.2.2 indicates that 0.6% of households own animal 
	cart. The results obtained from our work is 0.65% which matches the report.*/

tab motorbike [aw = weight]if id==1,m
	/* Table SR.2.2 indicates that 36.0% of households own bicycle. 
	The results obtained from our work is 36.04% which matches the report.*/

tab car [aw = weight] if id==1,m
	/* Table SR.2.2 indicates that 18,8% of households own car. 
	The results obtained from our work is 22.94% which includes
	jeep and truck ownership.*/



*** Standard MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own more than one of: radio, TV, telephone, bike, motorbike, 
refrigerator, computer or animal cart and does not own a car or truck.*/
*****************************************************************************
egen n_small_assets2 = rowtotal(television radio telephone refrigerator bicycle motorbike computer animal_cart), missing
lab var n_small_assets2 "Household Number of Small Assets Owned" 
   
gen hh_assets2 = (car==1 | n_small_assets2 > 1) 
replace hh_assets2 = . if car==. & n_small_assets2==.
lab var hh_assets2 "Household Asset Ownership: HH has car or more than 1 small assets incl computer & animal cart"
tab hh_assets2, miss


*** Destitution MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own any assets.*/
*****************************************************************************	
gen	hh_assets2_u = (car==1 | n_small_assets2>0)
replace hh_assets2_u = . if car==. & n_small_assets2==.
lab var hh_assets2_u "Household Asset Ownership: HH has car or at least 1 small assets incl computer & animal cart"


*** Harmonised MPI: DHS 2007 - MICS 2014 - MICS 2019 *** 
	/*The earlier survey lacked information on animal cart. As 
	such, the harmonised work excludes animal cart.*/
************************************************************************
gen animal_cart_01 = .
lab var animal_cart_01 "COT: Household has animal cart"

egen n_small_assets_01 = rowtotal(television radio telephone refrigerator bicycle motorbike computer), m
lab var n_small_assets_01 "COT: Number of small assets owned by HH" 
   
gen assets_01 = (car==1 | n_small_assets_01 > 1) 
replace assets_01 = . if car==. & n_small_assets_01==.
lab var assets_01 "COT: HH has assets"
tab n_small_assets_01 assets_01, miss
tab assets_01, miss


gen	assets_u_01 = (car==1 | n_small_assets_01>0)
replace assets_u_01 = . if car==. & n_small_assets_01==.
lab var assets_u_01 "COT-DST: HH has assets"
tab n_small_assets_01 assets_u_01, miss
tab assets_u_01, miss


********************************************************************************
**# Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************

	//Retain data on sampling design: 
desc psu stratum
clonevar strata = stratum
label var psu "Primary sampling unit"
label var strata "Sample strata"
	

	//Retain year, month & date of interview:
desc hh5y hh5m hh5d 
clonevar year_interview = hh5y 	
clonevar month_interview = hh5m 
clonevar date_interview = hh5d 


	//Generate presence of subsample
gen subsample = .
 

*** Keep main variables require for MPI calculation ***
keep hh_id ind_id subsample strata psu weight weight_ch sex age hhsize ///
area agec7 agec4 agec2 region headship relationship ///
no_fem_eligible no_male_eligible child_eligible no_child_eligible ///
marital_wom marital_men year_interview month_interview date_interview /// 
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet toilet_mdg shared_toilet toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
region_01 hh_nutr_01 hh_nutr_u_01 ///
animal_cart_01 n_small_assets_01 assets_01 assets_u_01

	 
*** Order file	***
order hh_id ind_id subsample strata psu weight weight_ch sex age hhsize ///
area agec7 agec4 agec2 region headship relationship ///
no_fem_eligible no_male_eligible child_eligible no_child_eligible ///
marital_wom marital_men year_interview month_interview date_interview /// 
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet toilet_mdg shared_toilet toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
region_01 hh_nutr_01 hh_nutr_u_01  ///
animal_cart_01 n_small_assets_01 assets_01 assets_u_01



*** Rename key global MPI indicators for estimation ***
recode hh_mortality_u18_5y  (0=1)(1=0) , gen(d_cm)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ)
recode electricity 			(0=1)(1=0) , gen(d_elct)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani)
recode housing_1 			(0=1)(1=0) , gen(d_hsg)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl)
recode hh_assets2 			(0=1)(1=0) , gen(d_asst)
 

*** Rename key global MPI indicators for destitution estimation ***
recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ)
recode electricity_u		(0=1)(1=0) , gen(dst_elct)
recode water_u 				(0=1)(1=0) , gen(dst_wtr)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl)
recode hh_assets2_u 		(0=1)(1=0) , gen(dst_asst) 


*** Rename indicators for changes over time estimation ***	
recode hh_mortality_u18_5y  (0=1)(1=0) , gen(d_cm_01)
recode hh_nutr_01			(0=1)(1=0) , gen(d_nutr_01)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt_01)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ_01)
recode electricity 			(0=1)(1=0) , gen(d_elct_01)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr_01)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani_01)
recode housing_1 			(0=1)(1=0) , gen(d_hsg_01)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl_01)
recode assets_01	 		(0=1)(1=0) , gen(d_asst_01)	
	

recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm_01)
recode hh_nutr_u_01 	 	(0=1)(1=0) , gen(dst_nutr_01)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt_01)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ_01)
recode electricity_u		(0=1)(1=0) , gen(dst_elct_01)
recode water_u	 			(0=1)(1=0) , gen(dst_wtr_01)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani_01)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg_01)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl_01)
recode assets_u_01 			(0=1)(1=0) , gen(dst_asst_01)



*** Eligibility for years of schooling indicator ***
gen educ_elig = 1 
replace educ_elig = 0 if age < 10 
replace educ_elig = 0 if (age==10 | age==11) & eduyears < 6 
	/*The variable "educ_eligible" was replaced with a '0' for ineligible 
	individuals, i.e.: those who have not completed 6 years of schooling 
	following their starting school age.*/
label define lab_educ_elig 0"ineligible" 1"eligible"  
label values educ_elig lab_educ_elig
lab var educ_elig "Individual is eligible for educ indicator"
tab eduyears educ_elig,m


*** Generate coutry and survey details for estimation ***
char _dta[cty] "Dominican Republic"
char _dta[ccty] "DOM"
char _dta[year] "2019" 	
char _dta[survey] "MICS"
char _dta[ccnum] "214"
char _dta[type] "micro"



*** Sort, compress and save data for estimation ***
sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/dom_mics19.dta", replace 
