********************************************************************************
/*
Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
Global Multidimensional Poverty Index - China Family Panel Studies 2014 
[STATA do-file]. Available from OPHI website: http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk

*/
********************************************************************************

clear all 
set more off
set maxvar 10000



*** Working Folder Path ***
global path_in "../rdta/China CFPS 2014" 
global path_out "cdta"
global path_ado "ado"


********************************************************************************
*** CHINA CFPS 2014 ***
********************************************************************************

/*
Note 1:
China CFPS 2014: Anthropometric information were recorded for all 
individuals aged 0-98 years (CFPS 2014 Questionnaire, Chinese version). However, 
to be consistent with the Global MPI criteria, we have used the nutritional 
information for adults up to the age of 70 years.

Note 2:
The data for the CFPS 2014 may be downloaded from the link appended below:
http://opendata.pku.edu.cn/dataverse/CFPS
*/


********************************************************************************
*** Step 1: Data preparation 
*** Merging adult, child and family dta files 
********************************************************************************
	
	//Data on adult members 
use  "$path_in/cfps2014_adult_20160601.dta", clear 
	/*For the purpose of the global MPI, we use the CFPS 2014 raw data 
	that was released in June 2016 */
gen data_adult = 1
save "$path_out/adult.dta", replace


	//Data on children
use "$path_in/cfps2014child_20160601.dta", clear 
	/*For the purpose of the global MPI, we use the CFPS 2014 raw data 
	that was released in June 2016 */
gen data_child = 1
save "$path_out/child.dta", replace


	//Data on family members
use "$path_in/cfps2014famconf_20160601.dta", clear 
	/*For the purpose of the global MPI, we use the CFPS 2014 raw data 
	that was released in June 2016 */
keep  if co_a14_p==1     
	*keep the members who have economic relations 
gen data_famconf = 1
save "$path_out/famconf.dta", replace


	//Data on family economy
use "$path_in/cfps2014famecon_20160601.dta", clear 
	/*For the purpose of the global MPI, we use the CFPS 2014 raw data 
	that was released in June 2016 */
gen data_famecon = 1
save "$path_out/famecon.dta", replace


	//Data on community
use "$path_in/Cfps2014comm_20160601.dta", clear 
	/*For the purpose of the global MPI, we use the CFPS 2014 raw data 
	that was released in June 2016 */
gen data_comm = 1
save "$path_out/comm.dta", replace



	//Merge relevant datasets
use "$path_out/adult.dta",clear   

merge 1:1 pid using "$path_out/child.dta",force
drop _merge
  
  
merge 1:1 pid using "$path_out/famconf.dta",force
drop _merge
  
  
merge m:1 fid14 using "$path_out/famecon.dta",force
drop if _merge == 2
drop _merge
  
 
merge m:1 cid14 using "$path_out/comm.dta",force
drop if _merge == 2
drop _merge  
  
keep if  data_child == 1 | data_adult == 1
  
	//Survey year
gen year = 2014
label var year "Survey year"  


count                                
	// 45,764 individuals

	//Check if relevant survey year
tab year, miss


	//Generate a household unique key variable at the household level
clonevar  hh_id = fid14
format    hh_id %20.0g
label var hh_id "Household ID"
codebook  hh_id  


	//Label individual id variable
clonevar  ind_id = pid	
format    ind_id %20.0g
label var ind_id "Individual ID"
duplicates report ind_id
	//No duplicates  



	//Age of household members 
clonevar birthm = tb1m_a_p
replace  birthm =. if tb1m_a_p<0 | tb1m_a_p==.
clonevar birthy = tb1y_a_p
replace  birthy =. if tb1y_a_p<0 | tb1y_a_p==.

gen month_temp=cmonth-birthm if birthm!=. 
gen     age_month=(cyear-birthy)*12+month_temp if month_temp>=0 & ///
                   month_temp!=. & cyear==2014 
				   //these children are older than their age
replace age_month=(cyear-1-birthy)*12+month_temp if month_temp>=0 & ///
                   month_temp!=. & cyear==2015 
				   //for cyear==2013
replace age_month=(cyear-birthy)*12+month_temp if month_temp<0 & cyear==2014 
                    //there children are younger than their age
replace age_month=(cyear-1-birthy-1)*12+(cmonth+(12-birthm)) if month_temp<0 ///
                   & cyear==2015 & (cyear-1-birthy)>=0  
				   //for cyear==2013
replace age_month=12+month_temp if month_temp<0 & cyear==2015 ///
                  & (cyear-1-birthy)<=0 
replace age_month=(cyear-birthy)*12 if month_temp==. & ///
                  cyear==2014 
replace age_month=(cyear-1-birthy)*12 if month_temp==. & cyear==2015 
                  //for cyear==2013

replace age_month=(cyear-birthy)*12+month_temp if month_temp>=0 & ///
                   month_temp!=. & cyear==2014 
				   //children are older than their age
replace age_month=(cyear-1-birthy)*12+month_temp if month_temp>=0 & ///
                   month_temp!=. & cyear==2015  
				   //for cyear==2013
								  
lab var age_month "age (month)"


	//Generate child's age who is 5 years old and above
gen     age_year=cyear-birthy
replace age_year=cfps2014_age if age_year==. 
replace age_year=tb1m_a_p if age_year==.
replace age_year=2014-birthy if age_year<0 
gen     age = age_year
lab var age  "Age of household member"


gen child5  = (age<=5)
gen child15 = (age<=15)                   
gen child19 = (age<=19)                   
gen adult19 = (age>19 & age!=.)    


count if age_year==.  

gen     age_group=1 if age_month<=60   
replace age_group=2 if age_month>60 & age_month<180   
replace age_group=3 if (age_month>=180 & age_year!=.) | age_year<=49
replace age_group=4 if age_year>=50 & age_year<.
tab     age_group, miss  
	//No missing observations

	
gen  province=provcd
capture drop sex
clonevar  sex=cfps_gender
clonevar  urban_1=urban14
clonevar  weight_ind=rswt_natcs14

bys age_group sex urban_1 province: sum weight_ind

gen 	prov = 1  if province==11
replace prov = 2  if province==12
replace prov = 3  if province==13
replace prov = 4  if province==14
replace prov = 5  if province==21
replace prov = 6  if province==22
replace prov = 7  if province==23
replace prov = 8  if province==31
replace prov = 9  if province==32
replace prov = 10 if province==33
replace prov = 11 if province==34
replace prov = 12 if province==35
replace prov = 13 if province==36
replace prov = 14 if province==37
replace prov = 15 if province==41
replace prov = 16 if province==42
replace prov = 17 if province==43
replace prov = 18 if province==44
replace prov = 19 if province==45
replace prov = 20 if province==50
replace prov = 21 if province==51
replace prov = 22 if province==52
replace prov = 23 if province==53
replace prov = 24 if province==61
replace prov = 25 if province==62

forvalues a=1(1)4 {
		forvalues s=0(1)1 {
				forvalues u=0(1)1 {
						forvalues p=1(1)25 {
								sum weight_ind if age_g==`a' & sex==`s' & ///
								urban_1==`u' & prov==`p'
								gen weight_a`a'_s`s'_u`u'_p`p' = r(mean)
						}
				}
		}
}

gen weight_core = weight_a1_s0_u0_p1 if age_g==1 & sex==0 & urban_1==0 & ///
                  prov==1 & coremember==1

forvalues a=1(1)4 {
		forvalues s=0(1)1 {
				forvalues u=0(1)1 {
						forvalues p=1(1)25 {
						        replace weight_core =  weight_a`a'_s`s'_u`u'_p`p' ///
								if age_g==`a' & sex==`s' & urban_1==`u' & ///
								   prov==`p' & coremember==1
						}
				}
		}
}


replace weight_ind=weight_core if coremember==1
lab var weight_ind "weight of individual"

egen double temp_1=sum(weight_ind)  
egen        temp_2=sum(weight_core)  

sum   temp_1 in 1
local mean: display %15.3f r(mean)
display "`mean'"

sum   temp_2 in 1
local mean: display %15.3f r(mean)
display "`mean'"
 
egen temp_3=sum(weight_ind)
sum  temp_3 in 1
local mean: display %15.3f r(mean)
display "`mean'"
 

clonevar weight_ind_pannel = rswt_natpn1014  

forvalues a=1(1)4 {
		forvalues s=0(1)1 {
				forvalues u=0(1)1 {
						forvalues p=1(1)25 {
								sum weight_ind_pannel if age_g==`a' ///
								    & sex==`s' & urban_1==`u' & prov==`p'
								gen weight_a`a'_s`s'_u`u'_p`p'_pannel = r(mean)
						}
				}
		}
}

gen weight_core_pannel = weight_a1_s0_u0_p1 if age_g==1 & sex==0 & urban_1==0 ///
                         & prov==1 & coremember==1

forvalues a=1(1)4 {
		forvalues s=0(1)1 {
				forvalues u=0(1)1 {
						forvalues p=1(1)25 {
						        replace weight_core_pannel =  ///
								weight_a`a'_s`s'_u`u'_p`p'_pannel if ///
								age_g==`a' & sex==`s' & urban_1==`u' ///
								& prov==`p' & coremember==1
						}
				}
		}
}

replace weight_ind_pannel=weight_core_pannel if coremember==1


	
	//Years of education of household members   
gen eduyears=.

	//For adults 
replace kr1=0 if data_adult==1 & wc01==1 & age_year>18 & kr1==2
replace kr1=0 if kr1==1

replace eduyears=6  if kr1==3 & data_adult==1 & wc01==1  
replace eduyears=9  if kr1==4 & data_adult==1 & wc01==1  
replace eduyears=12 if (kr1==5 | kr1==6) & data_adult==1 & wc01==1 
replace eduyears=16 if kr1==7 & data_adult==1 & wc01==1 
replace eduyears=18 if kr1==8 & data_adult==1 & wc01==1 
	//for adults who are attending school

replace eduyears=0  if tb4_a14_p==1 & data_adult==1 & wc01!=1 & kr1<=0
replace eduyears=6  if tb4_a14_p==2 & data_adult==1 & wc01!=1 & kr1<=0
replace eduyears=9  if tb4_a14_p==3 & data_adult==1 & wc01!=1 & kr1<=0
replace eduyears=12 if tb4_a14_p==4 & data_adult==1 & wc01!=1 & kr1<=0
replace eduyears=15 if tb4_a14_p==5 & data_adult==1 & wc01!=1 & kr1<=0
replace eduyears=16 if tb4_a14_p==6 & data_adult==1 & wc01!=1 & kr1<=0
replace eduyears=18 if tb4_a14_p==7 & data_adult==1 & wc01!=1 & kr1<=0
replace eduyears=22 if tb4_a14_p==8 & data_adult==1 & wc01!=1 & kr1<=0
	//for adults who are no longer atteding school

replace kw302=. if kw302<=0 | kw302==. 
replace kw2y=.  if kw2y<=0  | kw2y==. 
gen temp=kw2y-kw302		

replace eduyears=temp if tb4_a14_p==1 & temp<=6 & temp>0  
replace eduyears=temp if tb4_a14_p==2 & temp>6  & temp<=9
replace eduyears=temp if tb4_a14_p==3 & temp>9  & temp<=12 
replace eduyears=temp if tb4_a14_p==4 & temp>12 & temp<=16 
replace eduyears=temp if tb4_a14_p==5 & temp>12 & temp<=16 
replace eduyears=temp if tb4_a14_p==6 & temp>12 & temp<=16 
replace eduyears=temp if tb4_a14_p==7 & temp>16 & temp<=18
replace eduyears=temp if tb4_a14_p==8 & temp>18 & temp<=23


	//For children: cleaning inconsistencies
replace wc01=. if wc01<0
replace wc01=. if wc01==1 & kr1==-8  
	/*Note: There is 1 individuals who are report wc01==1(attending school) 
	but the status of school is not availble (wf3m==-8). As such we consider 
	them as missing value */
replace wf3m=.   if wf3m<0 	
replace kr1=.    if kr1<0 
replace wf301m=. if wf301m<0 
replace wf302=.  if wf302<0
replace wf303=.  if wf303<0

gen     schoollevel_a = 0 if age_month>=72 &  age_month!=. & wf301m<=2		
replace schoollevel_a = 1 if age_month>=72 &  age_month!=. & wf301m==3 
replace schoollevel_a = 2 if age_month>=72 &  age_month!=. & wf301m==4
replace schoollevel_a = 3 if age_month>=72 &  age_month!=. & wf301m==5
replace schoollevel_a = 4 if age_month>=72 &  age_month!=. & wf301m==6 

replace schoollevel_a=1 if age_month>=72 &  age_month!=. & wf301m==. & kr1==2 
replace schoollevel_a=2 if age_month>=72 &  age_month!=. & wf301m==. & kr1==3 
replace schoollevel_a=3 if age_month>=72 &  age_month!=. & wf301m==. & kr1==4

replace eduyears=wf302-1  if age_month>=72 &  age_month!=. ///
							 & schoollevel_a==1 & data_child==1
replace eduyears=wf302+5  if age_month>=72 &  age_month!=. ///
							 & schoollevel_a==2 & data_child==1
replace eduyears=wf302+8  if age_month>=72 &  age_month!=. ///
							 & schoollevel_a==3 & data_child==1
replace eduyears=wf302+11 if age_month>=72 &  age_month!=. ///
							 & schoollevel_a==4 & data_child==1
replace eduyears=. if age_year<=eduyears & age_month>=72 & eduyears!=. ///
                      & data_child==1
	//Note: Consider inconsistencies as missing

replace eduyears=(cyear-wf303)-1 if wf3m==1 & wf303!=. & cyear!=. ///
				 & age_month>=72 & age_month<=156
replace eduyears=0 if eduyears==-1
replace eduyears=(cyear-wf303)+5 if wf3m==1 & wf303!=. & cyear!=. ///
				 & age_month>152 & age_month<=180
replace eduyears=1 if cfps2012_latest_r1==2 & tb4_a14_p==1 & eduyears==0

lab var  eduyears "years of education for each individual"


	//School attendance
	//consider the attendance age from 7 to 15 
gen att_temp=(wf3m==1) if (age_month<192 & data_adult==1) | ( age_month>=84)
replace att_temp=. if wf3m==.
replace att_temp=wc01 if (age_month<192 & data_adult==1) | ///
						 (age_month>=84) & wf3m==. 
						 										  
gen attendance=att_temp 
replace attendance=0 if wf301m==1 | wf301m==2  
replace attendance=wc01 if attendance==. & ((age_month<192) | (age_month>=84))
drop temp

lab var attendance "Attending school"
label define lab_att 0 "Not attending" 1 "Yes, attending"
label values attendance lab_att         
tab age_year if age_month>=84 & age_month<192 & attendance==.,miss	
tab wc01 if age_month>=84 & age_month<192 & attendance==.,miss
tab attendance if age_month>=84 & age_month<192, miss

	
	
	//Anthropometric data: weight
codebook qp102, tab (9999)                        
codebook wa103, tab (9999)  	
clonevar weight=qp102
	//qp102=weight for adults
replace weight = wa103 if weight == .
	//wa103 is weight for children		
replace  weight=. if weight<0
replace  weight=weight/2  
label  var weight "Weight in kilograms"


                      
	//Anthropometric data: height 
codebook qp101, tab (9999)                       
codebook wa104, tab (9999)                       
clonevar height=qp101
	//qp101=height for adults
replace height = wa104 if (height == . | height<0)
	//wa103 is height for children
replace  height=. if height<0
lab var height "Height in centimeters"


	//Assigning individual weight
drop      weight_ind
clonevar  weight_ind=rswt_natcs14


	//Sex of individual
tab cfps_gender, miss 
	//"1" for male; "2" for female 
tab cfps_gender, nol  
gen gender = (cfps_gender == 1)
replace gender = 2 if gender == 0
label define lab_gen 2 "female" 1 "male" 
label values gender lab_gen 
label var gender "Sex of household member"
tab gender, miss	
	

	//Child mortality ever
 codebook alive_a14_c*

 forval i = 1(1)10 {
 foreach var in alive_a14_c`i' {
  gen `var'_temp=1 if `var'==0 
  replace `var'_temp=0 if `var'==-8 | `var'==1 
  //-8 not applicable; 1. still alive. Later we change them into "no child dead"
  replace `var'_temp=. if (`var'==. | `var'==-2 | `var'==-1)  
  // -2 refuse answer; -1 dk, . missing. Later change them into "missing value"
  }
  }

capture drop temp
egen dead_children_temp= /// 
                   rowtotal(alive_a14_c1_temp alive_a14_c2_temp ///
                   alive_a14_c3_temp alive_a14_c4_temp alive_a14_c5_temp ///
                   alive_a14_c6_temp alive_a14_c7_temp alive_a14_c8_temp ///
				   alive_a14_c9_temp alive_a14_c10_temp) 
tab dead_children_temp, miss	
	//Total child mortality reported by women & men			


forvalues i = 1(1)10 {

gen month_temp_`i' = cmonth-tb1m_a_c`i' if tb1m_a_c`i'>= 1 & tb1m_a_c`i'<= 12

gen age_deadm_`i'=(cyear - tb1y_a_c`i') * 12 + month_temp_`i' ///
 if month_temp_`i'>=0 & month_temp_`i' !=. & cyear==2014
 
replace age_deadm_`i'=(cyear-1 - tb1y_a_c`i') * 12+ month_temp_`i' ///
                      if month_temp_`i'>=0 & month_temp_`i' !=. & cyear==2015 
  
replace age_deadm_`i'=(cyear - tb1y_a_c`i')*12+ month_temp_`i' ///
                      if month_temp_`i'<0 & cyear==2014 
                       //there children are younger than their age
  
replace age_deadm_`i'=(cyear-1-tb1y_a_c`i'-1)*12+(cmonth+(12-tb1m_a_c`i'))  ///
                      if month_temp_`i'<0 & cyear==2015 & (cyear-1-tb1y_a_c`i')>=0  
replace age_deadm_`i'=12+month_temp_`i' if month_temp_`i'<0 & cyear==2015 ///
                      & (cyear-1-tb1y_a_c`i')<=0 
 
replace age_deadm_`i'=(cyear-tb1y_a_c`i')*12 if month_temp_`i'==. /// 
& cyear==2014 
 
replace age_deadm_`i'=(cyear-1-tb1y_a_c`i')*12 if month_temp_`i'==. & cyear==2015 
                 
replace age_deadm_`i'=(cyear-tb1y_a_c`i')*12+month_temp_`i' if month_temp_`i'>=0 ///
                       & month_temp_`i'!=. & cyear==2014 
      //children are older than their age
  
replace age_deadm_`i'=(cyear-1-tb1y_a_c`i')*12+month_temp_`i'  ///
                       if month_temp_`i'>=0 &  month_temp_`i'!=. & cyear==2015  
gen       age_death_`i' = age_deadm_`i' & alive_a14_c`i'==0
tab age_death_`i', miss
// check whether the age is in months
label var age_death_`i' "Age at death in months"
}

*	

	//Save master file
save "$path_out/cfps2014.dta", replace
count
	//45,764 individuals


	//Erase temp files from folder
erase "$path_out/adult.dta"
erase "$path_out/child.dta"
erase "$path_out/famconf.dta"
erase "$path_out/famecon.dta"
erase "$path_out/comm.dta"

clear


********************************************************************************
*** Step 1.1 Underweight, Stunting & Wasting for children 0-60 months in age
********************************************************************************

use "$path_out/cfps2014.dta", clear 

count if age_month<=60
keep  if age_month<=60
	//3,151 children 0-60 months

	
gen child_KR=1 
	//Generate identification variable for children 0-60 months in age


*** Check the variables that WHO ado needs to calculate the z-scores:
*** sex, age, weight, height, measurement, oedema & child sampling weight

*** Variable: SEX ***
desc gender
tab  gender, miss


*** Variable: AGE ***
summ age_month
gen  str6 ageunit = "months" 
lab var ageunit "Months"


*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook weight, tab (9999)
replace  weight = .  if weight<0  
sum weight


*** Variable: HEIGHT (CENTIMETERS) ***
codebook height, tab (9999)
replace  height = .  if height<0  
sum height


*** Variable: MEASURED STANDING/LYING DOWN ***
	/* For children aged below 24 months (<731 days), it makes a difference 
	whether the child was measured lying down (recumbent), in which case his/her 
	length was measured, than whether he was measured standing, in which his/her 
	measure is height. The ado can adjust for this difference.*/		
gen str1 measure=" "
tab measure

	
*** Variable: OEDEMA ***
gen  oedema = "n"  
	//It assumes no-one has oedema
tab oedema	


*** Variable: INDIVIDUAL CHILD SAMPLING WEIGHT ***
gen  sw = weight_ind
sum sw	
	
	
/* 
For this part of the do-file we use the WHO Anthro and macros. This is to 
calculate the z-scores of children under 5. 
Source of ado file: http://www.who.int/childgrowth/software/en/
*/
*** Next, indicate to STATA where the igrowup_restricted.ado file is stored
adopath + "$path_ado/igrowup_stata"


*** We will now proceed to create three nutritional variables: 
	*** weight-for-age (underweight),  
	*** weight-for-height (wasting) 
	*** height-for-age (stunting)

/* We use 'reflib' to specify the package directory where the .dta files 
containing the WHO Child Growth Standards are stored. Note that we use 
strX to specify the length of the path in string. If the path is long, 
you may specify str55 or more, so it will run. */	
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"


/* We use datalib to specify the working directory where the input STATA 
dataset containing the anthropometric measurement is stored. */
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"


/* We use datalab to specify the name that will prefix the output files that 
will be produced from using this ado file (datalab_z_r_rc and datalab_prev_rc)*/
gen str30 datalab = "children_nutri_chn" 
lab var datalab "Working file"

	
	//We now run the command to calculate the z-scores with the adofile 
igrowup_restricted reflib datalib datalab gender age_month ageunit ///
weight height measure oedema sw


/*We now turn to using the dta file that was created and that contains 
the calculated z-scores to create the child nutrition variables following WHO 
standards */
use "$path_out/children_nutri_chn_z_rc.dta", clear 


*** Standard MPI indicator ***
	//Takes value 1 if the child is under 2 stdev below the median & 0 otherwise
	
gen	underweight = (_zwei < -2.0) 
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "Child is undernourished (weight-for-age) 2sd - WHO"
tab underweight, miss

gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "Child is stunted (length/height-for-age) 2sd - WHO"
tab stunting, miss


gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "Child is wasted (weight-for-length/height) 2sd - WHO"
tab wasting, miss


*** Destitution indicator  ***
	//Takes value 1 if the child is under 3 stdev below the median & 0 otherwise
	
gen	underweight_u = (_zwei < -3.0) 
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "Child is undernourished (weight-for-age) 3sd - WHO"


gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "Child is stunted (length/height-for-age) 3sd - WHO"


gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "Child is wasted (weight-for-length/height) 3sd - WHO"
 
clonevar weight_ch = weight_ind
label var weight_ch "sample weight child under 5"
 
	//Retain relevant variables:
keep ind_id child_KR weight_ch underweight* stunting* wasting* 
order ind_id child_KR weight_ch underweight* stunting* wasting* 
sort ind_id
duplicates report ind_id

	//Erase files from folder:
erase "$path_out/children_nutri_chn_z_rc.xls"
erase "$path_out/children_nutri_chn_prev_rc.xls"
erase "$path_out/children_nutri_chn_z_rc.dta"

	
	//Save a temp file for merging later:
save "$path_out/CHN14_KR.dta", replace


********************************************************************************
*** Step 1.2 BMI-for-age for Young Children 61-228 age in months
********************************************************************************

use "$path_out/cfps2014.dta", clear 

count if age_month> 60 & age_month< 229
keep  if age_month> 60 & age_month< 229 	
	//6,945 individuals 61-228 age in months
	
***Variables required to calculate the z-scores to produce BMI-for-age:#

*** Variable: SEX ***
tab gender, miss

	
*** Variable: AGE UNIT ***
gen str6 ageunit = "months" 
lab var ageunit "Months"

		
*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook weight, tab (9999)
replace  weight = .  if weight<0  
sum weight


*** Variable: HEIGHT (CENTIMETERS) ***
codebook height, tab (9999)
replace  height = . if height<0 
sum height


*** Variable: OEDEMA ***
gen oedema = "n"  
tab oedema	


*** Variable: SAMPLING WEIGHT ***
gen  sw = weight_ind
sum sw	

			
/* 
For this part of the do-file we use the WHO AnthroPlus software. This is to 
calculate the z-scores for young individuals aged 15-19 years. 
Source of ado file: https://www.who.int/growthref/tools/en/
*/

*** Next, indicate to STATA where the igrowup_restricted.ado file is stored:
adopath + "$path_ado/who2007_stata"


/* We use 'reflib' to specify the package directory where the .dta files 
containing the WHO Growth reference are stored. Note that we use strX to specity 
the length of the path in string. */		
gen str100 reflib = "$path_ado/who2007_stata"
lab var reflib "Directory of reference tables"

/* We use datalib to specify the working directory where the input STATA data
set containing the anthropometric measurement is stored. */
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"

/* We use datalab to specify the name that will prefix the output files that 
will be produced from using this ado file*/
gen str30 datalab = "teen_nutri_chn" 
lab var datalab "Working file"
	

	//We now run the command to calculate the z-scores with the adofile 
who2007 reflib datalib datalab gender age_month ageunit weight height oedema sw


/*We now turn to using the dta file that was created and that contains 
the calculated z-scores to compute BMI-for-age*/
use "$path_out/teen_nutri_chn_z.dta", clear 

gen	z_bmi = _zbfa
replace z_bmi = . if _fbfa==1 
lab var z_bmi "z-score bmi-for-age WHO"


*** Standard MPI indicator ***
gen	low_bmiage = (z_bmi < -2.0) 
	/*Takes value 1 if BMI-for-age is under 2 stdev below the median & 0 
	otherwise */
replace low_bmiage = . if z_bmi==.
lab var low_bmiage "6-19 years low bmi 2sd - WHO"


*** Destitution indicator ***
gen	low_bmiage_u = (z_bmi < -3.0)
 	/*Takes value 1 if BMI-for-age is under 3 stdev below the median & 0 
	otherwise */
replace low_bmiage_u = . if z_bmi==.
lab var low_bmiage_u "6-19 years very low bmi 3sd - WHO"


tab low_bmiage, miss
tab low_bmiage_u, miss


	//Identification variable for children 61-228 age in months
gen teen_IR=1 
	

	//Retain relevant variables:	
keep ind_id teen_IR age_month low_bmiage*
order ind_id teen_IR age_month low_bmiage*
sort ind_id


	/*Append the nutrition information of children 61-228 months with 
	children 0-60 months in age */	
append using "$path_out/CHN14_KR.dta"
	
	//Save a temp file for merging later:
save "$path_out/CHN14_children.dta", replace


	//Erase files from folder:
erase "$path_out/teen_nutri_chn_z.xls"
erase "$path_out/teen_nutri_chn_prev.xls"
erase "$path_out/teen_nutri_chn_z.dta"	
erase "$path_out/CHN14_KR.dta"


********************************************************************************
*** Step 1.3 BMI for all individuals
********************************************************************************

use "$path_out/cfps2014.dta", clear 

count if age_month>=180
	//37,479 individuals who are 15 years and older
count if age_month>=229 & age_month<=840	
	// 29,445 individuals who are between 229-840 months in age


*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook weight, tab (10000)
replace  weight = .  if weight<0  
sum weight


*** Variable: HEIGHT (CENTIMETERS) ***
codebook height, tab (10000)
replace  height = . if height<0 
sum height


*** Variable: BMI MEASURE ***
gen bmi=(weight)/((height/100)^2)
lab var bmi "BMI"


gen low_bmi = (bmi<18.5)
replace low_bmi=. if bmi==.
replace low_bmi = . if age_month>840 & age_month<.
	/*Replace individuals who are older than 70 years with '.' even if they had 
	provided nutrition information since the global MPI does not take into 
	account of nutrition information of those above 70 years. */
lab var low_bmi "BMI <18.5"
lab define lab_low_bmi 1 "bmi<18.5" 0 "bmi>=18.5"
lab values low_bmi lab_low_bmi
tab low_bmi, miss


gen low_bmi_u = (bmi<17)
replace low_bmi_u = . if bmi==. 
replace low_bmi_u = . if age_month>840 & age_month<.
lab var low_bmi_u "BMI <17"
lab define lab_low_bmi_u 1 "bmi<17" 0 "bmi>=17"
lab values low_bmi_u lab_low_bmi_u
tab low_bmi_u, miss

sort ind_id


	/*Merge nutrition information from individuals 0-228 months.
	10,096 individuals merged, this matches the number of individuals 
	0-228 months from the earlier section. */
merge 1:1 ind_id using "$path_out/CHN14_children.dta"

drop _merge year
rename weight weight_kg
rename height height_cm


********************************************************************************
*** Step 1.4 RENAMING DEMOGRAPHIC VARIABLES ***
********************************************************************************

//Sample weight 
clonevar  weight = weight_ind
label var weight "Sample weight"


//Type of place of residency: urban or rural		
clonevar area = urban14   
replace  area = . if area == -9 
	//Redefine the coding and labels to 1/0
label define lab_area 1 "urban" 0 "rural"
label values area lab_area
label var area "Area: urban-rural"
tab area, miss


//Relationship to the head of household 
	/*Note: The CFPS 2014 does not have a variable that directly identifies the 
	relationship to household head. Hence the variable takes missing values*/
gen relationship = .
label var relationship "Relationship to the head of household"


//Sex of household member	
capture drop sex
clonevar  sex = gender
label var sex "Sex of household member"


//Household headship
bys	hh_id: egen missing_hhead = min(relationship)
tab missing_hhead,m 
gen household_head=.
replace household_head=1 if relationship==1 & sex==1 
replace household_head=2 if relationship==1 & sex==2
bysort hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
label define head 1"male-headed" 2"female-headed"
label values headship head
label var headship "Household headship"
tab headship, miss


//Age of household member
codebook  age, tab (999)
label var age "Age of household member"


//Age group 
recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+"), gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"


//Marital status of household member
clonevar marital = tb3_a14_p 
codebook marital, tab (10)
recode   marital (3=1) (5=3) (-9=.) (-1=.) (-8=.) (-2=.)
	/*Please ensure recoding follows these 5 broad categories: 1"never married" 
	2"currently married" 3"widowed" 4"divorced" 5"not living together" */
	
	/*In the case of CFPS 2014, 196 individuals reported as cohabitating. We 
	recoded this group as never married due to the very small observation*/	
label define lab_mar 1"never married" 2"currently married" 3"widowed" ///
4"divorced" 5"not living together" 
label values marital lab_mar	
label var marital "Marital status of household member"
tab tb3_a14_p marital, miss


//Total number of hh members in the household
gen member = 1
bysort fid14: egen hhsize = sum(member)
label var hhsize "Household size"
tab hhsize, miss
drop member	


//Subnational region
	/*NOTE: The sample for CFPS 2014 is not representative at the province  
	level but across the three major subregions. */ 
rename province province_2	
gen province = provcd14
label define lab_prov ///
11  "Beijing" ///
12  "Tianjin" ///
13  "Hebei" ///
14  "Shanxi" ///
15  "Inner Mongolia" ///
21  "Liaoning" ///
22  "Jilin" ///
23  "Heilongjiang" ///
31  "Shanghai" ///
32  "Jiangsu" ///
33  "Zhejiang" ///
34  "Anhui" ///
35  "Fujian" ///
36  "Jiangxi" ///
37  "Shandong" ///
41  "Henan" ///
42  "Hubei" ///
43  "Hunan" ///
44  "Guangdong" ///
45  "Guangxi" ///
46  "Hainan" ///
50  "Chongqing" ///
51  "Sichaun" ///
52 	"Guizhou" ///
53  "Yunnan" ///
54 	"Xizang (Tibet)" ///
61  "Shan'xi" ///
62  "Gansu" ///
63  "Qinghai" ///
64  "Ningxia" ///
65  "Xinjiang"
label values province lab_prov
lab var province "Provinces"
codebook province, tab (100)


//Group provinces into three major region
	
	//region 1 = 10 provinces
gen region = 1 if provcd14==11 | provcd14==12 | provcd14==13 | ///
				  provcd14==21 | provcd14==31 | provcd14==32 | ///
				  provcd14==33 | provcd14==35 | provcd14==37 | provcd14==44  

	//region 2 = 8 provinces
replace region = 2 if provcd14==14 | provcd14==22 | provcd14==23 | ///
					  provcd14==34 | provcd14==36 | provcd14==41 | ///
					  provcd14==42 | provcd14==43  

	//region 3 = 13 provinces
replace region = 3 if provcd14>=45 | provcd14==15 

lab define lab_reg 1 "East/Coastal Region" 2 "Central Region" 3 "Western Region"
label values region lab_reg
lab var region "Region for subnational decomposition"
codebook region, tab (100)
tab province region, miss


	/*In this survey, the harmonised 'region_01' variable is the 
	same as the standardised 'region' variable.*/	
clonevar region_01 = region 


********************************************************************************
***  Step 2 Data preparation  ***
***  Standardization of the 10 Global MPI indicators 
***  Identification of non-deprived & deprived individuals  
********************************************************************************


********************************************************************************
*** Step 2.1 Years of Schooling ***
********************************************************************************
tab eduyears, miss
	//Total number of years of education
replace eduyears = . if eduyears>30
	//Recode any unreasonable years of highest education as missing value
replace eduyears = . if eduyears>=age & age>0
	/*The variable "eduyears" was replaced with a '.' if total years of 
	education was more than individual's age */
replace eduyears = 0 if age < 10 
	/*The variable "eduyears" was replaced with a '0' given that the criteria 
	for this indicator is household member aged 10 years or older */

	
	/*A control variable is created on whether there is information on 
	years of education for at least 2/3 of the household members aged 10 years 
	and older */	
gen temp = 1 if eduyears!=. & age>=10 & age!=.
bysort	hh_id: egen no_missing_edu = sum(temp)
	/*Total household members who are 10 years and older with no missing 
	years of education */
gen temp2 = 1 if age>=10 & age!=.
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are 10 years and older 
replace no_missing_edu = no_missing_edu/hhs
replace no_missing_edu = (no_missing_edu>=2/3)
	/*Identify whether there is information on years of education for at 
	least 2/3 of the household members aged 10 years and older */
tab no_missing_edu, miss
label var no_missing_edu "No missing edu for at least 2/3 of the eligible HH members"
drop temp temp2 hhs


*** Standard MPI ***
/*The entire household is considered deprived if no household member aged 
10 years or older has completed SIX years of schooling.*/
******************************************************************* 
gen	 years_edu6 = (eduyears>=6)
	/* The years of schooling indicator takes a value of "1" if at least someone 
	in the hh has reported 6 years of education or more */
replace years_edu6 = . if eduyears==.
bysort hh_id: egen hh_years_edu6_1 = max(years_edu6)
gen	hh_years_edu6 = (hh_years_edu6_1==1)
replace hh_years_edu6 = . if hh_years_edu6_1==.
replace hh_years_edu6 = . if hh_years_edu6==0 & no_missing_edu==0 
	//Final variable missing if household has info for < 2/3 of members 
lab var hh_years_edu6 "Household has at least one member with 6 years of edu"
tab hh_years_edu6, miss

	
*** Destitution MPI ***
/*The entire household is considered deprived if no household member 
aged 10 years or older has completed at least one year of schooling.*/
******************************************************************* 
gen	years_edu1 = (eduyears>=1)
replace years_edu1 = . if eduyears==.
bysort	hh_id: egen hh_years_edu_u = max(years_edu1)
replace hh_years_edu_u = . if hh_years_edu_u==0 & no_missing_edu==0
lab var hh_years_edu_u "Household has at least one member with 1 year of edu"
tab hh_years_edu_u, miss


********************************************************************************
*** Step 2.2 Child School Attendance ***
********************************************************************************
codebook attendance, tab (10)


*** Standard MPI ***
/*The entire household is considered deprived if any school-aged child is not 
attending school up to class 8. */ 
******************************************************************* 
	/*Note: The UNESCO website (http://data.uis.unesco.org/?ReportId=163) is the 
	source of reference for start age of compulsory schooling at the country 
	level. The website indicates that compulsory schooling age in China begins 
	at the age of 6 years. However, in reality most children attend school at 
	the age of 7. As such, in the case of China, we follow 7 years rather than 
	6 years as the start age of compulsory schooling. 
	
	In addition, in Chinese culture, parents prefer to identify their child as 
	7 just after their child had/passed their 6th birthday (chinese calender).
	This is the reason we have age_year = 7 while their age_month is below 84. 
	By using age, it will include children whom seem eligible but they are 
	actually not attending school. It is preferable to use age in months to 
	identify eligible children of school age. The decision to use age in years
	for the standardised indicator of the global MPI was opted by HDRO team.*/
	
tab attendance if age>=7 & age<=15,miss

/*
     Attending |
        school |      Freq.     Percent        Cum.
---------------+-----------------------------------
 Not attending |        338        7.42        7.42
Yes, attending |      4,105       90.14       97.56
             . |        111        2.44      100.00
---------------+-----------------------------------
         Total |      4,554      100.00
*/


tab attendance if age_month>=84 & age_month<192,miss

/*
     Attending |
        school |      Freq.     Percent        Cum.
---------------+-----------------------------------
 Not attending |        245        5.44        5.44
Yes, attending |      4,257       94.54       99.98
             . |          1        0.02      100.00
---------------+-----------------------------------
         Total |      4,503      100.00
*/	
	

gen	child_schoolage = (age>=7 & age<=15)
	/*Note: In China, the official school entrance age is 7 years.  
	  So, age range is 7-15 (=7+8) or in months it is 84-191*/

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the school age children */
count if child_schoolage==1 & attendance==.
	//How many eligible school aged children are not attending school:111 child 
gen temp = 1 if child_schoolage==1 & attendance!=.
	/*Generate a variable that captures the number of eligible school aged 
	children who are attending school */
bysort hh_id: egen no_missing_atten = sum(temp)	
	/*Total school age children with no missing information on school 
	attendance */
gen temp2 = 1 if child_schoolage==1	
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are of school age
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)
	/*Identify whether there is missing information on school attendance for 
	more than 2/3 of the school age children */			
tab no_missing_atten, miss
	//Values for 0 (missing) is 0.01%
label var no_missing_atten "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs
		
bysort hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
	//Control variable: 
	//It takes value 1 if the household has children in school age
lab var hh_children_schoolage "Household has children in school age"


gen	child_not_atten = (attendance==0) if child_schoolage==1
replace child_not_atten = . if attendance==. & child_schoolage==1
bysort	hh_id: egen any_child_not_atten = max(child_not_atten)
gen	hh_child_atten = (any_child_not_atten==0) 
replace hh_child_atten = . if any_child_not_atten==.
replace hh_child_atten = 1 if hh_children_schoolage==0
replace hh_child_atten = . if hh_child_atten==1 & no_missing_atten==0 
	/*If the household has been intially identified as non-deprived, but has 
	missing school attendance for at least 2/3 of the school aged children, then 
	we replace this household with a value of '.' because there is insufficient 
	information to conclusively conclude that the household is not deprived */
lab var hh_child_atten "Household has all school age children up to class 8 in school"
tab hh_child_atten, miss



	/*For the purpose of harmonising with CFPS 2010, we construct the school 
	age variable using age in months instead of age in years as the former 
	is comparable to the 2010 survey.*/ 
	
gen	child_schoolage_c = (age_month>=84 & age_month<192)
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the school age children */
count if child_schoolage_c==1 & attendance==.
gen temp = 1 if child_schoolage_c==1 & attendance!=.
bysort hh_id: egen no_missing_atten_c = sum(temp)	
gen temp2 = 1 if child_schoolage_c==1	
bysort hh_id: egen hhs = sum(temp2)
replace no_missing_atten_c = no_missing_atten_c/hhs 
replace no_missing_atten_c = (no_missing_atten_c>=2/3)		
tab no_missing_atten_c, miss
label var no_missing_atten_c "No missing school attendance for at least 2/3 of the school aged children,c"		
drop temp temp2 hhs
		
bysort hh_id: egen hh_children_schoolage_c = sum(child_schoolage_c)
replace hh_children_schoolage_c = (hh_children_schoolage_c>0) 
lab var hh_children_schoolage_c "Household has children in school age,c"

gen	child_not_atten_c = (attendance==0) if child_schoolage_c==1
replace child_not_atten_c = . if attendance==. & child_schoolage_c==1
bysort	hh_id: egen any_child_not_atten_c = max(child_not_atten_c)
gen	hh_child_atten_c = (any_child_not_atten_c==0) 
replace hh_child_atten_c = . if any_child_not_atten_c==.
replace hh_child_atten_c = 1 if hh_children_schoolage_c==0
replace hh_child_atten_c = . if hh_child_atten_c==1 & no_missing_atten_c==0 
lab var hh_child_atten_c "Household has all school age children up to class 8 in school,c"
tab hh_child_atten_c, miss


*** Destitution MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 6. */ 
*******************************************************************  
gen	child_schoolage_6 = (age>=7 & age<=13)
	/*Note: In China, the official school entrance age is 7 years.  
	  So, age range is 7-13 (=7+6) or in months it is 84-168 */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the children attending school up to 
	class 6 */	
count if child_schoolage_6==1 & attendance==.
	//How many eligible school aged children are not attending school: 0 child 	
gen temp = 1 if child_schoolage_6==1 & attendance!=.
	/*Generate a variable that captures the number of eligible school aged 
	children who are attending school */
bysort hh_id: egen no_missing_atten_u = sum(temp)	
	/*Total school age children attending up to class 6 with no missing 
	information on school attendance */
gen temp2 = 1 if child_schoolage_6==1	
bysort hh_id: egen hhs = sum(temp2)
	/*Total number of household members who are of school age attending up to 
	class 6 */
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)
	/*Identify whether there is missing information on school attendance for 
	more than 2/3 of the school age children attending up to class 6 */			
tab no_missing_atten_u, miss
	//Values for 0 (missing) is 0%
label var no_missing_atten_u "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs		
	
	
bysort	hh_id: egen hh_children_schoolage_6 = sum(child_schoolage_6)
replace hh_children_schoolage_6 = (hh_children_schoolage_6>0) 
lab var hh_children_schoolage_6 "Household has children in school age (6 years of school)"

gen	child_atten_6 = (attendance==1) if child_schoolage_6==1
replace child_atten_6 = . if attendance==. & child_schoolage_6==1
bysort	hh_id: egen any_child_atten_6 = max(child_atten_6)
gen	hh_child_atten_u = (any_child_atten_6==1) 
replace hh_child_atten_u = . if any_child_atten_6==.
replace hh_child_atten_u = 1 if hh_children_schoolage_6==0
replace hh_child_atten_u = . if hh_child_atten_u==0 & no_missing_atten_u==0 
	/*If the household has been intially identified as deprived, but has 
	missing school attendance for at least 2/3 of the school aged children, then 
	we replace this household with a value of '.' because there is insufficient 
	information to conclusively conclude that the household is deprived */
lab var hh_child_atten_u "Household has at least one school age children up to class 6 in school"
tab hh_child_atten_u, miss



	/*For the purpose of harmonising with CFPS 2010, we construct the school 
	age (destitution) variable using age in months instead of age in years 
	as the former is comparable to the 2010 survey.*/ 
	
gen	child_schoolage_6_c = (age_month>=84 & age_month<168)
	/*Note: In China, the official school entrance age is 7 years.  
	  So, age range is 7-13 (=7+6) or in months it is 84-168 */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the children attending school up to 
	class 6 */	
count if child_schoolage_6_c==1 & attendance==.
gen temp = 1 if child_schoolage_6_c==1 & attendance!=.
bysort hh_id: egen no_missing_atten_u_c = sum(temp)	
gen temp2 = 1 if child_schoolage_6_c==1	
bysort hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u_c = no_missing_atten_u_c/hhs 
replace no_missing_atten_u_c = (no_missing_atten_u_c>=2/3)		
tab no_missing_atten_u_c, miss
label var no_missing_atten_u_c "No missing school attendance for at least 2/3 of the school aged children,c"		
drop temp temp2 hhs		
		
bysort	hh_id: egen hh_children_schoolage_6_c = sum(child_schoolage_6_c)
replace hh_children_schoolage_6_c = (hh_children_schoolage_6_c>0) 
lab var hh_children_schoolage_6_c "Household has children in school age (6 years of school),c"

gen	child_atten_6_c = (attendance==1) if child_schoolage_6_c==1
replace child_atten_6_c = . if attendance==. & child_schoolage_6_c==1
bysort	hh_id: egen any_child_atten_6_c = max(child_atten_6_c)
gen	hh_child_atten_u_c = (any_child_atten_6_c==1) 
replace hh_child_atten_u_c = . if any_child_atten_6_c==.
replace hh_child_atten_u_c = 1 if hh_children_schoolage_6_c==0
replace hh_child_atten_u_c = . if hh_child_atten_u_c==0 & no_missing_atten_u_c==0 
lab var hh_child_atten_u_c "Household has at least one school age children up to class 6 in school,c"
tab hh_child_atten_u_c, miss


********************************************************************************
*** Step 2.3 Nutrition ***
********************************************************************************
	/*Note: China CFPS 2014: Anthropometric information were recorded for all 
	individuals aged 0-98 years (CFPS 2014 Questionnaire). This departs from the 
	usual DHS surveys that tend to collect anthropometric data only from adults 
	between the age group of 15-49 or 15-59 years. For the purpose of the global 
	MPI, we have used nutrition data when the data is available for all 
	individuals but up to the age of 70 years. Hence, in the case of China, we 
	make use of the anthropometric data for children 0-60 months, young 
	individuals 61-228 months, and finally adults 229-840 months.*/ 

count if age_month!=.
	//42,089 individuals 

count if age_month<=840
	//39,541 individuals aged 0-70 years 
	
	
***As a first step, construct the eligibility criteria	
*** No Eligible Women, Men or Children for Nutrition
***********************************************	
gen nutri_eligible = age_month<=840
bysort	hh_id: egen n_nutri_eligible = sum(nutri_eligible) 	
gen	no_nutri_eligible = (n_nutri_eligible==0) 	
lab var no_nutri_eligible "Household has no eligible women, men, or children"
tab no_nutri_eligible, miss	

drop nutri_eligible n_nutri_eligible
	
	

********************************************************************************
*** Step 2.3a Nutrition for Adults 229-840 months in age ***
********************************************************************************


*** Standard MPI: BMI Indicator for Adults ***
******************************************************************* 
tab low_bmi if age_month>=229 & age_month<=840, miss
	//29,445 adults with BMI indicator (41 missing cases)
gen low_bmi_20 = low_bmi if age_month>=229 & age_month<=840
	/*In the context of CFPS 2014, we focus on BMI measure for individuals 
	229-840 months in age because BMI-for-age is applied for individuals 60-228 
	months in age */
bysort hh_id: egen temp = max(low_bmi_20)
tab temp, miss 
gen hh_no_low_bmi = (temp==0) 
	/*Under this section, households take a value of '1' if no adults in the 
	household has low bmi */
replace hh_no_low_bmi = . if temp==.
	/*Under this section, households take a value of '.' if there is no 
	information from adults*/
drop temp
lab var hh_no_low_bmi "Household has no adult with low BMI"
tab hh_no_low_bmi if age_month>=229 & age_month<=840, miss


*** Destitute MPI: BMI Indicator for Adults ***
******************************************************************* 
tab low_bmi_u if age_month>=229 & age_month<=840, miss
gen low_bmi_20_u = low_bmi_u if age_month>=229 & age_month<=840
	/*In the context of CFPS 2014, we focus on BMI measure for individuals 
	229-840 months in age because BMI-for-age is applied for individuals 
	60-228 months in age */	
bysort hh_id: egen temp = max(low_bmi_20_u)
tab temp, miss 
gen hh_no_low_bmi_u = (temp==0) 
	/*Under this section, households take a value of '1' if no adults in the 
	household has destitute bmi */
replace hh_no_low_bmi_u = . if temp==.
	/*Under this section, households take a value of '.' if there is no 
	information from adults*/	
drop temp
lab var hh_no_low_bmi_u "Household has no adult with low BMI (<17)"	
tab hh_no_low_bmi_u if age_month>=229 & age_month<=840, miss

		
********************************************************************************
*** Step 2.3b Nutrition for Young Children 60-228 months in age ***
********************************************************************************

*** Standard MPI: BMI-for-age for Young Children ***
******************************************************************* 	
count if age_month>60 & age_month<229
count if age_month>60 & age_month<229 & teen_IR==1
	//6,945 individuals who are 60-228 months in age 
	
tab low_bmiage, miss	
tab low_bmiage if teen_IR==1, miss	

bysort hh_id: egen temp = max(low_bmiage)  
tab temp, miss
gen	hh_no_low_bmiage = (temp==0) 
	/*Takes value 1 if no individuals who are 61-228 months in age in the 
	household has low bmi-for-age*/
replace hh_no_low_bmiage = . if temp==.
drop temp
lab var hh_no_low_bmiage "Household has no adult with low BMI-for-age"
tab hh_no_low_bmiage if age_month>60 & age_month<229, miss

tab low_bmi if hh_no_low_bmiage==. & age_month>60 & age_month<229,miss
	/* Note: 313 young children have missing nutrition information. However, 
	117 of these 313 children have BMI information. So we have used the BMI 
	information as the next best option for these young children. */
replace hh_no_low_bmiage=0 if low_bmi==1 & hh_no_low_bmiage==. & ///
							   age_month>60 & age_month<229
replace hh_no_low_bmiage=1 if low_bmi==0 & hh_no_low_bmiage==. & ///
							   age_month>60 & age_month<229
	

*** Destitute MPI: BMI-for-age for Young Children ***
******************************************************************* 
tab low_bmiage_u if age_month>60 & age_month<229, miss	
tab low_bmiage_u if teen_IR==1, miss	

bys	hh_id: egen temp = max(low_bmiage_u)  
tab temp, miss
gen	hh_no_low_bmiage_u = (temp==0) 
	/*The destitute measure takes a value of '1' if no individuals who are 
	61-228 months in age in the household has low bmi-for-age */
replace hh_no_low_bmiage_u = . if temp==.
drop temp
lab var hh_no_low_bmiage_u "Household has no adult with low BMI-for-age(-3sd)"
tab hh_no_low_bmiage_u if age_month>60 & age_month<229, miss

tab low_bmi_u if hh_no_low_bmiage_u==. & age_month>60 & age_month<229,miss
	/* Note: 313 young children have missing nutrition information. However, 
	117 of these 313 children have BMI information. So we have used the BMI 
	information as the next best option for these children. */
replace hh_no_low_bmiage_u=0 if low_bmi_u==1 & hh_no_low_bmiage_u==. & ///
							   age_month>60 & age_month<229
replace hh_no_low_bmiage_u=1 if low_bmi_u==0 & hh_no_low_bmiage_u==. & ///
							   age_month>60 & age_month<229	

	
********************************************************************************
*** Step 2.3c Child Nutrition ***
********************************************************************************


***As a first step, construct the eligibility criteria for children 0-60 months
*** No Eligible Children for Nutrition
***********************************************	
gen child_eligible = age_month<=60
bysort	hh_id: egen n_child_eligible = sum(child_eligible) 	
gen	no_child_eligible = (n_child_eligible==0) 	
lab var no_child_eligible "Household has no eligible children <59"
tab no_child_eligible, miss	

drop n_child_eligible


*** Standard MPI: Child Underweight Indicator ***
************************************************************************
tab underweight if child_KR==1, miss	
tab underweight if age_month<=60, miss	

bysort hh_id: egen temp = max(underweight)
tab temp, miss
gen	hh_no_underweight = (temp==0) 
	//Takes a value of '1' if no child in the household is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
drop temp
lab var hh_no_underweight "Household has no child underweight - 2 stdev"
tab hh_no_underweight, miss


*** Standard MPI: Child Stunting Indicator ***
************************************************************************
tab stunting if child_KR==1, miss
tab stunting if age_month<=60, miss

bysort hh_id: egen temp = max(stunting)
tab temp, miss
gen	hh_no_stunting = (temp==0) 
	//Takes a value of '1' if no child in the household is stunted
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
drop temp
lab var hh_no_stunting "Household has no child stunted - 2 stdev"
tab hh_no_stunting, miss


*** Standard MPI: Child Wasting Indicator ***
************************************************************************
tab wasting if child_KR==1, miss
tab wasting if age_month<=60, miss

bysort hh_id: egen temp = max(wasting)
tab temp, miss
gen	hh_no_wasting = (temp==0) 
	//Takes a value of '1' if no child in the household is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
drop temp
lab var hh_no_wasting "Household has no child wasted - 2 stdev"
tab hh_no_wasting, miss


*** Standard MPI: Child Either Stunted or Underweight Indicator ***
************************************************************************
gen uw_st = 1 if stunting==1 | underweight==1
replace uw_st = 0 if stunting==0 & underweight==0
replace uw_st = . if stunting==. & underweight==.

tab uw_st if child_KR==1, miss
tab uw_st if age_month<=60, miss

bysort hh_id: egen temp = max(uw_st)
tab temp, miss
gen	hh_no_uw_st = (temp==0) 
	/*Takes a value of '1' if no child in the household is underweight or 
	stunted */
replace hh_no_uw_st = . if temp==.
replace hh_no_uw_st = 1 if no_child_eligible==1
	//Households with no eligible children will receive a value of 1
drop temp
lab var hh_no_uw_st "Household has no child underweight or stunted"
tab hh_no_uw_st, miss


*** Destitute MPI: Child Underweight Indicator ***
************************************************************************
tab underweight_u if child_KR==1, miss	

bysort hh_id: egen temp = max(underweight_u)
tab temp, miss
gen	hh_no_underweight_u = (temp==0) 
	//Takes a value of '1' if no child in the household is severely underweight 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
drop temp
lab var hh_no_underweight_u "Destitute: Household has no child underweight"
tab hh_no_underweight_u, miss


*** Destitute MPI: MPI Child Stunting Indicator ***
************************************************************************
tab stunting_u if child_KR==1, miss

bysort hh_id: egen temp = max(stunting_u)
tab temp, miss
gen	hh_no_stunting_u = (temp==0) 
	//Takes a value of '1' if no child in the household is severely stunted
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1 
drop temp
lab var hh_no_stunting_u "Destitute: Household has no child stunted"
tab hh_no_stunting_u, miss


*** Destitute MPI: MPI Child Wasting Indicator ***
************************************************************************
tab wasting_u if child_KR==1, miss

bysort hh_id: egen temp = max(wasting_u)
tab temp, miss
gen	hh_no_wasting_u = (temp==0) 
	//Takes a value of '1' if no child in the household is severely stunted
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1 
drop temp
lab var hh_no_wasting_u "Destitute: Household has no child wasted"
tab hh_no_wasting_u, miss


*** Destitute MPI: Child Either Underweight or Stunted Indicator ***
************************************************************************
gen uw_st_u = 1 if stunting_u==1 | underweight_u==1
replace uw_st_u = 0 if stunting_u==0 & underweight_u==0
replace uw_st_u = . if stunting_u==. & underweight_u==.
	
tab uw_st_u if child_KR==1, miss

bysort hh_id: egen temp = max(uw_st_u)
tab temp, miss
gen	hh_no_uw_st_u = (temp==0) 
	/*Takes a value of '1' if no child in the household is severely underweight 
	or stunted */
replace hh_no_uw_st_u = . if temp==.
replace hh_no_uw_st_u = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1 
drop temp
lab var hh_no_uw_st_u "Destitute: Household has no child underweight or stunted"
tab hh_no_uw_st_u, miss


********************************************************************************
*** Step 2.3d Household Nutrition Indicator ***
********************************************************************************

********************************************************************************
*** Step 2.3d Household Nutrition Indicator ***
********************************************************************************

*** Standard MPI ***
/* The indicator takes value 1 if the household has no child under 5 who 
has either height-for-age or weight-for-age that is under 2 stdev below 
the median. It also takes value 1 for the households that have no eligible 
children. The indicator takes a value of missing only if all eligible 
children have missing information in their respective nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st = 1
replace hh_nutrition_uw_st = 0 if hh_no_low_bmi==0 | hh_no_low_bmiage==0 | ///
								  hh_no_uw_st==0
replace hh_nutrition_uw_st = . if hh_no_low_bmi==. & hh_no_low_bmiage==. & ///
								  hh_no_uw_st==.	
replace hh_nutrition_uw_st = 1 if no_nutri_eligible==1
 	/*We replace households that do not have the applicable population, that is, 
	individuals 0-840 months in age, as non-deprived in nutrition*/	
lab var hh_nutrition_uw_st "Household has no child underweight/stunted or adult deprived by BMI/BMI-for-age"
lab value hh_nutrition_uw_st lab_nutri 
tab hh_nutrition_uw_st, miss


*** Destitution MPI ***
/* The indicator takes value 1 if the household has no child under 5 who 
has either height-for-age or weight-for-age that is under 2 stdev below 
the median. It also takes value 1 for the households that have no eligible 
children. The indicator takes a value of missing only if all eligible 
children have missing information in their respective nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st_u = 1
replace hh_nutrition_uw_st_u = 0 if hh_no_low_bmi_u==0 | ///	
									hh_no_low_bmiage_u==0 | hh_no_uw_st_u==0
replace hh_nutrition_uw_st_u = . if hh_no_low_bmi_u==. & ///
									hh_no_low_bmiage_u==. & hh_no_uw_st_u==.
replace hh_nutrition_uw_st_u = 1 if no_nutri_eligible==1
 	/*We replace households that do not have the applicable population, that is, 
	individuals 0-840 months in age, as non-deprived in nutrition*/		 	
lab var hh_nutrition_uw_st_u "Household has no child underweight/stunted or adult deprived by BMI/BMI-for-age (destitute)"
lab value hh_nutrition_uw_st_u lab_nutri 
tab hh_nutrition_uw_st_u, miss


********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************
		
*** Standard MPI *** 
/*Deprived if children under 18 years (<216 months) died in the household 
in the last 5 years from the survey year */
************************************************************************

/*In CFPS 2010, there is no birth history data. We can identify all child 
mortality that was reported within household. But we can't identify when 
the death occured or the age of the child who died. However, in CFPS 2014, 
information on time of death and age of child who died is available. For the 
purpose of harmonising the data between 2010 and 2014, we only compare all 
reported child  mortality within household between both years. As such, we have
generated the hh_mortality_c variable for COT work.*/

tab dead_children_temp,miss
gen	hh_mortality = (dead_children_temp==0)
	/*Household is replaced with a value of "1" if there is no incidence of 
	child mortality*/
replace hh_mortality = . if dead_children_temp==.	
lab var hh_mortality "Household had no child mortality"
tab hh_mortality, miss

clonevar hh_mortality_c = hh_mortality

forvalues i = 1(1)10 {
gen mortality_5y_`i' = 1     if ta4y_a14_c`i'>=2010 & ta4y_a14_c`i'!=. & cyear==2015
replace mortality_5y_`i' = 0 if ta4y_a14_c`i'<2010  & ta4y_a14_c`i'==. & cyear==2015         
replace mortality_5y_`i' = 1 if ta4y_a14_c`i'>=2009 & ta4y_a14_c`i'!=. & cyear==2014
replace mortality_5y_`i' = 1 if ta4y_a14_c`i'<2009  & ta4y_a14_c`i'==. & cyear==2014
replace mortality_5y_`i' = . if ta4y_a14_c`i'==.
}	
egen dead_children_temp_5y = rowtotal(mortality_5y_1 mortality_5y_2 ///
								      mortality_5y_3 mortality_5y_4 ///
								      mortality_5y_5 mortality_5y_6 ///
								      mortality_5y_7 mortality_5y_8 ///
								      mortality_5y_9 mortality_5y_10)
									  
tab dead_children_temp_5y, miss	



forvalues i = 1(1)10 {
gen mortality_u185y_`i' = 1     if ta4y_a14_c`i'>=2010 & ta4y_a14_c`i'!=. & cyear==2015 & age_deadm_`i'>=0 & age_deadm_`i'<216
replace mortality_u185y_`i' = 0 if ta4y_a14_c`i'<2010  & ta4y_a14_c`i'==. & cyear==2015 & age_deadm_`i'>=0 & age_deadm_`i'<216       
replace mortality_u185y_`i' = 1 if ta4y_a14_c`i'>=2009 & ta4y_a14_c`i'!=. & cyear==2014 & age_deadm_`i'>=0 & age_deadm_`i'<216
replace mortality_u185y_`i' = 1 if ta4y_a14_c`i'<2009  & ta4y_a14_c`i'==. & cyear==2014 & age_deadm_`i'>=0 & age_deadm_`i'<216
replace mortality_u185y_`i' = . if ta4y_a14_c`i'==.
}	
egen dead_children_temp_u185y = 	  rowtotal(mortality_u185y_1 mortality_u185y_2 ///
								      mortality_u185y_3 mortality_u185y_4 ///
								      mortality_u185y_5 mortality_u185y_6 ///
								      mortality_u185y_7 mortality_u185y_8 ///
								      mortality_u185y_9 mortality_u185y_10)
									  
tab dead_children_temp_u185y, miss	


bysort hh_id: egen childu18_mortality_5y = max(dead_children_temp_u185y) 										
lab var childu18_mortality_5y "Total under 18 child mortality within household past 5 years"
tab childu18_mortality_5y, miss


gen hh_mortality_u18_5y = (childu18_mortality_5y==0) 
lab var hh_mortality_u18_5y "Household had no under 18 child mortality in the last 5 years"
tab hh_mortality_u18_5y, miss


*** Destitution MPI *** 
*** (same as standard MPI) ***
************************************************************************

gen hh_mortality_u = hh_mortality_u18_5y	
lab var hh_mortality_u "Household had no under 18 child mortality in the last 5 years"	
	
	
	
/*In CFPS 2010, there is no birth history data. We can identify all child 
mortality that was reported within household. But we can't identify when 
the death occured or the age of the child who died. However, in CFPS 2014, 
information on time of death and age of child who died is available. For the 
purpose of harmonising the data between 2010 and 2014, we only compare all 
reported child  mortality within household between both years. As such, we have
generated the hh_mortality_u_c variable for changes over time work.  */
gen hh_mortality_u_c = hh_mortality_c	
	
	
	
	
********************************************************************************
*** Step 2.5 Electricity ***
********************************************************************************

*** Standard MPI ***
/*Members of the household are considered 
deprived if the household has no electricity */
***************************************************
gen electricity = fa5
label define lab_elec  1 "No electricity" ///
					   2 "Frequent power outage" ///
					   3 "Occasional power outage" ///
					   4 "Almost no power outage at all" 
label values fa5 lab_elec
codebook fa5, tab (10)
replace electricity = 0 if electricity==1
replace electricity = 1 if electricity>1 & electricity!=.
replace electricity = . if fa5<0 
tab fa5 electricity, miss 
label var electricity "Household has electricity"


*** Destitution MPI  ***
*** (same as standard MPI) ***
***************************************************
gen electricity_u = electricity
label var electricity_u "Household has electricity"



********************************************************************************
*** Step 2.6 Sanitation ***
********************************************************************************
/*
Improved sanitation facilities include flush or pour flush toilets to sewer 
systems, septic tanks or pit latrines, ventilated improved pit latrines, pit 
latrines with a slab, and composting toilets. These facilities are only 
considered improved if it is private, that is, it is not shared with other 
households.
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-02-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/


clonevar toilet = fa8 
codebook toilet, tab(99)
label var toilet "Source of drinking water"


	/*There is no observation for shared toilet for CFPS 2010 & 2014.
	However, it is possible to identify from type of restroom/toilet whether 
	it is shared or private*/
gen shared_toilet =.


*** Standard MPI ***
/*Members of the household are considered deprived if the household's 
sanitation facility is not improved (according to the SDG guideline) 
or it is improved but shared with other households*/
********************************************************************
gen	toilet_mdg = (toilet==1 | toilet==2 | toilet==4 | toilet==5) 
	/*Household is assigned a value of '1' if it uses improved sanitation and 
	does not share toilet with other households  */
	
replace toilet_mdg = . if toilet==.  
	//Household is assigned a value of '.' if it has missing information 	

label define lab_toilet  1 "Improved sanitation" ///
						 0 "Non-improved sanitation" 
label values toilet_mdg lab_toilet	
lab var toilet_mdg "Household has improved sanitation with MDG Standards"
tab toilet toilet_mdg, miss


*** Destitution MPI ***
/*Members of the household are considered deprived if household practises 
open defecation or uses other unidentifiable sanitation practises */
********************************************************************
gen	toilet_u = 1

replace toilet_u = 0 if toilet==77 
	/*Household is assigned a value of '0' if it uses other unidentified
	type of toilet */
replace toilet_u = . if toilet==.  
	//Household is assigned a value of '.' if it has missing information 
	
label values toilet_u lab_toilet		
lab var toilet_u "Household does not practise open defecation or others"
tab toilet toilet_u, miss


********************************************************************************
*** Step 2.7 Drinking Water  ***
********************************************************************************
/*
Improved drinking water sources include the following: piped water into 
dwelling, yard or plot; public taps or standpipes; boreholes or tubewells; 
protected dug wells; protected springs; packaged water; delivered water and 
rainwater which is located on premises or is less than a 30-minute walk from 
home roundtrip. 
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-01-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

clonevar water = fa3   
codebook water, tab(99)
label define lab_water -1 "Unknown" ///
					   -2 "Refuse" ///
						1 "River/Lake water" ///
						2 "Well" 3 "Tap water" ///
						4 "Mineral/Purified/Filtered water" ///
						5 "Rainwater" ///
						6 "Cellar water" ///
						7 "Pond water/Spring water" ///
						77 "Other" 
label values water lab_water
label var water "Source of drinking water"
tab water, miss

	//NOTE: CFPS 2014 has no information on time to water 
gen timetowater = .

	//NOTE: CFPS 2014 has no information on non-drinking water
gen ndwater = .  

	
*** Standard MPI ***
/* Members of the household are considered deprived if the household 
does not have access to improved drinking water (according to the SDG 
guideline) or safe drinking water is at least a 30-minute walk from 
home, roundtrip */
********************************************************************

gen water_mdg = (fa3==3 | fa3==4 | fa3==5)
	/*Non deprived if water is "tap water", "mineral/purified/filtered water", 
								"rainwater"
	      Deprived if water is "river/lake water", "well", "cellar water"
						       "pond water/spring water", ""other" */
replace water_mdg = . if fa3==. | fa3==-1 | fa3==-2
lab var water_mdg "Household has drinking water with MDG standards"
tab water water_mdg, miss


*** Destitution MPI ***
/* Members of the household is identified as destitute if household 
does not have access to safe drinking water, or safe water is more 
than 45 minute walk from home, round trip.*/
********************************************************************
/*Note: Since CFPS 2014 has no observatin on time to water, hence, the 
destitute MPI will be the same as the standard MPI.  */

gen	water_u = water_mdg
lab var water_u "Household has drinking water with MDG standards"


********************************************************************************
*** Step 2.8 Housing ***
********************************************************************************
	/*Note: CFPS 2014 has no data on housing (floor, wall, or roof). 
	For the work related to the global MPI 2018, housing is replaced 
	as missing observation. This means the weight for the living standards 
	dimension (1/3) is equally distributed among the remaining 5 indicators: 
	electricity (1/15); sanitation (1/15); drinking water (1/15); 
	cooking fuel (1/15) & assets (1/15). */



gen floor = .
gen	floor_imp = .	
lab var floor_imp "Household has floor that it is not earth/sand/dung"

gen wall = .
gen	wall_imp = .	
lab var wall_imp "Household has wall that it is not of low quality materials"

gen roof = .
gen	roof_imp = .	
lab var roof_imp "Household has roof that it is not of low quality materials"

gen housing_1 = .
lab var housing_1 "Household has roof, floor & walls that it is not low quality material"

gen housing_u = .
lab var housing_u "Household has roof, floor or walls that it is not low quality material"


********************************************************************************
*** Step 2.9 Cooking Fuel ***
********************************************************************************
/*
Solid fuel are solid materials burned as fuels, which includes coal as well as 
solid biomass fuels (wood, animal dung, crop wastes and charcoal). 

Source: 
https://apps.who.int/iris/bitstream/handle/10665/141496/9789241548885_eng.pdf
*/

clonevar cookingfuel = fa4 
codebook cookingfuel, tab(20)
label define lab_fuel -1 "Unknown" ///
					  -2 "Refuse" ///
					   1 "Firewood/straw" ///
					   2 "Coal" ///
					   3 "Gas/Liquid" ///
					   4 "Natural gas/Pipe-line gas" ///
					   5 "Solar energy/Methane" ///
					   6 "Electricity" ///
					   77 "Other" 
label values cookingfuel lab_fuel
label var cookingfuel "Type of cooking fuel"
tab cookingfuel, miss


*** Standard MPI ***
/* Members of the household are considered deprived if the 
household uses solid fuels and solid biomass fuels for cooking. */
*****************************************************************
gen	cooking_mdg = 1
replace cooking_mdg = 0 if cookingfuel==77 | (cookingfuel>0 & cookingfuel<=2)
	/*Note: In China, households that responded as using other cooking fuel is 
	considered as deprived, following discussion with data provider */	
replace cooking_mdg = . if cookingfuel<0 | cookingfuel == .
lab var cooking_mdg "Household has cooking fuel by MDG standards"				
tab cookingfuel cooking_mdg, miss	


*** Destitution MPI ***
*** (same as standard MPI) ***
****************************************
gen	cooking_u = cooking_mdg
lab var cooking_u "Household uses clean fuels for cooking"


********************************************************************************
*** Step 2.10 Assets ownership ***
********************************************************************************

*** Television/LCD TV/plasma TV/color TV/black & white tv
gen tv=1 if fs6_s_1==6
forval i = 2(1)15 {
replace  tv=1 if fs6_s_`i'==6
}
replace tv=0 if tv==.
gen television = tv
replace television=. if fs6_s_1==.

***	Radio/walkman/stereo/kindle
gen radio = .

***	Handphone/telephone/iphone/mobilephone/ipod
gen telephone = .
gen mobiletelephone=1 if fs6_s_1==12
forval i = 2(1)15 {
replace  mobiletelephone=1 if fs6_s_`i'==12
}
replace mobiletelephone=0 if mobiletelephone==.
replace mobiletelephone=. if fs6_s_1==.
replace telephone = mobiletelephone


***	Refrigerator/icebox/fridge
gen refrigerator=1 if fs6_s_1==4
forval i = 2(1)15 {
replace  refrigerator=1 if fs6_s_`i'==4
}
replace refrigerator=0 if refrigerator==.
replace refrigerator=. if fs6_s_1==.

gen refrigerator_c = .


***	Car/van/lorry/truck
gen car=1 if fs6_s_1==1
replace car=1 if fp507>0 & fp507!=.  
	// fp504 "how much money do you spend on your car?"
replace car=0 if car==.
forval i = 2(1)15 {
replace  car=1 if fs6_s_`i'==1
}
replace car=. if fs6_s_1==.

***	Bicycle/cycle rickshaw
gen bicycle=1 if fs6_s_1==2
forval i = 2(1)15 {
replace  bicycle=1 if fs6_s_`i'==2
}
replace bicycle=0 if bicycle==.
replace bicycle=. if fs6_s_1==.


gen bicycle_c = .

***	Motorbike/motorized bike/autorickshaw/autobike
gen     motorbike=1 if fs6_s_1==3
forval i = 2(1)15 {
replace  motorbike=1 if fs6_s_`i'==3
}
replace motorbike=0 if motorbike==.
replace motorbike=. if fs6_s_1==.

gen motorbike_c = .


***	Computer/laptop/tablet
gen     computer=1 if fs6_s_1==7
forval i = 2(1)15 {
replace computer=1 if fs6_s_`i'==7
}
replace computer=0 if computer==.
replace computer=. if fs6_s_1==.

gen computer_c = .

***	Animal cart
gen animal_cart = .


	//label indicators
lab var television "Household has television"
lab var radio "Household has radio"	
lab var telephone "Household has telephone (landline/mobilephone)"	
lab var refrigerator "Household has refrigerator"
lab var car "Household has car"
lab var bicycle "Household has bicycle"	
lab var motorbike "Household has motorbike"
lab var computer "Household has computer"
lab var animal_cart "Household has animal cart"


*** Standard MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own more than one of: radio, TV, telephone, bike, motorbike, 
refrigerator, computer or animal cart and does not own a car or truck.*/
*****************************************************************************
egen n_small_assets2 = rowtotal(television radio telephone refrigerator bicycle motorbike computer animal_cart), missing
lab var n_small_assets2 "Household Number of Small Assets Owned" 
  
gen hh_assets2 = (car==1 | n_small_assets2 > 1) 
replace hh_assets2 = . if car==. & n_small_assets2==.
lab var hh_assets2 "Household Asset Ownership: HH has car or more than 1 small assets incl computer & animal cart"


/*
CFPS 2010
No data for 6 of the 8 small assets: radio, animal cart, 
refrigerator, bicycle, motorbike, computer

CFPS 2014 
No data for 2 of the 8 small assets: radio, animal cart

Harmonised 2010-2014
The harmonised assets consist of 2 small items that is present in 2010 & 2014 
(television & telephone). Hence the uncensored headcount for assets is higher 
for harmonised global MPI (7.66%) compared to standardised global MPI (2.55%).
*/

egen n_small_assets2_c = rowtotal(television radio telephone refrigerator_c bicycle_c motorbike_c computer_c animal_cart), missing
gen hh_assets2_c = (car==1 | n_small_assets2_c > 1) 
replace hh_assets2_c = . if car==. & n_small_assets2_c==.


*** Destitution MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own any assets.*/
*****************************************************************************	
gen	hh_assets2_u = (car==1 | n_small_assets2>0)
replace hh_assets2_u = . if car==. & n_small_assets2==.
lab var hh_assets2_u "Household Asset Ownership: HH has car or at least 1 small assets incl computer & animal cart"

//Harmonised 2010-2014 
gen	hh_assets2_u_c = (car==1 | n_small_assets2_c>0)
replace hh_assets2_u_c = . if car==. & n_small_assets2_c==.


********************************************************************************
*** Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************


//Retain data on sampling design: 	
gen strata=subpopulation 
gen psu=cid14
*svyset psu [iweight=weight_ind], strata(strata) singleunit(centered) 
label var psu "Primary sampling unit"
label var strata "Sample strata"


//Retain year, month & date of interview:
	//The variable on date of interview was not found in the CFPS 2014 raw data
clonevar year_interview = cyear 	
clonevar month_interview = cmonth 
gen date_interview = .


//No presence of subsample
gen subsample=.


*** Keep main variables require for MPI calculation ***
keep pid fid14 hh_id ind_id subsample strata psu weight weight_ch ///
area sex age agec7 agec4 agec2 headship marital relationship ///
hhsize province no_nutri_eligible child_eligible no_child_eligible /// 
region region_01 year_interview month_interview date_interview ///
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
age_month low_bmiage low_bmiage_u weight_kg height_cm low_bmiage low_bmiage_u ///
hh_no_low_bmi hh_no_low_bmi_u hh_no_low_bmiage hh_no_low_bmiage_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
dead_children_temp_u185y childu18_mortality_5y hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet shared_toilet toilet_mdg toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u hh_child_atten_c hh_child_atten_u_c ///
hh_mortality_c hh_mortality_u_c hh_assets2_c hh_assets2_u_c

  	 
*** Order file	***
order pid fid14 hh_id ind_id subsample strata psu weight weight_ch ///
area sex age agec7 agec4 agec2 headship marital relationship ///
hhsize province no_nutri_eligible child_eligible no_child_eligible /// 
region region_01 year_interview month_interview date_interview ///
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
age_month low_bmiage low_bmiage_u weight_kg height_cm low_bmiage low_bmiage_u ///
hh_no_low_bmi hh_no_low_bmi_u hh_no_low_bmiage hh_no_low_bmiage_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
dead_children_temp_u185y childu18_mortality_5y hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet shared_toilet toilet_mdg toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u hh_child_atten_c hh_child_atten_u_c ///
hh_mortality_c hh_mortality_u_c hh_assets2_c hh_assets2_u_c


*** Rename key global MPI indicators for estimation ***
recode hh_mortality_u18_5y	(0=1)(1=0) , gen(d_cm)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ)
recode electricity 			(0=1)(1=0) , gen(d_elct)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani)
recode housing_1 			(0=1)(1=0) , gen(d_hsg)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl)
recode hh_assets2 		    (0=1)(1=0) , gen(d_asst)
 

*** Rename key global MPI indicators for destitution estimation ***
recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ)
recode electricity_u		(0=1)(1=0) , gen(dst_elct)
recode water_u 				(0=1)(1=0) , gen(dst_wtr)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl)
recode hh_assets2_u 		(0=1)(1=0) , gen(dst_asst) 


*** Rename indicators for changes over time estimation ***	
recode hh_mortality_c	    (0=1)(1=0) , gen(d_cm_01)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr_01)
recode hh_child_atten_c 	(0=1)(1=0) , gen(d_satt_01)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ_01)
recode electricity 			(0=1)(1=0) , gen(d_elct_01)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr_01)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani_01)
recode housing_1 			(0=1)(1=0) , gen(d_hsg_01)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl_01)
recode hh_assets2_c 		(0=1)(1=0) , gen(d_asst_01)
 

recode hh_mortality_u_c     (0=1)(1=0) , gen(dst_cm_01)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr_01)
recode hh_child_atten_u_c 	(0=1)(1=0) , gen(dst_satt_01)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ_01)
recode electricity_u		(0=1)(1=0) , gen(dst_elct_01)
recode water_u 				(0=1)(1=0) , gen(dst_wtr_01)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani_01)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg_01)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl_01)
recode hh_assets2_u_c 		(0=1)(1=0) , gen(dst_asst_01) 



*** Eligibility for years of schooling indicator ***
gen educ_elig = 1 
replace educ_elig = 0 if age < 10 
label define lab_educ_elig 0"ineligible" 1"eligible"  
label values educ_elig lab_educ_elig
lab var educ_elig "Individual is eligible for educ indicator"
ta eduyears educ_elig,m
	

*** Generate coutry and survey details for estimation ***
char _dta[cty] "China"
char _dta[ccty] "CHN"
char _dta[year] "2014" 	
char _dta[survey] "CFPS"
char _dta[ccnum] "156"
char _dta[type] "micro"



*** Sort, compress and save data for estimation ***
sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/chn_cfps14.dta", replace 
erase "$path_out/cfps2014.dta"
erase "$path_out/CHN14_children.dta"
