
/* Global Multidimensional Poverty Index - 
Central African Republic MICS 2010 */

clear all 
set more off
set maxvar 10000



global path_in "../rdta/Central African Republic MICS 2010" 	 // folder path   
global path_out "cdta"
global path_ado "ado"



**# step 1: data cleaning


**## child under 5	

use "$path_in/ch.dta", clear

rename _all, lower


gen double ind_id = hh1*10000 + hh2*100 + ln 
format ind_id %20.0g


duplicates report ind_id								// qc: no duplicates



gen chelg = 1
lab var chelg "child eligible for measurement" 
count


codebook hl4, ta (9) 							// sex

clonevar gender = hl4						// qc: 1=male; 2=female 



codebook cage, ta(999) 		 			// qc: use age in mth

clonevar age = cage
su age								// qc: check min & max value


gen str6 ageunit = "months"
lab var ageunit "months"



codebook an3, ta (999) 					// qc: body weight in kg

clonevar weight = an3 if an3<99 		// qc: out of range value 
ta uf9 an3 if an3>=99, m  				// qc: why missing	
su weight



codebook an4, ta (999) 				// qc: height in cm

clonevar height = an4 if an4<900  	// qc: out of range value
ta uf9 an4 if an4>=999, m			// qc: why missing
su height
	

	
codebook an4a  							// how child was measured

gen measure = "l" if an4a==1 			// lying down
replace measure = "h" if an4a==2 		// standing up
replace measure = " " if an4a>=9  		// " " if unknown
ta measure 

		
gen str1 oedema = "n"  						// oedema


gen sw = 1								  // sampling weight


clonevar chswgt = chweight		
lab var chswgt "sample weight child u5"	



adopath + "$path_ado/igrowup_stata"			     		// compute z-score
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "ch_caf"
lab var datalab "Working file"


igrowup_restricted reflib datalib datalab gender age ///
ageunit weight height measure oedema sw


use "$path_out/ch_caf_z_rc.dta", clear 


foreach m of newlist wei len wfl {

	gen	`m'_a = (_z`m' < -2.0)    									// a = mpi
	replace `m'_a = . if _z`m' == . | _f`m'==1

	gen	`m'_u = (_z`m' < -3.0) 										// u = dst
	replace `m'_u = . if _z`m' == . | _f`m'==1
		
}


rename (wei_a len_a wfl_a) (underweight_a stunted_a wasted_a)
rename (wei_u len_u wfl_u) (underweight_u stunted_u wasted_u)

	
foreach i in underweight stunted wasted {	
	lab var `i'_a  "mpi: child is `i'"
	lab var `i'_u  "dst: child is `i'"
	
	ta `i'_a [aw=chswgt],m                      // qc: % underweight; stunted 
	ta `i'_u [aw=chswgt],m 
}


												// union underweight-stunted
foreach v in a u {								

gen uw_st_`v' = 1 if stunted_`v'==1 | underweight_`v'==1		
replace uw_st_`v' = 0 if stunted_`v'==0 & underweight_`v'==0
replace uw_st_`v' = . if stunted_`v'==. & underweight_`v'==.

}

lab var uw_st_a "mpi: child is uwgt/stunted"   			
lab var uw_st_u "dst: child is uwgt/stunted"  			



count if _fwei==1 | _flen==1  				// biologically implausible zscore 
  

keep *_id chelg chswgt underweight* stunted* wasted* uw_st*
order *_id chelg chswgt underweight* stunted* wasted* uw_st*
sort ind_id

save "$path_out/CAF10_CH.dta", replace


erase "$path_out/ch_caf_z_rc.xls" 						// erase files
erase "$path_out/ch_caf_prev_rc.xls"
erase "$path_out/ch_caf_z_rc.dta"



**## eligible female

use "$path_in/wm.dta", clear 
	
rename _all, lower	


gen double ind_id = hh1*10000 + hh2*100 + ln
format ind_id %20.0g

duplicates report ind_id 							// qc: no duplicates


gen fm_id = 1 
lab var fm_id "female recode file"


ta wb2,m 										 // qc: age 15-49 


codebook cm1 cm8 cm9a cm9b, ta (999)		// qc: check why missing values 


ta cm1 cm8, m					// never gave birth, but reported cm
	

	
codebook mstatus ma6, ta (9)
ta mstatus ma6, m 

gen marf = 1 if mstatus==3 						// never married 
replace marf = 2 if mstatus==1 			     	// married 
replace marf = 3 if ma6==1				 		// widowed
replace marf = 4 if ma6==2 			    		// divorced 
replace marf = 5 if ma6==3						// separated

lab def la_ma 1"never married" 2"currently married" ///
3"widowed" 4"divorced" 5"not living together"
lab val marf la_ma	
lab var marf "marital status female"
ta marf, m
ta ma6 marf, m
ta mstatus marf, m


keep *_id cm1 cm8 cm9a cm9b cm10 marf 
order *_id cm1 cm8 cm9a cm9b cm10 marf 
sort ind_id
save "$path_out/CAF10_FM.dta", replace



**## eligible male 	

use "$path_in/mn.dta", clear 

rename _all, lower


gen double ind_id = hh1*10000 + hh2*100 + ln 
format ind_id %20.0g	


duplicates report ind_id 						// qc: no duplicates


gen ml_id = 1 	
lab var ml_id "male recode file" 


ta mb2, m  								// qc: age of elg male 

	
codebook mmstatus mma6, ta (9)
ta mmstatus mma6, m

gen marm = 1 if mmstatus==3 					// never married
replace marm = 2 if mmstatus==1 & mma6>=9 	    // married
replace marm = 3 if mma6==1	 		 			// widowed	
replace marm = 4 if mma6==2			 			// divorced 
replace marm = 5 if mmstatus==2 & mma6>=3		// separated

lab def la_ma 1"never married" 2"currently married" ///
3"widowed" 4"divorced" 5"not living together"
lab val marm la_ma	
lab var marm "marital status male"
ta marm, m
ta mma6 marm, m
ta mmstatus marm, m


keep *_id marm
order *_id  marm
sort ind_id
save "$path_out/CAF10_ML.dta", replace



**## hh recode

use "$path_in/hh.dta", clear									

rename _all, lower	


gen	double hh_id = hh1*100 + hh2 
format	hh_id %20.0g

duplicates report hh_id					 	// qc: no duplicates


count									// qc: 11,966 hh  

save "$path_out/CAF10_HH.dta", replace



**## hh members

use "$path_in/hl.dta", clear 									

rename _all, lower	

										
gen double hh_id = hh1*100 + hh2 
format hh_id %20.0g
lab var hh_id "household id"


gen double ind_id = hh1*10000 + hh2*100 + hl1
format ind_id %20.0g
lab var ind_id "individual id"


duplicates report ind_id 


sort ind_id

 

**## data merging  
 
merge 1:1 ind_id using "$path_out/CAF10_FM.dta"			// fem recode			
drop _merge	
erase "$path_out/CAF10_FM.dta"


merge 1:1 ind_id using "$path_out/CAF10_ML.dta" 		// male recode
drop _merge
erase "$path_out/CAF10_ML.dta"


merge 1:1 ind_id using "$path_out/CAF10_CH.dta" 		// ch recode
drop _merge
erase "$path_out/CAF10_CH.dta"


merge m:1 hh_id using "$path_out/CAF10_HH.dta" 			// hh recode
ta hh9 if _m==2 
drop if _merge==2					// drop hh not interviewed 
drop _merge
erase "$path_out/CAF10_HH.dta"


sort ind_id


**## eligibility variables		
								
												// elg for cm
																
count if fm_id==1						
count if hl7>0 & hl7!=.			// qc: match between wm & hl datafiles 

bys hh_id: egen no_f_elg = min(fm_id==.)
lab var no_f_elg "hh has no elg female"
ta no_f_elg,m

												// elg for nutr

count if chelg==1
count if hl9>0 & hl9!=.		// qc: match between ch and hl datafiles

bys hh_id: egen no_ch_elg = min(chelg==.)
lab var no_ch_elg "hh has no elg child for nutr"
ta no_ch_elg,m



sort hh_id 
																


**## relevant variables 

clonevar weight = hhweight 						// sample weight 
lab var weight "sample weight"
				
				
lab var psu "primary sampling unit"				// sample design	

clonevar strata = stratum
lab var strata "sample strata"

svyset psu [pw=weight] , strata(strata)	 singleunit(centered)



desc hh5y hh5m hh5d 							// interview dates

clonevar intvw_y = hh5y 	
clonevar intvw_m = hh5m 
clonevar intvw_d = hh5d 




codebook hh6, ta (9)							// area: urban-rural
		
recode hh6 (1=1 "urban") (2=0 "rural"), gen (area)						 
lab var area "area: urban-rural"
ta hh6 area, m



codebook hl3, ta (99)							 // relationship

recode hl3 (1=1 "head")(2=2 "spouse")(3 13=3 "child") ///
(4/12=4 "extended family")(14=5 "not related")(96=6 "maid") ///
(98/99=.), gen (rship)
lab var rship "relationship"
ta hl3 rship, m	



codebook hl4 											// sex

recode hl4 (1=1 "male")	(2=2 "female"), gen (sex)						
lab var sex "sex"
ta hl4 sex,m



bys	hh_id: egen xhhead = min(rship) 				// headship
ta xhhead,m

gen hhead = (rship==1 & sex==1) | (xhhead==2 & rship==2 & sex==1)  
replace hhead = 2 if (rship==1 & sex==2) | (xhhead==2 & rship==2 & sex==2)

bys hh_id: egen hship = sum(hhead)

replace hship = . if xhhead > 2 | hship==0  	// hh, no head
lab def la_hs 1"male-headed" 2"female-headed"
lab val hship la_hs
lab var hship "gender of hh head"
ta hship, m



codebook hl6, ta (999) 								// age; age group

clonevar age = hl6  if hl6<98
lab var age "age"


recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+") , gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"



gen member = 1 										// hh size
bys hh_id: egen hhsize = sum(member)
lab var hhsize "household size"
compare hhsize hh11  					// qc: hh48 matches generated hhsize  
drop member



codebook hh7, ta (99) 										//  region

gen region = hh7
recode region (1 2 = 1) (3 4 5 = 2) (6 7 = 3)(8 9 10 = 4) (11 12 13 = 5) ///
(14 15 16 = 6)(17 = 7) 
lab def la_rg ///
1 "Ombella-M'Poko, Lobaye" 2 "Mambéré-Kadéï, Sangha-Mbaéré, Nana-Mambéré" ///
3 "Ouham, Ouham-Pendé" 4 "Kémo, Nana-Grébizi, Ouaka" ///
5 "Bamingui-Bangoran, Haute-Kotto, Vakaga" ///
6 "Basse-Kotto, Mbomou, Haut-Mbomou" 7 "Bangui" 
lab val region la_rg		
lab var region "subnational region"

codebook region, ta (99)


									// subnational region, harmonised over time

clonevar region_01 = region
lab var region_01 "hot: subnational region"

									
**# step 2: indicator construction


**## school attendance

codebook ed5 ed3, ta (99)

recode ed5 (1=1 "attending") (2=0 "not attending") (9=.), gen (att)

replace att = 0 if ed3==2 						// never attended school				
lab var att "sch att status"
ta att,m 										// qc: missing replaced as '.'



gen scheg_a = (schage>=6 & schage<=14)			// qc: mpi - age range of elg
lab var scheg_a "mpi: elg for school att"



gen scheg_u = (schage>=6 & schage<=12)			// qc: dst - age range of elg
lab var scheg_u "dst: elg for school att"


								// policy: >=2/3 elg pop have non-missing data 
								
foreach v in a u {										// a=mpi; u=dst	 
	 	
	gen tmp_`v' = 1 if scheg_`v'==1 & att!=. 		
	bys hh_id: egen hsattdt_`v' = sum(tmp_`v')	 	// elg & no missing data
	drop tmp_`v'
	
	bys hh_id: egen hscheg_`v' = sum(scheg_`v')  		// elg population

	replace hsattdt_`v' = hsattdt_`v'/hscheg_`v' 
	replace hsattdt_`v' = (hsattdt_`v'>=2/3)	 		
	ta hsattdt_`v', m							     	// qc: missing < 0.5% 	
}


foreach i in a u {							// school att indc (a=mpi; u=dst)

	gen tmp_`i' = (att==1) if scheg_`i'==1 & att!=.     		
	bys	hh_id: egen satt_`i' = min(tmp_`i') 
	replace satt_`i' = . if hsattdt_`i'==0	& satt_`i'==1			// policy
	replace satt_`i' = 1 if hscheg_`i'==0  						   // non-elg 
	drop tmp_`i'

	ta satt_`i', m
							
}

lab var satt_a "mpi: not deprived in satt"					
lab var satt_u "dst: not deprived in satt"


**## years of schooling 
 
codebook ed4a ed3, ta (99)

clonevar edulev = ed4a if ed4a<8 				// qc: missing value

replace edulev = 0 if ed3==2 					// never attended school 



codebook ed4b, ta (99)								

clonevar eduhyr = ed4b if ed4b < 90 			// qc: missing values

replace eduhyr = 0 if ed3==2 | ed4a<1			// never attended or preschool 



ta eduhyr edulev,m


gen eduyr = eduhyr

replace eduyr= 14 if eduhyr>=21 & eduhyr<=55
	// 14 years of educ for higher  

replace eduyr=6  if eduhyr==. & edulev==2
replace eduyr=14 if eduhyr==. & edulev==3

replace eduyr = . if age<=eduyr & age>0 

ta eduyr, m



	/* note: all those who should have completed 6 years 
	of schooling from their starting sch age are counted 
	as eligible, as well as early starters (from age of 10)
	who uniqely have up to six years of schooling are eligible. */
	

											// formal sch age is from 6 years
gen edueg = (age>=12 & age!=.) | ((age==10 | age==11) & ///
			(eduyr>=6 & eduyr!=.)) | (age==. & eduyr>=6 & eduyr!=.)	
								 
lab var edueg "elg for years of schooling"
ta edueg,m


								// policy: >=2/3 elg pop have non-missing data 
								
gen tmp = 1 if eduyr!=. & edueg==1
bys	hh_id: egen hedudt = sum(tmp)				// elg pop with no missing data
drop tmp				
	
	
bys hh_id: egen hedueg = sum(edueg==1) 			// elg population

replace hedudt = hedudt/hedueg
replace hedudt = (hedudt>=2/3)		
ta hedudt, m									// qc: missing < 0.5% 	
lab var hedudt "non-missing edu" 
							
							
									// mpi: completed six years of schooling
gen	tmp_a = (eduyr>=6 & eduyr!=. & edueg==1)   	
replace tmp_a = . if eduyr==. & edueg==1

									// dst: completed one year of schooling
gen	tmp_u = (eduyr>=1 & eduyr!=. & edueg==1)		
replace tmp_u = . if eduyr==. & edueg==1								
	
	
								
foreach i in a u {							
	bys hh_id: egen educ_`i' = max(tmp_`i')
	replace educ_`i' = . if educ_`i'==0 & hedudt==0 				// policy
	drop tmp_`i'
	
	ta educ_`i', m							
}
																
lab var educ_a "mpi: not deprived in yos"
lab var educ_u "dst: not deprived in yos"
				
									
**## nutrition

foreach msr in uw_st_a uw_st_u {	

	bys hh_id: egen tmp = max(`msr')
	gen	no_`msr' = (tmp==0)	if tmp!=.			
	drop tmp
	replace no_`msr' = 1 if no_ch_elg==1			// hh no elg child 

	ta no_`msr',m 
}

lab var no_uw_st_a "mpi: no underweight/stunted child in hh"
lab var no_uw_st_u "dst: no underweight/stunted child in hh"	
	
		
	
foreach i in a u {							 // nutr indc (a = mpi; u = dst)

gen	nutr1_`i' = 1  														
replace nutr1_`i' = 0 if no_uw_st_`i'==0
replace nutr1_`i' = . if no_uw_st_`i'==.	

ta nutr1_`i', m

}

lab var nutr1_a "mpi: not deprived in nutr (u5)"
lab var nutr1_u "dst: not deprived in nutr (u5)"



**## child mortality

codebook cm9a cm9b

egen tmp = rowtotal(cm9a cm9b), m

							// zero mortality = no child died, or non-elg fem 
ta cm1 cm8,m
	
replace tmp = 0 if cm8==2 | no_f_elg==1 		
														

bys hh_id: egen cm7 = sum(tmp),m			
	
	
recode cm7 (1/max=0) (0=1) if cm7!=.
lab var cm7 "not deprived in cm (no bh)"
ta cm7, m 												// mpi = dst

drop tmp 
		
		
		
**## electricity 

codebook hc8a, ta (9)

											// mpi = dst
											
recode hc8a  (1 = 1 "yes") (2 = 0 "no") (9=.), gen(elct)	 
lab var elct "not deprived in elct"							


svy: prop elct 					



**## sanitation

desc ws8 ws9

/*
(p.412)
Flush / manual flush
Connected to sewage system ...11
Connected to septic tank .....12
Connected to latrines ........13
Connected to something else....14
Connected to unknown/unsure/DK location where ...15

Pits/latrines
Improved ventilated latrines (LAV) ......21
Pit latrines with slab..................22
Hole latrines without slab/open hole....23
Hole latrine with slab................24
Composting toilets....................31
Buckets ...............................41
Wall-hung toilets/latrines...........51
No toilets, nature.................95
Other ........96
*/

codebook ws8, ta(99) 	
	
recode ws8 (11/13 15/22 24 31 = 1 "yes") ///
(14 23 41/96 = 0 "no") (99=.), gen(sani_a)   			// mpi


ta ws8 sani_a, m


codebook ws9

replace sani_a = 0 if ws9==1				
lab var sani_a "mpi: not deprived in sani"


							
recode ws8 (11/51 = 1 "yes") ///
(95 96 = 0 "no") (99=.), gen(sani_u)   				// dst

lab var sani_u "dst: not deprived in sani"

ta ws8 sani_u, m



**## drinking water 
/*
(p.411)
piped into home.... 11
piped courtyard or plot .....12
neighbor's faucet.....13
public tap / fountain stand .....14
drilling......21
pump well....22
manual pump well.....23
protected/developed well ....31
unprotected/undeveloped well ....32
protected/managed source...............41
unprotected/unmanaged source......42
rainwater.......51
tanker truck.........61
cart with small tank......71
surface water (river etc)....81
Bottled/mineral water ......91
Other...96
*/

desc ws1 ws4 

codebook ws1, ta(99)


recode ws1 (11/31 41 51/71 91 = 1 "yes") ///
(32 42 81 96 = 0 "no") (99=.), gen(wtr_sdg)

lab var wtr_sdg "improved source of dwtr sdg"

ta ws1 wtr_sdg,m



codebook ws4, ta (999)		// time to water


clonevar wtr_a = wtr_sdg

replace wtr_a = 0 if ws4 >=30 & ws4 <=900 		// mpi >=30 mins
lab var wtr_a "mpi: not deprived in dwtr"
ta wtr_a,m


clonevar wtr_u = wtr_sdg															   
replace wtr_u = 0 if ws4 >45 & ws4 <=900		// dst > 45 mins
lab var wtr_u "dst: not deprived in dwtr"
ta wtr_u,m 



**## housing

desc hc3 hc4 hc5

codebook hc3, ta (99)		// improved = rudimentary & finished floor 

/*
(p.413)
Natural material
Earth/sand....... 11
Dung........12

Rudimentary material
Wooden board ......21
Fins/bamboo....22

Finished material
Parquet or waxed wood ......31
Vinyl or asphalt.......32
Tiling ............33
Cement............34
Carpet ......35
Other 96
*/

recode hc3 (21/37 = 1 "yes") (11 12 96 = 0 "no") (99=.), gen(floor)
lab var floor "hh has improved floor"
ta hc3 floor,m

	
codebook hc5, ta (99)		// improved = finished walls 

/*
Natural material
No walls......11
Cane / Flippers / trunks ..12
Clods of earth....13

Rudimentary material
Bamboo with mud....21
Stone with mud.....22
Adobe uncovered..23
Plywood..24
Box .....25
Reclaimed wood .....26

Finished material
Cement......31
Stone with lime/cement..32
Bricks ....33
Cement blocks.......34
Adobe covered...35
Wooden planks/shingles.......36
Other _96
*/


recode hc5 (31/38 = 1 "yes") (11/27 96 = 0 "no") (99=.), gen(wall)
lab var wall "hh has improved wall"
ta hc5 wall,m

	
codebook hc4, ta (99)		// improved = finished roofing

/*
Natural material
No roof.........11
Thatch/palm leaf....12
Herbs/Straw....13

Rudimentary material
Mat ......21
Fins / Bamboo....22
Wooden boards .....23
Cardboard....24

Finished material
Metal/Sheet....31
Wood.....32
Zinc / Cement fiber....33
Tiles......34
Cement....35
Shingles....36
Other 96
*/
 
recode hc4 (31/37 = 1 "yes") (11/25 96 = 0 "no") (99=.), gen(roof)
lab var roof "hh has improved roof"
ta hc4 roof,m	


gen hsg_a = 1											// mpi
replace hsg_a = 0 if floor==0 | wall==0 | roof==0
replace hsg_a = . if floor==. & wall==. & roof==.
lab var hsg_a "mpi: not deprived in hsg"
ta hsg_a, m



gen hsg_u = 1											// dst
replace hsg_u = 0 if ///
(floor==0 & wall==0 & roof==1) | ///
(floor==0 & wall==1 & roof==0) | ///
(floor==1 & wall==0 & roof==0) | ///
(floor==0 & wall==0 & roof==0)

replace hsg_u = . if floor==. & wall==. & roof==.
lab var hsg_u "dst: not deprived in hsg"
ta hsg_u, m



**## cooking fuel 

codebook hc6, ta(99)

/*
(p.413)
Electricity.......01
Liquefied propane gas (LPG)... 02
Natural gas........03
Biogas.... 04
Kerosene...05
Coal / Lignite...... 06
Charcoal ......07
Wood...08
Straw/branches/grasses ... 09
Dung.........10
Agricultural residues ....... 11
No meals prepared in the household......95
Other 96
*/



													// mpi = dst

recode hc6 (1/5 95 96 = 1 "yes") (6/11 = 0 "no") (99 = .), gen(ckfl) 

lab var ckfl "not deprived in ckfl"	

ta hc6 ckfl,m				


	
**## assets

lookfor car van truck camion
codebook hc9f
clonevar car = hc9f  
lab var car "hh has car"	



lookfor tele phone téléphone mobilephone ipod telefone tele celular
ta hc8d hc9b,m nol 
gen phone = (hc8d==1 | hc9b==1)
replace phone = . if hc8d==9 & hc9b==9
lab var phone "hh has telephone"			
ta hc8d hc9b if phone==1,m



lookfor radio walkman stereo stéréo
codebook hc8b
clonevar radio = hc8b 
lab var radio "hh has radio"



lookfor tv television plasma lcd télé tele
codebook hc8c 
clonevar tv = hc8c
lab var tv "hh has television"



lookfor refrigerator réfrigérateur refri
codebook hc8e
clonevar fridge = hc8e
lab var fridge "hh has refrigerator"
	
	
	
lookfor bicycle bicyclette bicicleta
codebook hc9c 
clonevar bike = hc9c 
lab var bike "hh has bicycle"		
	
	
	
lookfor motorbike moto
codebook hc9d	
clonevar mbike = hc9d
lab var mbike "hh has motorbike"
	
	
	
lookfor brouette cart carro carreta animal
codebook hc9e
clonevar acart = hc9e
lab var acart "hh has animal cart"	

		
lookfor computer ordinateur laptop ipad tablet 
codebook hc8i
clonevar comp = hc8i
lab var comp "hh has computer"
	

	
lab def la_at 0"no" 1"yes" 
foreach var in car tv radio phone fridge bike mbike comp acart {
replace `var' = 0 if `var'==2 
lab val `var' la_at			   
replace `var' = . if `var'==9 | `var'==99 | `var'==8 | `var'==98 
}


egen n_bast = rowtotal (tv radio phone fridge bike mbike comp acart), m
lab var n_bast "small assets owned by hh" 
   
   
gen asst_a = (car==1 | n_bast > 1) 					// mpi
replace asst_a = . if car==. & n_bast==.   								
lab var asst_a "mpi: not deprived in assets"


gen	asst_u = (car==1 | n_bast >0)					// dst
replace asst_u = . if car==. & n_bast==.  								
lab var asst_u "dst: not deprived in assets"
			
			

**# step 3: spec & flav

recode cm7   	(0=1)(1=0) , gen(d_cm)				// mpi est 
recode nutr1_a 	(0=1)(1=0) , gen(d_nutr)
recode satt_a 	(0=1)(1=0) , gen(d_satt)
recode educ_a 	(0=1)(1=0) , gen(d_educ)
recode elct		(0=1)(1=0) , gen(d_elct)
recode sani_a	(0=1)(1=0) , gen(d_sani)
recode wtr_a 	(0=1)(1=0) , gen(d_wtr)
recode hsg_a 	(0=1)(1=0) , gen(d_hsg)
recode ckfl 	(0=1)(1=0) , gen(d_ckfl)
recode asst_a 	(0=1)(1=0) , gen(d_asst)


recode cm7   	(0=1)(1=0) , gen(d_cm_01)	 		 // hot mpi est 
recode nutr1_a 	(0=1)(1=0) , gen(d_nutr_01)
recode satt_a 	(0=1)(1=0) , gen(d_satt_01)
recode educ_a 	(0=1)(1=0) , gen(d_educ_01)
recode elct		(0=1)(1=0) , gen(d_elct_01)
recode wtr_a 	(0=1)(1=0) , gen(d_wtr_01)
recode sani_a	(0=1)(1=0) , gen(d_sani_01)
recode hsg_a 	(0=1)(1=0) , gen(d_hsg_01)
recode ckfl 	(0=1)(1=0) , gen(d_ckfl_01)
recode asst_a 	(0=1)(1=0) , gen(d_asst_01)



recode cm7   	(0=1)(1=0) , gen(dst_cm)			// dst est
recode nutr1_u 	(0=1)(1=0) , gen(dst_nutr)
recode satt_u 	(0=1)(1=0) , gen(dst_satt)
recode educ_u 	(0=1)(1=0) , gen(dst_educ)
recode elct 	(0=1)(1=0) , gen(dst_elct)
recode sani_u 	(0=1)(1=0) , gen(dst_sani)
recode wtr_u  	(0=1)(1=0) , gen(dst_wtr)
recode hsg_u  	(0=1)(1=0) , gen(dst_hsg)
recode ckfl 	(0=1)(1=0) , gen(dst_ckfl)
recode asst_u 	(0=1)(1=0) , gen(dst_asst) 


recode cm7   	(0=1)(1=0) , gen(dst_cm_01)			// hot dst est
recode nutr1_u	(0=1)(1=0) , gen(dst_nutr_01)
recode satt_u 	(0=1)(1=0) , gen(dst_satt_01)
recode educ_u 	(0=1)(1=0) , gen(dst_educ_01)
recode elct 	(0=1)(1=0) , gen(dst_elct_01)
recode wtr_u  	(0=1)(1=0) , gen(dst_wtr_01)
recode sani_u 	(0=1)(1=0) , gen(dst_sani_01)
recode hsg_u  	(0=1)(1=0) , gen(dst_hsg_01)
recode ckfl 	(0=1)(1=0) , gen(dst_ckfl_01)
recode asst_u 	(0=1)(1=0) , gen(dst_asst_01) 



lab def la_dp 1"yes" 0"no"
foreach var in d_* dst_* d_*_* dst_*_* {
lab val `var' la_dp
}

foreach var in cm nutr satt educ elct wtr sani hsg ckfl asst {
lab var d_`var' "deprived in `var'"
lab var d_`var'_01 "hot: deprived in `var'"
lab var dst_`var' "dst: deprived in `var'"
lab var dst_`var'_01 "hot-dst: deprived in `var'"
}


keep hh_id ind_id strata psu weight chswgt ///
sex age agec* area region* hship intvw_* ///
scheg_a att edueg eduyr chelg uw_st_a ///
d_* dst_* wtr_sdg ws4

	 
order hh_id ind_id strata psu weight chswgt ///
sex age agec* area region* hship intvw_* ///
scheg_a att edueg eduyr chelg uw_st_a ///
d_* dst_* wtr_sdg ws4


mdesc psu strata area age hship region* d_* dst_*
 
 
char _dta[cty] "Central African Republic"
char _dta[ccty] "CAF"
char _dta[year] "2010" 	
char _dta[survey] "MICS"
char _dta[ccnum] "140"
char _dta[type] "micro"

	
	
sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/caf_mics10.dta", replace 
									