********************************************************************************
/*
Citation:
Oxford Poverty and Human Development Initiative (OPHI), University of Oxford. 
Global Multidimensional Poverty Index - Bolivia EDSA 2016 [STATA do-file]. 
Available from OPHI website: http://ophi.org.uk/  

For further queries, contact: ophi@qeh.ox.ac.uk
*/
********************************************************************************

clear all 
set more off
set maxvar 10000


*** Working Folder Path ***
global path_in "../rdta/Bolivia EDSA 2016" 
global path_out "cdta"
global path_ado "ado"



********************************************************************************
*** Bolivia EDSA 2016 ***
********************************************************************************

	/*Bolivia EDSA 2016: The "Encuesta de Demografía y Salud" (EDSA 2016) is 
	a population-based survey, representative at the national, departmental, 
	areas (urban and rural) and regions (highlands, valleys and plains), 
	provides relevant information on maternal and child health, family 
	planning, fertility, mortality, lactation, nutrition and sexually 
	transmitted diseases, violence against women and men and other issues 
	that affect the health of the population. The survey is aimed at the 
	set of households across the 9 departments of the country in urban and 
	rural areas where women of childbearing age from 14 to 49 years of age, 
	children under six years of age and men from 15 to 64 year old reside. 
	The population residing in collective dwellings such as hospitals, 
	hotels, prisons, military compounds, nursing homes, educational units, 
	etc are excluded from the investigation; but they include households 
	that form an independent group and also reside within these establishments,
	for example: porters, caregivers, etc.
	Source: Documento metodologico (p.14)
	
	The microdata may be downloaded from the link appended below:
	http://anda.ine.gob.bo/index.php/auth/login/?destination=catalog/28/get-microdata
	
	
	There are 8 database that we use to compute the global MPI:
	
	ANTHROPOMETRICS
	EDSA16_PESO_TALLA_HEMOGLOBINA = Contains information on Weight and 
	Height measurements for all women ages 14-49 and children born since 2011, 
	in addition to the hemoglobin test result (19,060 obs)
	
	INDIVIDUALS
	EDSA16_HOGAR = Contains information on household members (all sections). 
	School attendance, years of schooling (51,024 obs)
	
	EDSA16_MUJER_ANTECEDENTES =	General information except the data of your 
	children. Contains information on the number of sons and daughters who 
	died and children ever born (12,274 obs)
	
	EDSA16_MUJER_PARTO_VACUNACION = Information on children born since 2011, 
	in relation to prenatal, postpartum delivery, breastfeeding, vaccination, 
	age.
	
	EDSA16_MUJER_CALENDARIO = Record of events since 2011 on pregnancies, 
	terminations, contraceptive methods.
	
	EDSA16_HOMBRES = Information such as age, education and others, similar 
	to the woman's background (except for the particularities of her children, 
	(number of sons and daughters death) (4,975 obs).
	
	
	HOUSEHOLDS
	EDSA16_VIVIENDA = It contains information on the characteristics of the 
	home of the home. Assets, sanitation, water, cookingfuel, electricity 
	(14,655 obs).
	
	BIRTH HISTORY
	EDSA16_MUJER_HISTORIA_NACIMIENTOS = information of date of birth, age, 
	sex, if he is alive or dead, single or multiple birth, if he is dead 
	the data of the age when he died (24,072 obs, 8,078 women).

	
	Anthropometric information is available for all children under 5 and 
	all eligible women 14-49 years old who live in the dwelling or 
	have slept there the night before the interview.
	
	There is  a subsample of male interviews. In the same male sub-sample, 
	hemoglobin data was collected from women and children.*/ 

	
********************************************************************************
*** Step 1: Data preparation 
*** Selecting variables from BR, IR, & MR recode & merging with PR recode 
********************************************************************************

********************************************************************************
*** Step 0 PR - BUILD THE INDIVIDUAL RECODE BY MERGING DIFFERENT FILES 
********************************************************************************

	/*Note: Bolivia EDSA 2016 provides the data in separate files. As a first 
	step, we constructed the PR - INDIVIDUAL RECODE - by merging the file for 
	children's recode on anthropometrics "EDSA16_PESO_TALLA_HEMOGLOBINA"  
	with the file for individual recode "EDSA16_HOGAR" and the data files 
	for household recode "EDSA16_VIVIENDA". We do this before computing the 
	child anthropometric indicators because variables that are required in this
	section is present in the different data files. Merging beforehand allow us 
	to join all the information before proceeding with the coding. Also, this 
	enable us to have a full “PR-INDIVIDUAL RECODE” that we will further use.*/

	
******************************
*** CHILDREN and WOMEN RECODE
******************************

use "$path_in/EDSA16_PESO_TALLA_HEMOGLOBINA.dta", clear 

rename _all, lower	

*** Generate individual unique key variable required for data merging using:
*** id  = Identificator variable of dwelling and household (6-digit long)
*** nro = Resident number 	
gen double hh_id=id	

gen double ind_id = id*100 + nro
format ind_id %8.0g
label var ind_id "Individual ID"
codebook ind_id

duplicates report ind_id 
	//19,060 observations


save "$path_out/BOL16_CHWM", replace


*********************
*** INDIVIDUAL RECODE 
*********************

use "$path_in/EDSA16_HOGAR.dta", clear 

rename _all, lower	

	
*** Generate a household unique key variable at the household level using: 
gen double hh_id=id	

gen double ind_id = id*100 + nro
format ind_id %8.0g
label var ind_id "Individual ID"
codebook ind_id

duplicates report ind_id 
	//51,024 observations, 14,655  households

save "$path_out/BOL16_PR_1", replace


**********************
*** HOUSEHOLD'S RECODE
********************** 

use "$path_in/EDSA16_VIVIENDA.DTA", clear 

rename _all, lower	

*** Generate a household unique key variable at the household level using: 
gen double hh_id = id
format hh_id %20.0g
label var hh_id "Household ID"
codebook hh_id  

sum e_01
di r(sum)
	//51,024 individuals
sum e_02
di r(sum)	
	//12,609 elegible women
sum e_03
di r(sum)
	//14,574 elegible men


merge 1:m hh_id using "$path_out/BOL16_PR_1.dta"
drop _merge
	//51,024 individuals merged (100%) 	



merge 1:1 hh_id ind_id using "$path_out/BOL16_CHWM.dta"
drop _merge
	//19,060 matched 
	

count 
	//51,024 individuals (of which 280 are non-permenant members)


save "$path_out/BOL16_PR.DTA", replace


********************************************************************************
*** Step 1.1a PR - INDIVIDUAL RECODE 
*** (Children under 5 years) 
********************************************************************************

use "$path_out/BOL16_PR.DTA", clear


*** Check the variables on eligible children 
count if hs06_0093!=. 
tab hs03_0003_1 if hs06_0093!=.,m
tab hs06_0093 if hs06_0093!=. 
	//Of the 6,451 children, 6,164 (95.6%) between 0-5 have been measured 
keep if hs06_0093!=. 
	//6,451 children 0-5 years

	
*** Variable: AGE ***
gen bdate = mdy(hs03_0003_2b, hs03_0003_2a, hs03_0003_2c) if hs03_0003_2a <= 31
gen mdate = mdy(dv_00484b, dv_00484a, dv_00484c)
gen age_months = (mdate-bdate)/30.4375 
	//Calculate age in months with days expressed as decimals 
replace age_months=0 if age_months<0	
sum age_months	

gen  str6 ageunit = "months" 
lab var ageunit "Months"


keep if age_months<61	
count
	/*The data indicate a total of 6,451 children aged 0-5 years. 
	In terms of age in months, these children are between 0-69 months old.
	We compute underweight and stunting measures for children up to 60 months. 
	We compute BMI-for-age for children from 61-69 months old. This is 
	done in the next section, that is, Step 1.1b
	
	The final sample count of children aged 0-60 months is 5,845 children*/

	
*** Variable: SEX ***
desc hs03_0005 
	//hs03_0005=sex from household roaster
tab hs03_0005, miss 
	//"1" for male ;"2" for female 
clonevar gender = hs03_0005
label define lab_sex 1 "male" 2 "female"
label values gender lab_sex
label var gender "Sex"
tab gender


*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook hs06_0090, tab (9999)
gen	weight = hs06_0090
tab hs06_0090 if hs06_0090>99.9, miss nol 
tab hs06_0093 if hs06_0093!=1
	//287 not measured. 
	//hs06_0093: result of the measurement
sum weight


*** Variable: HEIGHT (CENTIMETERS)
codebook hs06_0091, tab (9999)
gen	height = hs06_0091
tab hs06_0093 if height == .
	//287 not measured. Missing values are 9999
sum height


*** Variable: MEASURED STANDING/LYING DOWN ***	
codebook hs06_0092
gen measure = "l" if hs06_0092==1 
	//Child measured lying down
replace measure = "h" if hs06_0092==2 
	//Child measured standing up
replace measure = " " if hs06_0092==. 
	//Replace with " " if unknown
tab measure


*** Variable: OEDEMA ***
gen  oedema = "n"  


*** Variable: SAMPLING WEIGHT ***
	/* We don't require individual weight to compute the z-scores of a 
	child. So we assume all children in the sample have the same weight */
gen sw = 1	



*** Indicate to STATA where the igrowup_restricted.ado file is stored:
	***Source of ado file: http://www.who.int/childgrowth/software/en/
adopath + "$path_ado/igrowup_stata"

*** We will now proceed to create three nutritional variables: 
	*** weight-for-age (underweight),  
	*** weight-for-height (wasting) 
	*** height-for-age (stunting)

/* We use 'reflib' to specify the package directory where the .dta files 
containing the WHO Child Growth Standards are stored.*/	
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"

/* We use datalib to specify the working directory where the input STATA 
dataset containing the anthropometric measurement is stored. */
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"

/* We use datalab to specify the name that will prefix the output files that 
will be produced from using this ado file (datalab_z_r_rc and datalab_prev_rc)*/
gen str30 datalab = "children_nutri_bol" 
lab var datalab "Working file"

	
/*We now run the command to calculate the z-scores with the adofile */
igrowup_restricted reflib datalib datalab gender ///
age_months ageunit weight height measure oedema sw


/*We now turn to using the dta file that was created and that contains 
the calculated z-scores to create the child nutrition variables following WHO 
standards */
use "$path_out/children_nutri_bol_z_rc.dta", clear 

	
*** Standard MPI indicator ***	
gen	underweight = (_zwei < -2.0) 
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "Child is undernourished (weight-for-age) 2sd - WHO"
tab underweight, miss 
	

gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "Child is stunted (length/height-for-age) 2sd - WHO"
tab stunting, miss 


gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "Child is wasted (weight-for-length/height) 2sd - WHO"
tab wasting, miss 


*** Destitution MPI indicator  ***	
gen	underweight_u = (_zwei < -3.0) 
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "Child is undernourished (weight-for-age) 3sd - WHO"


gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "Child is stunted (length/height-for-age) 3sd - WHO"


gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "Child is wasted (weight-for-length/height) 3sd - WHO"
 
 
count if _fwei==1 | _flen==1 
	/*Bolivia EDSA 2016: 54 children were replaced as missing because
	they have extreme z-scores which are biologically implausible. */
 
 
gen child_CH = 1 

clonevar weight_ch = ponderadorh
label var weight_ch "sample weight child under 5" 

 
	//Retain relevant variables:
keep  ind_id child_CH weight_ch underweight* stunting* wasting* 
order ind_id child_CH weight_ch underweight* stunting* wasting* 
sort  ind_id
save "$path_out/BOL16_PR_child.dta", replace
	
	
	//Erase files from folder:
erase "$path_out/children_nutri_bol_z_rc.xls"
erase "$path_out/children_nutri_bol_prev_rc.xls"
erase "$path_out/children_nutri_bol_z_rc.dta"


********************************************************************************
*** Step 1.1b CHILDREN's RECODE (61-68 months)
********************************************************************************

use "$path_out/BOL16_PR.dta", clear 


*** Retain eligible children 
keep if hs06_0093!=. 
	//6,451 children 0-5 years
	

*** Variable: AGE ***
gen bdate = mdy(hs03_0003_2b, hs03_0003_2a, hs03_0003_2c) if hs03_0003_2a <= 31
gen mdate = mdy(dv_00484b, dv_00484a, dv_00484c)
gen age_months = (mdate-bdate)/30.4375 
	//Calculate age in months with days expressed as decimals 
replace age_months=0 if age_months<0	
sum age_months	

gen  str6 ageunit = "months" 
lab var ageunit "Months"


keep if age_months>=61 
count
	/*The data indicate a total of 6,451 children aged 0-5 years. 
	This includes 5,738 children aged 0-60 months and 606 children
	aged 61-69 months.*/
	

*** Check the variables to calculate the z-scores:

*** Variable: SEX *** 
clonevar gender = hs03_0005
label define lab_sex 1 "male" 2 "female"
label values gender lab_sex
label var gender "Sex"
tab gender

	
*** Variable: BODY WEIGHT (KILOGRAMS) ***
gen	weight = hs06_0090
sum weight


*** Variable: HEIGHT (CENTIMETERS)
gen	height = hs06_0091
sum height


*** Variable: MEASURED STANDING/LYING DOWN ***	
gen measure = "l" if hs06_0092==1 
	//Child measured lying down
replace measure = "h" if hs06_0092==2 
	//Child measured standing up
replace measure = " " if hs06_0092==. 
	//Replace with " " if unknown
tab measure


*** Variable: OEDEMA ***
gen  oedema = "n"  
	
*** Variable: SAMPLING WEIGHT ***
gen sw = 1	

	
/* 
For this part of the do-file we use the WHO AnthroPlus software. This is to 
calculate the z-scores for children aged 60-72 months. 
Source of ado file: https://www.who.int/growthref/tools/en/
*/
	
*** Indicate to STATA where the igrowup_restricted.ado file is stored:
adopath + "$path_ado/who2007_stata"


/* We use 'reflib' to specify the package directory where the .dta files 
containing the WHO Growth reference are stored. Note that we use strX to specity 
the length of the path in string. */		
gen str100 reflib = "$path_ado/who2007_stata"
lab var reflib "Directory of reference tables"


/* We use datalib to specify the working directory where the input STATA data
set containing the anthropometric measurement is stored. */
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"


/* We use datalab to specify the name that will prefix the output files that 
will be produced from using this ado file*/
gen str30 datalab = "children_nutri_bol" 
lab var datalab "Working file"


/*We now run the command to calculate the z-scores with the adofile */
who2007 reflib datalib datalab gender age_months ageunit weight ///
height oedema sw


/*We now turn to using the dta file that was created and that contains 
the calculated z-scores to compute BMI-for-age*/
use "$path_out/children_nutri_bol_z.dta", clear 

		
gen	z_bmi = _zbfa
replace z_bmi = . if _fbfa==1 
lab var z_bmi "z-score bmi-for-age WHO"


*** Standard MPI indicator ***
gen	low_bmiage_6y = (z_bmi < -2.0) 
replace low_bmiage_6y = . if z_bmi==.
lab var low_bmiage_6y "Young child low bmi 2sd - WHO"


*** Destitution indicator ***
gen	low_bmiage_6y_u = (z_bmi < -3.0)
replace low_bmiage_6y_u = . if z_bmi==.
lab var low_bmiage_6y_u "Young child low bmi 3sd - WHO"


tab low_bmiage_6y, miss
tab low_bmiage_6y_u, miss

rename age_months age_month_6y
gen child_CH_6y=1 


	//Retain relevant variables:	
keep ind_id child_CH_6y age_month_6y low_bmiage*
order ind_id child_CH_6y age_month_6y low_bmiage*
sort ind_id
save "$path_out/BOL16_CH_6Y.dta", replace


	//Erase files from folder:
erase "$path_out/children_nutri_bol_z.xls"
erase "$path_out/children_nutri_bol_prev.xls"
erase "$path_out/children_nutri_bol_z.dta"


********************************************************************************
*** Step 1.2  BR - BIRTH RECODE  
********************************************************************************


***EDSA16_MUJER_HISTORIA_NACIMIENTOS

use "$path_in/EDSA16_MUJER_HISTORIA_NACIMIENTOS.dta" , clear

rename _all, lower
count 
	//24,072 children 

gen double hh_id=id	

gen double ind_id = id*100 + nro
format ind_id %8.0g
label var ind_id "Individual ID"
codebook ind_id


gen sdate = mdy(ms01_0101_1b, ms01_0101_1a, ms01_0101_1c)
	//sdate = interview date

gen bdate = mdy(ms02_0219b, ms02_0219a, ms02_0219c) if ms02_0219a<= 31
	//bdate = date of birth
	
	
gen age_death     = ms02_0224_2 if ms02_0224_1==1
	//children who died within 30 days
replace age_death = ms02_0224_2*30.4375 if ms02_0224_1==2
	//children who died within 12 months
replace age_death = ms02_0224_2*30.4375*12 if ms02_0224_1==3
	//dates reported in years are converted to months
	
	
gen date_death = bdate + age_death	
	//Date of death = date of birth + age at death 
gen mdead_survey = sdate - date_death 	
	//Months dead from survey = Date of interview - date of death	
gen ydead_survey = mdead_survey/(30.4375*12)
	//Years dead from survey
sum ydead_survey	
	
	
codebook ms02_0220, tab (10)	
gen child_died = 1 if ms02_0220==2
replace child_died = 0 if ms02_0220==1
replace child_died = . if ms02_0220==.
label define lab_died 1 "child has died" 0 "child is alive"
label values child_died lab_died
tab ms02_0220 child_died, miss 
	//1,226 children are dead 
	

bysort ind_id: egen tot_child_died = sum(child_died) 
	//For each woman, sum the number of children who died
	
	
	//Identify child under 18 mortality in the last 5 years
gen child18_died = child_died 
replace child18_died=0 if age_death>=216*30.4375 & age_death<.
label define lab_u18died 1 "child u18 has died" 0 "child is alive/died but older"
label values child18_died lab_u18died
tab child18_died, miss


bysort ind_id: egen tot_child18_died_5y=sum(child18_died) if ydead_survey<=5
	/*Total number of children under 18 who died in the past 5 years 
	prior to the interview date */	
	
replace tot_child18_died_5y=0 if tot_child18_died_5y==. & tot_child_died>=0 & tot_child_died<.
	/*All children who are alive or who died longer than 5 years from the 
	interview date are replaced as '0'*/
	
replace tot_child18_died_5y=. if child18_died==1 & ydead_survey==.
	//Replace as '.' if there is no information on when the child died  

tab tot_child_died tot_child18_died_5y, miss

bysort ind_id: egen childu18_died_per_wom_5y = max(tot_child18_died_5y)
lab var childu18_died_per_wom_5y "Total child under 18 death for each women in the last 5 years (birth recode)"



	//Keep one observation per women
bysort ind_id: gen id2=1 if _n==1
count if id2==1 
keep if id2==1
drop id2
duplicates report ind_id 



gen women_BR = 1 
	//Identification variable for observations in BR recode

	
	//Retain relevant variables
keep ind_id women_BR childu18_died_per_wom_5y  
order ind_id women_BR childu18_died_per_wom_5y 
sort ind_id


save "$path_out/BOL16_BR.dta", replace	


********************************************************************************
*** Step 1.3  IR - WOMEN's RECODE  
*** (Eligible female 14-49 years in the household)
********************************************************************************

use "$path_in/EDSA16_MUJER_ANTECEDENTES.dta", clear


gen double hh_id=id	

gen double ind_id = id*100 + nro
format ind_id %8.0g
label var ind_id "Individual ID"
codebook ind_id

gen women_IR=1 
	//Identification variable for observations in IR recode


	//Retain relevant variables:
keep ind_id women_IR ms02_0201 ms02_0206_1 ms02_0207_1 ///
ms02_0207_2 ms06_0601 ms06_0602 ms06_0603 
order ind_id women_IR ms02_0201 ms02_0206_1 ms02_0207_1 ///
ms02_0207_2 ms02_0207_2 ms06_0601 ms06_0602 ms06_0603 
sort ind_id
save "$path_out/BOL16_IR.dta", replace
 

********************************************************************************
*** Step 1.4  PR - INDIVIDUAL RECODE  
*** (Girls 14-19 years in the household)
********************************************************************************

use "$path_out/BOL16_PR.dta", clear

	
*** Identify anthropometric sample for girls
tab  hs06_0084 if hs03_0003_1>=14 & hs03_0003_1<=19 & hs03_0005==2, miss 
	/*Total number of girls 14-19 years who were selected for anthropometric 
	measurement: 2,915  girls; 2,828 were measured.*/	
keep if hs03_0003_1>=14 & hs03_0003_1<=19 & hs03_0005==2 
count
	//Total girls 14-19 years: 2,915

	
***Variables required to calculate the z-scores to produce BMI-for-age:

*** Variable: SEX ***
codebook hs03_0005, tab (9)
clonevar gender = hs03_0005
	//2:female 


*** Variable: AGE ***
gen bdate = mdy(hs03_0003_2b, hs03_0003_2a, hs03_0003_2c) if hs03_0003_2a <= 31
gen mdate = mdy(dv_00484b, dv_00484a, dv_00484c)
gen age_months = (mdate-bdate)/30.4375 	
lab var age_month "Age in months, individuals 14-19 years (girls)"
sum age_month
count if age_month>228
	/*Note: 394 observations with the age in months beyond 228 months 
	(19 years). In this section, while calculating the z-scores, these 
	cases will be excluded. However, in section 2.3, we will take the 
	BMI information of these girls.*/

	
*** Variable: AGE UNIT ***
gen str6 ageunit = "months" 
lab var ageunit "Months"

			
*** Variable: BODY WEIGHT (KILOGRAMS) ***
codebook hs06_0082, tab (9999)
count if hs06_0082>9990 
	//87 observations 
tab hs06_0084 if hs06_0082>9990, miss 
	// 1 refused, 86 other 
gen weight = hs06_0082 if hs06_0082<9990
sum weight


*** Variable: HEIGHT (CENTIMETERS)	
codebook hs06_0083, tab (9999)
count if hs06_0083>9990 
	//87 observations 
tab hs06_0084 if hs06_0083>9990, miss 
	// 1 refused, 86 other 
gen height = hs06_0083 if hs06_0083<9990
sum height


*** Variable: OEDEMA
gen oedema = "n"  
	

*** Variable: SAMPLING WEIGHT ***
	/* We don't require individual weight to compute the z-scores. We 
	assume all individuals in the sample have the same sample weight */
gen sw = 1

					
/* 
For this part of the do-file we use the WHO AnthroPlus software. This is to 
calculate the z-scores for young individuals aged 15-19 years. 
Source of ado file: https://www.who.int/growthref/tools/en/
*/

*** Indicate to STATA where the igrowup_restricted.ado file is stored:	
adopath + "$path_ado/who2007_stata"

	
/* We use 'reflib' to specify the package directory where the .dta files 
containing the WHO Growth reference are stored. Note that we use strX to specify 
the length of the path in string. */		
gen str100 reflib = "$path_ado/who2007_stata"
lab var reflib "Directory of reference tables"


/* We use datalib to specify the working directory where the input STATA data
set containing the anthropometric measurement is stored. */
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"


/* We use datalab to specify the name that will prefix the output files that 
will be produced from using this ado file*/
gen str30 datalab = "girl_nutri_bol" 
lab var datalab "Working file"
	

/*We now run the command to calculate the z-scores with the adofile */
who2007 reflib datalib datalab gender age_month ageunit weight height oedema sw


/*We now turn to using the dta file that was created and that contains 
the calculated z-scores to compute BMI-for-age*/
use "$path_out/girl_nutri_bol_z.dta", clear 

	
gen	z_bmi = _zbfa
replace z_bmi = . if _fbfa==1 
lab var z_bmi "z-score bmi-for-age WHO"


*** Standard MPI indicator ***	
gen	low_bmiage = (z_bmi < -2.0) 
replace low_bmiage = . if z_bmi==.
lab var low_bmiage "Teenage low bmi 2sd - WHO"


*** Destitution indicator ***	
gen	low_bmiage_u = (z_bmi < -3.0)
replace low_bmiage_u = . if z_bmi==.
lab var low_bmiage_u "Teenage very low bmi 3sd - WHO"


tab low_bmiage, miss
tab low_bmiage_u, miss 
	/*Bolivia EDSA 2016: 15%  missing values, partially 
	due to age in months higher than 228 months.*/

gen girl_PR=1 
	//Identification variable for girls 14-19 years in PR recode 

rename age_month age_month_g	

	//Retain relevant variables:	
keep ind_id girl_PR age_month_g low_bmiage*
order ind_id girl_PR age_month_g low_bmiage*
sort ind_id
save "$path_out/BOL16_PR_girls.dta", replace


	//Erase files from folder:
erase "$path_out/girl_nutri_bol_z.xls"
erase "$path_out/girl_nutri_bol_prev.xls"
erase "$path_out/girl_nutri_bol_z.dta"


********************************************************************************
*** Step 1.4a  PR - INDIVIDUAL RECODE  
*** (Girls 15-19 years in the household)
*** For harmonised estimates
********************************************************************************

use "$path_out/BOL16_PR.dta", clear

count if hs03_0003_1>=14 & hs03_0003_1<=19 & hs03_0005==2  //2,915 girls 14-19
count if hs03_0003_1>=15 & hs03_0003_1<=19 & hs03_0005==2  //2,443 girls 15-19
keep if hs03_0003_1>=15 & hs03_0003_1<=19 & hs03_0005==2 
count



clonevar gender = hs03_0005
gen bdate = mdy(hs03_0003_2b, hs03_0003_2a, hs03_0003_2c) if hs03_0003_2a <= 31
gen mdate = mdy(dv_00484b, dv_00484a, dv_00484c)
gen age_months = (mdate-bdate)/30.4375 	
lab var age_month "Age in months, individuals 15-19 years (girls)"
gen str6 ageunit = "months" 
lab var ageunit "Months" 
gen weight = hs06_0082 if hs06_0082<9990
gen height = hs06_0083 if hs06_0083<9990
gen oedema = "n"  
gen sw = 1


adopath + "$path_ado/who2007_stata"	
gen str100 reflib = "$path_ado/who2007_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "girl_nutri_bol" 
lab var datalab "Working file"
who2007 reflib datalib datalab gender age_month ageunit weight height oedema sw
use "$path_out/girl_nutri_bol_z.dta", clear 

	
gen	z_bmi = _zbfa
replace z_bmi = . if _fbfa==1 
lab var z_bmi "z-score bmi-for-age WHO"


gen	low_bmiage_c = (z_bmi < -2.0) 
replace low_bmiage_c = . if z_bmi==.
lab var low_bmiage_c "HOT:Teenage low bmi 2sd - WHO"


gen	low_bmiage_u_c = (z_bmi < -3.0)
replace low_bmiage_u_c = . if z_bmi==.
lab var low_bmiage_u_c "HOT:Teenage very low bmi 3sd - WHO"


tab low_bmiage_c, miss
tab low_bmiage_u_c, miss 
gen girl_PR_c=1 

rename age_month age_month_c	

	//Retain relevant variables:	
keep ind_id girl_PR_c age_month_c low_bmiage*
order ind_id girl_PR_c age_month_c low_bmiage*
sort ind_id
save "$path_out/BOL16_PR_girls_c.dta", replace


	//Erase files from folder:
erase "$path_out/girl_nutri_bol_z.xls"
erase "$path_out/girl_nutri_bol_prev.xls"
erase "$path_out/girl_nutri_bol_z.dta"


********************************************************************************
*** Step 1.5  MR - MEN'S RECODE  
***(Eligible man 15-54 years in the household) 
********************************************************************************

use "$path_in/EDSA16_HOMBRES.dta", clear


gen double hh_id = id	

gen double ind_id = id*100 + nro
format ind_id %8.0g
label var ind_id "Individual ID"


duplicates report ind_id 

gen men_MN=1 
	//Identification variable for observations in mn recode
	


	//Retain relevant variables:
keep ind_id men_MN vs02_0201 vs02_0206 vs02_0207_1 vs02_0207_2 ///
vs04_0401 vs04_0402 vs04_0405 
order ind_id men_MN vs02_0201 vs02_0206 vs02_0207_1 vs02_0207_2 ///
vs04_0401 vs04_0402 vs04_0405 
sort ind_id
save "$path_out/BOL16_MN.dta", replace

	
********************************************************************************
*** Step 1.6  PR - INDIVIDUAL RECODE  
*** (Boys 15-19 years in the household)
********************************************************************************

	//No male anthropometric data.
	
********************************************************************************
*** Step 1.7  PR - HOUSEHOLD MEMBER'S RECODE 
********************************************************************************

use "$path_out/BOL16_PR.dta", clear

sort hh_id ind_id


********************************************************************************
*** Step 1.8 DATA MERGING 
******************************************************************************** 
 
 
*** Merging BR Recode 
*****************************************
merge 1:1 ind_id using "$path_out/BOL16_BR.dta" 
	// 8,078 women matched 
drop _merge


*** Merging IR Recode 
*****************************************
merge 1:1 ind_id using "$path_out/BOL16_IR.dta" 
	//  12,274 women matched 	
drop _merge


*** Merging 14-19 years: girls 
*****************************************
merge 1:1 ind_id using "$path_out/BOL16_PR_girls.dta" 
drop _merge


*** Merging 15-19 years: girls 
*** For harmonised estimates only
*****************************************
merge 1:1 ind_id using "$path_out/BOL16_PR_girls_c.dta" 
drop _merge



*** Merging MR Recode 
*****************************************
merge 1:1 ind_id using "$path_out/BOL16_MN.dta" 
	//4,975 matched observation
drop _merge


*** Merging 15-19 years: boys 
*****************************************
gen age_month_b = .
lab var age_month_b "Age in months, individuals 15-19 years (boys)"	

gen	low_bmiage_b = .
lab var low_bmiage_b "Teenage low bmi 2sd - WHO (boys)"

gen	low_bmiage_b_u = .
lab var low_bmiage_b_u "Teenage very low bmi 3sd - WHO (boys)"



*** Merging child under 5 
*****************************************
merge 1:1 ind_id using "$path_out/BOL16_PR_child.dta" 
	// 5,845 matched information
drop _merge


*** Merging children 61-72 months 
*****************************************
merge 1:1 ind_id using "$path_out/BOL16_CH_6Y.dta"
	// 606 matched information
drop _merge


sort ind_id


********************************************************************************
*** Step 1.9 KEEP ONLY DE JURE HOUSEHOLD MEMBERS ***
********************************************************************************

clonevar resident =  hs03_0010 
tab resident, miss 	
drop if resident!=1 
	//99.45% (50,744) of the household members are "de jure" members
label var resident "Permanent (de jure) household member"

	
********************************************************************************
*** 1.10 SUBSAMPLE VARIABLE ***
********************************************************************************

	/*In the context of Bolivia EDSA 2016, height and weight 
	measurements were collected from children (0-5) and women (15-49) 
	for the full sample of households.*/

gen subsample = .
label var subsample "Households selected as part of nutrition subsample" 


********************************************************************************
*** Step 1.11 CONTROL VARIABLES
********************************************************************************


*** No eligible women 14-49 years 
*** for adult nutrition indicator
***********************************************
count if hs06_0084!=.
	//12,533 women 14-49 eligible for anthropometric measure
tab hs06_0084 if hs03_0003_1>=14 & hs03_0003_1<=49 & hs03_0005==2, m	
	//12,124 women out of 12,533 are measured 

gen fem_nutri_eligible = (hs06_0084!=.)
tab fem_nutri_eligible, miss
bysort hh_id: egen hh_n_fem_nutri_eligible = sum(fem_nutri_eligible) 	
gen	no_fem_nutri_eligible = (hh_n_fem_nutri_eligible==0)
lab var no_fem_nutri_eligible "Household has no eligible women for anthropometric"	
drop hh_n_fem_nutri_eligible
tab no_fem_nutri_eligible, miss


*** No eligible women 14-49 years 
*** for child mortality indicator
*****************************************
count if hs03_0003_1>=14 & hs03_0003_1<=49 & hs03_0005==2 
count if women_IR==1
gen	fem_eligible = (hs03_0003_1>=14 & hs03_0003_1<=49 & hs03_0005==2)
tab fem_eligible, miss 
	//Eligible women 14-49 for child mortality: 12,394
bysort	hh_id: egen hh_n_fem_eligible = sum(fem_eligible) 	
gen	no_fem_eligible = (hh_n_fem_eligible==0) 									
lab var no_fem_eligible "Household has no eligible women for interview"
drop hh_n_fem_eligible 
tab no_fem_eligible, miss 


*** No eligible men 
*** for adult nutrition indicator (if relevant)
***********************************************
	//Bolivia EDSA 2016 has no male anthropometric data.
gen	male_nutri_eligible = .	
gen	no_male_nutri_eligible = .
lab var no_male_nutri_eligible "Household has no eligible men for anthropometric"	


*** No eligible men 15-64 years
*** for child mortality indicator (if relevant)
*****************************************
count if hs03_0003_1>=15 & hs03_0003_1<=64 & hs03_0005==1 & dv_0003_3==1 
count if men_MN==1
gen male_eligible = (hs03_0003_1>=15 & hs03_0003_1<=64 & hs03_0005==1 & dv_0003_3==1)
bysort	hh_id: egen hh_n_male_eligible = sum(male_eligible)  
gen	no_male_eligible = (hh_n_male_eligible==0) 	
lab var no_male_eligible "Household has no eligible man for interview"
drop hh_n_male_eligible
tab no_male_eligible, miss


*** No eligible children 0-60 months
*** for child nutrition indicator
*****************************************
count if hs06_0093!=.
count if child_CH==1 
count if child_CH_6y==1


gen	child_eligible = (child_CH==1)
bysort	hh_id: egen hh_n_children_eligible = sum(child_eligible)  
	//Number of eligible children for anthropometrics
gen	no_child_eligible = (hh_n_children_eligible==0) 
	//Takes value 1 if there were no eligible children for anthropometrics
lab var no_child_eligible "Household has no children eligible for anthropometric"
drop hh_n_children_eligible
tab no_child_eligible, miss


*** No Eligible Children 61-69 months
*****************************************
gen	child_eligible_6y = (child_CH_6y==1)
bysort	hh_id: egen hh_n_children_eligible_6y = sum(child_eligible_6y)  
	//Number of eligible children for anthropometrics
gen	no_child_eligible_6y = (hh_n_children_eligible_6y==0) 
	//Takes value 1 if there were no eligible children for anthropometrics
lab var no_child_eligible_6y "Household has no children 61-69 months eligible"	
tab no_child_eligible_6y, miss	
	

*** No eligible women and men 
*** for adult nutrition indicator
***********************************************
gen no_adults_eligible = (no_fem_nutri_eligible==1)
lab var no_adults_eligible "Household has no eligible women or men for anthropometrics"
tab no_adults_eligible, miss 


*** No Eligible Children and Women
*** for child and women nutrition indicator 
***********************************************
gen	no_child_fem_eligible = (no_child_eligible==1 & no_fem_nutri_eligible==1 & no_child_eligible_6y)
lab var no_child_fem_eligible "Household has no children or women eligible for anthropometric"
tab no_child_fem_eligible, miss 


*** No Eligible Women, Men or Children 
*** for nutrition indicator 
***********************************************
gen no_eligibles = (no_fem_nutri_eligible==1 & no_child_eligible==1 & no_child_eligible_6y)
lab var no_eligibles "Household has no eligible women, men, or children"
tab no_eligibles, miss 


sort hh_id ind_id


********************************************************************************
*** Step 1.12 RENAMING DEMOGRAPHIC VARIABLES ***
********************************************************************************

//Sample weight
desc ponderadorh
clonevar weight = ponderadorh
label var weight "Sample weight"


//Area: urban or rural	
desc area
codebook area, tab (5)		
replace area=0 if area==2  
label define lab_area 1 "urban" 0 "rural"
label values area lab_area
label var area "Area: urban-rural"


//Relationship to the head of household 
clonevar relationship = hs03_0004 
codebook relationship, tab (20)
recode relationship (1=1)(2=2)(3/4=3)(5/10=4)(11=5)(12=6)(13=5)(98=.)
label define lab_rel 1"head" 2"spouse" 3"child" 4"extended family" 5"not related" 6"maid"
label values relationship lab_rel
label var relationship "Relationship to the head of household"
tab hs03_0004 relationship, miss


//Sex of household member	
codebook hs03_0005
clonevar sex = hs03_0005
label define lab_sex 1"male" 2"female"
label values sex lab_sex
label var sex "Sex of household member"


//Household headship
bys	hh_id: egen missing_hhead = min(relationship)
tab missing_hhead,m 
gen household_head=.
replace household_head=1 if relationship==1 & sex==1 
replace household_head=2 if relationship==1 & sex==2
bysort hh_id: egen headship = sum(household_head)
replace headship = 1 if (missing_hhead==2 & sex==1)
replace headship = 2 if (missing_hhead==2 & sex==2)
replace headship = . if missing_hhead>2
label define head 1"male-headed" 2"female-headed"
label values headship head
label var headship "Household headship"
tab headship, miss


//Age of household member
codebook hs03_0003_1, tab (999)
clonevar age = hs03_0003_1  
label var age "Age of household member"


//Age group 
recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+"), gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"


//Marital status of household member
gen marital = 1 if ((ms06_0601 == 2 | ms06_0601 == 3) & (ms06_0602==2 | ms06_0602==3 | ms06_0602==.) & ms06_0603==. ) | ((vs04_0401 == 2 | vs04_0401 == 4) & (vs04_0402==. | vs04_0402==2 | vs04_0402==3) & (vs04_0405==4 | vs04_0405==.))
	//1: Never married
replace marital = 2 if (ms06_0601 == 1 & ms06_0602==. & ms06_0603==.) | (vs04_0401 == 1 & vs04_0402==. & vs04_0405==.)
	//2: Currently married
replace marital = 3 if (ms06_0601 == 3 & ms06_0602==1 & ms06_0603==1) | (vs04_0401 == 4 & vs04_0402==1 & vs04_0405==1)
	//3: Widowed	
replace marital = 4 if (ms06_0601 == 3 & ms06_0602==1 & ms06_0603==3) | (vs04_0401 == 4 & vs04_0402==1 & vs04_0405==3)
	//4: Divorced	
replace marital = 5 if (ms06_0601 == 3 & ms06_0602==1 & ms06_0603==2) | (vs04_0401 == 4 & vs04_0402==1 & vs04_0405==2)
	//5: Separated/not living together	
label define lab_marr 1"never married" 2"currently married" 3"widowed" ///
4"divorced" 5"not living together"
label values marital lab_marr	
label var marital "Marital status of household member"
tab marital, miss


//Total number of de jure hh members in the household
gen member = 1
bysort hh_id: egen hhsize = sum(member)
label var hhsize "Household size"
tab hhsize, miss
drop member


//Subnational region
	/*The Demographic and Health Survey - EDSA 2016- is a population-based
	survey, representative at the national, departmental (9), 
	areas (urban and rural) and regions (altiplano,valleys and plains), 
	provides relevant information on maternal and child health, family 
	planning, fertility, mortality, lactation, nutrition and sexually 
	transmitted diseases, violence against women and men and other issues 
	that affect the health of the population.(p.6 Documento metodologico).*/	
codebook region, tab (99)
rename region region_o
codebook departamento, tab (99)
decode departamento, gen(temp)
replace temp =  proper(temp)
encode temp, gen(region)
drop temp
codebook region, tab (99)
recode region (4=1)(5=2)(6=3)(7=4)(8=5)(9=6)(10=7)(11=8)(12=9)
label define lab_reg 1"Beni" 2"Chuquisaca" 3"Cochabamba" 4"La Paz" ///
5"Oruro" 6"Pando" 7"Potosi" 8"Santa Cruz" 9"Tarija"
label values region lab_reg
lab var region "Region for subnational decomposition"


	/*In this survey, the harmonised 'region_01' variable is the 
	same as the standardised 'region' variable.*/	
clonevar region_01 = region
lab var region_01 "HOT: Region for subnational decomposition"


********************************************************************************
***  Step 2 Data preparation  ***
***  Standardization of the 10 Global MPI indicators 
***  Identification of non-deprived & deprived individuals  
********************************************************************************


********************************************************************************
*** Step 2.1 Years of Schooling ***
********************************************************************************

codebook anioae, tab(30)
clonevar  eduyears = anioae   
	//Total number of years of education
replace eduyears = . if eduyears>30
tab anioae nivedu,miss
codebook nivedu, tab (99)
replace eduyears = . if eduyears>=age & age>0
	/*The variable "eduyears" was replaced with a '.' if total years of 
	education was more than individual's age*/
replace eduyears = 0 if age < 10 
replace eduyears = 0 if (age==10 | age==11 ) & eduyears < 6 
	/*The variable "eduyears" was replaced with a '0' for ineligible household 
	members, i.e.: those who have not completed 6 years of schooling following 
	their starting school age */


	/*A control variable is created on whether there is information on 
	years of education for at least 2/3 of the eligible household members*/	
gen temp = 1 if eduyears!=. & age>=12 & age!=.
replace temp = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp = 1 if age==11 & eduyears>=6 & eduyears<.
bysort	hh_id: egen no_missing_edu = sum(temp)
	//Total eligible household members with no missing years of education
gen temp2 = 1 if age>=12 & age!=.
replace temp2 = 1 if age==10 & eduyears>=6 & eduyears<.
replace temp2 = 1 if age==11 & eduyears>=6 & eduyears<.
bysort hh_id: egen hhs = sum(temp2)
	//Total number of eligible household members  
replace no_missing_edu = no_missing_edu/hhs
replace no_missing_edu = (no_missing_edu>=2/3)
	/*Identify whether there is information on years of education for at 
	least 2/3 of the eligible household members */
tab no_missing_edu, miss
	//The values for 0 (missing) is 0%
label var no_missing_edu "No missing edu for at least 2/3 of the eligible HH members"		
drop temp temp2 hhs


*** Standard MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed SIX years of schooling. */
******************************************************************* 
gen	 years_edu6 = (eduyears>=6)
replace years_edu6 = . if eduyears==.
bysort hh_id: egen hh_years_edu6_1 = max(years_edu6)
gen	hh_years_edu6 = (hh_years_edu6_1==1)
replace hh_years_edu6 = . if hh_years_edu6_1==.
replace hh_years_edu6 = . if hh_years_edu6==0 & no_missing_edu==0 
lab var hh_years_edu6 "Household has at least one member with 6 years of edu"
tab hh_years_edu6, miss

	
*** Destitution MPI ***
/*The entire household is considered deprived if no eligible 
household member has completed at least one year of schooling. */
*******************************************************************
gen	years_edu1 = (eduyears>=1)
replace years_edu1 = . if eduyears==.
bysort	hh_id: egen hh_years_edu_u = max(years_edu1)
replace hh_years_edu_u = . if hh_years_edu_u==0 & no_missing_edu==0
lab var hh_years_edu_u "Household has at least one member with 1 year of edu"
tab hh_years_edu_u, miss


********************************************************************************
*** Step 2.2 Child School Attendance ***
********************************************************************************

codebook hs04_c_0029, tab (99) 
	/*hs04_c_0029: attend the level and course that you 
	enrolled in this 2016*/

clonevar attendance = hs04_c_0029
recode  attendance (2=0)
	//1=attending, 0=not attending 
label define lab_attend 1 "currently attending" 0 "not currently attending"
label values attendance lab_attend
label var attendance "Attended school during current school year"
codebook attendance, tab (99)
replace attendance = 0 if attendance==. & hs04_b_0025==2 
replace attendance = 0 if attendance==. & hs04_c_0027==2


*** Standard MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 8. */ 
******************************************************************* 
gen	child_schoolage = (age>=6 & age<=14)
	/*In Bolivia, the official school entrance age to primary school is 
	6 years. So, age range is 6-14 (=6+8)*/

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the school age children */
count if child_schoolage==1 & attendance==.
	//How many eligible school aged children are not attending school: 230 children
gen temp = 1 if child_schoolage==1 & attendance!=.
	/*Generate a variable that captures the number of eligible school aged 
	children who are attending school */
bysort hh_id: egen no_missing_atten = sum(temp)	
	/*Total school age children with no missing information on school 
	attendance */
gen temp2 = 1 if child_schoolage==1	
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are of school age
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)
	/*Identify whether there is missing information on school attendance for 
	more than 2/3 of the school age children */			
tab no_missing_atten, miss
	//The values for 0 (missing) is 1.26%
label var no_missing_atten "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs
	
	
bysort hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
	//It takes value 1 if the household has children in school age
lab var hh_children_schoolage "Household has children in school age"


gen	child_not_atten = (attendance==0) if child_schoolage==1
replace child_not_atten = . if attendance==. & child_schoolage==1
bysort	hh_id: egen any_child_not_atten = max(child_not_atten)
gen	hh_child_atten = (any_child_not_atten==0) 
replace hh_child_atten = . if any_child_not_atten==.
replace hh_child_atten = 1 if hh_children_schoolage==0
replace hh_child_atten = . if hh_child_atten==1 & no_missing_atten==0 
lab var hh_child_atten "Household has all school age children up to class 8 in school"
tab hh_child_atten, miss 


*** Destitution MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 6. */ 
******************************************************************* 
gen	child_schoolage_6 = (age>=6 & age<=12) 
	/*In Bolivia, the official school entrance age is 6 years.  
	  So, age range for destitution measure is 6-12 (=6+6) */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the children attending school up to 
	class 6 */	
count if child_schoolage_6==1 & attendance==.	
gen temp = 1 if child_schoolage_6==1 & attendance!=.
bysort hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_6==1	
bysort hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)			
tab no_missing_atten_u, miss
label var no_missing_atten_u "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs		
		
bysort	hh_id: egen hh_children_schoolage_6 = sum(child_schoolage_6)
replace hh_children_schoolage_6 = (hh_children_schoolage_6>0) 
lab var hh_children_schoolage_6 "Household has children in school age (6 years of school)"

gen	child_atten_6 = (attendance==1) if child_schoolage_6==1
replace child_atten_6 = . if attendance==. & child_schoolage_6==1
bysort	hh_id: egen any_child_atten_6 = max(child_atten_6)
gen	hh_child_atten_u = (any_child_atten_6==1) 
replace hh_child_atten_u = . if any_child_atten_6==.
replace hh_child_atten_u = 1 if hh_children_schoolage_6==0
replace hh_child_atten_u = . if hh_child_atten_u==0 & no_missing_atten_u==0 
lab var hh_child_atten_u "Household has at least one school age children up to class 6 in school"
tab hh_child_atten_u, miss


********************************************************************************
*** Step 2.3 Nutrition ***
********************************************************************************

********************************************************************************
*** Step 2.3a Adult Nutrition ***
********************************************************************************
desc hs06_0082 hs06_0083
gen bmi = hs06_0082/(hs06_0083/100)^2
sum bmi  


foreach var in bmi {
			 gen inf_`var' = 1 if `var'!=.
			 bysort sex: tab age inf_`var' 
			 /*Bolivia EDSA 2016 has anthropometric data for all women 
			 14-49 years: 12,196 women 14-49 years old.*/
			 drop inf_`var'
			 }
***


*** BMI Indicator for Women 14-49 years ***
******************************************************************* 
gen	f_bmi = bmi
lab var f_bmi "Women's BMI"
gen	f_low_bmi = (f_bmi<18.5)
replace f_low_bmi = . if f_bmi==. | f_bmi>=99.97
lab var f_low_bmi "BMI of women < 18.5"

gen	f_low_bmi_u = (f_bmi<17)
replace f_low_bmi_u = . if f_bmi==. | f_bmi>=99.97
lab var f_low_bmi_u "BMI of women <17"

	

*** BMI Indicator for Men ***
******************************************************************* 
	//Note: Bolivia EDSA 2016 has no anthropometric data for men. 
	
gen m_bmi = .
lab var m_bmi "Male's BMI"
gen m_low_bmi = .
lab var m_low_bmi "BMI of male < 18.5"

gen m_low_bmi_u = .
lab var m_low_bmi_u "BMI of male <17"


*** Standard MPI: BMI-for-age for individuals 14-19 years 
*** 				  and BMI for individuals 20-49 years ***
*******************************************************************  
gen low_bmi_byage = 0
lab var low_bmi_byage "Individuals with low BMI or BMI-for-age"
replace low_bmi_byage = 1 if f_low_bmi==1
	//Replace variable "low_bmi_byage = 1" if eligible women have low BMI	
replace low_bmi_byage = 1 if low_bmi_byage==0 & m_low_bmi==1 
	/*Replace variable "low_bmi_byage = 1" if eligible men have low BMI. If 
	there is no male anthropometric data, then 0 changes are made.*/

	
/*Note: The following command replaces BMI with BMI-for-age for those between 
the age group of 14-19 by their age in months where information is available */
	//Replacement for girls: 
replace low_bmi_byage = 1 if low_bmiage==1 & age_month_g!=.
replace low_bmi_byage = 0 if low_bmiage==0 & age_month_g!=.
	/*Replacements for boys - if there is no male anthropometric data for boys, 
	then 0 changes are made: */
replace low_bmi_byage = 1 if low_bmiage_b==1 & age_month_b!=.
replace low_bmi_byage = 0 if low_bmiage_b==0 & age_month_b!=.
	
	
/*Note: The following control variable is applied when there is BMI information 
for adults and BMI-for-age for teenagers.*/	
replace low_bmi_byage = . if f_low_bmi==. & m_low_bmi==. & low_bmiage==. & low_bmiage_b==. 
		
bysort hh_id: egen low_bmi = max(low_bmi_byage)
gen	hh_no_low_bmiage = (low_bmi==0)
	/*Households take a value of '1' if all eligible adults and teenagers in the 
	household has normal bmi or bmi-for-age */	
replace hh_no_low_bmiage = . if low_bmi==.
	/*Households take a value of '.' if there is no information from eligible 
	individuals in the household */
replace hh_no_low_bmiage = 1 if no_adults_eligible==1	
	//Households take a value of '1' if there is no eligible adult population.
drop low_bmi
lab var hh_no_low_bmiage "Household has no adult with low BMI or BMI-for-age"
tab hh_no_low_bmiage, miss 

	
*** Destitution MPI: BMI-for-age for individuals 14-19 years 
*** 			     and BMI for individuals 20-49 years ***
********************************************************************************
gen low_bmi_byage_u = 0
replace low_bmi_byage_u = 1 if f_low_bmi_u==1
	/*Replace variable "low_bmi_byage_u = 1" if eligible women have low 
	BMI (destitute cutoff)*/	
replace low_bmi_byage_u = 1 if low_bmi_byage_u==0 & m_low_bmi_u==1 
	/*Replace variable "low_bmi_byage_u = 1" if eligible men have low 
	BMI (destitute cutoff). If there is no male anthropometric data, then 0 
	changes are made.*/

	
/*Note: The following command replaces BMI with BMI-for-age for those between 
the age group of 14-19 by their age in months where information is available */
	//Replacement for girls: 
replace low_bmi_byage_u = 1 if low_bmiage_u==1 & age_month_g!=.
replace low_bmi_byage_u = 0 if low_bmiage_u==0 & age_month_g!=.
	/*Replacements for boys - if there is no male anthropometric data for boys, 
	then 0 changes are made: */
replace low_bmi_byage_u = 1 if low_bmiage_b_u==1 & age_month_b!=.
replace low_bmi_byage_u = 0 if low_bmiage_b_u==0 & age_month_b!=.
	
	
/*Note: The following control variable is applied when there is BMI information 
for adults and BMI-for-age for teenagers. */
replace low_bmi_byage_u = . if f_low_bmi_u==. & low_bmiage_u==. & m_low_bmi_u==. & low_bmiage_b_u==. 

		
bysort hh_id: egen low_bmi = max(low_bmi_byage_u)
gen	hh_no_low_bmiage_u = (low_bmi==0)
	/*Households take a value of '1' if all eligible adults and teenagers in the 
	household has normal bmi or bmi-for-age (destitution cutoff) */
replace hh_no_low_bmiage_u = . if low_bmi==.
	/*Households take a value of '.' if there is no information from eligible 
	individuals in the household */
replace hh_no_low_bmiage_u = 1 if no_adults_eligible==1	
	//Households take a value of '1' if there is no eligible adult population.
drop low_bmi
lab var hh_no_low_bmiage_u "Household has no adult with low BMI or BMI-for-age(<17/-3sd)"
tab hh_no_low_bmiage_u, miss 


********************************************************************************
*** Step 2.3b Child Nutrition ***
********************************************************************************

*** Child Underweight Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(underweight)
gen	hh_no_underweight = (temp==0) 
	//Takes value 1 if no child in the hh is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_underweight "Household has no child underweight - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(underweight_u)
gen	hh_no_underweight_u = (temp==0) 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
lab var hh_no_underweight_u "Destitute: Household has no child underweight"
drop temp


*** Child Stunting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(stunting)
gen	hh_no_stunting = (temp==0) 
	//Takes value 1 if no child in the hh is stunted
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_stunting "Household has no child stunted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(stunting_u)
gen	hh_no_stunting_u = (temp==0) 
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
lab var hh_no_stunting_u "Destitute: Household has no child stunted"
drop temp


*** Child Wasting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(wasting)
gen	hh_no_wasting = (temp==0) 
	//Takes value 1 if no child in the hh is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_wasting "Household has no child wasted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(wasting_u)
gen	hh_no_wasting_u = (temp==0) 
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
lab var hh_no_wasting_u "Destitute: Household has no child wasted"
drop temp


*** Child Either Underweight or Stunted Indicator ***
************************************************************************

*** Standard MPI ***
gen uw_st = 1 if stunting==1 | underweight==1
	//Takes value 1 if child in the hh is stunted or underweight 
replace uw_st = 0 if stunting==0 & underweight==0
	//Takes value 0 if child in the hh is not stunted and not underweight 
replace uw_st = . if stunting==. & underweight==.
bysort hh_id: egen temp = max(uw_st)
gen hh_no_uw_st = (temp==0) 
replace hh_no_uw_st = . if temp==.
drop temp
replace hh_no_uw_st = 1 if no_child_eligible==1
	//Households with no eligible children will receive a value of 1 
lab var hh_no_uw_st "HH has no child underweight or stunted"


*** Destitution MPI  ***
gen uw_st_u = 1 if stunting_u==1 | underweight_u==1
replace uw_st_u = 0 if stunting_u==0 & underweight_u==0
replace uw_st_u = . if stunting_u==. & underweight_u==.
bysort hh_id: egen temp = max(uw_st_u)
gen hh_no_uw_st_u = (temp==0) 
replace hh_no_uw_st_u = . if temp==.
drop temp
replace hh_no_uw_st_u = 1 if no_child_eligible==1
lab var hh_no_uw_st_u "DST: HH has no child underweight or stunted"


********************************************************************************
*** Step 2.3c Children 61-69 months Nutrition ***
********************************************************************************

*** Standard MPI: BMI-for-age for children 61-69 months  ***
************************************************************************
bysort hh_id: egen temp = max(low_bmiage_6y)
gen	hh_no_low_bmiage_6y = (temp==0) 
	//Takes value 1 if no child 61-69 months in the hh has low BMI-for-age 
replace hh_no_low_bmiage_6y = . if temp==.
replace hh_no_low_bmiage_6y = 1 if no_child_eligible_6y==1 
	//Households with no eligible child 61-69 months will receive a value of 1 
lab var hh_no_low_bmiage_6y "HH has no children 61-69 months with low BMI-for-age"
drop temp


*** Destitute MPI: BMI-for-age for children 61-69 months ***
************************************************************************
bysort hh_id: egen temp = max(low_bmiage_6y_u)
gen	hh_no_low_bmiage_6y_u = (temp==0) 
	//Takes value 1 if no child 61-69 months in the hh has low BMI-for-age 
replace hh_no_low_bmiage_6y_u = . if temp==.
replace hh_no_low_bmiage_6y_u = 1 if no_child_eligible_6y==1 
	//Households with no eligible child 61-69 months will receive a value of 1 
lab var hh_no_low_bmiage_6y_u "DST: HH has no children 61-69 months with low BMI-for-age"
drop temp


********************************************************************************
*** Step 2.3d Household Nutrition Indicator ***
********************************************************************************

*** Standard MPI ***
/* Members of the household are considered deprived if the household 
has a child 0-60 months whose height-for-age or weight-for-age is under 
two standard deviation below the median, or has young people with 
BMI-for-age that is under two standard deviation below the median, 
or has adults with BMI threshold that is below 18.5 kg/m2.*/
************************************************************************
gen	hh_nutrition_uw_st = 1
replace hh_nutrition_uw_st = 0 if hh_no_low_bmiage==0 | hh_no_uw_st==0 | hh_no_low_bmiage_6y==0
replace hh_nutrition_uw_st = . if hh_no_low_bmiage==. & hh_no_uw_st==. & hh_no_low_bmiage_6y==.
	/*Replace indicator as missing if household has eligible 
	adult and child with missing nutrition information.*/								  
replace hh_nutrition_uw_st = . if no_adults_eligible==0 & hh_no_low_bmiage==. & ///
								  no_child_eligible_6y==1 & hh_no_low_bmiage_6y==1 & ///
								  no_child_eligible==1 & hh_no_uw_st==1								  
	/*Replace indicator as missing if household has eligible adult 
	with missing nutrition information and no eligible child 
	0-60 months or 61-69 months for anthropometric measures*/ 								  
replace hh_nutrition_uw_st = . if no_adults_eligible==1 & hh_no_low_bmiage==1 & ///
								  no_child_eligible_6y==1 & hh_no_low_bmiage_6y==1 & ///
								  no_child_eligible==0 & hh_no_uw_st==.								  					  
	/*Replace indicator as missing if household has eligible 
	child 0-60 months with missing nutrition information and 
	no eligible adult or child 61-69 months for anthropometric measures.*/ 		
replace hh_nutrition_uw_st = . if no_adults_eligible==1 & hh_no_low_bmiage==1 & ///
								  no_child_eligible_6y==0 & hh_no_low_bmiage_6y==. & ///
								  no_child_eligible==1 & hh_no_uw_st==1			
	/*Replace indicator as missing if household has eligible 
	child 61-69 months with missing nutrition information and 
	no eligible adult and child 0-60 months for anthropometric measures.*/ 	
replace hh_nutrition_uw_st = . if no_adults_eligible==0 & hh_no_low_bmiage==. & ///
								  no_child_eligible_6y==0 & hh_no_low_bmiage_6y==. & ///
								  no_child_eligible==1 & hh_no_uw_st==1
	/*Replace indicator as missing if household has eligible adult 
	 and child 61-69 with missing nutrition information and no eligible child 
	0-60 months for anthropometric measures*/ 
replace hh_nutrition_uw_st = . if no_adults_eligible==0 & hh_no_low_bmiage==. & ///
								  no_child_eligible_6y==1 & hh_no_low_bmiage_6y==1 & ///
								  no_child_eligible==0 & hh_no_uw_st==.
	/*Replace indicator as missing if household has eligible adult 
	 and child 0-60 with missing nutrition information and no eligible child 
	61-69 months for anthropometric measures*/  		
replace hh_nutrition_uw_st = . if no_adults_eligible==1 & hh_no_low_bmiage==1 & ///
								  no_child_eligible_6y==0 & hh_no_low_bmiage_6y==. & ///
								  no_child_eligible==0 & hh_no_uw_st==.
	/*Replace indicator as missing if household has eligible child 
	 0-69 with missing nutrition information and no eligible adult for 
	 anthropometric measures*/  		
replace hh_nutrition_uw_st = 1 if no_eligibles==1   	
lab var hh_nutrition_uw_st "Household has no individuals malnourished"
tab hh_nutrition_uw_st, miss


*** Destitution MPI ***
/* Members of the household are considered deprived if the household 
has a child under 5 whose height-for-age or weight-for-age is under 
three standard deviation below the median, or has young people with 
BMI-for-age that is under three standard deviation below the median, or 
has adults with BMI threshold that is below 17.0 kg/m2.*/
************************************************************************
gen	hh_nutrition_uw_st_u = 1
replace hh_nutrition_uw_st_u = 0 if hh_no_low_bmiage_u==0 | hh_no_uw_st_u==0 | hh_no_low_bmiage_6y_u==0
replace hh_nutrition_uw_st_u = . if hh_no_low_bmiage_u==. & hh_no_uw_st_u==. & hh_no_low_bmiage_6y_u==.
	/*Replace indicator as missing if household has eligible 
	adult and child with missing nutrition information.*/
replace hh_nutrition_uw_st_u = . if no_adults_eligible==0 & hh_no_low_bmiage_u==. & ///
								    no_child_eligible_6y==1 & hh_no_low_bmiage_6y_u==1 & ///
								    no_child_eligible==1 & hh_no_uw_st_u==1								  
	/*Replace indicator as missing if household has eligible adult 
	with missing nutrition information and no eligible child 
	0-60 months or 61-69 months for anthropometric measures*/ 								  
replace hh_nutrition_uw_st_u = . if no_adults_eligible==1 & hh_no_low_bmiage_u==1 & ///
								  no_child_eligible_6y==1 & hh_no_low_bmiage_6y_u==1 & ///
								  no_child_eligible==0 & hh_no_uw_st_u==.								  					  
	/*Replace indicator as missing if household has eligible 
	child 0-60 months with missing nutrition information and 
	no eligible adult or child 61-69 months for anthropometric measures.*/ 		
replace hh_nutrition_uw_st_u = . if no_adults_eligible==1 & hh_no_low_bmiage_u==1 & ///
								  no_child_eligible_6y==0 & hh_no_low_bmiage_6y_u==. & ///
								  no_child_eligible==1 & hh_no_uw_st_u==1			
	/*Replace indicator as missing if household has eligible 
	child 61-69 months with missing nutrition information and 
	no eligible adult and child 0-60 months for anthropometric measures.*/ 	
replace hh_nutrition_uw_st_u = . if no_adults_eligible==0 & hh_no_low_bmiage_u==. & ///
								  no_child_eligible_6y==0 & hh_no_low_bmiage_6y_u==. & ///
								  no_child_eligible==1 & hh_no_uw_st_u==1
	/*Replace indicator as missing if household has eligible adult 
	 and child 61-69 with missing nutrition information and no eligible child 
	0-60 months for anthropometric measures*/ 
replace hh_nutrition_uw_st = . if no_adults_eligible==0 & hh_no_low_bmiage_u==. & ///
								  no_child_eligible_6y==1 & hh_no_low_bmiage_6y_u==1 & ///
								  no_child_eligible==0 & hh_no_uw_st_u==.
	/*Replace indicator as missing if household has eligible adult 
	 and child 0-60 with missing nutrition information and no eligible child 
	61-69 months for anthropometric measures*/  		
replace hh_nutrition_uw_st = . if no_adults_eligible==1 & hh_no_low_bmiage_u==1 & ///
								  no_child_eligible_6y==0 & hh_no_low_bmiage_6y_u==. & ///
								  no_child_eligible==0 & hh_no_uw_st_u==.
	/*Replace indicator as missing if household has eligible child 
	 0-69 with missing nutrition information and no eligible adult for 
	 anthropometric measures*/  		
replace hh_nutrition_uw_st_u = 1 if no_eligibles==1   	
lab var hh_nutrition_uw_st_u "Household has no individuals malnourished"
tab hh_nutrition_uw_st_u, miss



*** Harmonised MPI *** 
	/*In this survey, anthropometric measurements were collected from 
	children under 0-5 years and women 14-49. For harmonization purposes, 
	we use the anthropometric information from only children under 5 and women 
	15-49. We only count women between the age of 15-49 (rather than 14-49), 
	to accord with the earlier surveys.*/
************************************************************************

//Revised eligibility variable for women (15-19)

count if hs03_0003_1>=14 & hs03_0003_1<=49 & hs03_0005==2
count if hs06_0084!=.
	//12,533 women 14-49 

count if hs03_0003_1>=15 & hs03_0003_1<=49 & hs03_0005==2
count if hs06_0084!=. & hs03_0003_1>=15 & hs03_0003_1<=49 & hs03_0005==2
	//12,063 women 15-49
	
gen fem_nutri_eligible_c = (hs06_0084!=. & hs03_0003_1>=15 & hs03_0003_1<=49 & ///
						  hs03_0005==2)
tab fem_nutri_eligible_c, miss
bysort hh_id: egen hh_n_fem_nutri_eligible_c = sum(fem_nutri_eligible_c) 	
gen	no_fem_nutri_eligible_c = (hh_n_fem_nutri_eligible_c==0)
lab var no_fem_nutri_eligible_c "HOT:HH has no eligible women for anthropometric"	
drop hh_n_fem_nutri_eligible_c
tab no_fem_nutri_eligible_c, miss


gen no_adults_eligible_c = (no_fem_nutri_eligible_c==1)
lab var no_adults_eligible_c "Household has no eligible women or men for anthropometrics"

gen no_eligibles_c = (no_fem_nutri_eligible_c==1 & no_child_eligible==1)
lab var no_eligibles_c "HOT:HH has no eligible women, men, or children"


gen	f_bmi_c = bmi
lab var f_bmi_c "HOT: Women's BMI"
replace f_bmi_c = . if hs03_0003_1<=14 & hs03_0005==2
gen	f_low_bmi_c = (f_bmi_c<18.5)
replace f_low_bmi_c = . if f_bmi_c==. | f_bmi_c>=99.97
lab var f_low_bmi_c "HOT: BMI of women < 18.5"

gen	f_low_bmi_u_c = (f_bmi_c<17)
replace f_low_bmi_u_c = . if hs03_0003_1<=14 & hs03_0005==2
replace f_low_bmi_u_c = . if f_bmi_c==. | f_bmi_c>=99.97
lab var f_low_bmi_u_c "DST-HOT:BMI of women <17"


gen low_bmi_byage_c = 0
lab var low_bmi_byage_c "HOT:Individuals with low BMI or BMI-for-age"
replace low_bmi_byage_c = 1 if f_low_bmi_c==1	
replace low_bmi_byage_c = 1 if low_bmiage_c==1 & age_month_c!=.
replace low_bmi_byage_c = 0 if low_bmiage_c==0 & age_month_c!=.
replace low_bmi_byage_c = . if f_low_bmi_c==. & low_bmiage_c==. 
		
bysort hh_id: egen low_bmi = max(low_bmi_byage_c)
gen	hh_no_low_bmiage_c = (low_bmi==0)
replace hh_no_low_bmiage_c = . if low_bmi==.
replace hh_no_low_bmiage_c = 1 if no_adults_eligible_c==1	
drop low_bmi
lab var hh_no_low_bmiage_c "DST-HOT: HH has no adult with low BMI or BMI-for-age"

tab hh_no_low_bmiage, miss
tab hh_no_low_bmiage_c, miss 


gen low_bmi_byage_u_c = 0
replace low_bmi_byage_u_c = 1 if f_low_bmi_u_c==1	
replace low_bmi_byage_u_c = 1 if low_bmiage_u_c==1 & age_month_c!=.
replace low_bmi_byage_u_c = 0 if low_bmiage_u_c==0 & age_month_c!=.
replace low_bmi_byage_u_c = . if f_low_bmi_u_c==. & low_bmiage_u_c==. 
		
bysort hh_id: egen low_bmi = max(low_bmi_byage_u_c)
gen	hh_no_low_bmiage_u_c = (low_bmi==0)
replace hh_no_low_bmiage_u_c = . if low_bmi==.
replace hh_no_low_bmiage_u_c = 1 if no_adults_eligible_c==1	
drop low_bmi
lab var hh_no_low_bmiage_u_c "DST-HOT: HH has no adult with low BMI or BMI-for-age(<17/-3sd)"

tab hh_no_low_bmiage_u, miss 
tab hh_no_low_bmiage_u_c, miss 


gen	hh_nutrition_uw_st_c = 1
replace hh_nutrition_uw_st_c = 0 if hh_no_low_bmiage_c==0 | hh_no_uw_st==0
replace hh_nutrition_uw_st_c = . if hh_no_low_bmiage_c==. & hh_no_uw_st==.
replace hh_nutrition_uw_st_c = . if hh_no_low_bmiage_c==. & hh_no_uw_st==1 & no_child_eligible==1
replace hh_nutrition_uw_st_c = . if hh_no_uw_st==. & hh_no_low_bmiage_c==1 & no_adults_eligible_c==1
replace hh_nutrition_uw_st_c = 1 if no_eligibles_c==1  	
lab var hh_nutrition_uw_st_c "HOT:HH has no individuals malnourished"

tab hh_nutrition_uw_st, miss 
tab hh_nutrition_uw_st_c, miss 

gen	hh_nutrition_uw_st_u_c = 1
replace hh_nutrition_uw_st_u_c = 0 if hh_no_low_bmiage_u_c==0 | hh_no_uw_st_u==0
replace hh_nutrition_uw_st_u_c = . if hh_no_low_bmiage_u_c==. & hh_no_uw_st_u==.
replace hh_nutrition_uw_st_u_c = . if hh_no_low_bmiage_u_c==. & hh_no_uw_st_u==1 & no_child_eligible==1
replace hh_nutrition_uw_st_u_c = . if hh_no_uw_st_u==. & hh_no_low_bmiage_u_c==1 & no_adults_eligible_c==1
replace hh_nutrition_uw_st_u_c = 1 if no_eligibles_c==1   		 	 	
lab var hh_nutrition_uw_st_u_c "DST-HOT:HH has no individuals malnourished"

tab hh_nutrition_uw_st_u, miss 
tab hh_nutrition_uw_st_u_c, miss 


********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************
codebook ms02_0207_1 ms02_0207_2 vs02_0207_1 vs02_0207_2
	/*ms02_0207_1 vs02_0207_1: number of sons who have died 
	  ms02_0207_2 vs02_0207_2: number of daughters who have died */
	  	
egen temp_f = rowtotal(ms02_0207_1 ms02_0207_2), missing
replace temp_f = 0 if ms02_0206_1==2
bysort	hh_id: egen child_mortality_f = sum(temp_f), missing
lab var child_mortality_f "Occurrence of child mortality reported by women"
tab child_mortality_f, miss


egen temp_m = rowtotal(vs02_0207_1 vs02_0207_2), missing
replace temp_m = 0 if vs02_0206==2	
bysort	hh_id: egen child_mortality_m = sum(temp_m), missing
lab var child_mortality_m "Occurrence of child mortality reported by men"
tab child_mortality_m, miss
drop temp_m


egen child_mortality = rowmax(child_mortality_f child_mortality_m)
lab var child_mortality "Total child mortality within household"
tab child_mortality, miss	


		
*** Standard MPI *** 
/* Members of the household are considered deprived if women in the household 
reported mortality among children under 18 in the last 5 years from the survey 
year.*/
************************************************************************
tab childu18_died_per_wom_5y, miss
gen childu18_died_per_wom_5y_c = childu18_died_per_wom_5y

replace childu18_died_per_wom_5y = 0 if ms02_0206_1==2
	/*Assign a value of "0" for:
	- all eligible women who never ever gave birth */
replace childu18_died_per_wom_5y = 0 if no_fem_eligible==1 
	/*Assign a value of "0" for:
	- individuals living in households that have non-eligible women */	
	
bysort hh_id: egen childu18_mortality_5y = sum(childu18_died_per_wom_5y), missing
replace childu18_mortality_5y = 0 if childu18_mortality_5y==. & child_mortality==0
	/*Replace all households as 0 death if women has missing value and men 
	reported no death in those households */
label var childu18_mortality_5y "Under 18 child mortality within household past 5 years reported by women"
tab childu18_mortality_5y, miss		
	
gen hh_mortality_u18_5y = (childu18_mortality_5y==0)
replace hh_mortality_u18_5y = . if childu18_mortality_5y==.
lab var hh_mortality_u18_5y "Household had no under 18 child mortality in the last 5 years"
tab hh_mortality_u18_5y, miss 


*** Destitution MPI *** 
*** (same as standard MPI) ***
************************************************************************
gen hh_mortality_u = hh_mortality_u18_5y	
lab var hh_mortality_u "Household had no under 18 child mortality in the last 5 years"	



*** Harmonised MPI *** 
	/*In this survey, data on child mortality was collected from 
	women 14-49. For harmonization purposes, we use the data from 
	women 15-49. We only count women between the age of 15-49 
	(rather than 14-49) to accord with the earlier survey.*/
************************************************************************

//Revised the eligiblity

count if hs03_0003_1>=15 & hs03_0003_1<=49 & hs03_0005==2 

gen	fem_eligible_c = (hs03_0003_1>=15 & hs03_0003_1<=49 & hs03_0005==2)
tab fem_eligible_c, miss 
	//Eligible women 15-49 for child mortality: 12,063
bysort	hh_id: egen hh_n_fem_eligible_c = sum(fem_eligible_c) 	
gen	no_fem_eligible_c = (hh_n_fem_eligible_c==0) 									
lab var no_fem_eligible_c "HOT: HH has no eligible women for interview"
drop hh_n_fem_eligible_c 

tab no_fem_eligible, miss 
tab no_fem_eligible_c, miss 


replace temp_f = 0 if hs03_0003_1<=14 & hs03_0005==2	
bysort	hh_id: egen child_mortality_f_c = sum(temp_f), missing
lab var child_mortality_f_c "HOT:Occurrence of child mortality reported by women"
tab child_mortality_f_c, miss
drop temp_f

		
egen child_mortality_c = rowmax(child_mortality_f_c)
lab var child_mortality_c "HOT:Total child mortality within household"

tab child_mortality, miss	
tab child_mortality_c, miss


replace childu18_died_per_wom_5y_c = 0 if ms02_0206_1==2
replace childu18_died_per_wom_5y_c = 0 if no_fem_eligible_c==1 
	
bysort hh_id: egen childu18_mortality_5y_c = sum(childu18_died_per_wom_5y_c), missing
replace childu18_mortality_5y_c = 0 if childu18_mortality_5y_c==. & child_mortality_c==0
label var childu18_mortality_5y_c "HOT: Under 18 child mortality within HH past 5 years reported by women"
tab childu18_mortality_5y_c, miss		
	
gen hh_mortality_u18_5y_c = (childu18_mortality_5y_c==0)
replace hh_mortality_u18_5y_c = . if childu18_mortality_5y_c==.
lab var hh_mortality_u18_5y_c "HOT: HH had no under 18 child mortality in the last 5 years"

tab hh_mortality_u18_5y, miss 
tab hh_mortality_u18_5y_c, miss 


gen hh_mortality_u_c = hh_mortality_u18_5y_c	
lab var hh_mortality_u_c "HOT:HH had no under 18 child mortality in the last 5 years"



********************************************************************************
*** Step 2.5 Electricity ***
********************************************************************************

*** Standard MPI ***
/*Members of the household are considered 
deprived if the household has no electricity */
***************************************************
codebook dv_0053, tab (9)
clonevar electricity = dv_0053 
recode electricity (2=0)
label define lab_elec 1 "have electricity" 0 "no electricity"
label values electricity lab_elec		
label var electricity "Household has electricity"
tab electricity, miss


*** Destitution MPI  ***
*** (same as standard MPI) ***
***************************************************
gen electricity_u = electricity
label var electricity_u "Household has electricity"


********************************************************************************
*** Step 2.6 Sanitation ***
********************************************************************************

/*
Improved sanitation facilities include flush or pour flush toilets to sewer 
systems, septic tanks or pit latrines, ventilated improved pit latrines, pit 
latrines with a slab, and composting toilets. These facilities are only 
considered improved if it is private, that is, it is not shared with other 
households.
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-02-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

desc dv_0060 dv_0061 dv_0062
clonevar toilet = dv_0061  
clonevar shared_toilet = dv_0062
codebook shared_toilet, tab(99)  
	//1=no;2=yes;.=missing

	
*** Standard MPI ***
/*Members of the household are considered deprived if the household's 
sanitation facility is not improved (according to the SDG guideline) 
or it is improved but shared with other households*/
********************************************************************
codebook toilet, tab(99)

/*dv_0060 indicates if the dwelling has any sanitation facility 1.yes 2.no
toilet label 
1 To the sewer network?              		NOT DEPRIVED
2 To a septic tank?							NOT DEPRIVED
3 To a pump / suction well?	(Cesspit)		DEPRIVED
4 To the surface (street/creek /river)?	    DEPRIVED
5.Do not know where 						NOT DEPRIVED
*/
						   
gen	toilet_mdg = ((toilet<=5) & shared_toilet!=2) 	
replace toilet_mdg = 0 if dv_0060==2 | toilet==3 | toilet==4
lab var toilet_mdg "Household has improved sanitation"
tab toilet toilet_mdg, miss
tab toilet_mdg, miss


*** Destitution MPI ***
/*Members of the household are considered deprived if household practises 
open defecation or uses other unidentifiable sanitation practises */
********************************************************************
gen	toilet_u = .
replace toilet_u = 0 if dv_0060==2 
replace toilet_u = 1 if dv_0060==1
lab var toilet_u "Household does not practise open defecation or others"
tab toilet_u, miss


********************************************************************************
*** Step 2.7 Drinking Water  ***
********************************************************************************
/*
Improved drinking water sources include the following: piped water into 
dwelling, yard or plot; public taps or standpipes; boreholes or tubewells; 
protected dug wells; protected springs; packaged water; delivered water and 
rainwater which is located on premises or is less than a 30-minute walk from 
home roundtrip. 
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-01-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

desc dv_0055 dv_0055_a dv_0058*
clonevar water = dv_0055  
clonevar timetowater = dv_0055_a  
gen ndwater = .  	


*** Standard MPI ***
/* Members of the household are considered deprived if the household 
does not have access to improved drinking water (according to the SDG 
guideline) or safe drinking water is at least a 30-minute walk from 
home, roundtrip */
********************************************************************
codebook water, tab(99)
/*
1. Piped into dwelling) 								NOT DEPRIVED
2. Piped outside the dwelling but inside the grounds 	NOT DEPRIVED
3. Public tap/standpipe									NOT DEPRIVED
4. Boreholes or tubewells		 						NOT DEPRIVED
5. Protected dug wells with pump 						NOT DEPRIVED
6. Protected dug wells without pump 					NOT DEPRIVED
7. Unprotected well										DEPRIVED
8. Protected springs 									NOT DEPRIVED
9. Unprotected springs 									DEPRIVED
10.Bottled water /packed water? 						NOT DEPRIVED
11.Tanker truck 										NOT DEPRIVED
12.Other 												DEPRIVED
*/
		
gen	water_mdg = 1 if water==1 | water==2  | water==3 | /// 
					 water==4 | water==5  | water==6 | ///
					 water==8 | water==10 | water==11
					 
replace water_mdg = 0 if water==7 | water==9 | water==12 

codebook timetowater, tab(999)	
replace water_mdg = 0 if water_mdg==1 & timetowater >= 30 & timetowater!=. & ///
						 timetowater!=996 & timetowater!=999 
replace water_mdg = . if water==. 
lab var water_mdg "Household has drinking water with MDG standards (considering distance)"
tab water water_mdg, miss
tab water_mdg, miss


*** Destitution MPI ***
/* Members of the household is identified as destitute if household 
does not have access to safe drinking water, or safe water is more 
than 45 minute walk from home, round trip.*/
********************************************************************
gen	water_u = .
replace water_u = 1 if water==1 | water==2  | water==3 | /// 
					   water==4 | water==5  | water==6 | ///
					   water==8 | water==10 | water==11
						   
replace water_u = 0 if water==7 | water==9 | water==12 
						   
replace water_u = 0 if water_u==1 & timetowater>45 & timetowater!=. & ///
					   timetowater!=996 &  timetowater!=999 					  					   
replace water_u = . if water==.
lab var water_u "Household has drinking water with MDG standards (45 minutes distance)"
tab water water_u, miss
tab water_u, miss


********************************************************************************
*** Step 2.8 Housing ***
********************************************************************************

/* Members of the household are considered deprived 
if the household has a dirt, sand or dung floor*/
clonevar floor = dv_0052  
codebook floor, tab(99)
/*
1   Earth/sand 						DEPRIVED
2   Wood 							NOT DEPRIVED
3   Parquet or polished wood 		NOT DEPRIVED
4   Carpet 							NOT DEPRIVED
5   Cement 							NOT DEPRIVED
6   Ceramic tiles 					NOT DEPRIVED
7   Bricks 							NOT DEPRIVED
96  Other 							DEPRIVED
*/						 				
gen	floor_imp = 1
replace floor_imp = 0 if floor<=1 | floor==96  	
replace floor_imp = . if floor==. 
lab var floor_imp "Household has floor that it is not earth/sand/dung"
tab floor floor_imp, miss		

codebook dv_0052_1cod if floor==96
clonevar floor_other = dv_0052_1cod
destring floor_other, replace force
codebook floor_other, tab (999)


/*Members of the household are considered deprived if the 
household has walls made of natural or rudimentary materials.*/
lookfor pared
clonevar wall = dv_0049_1 
codebook wall, tab(99)
/* 
1. BRICK/CEMENT /CONCRETE BLOCK					 	Improved
2. Adobe/bricks with mud 						 	Improved
3. Cane or bark of wood that is filled with mud  	Non-improved
4. Stone 											Non-improved
5. Wood 											Non-improved
6. CANE / PALM / TRUNK								Non-improved
96.Other											Non-improved
*/
gen	wall_imp = 0
replace wall_imp = 1 if wall==1 | wall==2 
replace wall_imp = . if wall==. 
lab var wall_imp "Household has wall that it is not of low quality materials"
tab wall wall_imp, miss	

	
codebook dv_0049_2cod if wall==96
clonevar wall_other = dv_0049_2cod
destring wall_other, replace force
codebook wall_other, tab (999)	
	
	
	
/* Members of the household are considered deprived if the 
household has roof made of natural or rudimentary materials. */
lookfor techo
clonevar roof = dv_0051
codebook roof, tab(99)
/*
1.  Calamine  							Improved
2.  TILES 								Improved
3.  REINFORCED CONCRETE SLAB 			Improved
4.  STRAW / CANE / PALM / MUD)			Non-improved
96. OTRO (other) 						Non-improved
*/
	
gen	roof_imp = 1  
replace roof_imp = 0 if roof==4 | roof==96   
replace roof_imp = . if roof==. 
lab var roof_imp "Household has roof that it is not of low quality materials"
tab roof roof_imp, miss


codebook dv_0051_1cod if roof==96
clonevar roof_other = dv_0051_1cod
destring roof_other, replace force
codebook roof_other, tab (999)	



*** Standard MPI ***
/* Members of the household is deprived in housing if the roof, 
floor OR walls are constructed from low quality materials.*/
**************************************************************
gen housing_1 = 1
replace housing_1 = 0 if floor_imp==0 | wall_imp==0 | roof_imp==0
replace housing_1 = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_1 "Household has roof, floor & walls that it is not low quality material"
tab housing_1, miss 


*** Destitution MPI ***
/* Members of the household is deprived in housing if two out 
of three components (roof and walls; OR floor and walls; OR 
roof and floor) the are constructed from low quality materials. */
**************************************************************
gen housing_u = 1
replace housing_u = 0 if (floor_imp==0 & wall_imp==0 & roof_imp==1) | ///
						 (floor_imp==0 & wall_imp==1 & roof_imp==0) | ///
						 (floor_imp==1 & wall_imp==0 & roof_imp==0) | ///
						 (floor_imp==0 & wall_imp==0 & roof_imp==0)
replace housing_u = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_u "Household has one of three aspects(either roof,floor/walls) that is not low quality material"
tab housing_u, miss
	

********************************************************************************
*** Step 2.9 Cooking Fuel ***
********************************************************************************
/*
Solid fuel are solid materials burned as fuels, which includes coal as well as 
solid biomass fuels (wood, animal dung, crop wastes and charcoal). 

Source: 
https://apps.who.int/iris/bitstream/handle/10665/141496/9789241548885_eng.pdf
*/

lookfor combustible
clonevar cookingfuel = dv_0063



*** Standard MPI ***
/* Members of the household are considered deprived if the 
household uses solid fuels and solid biomass fuels for cooking. */
*****************************************************************
codebook cookingfuel, tab(99)
/* 
1. Wood 					DEPRIVED
2. Animal waste 			DEPRIVED
3. LPG 						NOT DEPRIVED
4. Natural gas 				NOT DEPRIVED
5. Electricity 				NOT DEPRIVED
6. Don't cook 				NOT DEPRIVED
7. Other 					NOT DEPRIVED 
*/

gen	cooking_mdg = 1
replace cooking_mdg = 0 if cookingfuel==1 | cookingfuel==2
replace cooking_mdg = . if cookingfuel==. 
lab var cooking_mdg "Household uses clean fuels for cooking"			 
tab cookingfuel cooking_mdg, miss	
tab cooking_mdg, miss	


*** Destitution MPI ***
*** (same as standard MPI) ***
****************************************
gen	cooking_u = cooking_mdg
lab var cooking_u "Household uses clean fuels for cooking"


********************************************************************************
*** Step 2.10 Assets ownership ***
********************************************************************************


*** Television/LCD TV/plasma TV/color TV/black & white tv
lookfor televisor	
codebook dv_0068_b 
	//1=yes; 2=no
clonevar television = dv_0068_b 
lab var television "Household has television"	

	
***	Radio/walkman/stereo/kindle
lookfor radio walkman stereo
codebook dv_0068_a
	//1=yes; 2=no
clonevar radio = dv_0068_a 
lab var radio "Household has radio"	

	
***	Handphone/telephone/iphone/mobilephone/ipod
lookfor tel celular ipod
desc dv_0068_j dv_0068_k 
codebook dv_0068_j dv_0068_k 
	//1=yes; 2=no
clonevar telephone = dv_0068_j
replace telephone=1 if telephone!=1 & dv_0068_k==1	
	//dv_0068_k=mobilephone. Combine information on telephone and mobilephone.	
tab dv_0068_a dv_0068_k if telephone==1,miss
lab var telephone "Household has telephone (landline/mobilephone)"	

	
***	Refrigerator/icebox/fridge
lookfor refri
codebook  dv_0068_d
	//1=yes; 2=no
clonevar refrigerator = dv_0068_d 
lab var refrigerator "Household has refrigerator"


***	Car/van/lorry/truck
lookfor carro camion auto
codebook dv_0068_p 
	//1=yes; 2=no
clonevar car = dv_0068_p   
lab var car "Household has car"		

		
***	Bicycle/cycle rickshaw
lookfor bicicleta
codebook dv_0068_n 
	//1=yes; 2=no	
clonevar bicycle = dv_0068_n  
lab var bicycle "Household has bicycle"	


***	Motorbike/motorized bike/autorickshaw
lookfor motocicleta
codebook dv_0068_o
	//1=yes; 2=no	
clonevar motorbike = dv_0068_o 
lab var motorbike "Household has motorbike"	

	
***	Computer/laptop/tablet
lookfor computadora
codebook dv_0068_f 
	//1=yes; 2=no
clonevar computer = dv_0068_f
lab var computer "Household has computer"
	
	
***	Animal cart
lookfor carreta
gen animal_cart = .
lab var animal_cart "Household has animal cart"	


foreach var in television radio telephone refrigerator car ///
			   bicycle motorbike computer animal_cart {
			   replace `var' = 0 if `var'==2 
			   label define lab_`var' 0"No" 1"Yes"
			   label values `var' lab_`var'	
}
	//Missing values replaced



*** Standard MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own more than one of: radio, TV, telephone, bike, motorbike, 
refrigerator, computer or animal cart and does not own a car or truck.*/
*****************************************************************************
egen n_small_assets2 = rowtotal(television radio telephone refrigerator bicycle motorbike computer animal_cart), missing
lab var n_small_assets2 "Household Number of Small Assets Owned" 
   
gen hh_assets2 = (car==1 | n_small_assets2 > 1) 
replace hh_assets2 = . if car==. & n_small_assets2==.
lab var hh_assets2 "Household Asset Ownership: HH has car or more than 1 small assets incl computer & animal cart"
tab hh_assets2, miss


*** Destitution MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own any assets.*/
*****************************************************************************	
gen	hh_assets2_u = (car==1 | n_small_assets2>0)
replace hh_assets2_u = . if car==. & n_small_assets2==.
lab var hh_assets2_u "Household Asset Ownership: HH has car or at least 1 small assets incl computer & animal cart"

		
********************************************************************************
*** Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************

	//Retain data on sampling design: 
desc estrato upm	
egen psu = group(upm)
egen strata = group(estrato)
label var psu "Primary sampling unit"
label var strata "Sample strata"
desc psu strata
	
	//Retain year, month & date of interview:
desc dv_00484c dv_00484b dv_00484a 

gen year_interview = dv_00484c	
gen month_interview = dv_00484b
clonevar date_interview = dv_00484a


		
*** Keep main variables require for MPI calculation ***
keep hh_id ind_id subsample strata psu weight weight_ch sex age hhsize ///
area agec7 agec4 agec2 headship region marital relationship ///
region region_01 year_interview month_interview date_interview /// 
fem_nutri_eligible no_fem_nutri_eligible no_fem_eligible ///
male_nutri_eligible no_male_nutri_eligible no_male_eligible ///
child_eligible no_child_eligible no_adults_eligible ///
no_child_fem_eligible no_eligibles ///
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
hh_no_low_bmiage hh_no_low_bmiage_u ///
low_bmiage low_bmiage_u low_bmi_byage f_bmi m_bmi ///
hh_no_low_bmiage hh_no_low_bmiage_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet shared_toilet toilet_mdg toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
hh_mortality_u18_5y_c hh_mortality_u_c ///
hh_nutrition_uw_st_c hh_nutrition_uw_st_u_c
	
	 
*** Order file	***
order hh_id ind_id subsample strata psu weight weight_ch sex age hhsize ///
area agec7 agec4 agec2 headship region marital relationship ///
region region_01 year_interview month_interview date_interview /// 
fem_nutri_eligible no_fem_nutri_eligible no_fem_eligible ///
male_nutri_eligible no_male_nutri_eligible no_male_eligible ///
child_eligible no_child_eligible no_adults_eligible ///
no_child_fem_eligible no_eligibles ///
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
hh_no_low_bmiage hh_no_low_bmiage_u ///
low_bmiage low_bmiage_u low_bmi_byage f_bmi m_bmi ///
hh_no_low_bmiage hh_no_low_bmiage_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet shared_toilet toilet_mdg toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u ///
hh_mortality_u18_5y_c hh_mortality_u_c ///
hh_nutrition_uw_st_c hh_nutrition_uw_st_u_c


*** Rename key global MPI indicators for estimation ***
recode hh_mortality_u18_5y  (0=1)(1=0) , gen(d_cm)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ)
recode electricity 			(0=1)(1=0) , gen(d_elct)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani)
recode housing_1 			(0=1)(1=0) , gen(d_hsg)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl)
recode hh_assets2 			(0=1)(1=0) , gen(d_asst)
 

*** Rename key global MPI indicators for destitution estimation ***
recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ)
recode electricity_u		(0=1)(1=0) , gen(dst_elct)
recode water_u 				(0=1)(1=0) , gen(dst_wtr)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl)
recode hh_assets2_u 		(0=1)(1=0) , gen(dst_asst) 


*** Rename indicators for changes over time estimation ***	
recode hh_mortality_u18_5y_c  (0=1)(1=0) , gen(d_cm_01)
recode hh_nutrition_uw_st_c   (0=1)(1=0) , gen(d_nutr_01)
recode hh_child_atten 		  (0=1)(1=0) , gen(d_satt_01)
recode hh_years_edu6 		  (0=1)(1=0) , gen(d_educ_01)
recode electricity 			  (0=1)(1=0) , gen(d_elct_01)
recode water_mdg 			  (0=1)(1=0) , gen(d_wtr_01)
recode toilet_mdg 			  (0=1)(1=0) , gen(d_sani_01)
recode housing_1 			  (0=1)(1=0) , gen(d_hsg_01)
recode cooking_mdg 			  (0=1)(1=0) , gen(d_ckfl_01)
recode hh_assets2 			  (0=1)(1=0) , gen(d_asst_01)	
	

recode hh_mortality_u_c       (0=1)(1=0) , gen(dst_cm_01)
recode hh_nutrition_uw_st_u_c (0=1)(1=0) , gen(dst_nutr_01)
recode hh_child_atten_u 	  (0=1)(1=0) , gen(dst_satt_01)
recode hh_years_edu_u 		  (0=1)(1=0) , gen(dst_educ_01)
recode electricity_u		  (0=1)(1=0) , gen(dst_elct_01)
recode water_u	 			  (0=1)(1=0) , gen(dst_wtr_01)
recode toilet_u 			  (0=1)(1=0) , gen(dst_sani_01)
recode housing_u 			  (0=1)(1=0) , gen(dst_hsg_01)
recode cooking_u			  (0=1)(1=0) , gen(dst_ckfl_01)
recode hh_assets2_u 		  (0=1)(1=0) , gen(dst_asst_01)


*** Eligibility for years of schooling indicator ***
gen educ_elig = 1 
replace educ_elig = 0 if age < 10 
replace educ_elig = 0 if (age==10 | age==11) & eduyears < 6 
label define lab_educ_elig 0"ineligible" 1"eligible"  
label values educ_elig lab_educ_elig
lab var educ_elig "Individual is eligible for educ indicator"
ta eduyears educ_elig,m

			
*** Generate coutry and survey details for estimation ***
char _dta[cty] "Bolivia"
char _dta[ccty] "BOL"
char _dta[year] "2016" 	
char _dta[survey] "EDSA"
char _dta[ccnum] "68"
char _dta[type] "micro"

	
*** Sort, compress and save data for estimation ***
sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."
save "$path_out/bol_edsa16.dta", replace 


erase "$path_out/BOL16_BR.dta"
erase "$path_out/BOL16_IR.dta"
erase "$path_out/BOL16_PR_girls.dta"
erase "$path_out/BOL16_PR_girls_c.dta"	
erase "$path_out/BOL16_MN.dta" 
erase "$path_out/BOL16_PR_child.dta"
erase "$path_out/BOL16_CH_6Y.dta"
erase "$path_out/BOL16_PR_1.dta"	
erase "$path_out/BOL16_CHWM.dta"
erase "$path_out/BOL16_PR.dta"