********************************************************************************
/*
Suggested citation:
Oxford Poverty and Human Development Initiative (OPHI). 
Global Multidimensional Poverty Index - Albania DHS 2017-2018 
[STATA do-file]. OPHI, University of Oxford. 
*/
********************************************************************************

clear all 
set more off
set maxvar 10000


global path_in "../rdta/Albania DHS 2017-18"	//working folder path  
global path_out "cdta"
global path_ado "ado"
	

********************************************************************************
*** Step 1: Data preparation 
********************************************************************************

********************************************************************************
*** Step 1.1 PR - INDIVIDUAL RECODE (child under 5 years) 
********************************************************************************
use "$path_in/ALPR71FL.DTA", clear 

gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g
lab var ind_id "individual id"
codebook ind_id

duplicates report ind_id

keep if hv120==1
count							//2,907 children under 5		
	

desc hc27 hv104					//sex
compare hc27 hv104				//quality check:hc27 match hv104
tab hc27, m						//quality check: 1=male; 2=female 
tab hc27, nol

clonevar gender = hc27
tab gender,m					


codebook hc1						//age				
clonevar age_months = hc1  
sum age_months

gen  str6 ageunit = "months" 		//age unit
lab var ageunit "months"

gen mdate = mdy(hc18, hc17, hc19)

gen bdate = mdy(hc30, hc16, hc31) if hc16 <= 31	
replace bdate = mdy(hc30, 15, hc31) if hc16 > 31	
	
gen age = (mdate-bdate)/30.4375 			//age in months, days in decimals

	
codebook hc2 								//body weight (kg)
gen	weight = hc2/10 
tab hc2 if hc2>990, m nol  
replace weight = . if hc2>=990				//missing values/out of range 
tab	hc13 hc2 if hc2>=990 | hc2==., m 
sum weight



codebook hc3								//height (cm)
gen	height = hc3/10 
tab hc3 if hc3>9990, m nol   
replace height = . if hc3>=9990				//missing values/out of range 
tab	hc13 hc3   if hc3>=9990 | hc3==., m
sum height


codebook hc15												//measurement
gen measure = "l" if hc15==1								//lying down
replace measure = "h" if hc15==2 							//standing up
replace measure = " " if hc15==9 | hc15==0 | hc15==.		//unknown
tab measure


lookfor oedema								//oedema
gen  oedema = "n"  							//assumes no-one has oedema
	

gen  sw = 1									//sample weight

gen ch_sw = hv005/1000000
lab var ch_sw "sample weight - child under 5" 


adopath + "$path_ado/igrowup_stata" 				//z-score
gen str100 reflib = "$path_ado/igrowup_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "children_nutri_alb" 
lab var datalab "Working file"

igrowup_restricted reflib datalib datalab gender ///
age ageunit weight height measure oedema sw

use "$path_out/children_nutri_alb_z_rc.dta", clear 

	
gen	underweight = (_zwei < -2.0)					//mpi cutoff
replace underweight = . if _zwei == . | _fwei==1
lab var underweight  "child is underweight"

gen stunting = (_zlen < -2.0)
replace stunting = . if _zlen == . | _flen==1
lab var stunting "child is stunted"

gen wasting = (_zwfl < - 2.0)
replace wasting = . if _zwfl == . | _fwfl == 1
lab var wasting  "child is wasted"


gen	underweight_u = (_zwei < -3.0) 					//dst cutoff
replace underweight_u = . if _zwei == . | _fwei==1
lab var underweight_u  "dst: child is underweight"

gen stunting_u = (_zlen < -3.0)
replace stunting_u = . if _zlen == . | _flen==1
lab var stunting_u "dst: child is stunted"

gen wasting_u = (_zwfl < - 3.0)
replace wasting_u = . if _zwfl == . | _fwfl == 1
lab var wasting_u  "dst: child is wasted"
 
 
count if _fwei==1 | _flen==1	//27 obs, z-scores biologically implausible
 
keep ind_id ch_sw underweight* stunting* wasting* 
order ind_id ch_sw underweight* stunting* wasting* 
sort ind_id
duplicates report ind_id
save "$path_out/ALB17-18_PR_child.dta", replace

	
erase "$path_out/children_nutri_alb_z_rc.xls"
erase "$path_out/children_nutri_alb_prev_rc.xls"
erase "$path_out/children_nutri_alb_z_rc.dta"

	
********************************************************************************
*** Step 1.2  BR - BIRTH RECODE (fem 15-49 years who ever gave birth)  
********************************************************************************
use "$path_in/ALBR71FL.dta", clear
	
gen double ind_id = v001*1000000 + v002*100 + v003 
format ind_id %20.0g
lab var ind_id "individual id"

desc b3 b7	
gen date_death = b3 + b7		//date of death 

gen mdead_survey = v008 - date_death
		
gen ydead_survey = mdead_survey/12
	
gen age_death = b7	
label var age_death "age at death in months"
tab age_death, m
	
		
codebook b5, tab (9)	
gen child_died = 1 if b5==0
replace child_died = 0 if b5==1
replace child_died = . if b5==.
lab define lab_died 1 "child has died" 0 "child is alive"
lab values child_died lab_died
tab b5 child_died, m
	
bysort ind_id: egen tot_child_died = sum(child_died) 
egen tot_child_died_2 = rsum(v206 v207)		//quality check with v206 & v207
compare tot_child_died tot_child_died_2		//figures are identical
			
gen child18_died = child_died 
replace child18_died=0 if age_death>=216 & age_death<.
label values child18_died lab_died
tab child18_died, m				//child under 18 mortality,ast 5 years
			
bysort ind_id: egen tot_child18_died_5y=sum(child18_died) if ydead_survey<=5
	
replace tot_child18_died_5y=0 if tot_child18_died_5y==. & ///
								 tot_child_died>=0 & tot_child_died<.
								 
replace tot_child18_died_5y=. if child18_died==1 & ydead_survey==.

tab tot_child_died tot_child18_died_5y, m

bysort ind_id: egen childu18_died_per_wom_5y = max(tot_child18_died_5y)
lab var childu18_died_per_wom_5y "total child mortality of each women"
	

bysort ind_id: gen id=1 if _n==1		//keep one observation per women
keep if id==1
drop id
duplicates report ind_id 

gen women_BR = 1 
	
keep ind_id women_BR childu18_died_per_wom_5y 
order ind_id women_BR childu18_died_per_wom_5y
sort ind_id
save "$path_out/ALB17-18_BR.dta", replace	
	
	
********************************************************************************
*** Step 1.3  IR - WOMEN's RECODE (elig fem 15-49 years)
********************************************************************************
use "$path_in/ALIR71FL.dta", clear
		
gen double ind_id = v001*1000000 + v002*100 + v003 
format ind_id %20.0g
lab var ind_id "individual id"

duplicates report ind_id


codebook v012 v201 v206 v207,tab (999)  //quality check

tab v012, m						//women 15-59 years
tab v012 if v201==., m			//50-59 years interviewed, no fertility data  
keep if v012>=15 & v012<=49		//keep only 15-49 years
count							//10,860 women 15-49 years

gen women_IR=1 
		
keep ind_id women_IR v003 v005 v012 v201 v206 v207 
order ind_id women_IR v003 v005 v012 v201 v206 v207 
sort ind_id
save "$path_out/ALB17-18_IR.dta", replace


********************************************************************************
*** Step 1.4  PR - INDIVIDUAL RECODE (fem 15-19 years)
********************************************************************************
use "$path_in/ALPR71FL.dta", clear
		
gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g
lab var ind_id "individual id"


tab ha13 if hv105>=15 & hv105<=19 & hv104==2, m		
tab ha13 hv117 if hv105>=15 & hv105<=19 & hv104==2, m


keep if hv105>=15 & hv105<=19 & hv104==2 
count			//female 15-19 years: 1,868   

	
codebook hv104, tab (9)		//sex
clonevar gender = hv104		//quality check: only female 


desc hv807c ha32			 //age
gen age_month = hv807c - ha32
lab var age_month "age in months, 15-19 years (fem)"
tab age_month, m			//>228 months, excluded from z-score but have BMI 

	
gen str6 ageunit = "months"			//age unit 
lab var ageunit "months"

			
codebook ha2, tab (9999)			//body weight (kg)
count if ha2>9990 
tab ha13 if ha2>9990, m
gen weight = ha2/10 if ha2<9990	
sum weight


codebook ha3, tab (9999)			// height (cm)
count if ha3>9990 
tab ha13 if ha3>9990, m
gen height = ha3/10 if ha3<9990
sum height


gen oedema = "n" 					//oedema 


gen sw = 1 							//sample weight

					
adopath + "$path_ado/who2007_stata"			//bmi-for-age	
gen str100 reflib = "$path_ado/who2007_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "girl_nutri_alb" 
lab var datalab "Working file"
	
who2007 reflib datalib datalab gender ///
age_month ageunit weight height oedema sw

use "$path_out/girl_nutri_alb_z.dta", clear 
	
gen	z_bmi = _zbfa
replace z_bmi = . if _fbfa==1 
lab var z_bmi "z-score bmi-for-age"


gen	low_bmiage = (z_bmi < -2.0)			//mpi cutoff
replace low_bmiage = . if z_bmi==.
lab var low_bmiage "low bmi-for-age (fem)"

	
gen	low_bmiage_u = (z_bmi < -3.0)		//dst cutoff
replace low_bmiage_u = . if z_bmi==.
lab var low_bmiage_u "dst: low bmi-for-age (fem)"

tab low_bmiage, m
tab low_bmiage_u, m

gen girl_PR=1 

keep ind_id girl_PR age_month low_bmiage*
order ind_id girl_PR age_month low_bmiage*
sort ind_id
save "$path_out/ALB17-18_PR_girls.dta", replace


erase "$path_out/girl_nutri_alb_z.xls"
erase "$path_out/girl_nutri_alb_prev.xls"
erase "$path_out/girl_nutri_alb_z.dta"


********************************************************************************
*** Step 1.5  MR - MEN'S RECODE  
********************************************************************************
use "$path_in/ALMR71FL.dta", clear 
	
gen double ind_id = mv001*1000000 + mv002*100 + mv003 	
format ind_id %20.0g
lab var ind_id "individual id"

duplicates report ind_id

codebook mv012 mv201 mv206 mv207,tab (999)		//quality check

gen men_MR=1 	
	
keep ind_id men_MR mv003 mv005 mv012 mv201 mv206 mv207  
order ind_id men_MR mv003 mv005 mv012 mv201 mv206 mv207   
sort ind_id
save "$path_out/ALB17-18_MR.dta", replace


********************************************************************************
*** Step 1.6  PR - INDIVIDUAL RECODE (male 15-19 years)
********************************************************************************
use "$path_in/ALPR71FL.dta", clear 
	
gen double ind_id = hv001*1000000 + hv002*100 + hvidx 
format ind_id %20.0g
lab var ind_id "individual id"


tab hb13 if hv105>=15 & hv105<=19 & hv104==1, m 
tab hb13 if hv105>=15 & hv105<=19 & hv104==1 & hv027==1, m 
tab hb13 hv118 if hv105>=15 & hv105<=19 & hv104==1 & hv027==1, m

	
keep if hv105>=15 & hv105<=19 & hv104==1 & hv027==1 
count						//male 15-19 years: 872


codebook hv104, tab (9)			//sex
clonevar gender = hv104			//quality check: only male
 

desc hv807c hb32				//age
gen age_month_b = hv807c - hb32
lab var age_month_b "age in months, 15-19 years (male)"
tab age_month_b, m 			//>228 months, excluded from z-score but have BMI 

	
gen str6 ageunit = "months" 			//age unit
lab var ageunit "months"

			
codebook hb2, tab (9999)				//body weight (kg)
count if hb2>9990 
tab hb13 if hb2>9990, m
gen weight = hb2/10 if hb2<9990
sum weight


codebook hb3, tab (9999)				//height (cm)
count if hb3>9990 
tab hb13 if hb3>9990, m
gen height = hb3/10 if hb3<9990
sum height


gen oedema = "n"  						//oedema



gen sw = 1								//sample weight


adopath + "$path_ado/who2007_stata"					//bmi-for-age	
gen str100 reflib = "$path_ado/who2007_stata"
lab var reflib "Directory of reference tables"
gen str100 datalib = "$path_out" 
lab var datalib "Directory for datafiles"
gen str30 datalab = "boy_nutri_alb" 
lab var datalab "Working file"

	
who2007 reflib datalib datalab gender ///
age_month_b ageunit weight height oedema sw

use "$path_out/boy_nutri_alb_z.dta", clear 

	
gen	z_bmi = _zbfa
replace z_bmi = . if _fbfa==1 
lab var z_bmi "z-score bmi-for-age"

	
gen	low_bmiage_b = (z_bmi < -2.0) 					//mpi cutoff
replace low_bmiage_b = . if z_bmi==.
lab var low_bmiage_b "low bmi-for-age (male)"


gen	low_bmiage_b_u = (z_bmi < -3.0)					//dst cutoff
replace low_bmiage_b_u = . if z_bmi==.
lab var low_bmiage_b_u "dst: low bmi-for-age (male)"

tab low_bmiage_b, m
tab low_bmiage_b_u, m

gen boy_PR=1 

keep ind_id boy_PR age_month_b low_bmiage*
order ind_id boy_PR age_month_b low_bmiage*
sort ind_id
save "$path_out/ALB17-18_PR_boys.dta", replace


erase "$path_out/boy_nutri_alb_z.xls"
erase "$path_out/boy_nutri_alb_prev.xls"
erase "$path_out/boy_nutri_alb_z.dta"

	
********************************************************************************
*** Step 1.7  PR - HOUSEHOLD MEMBER'S RECODE 
********************************************************************************
use "$path_in/ALPR71FL.dta", clear

	
gen double hh_id = hv001*10000 + hv002 
format hh_id %20.0g
lab var hh_id "household id"


gen double ind_id = hv001*1000000 + hv002*100 + hvidx  
format ind_id %20.0g
lab var ind_id "individual id"


sort hh_id ind_id

	
********************************************************************************
*** Step 1.8 DATA MERGING 
******************************************************************************** 
 
 
* BR recode 
merge 1:1 ind_id using "$path_out/ALB17-18_BR.dta"
drop _merge
erase "$path_out/ALB17-18_BR.dta"


* IR recode 
merge 1:1 ind_id using "$path_out/ALB17-18_IR.dta"
tab women_IR hv117, m col
tab ha65 if hv117==1 & women_IR ==., m		//elig fem not interviewed
drop _merge
erase "$path_out/ALB17-18_IR.dta"


* 15-19 years: female 
merge 1:1 ind_id using "$path_out/ALB17-18_PR_girls.dta"
tab ha13 girl_PR if hv105>=15 & hv105<=19 & hv104==2, m col
drop _merge
erase "$path_out/ALB17-18_PR_girls.dta"		
	
	
* MR recode 
merge 1:1 ind_id using "$path_out/ALB17-18_MR.dta"
tab men_MR hv118 if hv027==1, m col
tab hb65 if hv118==1 & men_MR ==.			//elig male not interviewed
drop _merge
erase "$path_out/ALB17-18_MR.dta"


* 15-19 years: male 
merge 1:1 ind_id using "$path_out/ALB17-18_PR_boys.dta"
tab boy_PR hv027 if hv105>=15 & hv105<=19 & hv104==1, m col
drop _merge
erase "$path_out/ALB17-18_PR_boys.dta"


* child under 5 
merge 1:1 ind_id using "$path_out/ALB17-18_PR_child.dta"
tab hv120, m  
tab hc13 if hv120==1, m
drop _merge
erase "$path_out/ALB17-18_PR_child.dta"


sort ind_id


********************************************************************************
*** Step 1.9 USUAL HOUSEHOLD MEMBERS ***
********************************************************************************
codebook hv102
drop if hv102!=1 

	
********************************************************************************
*** Step 1.10 CONTROL VARIABLES
********************************************************************************

* no elig wom (nutr indic)
tab ha13, miss
tab ha13 if hv105>=15 & hv105<=59 & hv104==2, m

gen fem_nutri_eligible = (ha13!=.)
tab fem_nutri_eligible, m

bysort hh_id: egen hh_n_fem_nutri_eligible = sum(fem_nutri_eligible) 
	
gen	no_fem_nutri_eligible = (hh_n_fem_nutri_eligible==0)
lab var no_fem_nutri_eligible "hh no elig wom for nutr"		
drop hh_n_fem_nutri_eligible
tab no_fem_nutri_eligible, m


* no elig wom (cm indic)
gen	fem_eligible = (hv117==1)

bysort	hh_id: egen hh_n_fem_eligible = sum(fem_eligible) 
	
gen	no_fem_eligible = (hh_n_fem_eligible==0) 									
lab var no_fem_eligible "hh no elig wom"
drop fem_eligible hh_n_fem_eligible 
tab no_fem_eligible, m


* no elig man (nutr indic, if relevant)
tab hb13, miss
tab hb13 hv027 if hv105>=15 & hv105<=59 & hv104==1, m

gen	male_nutri_eligible = (hb13!=.)
tab male_nutri_eligible,m

bysort hh_id: egen hh_n_male_nutri_eligible = sum(male_nutri_eligible)
  
gen	no_male_nutri_eligible = (hh_n_male_nutri_eligible==0)
drop hh_n_male_nutri_eligible
lab var no_male_nutri_eligible "hh no elig man for nutr"	
tab no_male_nutri_eligible, m


* no elig man (cm indic, if relevant)
gen	male_eligible = (hv118==1)

bysort	hh_id: egen hh_n_male_eligible = sum(male_eligible)  

gen	no_male_eligible = (hh_n_male_eligible==0) 	
lab var no_male_eligible "hh no elig man"
drop male_eligible hh_n_male_eligible
tab no_male_eligible, m


* no elig child under 5 (child nutr indic)
gen	child_eligible = (hv120==1) 

bysort	hh_id: egen hh_n_children_eligible = sum(child_eligible) 
 
gen	no_child_eligible = (hh_n_children_eligible==0) 
lab var no_child_eligible "hh no elig child under 5"
drop hh_n_children_eligible
tab no_child_eligible, m


* no elig wom & man (nutr indic)
gen no_adults_eligible = (no_fem_nutri_eligible==1 & no_male_nutri_eligible==1) 
lab var no_adults_eligible "hh no elig wom or man for nutr"
tab no_adults_eligible, m 


* no elig child & wom (child and wom nutr indic) 
gen	no_child_fem_eligible = (no_child_eligible==1 & no_fem_nutri_eligible==1)
lab var no_child_fem_eligible "hh no elig child or wom for nutr"
tab no_child_fem_eligible, m 


* no elig wom, man or child (nutr indic) 
gen no_eligibles = (no_fem_nutri_eligible==1 & ///
					no_male_nutri_eligible==1 & no_child_eligible==1)
lab var no_eligibles "hh no one elig for nutr"
tab no_eligibles, m


sort hh_id ind_id


********************************************************************************
*** Step 1.11 DEMOGRAPHIC VARIABLES ***
********************************************************************************

codebook hv005									//hh sample weight
clonevar weight = hv005
replace weight = weight/1000000 
lab var weight "hh sample weight"

									
codebook hv025, tab (5)							//urban-rural	
clonevar area = hv025  
replace area=0 if area==2  
lab define lab_area 1 "urban" 0 "rural"
lab values area lab_area
lab var area "urban-rural area"


clonevar rship = hv101							//rship to hh head
codebook rship, tab (99)
recode rship (1=1)(2=2)(3=3)(11=3)(4/10 13/14=4)(15=2)(12=5)(98=.)
lab define lab_rel 1"head" 2"spouse" 3"child" ///
				   4"extended family" 5"not related" 6"maid"
lab values rship lab_rel
lab var rship "relationship to hh head"
tab hv101 rship, m

	
codebook hv104									//sex
clonevar sex = hv104 
lab var sex "sex"


bys	hh_id: egen miss_hhead = min(rship) 		//headship
tab miss_hhead,m 
gen hh_head=.
replace hh_head=1 if rship==1 & sex==1 
replace hh_head=2 if rship==1 & sex==2
bysort hh_id: egen hship = sum(hh_head)
replace hship = 1 if (miss_hhead==2 & sex==1)
replace hship = 2 if (miss_hhead==2 & sex==2)
replace hship = . if miss_hhead>2
lab define lab_hs 1"male-headed" 2"female-headed"
lab values hship lab_hs
lab var hship "gender of hh head"
tab hship, m


codebook hv105, tab (999)						//age & age group
clonevar age = hv105  
replace age = . if age>=98
lab var age "age"

recode age (0/4 = 1 "0-4")(5/9 = 2 "5-9")(10/14 = 3 "10-14") ///
		   (15/17 = 4 "15-17")(18/59 = 5 "18-59")(60/max=6 "60+"), gen(agec7)
lab var agec7 "age groups (7 groups)"	
	   
recode age (0/9 = 1 "0-9") (10/17 = 2 "10-17")(18/59 = 3 "18-59") ///
		   (60/max=4 "60+"), gen(agec4)
lab var agec4 "age groups (4 groups)"

recode age (0/17 = 1 "0-17") (18/max = 2 "18+"), gen(agec2)		 		   
lab var agec2 "age groups (2 groups)"


clonevar marital = hv115						//marital 
codebook marital, tab (9)
recode marital (0=1)(1=2)
label define lab_mar 1"never married" 2"currently married" ///
					 3"widowed" 4"divorced" 5"not living together"
label values marital lab_mar	
label var marital "marital status"
tab hv115 marital, m


gen member = 1									//hhsize
bysort hh_id: egen hhsize = sum(member)
label var hhsize "hh size"
tab hhsize, m
drop member

   
codebook hv024, tab (99)						//subnational region
clonevar region = hv024
lab define lab_reg ///
1  "Berat" ///
2  "Dibër" ///
3  "Durrës" ///
4  "Elbasan" ///
5  "Fier" ///
6  "Gjirokastër" ///
7  "Korçë" ///
8  "Kukes" ///
9  "Lezhë" ///
10 "Shkodër" ///
11 "Tirane" ///
12 "Vlorë" 
lab values region lab_reg
lab var region "subnational region"
codebook region, tab (99)


clonevar region_01 = region 
lab var region "hot: subnational region"
 

********************************************************************************
***  Step 2 Data preparation  ***
***  standardisation of the 10 global MPI indicators 
***  identification of non-deprived & deprived individuals  
********************************************************************************


********************************************************************************
*** Step 2.1 Years of Schooling ***
********************************************************************************

codebook hv108, tab(30)
clonevar  eduyears = hv108   
	//Total number of years of education
replace eduyears = . if eduyears>30
	//Recode any unreasonable years of highest education as missing value
replace eduyears = . if eduyears>=age & age>0
	/*The variable "eduyears" was replaced with a '.' if total years of 
	education was more than individual's age */
replace eduyears = 0 if age < 10 
	/*The variable "eduyears" was replaced with a '0' given that the criteria 
	for this indicator is household member aged 10 years or older */

	
	/*A control variable is created on whether there is information on 
	years of education for at least 2/3 of the household members aged 10 years 
	and older */	
gen temp = 1 if eduyears!=. & age>=10 & age!=.
bysort	hh_id: egen no_missing_edu = sum(temp)
	/*Total household members who are 10 years and older with no missing 
	years of education */
gen temp2 = 1 if age>=10 & age!=.
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are 10 years and older 
replace no_missing_edu = no_missing_edu/hhs
replace no_missing_edu = (no_missing_edu>=2/3)
	/*Identify whether there is information on years of education for at 
	least 2/3 of the household members aged 10 years and older */
tab no_missing_edu, miss
label var no_missing_edu "No missing edu for at least 2/3 of the HH members aged 10 years & older"		
drop temp temp2 hhs


*** Standard MPI ***
/*The entire household is considered deprived if no household member aged 
10 years or older has completed SIX years of schooling.*/
******************************************************************* 
gen	 years_edu6 = (eduyears>=6)
	/* The years of schooling indicator takes a value of "1" if at least someone 
	in the hh has reported 6 years of education or more */
replace years_edu6 = . if eduyears==.
bysort hh_id: egen hh_years_edu6_1 = max(years_edu6)
gen	hh_years_edu6 = (hh_years_edu6_1==1)
replace hh_years_edu6 = . if hh_years_edu6_1==.
replace hh_years_edu6 = . if hh_years_edu6==0 & no_missing_edu==0 
lab var hh_years_edu6 "Household has at least one member with 6 years of edu"


	
*** Destitution MPI ***
/*The entire household is considered deprived if no household member 
aged 10 years or older has completed at least one year of schooling.*/
******************************************************************* 
gen	years_edu1 = (eduyears>=1)
replace years_edu1 = . if eduyears==.
bysort	hh_id: egen hh_years_edu_u = max(years_edu1)
replace hh_years_edu_u = . if hh_years_edu_u==0 & no_missing_edu==0
lab var hh_years_edu_u "Household has at least one member with 1 year of edu"


********************************************************************************
*** Step 2.2 Child School Attendance ***
********************************************************************************

codebook hv121, tab (99)
clonevar attendance = hv121 
recode attendance (2=1) 
label define lab_attend 1 "currently attending" 0 "not currently attending"
label values attendance lab_attend
label var attendance "Attended school during current school year"
codebook attendance, tab (99)
	

*** Standard MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 8. */ 
******************************************************************* 
gen	child_schoolage = (age>=6 & age<=14)
	/* Note: In Albania, the official school entrance age to primary school is 
	6 years. So, age range is 6-14 (=6+8) 
	Source: Country report p.12 and http://data.uis.unesco.org/?ReportId=163 */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the school age children */
count if child_schoolage==1 & attendance==.
	//Understand how many eligible school aged children are not attending school 
gen temp = 1 if child_schoolage==1 & attendance!=.
	/*Generate a variable that captures the number of eligible school aged 
	children who are attending school */
bysort hh_id: egen no_missing_atten = sum(temp)	
	/*Total school age children with no missing information on school 
	attendance */
gen temp2 = 1 if child_schoolage==1	
bysort hh_id: egen hhs = sum(temp2)
	//Total number of household members who are of school age
replace no_missing_atten = no_missing_atten/hhs 
replace no_missing_atten = (no_missing_atten>=2/3)
	/*Identify whether there is missing information on school attendance for 
	more than 2/3 of the school age children */			
tab no_missing_atten, miss
label var no_missing_atten "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs
		
bysort hh_id: egen hh_children_schoolage = sum(child_schoolage)
replace hh_children_schoolage = (hh_children_schoolage>0) 
	//It takes value 1 if the household has children in school age
lab var hh_children_schoolage "Household has children in school age"

gen	child_not_atten = (attendance==0) if child_schoolage==1
replace child_not_atten = . if attendance==. & child_schoolage==1
bysort	hh_id: egen any_child_not_atten = max(child_not_atten)
gen	hh_child_atten = (any_child_not_atten==0) 
replace hh_child_atten = . if any_child_not_atten==.
replace hh_child_atten = 1 if hh_children_schoolage==0
replace hh_child_atten = . if hh_child_atten==1 & no_missing_atten==0 
	/*If the household has been intially identified as non-deprived, but has 
	missing school attendance for at least 2/3 of the school aged children, then 
	we replace this household with a value of '.' because there is insufficient 
	information to conclusively conclude that the household is not deprived */
lab var hh_child_atten "Household has all school age children up to class 8 in school"
tab hh_child_atten, miss


	
*** Destitution MPI ***
/*The entire household is considered deprived if any school-aged 
child is not attending school up to class 6. */ 
******************************************************************* 

gen	child_schoolage_6 = (age>=6 & age<=12) 
	/*Note: In Albania, the official school entrance age is 6 years.  
	  So, age range for destitution measure is 6-12 (=6+6) */

	
	/*A control variable is created on whether there is no information on 
	school attendance for at least 2/3 of the children attending school up to 
	class 6 */	
count if child_schoolage_6==1 & attendance==.
gen temp = 1 if child_schoolage_6==1 & attendance!=.
bysort hh_id: egen no_missing_atten_u = sum(temp)	
gen temp2 = 1 if child_schoolage_6==1	
bysort hh_id: egen hhs = sum(temp2)
replace no_missing_atten_u = no_missing_atten_u/hhs 
replace no_missing_atten_u = (no_missing_atten_u>=2/3)		
tab no_missing_atten_u, miss
label var no_missing_atten_u "No missing school attendance for at least 2/3 of the school aged children"		
drop temp temp2 hhs		
		
bysort	hh_id: egen hh_children_schoolage_6 = sum(child_schoolage_6)
replace hh_children_schoolage_6 = (hh_children_schoolage_6>0) 
lab var hh_children_schoolage_6 "Household has children in school age (6 years of school)"

gen	child_atten_6 = (attendance==1) if child_schoolage_6==1
replace child_atten_6 = . if attendance==. & child_schoolage_6==1
bysort	hh_id: egen any_child_atten_6 = max(child_atten_6)
gen	hh_child_atten_u = (any_child_atten_6==1) 
replace hh_child_atten_u = . if any_child_atten_6==.
replace hh_child_atten_u = 1 if hh_children_schoolage_6==0
replace hh_child_atten_u = . if hh_child_atten_u==0 & no_missing_atten_u==0 
lab var hh_child_atten_u "Household has at least one school age children up to class 6 in school"
tab hh_child_atten_u, miss



********************************************************************************
*** Step 2.3 Nutrition ***
********************************************************************************


********************************************************************************
*** Step 2.3a Adult Nutrition ***
********************************************************************************
 
codebook ha40 hb40 


foreach var in ha40 hb40 {
			 gen inf_`var' = 1 if `var'!=.
			 bysort sex: tab age inf_`var' 
			 //Albania DHS 2016-17: women 15-59 years; men 15-59 years 
			 drop inf_`var'
			 }
***

*** BMI Indicator for Women 15-59 years ***
******************************************************************* 
gen	f_bmi = ha40/100
lab var f_bmi "Women's BMI"
gen	f_low_bmi = (f_bmi<18.5)
replace f_low_bmi = . if f_bmi==. | f_bmi>=99.97
lab var f_low_bmi "BMI of women < 18.5"

gen	f_low_bmi_u = (f_bmi<17)
replace f_low_bmi_u = . if f_bmi==. | f_bmi>=99.97
lab var f_low_bmi_u "BMI of women <17"



*** BMI Indicator for Men 15-59 years ***
******************************************************************* 
gen m_bmi = hb40/100
lab var m_bmi "Male's BMI"
gen m_low_bmi = (m_bmi<18.5)
replace m_low_bmi = . if m_bmi==. | m_bmi>=99.97 
lab var m_low_bmi "BMI of male < 18.5"


gen	m_low_bmi_u = (m_bmi<17)
replace m_low_bmi_u = . if m_bmi==. | m_bmi>=99.97
lab var m_low_bmi_u "BMI of male <17"


 
*** Standard MPI: BMI-for-age for individuals 15-19 years 
*** 			  and BMI for individuals 20-59 years ***
******************************************************************* 
gen low_bmi_byage = 0
lab var low_bmi_byage "Individuals with low BMI or BMI-for-age"
replace low_bmi_byage = 1 if f_low_bmi==1
	//Replace variable "low_bmi_byage = 1" if eligible women have low BMI	
replace low_bmi_byage = 1 if low_bmi_byage==0 & m_low_bmi==1 
	//Replace variable "low_bmi_byage = 1" if eligible men have low BMI. 

	
/*Note: The following command replaces BMI with BMI-for-age for those between 
the age group of 15-19 by their age in months where information is available */
	//Replacement for girls: 
replace low_bmi_byage = 1 if low_bmiage==1 & age_month!=.
replace low_bmi_byage = 0 if low_bmiage==0 & age_month!=.
	/*Replacements for boys - if there is no male anthropometric data for boys, 
	then 0 changes are made: */
replace low_bmi_byage = 1 if low_bmiage_b==1 & age_month_b!=.
replace low_bmi_byage = 0 if low_bmiage_b==0 & age_month_b!=.
	
	
/*Note: The following control variable is applied when there is BMI information 
for adults and BMI-for-age for teenagers.*/	
replace low_bmi_byage = . if f_low_bmi==. & m_low_bmi==. & low_bmiage==. & low_bmiage_b==. 
		
bysort hh_id: egen low_bmi = max(low_bmi_byage)
gen	hh_no_low_bmiage = (low_bmi==0)
	/*Households take a value of '1' if all eligible adults and teenagers in the 
	household has normal bmi or bmi-for-age */	
replace hh_no_low_bmiage = . if low_bmi==.
	/*Households take a value of '.' if there is no information from eligible 
	individuals in the household */
replace hh_no_low_bmiage = 1 if no_adults_eligible==1	
	//Households take a value of '1' if there is no eligible adult population.
drop low_bmi	
lab var hh_no_low_bmiage "Household has no adult with low BMI or BMI-for-age"
tab hh_no_low_bmiage, miss	


*** DST MPI: BMI-for-age for individuals 15-19 years 
*** 			 and BMI for individuals 20-59 years ***
********************************************************************************
gen low_bmi_byage_u = 0
replace low_bmi_byage_u = 1 if f_low_bmi_u==1
	/*Replace variable "low_bmi_byage_u = 1" if eligible women have low 
	BMI (destitute cutoff)*/	
replace low_bmi_byage_u = 1 if low_bmi_byage_u==0 & m_low_bmi_u==1 
	/*Replace variable "low_bmi_byage_u = 1" if eligible men have low 
	BMI (destitute cutoff).*/

	
/*The following command replaces BMI with BMI-for-age for those between 
the age group of 15-19 by their age in months where information is available */
	//Replacement for girls: 
replace low_bmi_byage_u = 1 if low_bmiage_u==1 & age_month!=.
replace low_bmi_byage_u = 0 if low_bmiage_u==0 & age_month!=.
	/*Replacements for boys - if there is no male anthropometric data for boys, 
	then 0 changes are made: */
replace low_bmi_byage_u = 1 if low_bmiage_b_u==1 & age_month_b!=.
replace low_bmi_byage_u = 0 if low_bmiage_b_u==0 & age_month_b!=.
	
	
/*The following control variable is applied when there is BMI information 
for adults and BMI-for-age for teenagers. */
replace low_bmi_byage_u = . if f_low_bmi_u==. & low_bmiage_u==. & m_low_bmi_u==. & low_bmiage_b_u==. 

		
bysort hh_id: egen low_bmi = max(low_bmi_byage_u)
gen	hh_no_low_bmiage_u = (low_bmi==0)
	/*Households take a value of '1' if all eligible adults and teenagers in the 
	household has normal bmi or bmi-for-age (destitution cutoff) */
replace hh_no_low_bmiage_u = . if low_bmi==.
	/*Households take a value of '.' if there is no information from eligible 
	individuals in the household */
replace hh_no_low_bmiage_u = 1 if no_adults_eligible==1	
	//Households take a value of '1' if there is no eligible adult population.
drop low_bmi	
lab var hh_no_low_bmiage_u "Household has no adult with low BMI or BMI-for-age(<17/-3sd)"
tab hh_no_low_bmiage_u, miss	

	
********************************************************************************
*** Step 2.3b Child Nutrition ***
********************************************************************************

*** Child Underweight Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(underweight)
gen	hh_no_underweight = (temp==0) 
	//Takes value 1 if no child in the hh is underweight 
replace hh_no_underweight = . if temp==.
replace hh_no_underweight = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_underweight "Household has no child underweight - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(underweight_u)
gen	hh_no_underweight_u = (temp==0) 
replace hh_no_underweight_u = . if temp==.
replace hh_no_underweight_u = 1 if no_child_eligible==1 
lab var hh_no_underweight_u "Destitute: Household has no child underweight"
drop temp


*** Child Stunting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(stunting)
gen	hh_no_stunting = (temp==0) 
	//Takes value 1 if no child in the hh is stunted
replace hh_no_stunting = . if temp==.
replace hh_no_stunting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_stunting "Household has no child stunted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(stunting_u)
gen	hh_no_stunting_u = (temp==0) 
replace hh_no_stunting_u = . if temp==.
replace hh_no_stunting_u = 1 if no_child_eligible==1 
lab var hh_no_stunting_u "Destitute: Household has no child stunted"
drop temp


*** Child Wasting Indicator ***
************************************************************************

*** Standard MPI ***
bysort hh_id: egen temp = max(wasting)
gen	hh_no_wasting = (temp==0) 
	//Takes value 1 if no child in the hh is wasted
replace hh_no_wasting = . if temp==.
replace hh_no_wasting = 1 if no_child_eligible==1 
	//Households with no eligible children will receive a value of 1
lab var hh_no_wasting "Household has no child wasted - 2 stdev"
drop temp


*** Destitution MPI  ***
bysort hh_id: egen temp = max(wasting_u)
gen	hh_no_wasting_u = (temp==0) 
replace hh_no_wasting_u = . if temp==.
replace hh_no_wasting_u = 1 if no_child_eligible==1 
lab var hh_no_wasting_u "Destitute: Household has no child wasted"
drop temp


*** Child Either Underweight or Stunted Indicator ***
************************************************************************

*** Standard MPI ***
gen hh_no_uw_st = 1 if hh_no_stunting==1 & hh_no_underweight==1
replace hh_no_uw_st = 0 if hh_no_stunting==0 | hh_no_underweight==0
	//Takes value 0 if child in the hh is stunted or underweight 
replace hh_no_uw_st = . if hh_no_stunting==. & hh_no_underweight==.
replace hh_no_uw_st = 1 if no_child_eligible==1
	//Households with no eligible children will receive a value of 1 
lab var hh_no_uw_st "Household has no child underweight or stunted"


*** Destitution MPI  ***
gen hh_no_uw_st_u = 1 if hh_no_stunting_u==1 & hh_no_underweight_u==1
replace hh_no_uw_st_u = 0 if hh_no_stunting_u==0 | hh_no_underweight_u==0
replace hh_no_uw_st_u = . if hh_no_stunting_u==. & hh_no_underweight_u==.
replace hh_no_uw_st_u = 1 if no_child_eligible==1 
lab var hh_no_uw_st_u "Destitute: Household has no child underweight or stunted"


********************************************************************************
*** Step 2.3c Household Nutrition Indicator ***
********************************************************************************

*** Standard MPI ***
/* Members of the household are considered deprived if the household has a 
child under 5 whose height-for-age or weight-for-age is under two standard 
deviation below the median, or has teenager with BMI-for-age that is under two 
standard deviation below the median, or has adults with BMI threshold that is 
below 18.5 kg/m2. Households that have no eligible adult AND no eligible 
children are considered non-deprived. The indicator takes a value of missing 
only if all eligible adults and eligible children have missing information 
in their respective nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st = 1
replace hh_nutrition_uw_st = 0 if hh_no_low_bmiage==0 | hh_no_uw_st==0
replace hh_nutrition_uw_st = . if hh_no_low_bmiage==. & hh_no_uw_st==.
	/*Replace indicator as missing if household has eligible adult and child 
	with missing nutrition information */
replace hh_nutrition_uw_st = . if hh_no_low_bmiage==. & hh_no_uw_st==1 & no_child_eligible==1
	/*Replace indicator as missing if household has eligible adult with missing 
	nutrition information and no eligible child for anthropometric measures */ 
replace hh_nutrition_uw_st = . if hh_no_uw_st==. & hh_no_low_bmiage==1 & no_adults_eligible==1
	/*Replace indicator as missing if household has eligible child with missing 
	nutrition information and no eligible adult for anthropometric measures */ 
replace hh_nutrition_uw_st = 1 if no_eligibles==1  
 	/*We replace households that do not have the applicable population, that is, 
	women 15-49 & children 0-5, as non-deprived in nutrition*/
lab var hh_nutrition_uw_st "Household has no individuals malnourished"
tab hh_nutrition_uw_st, miss


*** Destitution MPI ***
/* Members of the household are considered deprived if the household has a 
child under 5 whose height-for-age or weight-for-age is under three standard 
deviation below the median, or has teenager with BMI-for-age that is under three 
standard deviation below the median, or has adults with BMI threshold that is 
below 17.0 kg/m2. Households that have no eligible adult AND no eligible 
children are considered non-deprived. The indicator takes a value of missing 
only if all eligible adults and eligible children have missing information 
in their respective nutrition variable. */
************************************************************************
gen	hh_nutrition_uw_st_u = 1
replace hh_nutrition_uw_st_u = 0 if hh_no_low_bmiage_u==0 | hh_no_uw_st_u==0
replace hh_nutrition_uw_st_u = . if hh_no_low_bmiage_u==. & hh_no_uw_st_u==.
	/*Replace indicator as missing if household has eligible adult and child 
	with missing nutrition information */
replace hh_nutrition_uw_st_u = . if hh_no_low_bmiage_u==. & hh_no_uw_st_u==1 & no_child_eligible==1
	/*Replace indicator as missing if household has eligible adult with missing 
	nutrition information and no eligible child for anthropometric measures */ 
replace hh_nutrition_uw_st_u = . if hh_no_uw_st_u==. & hh_no_low_bmiage_u==1 & no_adults_eligible==1
	/*Replace indicator as missing if household has eligible child with missing 
	nutrition information and no eligible adult for anthropometric measures */ 
replace hh_nutrition_uw_st_u = 1 if no_eligibles==1   
 	/*We replace households that do not have the applicable population, that is, 
	women 15-49 & children 0-5, as non-deprived in nutrition*/	
lab var hh_nutrition_uw_st_u "Household has no individuals malnourished (destitution)"
tab hh_nutrition_uw_st_u, miss



*** Harmonised MPI ***
	/*Anthropometric measurements were collected from women aged 15 to 59 and 
	children aged under 5 years living in all sampled households, and from
	men 15-59 years living in half of sampled households. For harmonization 
	purposes, we only include anthropometric information from women 15-49
	and men 15-49 years to match with the information available in the 
	earlier survey. In other word, we exclude adults 50-59 from the 
	harmonised estiamtes. */
************************************************************************

//Eligibility for women and men 15-49
tab ha13, miss	
tab ha13 if hv105>=15 & hv105<=59 & hv104==2, miss	
tab ha13 if hv105>=15 & hv105<=49 & hv104==2, miss	
gen fem_nutri_eligible_c = (ha13!=.) if hv105>=15 & hv105<=49 & hv104==2
bysort hh_id: egen hh_n_fem_nutri_eligible_c = sum(fem_nutri_eligible_c) 	
gen	no_fem_nutri_eligible_c = (hh_n_fem_nutri_eligible_c==0)
lab var no_fem_nutri_eligible_c "COT: HH has no eligible women for anthropometric"	
tab no_fem_nutri_eligible no_fem_nutri_eligible_c, miss


tab hb13 hv027 if hv105>=15 & hv105<=59 & hv104==1, miss
tab hb13 hv027 if hv105>=15 & hv105<=49 & hv104==1, miss
gen	male_nutri_eligible_c = (hb13!=.) if hv105>=15 & hv105<=49 & hv104==1
bysort hh_id: egen hh_n_male_nutri_eligible_c = sum(male_nutri_eligible_c)  
gen	no_male_nutri_eligible_c = (hh_n_male_nutri_eligible_c==0)
lab var no_male_nutri_eligible_c "COT: HH has no eligible men for anthropometric"	
tab no_male_nutri_eligible no_male_nutri_eligible_c, miss


gen no_adults_eligible_c = (no_fem_nutri_eligible_c==1 & no_male_nutri_eligible_c==1) 
lab var no_adults_eligible_c "COT: HH has no eligible women or men for anthropometrics"
tab no_adults_eligible_c, miss 

 
gen no_eligibles_c = (no_fem_nutri_eligible_c==1 & no_male_nutri_eligible_c==1 & no_child_eligible==1)
lab var no_eligibles_c "COT: HH has no eligible women, men, or children"
tab no_eligibles_c, miss

 
//BMI indicator for women 15-49
gen	f_bmi_c = ha40/100 if fem_nutri_eligible_c==1  
lab var f_bmi_c "COT:Women's BMI"
gen	f_low_bmi_c = (f_bmi_c<18.5)
replace f_low_bmi_c = . if f_bmi_c==. | f_bmi_c>=99.97
lab var f_low_bmi_c "COT:BMI of women < 18.5"

gen	f_low_bmi_u_c = (f_bmi_c<17)
replace f_low_bmi_u_c = . if f_bmi_c==. | f_bmi_c>=99.97
lab var f_low_bmi_u_c "COT:BMI of women <17"

 
//BMI indicator for men 15-49 
gen m_bmi_c = hb40/100 if male_nutri_eligible_c==1 
lab var m_bmi_c "COT:Male's BMI"
gen m_low_bmi_c = (m_bmi_c<18.5)
replace m_low_bmi_c = . if m_bmi_c==. | m_bmi_c>=99.97 
lab var m_low_bmi_c "COT:BMI of male < 18.5"


gen	m_low_bmi_u_c = (m_bmi_c<17)
replace m_low_bmi_u_c = . if m_bmi_c==. | m_bmi_c>=99.97
lab var m_low_bmi_u_c "COT:BMI of male <17" 
 
 
//Combine BMI and BMI-by-age for adults 15-49
gen low_bmi_byage_c = 0
replace low_bmi_byage_c = 1 if f_low_bmi_c==1	
replace low_bmi_byage_c = 1 if low_bmi_byage_c==0 & m_low_bmi_c==1 
replace low_bmi_byage_c = 1 if low_bmiage==1 & age_month!=. //girls
replace low_bmi_byage_c = 0 if low_bmiage==0 & age_month!=. //girls
replace low_bmi_byage_c = 1 if low_bmiage_b==1 & age_month_b!=. //boys
replace low_bmi_byage_c = 0 if low_bmiage_b==0 & age_month_b!=. //boys
replace low_bmi_byage_c = . if f_low_bmi_c==. & m_low_bmi_c==. & low_bmiage==. & low_bmiage_b==. 		
bysort hh_id: egen low_bmi = max(low_bmi_byage_c)
gen	hh_no_low_bmiage_c = (low_bmi==0)
replace hh_no_low_bmiage_c = . if low_bmi==.
replace hh_no_low_bmiage_c = 1 if no_adults_eligible_c==1	
drop low_bmi
tab hh_no_low_bmiage_c, miss	


gen low_bmi_byage_u_c = 0
replace low_bmi_byage_u_c = 1 if f_low_bmi_u_c==1 
replace low_bmi_byage_u_c = 1 if low_bmi_byage_u_c==0 & m_low_bmi_u_c==1 
replace low_bmi_byage_u_c = 1 if low_bmiage_u==1 & age_month!=. //girls
replace low_bmi_byage_u_c = 0 if low_bmiage_u==0 & age_month!=. //girls
replace low_bmi_byage_u_c = 1 if low_bmiage_b_u==1 & age_month_b!=. //boys
replace low_bmi_byage_u_c = 0 if low_bmiage_b_u==0 & age_month_b!=. //boys
replace low_bmi_byage_u_c = . if f_low_bmi_u_c==. & low_bmiage_u==. & m_low_bmi_u_c==. & low_bmiage_b_u==. 	
bysort hh_id: egen low_bmi = max(low_bmi_byage_u_c)
gen	hh_no_low_bmiage_u_c = (low_bmi==0)
replace hh_no_low_bmiage_u_c = . if low_bmi==.
replace hh_no_low_bmiage_u_c = 1 if no_adults_eligible_c==1	
drop low_bmi
tab hh_no_low_bmiage_u_c, miss	


//Final harmonised nutrition indicator
gen	hh_nutrition_uw_st_c = 1
replace hh_nutrition_uw_st_c = 0 if hh_no_low_bmiage_c==0 | hh_no_uw_st==0
replace hh_nutrition_uw_st_c = . if hh_no_low_bmiage_c==. & hh_no_uw_st==.
replace hh_nutrition_uw_st_c = . if hh_no_low_bmiage_c==. & hh_no_uw_st==1 & no_child_eligible==1 
replace hh_nutrition_uw_st_c = . if hh_no_uw_st==. & hh_no_low_bmiage_c==1 & no_adults_eligible_c==1 
replace hh_nutrition_uw_st_c = 1 if no_child_eligible==1 & no_adults_eligible_c==1  	
lab var hh_nutrition_uw_st_c "COT: HH has no individuals malnourished"
tab hh_nutrition_uw_st_c, miss
tab hh_nutrition_uw_st, miss


gen	hh_nutrition_uw_st_u_c = 1
replace hh_nutrition_uw_st_u_c = 0 if hh_no_low_bmiage_u_c==0 | hh_no_uw_st_u==0
replace hh_nutrition_uw_st_u_c = . if hh_no_low_bmiage_u_c==. & hh_no_uw_st_u==.
replace hh_nutrition_uw_st_u_c = . if hh_no_low_bmiage_u_c==. & hh_no_uw_st_u==1 & no_child_eligible==1 
replace hh_nutrition_uw_st_u_c = . if hh_no_uw_st_u==. & hh_no_low_bmiage_u_c==1 & no_adults_eligible_c==1 
replace hh_nutrition_uw_st_u_c = 1 if no_child_eligible==1 & no_adults_eligible_c==1  	 		
lab var hh_nutrition_uw_st_u_c "COT-DST: HH has no individuals malnourished"
tab hh_nutrition_uw_st_u_c, miss
tab hh_nutrition_uw_st_u, miss



********************************************************************************
*** Step 2.4 Child Mortality ***
********************************************************************************
	
codebook v206 v207 mv206 mv207
	/*v206 or mv206: number of sons who have died 
	  v207 or mv207: number of daughters who have died */
	
egen temp_f = rowtotal(v206 v207), missing
	//Total child mortality reported by eligible women
replace temp_f = 0 if v201==0
	//This line replaces women who have never given birth	
bysort	hh_id: egen child_mortality_f = sum(temp_f), missing
lab var child_mortality_f "Occurrence of child mortality reported by women"
tab child_mortality_f, miss
drop temp_f
	
	//Total child mortality reported by eligible men	
egen temp_m = rowtotal(mv206 mv207), missing
replace temp_m = 0 if mv201==0
bysort	hh_id: egen child_mortality_m = sum(temp_m), missing
lab var child_mortality_m "Occurrence of child mortality reported by men"
tab child_mortality_m, miss
drop temp_m

egen child_mortality = rowmax(child_mortality_f child_mortality_m)
lab var child_mortality "Total child mortality within household"
tab child_mortality, miss	

		
*** Standard MPI *** 
/* Members of the household are considered deprived if women in the household 
reported mortality among children under 18 in the last 5 years from the survey 
year. Members of the household is considered non-deprived if eligible women 
within the household reported (i) no child mortality or (ii) if any child died 
longer than 5 years from the survey year or (iii) if any child 18 years and 
older died in the last 5 years. In adddition, members of the household were 
identified as non-deprived if eligible men within the household reported no 
child mortality in the absence of information from women. Households that have 
no eligible women or adult are considered non-deprived. The indicator takes 
a missing value if there was missing information on reported death from 
eligible individuals. */
************************************************************************

tab childu18_died_per_wom_5y, miss
	/* The 'childu18_died_per_wom_5y' variable was constructed in Step 1.2 using 
	information from individual women who ever gave birth in the BR file. The 
	missing values represent eligible woman who have never ever given birth and 
	so are not present in the BR file. But these 'missing women' may be living 
	in households where there are other women with child mortality information 
	from the BR file. So at this stage, it is important that we aggregate the 
	information that was obtained from the BR file at the household level. This
	ensures that women who were not present in the BR file is assigned with a 
	value, following the information provided by other women in the household.*/
replace childu18_died_per_wom_5y = 0 if v201==0 
	/*Assign a value of "0" for:
	- all eligible women who never ever gave birth */	
replace childu18_died_per_wom_5y = 0 if no_fem_eligible==1 
	/*Assign a value of "0" for:
	- individuals living in households that have non-eligible women */	
	
bysort hh_id: egen childu18_mortality_5y = sum(childu18_died_per_wom_5y), missing
replace childu18_mortality_5y = 0 if childu18_mortality_5y==. & child_mortality==0
	/*Replace all households as 0 death if women has missing value and men 
	reported no death in those households */
label var childu18_mortality_5y "Under 18 child mortality within household past 5 years reported by women"
tab childu18_mortality_5y, miss		
	
gen hh_mortality_u18_5y = (childu18_mortality_5y==0)
replace hh_mortality_u18_5y = . if childu18_mortality_5y==.
lab var hh_mortality_u18_5y "Household had no under 18 child mortality in the last 5 years"
tab hh_mortality_u18_5y, miss 


*** Destitution MPI *** 
*** (same as standard MPI) ***
************************************************************************
gen hh_mortality_u = hh_mortality_u18_5y	
lab var hh_mortality_u "Household had no under 18 child mortality in the last 5 years"	


********************************************************************************
*** Step 2.5 Electricity ***
********************************************************************************
	/*Albania DHS 2017-18: The survey did not collect information on 
	electricity. We assume all individuals have electricity. 
	
	There is sufficient evidence to indicate that 100% of the population in 
	Albania has access to electricity in 1990 and 2014. 
	Source 1: https://data.worldbank.org/indicator/eg.elc.accs.zs
	Source 2: https://www.ceicdata.com/en/albania/energy-production-and-consumption
	
	A UN report in 2013 indicate that 0% of the people in Albania are without 
	access to electricity (p.14). 
	Source: https://sustainabledevelopment.un.org/content/documents/1272A%20Survey%20of%20International%20Activities%20in%20Energy%20Access%20and%20Electrification.pdf
	*/

*** Standard MPI ***
/*Members of the household are considered 
deprived if the household has no electricity */
***************************************************
gen electricity = 1
label var electricity "Household has electricity"


*** Destitution MPI  ***
*** (same as standard MPI) ***
***************************************************
gen electricity_u = electricity
label var electricity_u "Household has electricity"


********************************************************************************
*** Step 2.6 Sanitation ***
********************************************************************************

/*
Improved sanitation facilities include flush or pour flush toilets to sewer 
systems, septic tanks or pit latrines, ventilated improved pit latrines, pit 
latrines with a slab, and composting toilets. These facilities are only 
considered improved if it is private, that is, it is not shared with other 
households.
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-02-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

desc hv205 hv225
clonevar toilet = hv205  
clonevar shared_toilet = hv225
codebook shared_toilet, tab(99)  

	
*** Standard MPI ***
/*Members of the household are considered deprived if the household's 
sanitation facility is not improved (according to the SDG guideline) 
or it is improved but shared with other households*/
********************************************************************
codebook toilet, tab(99)
gen	toilet_mdg = ((toilet<23 | toilet==41) & shared_toilet!=1) 
	/*Household is assigned a value of '1' if it uses improved sanitation and 
	does not share toilet with other households  */	
replace toilet_mdg = 0 if (toilet<23 | toilet==41)  & shared_toilet==1   
	/*Household is assigned a value of '0' if it uses improved sanitation 
	but shares toilet with other households  */		
replace toilet_mdg = 0 if toilet == 14 | toilet == 15
	/*Household is assigned a value of '0' if it uses non-improved sanitation: 
	"flush to somewhere else" and "flush don't know where"  */	
replace toilet_mdg = . if toilet==.  | toilet==99
	//Household is assigned a value of '.' if it has missing information 	
	
lab var toilet_mdg "Household has improved sanitation"
tab toilet toilet_mdg, miss


*** Destitution MPI ***
/*Members of the household are considered deprived if household practises 
open defecation or uses other unidentifiable sanitation practises */
********************************************************************
gen	toilet_u = .
replace toilet_u = 0 if toilet==31 | toilet==96 
	/*Household is assigned a value of '0' if it practises open defecation or 
	others */	
replace toilet_u = 1 if toilet!=31 & toilet!=96 & toilet!=. & toilet!=99
	/*Household is assigned a value of '1' if it does not practise open 
	defecation or others  */	
lab var toilet_u "Household does not practise open defecation or others"
tab toilet toilet_u, miss


********************************************************************************
*** Step 2.7 Drinking Water  ***
********************************************************************************
/*
Improved drinking water sources include the following: piped water into 
dwelling, yard or plot; public taps or standpipes; boreholes or tubewells; 
protected dug wells; protected springs; packaged water; delivered water and 
rainwater which is located on premises or is less than a 30-minute walk from 
home roundtrip. 
Source: https://unstats.un.org/sdgs/metadata/files/Metadata-06-01-01.pdf

Note: In cases of mismatch between the country report and the internationally 
agreed guideline, we followed the report.
*/

desc hv201 hv204 hv202
clonevar water = hv201  
clonevar timetowater = hv204  
clonevar ndwater = hv202  	
tab hv202 if water==71, miss 	
/*Households using bottled water are only considered to be using 
improved water when they use water from an improved source for cooking and 
personal hygiene. This is because the quality of bottled water is not known. */	


*** Standard MPI ***
/* Members of the household are considered deprived if the household 
does not have access to improved drinking water (according to the SDG 
guideline) or safe drinking water is at least a 30-minute walk from 
home, roundtrip */
********************************************************************
codebook water, tab(99)
gen	water_mdg = 1 if water==11 | water==12 | water==14 | water==21 | ///
					 water==31 | water==41 | water==71 
					 
replace water_mdg = 0 if water==32 | water==42 | water==61 | ///
						 water==62 | water==96 
	/*Deprived if it is "unprotected well", "unprotected spring", 
	"tanker truck", "cart with small tank","other" */

codebook timetowater, tab(9999)	
replace water_mdg = 0 if water_mdg==1 & timetowater >= 30 & timetowater!=. & ///
						 timetowater!=996 & timetowater!=998 
	/*Deprived if water source is 30 minutes or more from home, roundtrip. 
	Please check the value assigned to 'in premises'. If this is different from 
	996, add to the condition accordingly */
	
replace water_mdg = . if water==. | water==99
replace water_mdg = 0 if water==71 & ///
						(ndwater==32 | ndwater==42 | /// 
						 ndwater==61 | ndwater==62 | ndwater==96)
	/*Households using bottled water for drinking are classified as using an 
	improved or unimproved source according to their water source for 
	non-drinking activities	*/
lab var water_mdg "Household has drinking water with MDG standards (considering distance)"
tab water water_mdg, miss


*** Destitution MPI ***
/* Members of the household is identified as destitute if household 
does not have access to safe drinking water, or safe water is more 
than 45 minute walk from home, round trip.*/
********************************************************************
gen	water_u = .
replace water_u = 1 if water==11 | water==12 | water==14 | ///
					   water==21 | water==31 | water==41 | water==71 
						   
replace water_u = 0 if water==32 | water==42 | water==61 | ///
					   water==62 | water==96
						   
replace water_u = 0 if water_u==1 & timetowater>45 & timetowater!=. & ///
					   timetowater!=996 & timetowater!=998
						   
replace water_u = . if water==. | water==99

replace water_u = 0 if water==71 & ///
					   (ndwater==32 | ndwater==42 | ndwater==43 | /// 
					    ndwater==61 | ndwater==62 | ndwater==96) 
						  
lab var water_u "Household has drinking water with MDG standards (45 minutes distance)"
tab water water_u, miss


********************************************************************************
*** Step 2.8 Housing ***
********************************************************************************

/* Members of the household are considered deprived if the household 
has a dirt, sand or dung floor */
lookfor floor
clonevar floor = hv213 
codebook floor, tab(99)
gen	floor_imp = 1
replace floor_imp = 0 if floor<=12 | floor==96  
	//Deprived if mud/earth, sand, dung, other 	
replace floor_imp = . if floor==. | floor==99 
lab var floor_imp "Household has floor that it is not earth/sand/dung"
tab floor floor_imp, miss		


/* Members of the household are considered deprived if the household has walls 
made of natural or rudimentary materials. We followed the report's definitions
of natural or rudimentary materials. */
lookfor wall
clonevar wall = hv214 
codebook wall, tab(99)	
gen	wall_imp = 1 
replace wall_imp = 0 if wall<=26 | wall==96  
	/*Deprived if stone with mud, plywood, cardboard, 
	uncovered adobe, reused wood, other*/	
replace wall_imp = . if wall==. | wall==99 	
lab var wall_imp "Household has wall that it is not of low quality materials"
tab wall wall_imp, miss	
	

/* Members of the household are considered deprived if the household has roof 
made of natural or rudimentary materials. We followed the report's definitions
of natural and rudimentary materials. */
lookfor roof
clonevar roof = hv215
codebook roof, tab(99)		
gen	roof_imp = 1 
replace roof_imp = 0 if roof<=24 | roof==96  
	//Deprived if rustic mat, cardboard wood planks, other	
replace roof_imp = . if roof==. | roof==99 	
lab var roof_imp "Household has roof that it is not of low quality materials"
tab roof roof_imp, miss


*** Standard MPI ***
/* Members of the household is deprived in housing if the roof, 
floor OR walls are constructed from low quality materials.*/
**************************************************************
gen housing_1 = 1
replace housing_1 = 0 if floor_imp==0 | wall_imp==0 | roof_imp==0
replace housing_1 = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_1 "Household has roof, floor & walls that it is not low quality material"
tab housing_1, miss


*** Destitution MPI ***
/* Members of the household is deprived in housing if two out 
of three components (roof and walls; OR floor and walls; OR 
roof and floor) the are constructed from low quality materials. */
**************************************************************

gen housing_u = 1
replace housing_u = 0 if (floor_imp==0 & wall_imp==0 & roof_imp==1) | ///
						 (floor_imp==0 & wall_imp==1 & roof_imp==0) | ///
						 (floor_imp==1 & wall_imp==0 & roof_imp==0) | ///
						 (floor_imp==0 & wall_imp==0 & roof_imp==0)
replace housing_u = . if floor_imp==. & wall_imp==. & roof_imp==.
lab var housing_u "Household has one of three aspects(either roof,floor/walls) that is not low quality material"
tab housing_u, miss


********************************************************************************
*** Step 2.9 Cooking Fuel ***
********************************************************************************

/*
Solid fuel are solid materials burned as fuels, which includes coal as well as 
solid biomass fuels (wood, animal dung, crop wastes and charcoal). 

Source: 
https://apps.who.int/iris/bitstream/handle/10665/141496/9789241548885_eng.pdf
*/

lookfor cooking combustible
clonevar cookingfuel = hv226


*** Standard MPI ***
/* Members of the household are considered deprived if the 
household uses solid fuels and solid biomass fuels for cooking. */
*****************************************************************
codebook cookingfuel, tab(99)
gen	cooking_mdg = 1
replace cooking_mdg = 0 if cookingfuel>5 & cookingfuel<95 
replace cooking_mdg = . if cookingfuel==. | cookingfuel==99
lab var cooking_mdg "Household cooks with clean fuels"
	//Deprived if: coal/lignite, charcoal, wood, agricultural crop 		 
tab cookingfuel cooking_mdg, miss	

	
*** Destitution MPI ***
*** (same as standard MPI) ***
****************************************
gen	cooking_u = cooking_mdg
lab var cooking_u "Household uses clean fuels for cooking"


********************************************************************************
*** Step 2.10 Assets ownership ***
********************************************************************************
/*Assets that are included in the global MPI: Radio, TV, telephone, bicycle, 
motorbike, refrigerator, car, computer and animal cart */


*** Television/LCD TV/plasma TV/color TV/black & white tv
lookfor tv television plasma lcd	
codebook hv208
clonevar television = hv208 
lab var television "Household has television"	


***	Radio/walkman/stereo/kindle
lookfor radio walkman stereo
codebook hv207
clonevar radio = hv207 
lab var radio "Household has radio"


***	Handphone/telephone/iphone/mobilephone/ipod
lookfor telephone téléphone mobilephone ipod
codebook hv221 hv243a
clonevar telephone = hv221
replace telephone=1 if telephone!=1 & hv243a==1	
	//hv243a=mobilephone. Combine information on telephone and mobilephone.	
tab hv243a hv221 if telephone==1,miss
lab var telephone "Household has telephone (landline/mobilephone)"	


***	Refrigerator/icebox/fridge
lookfor refrigerator réfrigérateur
codebook hv209
clonevar refrigerator = hv209 
lab var refrigerator "Household has refrigerator"


***	Car/van/lorry/truck
lookfor car voiture truck van
codebook hv212
clonevar car = hv212  
lab var car "Household has car"	


***	Bicycle/cycle rickshaw
lookfor bicycle bicyclette
codebook hv210
clonevar bicycle = hv210 
lab var bicycle "Household has bicycle"	


***	Motorbike/motorized bike/autorickshaw
lookfor motorbike moto
codebook hv211	
clonevar motorbike = hv211 
lab var motorbike "Household has motorbike"	


***	Computer/laptop/tablet
lookfor computer ordinateur laptop ipad tablet
codebook hv243e
clonevar computer = hv243e
lab var computer "Household has computer"

	
***	Animal cart
lookfor cart 
codebook hv243c
clonevar animal_cart = hv243c
lab var animal_cart "Household has animal cart"	



*** Standard MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own more than one of: radio, TV, telephone, bike, motorbike, 
refrigerator, computer or animal cart and does not own a car or truck.*/
*****************************************************************************
egen n_small_assets2 = rowtotal(television radio telephone refrigerator bicycle motorbike computer animal_cart), missing
lab var n_small_assets2 "Household Number of Small Assets Owned" 
   
gen hh_assets2 = (car==1 | n_small_assets2 > 1) 
replace hh_assets2 = . if car==. & n_small_assets2==.
lab var hh_assets2 "Household Asset Ownership: HH has car or more than 1 small assets incl computer & animal cart"



*** Destitution MPI ***
/* Members of the household are considered deprived in assets if the household 
does not own any assets.*/
*****************************************************************************	
gen	hh_assets2_u = (car==1 | n_small_assets2>0)
replace hh_assets2_u = . if car==. & n_small_assets2==.
lab var hh_assets2_u "Household Asset Ownership: HH has car or at least 1 small assets incl computer & animal cart"


********************************************************************************
*** Step 2.11 Rename and keep variables for MPI calculation 
********************************************************************************


	//Retain data on sampling design: 
desc hv022 hv021	
clonevar strata = hv022
clonevar psu = hv021
label var psu "Primary sampling unit"
label var strata "Sample strata"

	
	//Retain year, month & date of interview:
desc hv007 hv006 hv008
clonevar year_interview = hv007 	
clonevar month_interview = hv006 
clonevar date_interview = hv008
 
		
*** Keep main variables require for MPI calculation ***
keep hh_id ind_id strata psu weight weight_ch sex age hhsize ///
area agec7 agec4 agec2 headship region marital relationship ///
region region_01 year_interview month_interview date_interview /// 
fem_nutri_eligible no_fem_nutri_eligible no_fem_eligible ///
male_nutri_eligible no_male_nutri_eligible no_male_eligible ///
child_eligible no_child_eligible no_adults_eligible ///
no_child_fem_eligible no_eligibles ///
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
low_bmiage low_bmiage_u low_bmi_byage f_bmi m_bmi ///
hh_no_low_bmiage hh_no_low_bmiage_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet shared_toilet toilet_mdg toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u  ///
hh_nutrition_uw_st_c hh_nutrition_uw_st_u_c
	
	 
*** Order file	***
order hh_id ind_id strata psu weight weight_ch sex age hhsize ///
area agec7 agec4 agec2 headship region marital relationship ///
region region_01 year_interview month_interview date_interview /// 
fem_nutri_eligible no_fem_nutri_eligible no_fem_eligible ///
male_nutri_eligible no_male_nutri_eligible no_male_eligible ///
child_eligible no_child_eligible no_adults_eligible ///
no_child_fem_eligible no_eligibles ///
eduyears no_missing_edu hh_years_edu6 hh_years_edu_u ///
attendance child_schoolage no_missing_atten hh_child_atten hh_child_atten_u ///
underweight stunting wasting underweight_u stunting_u wasting_u ///
low_bmiage low_bmiage_u low_bmi_byage f_bmi m_bmi ///
hh_no_low_bmiage hh_no_low_bmiage_u ///
hh_no_underweight hh_no_stunting hh_no_wasting hh_no_uw_st ///
hh_no_underweight_u hh_no_stunting_u hh_no_wasting_u hh_no_uw_st_u ///
hh_nutrition_uw_st hh_nutrition_uw_st_u ///
child_mortality hh_mortality_u18_5y hh_mortality_u ///
electricity electricity_u toilet shared_toilet toilet_mdg toilet_u ///
water timetowater ndwater water_mdg water_u floor wall roof ///
floor_imp wall_imp roof_imp housing_1 housing_u ///
cookingfuel cooking_mdg cooking_u television radio telephone ///
refrigerator car bicycle motorbike animal_cart computer ///
n_small_assets2 hh_assets2 hh_assets2_u  ///
hh_nutrition_uw_st_c hh_nutrition_uw_st_u_c


*** Rename key global MPI indicators for estimation ***
recode hh_mortality_u18_5y  (0=1)(1=0) , gen(d_cm)
recode hh_nutrition_uw_st 	(0=1)(1=0) , gen(d_nutr)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ)
recode electricity 			(0=1)(1=0) , gen(d_elct)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani)
recode housing_1 			(0=1)(1=0) , gen(d_hsg)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl)
recode hh_assets2 			(0=1)(1=0) , gen(d_asst)
 

*** Rename key global MPI indicators for destitution estimation ***
recode hh_mortality_u       (0=1)(1=0) , gen(dst_cm)
recode hh_nutrition_uw_st_u (0=1)(1=0) , gen(dst_nutr)
recode hh_child_atten_u 	(0=1)(1=0) , gen(dst_satt)
recode hh_years_edu_u 		(0=1)(1=0) , gen(dst_educ)
recode electricity_u		(0=1)(1=0) , gen(dst_elct)
recode water_u 				(0=1)(1=0) , gen(dst_wtr)
recode toilet_u 			(0=1)(1=0) , gen(dst_sani)
recode housing_u 			(0=1)(1=0) , gen(dst_hsg)
recode cooking_u			(0=1)(1=0) , gen(dst_ckfl)
recode hh_assets2_u 		(0=1)(1=0) , gen(dst_asst) 
 
 
*** Rename indicators for changes over time estimation ***	
recode hh_mortality_u18_5y  (0=1)(1=0) , gen(d_cm_01)
recode hh_nutrition_uw_st_c (0=1)(1=0) , gen(d_nutr_01)
recode hh_child_atten 		(0=1)(1=0) , gen(d_satt_01)
recode hh_years_edu6 		(0=1)(1=0) , gen(d_educ_01)
recode electricity 			(0=1)(1=0) , gen(d_elct_01)
recode water_mdg 			(0=1)(1=0) , gen(d_wtr_01)
recode toilet_mdg 			(0=1)(1=0) , gen(d_sani_01)
recode housing_1 			(0=1)(1=0) , gen(d_hsg_01)
recode cooking_mdg 			(0=1)(1=0) , gen(d_ckfl_01)
recode hh_assets2 			(0=1)(1=0) , gen(d_asst_01)	
	

recode hh_mortality_u         (0=1)(1=0) , gen(dst_cm_01)
recode hh_nutrition_uw_st_u_c (0=1)(1=0) , gen(dst_nutr_01)
recode hh_child_atten_u 	  (0=1)(1=0) , gen(dst_satt_01)
recode hh_years_edu_u 		  (0=1)(1=0) , gen(dst_educ_01)
recode electricity_u		  (0=1)(1=0) , gen(dst_elct_01)
recode water_u	 			  (0=1)(1=0) , gen(dst_wtr_01)
recode toilet_u 			  (0=1)(1=0) , gen(dst_sani_01)
recode housing_u 			  (0=1)(1=0) , gen(dst_hsg_01)
recode cooking_u			  (0=1)(1=0) , gen(dst_ckfl_01)
recode hh_assets2_u 		  (0=1)(1=0) , gen(dst_asst_01) 
 
 
*** Eligibility for years of schooling indicator ***
gen educ_elig = 1 
replace educ_elig = 0 if age < 10 
label define lab_educ_elig 0"ineligible" 1"eligible"  
label values educ_elig lab_educ_elig
lab var educ_elig "Individual is eligible for educ indicator"
ta eduyears educ_elig,m 
 
 
*** Generate coutry and survey details for estimation ***
char _dta[cty] "Albania"
char _dta[ccty] "ALB"
char _dta[year] "2017-2018" 	
char _dta[survey] "DHS"
char _dta[ccnum] "008"
char _dta[type] "micro"

	
*** Sort, compress and save data for estimation ***
sort ind_id
compress
la da "Micro data for `_dta[ccty]' (`_dta[ccnum]') from `c(current_date)' (`c(current_time)')."	
save "$path_out/alb_dhs17-18.dta", replace 

