# -*- coding: utf-8 -*-
'''
###############################################################################
Code written by Jacob Blackmore and Shaobo Gao for Arxiv 2112:05795

This code generates figure 5 showing the transmission for various scattering
losses with geometry.

outputs are shown to the user using matplotlib and saved in png and pdf formats

requires the cavity module from /dist or DOI:10.5281/zenodo.7020047

optionally uses the ox_plot plotting module, which is included.
###############################################################################
'''

from matplotlib import pyplot as plt
from matplotlib import ticker,gridspec
from matplotlib.colors import ListedColormap,LogNorm
import numpy as np
from jqc import jqc_plot,Ox_plot
from Figures import draw_brace,jqc_red_blue

from matplotlib.text import TextPath
from matplotlib.patches import PathPatch

import os

cwd = os.path.dirname(os.path.abspath(__file__))

Ox_plot.plot_style()

fontsize = 15

N = 128

Theta = np.linspace(0,9,N)
scatter = np.linspace(0,400,N)*1e-6
alpha = 1/20

X,Y = np.meshgrid(Theta,scatter/2)

C_in = 6*alpha*(np.deg2rad(X))**2/(Y)
probability = 1 - 2/(1+np.sqrt(1+2*C_in))
probability[np.isnan(probability)] = 1

transmission_opt = Y*np.sqrt(1+2*C_in)
transmission_opt[np.isnan(transmission_opt)] = 0


fig = plt.figure("fig 5")


size = fig.get_size_inches()
size[1]=0.9*size[1]
fig.set_size_inches(size,forward=True)


gs =gridspec.GridSpec(2,2,height_ratios = [.05,1])

axs = [fig.add_subplot(gs[1,0]),fig.add_subplot(gs[1,1])]

'''axis 0 is probability '''

im = axs[0].imshow(probability, aspect = 'auto', vmin=0,vmax=1,
                   extent=[Theta[0],Theta[-1],scatter[-1]*1e6,scatter[0]*1e6])

axs[0].set_xticks([1,3,5,7,9])
axs[0].set_yticks(np.linspace(0,400,5))

axs[0].set_ylabel(r'Scattering Losses, $\mathcal{L}_\mathrm{scat}$ (ppm)')

cbar_ax = fig.add_subplot(gs[0,0])
tb = fig.colorbar(im, cax=cbar_ax,orientation = 'horizontal',shrink=0.8, pad=0.15)
cbar_ax.xaxis.set_ticks_position('top')
tb.set_ticks([0,0.2,0.4,0.6,0.8,1])
#tb.set_ticklabels(["0","0.25","0.50","0.75","1.0"])

tb.ax.set_title(r'Extraction Probability, $P_\mathrm{ext}$',fontsize=15,pad=15)
levels =[0.2,0.4,0.6,0.8,0.9]
ct = axs[0].contour(probability,levels, colors='k', origin='image',
            extent=[Theta[0],Theta[-1],scatter[-1]*1e6,scatter[0]*1e6])
axs[0].clabel(ct,inline=True,fmt='%1.1f',
    manual=[(1.71,350),(3.36,350),(5.3,300),(7.5,105),(7.8,25)])

''' axis 1 is transmission '''


min = 50
max = 2000
print(np.amin(transmission_opt*2e6))
im = axs[1].imshow(transmission_opt*2e6, cmap=jqc_red_blue, aspect = 'auto',
                   extent=[Theta[0],Theta[-1],scatter[-1]*1e6,scatter[0]*1e6],
                   norm=LogNorm(min,max))


axs[1].set_xticks(axs[0].get_xticks())

axs[1].tick_params(labelleft=False)


cbar_ax = fig.add_subplot(gs[0,1])
tb = fig.colorbar(im, cax=cbar_ax,orientation = 'horizontal',
                  shrink=0.8, pad=0.15,extend='both')
cbar_ax.xaxis.set_ticks_position('top')

tb.ax.set_title(r'Transmission, $T_\mathrm{opt}$ (ppm)',fontsize=15)
#tb.ax.locator_params(nbins=5)
levels = [150,300,600,1200]

ct = axs[1].contour(transmission_opt*2e6,levels, colors='k', origin='image',
                    extent=[Theta[0],Theta[-1],scatter[-1]*1e6,scatter[0]*1e6])

axs[1].clabel(ct,inline=True,fmt='%d',fontsize=fontsize,
    manual =[(6,267),(4,180),(2.6,90),(1.4,60)])


axs[0].text(0.85,0.05,"(a)",fontsize=20,transform=axs[0].transAxes,color='w')
axs[1].text(0.85,0.05,"(b)",fontsize=20,transform=axs[1].transAxes,color='w')

draw_brace(axs[1],(-1.15,1),-0.13,"Divergence Half-Angle, $\\theta$ (deg)",Radius=150)

plt.subplots_adjust(left=0.15,
                    bottom=0.2,
                    right=0.95,
                    top=0.82,
                    wspace=0.14,
                    hspace=0.1)

fig.savefig(cwd+'\\pdf\\transmission.pdf')
fig.savefig(cwd+'\\png\\transmission.png')


plt.show()
