'''
Script that calculates the optimum cavity over a range of volumes,
scattering losses and misalignments. Saves the data and then returns a pickle
file.
'''

import numpy
import os


from cavity import core

cwd =os.path.dirname(os.path.abspath(__file__))
header = "length (m), ROC (m), Diameter (m), Pext, transmission"

def _cavity_opt(pars):
    ''' function to be passed to each of the workers in the pool '''
    V,S,M,Lmin = pars
    result = core.optimizer(V,S,M,L_lims=(Lmin,3e-3))

    L = result['length']
    R = result['roc']
    D = result['diameter']
    P = result['probability']
    T = result['transmission']

    data =[L,R,D,P,T]
    path = cwd+"\\data\\3D\\Lmin{:.0f} um\\".format(Lmin*1e6)
    fname = "V{:.1f}m3_S{:.1f}ppm_M{:.1f}um.csv".format(numpy.log10(V),S*1e6,M*1e6)
    numpy.savetxt(path+fname,data,header=header,delimiter=',')
    return

if __name__ == "__main__":
    import multiprocessing
    import pickle
    N = 10

    outputpath = cwd+"\\data\\3D"
    if not os.path.exists(outputpath):
        os.makedirs(outputpath)

    vol = numpy.logspace(-15,-12,N)
    scatter = [100e-6]
    mis = [0,5e-6,10e-6]#numpy.linspace(0,25e-6,N-1)

    mins = numpy.linspace(100,500,N-1)*1e-6

    for Lmin in mins:
        dir = cwd+"\\data\\3D\\Lmin{:.0f} um\\".format(Lmin*1e6)
        if not os.path.exists(dir):
            os.makedirs(dir)

    pars = [(v,s,m,l) for v in vol for s in scatter for m in mis for l in mins]

    with multiprocessing.Pool(5) as p:
        p.map(_cavity_opt,pars)


    length = numpy.zeros((len(mins),len(vol),len(scatter),len(mis)))
    roc = numpy.zeros(length.shape)
    diameter = numpy.zeros(length.shape)
    probability = numpy.zeros(length.shape)
    transmission = numpy.zeros(length.shape)
    for l,Lmin in enumerate(mins):
        dir = cwd+"\\data\\3D\\Lmin{:.0f} um\\".format(Lmin*1e6)
        for i,v in enumerate(vol):
            for j,s in enumerate(scatter):
                for k,m in enumerate(mis):
                    fname = "V{:.1f}m3_S{:.1f}ppm_M{:.1f}um.csv".format(numpy.log10(v),s*1e6,m*1e6)
                    dat = numpy.genfromtxt(dir+fname,skip_header=1,delimiter=',')

                    length[l,i,j,k]   = dat[0]
                    roc[l,i,j,k]      = dat[1]
                    diameter[l,i,j,k] = dat[2]

                    probability[l,i,j,k]  = dat[3]
                    transmission[l,i,j,k] = dat[4]

    output =   {"volume":vol,
                "scattering loss":scatter,
                "misalignment": mis,
                "min_length":mins,
                "length":length,
                "roc":roc,
                "diameter":diameter,
                "probability":probability,
                "transmission":transmission}

    outputfile = outputpath+"\\3D_data.pkl"
    with open(outputfile,"wb") as file:
        pickle.dump(output,file)
