function [Dia13, Dia50] = ProfileWidthMain(LineProfile, PixelSize, Display, Scaling)

% Usage:
% Find the left and right cutoffs of a symmetric object along the vertical axis (column) for each line.
% Calculate the 13.53% and 50.00% radius using the left cutoff and peak
% position.
%
% Inputs:
%
%   LineProfile: Horizontal array on which cutoffs will be searched
%   PixelSize: pixel size [um/pixel]
%   Display = 0 no Display, 1 Display
%   Scaling: Scaling factor, required for accuracy
%  
%   All inputs are required.
%
% Output: radii at 13.5% and 50% of the line profile peak.
%
% By Raphael Turcotte (2016)
% Janelia Research Campus
% turcotter@janelia.hhmi.org
%
% Updated
% By Raphael Turcotte (2020)
% University of Oxford
% raphael.turcotte@pharm.ox.ac.uk

%%% ------------------------------------------------------------------- %%%
% Resize the image of the profile
Im1 = ImageResizer(uint32(LineProfile'), Scaling);

%%% ------------------------------------------------------------------- %%%
% Load the resized profile as data
MultiProfile = double(Im1);

%Calculate background
Background = min(MultiProfile);

% Substract background
 MultiProfile = MultiProfile - Background;

% Find max values of profile for normalization normalization.
[Norm,middleTemp] = max(MultiProfile);
maxSet = find(MultiProfile == Norm);
middle = middleTemp + floor(length(maxSet)/2);
MultiProfile = MultiProfile/Norm;

%%% ------------------------------------------------------------------- %%%

% Finds the relevant point in the normalized profile
% All points are in pixels.

CutoffVector = [0.135,0.5,0.865];
[CutoffArray] = FindThreshold(MultiProfile, CutoffVector, middle);

%%% ------------------------------------------------------------------- %%%
% Computes the morphology of Vessel in the longitudinal plane

[Dia13, Dia50] = ComputeDia(CutoffArray(1),CutoffArray(2),CutoffArray(3),CutoffArray(4),CutoffArray(5),CutoffArray(6), PixelSize, Scaling);

%% Output
if Display == 1
    figure('units','normalized','outerposition',[0 0 1 1])
    plot((1:length(MultiProfile)),MultiProfile,'-r','LineWidth',3);
    hold on
    plot([CutoffArray(2),CutoffArray(5)], [0.5,0.5],'bo', 'MarkerSize',15);
    plot([CutoffArray(1),CutoffArray(4)], [0.135,0.135],'go', 'MarkerSize',15);
    plot([middle], [1],'go', 'MarkerSize',20);
    hold off
    pause;
end
