function [CutoffArray] = FindThreshold(MultiProfile, CutoffVector, middle)

% Usage:
% Find the left and right cutoffs of a symmetric object along the vertical axis (column) for each line.
% The cutoffs vectors are returned as a structure.
%
% Output:
%
% Array in which each column is a cutoff vectors
%
%       [CutoffArray] = FindThreshold(MultiProfile, CutoffVector, TimeSampling)
%
%
% Inputs:
%
%    MultiProfile = a 2D array containing a symmetric object along the vertical axis.
%    CutoffVector = a vector of numerical values between 0 and 1.
%    TimeSampling = a numerical value indicating the line sampling number (1/sampling frequency).
%                
%   All inputs are required.
%
% By Raphael Turcotte (2012)
% Wellman Center for Photomedicine
% rturcotte861@gmail.com


% Initialization
LengthOfDim2 = size(MultiProfile);

CutoffArray = zeros(1,2*length(CutoffVector));

% Search for cutoffs
  for j = 1:length(CutoffVector)       
       
        % MultiProfile is divided in two to distinguish left and right cutoff
        % The middle is not to be defined a the center of the image, but rather the
        % center of the vessel
        
        % Find Left cutoff 
        flipped = fliplr(MultiProfile(1:middle));
        position = FindNearest(CutoffVector(j), flipped);
 
        if isempty(position)
            CutoffArray(j) = 0;
        else
            CutoffArray(j) = middle - position(1);
        end
        
        % Find Right cutoff
        position = middle + FindNearest(CutoffVector(j), MultiProfile((middle+1):end));
        if isempty(position)
            CutoffArray(j+length(CutoffVector)) = 0;
            %fprintf('fail');
        else
            CutoffArray(j+length(CutoffVector)) = position(1);
        end
   end
end