#!/usr/bin/env python

# find average structure of a 2D or 3D origami. Find twist and curl

"""
3d twist:
file edge_vhelices specifies top and bottom vhelices for 3d twist calculation. Format top_vhelix1_id, top_vhelix2_id, ...; bottom_vhelix1_id, bottom_vhelix2_id, ...
"""

import readers
import base
import sys
import numpy as np
import os
import origami_utils2 as oru
import pickle
import subprocess

PROCESSDIR = os.path.join(os.path.dirname(__file__), "process_data/")
CONF_SKIP = 0
IGNORE_UNBONDED = False
DEBUG = "trajectory"
DEBUG_TYPE = "reference"
H_BONDS = True

colourindex = range(1000)
colourindex[:3] = [0,3,7]

def angle_sense(v1, v2, axis):
    v1 = oru.norm(v1)
    v2 = oru.norm(v2)
    axis = oru.norm(axis)
    angle = np.arccos(np.dot(v1,v2))
    if np.dot(oru.norm(np.cross(v1,v2)),axis) < 0:
        angle *= -1 # attempt to check the 'sense' of the angle w.r.t. an axis
    return angle

def tcl_colour_preamble(f, colour_scale):
    f.write("color Display Background white\n")
    f.write("mol new\n")
    f.write("color scale method %s\n" % colour_scale)
    for i in range(33, 1056):
        f.write("graphics 0 color %d\n" % i)
        f.write("graphics 0 line {%f %f %f} {%f %f %f} width 1\n" % (20,float(i)/100,0,21,float(i)/100,0))


def tcl_sigma_vec_preamble(f):
    f.write("color Display Background white\n")
    f.write("mol new\n")
    f.write("graphics 0 color 0\n")
    f.write("mol new\n")
    f.write("graphics 1 color 1\n")
    f.write("mol new\n")
    f.write("graphics 2 color 2\n")

def component(v, dir):
    a = np.zeros(3)
    a[dir] = v[dir]
    return a

def writeline_sigma(f, vh_1, vb_1, vh_2, vb_2, dir):
    bbm1 = bbms_av[vh_1][vb_1]
    bbm2 = bbms_av[vh_2][vb_2]
    bbm1_sd1 = bbms_av[vh_1][vb_1] - component(bbms_sigma_vec[vh_1][vb_1], dir)
    bbm1_sd2 = bbms_av[vh_1][vb_1] + component(bbms_sigma_vec[vh_1][vb_1], dir)
    bbm2_sd1 = bbms_av[vh_2][vb_2] - component(bbms_sigma_vec[vh_2][vb_2], dir)
    bbm2_sd2 = bbms_av[vh_2][vb_2] + component(bbms_sigma_vec[vh_2][vb_2], dir)
    f.write("graphics 0 line {%f %f %f} {%f %f %f} width 1\n" % (bbm1[0], bbm1[1], bbm1[2], bbm2[0], bbm2[1], bbm2[2]))
    f.write("graphics 1 line {%f %f %f} {%f %f %f} width 1\n" % (bbm1_sd1[0], bbm1_sd1[1], bbm1_sd1[2], bbm2_sd1[0], bbm2_sd1[1], bbm2_sd1[2]))
    f.write("graphics 2 line {%f %f %f} {%f %f %f} width 1\n" % (bbm1_sd2[0], bbm1_sd2[1], bbm1_sd2[2], bbm2_sd2[0], bbm2_sd2[1], bbm2_sd2[2]))
    
def writeline_colour(f, vh_1, vb_1, vh_2, vb_2, sigmamax, sigmamin, type):
    bbm1 = bbms_av[vh_1][vb_1]
    bbm2 = bbms_av[vh_2][vb_2]
    mid = (bbm1 + bbm2) / 2
    if type == "sigma":
        mag1 = bbms_sigma[vh_1][vb_1]
        mag2 = bbms_sigma[vh_2][vb_2]
        mag1 = float(mag1 - sigmamin)/(sigmamax - sigmamin) * 1023 + 33
        mag2 = float(mag2 - sigmamin)/(sigmamax - sigmamin) * 1023 + 33
    elif type == "unbonded":
        mag1 = min(1056, bbms_unbonded[vh_1][vb_1] * 5 * 1023 + 33)
        mag2 = min(1056, bbms_unbonded[vh_2][vb_2] * 5 * 1023 + 33)
    elif type == "misbonded":
        mag1 = min(1056, bbms_misbonded[vh_1][vb_1] * 5 * 1023 + 33)
        mag2 = min(1056, bbms_misbonded[vh_2][vb_2] * 5 * 1023 + 33)
        # no idea what this was for
#        if mag1 >= 1056:
#            nn = origami.get_nucleotides(origami.vhelix_indices[vh_1],origami.vh_vbase_indices[vh_1][vb_1])[0]
#            nnc = origami.complementary_list[nn]
#        if mag2 >= 1056:
#            nn = origami.get_nucleotides(origami.vhelix_indices[vh_2],origami.vh_vbase_indices[vh_2][vb_2])[0]
#            nnc = origami.complementary_list[nn]
    else:
        print "error in writeline_colour"
        sys.exit()
    f.write("graphics 0 color %d\n" % mag1)
    f.write("graphics 0 line {%f %f %f} {%f %f %f} width 2\n" % (bbm1[0], bbm1[1], bbm1[2], mid[0], mid[1], mid[2]))
    f.write("graphics 0 color %d\n" % mag2)
    f.write("graphics 0 line {%f %f %f} {%f %f %f} width 2\n" % (mid[0], mid[1], mid[2], bbm2[0], bbm2[1], bbm2[2]))

### begin program ###
if (len(sys.argv) < 6):
    print 'Usage %s trajectory topology input output 2d|3d sq|he [-vh_extent] [-analyse] [-no_loops]' % sys.argv[0]
    sys.exit()

conffile = sys.argv[1]
topologyfile = sys.argv[2]
infile = sys.argv[3]
outfile = sys.argv[4]
raw_type = sys.argv[5]
raw_lattice = sys.argv[6]

if raw_type in ("3d","2d"):
    origami_type = raw_type
else:
    base.Logger.log("unrecognised type %s, please enter either 2d or 3d" % raw_type, base.Logger.CRITICAL)
    sys.exit()

if raw_lattice in ("sq", "he"):
    lattice_type = raw_lattice
else:
    base.Logger.log("unrecognised lattice type %s, please enter either sq or he" % raw_type, base.Logger.CRITICAL)
    sys.exit()

if not os.path.isfile(infile):
    base.Logger.log("unable to open file %s" % infile, base.Logger.CRITICAL)
    sys.exit()

myreader = readers.LorenzoReader(conffile,topologyfile)
base.Logger.log("reading conf 1", base.Logger.INFO)
s = myreader.get_system()
s_retain = s.copy()
origami = oru.Origami(s, "virt2nuc")
vh_neighbours = origami.get_vhelix_neighbours(lattice_type)

vhelix_extent = False
if "-vh_extent" in sys.argv[6:]:
    base.Logger.log("Using vh_extent file; only the virtual bases in this region will be analysed", base.Logger.INFO)
    vhelix_extent = []
    # should contain 2 numbers corresponding to the first and last virtual base index of the origami region required
    f = open("vh_extent", "rb")
    for str in f.readline().replace(" ", "").split(","):
        i = int(str)
        if i < 0:
            i = s._strands[0].get_length() + i
        vhelix_extent.append(i)
    vh_begin_raw = vhelix_extent[0]
    vh_end_raw = vhelix_extent[1]

analyse = False
if "-analyse" in sys.argv[6:]:
    base.Logger.log("-analyse: using previously calculated average structure from file bbms_av", base.Logger.INFO)
    f = open(outfile + "_bbms_av", "r")
    pickled = pickle.load(f)
    # deal with all the different pickle files created with older versions of this script
    if len(pickled) == 2:
        bbms_av, bbms_av2 = pickled
        bbms_unbonded = bbms_misbonded = [[0 for y in range(500)] for y in range(60)]
    elif len(pickled) == 4:
        bbms_av, bbms_av2, bbms_unbonded, bbms_misbonded = pickled
        bbvs_av = False
    elif len(pickled) == 6:
        bbms_av, bbms_av2, bbms_unbonded, bbms_misbonded, bbvs_av, bkbkvs_av = pickled
    elif  len(pickled) == 9:
        bbms_av, bbms_av2, bbms_unbonded, bbms_misbonded, bbvs_av, bkbkvs_av, a1_av, a3_av, cm_pos_av = pickled
        bkbkms_av = False
    else:
        bbms_av, bbms_av2, bbms_unbonded, bbms_misbonded, bbvs_av, bkbkvs_av, bkbkms_av, a1_av, a3_av, cm_pos_av = pickled
    f.close()
    analyse = True

no_loops = False
if "-no_loops" in sys.argv[6:]:
    # the program will use the furthest extent of the double helices on each virtual helix as the extent of the origami to be analysed
    base.Logger.log("-no_loops flag detected; ignoring staple/scaffold loops at edge of origami")
    no_loops = True

if vhelix_extent:
    vh_begin = [vh_begin_raw for x in origami.vvib]
    vh_end = [vh_end_raw for x in origami.vvib]
else:
    if no_loops:
        # use extent of double stranded regions
        vh_begin = [x[0] for x in origami.vvib]
#        vh_end = [x[-1] for x in origami.vvib]
        vh_end = [0 for x in origami.vhelix_indices]
        for vhi in range(len(origami.vhelix_indices)):
            vh_end[vhi] = origami.get_vhelix_ds_length(vhi) + vh_begin[vhi] - 1
    else:
        # use extent of scaffold regions
        vh_begin = [x[0] for x in origami.vh_vbase_indices]
        vh_end = [x[-1] for x in origami.vh_vbase_indices]
    
# skip configurations
for counter in range(CONF_SKIP):
    s = myreader.get_system()

# debugging something
if DEBUG == "highlight vhelix":
    myvh = 0
    myvh_id = origami.vhelix_indices.index(myvh)
    scaf_nuc_ids = []
    for vb in origami.vh_vbase_indices[myvh_id]:
        for nuc in origami.get_nucleotides(myvh, vb):
            scaf_nuc_ids.append(nuc)

if not analyse:
    if DEBUG == "trajectory":
        f = open(outfile + "_traj.xyz", "w")
        f.write("")
        f.close()
    
    # find average base-base midpoints by averaging over the whole trajectory
    a1_av = [0 for x in range(s._N)]
    a3_av = [0 for x in range(s._N)]
    cm_pos_av = [np.zeros(3) for x in range(s._N)]
    nuc_conf_count = [[0 for x in range(origami.get_vhelix_ds_length(vhi))] for vhi in range(len(origami.vhelix_indices))]
    bbvs_av = [[np.zeros(3) for x in range(origami.get_vhelix_ds_length(vhi))] for vhi in range(len(origami.vhelix_indices))]
    bkbkvs_av = [[np.zeros(3) for x in range(origami.get_vhelix_ds_length(vhi))] for vhi in range(len(origami.vhelix_indices))]
    bkbkms_av = [[np.zeros(3) for x in range(origami.get_vhelix_ds_length(vhi))] for vhi in range(len(origami.vhelix_indices))]
    bbms_av = [[np.zeros(3) for x in range(origami.get_vhelix_ds_length(vhi))] for vhi in range(len(origami.vhelix_indices))]
    bbms_av2 = [[np.zeros(3) for x in range(origami.get_vhelix_ds_length(vhi))] for vhi in range(len(origami.vhelix_indices))]
    bbms_unbonded = [[0 for x in range(origami.get_vhelix_ds_length(vhi))] for vhi in range(len(origami.vhelix_indices))]
    bbms_misbonded = [[0 for x in range(origami.get_vhelix_ds_length(vhi))] for vhi in range(len(origami.vhelix_indices))]
    conf_counter = 1
    while s != False:
        if H_BONDS:
            origami.get_h_bond_list(infile, conffile, conf_counter-1)
        # find the co-ordinate system that is "invariant under rotation" (hopefully) for this configuration
        if origami_type == "3d":
            # 3D origami
            rm = origami.get_3d_vecs(vhelix_extent = (vh_begin, vh_end), discard_unbonded = IGNORE_UNBONDED)
        elif len(origami.vhelix_indices) < 2:
            # ds/ssDNA
            rm = origami.get_1d_vecs(discard_unbonded = IGNORE_UNBONDED)

        else:
            # 2D origami (sheet)

            origami.get_plane_vecs(discard_unbonded = IGNORE_UNBONDED)
            zprime = np.cross(origami.vec_long, origami.vec_lat)
            zprime /= np.sqrt(np.dot(zprime, zprime))
            xprime = origami.vec_long
            yprime = -np.cross(xprime, zprime)
            yprime /= np.sqrt(np.dot(yprime, yprime))
            rm = np.array([xprime, yprime, zprime])
            
        # TODO : make xprime,yprime,zprime into a matrix

        # find centre of mass in "invariant" co-ordinate system
        com = np.zeros(3)
        bbm_count = 0
        for vh in origami.vhelix_indices:
            for vb in origami.vvib[origami.vhelix_indices.index(vh)]:
                for nuc in origami.get_nucleotides(vh, vb):
                    bbm = origami.get_bb_midpoint(nuc)
                    bbm_count += 1
                    com += np.dot(rm, bbm)

        com /= bbm_count

        # use the "invariant" co-ordinate system and com correction to add the current base-base midpoint position vectors to the record
        if DEBUG == "trajectory":
            f = open(outfile + "_traj.xyz", "a")
            f.write("%d\n" % (bbm_count))
            f.write("Trajectory with invariant co-ordinates\n")

        for ii in range(s._N):
            nuc = s._nucleotides[ii]
            a1 = nuc._a1
            a3 = nuc._a3
            cmp = np.dot(rm, nuc.cm_pos)
            
            cmp -= com
            # matrices would be nicer
            a1 = np.dot(rm, a1)
            a3 = np.dot(rm, a3)

            a1_av[ii] += a1
            a3_av[ii] += a3
            cm_pos_av[ii] += cmp

        for vh_i in range(len(origami.vhelix_indices)):
            nuci = 0
            for vb_i in range(len(origami.vh_vbase_indices[vh_i])):
                vh = origami.vhelix_indices[vh_i]
                vb = origami.vh_vbase_indices[vh_i][vb_i]
                # work out the bbms in invariant co-ordinates
                for nuc in origami.get_nucleotides(vh, vb):
                    bbm = origami.get_bb_midpoint(nuc)
                    bbv = origami.get_bb_vec(nuc)
                    bkbkv = origami.get_backback_vec(nuc)
                    bkbkm = origami.get_backback_midpoint(nuc)
                    if isinstance(bbm, np.ndarray):
                        # rotate+translate to new co-ordinate system
                        bbm = np.dot(rm, bbm) - com
                        bbv = np.dot(rm, bbv)
                        bkbkv = np.dot(rm, bkbkv)
                        bkbkm = np.dot(rm, bkbkm)
                        if DEBUG == "trajectory":
                                f.write("C %f %f %f\n" % (bbm[0], bbm[1], bbm[2]))
                        # add to average
                        bbms_av2[vh_i][nuci] += np.dot(bbm, bbm) # this should be right.....
                        bbms_av[vh_i][nuci] += bbm
                        bbvs_av[vh_i][nuci] += bbv
                        bkbkvs_av[vh_i][nuci] += bkbkv
                        bkbkms_av[vh_i][nuci] += bkbkm
                        # check for misbonding/unbonded
                        if H_BONDS:
                            if len(origami.interaction_list[nuc]) == 0:
                                bbms_unbonded[vh_i][nuci] += 1
                            elif origami.interaction_list[nuc][0] != origami.complementary_list[nuc]:
                                bbms_misbonded[vh_i][nuci] += 1
                        nuc_conf_count[vh_i][nuci] += 1
                        nuci += 1

        s = myreader.get_system()
        if s:
            conf_counter += 1
            base.Logger.log("reading conf %d" % conf_counter, base.Logger.INFO)
            origami.update_system(s)

    if DEBUG == "trajectory":
        f.close()
        base.Logger.log("wrote invariant co-ordinate xyz trajectory file %s" % outfile+"_traj.xyz", base.Logger.INFO)
    if H_BONDS:
        bbms_unbonded = [[float(y)/conf_counter for y in x] for x in bbms_unbonded]
        bbms_misbonded = [[float(y)/conf_counter for y in x] for x in bbms_misbonded]

    for i in range(len(bbms_av)):
        for j in range(len(bbms_av[i])):
            bbms_av[i][j] /= nuc_conf_count[i][j]
            bbms_av2[i][j] /= nuc_conf_count[i][j]
            bbvs_av[i][j] /= nuc_conf_count[i][j]
            bkbkvs_av[i][j] /= nuc_conf_count[i][j]
            bkbkms_av[i][j] /= nuc_conf_count[i][j]
            
    for ii in range(s_retain._N):
        a1_av[ii] = oru.norm(a1_av[ii])
        a3_av[ii] = oru.norm(a3_av[ii])
        cm_pos_av[ii] /= conf_counter
        
    f = open(outfile+"_bbms_av", "wb")
    pickle.dump((bbms_av, bbms_av2, bbms_unbonded, bbms_misbonded, bbvs_av, bkbkvs_av, bkbkms_av, a1_av, a3_av, cm_pos_av), f)
    f.close()
    base.Logger.log("dumped a load of useful stuff to pickle file %s_bbms_av - in future use this file by using -analyse flag" % outfile, base.Logger.INFO)

# standard deviation - here defined for a point P as the sqrt of (the sum of the variances of the co-ordinates of P in each dimension minus the squared distance of P from the origin)
bbms_sigma = [[0 for x in range(origami.get_vhelix_ds_length(vhi))] for vhi in range(len(origami.vhelix_indices))]
bbms_sigma_vec = [[np.zeros(3) for x in range(origami.get_vhelix_ds_length(vhi))] for vhi in range(len(origami.vhelix_indices))]
for vhi in range(len(bbms_av)):
    for vbfi in range(len(bbms_av[vhi])):
        for j in range(3):
            bbms_sigma[vhi][vbfi] += bbms_av2[vhi][vbfi][j] - bbms_av[vhi][vbfi][j]*bbms_av[vhi][vbfi][j]
            bbms_sigma_vec[vhi][vbfi][j] = np.sqrt(bbms_av2[vhi][vbfi][j] - bbms_av[vhi][vbfi][j]*bbms_av[vhi][vbfi][j])
        bbms_sigma[vhi][vbfi] = np.sqrt(bbms_sigma[vhi][vbfi])

working_list = [0 for x in bbms_sigma]
for vh_i in range(len(working_list)):
    working_list[vh_i] = bbms_sigma[vh_i][origami.vh_vbase_indices[vh_i].index(vh_begin[vh_i]):origami.vh_vbase_indices[vh_i].index(vh_end[vh_i])+1]

mins = [min(x) for x in working_list]
sigma_min = min(mins)
maxs = [max(x) for x in working_list]
sigma_max = max(maxs)

av_helix_axis = np.zeros(3)
for i in range(len(bbms_av)):
    av_helix_axis += bbms_av[i][-1] - bbms_av[i][0]
av_helix_axis /= len(bbms_av)
    
# twist and curl
if origami_type == "3d":
    # twist
    try:
        f = open("edge_vhelices", "r")
    except IOError:
        base.Logger.log("could not find file edge_vhelices; skipping twist/curl calculation", base.Logger.INFO)
    else:
        str1, str2 = f.readline().replace(" ","").split(";")
        vhelices1 = [int(x) for x in str1.split(",")]
        vhelices2 = [int(x) for x in str2.split(",")]
        top = np.zeros(3)
        bottom = np.zeros(3)
        top_left = np.zeros(3)
        top_right = np.zeros(3)
        bottom_left = np.zeros(3)
        bottom_right = np.zeros(3)
        for vh in vhelices1:
            vh_i = origami.vhelix_indices.index(vh)
            vec1 = bbms_av[vh_i][origami.vh_vbase_indices[vh_i].index(vh_begin[vh_i])]
            vec2 = bbms_av[vh_i][origami.vh_vbase_indices[vh_i].index(vh_end[vh_i])]
            top_left += vec1
            top_right += vec2
            top += vec2 - vec1

        for vh in vhelices2:
            vh_i = origami.vhelix_indices.index(vh)
            vec1 = bbms_av[vh_i][origami.vh_vbase_indices[vh_i].index(vh_begin[vh_i])]
            vec2 = bbms_av[vh_i][origami.vh_vbase_indices[vh_i].index(vh_end[vh_i])]
            bottom_left += vec1
            bottom_right += vec2
            bottom += vec2 - vec1

        top /= len(vhelices1)
        top_left /= len(vhelices1)
        top_right /= len(vhelices1)
        bottom /= len(vhelices2)
        bottom_left /= len(vhelices2)
        bottom_right /= len(vhelices2)

        left = top_left - bottom_left
        right = top_right - bottom_right

        twist1 = angle_sense(top,bottom,left+right)
        twist2 = angle_sense(left,right,av_helix_axis) # possibly twist2 doesn't allow -ve angles as it's currently defined???
        twist1 *= 180./np.pi
        twist2 *= 180./np.pi

        print twist1, twist2
else:
    aa = range(len(origami.vhelix_indices))
    bb = range(len(origami.vhelix_indices))

    for i in range(len(origami.vhelix_indices)):
        aa[i] = bbms_av[i][origami.vvib[i].index(vh_begin[i])]
        bb[i] = bbms_av[i][origami.vvib[i].index(vh_end[i])]

    # twist
    left = aa[0]-aa[-1]
    right = bb[0]-bb[-1]
    top = aa[0] - bb[0]
    bottom = aa[-1] - bb[-1]
    twist = angle_sense(right,left,av_helix_axis)
    twist2 = angle_sense(top, bottom, left+right)
    twist *= 180./np.pi
    twist2 *= 180./np.pi

    # curl
    curl = "n/a"
    if len(aa) > 2:
        curl_a = 0
        for i in range(len(aa)-2):
            n1 = oru.norm(aa[i+2] - aa[i+1])
            n2 = oru.norm(aa[i+1] - aa[i])
            curl_a += angle_sense(n1,n2,av_helix_axis)

        curl_b = 0
        for i in range(len(bb)-2):
            n1 = oru.norm(bb[i+2] - bb[i+1])
            n2 = oru.norm(bb[i+1] - bb[i])
            curl_b += angle_sense(n1,n2,av_helix_axis)

        if curl_a * curl_b == -1:
            base.Logger.log("curl in opposite directions", base.Logger.WARNING)
            sys.exit()

        curl = (curl_a + curl_b)/2 
        curl *= 180./np.pi
        curl /= len(aa)-2

    print twist, twist2, curl

# calculate bp by bp twist
if bbvs_av and bkbkms_av:
    f_bp_twist = open("bp_twist.dat", "w")
    f_bp_twist_back = open("bp_twist_back.dat", "w")
    for i in range(len(bbvs_av)):
        for j in range(len(bbvs_av[i])-1):
            # use normalised vector between consecutive base-base midpoints n, normalised vectors between each base pair bbv1 and bbv2
            n = bbms_av[i][j+1] - bbms_av[i][j]#bbms_av[0][-4] - bbms_av[0][3]
            bbv1 = bbvs_av[i][j]
            bbv2 = bbvs_av[i][j+1]
            n = oru.norm(n)
            # find bbv1', bbv2', the components of bbv1 bbv2 that are in the plane perpendicular to n
            bbv1prime = bbv1 - np.dot(bbv1, n) * n
            bbv2prime = bbv2 - np.dot(bbv2, n) * n
            bbv1prime = oru.norm(bbv1prime)
            bbv2prime = oru.norm(bbv2prime)
            # finally find their dot product
            bp_twist =  np.arccos(np.dot(bbv1prime, bbv2prime))
            bp_twist *= 180./np.pi
            f_bp_twist.write("%d %d %f\n" % (i, j, bp_twist))

            ### backbone-backbone version
            n = bkbkms_av[i][j+1] - bkbkms_av[i][j]#n = bkbkms_av[0][-4] - bkbkms_av[0][3]
            n = oru.norm(n)
            bbv1 = bkbkvs_av[i][j]
            bbv2 = bkbkvs_av[i][j+1]
            # find bbv1', bbv2', the components of bbv1 bbv2 that are in the plane perpendicular to n
            bbv1prime = bbv1 - np.dot(bbv1, n) * n
            bbv2prime = bbv2 - np.dot(bbv2, n) * n
            bbv1prime = oru.norm(bbv1prime)
            bbv2prime = oru.norm(bbv2prime)
            # finally find their dot product
            bp_twist =  np.arccos(np.dot(bbv1prime, bbv2prime))
            bp_twist *= 180./np.pi
            f_bp_twist_back.write("%d %d %f\n" % (i, j, bp_twist))

    base.Logger.log("wrote bp by bp twist file %s" % "bp_twist.dat", base.Logger.INFO)
    base.Logger.log("wrote bp by bp twist file using backbones instead of bases %s" % "bp_twist_backbone.dat", base.Logger.INFO)
    f_bp_twist.close()
else:
    base.Logger.log("rerun without -analyse flag to find bp by bp twist data", base.Logger.INFO)


# print detailed average structure - currently we take a configuration from the trajectory (could be first or last one) and change all the positions and orientations of the nucleotides
s_new = base.System(s_retain._box)
nuci = 0
for strand in s_retain._strands:
    strand_new = base.Strand()
    for nuc in strand._nucleotides:
        nuc_new = base.Nucleotide(cm_pos_av[nuci], a1_av[nuci], a3_av[nuci], 0)
        strand_new.add_nucleotide(nuc_new)
        nuci += 1
    s_new.add_strand(strand_new, check_overlap = False)
s_new.print_tcl_detailed_output(outfile+"_detailed.tcl", labels=True)
s_new.print_pdb_output_chimera(outfile+"_detailed.pdb")
s_new.print_tcl_cylinder_output(outfile+"_cylinder.tcl", show_labels=False, colour=True, show_box=False)
s_new.print_lorenzo_output(outfile+"_lorenzo.conf", outfile+"_lorenzo.top")
base.Logger.log("wrote detailed tcl file %s" % outfile+"_detailed.tcl", base.Logger.INFO)
base.Logger.log("wrote detailed pdb file %s" % outfile+"_detailed.pdb", base.Logger.INFO)
base.Logger.log("wrote cylinder tcl file %s" % outfile+"_cylinder.tcl", base.Logger.INFO)
base.Logger.log("wrote lorenzo files %s %s" % (outfile+"_lorenzo.conf", outfile+"_lorenzo.top"), base.Logger.INFO)

# print various files containing information from the average base-base midpoints and their standard deviation
f0 = open(outfile + "_sigma.tcl", "w")
tcl_colour_preamble(f0, "BGR")
f_un = open(outfile + "_un.tcl", "w")
tcl_colour_preamble(f_un, "BGR")
f_mis = open(outfile + "_mis.tcl", "w")
tcl_colour_preamble(f_mis, "BGR")

fsx = open(outfile + "_sigmax.tcl", "w")
fsy = open(outfile + "_sigmay.tcl", "w")
fsz = open(outfile + "_sigmaz.tcl", "w")
tcl_sigma_vec_preamble(fsx)
tcl_sigma_vec_preamble(fsy)
tcl_sigma_vec_preamble(fsz)

f2 = open(outfile + ".xyz", "w")
f2.write("%d\n" % (origami.num_vh * origami.width))
f2.write("Average structure of an origami\n")
for bbms_row in bbms_av:
    for bbm in bbms_row:
        f2.write("C %f %f %f\n" % (bbm[0], bbm[1], bbm[2]))
f2.close()
base.Logger.log("wrote base-base midpoint xyz file %s" % outfile+".xyz", base.Logger.INFO)

f3 = open(outfile + ".tcl", "w")
f3.write("color Display Background white\n")
f3.write("mol new\n")
f3.write("graphics 0 color 0\n")
f4 = open(outfile + "_vhelix.tcl", "w")
f4.write("color Display Background white\n")
f4.write("mol new\n")
f4.write("graphics 0 color 0\n")

# to highlight crossovers in mesh tcl representation
xovers = False
if len(origami.vhelix_indices) > 1:
    xovers = origami.get_holliday_junctions(single=True)

# draw all the lines for the output mesh files
# draw lines across vhelices
for vh_i in range(len(bbms_av[:-1])):
    vh = origami.vhelix_indices[vh_i]
    # we draw lines for all neighbours so I guess most of the lines are drawn twice - I don't think this matters very much
    for vh_n in vh_neighbours[vh]:
        vh_in = origami.vhelix_indices.index(vh_n)
        # for now we demand that skips line up, and we ignore loops (do not draw them)
        for vb_i in range(len(origami.vvib[vh_i])):
            vb = origami.vvib[vh_i][vb_i]
            lnucs = len(origami.get_nucleotides(vh, vb))
            if lnucs > 0:
                if vb in origami.vvib[vh_in]:
                    nuci1 = origami.vb2nuci(vh_i,vb)
                    nuci2 = origami.vb2nuci(vh_in,vb)
                    bbm1 = bbms_av[vh_i][nuci1]
                    bbm2 = bbms_av[vh_in][nuci2]
                    # check for crossovers - we highlight them red
                    if xovers and ([vh, origami.vhelix_indices[vh_in], vb] in xovers or [origami.vhelix_indices[vh_in], vh, vb] in xovers):
                        f3.write("graphics 0 color 1\n")
                        f3.write("graphics 0 line {%f %f %f} {%f %f %f} width 2\n" % (bbm1[0], bbm1[1], bbm1[2], bbm2[0], bbm2[1], bbm2[2]))
                        f3.write("graphics 0 color 0\n")
                        f4.write("graphics 0 color 1\n")
                        f4.write("graphics 0 line {%f %f %f} {%f %f %f} width 10\n" % (bbm1[0], bbm1[1], bbm1[2], bbm2[0], bbm2[1], bbm2[2]))
                        f4.write("graphics 0 color 0\n")
                    else:
                        f3.write("graphics 0 line {%f %f %f} {%f %f %f} width 1\n" % (bbm1[0], bbm1[1], bbm1[2], bbm2[0], bbm2[1], bbm2[2]))
                    writeline_colour(f0, vh_i, nuci1, vh_in, nuci2, sigma_max, sigma_min, "sigma")
                    writeline_colour(f_un, vh_i, nuci1, vh_in, nuci2, sigma_max, sigma_min, "unbonded")
                    writeline_colour(f_mis, vh_i, nuci1, vh_in, nuci2, sigma_max, sigma_min, "misbonded")
                    writeline_sigma(fsx, vh_i, nuci1, vh_in, nuci2, 0)
                    writeline_sigma(fsy, vh_i, nuci1, vh_in, nuci2, 1)
                    writeline_sigma(fsz, vh_i, nuci1, vh_in, nuci2, 2)
# draw lines along vhelices
colourid = 0
for i in range(len(bbms_av)):
    for j in range(len(bbms_av[i][:-1])):
        if origami.vvib[i][j] >= vh_begin[i] and origami.vvib[i][j] < vh_end[i]:
            bbm1 = bbms_av[i][j]
            bbm2 = bbms_av[i][j+1]
            if DEBUG == "highlight vhelix" and origami.get_nucleotides(origami.vhelix_indices[i],origami.vvib[i][j])[0] in scaf_nuc_ids:
                f3.write("graphics 0 color 1\n")
                f3.write("graphics 0 line {%f %f %f} {%f %f %f} width 2\n" % (bbm1[0], bbm1[1], bbm1[2], bbm2[0], bbm2[1], bbm2[2]))
                f3.write("graphics 0 color 0\n")
            else:
                f4.write("graphics 0 color %d\n" % colourindex[colourid])
                f4.write("graphics 0 line {%f %f %f} {%f %f %f} width 10\n" % (bbm1[0], bbm1[1], bbm1[2], bbm2[0], bbm2[1], bbm2[2]))
                f3.write("graphics 0 line {%f %f %f} {%f %f %f} width 1\n" % (bbm1[0], bbm1[1], bbm1[2], bbm2[0], bbm2[1], bbm2[2]))
            writeline_colour(f0, i, j, i, j+1, sigma_max, sigma_min, "sigma")
            writeline_colour(f_un, i, j, i, j+1, sigma_max, sigma_min, "unbonded")
            writeline_colour(f_mis, i, j, i, j+1, sigma_max, sigma_min, "misbonded")
            writeline_sigma(fsx, i, j, i, j+1, 0)
            writeline_sigma(fsy, i, j, i, j+1, 1)
            writeline_sigma(fsz, i, j, i, j+1, 2)
    colourid += 1
f3.close()
base.Logger.log("wrote vhelix bbm file %s" % outfile+"_vhelix.tcl", base.Logger.INFO)
base.Logger.log("wrote mesh tcl file %s" % outfile+".tcl", base.Logger.INFO)
base.Logger.log("wrote standard deviation mesh tcl file %s" % outfile+"_sigma.tcl", base.Logger.INFO)
base.Logger.log("wrote standard deviation mesh tcl file %s" % outfile+"_sigmax.tcl", base.Logger.INFO)
base.Logger.log("wrote standard deviation mesh tcl file %s" % outfile+"_sigmay.tcl", base.Logger.INFO)
base.Logger.log("wrote standard deviation mesh tcl file %s" % outfile+"_sigmaz.tcl", base.Logger.INFO)


f4 = open(outfile + ".var", "w")
for i in range(len(bbms_sigma)):
    for j in range(len(bbms_sigma[i])):
        f4.write("%d %d %f\n" % (i, j, bbms_sigma[i][j]))
f4.close()
base.Logger.log("wrote standard deviation file %s" % outfile+".var", base.Logger.INFO)

