#!/usr/bin/env python

import sys
import matplotlib.pyplot as plt
import numpy as np
import pylab
from matplotlib import rcParams, rc
from scipy import ma
from matplotlib.ticker import NullFormatter
import parse_pt_fe as ppf

def plot_window():
	plt.rc('font', **{'family':'serif', 'serif':['Computer Modern Roman'],
                  'monospace':['Computer Modern Typewriter']})
	fig_width_pt = 246.0  # Get this from LaTeX using \showthe\columnwidth
	inches_per_pt = 1.0/72.27               # Convert pt to inch
	golden_mean = (np.sqrt(5)-1.0)/2.0         # Aesthetic ratio
	fig_width = fig_width_pt*inches_per_pt  # width in inches
	fig_height = fig_width*golden_mean      # height in inches
	fig_size =  [fig_width,fig_height]
	params = {'backend': 'ps',
	          'axes.labelsize': 11,
	          'text.fontsize': 11,
	          'legend.fontsize': 10,
	          'xtick.labelsize': 10,
	          'ytick.labelsize': 10,
	          'xtick.major.size' : 2.5,
			  'xtick.major.width' : 0.5,
			  'ytick.major.size' : 2.5,
              'ytick.major.width' : 0.5,
	          'text.usetex': True,
	          #'figure.figsize': fig_size,
	          'text.latex.preamble': [r"\usepackage{amstext}", r"\usepackage{mathpazo}"]}
	fig = plt.figure(1, figsize = (6,6), facecolor='w', edgecolor='k')
	rcParams.update(params)
	return fig

save = False
if len(sys.argv) > 1 and sys.argv[1] == "-save":
	save = True

# plotting histograms alongside a 2d histogram: from http://matplotlib.org/examples/pylab_examples/scatter_hist.html
# definitions for the axes
left, width = 0.1, 0.65
bottom, height = 0.1, 0.65
left_h = left + width + 0.02
bottom_h = bottom + height + 0.02

rect_2d = [left, bottom, width, height]
rect_histx = [left, bottom_h, width, 0.2]
rect_histy = [left_h, bottom, 0.2, height]

# start with a rectangular Figure
fig = plot_window()#1, figsize=(12,8))

ax2d = plt.axes(rect_2d)

sss1, miny, maxy, xvals, yvals, xticks, yticks, xticklabels, yticklabels = ppf.load_2d_file("phi_theta_fe.dat")
im = pylab.imshow( sss1, interpolation='nearest', aspect="auto", origin = 'center', vmin=miny, vmax=maxy, cmap='hot' )

axHistx = plt.axes(rect_histx)
axHisty = plt.axes(rect_histy)

# no labels
axHistx.xaxis.set_major_formatter(NullFormatter())
axHisty.yaxis.set_major_formatter(NullFormatter())

# make the colourbar
cb_pad = 0.03
cb_left = 0.1 + cb_pad
cb_bottom = 0.65
cb_width = 0.3
cb_height = bottom + height - cb_bottom - cb_pad
cax = fig.add_axes([cb_left, cb_bottom, cb_width, cb_height])
cbar = fig.colorbar(im, cax=cax, orientation='horizontal')#, shrink=0.5, aspect=20, fraction=.12,pad=.02)
cbar.set_label(r"$\Delta G / (k_{\rm B}T)$")
cbar.set_clim(miny, maxy)

# main plot ticks
ax2d.set_xlabel(r'$\phi$')
ax2d.set_xticks(xticks)
ax2d.set_xticklabels(xticklabels)
ax2d.set_ylabel(r'$\theta$')
ax2d.set_yticks(yticks)
ax2d.set_yticklabels(yticklabels)

# x-histogram
xdat = ppf.load_phi_hist("phi_fe.dat")
axHistx.plot(xdat[0], xdat[1])
axHistx.set_xticks(range(0,360,45))
axHistx.set_xlim([min(xvals)-0.5, max(xvals)+1.5])
axHistx.set_ylabel(r"$\Delta G / (k_{\rm B}T)$")

# y-histogram
ydat_all = ppf.load_theta_hist("theta_fe.dat")
ydat_phi_lt_180 = ppf.load_theta_hist("theta_fe_phi_lt_180.dat")
ydat_phi_gt_180 = ppf.load_theta_hist("theta_fe_phi_gt_180.dat")

axHisty.plot(ydat_all[1], ydat_all[0])
axHisty.plot(ydat_phi_lt_180[1], ydat_phi_lt_180[0])
axHisty.plot(ydat_phi_gt_180[1], ydat_phi_gt_180[0], linestyle = "dashed")

axHisty.set_yticks(range(-180,180,10))
axHisty.set_ylim([min(yvals)-0.5, 30.0+1])
axHisty.set_xlabel(r"$\Delta G / (k_{\rm B}T)$")

# vertical line
ax2d.axvline(180-6.5, color = "blue", linestyle = '--')

if save:
	plt.savefig("phi_theta_fe.pdf", bbox_inches = "tight", transparent = True)
else:
	plt.show()
