""" Rotational states for a symmetric top including inversion.
"""

import scipy.constants as c

class RotationalStates(object):
    def energy(self, J, K, p): 
        """ Energy of a state defined by `(J, K, p)` in J.  Parity `p` is
        0 for `-` and 1 for `+`.
        """

        return c.h * 1e6 * (self.ezero[p] + self.brot[p]*J**2 + self.cb[p]*K**2 +
                self.ej[p]*J**4 + self.ejk[p]*J**2*K**2 + self.ek[p]*K**4 +
                self.hj[p]*J**6 + self.hjjk[p]*J**4*K**2 + 
                self.hjkk[p]*J**2*K**4 + self.hk[p]*K**6)


    @staticmethod
    def degeneracy(J, K):
        if K==0:
            return 12 * J + 6
        elif K%3==0:
            return 24*J+12
        else:
            return 12*J+6

    @staticmethod
    def generateStates(jmax=8):
        """ Returns a generator for a list of the rotational state quantum
        numbers in the same order used in the calculation.
        """

        for j in range(jmax+1):
            for k in range(j+1):
                sym = 'A' if k%3==0 else 'E'
                for p in range(2):
                    yield (j, k, p, sym)

 
class NH3States(RotationalStates):
    """ Rotational constants for NH3.
    """
    ezero=(-23785.895135, 0.)
    brot=(298192.92072, 298041.38282)
    cb=(-111497.0613, -111285.5732)
    ej=(25.461658, 24.956414)
    ejk=(-47.27790, -45.88426)
    ek=(27.35488, 26.40027)
    hj=(7.1836e-3, 6.0083e-3)
    hjjk=(-0.021997, -0.017179)
    hjkk=(0.023479, 0.016947)
    hk=(-8.154e-3, -5.22463e-3)
