import numpy as np
import scipy.constants as c

class MolecularBeam(object):
    def __init__(self, m, cp, molfrac, d, p0, t0, sigma):
        """ Initialise a molecular beam composed of a mixture of species.

        Species parameters are stored in lists `m`, `cp`, and `molfrac`, and
        averaged parameters are calculated and used for the remainder of the
        model.

        Args:
          m (float, list) : List of species masses in u.
          cp (float, list) : Species heat capacity (J/mol/K)
          molfrac (float, list) : Mole fraction of each species.
          d (float) : Nozzle diameter (mm).
          p0 (float) : Source pressure (bar).
          t0 (float) : Source temperature (bar).
          sigma (float) : Hard-sphere collision cross section in m^2
        """

        self._mbar = np.sum(np.array(m) * np.array(molfrac)) * c.u
        cpbar = np.sum(np.array(cp) * np.array(molfrac))
        self._gammabar = 1/(1 - (c.R/cpbar))

        self.d = d 
        self._t0 = t0
        self.p0 = p0 

        self._sigma = sigma

        # Molecular beam constants from Ashkenas and Sherman
        self._a = 3.375 # 3.26
        self._z0 = 8.534e-5 # 0.075e-3


    def mach(self, z):
        """ Mach number at distance `z`.
        """
        return self._a * ((z - self._z0)/self._d)**(self._gammabar-1) - 0.5 * (
                ((self._gammabar+1)/(self._gammabar-1)) / 
                (self._a * ((z-self._z0)/self._d))**(self._gammabar-1))


    def t(self, z):
        """ Beam temperature at distance `z`.
        """
        return self._t0 / (1 + self.mach(z)**2 * (self._gammabar-1)/2)


    def density(self, z):
        """ Beam density at distance `z` in cm^{-3}.
        """
        dens = self._dens0 * (1 + self.mach(z)**2 * (self._gammabar-1)/2)**(
                -1/(self._gammabar-1))
        return dens / 1e6


    def u(self, z):
        """ Beam velocity at distance `z` in m s^{-1}.
        """
        return (self.mach(z) *
                np.sqrt(self._gammabar * c.k * self._t0/self._mbar) *
                (1 + ((self._gammabar-1)/2 * self.mach(z)**2))**(-0.5))


    def colFreq(self, z):
        """ Hard-sphere collision frequency at distance `z` in s^{-1}.
        """
        return np.sqrt(2) * self._sigma * np.sqrt(16 * c.k * self.t(z) /
                (c.pi * self._mbar)) * self.density(z) * 1e6


    # Source gas parameters are entered in units of bar and mm for pressure and
    # nozzle diameter. The source density also needs to be recalculated if
    # pressure or temperature change.
    def updateSourceDensity(self):
        self._dens0 = self._p0 / (c.k * self._t0)

    @property
    def p0(self):
        return self._p0 / c.atmosphere

    @p0.setter
    def p0(self, val):
        self._p0 = val * c.atmosphere
        self.updateSourceDensity()

    @property
    def t0(self):
        return self._t0

    @t0.setter
    def t0(self, val):
        self._t0 = val
        self.updateSourceDensity()

    @property
    def d(self):
        return self._d / 1e-3

    @d.setter
    def d(self, val):
        self._d = val * 1e-3
