""" Generate a paper-sized plot of the velocity of the molecular beam as a
function of distance from the nozzle, as given by the DSMC model. Overlay a
comparison with the analytical molecular beam model.
"""
import h5py
import matplotlib.pyplot as plt
import numpy as np

from FigureSetup import he_dsmc_line, nh3_dsmc_line
from Parameters import NH3HeBeam
from molecularBeam import MolecularBeam
import FigureSetup


h5f = h5py.File('2502.hdf5', 'r')

def plot_speed(d, ax, plotdict, std_alpha=0.25):
    # Averaging over nXn central subcells.
    avg_n = 5
    centre_slice = slice(200-avg_n, 200+avg_n)

    speed_avg = np.nanmean(d[:,centre_slice, centre_slice, 0], (1, 2))
    speed_std = np.nanstd(d[:,centre_slice, centre_slice, 0], (1, 2))

    x = np.arange(d.shape[0]) * d.attrs['boxdx'] * 1000
    ax.fill_between(x, speed_avg-speed_std, speed_avg+speed_std, 
            alpha=std_alpha, color=plotdict['color'], linewidth=0)
    ax.plot(x, speed_avg, **plotdict)


fig, ax = FigureSetup.new_figure()


plot_speed(h5f['velocity/he'], ax, he_dsmc_line)
plot_speed(h5f['velocity/nh3'], ax, nh3_dsmc_line)

# Analytical model
mb = MolecularBeam(NH3HeBeam.m, NH3HeBeam.cp, NH3HeBeam.molfrac, 
        NH3HeBeam.d, NH3HeBeam.p0, NH3HeBeam.t0, NH3HeBeam.sigma)

x = np.linspace(1.0e-3, h5f.attrs['boxc1'], 100)
ax.plot(x*1000, mb.u(x), label='Analytical',
        linestyle='-', dashes=[4, 10], color='black')

ax.set_ylim(0, 1800)
ax.set_xlabel(r'Distance (mm)')
ax.set_ylabel(r'Speed (m s$^{-1}$)')
legend = ax.legend(loc=4)
legend.get_frame().set_edgecolor('0.9')
legend.get_frame().set_facecolor('0.9')

fig.set_tight_layout(pad=0.1)
fig.savefig('velocity.pdf')
plt.show()
