""" Generate a paper-sized plot of the density of He and NH3 as a function of
distance from the nozzle given by the DSMC model. Overlay with the predicion of
an analytical molecular beam model, scaled by the mole fraction of each
species.
"""
import h5py
import matplotlib.pyplot as plt
import numpy as np

from FigureSetup import he_model_line, he_dsmc_line, nh3_model_line, nh3_dsmc_line
from Parameters import NH3HeBeam
from molecularBeam import MolecularBeam
import FigureSetup

def plot_density(d, ax, plotdict, std_alpha=0.25, x_pos=1.5e-3):
    """ Plot the transverse density distribution at location x_pos from nozzle.
    Average dataset `d` over central region and plot as a line in `ax` with
    shading to +- 1 standard deviation.
    """
    # Averaging over nXn central subcells.
    avg_n = 5

    nx = int(x_pos/d.attrs['boxdx'])
    mid_y = d.shape[2]/2

    centre_slice = slice(nx-avg_n, nx+avg_n)

    scale = d.attrs['fn']/(d.attrs['boxdx']*d.attrs['boxdy']*d.attrs['boxdz'])
    dens_avg = np.nanmean(d[centre_slice, mid_y, :], (0)) * 1e-6 * scale
    dens_std = np.nanstd(d[centre_slice, mid_y, :], (0)) * 1e-6 * scale

    x = np.arange(-mid_y, mid_y) * d.attrs['boxdx'] * 1000
    ax.fill_between(x, dens_avg-dens_std, dens_avg+dens_std, 
            alpha=std_alpha, color=plotdict['color'], linewidth=0)
    ax.plot(x, dens_avg, **plotdict)


fig, ax = FigureSetup.new_figure()

# Load DSMC data from HDF5 file.
h5f = h5py.File('2502.hdf5', 'r')
plot_density(h5f['density/he'], ax, he_dsmc_line)
plot_density(h5f['density/nh3'], ax, nh3_dsmc_line)

# Graph parameters.
ax.set_yscale('log')
ax.set_ylim(ymin=5e14,ymax=1e20)
ax.set_xlabel(r'Distance (mm)')
ax.set_ylabel(r'Density (cm$^{-3}$)')
legend = ax.legend(ncol=2, loc=1)
legend.get_frame().set_edgecolor('0.9')
legend.get_frame().set_facecolor('0.9')

fig.tight_layout(pad=0.1)
fig.savefig('density.pdf')
plt.show()
