import h5py
import matplotlib.pyplot as plt
import scipy.constants as C
import numpy as np
from scipy import stats, optimize, integrate

from FigureSetup import he_dsmc_line, nh3_dsmc_line
from Parameters import NH3HeBeam
from molecularBeam import MolecularBeam
from rotationalStates import NH3States
import FigureSetup

# Load data from HDF5 file.
h5f = h5py.File('2502.hdf5', 'r')
population = h5f['rotation/population']

# Generate lists of the energy, degenercy and symmetry of each state.
nh3 = NH3States()
indA = [i for i, s in enumerate(nh3.generateStates()) if s[3]=='A']
indE = [i for i, s in enumerate(nh3.generateStates()) if s[3]=='E']
energy = np.array([nh3.energy(j, k, p) for j, k, p, s in nh3.generateStates()])
degeneracy = np.array([nh3.degeneracy(j, k) for j, k, p, s in nh3.generateStates()])


def tRotAt(z, sym):
    """ Rotational temperature for symmetry specified by character sym='A', 'E'
    at index z. Perform a Boltzmann fit for states in the selected symmetry
    within each 6x6 central slice of population data. Take the mean and
    standard deviation of the temperatures over this central region.
    """
    assert sym in ['A', 'E'], 'Symmetry is not A or E'
    g = degeneracy[eval('ind'+sym)]
    e = energy[eval('ind'+sym)]

    T_cell = np.zeros((6,6))
    for i in xrange(6):
        for j in xrange(6):
            p = population[z,i,j,eval('ind'+sym)]
            y = np.log(p/g)

            grad, _, _, _, _= stats.linregress(e, y)

            T_cell[i,j] = -1/(C.k * grad)

    T = np.nanmean(T_cell)
    dT = np.nanstd(T_cell)

    return T, dT


fig, ax = FigureSetup.new_figure()
ax.hold(True)

x = np.arange(population.shape[0]) * population.attrs['boxdx']
a_color = '#ff800e'
e_color = '#5f9ed1'
std_alpha = 0.25

# Generate A state plot.
t_rot = np.array([tRotAt(z, 'A')[0] for z in range(len(x))])
t_rot_std = np.array([tRotAt(z, 'A')[1] for z in range(len(x))])
ax.fill_between(x*1000, t_rot-t_rot_std, t_rot+t_rot_std,
        alpha=std_alpha, color=a_color, linewidth=0)
ax.plot (x*1000, t_rot, label='A', color=a_color)

# Generate E state plot
t_rot = np.array([tRotAt(z, 'E')[0] for z in range(len(x))])
t_rot_std = np.array([tRotAt(z, 'E')[1] for z in range(len(x))])
ax.fill_between(x*1000, t_rot-t_rot_std, t_rot+t_rot_std,
        alpha=std_alpha, color=e_color, linewidth=0)
ax.plot (x*1000, t_rot, label='E', color=e_color)

# Set graph parameters.
ax.set_xlabel('Distance (mm)')
ax.set_ylabel('T$_{\mathrm{rot}}$ (K)')

ax.set_ylim((20, 65))
legend = ax.legend()
legend.get_frame().set_edgecolor('0.9')
legend.get_frame().set_facecolor('0.9')

fig.tight_layout(pad=0.1)
fig.savefig('rotationalTemperature.pdf')

plt.show()

